#ifndef __AUTHNLIB_NSSUTIL_H__
#define __AUTHNLIB_NSSUTIL_H__

#include <string>

//#include "cache.h"
#include "canlxx.h"
#include "datetime.h"

namespace AuthN {
namespace NSS {

  class CSRContext;

  // A wrapper class for access nss interface, it includes two 
  // functionalities: 
  // 1, import a certificate or key from external source to
  // the nss database: ImportCertFrom*ToDB, ImportCertFrom*ToDB. 
  // Note that the proxy certificate can also be import into nss db, then 
  // by using the method SSL_CTX_use_pkcs11, the proxy credential in side
  // nss db can be used for the SSL communication. By this way, we can
  // remove the requirement of the original file-based proxy.
  // Note that trusted CA certificates can also be import into nss db,
  // then we can remove the requirement of original file-based CA directory.
  // 2. use the CA certificates in nss database to verify the peer
  // certificate (proxy), by using nss' pkix interface.
 
  class CertContext {
    private:
      std::string configdir_;
      std::string slotpass_;
      std::string certname_;
      std::string trusts_;
      std::string cert_str_;
      std::vector<std::string> cert_chain_str_;
      Context* context_;
      bool nss_initialized_;
      Status last_error_;

      //AuthN::Utils::Cache* ocsp_cache_;
      //std::string ocsp_cache_file;
      //AuthN::Utils::Cache* create_cache();
      //void remove_cache();

      CertContext():context_(NULL), nss_initialized_(false) {}; //, ocsp_cache_(NULL) {};

    public:
      /// Standard constructor
      CertContext(Context& ctx);

      /// Constructor
      CertContext(Context& ctx, const std::string& configdir);

      /// Deconstructor
      ~CertContext();

      // old_pass is not needed when setting the pass for a new db
      bool SetNSSDBPass(const std::string& new_pass, const std::string& old_pass = "");

      /// Sign a certificate based on the given options and certificate request
      void signRequest(CSRContext& csr, int duration, std::string& cert_str);

      void certToDER(std::string& out) const;

      void certToPEM(std::string& out) const;

      bool certFromDER(const std::string& in, const std::string& certname, 
        const std::string& trusts = "");

      bool certFromPEM(const std::string& in, const std::string& certname, 
        const std::string& trusts = "");

      bool certFromStr(const std::string& in, const std::string& certname, 
        const std::string& trusts = "");

      bool certFromFile(const char* certfile, const std::string& certname, 
        const std::string& trusts = "");

      bool keyFromDER(const std::string& keystr, const std::string& keyname);
     
      bool keyFromPEM(const std::string& keystr, const std::string& keyname);

      // Import key from outside sources into nss db. 
      // Note if both key and cert needs to be imported, the key should
      // be imported before the cert, otherwise PK11_FindKeyByAnyCert (if
      // called under the same nss session) will not succeed to retrive 
      // private key based on cert.
      bool ImportKeyFromStrToDB(const std::string& in, const std::string& keyname);

      bool ImportKeyFromFileToDB(const char* keyfile, const std::string& keyname);

      bool ImportCertFromStrToDB(const std::string& certstr, 
        const std::string& certname, const std::string& trusts = "");

      bool ImportCertFromFileToDB(const char* certfile, 
        const std::string& certname, const std::string& trusts = "");

      /// Import CRL from a directory, this only happens when 
      /// importing the CA certificates in a directory into nss db
      bool ImportCertFromDirToDB(const char* certdir, const std::string& trusts = "");

      CertContext* getChainCert(int pos);

      bool getCertExtension(int pos, AuthN::Credentials::Extension& ext) const;

      bool getCertExtension(const std::string& name, AuthN::Credentials::Extension& ext) const;

      bool getCertAttributes(const std::string& name, std::list<std::string>& attrs) const;

      /// Check if this certificate is equal to another certificate,
      bool operator==(const CertContext& other);

      /// Check if this certificate is an issuer of another
      bool isIssuerOf(const CertContext *other) const;

      /// Check the validity of this certificate, the CA and CRL will
      /// be fetched from files that configured in Context
      Status validate(const AuthN::Validator::ValidationMode& mode, AuthN::Context* ctx = NULL);

      operator bool(void) const;

      bool operator!(void) const;

  };

  // A wrapper class for generating key pair and certificate request.
  // In the principal of pkcs11, the private key is supposed to be 
  // inside the certificate database on a software token (e.g. nss db) or
  // hardware token, and not to be exposed to external.
  // But we provide the possibility to expose the private key, so that 
  // the origianl file-based credential for proxy (RFC 3820) can be composed
  // (with private key as part of the proxy content).
 
  class CSRContext {
    private:
      std::string configdir_;
      std::string privkey_name_;
      std::string slotpass_;
      std::string csr_str_;
      std::string privkey_str_;
      Context* context_;
      bool nss_initialized_;
      Status last_error_;

      /// Standard constructor
      CSRContext() : context_(NULL) { };

    public:
      /// Constructor
      CSRContext(Context& ctx, const std::string& configdir, const std::string& passwd);

      /// Deconstructor
      ~CSRContext();

      /// Create a certificate request
      bool createRequest(const std::string& privkey_name);

      /// Set CSR
      void setRequest(const char* csr_filename);

      /// Set CSR
      void setRequest(const std::string& csrstr);

      /// Get CSR string
      void getRequest(std::string& req_str) { if(!csr_str_.empty()) req_str.assign(csr_str_); };

      /// Get priv key string
      void getPrivKey(std::string& privk_str) { if(!privkey_str_.empty()) privk_str.assign(privkey_str_); };

      /// Check if this csr is equal to another csr
      bool operator==(const CSRContext& other) { return ((!csr_str_.empty()) && (!csr_str_.compare(other.csr_str_))); };

      operator bool(void) const {
        return !csr_str_.empty();
      };

      bool operator!(void) const {
        return csr_str_.empty();
      };
  };

  // A wrapper class for importing CRL files (usaually inside the CA directory) 
  // into nss db, so that CRL can be checked when verifying the peer certificate.

  class CRLContext {
    private:
      std::string configdir_;
      Context* context_;
      bool nss_initialized_;
      Status last_error_;

      /// Standard constructor
      CRLContext() : context_(NULL) { };

    public:
      /// Constructor
      CRLContext(Context& ctx, const std::string& configdir);

      /// Deconstructor
      ~CRLContext();

      /// Import CRL from a string
      bool ImportCRLFromStrToDB(const std::string& crlstr, const std::string& crl_url = "");

      /// Import CRL from a file      
      bool ImportCRLFromFileToDB(const char* crlfile, const std::string& crl_url = "");

      /// Import CRL from a directory  
      bool ImportCRLFromDirToDB(const char* crldir);
  };

  struct certInfo {
    std::string certname;
    std::string subject_dn;
    std::string issuer_dn;
    unsigned long serial;
    AuthN::Utils::Time start;
    AuthN::Utils::Time end;
  };

  /**
   * Initializes nss library
   * @param configdir   full path to the nss db
   */
  bool nssInit(Context& context, const std::string& configdir);  

  void nssFinish();

  bool nssExportCertificate(Context& context, const std::string& certname, const char* certfile, bool ascii = true);

  bool nssExportCertificate(Context& context, const std::string& certname, std::string& certstr, bool ascii = true);

  bool nssOutputPKCS12(Context& context, const std::string& certname, const char* outfile, const char* slotpw, const char* p12pw);

  bool nssOutputPKCS12(Context& context, const std::string& certname, std::string& outstr, const char* slotpw, const char* p12pw);

  bool nssGenerateCSR(Context& context, const std::string& privkey_name, const std::string& dn, const char* slotpw, const char* outfile, std::string& privk_str, bool ascii = true);

  bool nssGenerateCSR(Context& context, const std::string& privkey_name, const std::string& dn, const char* slotpw, std::string& outstr, std::string& privk_str, bool ascii = true);

  void nssListUserCertificatesInfo(std::list<certInfo>& certInfolist);

  bool nssCreateCert(Context& context, const char* csrfile, const std::string& issuername, const char* passwd, const int duration, const std::string& vomsacseq, const char* outfile, bool ascii = true);

  bool nssCreateCert(Context& context, const std::string& csrstr, const std::string& issuername, const char* passwd, const int duration, const std::string& vomsacseq, std::string& outstr, bool ascii = true);

  bool nssImportCertAndPrivateKey(Context& context, const char* slotpw, const char* keyfile, const std::string& keyname, const char* certfile, const std::string& certname, const char* trusts = NULL, bool ascii = true);

  bool nssImportCertAndPrivateKey(Context& context, const char* slotpw, const std::string& keystr, const std::string& keyname, const std::string& certstr, const std::string& certname, const char* trusts, bool ascii = true);

  bool nssImportCert(Context& context, const char* slotpw, const char* certfile, const std::string& name, const char* trusts = NULL, bool ascii = true);
  bool nssImportCert(Context& context, const char* slotpw, const std::string& certstr, const std::string& name, const char* trusts, bool ascii = true);

  bool nssImportPrivateKey(Context& context, const char* slotpw, const char* keyfile, const std::string& nick_str);

  bool nssImportPrivateKey(Context& context, const char* slotpw, const std::string& keystr, const std::string& nick_str);

  bool nssVerifyCert(Context& context, const char* slotpw, const char* certfile, bool use_pkix, const std::vector<const char*>& untrusted_certfiles, const std::vector<const char*>& trusted_certfiles, const std::string& cert_policy);

  bool nssVerifyCert(Context& context, const char* slotpw, const std::string& certstr, bool use_pkix, const std::vector<std::string>& untrusted_certs, const std::vector<std::string>& trusted_certs, const std::string& cert_policy);

  bool nssImportCRL(Context& context, const char* slotpw, const char* crlfile, const std::string& crl_url);

  bool nssImportCRL(Context& context, const char* slotpw, const std::string& certstr, const std::string& crl_url);


  int SSL_CTX_use_pkcs11 (Context* context, SSL_CTX* ssl_ctx, const std::string& pkcs11_id);


}
}

#endif // __AUTHNLIB_NSSUTIL_H__
