#ifndef __AUTHNLIB_CACHE_H__
#define __AUTHNLIB_CACHE_H__

#include <string>
#include <db_cxx.h>

#include "canlxx.h"
#include "datetime.h"

namespace AuthN {
namespace Utils {

  typedef bool cursor_delete_cb(const std::string& key, const std::string& value);

class BDB {
  public:
    //The db_home parameter is the database environment's home directory
    //The db_name parameter is used as the name of an underlying file that will be used to back the database
    BDB(const std::string &db_home, const std::string &db_name, AuthN::Context* context);

    ~BDB();
    bool put(const std::string &name, const std::string &content);
    std::string get(const std::string &name);
    void cursor_init();
    bool cursor_walk(std::string& name, std::string& value);
    bool cursor_del(); // Delete the item at current cursor
    void cursor_close();
    void del(const std::string &name);
    void del_with_cb(cursor_delete_cb* cb);
    std::vector<std::string> get_doc_names();
    operator bool(void) {return db_ != NULL;}
    bool operator!(void) {return db_ == NULL;}
   
  private:
    DbEnv* env_;
    Db* db_;
    Dbc* cursor_;
    std::string db_home_;
    std::string db_name_;
    AuthN::Context* context_;
};

class Cache {
  public:
    enum CacheType{
      CacheInMemory = 0x01,
      CacheInFile     = 0x02
    };
    // In Memory Cache, use the "In-memory database" of BDB, 
    // in this case the db_name is empty
    // Note some log files of BDB are still written onto file system
    Cache(const CacheType& type) : cache_type(type) {};
    //In File Cache
    Cache(const std::string& db_home, const std::string& db_name, AuthN::Context* ctx);
    //In File Cache
    Cache(const std::string& cache_file, AuthN::Context* ctx);

    Cache(AuthN::Context* ctx);
    ~Cache() { if(cache_db != NULL) delete cache_db; };

    void SetCacheType(const CacheType& type) { cache_type = type; };
    void SetCacheCheckInterval(const long interval) { cache_check_interval = interval; };

    void StoreItem(const std::string& id, const std::string& data, long till);
    std::string RetrieveItem(const std::string& id);
 
    operator bool(void) {return cache_db != NULL;}
    bool operator!(void) {return cache_db == NULL;}

    static bool record_del_cb(const std::string& key, const std::string& value); 

  private:
    void expiration_check();
    BDB* cache_db;
    CacheType cache_type;
    long last_cached_check;
    long cache_check_interval;
    AuthN::Context* context; 
};  


} // namespace Utils
} // namespace AuthN

#endif // __AUTHNLIB_CACHE_H__

