/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017-2020 Yegor Bugayenko
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.cactoos.io;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * Files and folders in a directory.
 *
 * <p>There is no thread-safety guarantee.
 *
 * @since 0.21
 */
public final class Directory implements Iterable<Path> {

    /**
     * Path of the directory.
     */
    private final Path dir;

    /**
     * Ctor.
     * @param file File as a path to directory.
     */
    public Directory(final File file) {
        this(file.toPath());
    }

    /**
     * Ctor.
     * @param path Path of the dir
     */
    public Directory(final Path path) {
        this.dir = path;
    }

    @Override
    public Iterator<Path> iterator() {
        try (Stream<Path> files = Files.walk(this.dir)) {
            return files.collect(Collectors.toList()).iterator();
        } catch (final IOException ex) {
            throw new IllegalStateException(ex);
        }
    }

}
