/** BEGIN COPYRIGHT BLOCK
 * Copyright (C) 2001 Sun Microsystems, Inc. Used by permission.
 * Copyright (C) 2005 Red Hat, Inc.
 * All rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * END COPYRIGHT BLOCK **/

package com.netscape.admin.dirserv.panel.replication;

import java.util.StringTokenizer;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import netscape.ldap.*;
import com.netscape.admin.dirserv.*;
import com.netscape.admin.dirserv.panel.*;
import com.netscape.management.client.*;
import com.netscape.management.client.util.*;

/**
 * Panel for Directory Server Consumer Replication Setting
 *
 * @author  jpanchen
 * @version %I%, %G%
 * @date	 	9/15/97
 * @see     com.netscape.admin.dirserv.panel.replication
 */
public class LegacyConsumerSettingPanel extends BlankPanel {
    
    /*==========================================================
     * constructors
     *==========================================================*/
     
    /**
     * public constructor
     * construction is delayed until selected.
     * @param parent parent panel
     */
    public LegacyConsumerSettingPanel(IDSModel model) {
		super(model, "replication");
		setTitle(_resource.getString(_section+"-setting","title"));
		_model = model;
		_helpToken = "configuration-replication-legacyconsumersettings-help";
		_refreshWhenSelect = false;
    }
	
    /*==========================================================
     * public methods
     *==========================================================*/
         
    /**
     * Actual panel construction
     */
    public void init() {
		if (_isInitialized) {
			return;
		}
		_entryExists = false;
		try {
			LDAPConnection ldc = _model.getServerInfo().getLDAPConnection();
			_legacyEntry = ldc.read(REPSETTINGS_DN);
			if (_legacyEntry != null) {
				_entryExists = true;
			}
		} catch (LDAPException lde) {
		}
		
		_isLegacyConsumerEnabled = _entryExists;
		
		GridBagConstraints gbc = getGBC();	
		
		_myPanel.setLayout(new GridBagLayout());
		_myPanel.setBackground(getBackground());
		_myPanel.setPreferredSize(ReplicationTool.DEFAULT_PANEL_SIZE);
		_myPanel.setMaximumSize(ReplicationTool.DEFAULT_PANEL_SIZE);
		
		//add Description label
        JTextArea desc = makeMultiLineLabel( 2, 60,
											 _resource.getString(
																 _section,"setting-desc") );
		gbc.anchor = gbc.NORTHWEST;
        gbc.gridwidth = 2;
		gbc.weightx    = 1.0;
		gbc.fill = gbc.HORIZONTAL;
        _myPanel.add(desc, gbc);
		gbc.weightx    = 0.0;
		gbc.fill = gbc.HORIZONTAL;
		gbc.gridwidth = gbc.REMAINDER;
		_myPanel.add(Box.createGlue(),gbc);
		
		// Switch on/off
		_cbEnable = makeJCheckBox( _section,
								   "enable",
								   _entryExists,
								   _resource);
		_cbEnable.setSelected( _entryExists );
		_cbEnable.addItemListener(this);
		gbc.gridwidth = 2;
		gbc.weightx    = 0;
		gbc.fill = gbc.NONE;
		_myPanel.add(_cbEnable,gbc);
		gbc.weightx    = 1.0;
		gbc.fill = gbc.HORIZONTAL;
		gbc.gridwidth = gbc.REMAINDER;
		_myPanel.add(Box.createGlue(),gbc);	    
		
		// Auth param
		ReplicationTool.resetGBC(gbc);
		gbc.insets = (Insets) ReplicationTool.BOTTOM_INSETS.clone();
        gbc.anchor = gbc.NORTH;
        gbc.gridwidth = gbc.REMAINDER;
        gbc.weightx=1.0;
		_myPanel.add(createAuth(), gbc);

		checkEnabledField();
		addBottomGlue ();
        _isInitialized = true;
    }
	
    private JPanel createAuth(){
        //setup Normal Auth panel		
		String title = _resource.getString(_section+"-normalAuth","label");
        JPanel panel = new GroupPanel(title);        
        
		//add supplier DN
        _dnLabel = makeJLabel(_resource.getString(_section+"-supplierDN","label"));				
        _dnLabel.setToolTipText(_resource.getString(_section+"-supplierDN","ttip"));
		_dnLabel.resetKeyboardActions();
		
		GridBagConstraints gbc = new GridBagConstraints();
   		ReplicationTool.resetGBC(gbc);
		gbc.insets = (Insets) ReplicationTool.BOTTOM_INSETS.clone();
        gbc.anchor = gbc.NORTHEAST;
		gbc.gridwidth  = gbc.RELATIVE;
		gbc.fill = gbc.NONE;
		gbc.weightx = 0.0;		
        panel.add(_dnLabel, gbc);
		
        _dnText= makeJTextField(_section, "supplierDN", _resource);
		_dnLabel.setLabelFor(_dnText);
		_saveDN = DSUtil.getAttrValue( _legacyEntry, UPDATEDN_ATTR_NAME );
		if( _saveDN != null ) {
			_dnText.setText( _saveDN );
		} else {
			_dnText.setText("");
		}		

		gbc.gridwidth  = gbc.REMAINDER;
        gbc.anchor = gbc.NORTHWEST;
		gbc.fill = gbc.HORIZONTAL;
		gbc.weightx = 1.0;
        panel.add(_dnText, gbc);
		
        
        // passwd 
        _pwdLabel = makeJLabel(_resource.getString(_section+"-SupplierNewPwd","label"));
        _pwdLabel.setToolTipText(_resource.getString(_section+"-SupplierNewPwd","ttip"));
		_pwdLabel.resetKeyboardActions();
 
        gbc.anchor = gbc.NORTHEAST;
		gbc.gridwidth  = gbc.RELATIVE;
		gbc.fill = gbc.NONE;
		gbc.weightx = 0.0;
        panel.add(_pwdLabel, gbc);
		
        _pwdText= makeJPasswordField(_section, "SupplierNewPwd",
									 "", 10, _resource);
		_pwdLabel.setLabelFor(_pwdText);
		_savePwd = DSUtil.getAttrValue( _legacyEntry, UPDATEDN_PASSWORD_ATTR_NAME);
		if( _savePwd != null){
			_pwdText.setText( _savePwd );
		} else {
			_savePwd = "";
			_pwdText.setText("");
		}

        gbc.anchor = gbc.NORTHWEST;
		gbc.gridwidth  = gbc.REMAINDER;
		gbc.fill = gbc.HORIZONTAL;
		gbc.weightx = 1.0;
        panel.add(_pwdText, gbc);
		
        // pwd again
        _againLabel= makeJLabel(_resource.getString(_section+"-SupplierNewPwd","again"));
		_againLabel.resetKeyboardActions(); 
		
		gbc.anchor = gbc.NORTHEAST;
		gbc.gridwidth  = gbc.RELATIVE;
		gbc.fill = gbc.NONE;
		gbc.weightx = 0.0;
        panel.add(_againLabel, gbc);
		
		_againText= makeJPasswordField(_section, "SupplierNewPwd",
									   "", 10, _resource);		
		_againLabel.setLabelFor(_againText);
        gbc.anchor = gbc.NORTHWEST;
		gbc.fill = gbc.HORIZONTAL;
		gbc.gridwidth  = gbc.REMAINDER;
		// empty passwd is not valid
		_againText.setText( _savePwd );
        panel.add(_againText, gbc);        
		
		return(panel);
    }

   /**
     * Update on-screen data from Directory.
     *
     **/
    public boolean refresh () {
		Debug.println("LegacyConsumerPanel.refresh()");
		/* We re-read the entry from the server */
		_entryExists = false;
		try {
			LDAPConnection ldc = _model.getServerInfo().getLDAPConnection();
			_legacyEntry = ldc.read(REPSETTINGS_DN);
			if (_legacyEntry != null) {
				_entryExists = true;
			}
		} catch (LDAPException lde) {
			_legacyEntry = null;
		}
		_isLegacyConsumerEnabled = _entryExists;
		_saveDN = DSUtil.getAttrValue( _legacyEntry, UPDATEDN_ATTR_NAME );		
		_savePwd = DSUtil.getAttrValue( _legacyEntry, UPDATEDN_PASSWORD_ATTR_NAME);

		checkOkay();
		return true;
	}

    /**
     * Update on-screen data from Directory.
	 *
	 * Note: we overwrite the data that the user may have modified.  This is done in order to keep
	 * the coherency between the refresh behaviour of the different panels of the configuration tab.
     *
     **/
    public void refreshFromServer () {
		Debug.println("LegacyConsumerPanel.refreshFromServer()");
		resetCallback();
	}
	

    public void resetCallback() {
		/* We re-read the entry from the server */
		_entryExists = false;
		try {
			LDAPConnection ldc = _model.getServerInfo().getLDAPConnection();
			_legacyEntry = ldc.read(REPSETTINGS_DN);
			if (_legacyEntry != null) {
				_entryExists = true;
			}
		} catch (LDAPException lde) {
			_legacyEntry = null;
		}
		_isLegacyConsumerEnabled = _entryExists;
		_saveDN = DSUtil.getAttrValue( _legacyEntry, UPDATEDN_ATTR_NAME );		
		_savePwd = DSUtil.getAttrValue( _legacyEntry, UPDATEDN_PASSWORD_ATTR_NAME);

		if( _saveDN != null){
			_dnText.setText( _saveDN );
		} else {
			_dnText.setText("");
		}
		_cbEnable.setSelected( _entryExists );
		_isDNDirty = false;
		_pwdText.setText( _savePwd );
		_againText.setText( _savePwd );
		_isPwdDirty = false;	
		checkOkay();
    }

    public void okCallback() {
		if (!_isValid ||
			!_isDirty) {
			return;
		}
		LDAPConnection ldc = _model.getServerInfo().getLDAPConnection();
		/* We read again the entry to check if it exists ...*/
		_entryExists = false;
		try {
			_legacyEntry = ldc.read(REPSETTINGS_DN);
			if (_legacyEntry != null) {
				_entryExists = true;
			}
		} catch (LDAPException lde) {
			_legacyEntry = null;
		}
		checkStatus();

		String pwd = new String( _pwdText.getPassword() );		
		
		if( _mustDelete ) {
			try {
				ldc.delete(  REPSETTINGS_DN );
				_entryExists = false;
				_mustDelete = false;
				_dnText.setText("");
				_pwdText.setText("");
				_againText.setText("");
			} catch (LDAPException e) {
				/* If the entry does not exists, it's already removed, and we don't display any error message */
				if (e.getLDAPResultCode() == LDAPException.NO_SUCH_OBJECT) {
					_entryExists = false;
					_mustDelete = false;
					_dnText.setText("");
					_pwdText.setText("");
					_againText.setText("");
				} else {
					String ldapError =  e.errorCodeToString();
					String ldapMessage = e.getLDAPErrorMessage();
					if ((ldapMessage != null) &&
						(ldapMessage.length() > 0)) {
						ldapError = ldapError + ". "+ldapMessage;
					}
					String[] args_m = { ldapError };
					DSUtil.showErrorDialog( _model.getFrame(),
											"error-del-legacy",
											args_m,
											_section,
											_resource);
					checkOkay();
					return;
				}
			}
		} else if (!_entryExists) {
			
			LDAPAttributeSet las = new LDAPAttributeSet();
			
			// add the more well known attributes and values
			las.add(new LDAPAttribute("objectclass",
									  REPSETTINGS_OBJECTCLASSES));
			las.add(new LDAPAttribute("cn",
									  REPSETTINGS_CN));
			las.add(new LDAPAttribute(UPDATEDN_ATTR_NAME,
									  _dnText.getText()));
			las.add(new LDAPAttribute(UPDATEDN_PASSWORD_ATTR_NAME,
									  pwd));	
			LDAPEntry newEntry = new LDAPEntry(REPSETTINGS_DN, las);
			
			try {
				ldc.add( newEntry );				
			} catch(LDAPException e) {
				String ldapError =  e.errorCodeToString();
				String ldapMessage = e.getLDAPErrorMessage();
				if ((ldapMessage != null) &&
					(ldapMessage.length() > 0)) {
					ldapError = ldapError + ". "+ldapMessage;
				}
				String[] args_m = { ldapError };
				DSUtil.showErrorDialog( _model.getFrame(),
										"error-add-legacy",
										args_m,
										_section,
										_resource);
				checkOkay();
				return;
			}	    
			_legacyEntry = newEntry;
			_entryExists = true;		    
		} else {
			LDAPModificationSet attrs = new LDAPModificationSet();
			attrs.add(LDAPModification.REPLACE,
					  new LDAPAttribute("nsslapd-legacy-updatedn",
										_dnText.getText()));
			attrs.add(LDAPModification.REPLACE,
					  new LDAPAttribute("nsslapd-legacy-updatepw",
										pwd));	
			try {
				ldc.modify( _legacyEntry.getDN(), attrs );
			} catch(LDAPException e) {				
				String ldapError =  e.errorCodeToString();
				String ldapMessage = e.getLDAPErrorMessage();
				if ((ldapMessage != null) &&
					(ldapMessage.length() > 0)) {
					ldapError = ldapError + ". "+ldapMessage;
				}
				String[] args_m = { ldapError };			
				DSUtil.showErrorDialog( _model.getFrame(),
										"error-mod-legacy",
										args_m,
										_section,
										_resource);
				checkOkay();
				return;
			}
		}
		_saveDN = _dnText.getText();
		_savePwd = new String(_pwdText.getPassword());
		checkOkay();
    }


    public void changedUpdate(DocumentEvent e) {
		insertUpdate(e);
    }
	
    public void insertUpdate(DocumentEvent e) {
		if (!_isInitialized) return;
		checkOkay();		
    }
	
    public void removeUpdate(DocumentEvent e) {
		changedUpdate(e);
    }

	public void itemStateChanged(ItemEvent e) {
		if(e.getSource().equals(_cbEnable)) {			
			checkOkay();			
		}
	}

	public void actionPerformed(ActionEvent e) {
		checkOkay();		
	}

    private void checkPwd(){
		String p1 = new String( _pwdText.getPassword() );
		String p2 = new String( _againText.getPassword() );
		
		// (p1.length() == 0) && ( p2.length() == 0) in case of SSL strong auth
		// for the rest i think there is a min and of course they must be the same
		_isPasswdOK = (((p1.length() == 0) && ( p2.length() == 0)) ||
					   (( p1.length() >= _pwdMinLength ) && 
						( p2.length() >= _pwdMinLength ) &&
						( p1.compareTo( p2 ) == 0)));
		
		_isPwdDirty = !p1.equals(_savePwd);
		
		if( _isPasswdOK ) {
			if( _isPwdDirty){
				setChangeState( _pwdLabel, CHANGE_STATE_MODIFIED);
				setChangeState( _againLabel, CHANGE_STATE_MODIFIED);
			} else {
				setChangeState( _pwdLabel, CHANGE_STATE_UNMODIFIED);
				setChangeState( _againLabel, CHANGE_STATE_UNMODIFIED);
			}
		} else {
			setChangeState( _pwdLabel, CHANGE_STATE_ERROR);
			setChangeState( _againLabel, CHANGE_STATE_ERROR);
		}
    }
	
    private void checkDN(){
		_isDNDirty = !DSUtil.equalDNs(_dnText.getText(), _saveDN);
		_isDNOK = DSUtil.isValidDN( _dnText.getText());
		if( _isDNOK ) {
			if( _isDNDirty ) {
				setChangeState( _dnLabel, CHANGE_STATE_MODIFIED);
			} else {
				setChangeState( _dnLabel, CHANGE_STATE_UNMODIFIED );
			}
		} else {
			setChangeState( _dnLabel, CHANGE_STATE_ERROR);
		}
    }

	private void checkEnable() {
		_isEnableDirty = (_cbEnable.isSelected() != _entryExists);
		if (_isEnableDirty) {
			setChangeState( _cbEnable, CHANGE_STATE_MODIFIED);
		} else {
			setChangeState( _cbEnable, CHANGE_STATE_UNMODIFIED );
		}
	}
    
    private void checkOkay() {				
		checkEnabledField();
		checkPwd();
		checkDN();
		checkEnable();
		
		/* if the check box enable is not selected, what we have is always valid */
		_isValid = _isPasswdOK && _isDNOK || !_cbEnable.isSelected();
		_isDirty = (_isPwdDirty || _isDNDirty || _isEnableDirty)  &&
			(_isEnableDirty || _cbEnable.isSelected());  /* We don't consider dirty if the user has kept the 
															legacy disabled ... even if the other fields have been 
															modified */
		if( _mustDelete ) {
			setDirtyFlag();
			setValidFlag();
		} else {
			if( _isDirty ){				
				setDirtyFlag();
			} else {				
				clearDirtyFlag();
			}
			if (_isValid) {				
				setValidFlag();				
			} else {		
				clearValidFlag();				
			}
		}
    }


    private void checkStatus() {
		_mustDelete = _entryExists && !_cbEnable.isSelected();
		_isLegacyConsumerEnabled = _cbEnable.isSelected();
    }
    
    /*==========================================================
     * private methods
     *==========================================================*/	 
    private void clearPWDFields() {
        _pwdText.setText("");
        _againText.setText("");	    
    }	    	  

    private void checkEnabledField(){
		checkStatus();
		_dnLabel.setEnabled( _isLegacyConsumerEnabled );
		_dnText.setEnabled( _isLegacyConsumerEnabled );
		_pwdText.setBackground( _dnText.getBackground());
		_againText.setBackground( _dnText.getBackground());
		_pwdLabel.setEnabled( _isLegacyConsumerEnabled );
		_pwdText.setEnabled( _isLegacyConsumerEnabled );
		_againLabel.setEnabled( _isLegacyConsumerEnabled );
		_againText.setEnabled( _isLegacyConsumerEnabled );
    }
    
    /*==========================================================
     * variables
     *==========================================================*/
    private IDSModel	_model;
    private boolean		_entryExists;
	private boolean		_isValid = true;
	private boolean		_isDirty = false;
    private boolean		_isDNDirty = false;
    private boolean		_isDNOK = true;
    private boolean		_isPwdDirty = false;
    private boolean		_isPasswdOK = true;
	private boolean     _isEnableDirty = false;
    private boolean		_mustDelete;
    private boolean		_isLegacyConsumerEnabled;
	
    private JTextField		_dnText;
    private JPasswordField	_pwdText;
    private JPasswordField	_againText;
    private JCheckBox		_cbEnable = null;
    private JLabel		_dnLabel;
    private JLabel		_againLabel;
    private JLabel		_pwdLabel;
    private String		_savePwd;
    private String		_saveDN;
    
//     DSEntryConfirmPassword _againDSEntry;
//     DSEntryPassword _pwdDSEntry;
    private LDAPEntry		_legacyEntry;

    //get resource bundle
    private static ResourceSet _resource =
	new ResourceSet("com.netscape.admin.dirserv.panel.replication.replication");
    private static final String _section = "replication-legacyconsumer";

    private static final int _pwdMinLength = 8;

    private static final String REPSETTINGS_CN = "legacy consumer";
    public static final String REPSETTINGS_DN = "cn=" + REPSETTINGS_CN +
	",cn=replication,cn=config";
    private static final String[] REPSETTINGS_OBJECTCLASSES =
    {"top", "extensibleObject"};
    private static final String UPDATEDN_ATTR_NAME =
	"nsslapd-legacy-updatedn";
    private static final String UPDATEDN_PASSWORD_ATTR_NAME =
	"nsslapd-legacy-updatepw";
}

