if [ "$(basename "$0")" != 'libplayit2.sh' ] && [ -z "$LIB_ONLY" ]; then

	# Exit immediately on error
	set -o errexit

	# Exit if the current process has been spawned by the root user
	## This check can be skipped by setting the following environment variable:
	## PLAYIT_OPTION_RUN_AS_ROOT=1
	if
		[ "${PLAYIT_OPTION_RUN_AS_ROOT:-0}" -eq 0 ] &&
		check_is_running_as_root
	then
		error_run_as_root
		exit 1
	fi

	# Error out (and exit) when trying to expand an unset variable
	## Only for game scripts targeting ./play.it ≥ 2.23
	if compatibility_level_is_at_least '2.23'; then
		set -o nounset
	fi

	# Set input field separator to default value (space, tab, newline)
	unset IFS

	# Force umask to ensure all paths are created with correct permissions
	umask 0022

	# Check early if ./play.it has been called in games listing mode,
	# since most of the initialization steps are not required in this mode.

	option_list_supported_games=$(option_value 'list-supported-games')
	if [ "${option_list_supported_games:-0}" -eq 1 ]; then
		## Set the default package, used by the context system when no explicit package context is set.
		## It is required by the context system, but is usually set later, when the current archive has been set already.
		packages_list=$(packages_list)
		default_package=$(printf '%s' "$packages_list" | head --lines=1)
		set_default_package "$default_package"
		unset packages_list default_package

		games_list_supported
		exit 0
	fi
	unset option_list_supported_games

	# Unset variables that we do not want to import from the user environment

	unset SOURCE_ARCHIVE_PATH
	unset PLAYIT_WORKDIR

	# Set URLs for error messages

	PLAYIT_GAMES_BUG_TRACKER_URL='https://forge.dotslashplay.it/play.it/games/issues'
	PLAYIT_BUG_TRACKER_URL='https://forge.dotslashplay.it/play.it/scripts/issues'

	# Check the current library version against the expected compatibility level

	COMPATIBILITY_LEVEL=$(compatibility_level)
	COMPATIBILITY_LEVEL_MAJOR=$(printf '%s' "$COMPATIBILITY_LEVEL" | cut --delimiter='.' --fields=1)
	LIBRARY_VERSION_MAJOR=$(printf '%s' "$LIBRARY_VERSION" | cut --delimiter='.' --fields=1)
	if [ "$LIBRARY_VERSION_MAJOR" -lt "$COMPATIBILITY_LEVEL_MAJOR" ]; then
		error_incompatible_versions
		exit 1
	elif [ "$LIBRARY_VERSION_MAJOR" -eq "$COMPATIBILITY_LEVEL_MAJOR" ]; then
		COMPATIBILITY_LEVEL_MINOR=$(printf '%s' "$COMPATIBILITY_LEVEL" | cut --delimiter='.' --fields=2)
		LIBRARY_VERSION_MINOR=$(printf '%s' "$LIBRARY_VERSION" | cut --delimiter='.' --fields=2)
		if [ "$LIBRARY_VERSION_MINOR" -lt "$COMPATIBILITY_LEVEL_MINOR" ]; then
			error_incompatible_versions
			exit 1
		fi
	fi
	unset \
		COMPATIBILITY_LEVEL \
		COMPATIBILITY_LEVEL_MAJOR COMPATIBILITY_LEVEL_MINOR \
		LIBRARY_VERSION_MAJOR LIBRARY_VERSION_MINOR

	# Set options

	## Set hardcoded defaults
	options_init_default

	## Load defaults from the configuration file
	config_file_path=$(find_configuration_file "$@")
	load_configuration_file "$config_file_path"

	## Set options from the command-line
	parse_arguments "$@"

	# Display the help message,
	# if called with --help.

	option_help=$(option_value 'help')
	if [ "$option_help" -eq 1 ]; then
		help
		exit 0
	fi
	unset option_help

	# Display the version string,
	# if called with --version.

	option_version=$(option_value 'version')
	if [ "$option_version" -eq 1 ]; then
		printf '%s\n' "$LIBRARY_VERSION"
		exit 0
	fi
	unset option_version

	# If called with --list-supported-games,
	# print the list of games supported by the current game script,
	# then exit early.

	option_list_supported_games=$(option_value 'list-supported-games')
	if [ "$option_list_supported_games" -eq 1 ]; then
		games_list_supported
		exit 0
	fi
	unset option_list_supported_games

	# Check the validity of all options

	options_validity_check
	options_compatibility_check

	# Set the base archive that is going to be used.

	archive_initialize_base

	# Set the default package, used by the context system when no explicit package context is set

	packages_list=$(packages_list)
	default_package=$(printf '%s' "$packages_list" | head --lines=1)
	set_default_package "$default_package"
	unset packages_list default_package

	# Set the path to the temporary directory used by ./play.it.
	# This can not be set earlier, because GAME_ID (required to compute this path) might be unset before the call to archive_initialize_base.

	set_temp_directories

	# Display the path to the game script,
	# if called with --show-game-script.

	option_show_game_script=$(option_value 'show-game-script')
	if [ "$option_show_game_script" -eq 1 ]; then
		printf '%s\n' "$(realpath "$0")"
		## Delete temporary files
		rm --force --recursive "${PLAYIT_WORKDIR:-}"
		exit 0
	fi
	unset option_show_game_script

	# If called with --list-packages,
	# print the list of packages that would be generated from the given archive
	# then exit early.

	option_list_packages=$(option_value 'list-packages')
	if [ "$option_list_packages" -eq 1 ]; then
		archive=$(current_archive)
		packages_print_list "$archive"
		## Delete temporary files
		rm --force --recursive "${PLAYIT_WORKDIR:-}"
		exit 0
	fi
	unset option_list_packages

	# If called with --list-requirements,
	# print the list of commands required to run the current game script
	# then exit early.

	option_list_requirements=$(option_value 'list-requirements')
	if [ "$option_list_requirements" -eq 1 ]; then
		requirements_list
		## Delete temporary files
		rm --force --recursive "${PLAYIT_WORKDIR:-}"
		exit 0
	fi
	unset option_list_requirements

	# Check the list of packages that would be built,
	# exit early if all are already there.

	option_overwrite=$(option_value 'overwrite')
	if [ "$option_overwrite" -eq 0 ]; then
		option_output_dir=$(option_value 'output-dir')
		archive=$(current_archive)
		generated_packages_list=$(packages_print_list "$archive")
		missing_package=0
		while read -r generated_package; do
			if [ ! -e "${option_output_dir}/${generated_package}" ]; then
				missing_package=1
				break
			fi
		done <<- EOL
		$(printf '%s' "$generated_packages_list")
		EOL
		if [ "$missing_package" -eq 0 ]; then
			info_all_packages_already_built
			## Delete temporary files
			rm --force --recursive "${PLAYIT_WORKDIR:-}"
			exit 0
		fi
		unset option_output_dir generated_packages_list generated_package missing_package
	fi
	unset option_overwrite

	# Check the presence of required tools

	{
		check_deps
		check_deps_status=$?
	} || true
	if [ $check_deps_status -ne 0 ]; then
		## Delete temporary files
		rm --force --recursive "${PLAYIT_WORKDIR:-}"
		exit 1
	fi
	unset check_deps_status
	archive=$(current_archive)
	{
		archive_dependencies_check "$archive"
		archive_dependencies_check_status=$?
	} || true
	if [ $archive_dependencies_check_status -ne 0 ]; then
		## Delete temporary files
		rm --force --recursive "${PLAYIT_WORKDIR:-}"
		exit 1
	fi
	unset archive archive_dependencies_check_status

	# Check for the presence of extra required archives

	{
		archives_required_extra_presence_check
		archives_required_extra_presence_check_status=$?
	} || true
	if [ $archives_required_extra_presence_check_status -ne 0 ]; then
		## Delete temporary files
		rm --force --recursive "${PLAYIT_WORKDIR:-}"
		exit 1
	fi
	unset archives_required_extra_presence_check_status

	# Check for the presence of extra optional archives

	archives_optional_extra_presence_check

	# Check the archives integrity

	{
		archives_integrity_check
		archives_integrity_check_status=$?
	} || true
	if [ $archives_integrity_check_status -ne 0 ]; then
		## Delete temporary files
		rm --force --recursive "${PLAYIT_WORKDIR:-}"
		exit 1
	fi
	unset archives_integrity_check_status
fi

