#!/usr/bin/perl

#   Copyright (C) 2011 Mauro Carvalho Chehab <mchehab@redhat.com>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, version 2 of the License.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
# This small script parses register dumps generated by em28xx driver
# with debug options enabled, generating a source code with the results
# of the dump.
#
# It is a variant of parse_az6007.pl, meant to be used to decode az6007
# register read/writes, with DRX-K
#

use strict;
use Getopt::Long;

my $show_other_lines = 0;	# Useful on some cases
my $show_timestamp = 0;
my $show_all = 0;

my $argerr = "Invalid arguments.\nUse $0 [--show_all] [--show_other_lines] [--show_timestamp]\n";

GetOptions(
	'show_other_lines' => \$show_other_lines,
	'show_timestamp' => \$show_timestamp,
	'show_all' => \$show_all,
) or die $argerr;

if ($show_all) {
	$show_other_lines = 1;
	$show_timestamp = 1;
}

sub add_hex_mark($)
{
	my $data = shift;
	my $out ="{";

	return "NULL" if ($data eq "");
	return "NULL due to $data " if ($data =~ /[Ee][Rr][Rr][Oo][Rr]/);

	# Check if the string has the correct syntax. If not, just return it as-is
	for (my $i = 0; $i++; $i < length($data)) {
		if ((($i + 1) % 3) == 0) {
			return $data if (substr($data, $i, 1) ne ' ');
		} else {
			return $data if (!(substr($data, $i, 1) =~ /0-9A-Fa-f/));
		}
	}

	while ($data) {
		$out .= "0x" . substr($data, 0, 2) . ", ";
		$data = substr($data, 3);
	}
	$out =~ s/, $//;
	$out .= "}";

	return $out;
}

#	0 => "Set IR value",
#	1 => "Set Wakeup IRcode",
#define REQ_02_GET_IR_CODE              2
#define REQ_03_SET_GET_MCU_PIN          3
#define REQ_04_EN_DISABLE_MCU_INT	4
#define REQ_05_SET_GET_USBREG           5
#define REQ_06_SET_GET_USBREG_BIT	6
#define REQ_07_SET_GET_AVREG            7
#define REQ_08_SET_GET_AVREG_BIT        8
#define REQ_09_SET_GET_TUNER_FQ         9
#define REQ_10_SET_TUNER_SYSTEM         10
#define REQ_11_SET_EEPROM_ADDR          11
#define REQ_12_SET_GET_EEPROMBYTE	12
#define REQ_13_GET_EEPROM_SEQREAD	13
#define REQ_14_SET_GET_I2C_WR2_RDN	14
#define REQ_15_SET_GET_I2CBYTE          15

my %regs = (
	0x0500 => "TM6010_REQ05_R00_MAIN_CTRL",
	0x0501 => "TM6010_REQ05_R01_DEVADDR",
	0x0502 => "TM6010_REQ05_R02_TEST",
	0x0504 => "TM6010_REQ05_R04_SOFN0",
	0x0505 => "TM6010_REQ05_R05_SOFN1",
	0x0506 => "TM6010_REQ05_R06_SOFTM0",
	0x0507 => "TM6010_REQ05_R07_SOFTM1",
	0x0508 => "TM6010_REQ05_R08_PHY_TEST",
	0x0509 => "TM6010_REQ05_R09_VCTL",
	0x050a => "TM6010_REQ05_R0A_VSTA",
	0x050b => "TM6010_REQ05_R0B_CX_CFG",
	0x050c => "TM6010_REQ05_R0C_ENDP0_REG0",
	0x0510 => "TM6010_REQ05_R10_GMASK",
	0x0511 => "TM6010_REQ05_R11_IMASK0",
	0x0512 => "TM6010_REQ05_R12_IMASK1",
	0x0513 => "TM6010_REQ05_R13_IMASK2",
	0x0514 => "TM6010_REQ05_R14_IMASK3",
	0x0515 => "TM6010_REQ05_R15_IMASK4",
	0x0516 => "TM6010_REQ05_R16_IMASK5",
	0x0517 => "TM6010_REQ05_R17_IMASK6",
	0x0518 => "TM6010_REQ05_R18_IMASK7",
	0x0519 => "TM6010_REQ05_R19_ZEROP0",
	0x051a => "TM6010_REQ05_R1A_ZEROP1",
	0x051c => "TM6010_REQ05_R1C_FIFO_EMP0",
	0x051d => "TM6010_REQ05_R1D_FIFO_EMP1",
	0x0520 => "TM6010_REQ05_R20_IRQ_GROUP",
	0x0521 => "TM6010_REQ05_R21_IRQ_SOURCE0",
	0x0522 => "TM6010_REQ05_R22_IRQ_SOURCE1",
	0x0523 => "TM6010_REQ05_R23_IRQ_SOURCE2",
	0x0524 => "TM6010_REQ05_R24_IRQ_SOURCE3",
	0x0525 => "TM6010_REQ05_R25_IRQ_SOURCE4",
	0x0526 => "TM6010_REQ05_R26_IRQ_SOURCE5",
	0x0527 => "TM6010_REQ05_R27_IRQ_SOURCE6",
	0x0528 => "TM6010_REQ05_R28_IRQ_SOURCE7",
	0x0529 => "TM6010_REQ05_R29_SEQ_ERR0",
	0x052a => "TM6010_REQ05_R2A_SEQ_ERR1",
	0x052b => "TM6010_REQ05_R2B_SEQ_ABORT0",
	0x052c => "TM6010_REQ05_R2C_SEQ_ABORT1",
	0x052d => "TM6010_REQ05_R2D_TX_ZERO0",
	0x052e => "TM6010_REQ05_R2E_TX_ZERO1",
	0x052f => "TM6010_REQ05_R2F_IDLE_CNT",
	0x0530 => "TM6010_REQ05_R30_FNO_P1",
	0x0531 => "TM6010_REQ05_R31_FNO_P2",
	0x0532 => "TM6010_REQ05_R32_FNO_P3",
	0x0533 => "TM6010_REQ05_R33_FNO_P4",
	0x0534 => "TM6010_REQ05_R34_FNO_P5",
	0x0535 => "TM6010_REQ05_R35_FNO_P6",
	0x0536 => "TM6010_REQ05_R36_FNO_P7",
	0x0537 => "TM6010_REQ05_R37_FNO_P8",
	0x0538 => "TM6010_REQ05_R38_FNO_P9",
	0x0539 => "TM6010_REQ05_R30_FNO_P10",
	0x053a => "TM6010_REQ05_R30_FNO_P11",
	0x053b => "TM6010_REQ05_R30_FNO_P12",
	0x053c => "TM6010_REQ05_R30_FNO_P13",
	0x053d => "TM6010_REQ05_R30_FNO_P14",
	0x053e => "TM6010_REQ05_R30_FNO_P15",
	0x0540 => "TM6010_REQ05_R40_IN_MAXPS_LOW1",
	0x0541 => "TM6010_REQ05_R41_IN_MAXPS_HIGH1",
	0x0542 => "TM6010_REQ05_R42_IN_MAXPS_LOW2",
	0x0543 => "TM6010_REQ05_R43_IN_MAXPS_HIGH2",
	0x0544 => "TM6010_REQ05_R44_IN_MAXPS_LOW3",
	0x0545 => "TM6010_REQ05_R45_IN_MAXPS_HIGH3",
	0x0546 => "TM6010_REQ05_R46_IN_MAXPS_LOW4",
	0x0547 => "TM6010_REQ05_R47_IN_MAXPS_HIGH4",
	0x0548 => "TM6010_REQ05_R48_IN_MAXPS_LOW5",
	0x0549 => "TM6010_REQ05_R49_IN_MAXPS_HIGH5",
	0x054a => "TM6010_REQ05_R4A_IN_MAXPS_LOW6",
	0x054b => "TM6010_REQ05_R4B_IN_MAXPS_HIGH6",
	0x054c => "TM6010_REQ05_R4C_IN_MAXPS_LOW7",
	0x054d => "TM6010_REQ05_R4D_IN_MAXPS_HIGH7",
	0x054e => "TM6010_REQ05_R4E_IN_MAXPS_LOW8",
	0x054f => "TM6010_REQ05_R4F_IN_MAXPS_HIGH8",
	0x0550 => "TM6010_REQ05_R50_IN_MAXPS_LOW9",
	0x0551 => "TM6010_REQ05_R51_IN_MAXPS_HIGH9",
	0x0552 => "TM6010_REQ05_R40_IN_MAXPS_LOW10",
	0x0553 => "TM6010_REQ05_R41_IN_MAXPS_HIGH10",
	0x0554 => "TM6010_REQ05_R40_IN_MAXPS_LOW11",
	0x0555 => "TM6010_REQ05_R41_IN_MAXPS_HIGH11",
	0x0556 => "TM6010_REQ05_R40_IN_MAXPS_LOW12",
	0x0557 => "TM6010_REQ05_R41_IN_MAXPS_HIGH12",
	0x0558 => "TM6010_REQ05_R40_IN_MAXPS_LOW13",
	0x0559 => "TM6010_REQ05_R41_IN_MAXPS_HIGH13",
	0x055a => "TM6010_REQ05_R40_IN_MAXPS_LOW14",
	0x055b => "TM6010_REQ05_R41_IN_MAXPS_HIGH14",
	0x055c => "TM6010_REQ05_R40_IN_MAXPS_LOW15",
	0x055d => "TM6010_REQ05_R41_IN_MAXPS_HIGH15",
	0x0560 => "TM6010_REQ05_R60_OUT_MAXPS_LOW1",
	0x0561 => "TM6010_REQ05_R61_OUT_MAXPS_HIGH1",
	0x0562 => "TM6010_REQ05_R62_OUT_MAXPS_LOW2",
	0x0563 => "TM6010_REQ05_R63_OUT_MAXPS_HIGH2",
	0x0564 => "TM6010_REQ05_R64_OUT_MAXPS_LOW3",
	0x0565 => "TM6010_REQ05_R65_OUT_MAXPS_HIGH3",
	0x0566 => "TM6010_REQ05_R66_OUT_MAXPS_LOW4",
	0x0567 => "TM6010_REQ05_R67_OUT_MAXPS_HIGH4",
	0x0568 => "TM6010_REQ05_R68_OUT_MAXPS_LOW5",
	0x0569 => "TM6010_REQ05_R69_OUT_MAXPS_HIGH5",
	0x056a => "TM6010_REQ05_R6A_OUT_MAXPS_LOW6",
	0x056b => "TM6010_REQ05_R6B_OUT_MAXPS_HIGH6",
	0x056c => "TM6010_REQ05_R6C_OUT_MAXPS_LOW7",
	0x056d => "TM6010_REQ05_R6D_OUT_MAXPS_HIGH7",
	0x056e => "TM6010_REQ05_R6E_OUT_MAXPS_LOW8",
	0x056f => "TM6010_REQ05_R6F_OUT_MAXPS_HIGH8",
	0x0570 => "TM6010_REQ05_R70_OUT_MAXPS_LOW9",
	0x0571 => "TM6010_REQ05_R71_OUT_MAXPS_HIGH9",
	0x0572 => "TM6010_REQ05_R60_OUT_MAXPS_LOW10",
	0x0573 => "TM6010_REQ05_R61_OUT_MAXPS_HIGH10",
	0x0574 => "TM6010_REQ05_R60_OUT_MAXPS_LOW11",
	0x0575 => "TM6010_REQ05_R61_OUT_MAXPS_HIGH11",
	0x0576 => "TM6010_REQ05_R60_OUT_MAXPS_LOW12",
	0x0577 => "TM6010_REQ05_R61_OUT_MAXPS_HIGH12",
	0x0578 => "TM6010_REQ05_R60_OUT_MAXPS_LOW13",
	0x0579 => "TM6010_REQ05_R61_OUT_MAXPS_HIGH13",
	0x057a => "TM6010_REQ05_R60_OUT_MAXPS_LOW14",
	0x057b => "TM6010_REQ05_R61_OUT_MAXPS_HIGH14",
	0x057c => "TM6010_REQ05_R60_OUT_MAXPS_LOW15",
	0x057d => "TM6010_REQ05_R61_OUT_MAXPS_HIGH15",
	0x0580 => "TM6010_REQ05_R80_FIFO0",
	0x0581 => "TM6010_REQ05_R81_FIFO1",
	0x0582 => "TM6010_REQ05_R82_FIFO2",
	0x0583 => "TM6010_REQ05_R83_FIFO3",
	0x0584 => "TM6010_REQ05_R84_FIFO4",
	0x0585 => "TM6010_REQ05_R85_FIFO5",
	0x0586 => "TM6010_REQ05_R86_FIFO6",
	0x0587 => "TM6010_REQ05_R87_FIFO7",
	0x0588 => "TM6010_REQ05_R88_FIFO8",
	0x0589 => "TM6010_REQ05_R89_FIFO9",
	0x058a => "TM6010_REQ05_R81_FIFO10",
	0x058b => "TM6010_REQ05_R81_FIFO11",
	0x058c => "TM6010_REQ05_R81_FIFO12",
	0x058d => "TM6010_REQ05_R81_FIFO13",
	0x058e => "TM6010_REQ05_R81_FIFO14",
	0x058f => "TM6010_REQ05_R81_FIFO15",
	0x0590 => "TM6010_REQ05_R90_CFG_FIFO0",
	0x0591 => "TM6010_REQ05_R91_CFG_FIFO1",
	0x0592 => "TM6010_REQ05_R92_CFG_FIFO2",
	0x0593 => "TM6010_REQ05_R93_CFG_FIFO3",
	0x0594 => "TM6010_REQ05_R94_CFG_FIFO4",
	0x0595 => "TM6010_REQ05_R95_CFG_FIFO5",
	0x0596 => "TM6010_REQ05_R96_CFG_FIFO6",
	0x0597 => "TM6010_REQ05_R97_CFG_FIFO7",
	0x0598 => "TM6010_REQ05_R98_CFG_FIFO8",
	0x0599 => "TM6010_REQ05_R99_CFG_FIFO9",
	0x059a => "TM6010_REQ05_R91_CFG_FIFO10",
	0x059b => "TM6010_REQ05_R91_CFG_FIFO11",
	0x059c => "TM6010_REQ05_R91_CFG_FIFO12",
	0x059d => "TM6010_REQ05_R91_CFG_FIFO13",
	0x059e => "TM6010_REQ05_R91_CFG_FIFO14",
	0x059f => "TM6010_REQ05_R91_CFG_FIFO15",
	0x05a0 => "TM6010_REQ05_RA0_CTL_FIFO0",
	0x05a1 => "TM6010_REQ05_RA1_CTL_FIFO1",
	0x05a2 => "TM6010_REQ05_RA2_CTL_FIFO2",
	0x05a3 => "TM6010_REQ05_RA3_CTL_FIFO3",
	0x05a4 => "TM6010_REQ05_RA4_CTL_FIFO4",
	0x05a5 => "TM6010_REQ05_RA5_CTL_FIFO5",
	0x05a6 => "TM6010_REQ05_RA6_CTL_FIFO6",
	0x05a7 => "TM6010_REQ05_RA7_CTL_FIFO7",
	0x05a8 => "TM6010_REQ05_RA8_CTL_FIFO8",
	0x05a9 => "TM6010_REQ05_RA9_CTL_FIFO9",
	0x05aa => "TM6010_REQ05_RA1_CTL_FIFO10",
	0x05ab => "TM6010_REQ05_RA1_CTL_FIFO11",
	0x05ac => "TM6010_REQ05_RA1_CTL_FIFO12",
	0x05ad => "TM6010_REQ05_RA1_CTL_FIFO13",
	0x05ae => "TM6010_REQ05_RA1_CTL_FIFO14",
	0x05af => "TM6010_REQ05_RA1_CTL_FIFO15",
	0x05b0 => "TM6010_REQ05_RB0_BC_LOW_FIFO0",
	0x05b1 => "TM6010_REQ05_RB1_BC_LOW_FIFO1",
	0x05b2 => "TM6010_REQ05_RB2_BC_LOW_FIFO2",
	0x05b3 => "TM6010_REQ05_RB3_BC_LOW_FIFO3",
	0x05b4 => "TM6010_REQ05_RB4_BC_LOW_FIFO4",
	0x05b5 => "TM6010_REQ05_RB5_BC_LOW_FIFO5",
	0x05b6 => "TM6010_REQ05_RB6_BC_LOW_FIFO6",
	0x05b7 => "TM6010_REQ05_RB7_BC_LOW_FIFO7",
	0x05b8 => "TM6010_REQ05_RB8_BC_LOW_FIFO8",
	0x05b9 => "TM6010_REQ05_RB9_BC_LOW_FIFO9",
	0x05ba => "TM6010_REQ05_RB1_BC_LOW_FIFO10",
	0x05bb => "TM6010_REQ05_RB1_BC_LOW_FIFO11",
	0x05bc => "TM6010_REQ05_RB1_BC_LOW_FIFO12",
	0x05bd => "TM6010_REQ05_RB1_BC_LOW_FIFO13",
	0x05be => "TM6010_REQ05_RB1_BC_LOW_FIFO14",
	0x05bf => "TM6010_REQ05_RB1_BC_LOW_FIFO15",
	0x05c0 => "TM6010_REQ05_RC0_DATA_FIFO0",
	0x05c4 => "TM6010_REQ05_RC4_DATA_FIFO1",
	0x05c8 => "TM6010_REQ05_RC8_DATA_FIFO2",
	0x05cc => "TM6010_REQ05_RCC_DATA_FIFO3",
	0x05d0 => "TM6010_REQ05_RD0_DATA_FIFO4",
	0x05d4 => "TM6010_REQ05_RD4_DATA_FIFO5",
	0x05d8 => "TM6010_REQ05_RD8_DATA_FIFO6",
	0x05dc => "TM6010_REQ05_RDC_DATA_FIFO7",
	0x05e0 => "TM6010_REQ05_RE0_DATA_FIFO8",
	0x05e4 => "TM6010_REQ05_RE4_DATA_FIFO9",
	0x05e8 => "TM6010_REQ05_RC4_DATA_FIFO10",
	0x05ec => "TM6010_REQ05_RC4_DATA_FIFO11",
	0x05f0 => "TM6010_REQ05_RC4_DATA_FIFO12",
	0x05f4 => "TM6010_REQ05_RC4_DATA_FIFO13",
	0x05f8 => "TM6010_REQ05_RC4_DATA_FIFO14",
	0x05fc => "TM6010_REQ05_RC4_DATA_FIFO15",
	0x0700 => "TM6010_REQ07_R00_VIDEO_CONTROL0",
	0x0701 => "TM6010_REQ07_R01_VIDEO_CONTROL1",
	0x0702 => "TM6010_REQ07_R02_VIDEO_CONTROL2",
	0x0703 => "TM6010_REQ07_R03_YC_SEP_CONTROL",
	0x0704 => "TM6010_REQ07_R04_LUMA_HAGC_CONTROL",
	0x0705 => "TM6010_REQ07_R05_NOISE_THRESHOLD",
	0x0706 => "TM6010_REQ07_R06_AGC_GATE_THRESHOLD",
	0x0707 => "TM6010_REQ07_R07_OUTPUT_CONTROL",
	0x0708 => "TM6010_REQ07_R08_LUMA_CONTRAST_ADJ",
	0x0709 => "TM6010_REQ07_R09_LUMA_BRIGHTNESS_ADJ",
	0x070a => "TM6010_REQ07_R0A_CHROMA_SATURATION_ADJ",
	0x070b => "TM6010_REQ07_R0B_CHROMA_HUE_PHASE_ADJ",
	0x070c => "TM6010_REQ07_R0C_CHROMA_AGC_CONTROL",
	0x070d => "TM6010_REQ07_R0D_CHROMA_KILL_LEVEL",
	0x070f => "TM6010_REQ07_R0F_CHROMA_AUTO_POSITION",
	0x0710 => "TM6010_REQ07_R10_AGC_PEAK_NOMINAL",
	0x0711 => "TM6010_REQ07_R11_AGC_PEAK_CONTROL",
	0x0712 => "TM6010_REQ07_R12_AGC_GATE_STARTH",
	0x0713 => "TM6010_REQ07_R13_AGC_GATE_STARTL",
	0x0714 => "TM6010_REQ07_R14_AGC_GATE_WIDTH",
	0x0715 => "TM6010_REQ07_R15_AGC_BP_DELAY",
	0x0716 => "TM6010_REQ07_R16_LOCK_COUNT",
	0x0717 => "TM6010_REQ07_R17_HLOOP_MAXSTATE",
	0x0718 => "TM6010_REQ07_R18_CHROMA_DTO_INCREMENT3",
	0x0719 => "TM6010_REQ07_R19_CHROMA_DTO_INCREMENT2",
	0x071a => "TM6010_REQ07_R1A_CHROMA_DTO_INCREMENT1",
	0x071b => "TM6010_REQ07_R1B_CHROMA_DTO_INCREMENT0",
	0x071c => "TM6010_REQ07_R1C_HSYNC_DTO_INCREMENT3",
	0x071d => "TM6010_REQ07_R1D_HSYNC_DTO_INCREMENT2",
	0x071e => "TM6010_REQ07_R1E_HSYNC_DTO_INCREMENT1",
	0x071f => "TM6010_REQ07_R1F_HSYNC_DTO_INCREMENT0",
	0x0720 => "TM6010_REQ07_R20_HSYNC_RISING_EDGE_TIME",
	0x0721 => "TM6010_REQ07_R21_HSYNC_PHASE_OFFSET",
	0x0722 => "TM6010_REQ07_R22_HSYNC_PLL_START_TIME",
	0x0723 => "TM6010_REQ07_R23_HSYNC_PLL_END_TIME",
	0x0724 => "TM6010_REQ07_R24_HSYNC_TIP_START_TIME",
	0x0725 => "TM6010_REQ07_R25_HSYNC_TIP_END_TIME",
	0x0726 => "TM6010_REQ07_R26_HSYNC_RISING_EDGE_START",
	0x0727 => "TM6010_REQ07_R27_HSYNC_RISING_EDGE_END",
	0x0728 => "TM6010_REQ07_R28_BACKPORCH_START",
	0x0729 => "TM6010_REQ07_R29_BACKPORCH_END",
	0x072a => "TM6010_REQ07_R2A_HSYNC_FILTER_START",
	0x072b => "TM6010_REQ07_R2B_HSYNC_FILTER_END",
	0x072c => "TM6010_REQ07_R2C_CHROMA_BURST_START",
	0x072d => "TM6010_REQ07_R2D_CHROMA_BURST_END",
	0x072e => "TM6010_REQ07_R2E_ACTIVE_VIDEO_HSTART",
	0x072f => "TM6010_REQ07_R2F_ACTIVE_VIDEO_HWIDTH",
	0x0730 => "TM6010_REQ07_R30_ACTIVE_VIDEO_VSTART",
	0x0731 => "TM6010_REQ07_R31_ACTIVE_VIDEO_VHIGHT",
	0x0732 => "TM6010_REQ07_R32_VSYNC_HLOCK_MIN",
	0x0733 => "TM6010_REQ07_R33_VSYNC_HLOCK_MAX",
	0x0734 => "TM6010_REQ07_R34_VSYNC_AGC_MIN",
	0x0735 => "TM6010_REQ07_R35_VSYNC_AGC_MAX",
	0x0736 => "TM6010_REQ07_R36_VSYNC_VBI_MIN",
	0x0737 => "TM6010_REQ07_R37_VSYNC_VBI_MAX",
	0x0738 => "TM6010_REQ07_R38_VSYNC_THRESHOLD",
	0x0739 => "TM6010_REQ07_R39_VSYNC_TIME_CONSTANT",
	0x073a => "TM6010_REQ07_R3A_STATUS1",
	0x073b => "TM6010_REQ07_R3B_STATUS2",
	0x073c => "TM6010_REQ07_R3C_STATUS3",
	0x073f => "TM6010_REQ07_R3F_RESET",
	0x0740 => "TM6010_REQ07_R40_TELETEXT_VBI_CODE0",
	0x0741 => "TM6010_REQ07_R41_TELETEXT_VBI_CODE1",
	0x0742 => "TM6010_REQ07_R42_VBI_DATA_HIGH_LEVEL",
	0x0743 => "TM6010_REQ07_R43_VBI_DATA_TYPE_LINE7",
	0x0744 => "TM6010_REQ07_R44_VBI_DATA_TYPE_LINE8",
	0x0745 => "TM6010_REQ07_R45_VBI_DATA_TYPE_LINE9",
	0x0746 => "TM6010_REQ07_R46_VBI_DATA_TYPE_LINE10",
	0x0747 => "TM6010_REQ07_R47_VBI_DATA_TYPE_LINE11",
	0x0748 => "TM6010_REQ07_R48_VBI_DATA_TYPE_LINE12",
	0x0749 => "TM6010_REQ07_R49_VBI_DATA_TYPE_LINE13",
	0x074a => "TM6010_REQ07_R4A_VBI_DATA_TYPE_LINE14",
	0x074b => "TM6010_REQ07_R4B_VBI_DATA_TYPE_LINE15",
	0x074c => "TM6010_REQ07_R4C_VBI_DATA_TYPE_LINE16",
	0x074d => "TM6010_REQ07_R4D_VBI_DATA_TYPE_LINE17",
	0x074e => "TM6010_REQ07_R4E_VBI_DATA_TYPE_LINE18",
	0x074f => "TM6010_REQ07_R4F_VBI_DATA_TYPE_LINE19",
	0x0750 => "TM6010_REQ07_R50_VBI_DATA_TYPE_LINE20",
	0x0751 => "TM6010_REQ07_R51_VBI_DATA_TYPE_LINE21",
	0x0752 => "TM6010_REQ07_R52_VBI_DATA_TYPE_LINE22",
	0x0753 => "TM6010_REQ07_R53_VBI_DATA_TYPE_LINE23",
	0x0754 => "TM6010_REQ07_R54_VBI_DATA_TYPE_RLINES",
	0x0755 => "TM6010_REQ07_R55_VBI_LOOP_FILTER_GAIN",
	0x0756 => "TM6010_REQ07_R56_VBI_LOOP_FILTER_I_GAIN",
	0x0757 => "TM6010_REQ07_R57_VBI_LOOP_FILTER_P_GAIN",
	0x0758 => "TM6010_REQ07_R58_VBI_CAPTION_DTO1",
	0x0759 => "TM6010_REQ07_R59_VBI_CAPTION_DTO0",
	0x075a => "TM6010_REQ07_R5A_VBI_TELETEXT_DTO1",
	0x075b => "TM6010_REQ07_R5B_VBI_TELETEXT_DTO0",
	0x075c => "TM6010_REQ07_R5C_VBI_WSS625_DTO1",
	0x075d => "TM6010_REQ07_R5D_VBI_WSS625_DTO0",
	0x075e => "TM6010_REQ07_R5E_VBI_CAPTION_FRAME_START",
	0x075f => "TM6010_REQ07_R5F_VBI_WSS625_FRAME_START",
	0x0760 => "TM6010_REQ07_R60_TELETEXT_FRAME_START",
	0x0761 => "TM6010_REQ07_R61_VBI_CCDATA1",
	0x0762 => "TM6010_REQ07_R62_VBI_CCDATA2",
	0x0763 => "TM6010_REQ07_R63_VBI_WSS625_DATA1",
	0x0764 => "TM6010_REQ07_R64_VBI_WSS625_DATA2",
	0x0765 => "TM6010_REQ07_R65_VBI_DATA_STATUS",
	0x0766 => "TM6010_REQ07_R66_VBI_CAPTION_START",
	0x0767 => "TM6010_REQ07_R67_VBI_WSS625_START",
	0x0768 => "TM6010_REQ07_R68_VBI_TELETEXT_START",
	0x0770 => "TM6010_REQ07_R70_HSYNC_DTO_INC_STATUS3",
	0x0771 => "TM6010_REQ07_R71_HSYNC_DTO_INC_STATUS2",
	0x0772 => "TM6010_REQ07_R72_HSYNC_DTO_INC_STATUS1",
	0x0773 => "TM6010_REQ07_R73_HSYNC_DTO_INC_STATUS0",
	0x0774 => "TM6010_REQ07_R74_CHROMA_DTO_INC_STATUS3",
	0x0775 => "TM6010_REQ07_R75_CHROMA_DTO_INC_STATUS2",
	0x0776 => "TM6010_REQ07_R76_CHROMA_DTO_INC_STATUS1",
	0x0777 => "TM6010_REQ07_R77_CHROMA_DTO_INC_STATUS0",
	0x0778 => "TM6010_REQ07_R78_AGC_AGAIN_STATUS",
	0x0779 => "TM6010_REQ07_R79_AGC_DGAIN_STATUS",
	0x077a => "TM6010_REQ07_R7A_CHROMA_MAG_STATUS",
	0x077b => "TM6010_REQ07_R7B_CHROMA_GAIN_STATUS1",
	0x077c => "TM6010_REQ07_R7C_CHROMA_GAIN_STATUS0",
	0x077d => "TM6010_REQ07_R7D_CORDIC_FREQ_STATUS",
	0x077f => "TM6010_REQ07_R7F_STATUS_NOISE",
	0x0780 => "TM6010_REQ07_R80_COMB_FILTER_TRESHOLD",
	0x0782 => "TM6010_REQ07_R82_COMB_FILTER_CONFIG",
	0x0783 => "TM6010_REQ07_R83_CHROMA_LOCK_CONFIG",
	0x0784 => "TM6010_REQ07_R84_NOISE_NTSC_C",
	0x0785 => "TM6010_REQ07_R85_NOISE_PAL_C",
	0x0786 => "TM6010_REQ07_R86_NOISE_PHASE_C",
	0x0787 => "TM6010_REQ07_R87_NOISE_PHASE_Y",
	0x078a => "TM6010_REQ07_R8A_CHROMA_LOOPFILTER_STATE",
	0x078b => "TM6010_REQ07_R8B_CHROMA_HRESAMPLER",
	0x078d => "TM6010_REQ07_R8D_CPUMP_DELAY_ADJ",
	0x078e => "TM6010_REQ07_R8E_CPUMP_ADJ",
	0x078f => "TM6010_REQ07_R8F_CPUMP_DELAY",
	0x07c0 => "TM6010_REQ07_RC0_ACTIVE_VIDEO_SOURCE",
	0x07c1 => "TM6010_REQ07_RC1_TRESHOLD",
	0x07c2 => "TM6010_REQ07_RC2_HSYNC_WIDTH",
	0x07c3 => "TM6010_REQ07_RC3_HSTART1",
	0x07c4 => "TM6010_REQ07_RC4_HSTART0",
	0x07c5 => "TM6010_REQ07_RC5_HEND1",
	0x07c6 => "TM6010_REQ07_RC6_HEND0",
	0x07c7 => "TM6010_REQ07_RC7_VSTART1",
	0x07c8 => "TM6010_REQ07_RC8_VSTART0",
	0x07c9 => "TM6010_REQ07_RC9_VEND1",
	0x07ca => "TM6010_REQ07_RCA_VEND0",
	0x07cb => "TM6010_REQ07_RCB_DELAY",
	0x07cc => "TM6010_REQ07_RCC_ACTIVE_IF",
	0x07d0 => "TM6010_REQ07_RD0_USB_PERIPHERY_CONTROL",
	0x07d1 => "TM6010_REQ07_RD1_ADDR_FOR_REQ1",
	0x07d2 => "TM6010_REQ07_RD2_ADDR_FOR_REQ2",
	0x07d3 => "TM6010_REQ07_RD3_ADDR_FOR_REQ3",
	0x07d4 => "TM6010_REQ07_RD4_ADDR_FOR_REQ4",
	0x07d5 => "TM6010_REQ07_RD5_POWERSAVE",
	0x07d6 => "TM6010_REQ07_RD6_ENDP_REQ1_REQ2",
	0x07d7 => "TM6010_REQ07_RD7_ENDP_REQ3_REQ4",
	0x07d8 => "TM6000_REQ07_RD8_TEST_SEL",
	0x07d8 => "TM6010_REQ07_RD8_IR",
	0x07d9 => "TM6000_REQ07_RD9_A_SIM_SEL",
	0x07d9 => "TM6010_REQ07_RD9_IR_BSIZE",
	0x07da => "TM6000_REQ07_RDA_CLK_SEL",
	0x07da => "TM6010_REQ07_RDA_IR_WAKEUP_SEL",
	0x07db => "TM6000_REQ07_RDB_OUT_SEL",
	0x07db => "TM6010_REQ07_RDB_IR_WAKEUP_ADD",
	0x07dc => "TM6000_REQ07_RDC_NSEL_I2S",
	0x07dc => "TM6010_REQ07_RDC_IR_LEADER1",
	0x07dd => "TM6000_REQ07_RDD_GPIO2_MDRV",
	0x07dd => "TM6010_REQ07_RDD_IR_LEADER0",
	0x07de => "TM6000_REQ07_RDE_GPIO1_MDRV",
	0x07de => "TM6010_REQ07_RDE_IR_PULSE_CNT1",
	0x07df => "TM6000_REQ07_RDF_PWDOWN_ACLK",
	0x07df => "TM6010_REQ07_RDF_IR_PULSE_CNT0",
	0x07e0 => "TM6000_REQ07_RE0_VADC_REF_CTL",
	0x07e0 => "TM6010_REQ07_RE0_DVIDEO_SOURCE",
	0x07e1 => "TM6000_REQ07_RE1_VADC_DACLIMP",
	0x07e1 => "TM6010_REQ07_RE0_DVIDEO_SOURCE_IF",
	0x07e2 => "TM6000_REQ07_RE2_VADC_STATUS_CTL",
	0x07e2 => "TM6010_REQ07_RE2_OUT_SEL2",
	0x07e3 => "TM6000_REQ07_RE3_VADC_INP_LPF_SEL1",
	0x07e3 => "TM6010_REQ07_RE3_OUT_SEL1",
	0x07e4 => "TM6000_REQ07_RE4_VADC_TARGET1",
	0x07e4 => "TM6010_REQ07_RE4_OUT_SEL0",
	0x07e5 => "TM6000_REQ07_RE5_VADC_INP_LPF_SEL2",
	0x07e5 => "TM6010_REQ07_RE5_REMOTE_WAKEUP",
	0x07e6 => "TM6000_REQ07_RE6_VADC_TARGET2",
	0x07e7 => "TM6000_REQ07_RE7_VADC_AGAIN_CTL",
	0x07e7 => "TM6010_REQ07_RE7_PUB_GPIO",
	0x07e8 => "TM6000_REQ07_RE8_VADC_PWDOWN_CTL",
	0x07e8 => "TM6010_REQ07_RE8_TYPESEL_MOS_I2S",
	0x07e9 => "TM6000_REQ07_RE9_VADC_INPUT_CTL1",
	0x07e9 => "TM6010_REQ07_RE9_TYPESEL_MOS_TS",
	0x07ea => "TM6000_REQ07_REA_VADC_INPUT_CTL2",
	0x07ea => "TM6010_REQ07_REA_TYPESEL_MOS_CCIR",
	0x07eb => "TM6000_REQ07_REB_VADC_AADC_MODE",
	0x07ec => "TM6000_REQ07_REC_VADC_AADC_LVOL",
	0x07ed => "TM6000_REQ07_RED_VADC_AADC_RVOL",
	0x07ee => "TM6000_REQ07_REE_VADC_CTRL_SEL_CONTROL",
	0x07ef => "TM6000_REQ07_REF_VADC_GAIN_MAP_CTL",
	0x07f0 => "TM6010_REQ07_RF0_BIST_CRC_RESULT0",
	0x07f1 => "TM6010_REQ07_RF1_BIST_CRC_RESULT1",
	0x07f2 => "TM6010_REQ07_RF2_BIST_CRC_RESULT2",
	0x07f3 => "TM6010_REQ07_RF3_BIST_CRC_RESULT3",
	0x07f4 => "TM6010_REQ07_RF4_BIST_ERR_VST2",
	0x07f5 => "TM6010_REQ07_RF5_BIST_ERR_VST1",
	0x07f6 => "TM6010_REQ07_RF6_BIST_ERR_VST0",
	0x07f7 => "TM6010_REQ07_RF7_BIST",
	0x07fd => "TM6000_REQ07_RFD_BIST_ERR_VST_LOW",
	0x07fe => "TM6000_REQ07_RFE_BIST_ERR_VST_HIGH",
	0x07fe => "TM6010_REQ07_RFE_POWER_DOWN",
	0x07ff => "TM6010_REQ07_RFF_SOFT_RESET",
	0x0800 => "TM6010_REQ08_R00_A_VERSION",
	0x0801 => "TM6010_REQ08_R01_A_INIT",
	0x0802 => "TM6010_REQ08_R02_A_FIX_GAIN_CTRL",
	0x0803 => "TM6010_REQ08_R03_A_AUTO_GAIN_CTRL",
	0x0804 => "TM6010_REQ08_R04_A_SIF_AMP_CTRL",
	0x0805 => "TM6010_REQ08_R05_A_STANDARD_MOD",
	0x0806 => "TM6010_REQ08_R06_A_SOUND_MOD",
	0x0807 => "TM6010_REQ08_R07_A_LEFT_VOL",
	0x0808 => "TM6010_REQ08_R08_A_RIGHT_VOL",
	0x0809 => "TM6010_REQ08_R09_A_MAIN_VOL",
	0x080a => "TM6010_REQ08_R0A_A_I2S_MOD",
	0x080b => "TM6010_REQ08_R0B_A_ASD_THRES1",
	0x080c => "TM6010_REQ08_R0C_A_ASD_THRES2",
	0x080d => "TM6010_REQ08_R0D_A_AMD_THRES",
	0x080e => "TM6010_REQ08_R0E_A_MONO_THRES1",
	0x080f => "TM6010_REQ08_R0F_A_MONO_THRES2",
	0x0810 => "TM6010_REQ08_R10_A_MUTE_THRES1",
	0x0811 => "TM6010_REQ08_R11_A_MUTE_THRES2",
	0x0812 => "TM6010_REQ08_R12_A_AGC_U",
	0x0813 => "TM6010_REQ08_R13_A_AGC_ERR_T",
	0x0814 => "TM6010_REQ08_R14_A_AGC_GAIN_INIT",
	0x0815 => "TM6010_REQ08_R15_A_AGC_STEP_THR",
	0x0816 => "TM6010_REQ08_R16_A_AGC_GAIN_MAX",
	0x0817 => "TM6010_REQ08_R17_A_AGC_GAIN_MIN",
	0x0818 => "TM6010_REQ08_R18_A_TR_CTRL",
	0x0819 => "TM6010_REQ08_R19_A_FH_2FH_GAIN",
	0x081a => "TM6010_REQ08_R1A_A_NICAM_SER_MAX",
	0x081b => "TM6010_REQ08_R1B_A_NICAM_SER_MIN",
	0x081e => "TM6010_REQ08_R1E_A_GAIN_DEEMPH_OUT",
	0x081f => "TM6010_REQ08_R1F_A_TEST_INTF_SEL",
	0x0820 => "TM6010_REQ08_R20_A_TEST_PIN_SEL",
	0x0821 => "TM6010_REQ08_R21_A_AGC_ERR",
	0x0822 => "TM6010_REQ08_R22_A_AGC_GAIN",
	0x0823 => "TM6010_REQ08_R23_A_NICAM_INFO",
	0x0824 => "TM6010_REQ08_R24_A_SER",
	0x0825 => "TM6010_REQ08_R25_A_C1_AMP",
	0x0826 => "TM6010_REQ08_R26_A_C2_AMP",
	0x0827 => "TM6010_REQ08_R27_A_NOISE_AMP",
	0x0828 => "TM6010_REQ08_R28_A_AUDIO_MODE_RES",
	0x08e0 => "TM6010_REQ08_RE0_ADC_REF",
	0x08e1 => "TM6010_REQ08_RE1_DAC_CLMP",
	0x08e2 => "TM6010_REQ08_RE2_POWER_DOWN_CTRL1",
	0x08e3 => "TM6010_REQ08_RE3_ADC_IN1_SEL",
	0x08e4 => "TM6010_REQ08_RE4_ADC_IN2_SEL",
	0x08e5 => "TM6010_REQ08_RE5_GAIN_PARAM",
	0x08e6 => "TM6010_REQ08_RE6_POWER_DOWN_CTRL2",
	0x08e7 => "TM6010_REQ08_RE7_REG_GAIN_Y",
	0x08e8 => "TM6010_REQ08_RE8_REG_GAIN_C",
	0x08e9 => "TM6010_REQ08_RE9_BIAS_CTRL",
	0x08ea => "TM6010_REQ08_REA_BUFF_DRV_CTRL",
	0x08eb => "TM6010_REQ08_REB_SIF_GAIN_CTRL",
	0x08ec => "TM6010_REQ08_REC_REVERSE_YC_CTRL",
	0x08ed => "TM6010_REQ08_RED_GAIN_SEL",
	0x08f0 => "TM6010_REQ08_RF0_DAUDIO_INPUT_CONFIG",
	0x08f1 => "TM6010_REQ08_RF1_AADC_POWER_DOWN",
	0x08f2 => "TM6010_REQ08_RF2_LEFT_CHANNEL_VOL",
	0x08f3 => "TM6010_REQ08_RF3_RIGHT_CHANNEL_VOL",
);

my %gpio = (
	0x102	=> "GPIO_0",
	0x103	=> "GPIO_1",
	0x104	=> "GPIO_2",
	0x105	=> "GPIO_3",
	0x106	=> "GPIO_4",
	0x107	=> "GPIO_5",
	0x300	=> "GPIO_6",
	0x301	=> "GPIO_7",
	0x305	=> "GPIO_9",
);

##############
# Main program
##############

while (<>) {
	my $org_line = $_;

	if (m/(.*)([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].) ([0-9a-fA-F].)\s*([\<\>]+)\s*(.*)/) {
		my $timestamp = $1;
		my $reqtype = hex($2);
		my $req = hex($3);
		my $wvalue = hex("$5$4");
		my $windex = hex("$7$6");
		my $wlen = hex("$9$8");
		my $dir = $10;
		my $payload = $11;

		if ($req == 0 && $reqtype < 0x80 && !$wvalue && !$windex) {
			printf("tm6000_set_ir_value(udev, %s, %s);\n",
			       add_hex_mark($payload), $wlen);
		} elsif ($req == 1 && $reqtype < 0x80 && !$wvalue && !$windex) {
			printf("tm6000_set_ir_wakeup_code(udev, %s, %s);\n",
			       add_hex_mark($payload), $wlen);
		} elsif ($req == 2 && $reqtype >= 0x80 && !$wvalue && !$windex) {
			printf("tm6000_get_ir(udev, &data, %s); /* %s */\n",
				$wlen, add_hex_mark($payload));
		} elsif ($req == 3) {
			my $cur_gpio;
			if (defined($gpio{$wvalue})) {
				$cur_gpio = sprintf "%s", $gpio{$wvalue};
			} else {
				$cur_gpio = sprintf "0x%02x,", $wvalue;
			}
			if ($reqtype >= 0x80) {
				printf("tm6000_read_gpio(udev, %s, &data, %s); /* %s */\n",
					$cur_gpio, $wlen, add_hex_mark($payload));
			} else {
				if ($wlen) {
					printf("tm6000_write_gpio(udev, %s, 0x%02x, %s);\n",
						$cur_gpio, $windex, add_hex_mark($payload), $wlen);
				} else {
					printf("tm6000_write_gpio(udev, %s, 0x%02x);\n",
						$cur_gpio, $windex);
				}
			}
		} elsif ($req == 4 && $reqtype < 0x80 && !$wlen) {
			printf("tm6000_write(udev, 0x04, EN_DISABLE_MCU_INT, 0x%02x, 0x%02x);\n",
				$wvalue, $windex);
		} elsif (($req == 5 || $req == 7 || $req == 8) && $wvalue < 256) {
			my $register = $req << 8 | $wvalue;
			if (defined($regs{$register})) {
				$register = sprintf "%s", $regs{$register};
			} else {
				$register = sprintf "0x%02x, 0x%02x,", $req, $wvalue;
			}
			if ($reqtype >= 0x80) {
				printf("tm6000_read(udev, %s, %02x, &data, %s); /* %s */\n",
					$register, $windex, $wlen, add_hex_mark($payload));
			} else {
				if ($wlen) {
					printf("tm6000_write(udev, %s, 0x%02x, %s, %s);\n",
						$register, $windex, add_hex_mark($payload), $wlen);
				} else {
					printf("tm6000_write(udev, %s, 0x%02x);\n",
						$register, $windex);
				}
			}
		} else {
			printf "$timestamp " if ($show_timestamp);

			if ($reqtype >= 0x80) {
				printf("tm6000_read(udev, 0x%02x, 0x%02x, %d, &data, %s); /* %s */\n",
					$req, $wvalue, $windex, $wlen, add_hex_mark($payload));
			} else {
				printf("tm6000_write(udev, 0x%02x, 0x%02x, 0x%02x, %s, %d);\n",
					$req, $wvalue, $windex, add_hex_mark($payload), $wlen);
			}
		}
		next;
	}
	print $org_line if ($show_other_lines);
}
