/**
 * @brief Module to control Display Brightness and Keyboard Illumination
 * 
 * This module contains the machine independent logic to control the
 * display brightness and the keyboard illumination. If an ambient light
 * sensor is available in the system, it is taken into account too.
 *
 * This module does not access the display or sensors directly. This is
 * done by the hardware abstraction layer of the current machine which
 * contains appropriate driver for the display and the sensors. The
 * display module sends all hardware requests to this hardware drivers.
 *
 * The display module has several properties that could be accessed from
 * the outside. The properties could be accessed through the function
 * display_handle_tags(). The properties are described there.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/module_display.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <strings.h>
#include <time.h>
#include <math.h>
#include "pbbinput.h"
#include <pbb.h>

#include "input_manager.h"
#include "module_display.h"
#include "support.h"
#include "gettext_macros.h"

#include "class_config.h"

#include <linux/radeonfb.h>

#include "debug.h"

#define VAL2PERCENT(x,max)  (x * 100 / max)
#define PERCENT2VAL(x,max)  (x * max / 100)

/**
 * @brief  Section name for the configuration file
 */
#define SECTION "MODULE DISPLAY"

/**
 * @brief Private module data structure
 */
struct moddata_display {
	char *fbdev;         /* pathname of the framebuffer device */
	struct display_flags flags;
	struct display_light lcdlight; /* structures for LCD light control */
	struct display_light kbdlight; /* structures for keyboard light control */

	int lcdfeedback;        /* feedback of the LCD on the ambient sensor */
	int kbdon_brightness;   /* keyboard brightness after switch on */
	
	unsigned short keylcdup;  /* keycodes and modifiers */
	unsigned short modlcdup;   /*     for LCD backlight controls */
	unsigned short keylcddn;
	unsigned short modlcddn;
	unsigned short keykbdup;    /* keycodes and modifiers */
	unsigned short modkbdup;    /*     for keyboard illumination controls */
	unsigned short keykbddn;
	unsigned short modkbddn;
	unsigned short keykbdon;
	unsigned short modkbdon;
	unsigned short keymirror;  /* keycode for CRT mirror key */
	unsigned short modmirror;
} modbase_display;

static const char *AutoadjModeList[] = { "off", "linear", "hysteresis", NULL };

/**
 * @brief  Initialize the display module
 *
 * This function does everything to get the display module up. It initializes
 * the private module data, read and store the module configuration, check
 * for hardware and path names and finally register some callbacks at the
 * pbbuttonsd framework.
 *
 * @return Zero if everything went well or the error code. Only really fatal
 *         errors throw an error code here because if one of the initialization
 *         functions fail, pbbuttonsd will quit.
 */
int
display_init ()
{
	struct moddata_display *base = &modbase_display;
	struct tagitem args[] = {{ TAG_BACKLIGHTMAX, 0 },
	                         { TAG_BACKLIGHTLEVEL, -1 },
	                         { TAG_KEYBLIGHTMAX, 0 },
	                         { TAG_AMBIENTLIGHT, -1 },
	                         { TAG_END, 0 }};
	int level, ambient;

	base->fbdev = config_get_string (SECTION, "Device_FB", DEFAULT_FB_DEVICE);

	/* get some information from hardware module */
	process_queue (QUERYQUEUE, args);

	/* if no backlight controller was registered TAG_BACKLIGHTLEVEL
	 * would return -1 which must be converted to be in brightness
	 * range. furtheron a flag saves this information.
	 */
	base->flags.nobacklight = 0;
	if ((level = (int) tagfind (args, TAG_BACKLIGHTLEVEL, -1)) == -1) {
		level = (int) tagfind (args, TAG_BACKLIGHTMAX, 0);
		base->flags.nobacklight = 1;
	}

	base->lcdlight.current     = level;
	base->lcdlight.target      = level;
	base->lcdlight.max         = (int) tagfind (args, TAG_BACKLIGHTMAX, 0);
	base->lcdlight.backup      = level;
	base->lcdlight.offset      = 0;
	base->lcdlight.isrunning   = 0;
	display_set_fading (&base->lcdlight, config_get_int (SECTION, "LCD_FadingSpeed", 0));
	base->lcdlight.autoadj_mode =
			config_get_option (SECTION, "LCD_AutoadjMode", AutoadjModeList, AUTOADJ_LIN);
	base->lcdlight.autoadj_hyst_out = 0;
	display_set_autoadj_parameter (&base->lcdlight, BATTERY,
			config_get_intlist (SECTION, "LCD_AutoadjParm_onBattery", 4, 0,   1, 94,  54));
	display_set_autoadj_parameter (&base->lcdlight, AC,
			config_get_intlist (SECTION, "LCD_AutoadjParm_onAC",      4, 0,   1, 94, 100));

	base->kbdlight.current     = 0;
	base->kbdlight.target      = 0;
	base->kbdlight.max         = (int) tagfind (args, TAG_KEYBLIGHTMAX, 0);
	base->kbdlight.backup      = 0;
	base->kbdlight.offset      = 0;
	base->kbdlight.isrunning   = 0;
	display_set_fading (&base->kbdlight, config_get_int (SECTION, "KBD_FadingSpeed", 0));
	base->kbdlight.autoadj_mode =
			config_get_option (SECTION, "KBD_AutoadjMode", AutoadjModeList, AUTOADJ_HYS);
	base->kbdlight.autoadj_hyst_out = 0;
	display_set_autoadj_parameter (&base->kbdlight, BATTERY,
			config_get_intlist (SECTION, "KBD_AutoadjParm_onBattery", 4, 10, 100, 28,   0));
	display_set_autoadj_parameter (&base->kbdlight, AC,
			config_get_intlist (SECTION, "KBD_AutoadjParm_onAC",      4, 10, 100, 28,   0));

	process_queue_single (CONFIGQUEUE, TAG_KEYBLIGHTLEVEL, base->kbdlight.current);

	base->flags.status 	   = STATUS_NORMAL;
	base->flags.coveropen  = 1;  /* cover open by default same as in laptop module */

	/* If we have an Ambient light sensor in this Laptop, we must measure the LCD
	 * feedback to this sensor in current state. To do this we switch the backlight
	 * off for a moment and measure the real ambient light level.
	 */
	ambient                = tagfind(args, TAG_AMBIENTLIGHT, -1);
	base->flags.lmu_enable = ambient == -1 ? 0 : 1;
	base->lcdfeedback      = 0;

	if (base->flags.lmu_enable) {
		print_msg (PBB_INFO, _("Ambient light sensor found.\n"));
		process_queue_single (CONFIGQUEUE, TAG_BACKLIGHTLEVEL, 0);
		base->lcdfeedback = ambient - process_queue_single (QUERYQUEUE, TAG_AMBIENTLIGHT, 0);
		process_queue_single (CONFIGQUEUE, TAG_BACKLIGHTLEVEL, base->lcdlight.current);
#if defined(DEBUG) && AMBIENT
		printf("Init: Ambient = %d, LCD feedback = %d\n", ambient, base->lcdfeedback);
#endif
	}

	/* change the current brightness levels for LCD backlight and keyboard
	 * illumination if there are some set in the configuration file.
	 */
	if ((level = config_get_percent (SECTION, "LCD_Brightness", -1)) != -1)
		display_set_lcdbrightness (PERCENT2VAL(level, base->lcdlight.max));
	if ((level = config_get_percent (SECTION, "KBD_Brightness", -1)) != -1)
		display_set_kbdbrightness (PERCENT2VAL(level, base->kbdlight.max));

	base->keylcdup	= config_get_keycode (SECTION, "LCD_IllumUpKey", KEY_BRIGHTNESSUP);
	base->modlcdup	= config_get_modmask (SECTION, "LCD_IllumUpKey", MOD_NONE);
	base->keylcddn	= config_get_keycode (SECTION, "LCD_IllumDownKey", KEY_BRIGHTNESSDOWN);
	base->modlcddn	= config_get_modmask (SECTION, "LCD_IllumDownKey", MOD_NONE);
	base->keykbdup	= config_get_keycode (SECTION, "KBD_IllumUpKey", KEY_KBDILLUMUP);
	base->modkbdup	= config_get_modmask (SECTION, "KBD_IllumUpKey", MOD_NONE);
	base->keykbddn	= config_get_keycode (SECTION, "KBD_IllumDownKey", KEY_KBDILLUMDOWN);
	base->modkbddn	= config_get_modmask (SECTION, "KBD_IllumDownKey", MOD_NONE);
	base->keykbdon	= config_get_keycode (SECTION, "KBD_IllumOnKey", KEY_KBDILLUMTOGGLE);
	base->modkbdon	= config_get_modmask (SECTION, "KBD_IllumOnKey", MOD_NONE);
	base->keymirror = config_get_keycode (SECTION, "CRT_MirrorKey", KEY_F7);
	base->modmirror = config_get_modmask (SECTION, "CRT_MirrorKey", MOD_CTRL);

	/* Saved brightness for kbd on/off key
	 * The KBD_OnBrightness value is configured as percentage and must
	 * be transformed to a real backlight level the display module work
	 * with. This is done here and also in the function handle_tags()
	 */
	level = config_get_percent (SECTION, "KBD_OnBrightness", 0);
	base->kbdon_brightness = PERCENT2VAL(level, base->kbdlight.max);
	if (base->kbdon_brightness < KBD_BRIGHTNESS_MIN)
		base->kbdon_brightness = KBD_BRIGHTNESS_MIN;
	
	/* framebuffer disabled by default  */
	base->flags.ctrl_fb = config_get_boolean (SECTION, "UseFBBlank", FALSE); 

	/* don't dim to complete darkness by default */
	base->flags.dimfullydark = config_get_boolean (SECTION, "DimFullyDark", FALSE); 

	/* This value must be set to 1 if the initial value of kbd_onbrightness should
	 * take effect but this will block KBD autoadjustment mode. In KBD autoadjustement
	 * mode the initial value makes no sense so we set the flag dependent of the
	 * users autoadjustment settings.
	 * In case somebody set the keyboard brightness from extern the kbd_switchedoff
	 * flag will always be 0 except the current brightness is zero (dark). This flag
	 * status is set by display_handle_tags(), TAG_KBDBRIGHTNESS.
	 */
	if (base->kbdlight.current == 0)
		base->flags.kbd_switchedoff = base->kbdlight.autoadj_mode == AUTOADJ_OFF ? 1 : 0; 

	register_function (KBDQUEUE, display_keyboard);
	register_function (QUERYQUEUE, display_query);
	register_function (CONFIGQUEUE, display_configure);
	register_function (T1000QUEUE, display_ambienttimer);
	return 0;
}

/**
 * @brief Frees all ressources allocated by this module
 */
void
display_exit ()
{
	struct moddata_display *base = &modbase_display;

	g_free (base->fbdev);
}

/**
 * @brief  Handles keyboard events
 *
 * This function is called whenever keyboard events need
 * processing. It looks for certain keys and if they are
 * found predefined actions will take place.
 *
 * Special keys catched by this function are:
 *   @li LCD brightness up/down
 *   @li Keyboard brightness up/down
 *   @li Keyboard illumination on/off
 *   @li CRT Mirror on/off (switch the external video output)
 *
 * The appropritate keys change their meaning in autoadjust mode.
 * Instead of modifying the brightness directly the keys now modify
 * an offset that will shift the characteristic curve for automatic
 * brightness adjustment up or down a bit. Although automatic
 * adjustment ist enable the user keep control over the brightness.
 *
 * Because the brightness range could be very big, the change step 
 * for each key trigger is calculated automatically in that way
 * that always 16 key strokes are needed to switch from lowest to
 * highest brightness level.
 *
 * Functions using the shift modifier is disabled during automatic
 * control.
 *
 * @param  taglist   taglist containing the key information
 */
void
display_keyboard (struct tagitem *taglist)
{
	struct moddata_display *base = &modbase_display;
	int code, value, mod, lcdstep, kbdstep, tmp;
	int level = 0, changed = 0;

	code = (int) tagfind (taglist, TAG_KEYCODE, 0);
	value = (int) tagfind (taglist, TAG_KEYREPEAT, 0);
	mod = (int) tagfind (taglist, TAG_MODIFIER, 0);

	if (value && base->flags.coveropen) {
		lcdstep = (base->lcdlight.max >> 4) + 1;
		kbdstep = (base->kbdlight.max >> 4) + 1;

		/* key for video mirroring */
		if ((code == base->keymirror) && (mod == base->modmirror))
			if ((tmp = display_switchmirror ()) != -1)
				singletag_to_clients(CHANGEVALUE, TAG_CRTMIRROR, tmp);
		
		/* keys for LCD brightness control */
		if ((code == base->keylcdup) && ((mod & ~MOD_SHIFT) == base->modlcdup)) {
			/* somebody might have changed the brightness level directly so
			 * check if we are still synchron here
			 */
			display_sync_lcdbrightness ();
			
			/* .target and .current will normally be the same value synchronized
			 * by the display_timer() function. We use .target here because the
			 * powersave module might have given an order to change brightness.
			 * In case fading was activated this order would not be fullfilled
			 * yet (only .target has been set accordingly) and if we used
			 * .current here we would overwrite it.
			 */
			level = mod & MOD_SHIFT ? base->lcdlight.max : base->lcdlight.target + lcdstep;
			changed = 1;
		} else if ((code == base->keylcddn) && ((mod & ~MOD_SHIFT) == base->modlcddn)) {
			display_sync_lcdbrightness ();  /* same as brightness up */
			level = mod & MOD_SHIFT ? LCD_BRIGHTNESS_MIN : base->lcdlight.target - lcdstep;
			changed = 1;
		}
		
		/* keys for KBD illumination control */
		if ((code == base->keykbdup) && ((mod & ~MOD_SHIFT) == base->modkbdup)) {
			if (base->flags.kbd_switchedoff) {
				base->kbdlight.current = base->kbdon_brightness;
				base->flags.kbd_switchedoff = 0;
			}
			level = mod & MOD_SHIFT ? base->kbdlight.max : base->kbdlight.current + kbdstep;
			changed = 2;
		} else if ((code == base->keykbddn) && ((mod & ~MOD_SHIFT) == base->modkbddn)) {
			if (base->flags.kbd_switchedoff) {
				base->kbdlight.current = base->kbdon_brightness;
				base->flags.kbd_switchedoff = 0;
			}
			level = mod & MOD_SHIFT ? KBD_BRIGHTNESS_MIN : base->kbdlight.current - kbdstep;
			changed = 2;
		} else if (base->flags.lmu_enable && (code == base->keykbdon) && (mod == base->modkbdon)) {
			if (base->flags.kbd_switchedoff) {
				level = base->kbdon_brightness;
				base->flags.kbd_switchedoff = 0;
			} else {
				base->kbdon_brightness = base->kbdlight.current;
				base->flags.kbd_switchedoff = 1;
				level = KBD_BRIGHTNESS_OFF;
			}
			level = display_set_kbdbrightness(level);
			singletag_to_clients(CHANGEVALUE, TAG_KBDBRIGHTNESS,
					             VAL2PERCENT(level, base->kbdlight.max));
		}

		switch (changed) {
		case 1:  /* LCD brightness level changed by keys */
			if (base->flags.lmu_enable && base->lcdlight.autoadj_mode != AUTOADJ_OFF) {
				if (mod & MOD_SHIFT)
					break;   /* shift modifier disabled in automatic mode */
				tmp = base->lcdlight.current - base->lcdlight.offset;
				base->lcdlight.offset += (level - base->lcdlight.current) > 0 ? lcdstep : -lcdstep;
				if (tmp + base->lcdlight.offset > base->lcdlight.max)
					base->lcdlight.offset = base->lcdlight.max - tmp;
				if (tmp + base->lcdlight.offset < 1)
					base->lcdlight.offset = -tmp;
				level = tmp + base->lcdlight.offset;
			}
#if defined (DEBUG) && AMBIENT
			printf ("               Offset: %3d, level: %3d\n", base->lcdlight.offset, level);
#endif
			level = display_set_lcdbrightness(level);
			singletag_to_clients(CHANGEVALUE, TAG_LCDBRIGHTNESS,
					             VAL2PERCENT(level, base->lcdlight.max));
			break;
		case 2:  /* Keyboard illumination level changed by keys */
			if (base->flags.lmu_enable) {
				if (base->kbdlight.autoadj_mode != AUTOADJ_OFF) {
					if (mod & MOD_SHIFT)
						break;   /* shift modifier disabled in automatic mode */
					tmp = base->kbdlight.current - base->kbdlight.offset;
					base->kbdlight.offset += (level - base->kbdlight.current) > 0 ? kbdstep : -kbdstep;
					if (tmp + base->kbdlight.offset > base->kbdlight.max)
						base->kbdlight.offset = base->kbdlight.max - tmp;
					if (tmp + base->kbdlight.offset < 0)
						base->kbdlight.offset = -tmp;
					level = tmp + base->kbdlight.offset;
				}
				level = display_set_kbdbrightness(level);
				singletag_to_clients(CHANGEVALUE, TAG_KBDBRIGHTNESS,
						             VAL2PERCENT(level, base->kbdlight.max));
			}
			break;
		}
	}
}

/**
 * @brief  LCD brightness timer function
 *
 * The following two timer functions will set the brightness
 * level for the LCD and keyboard backlight. The timer will
 * be dynamically configured to be able to switch the brightness
 * hard or smooth from one level to another. 
 *
 * When no further change requests are pending the timer disables
 * itself. A new brightness change request will launch another
 * timer.
 *
 * To track the influence of the LCD on the ambient light sensors,
 * the raw sensor values, before and after the new backlight level
 * has been set, are read. if one of the ambient values is invalid,
 * the lcdfeedback won't change. In this case lcdfeedback become
 * slightly out of sync but will be synchronized again the next time
 * the display will be switched off.
 *
 * @see display_kbdtimer()
 *
 * @param  data  standard callback parameter not used in this function
 * @return TRUE, if the timer should be called again or FALSE if the
 *         job has been finished.
 */
gboolean
display_lcdtimer (gpointer data)	
{
	struct moddata_display *base = &modbase_display;
	struct display_light *illu = &base->lcdlight;
	int preambient, postambient;

	if (illu->current != illu->target) {
		if (illu->fading && abs(illu->current - illu->target) >= illu->fadingstep) {
			if (illu->current > illu->target)  /* target is darker as current */
				illu->current -= illu->fadingstep; 
			else                               /* target is brighter as current */
				illu->current += illu->fadingstep;
		} else
			illu->current = illu->target;

		preambient = process_queue_single (QUERYQUEUE, TAG_AMBIENTLIGHT, -1);

#if defined(DEBUG) && FADING
		printf ("LCD timer: Target=%3d, Current=%3d, RawAmbient (%4d) - LCD feedback (%4d) = Ambient (%d)\n",
				illu->target, illu->current, preambient, base->lcdfeedback, preambient - base->lcdfeedback);
#endif

		process_queue_single (CONFIGQUEUE, TAG_BACKLIGHTLEVEL, illu->current);
		display_framebuffer (illu->current ? 1: 0);
		
		postambient = process_queue_single (QUERYQUEUE, TAG_AMBIENTLIGHT, -1);

		if (preambient != -1 && postambient != -1) {
			if (illu->current) {
				base->lcdfeedback += postambient - preambient;
				if (base->lcdfeedback < 0)
					base->lcdfeedback = 0; /* lower than zero is not valid */
			} else
				base->lcdfeedback = 0;
		}

#if defined(DEBUG) && FADING
		printf ("                                    RawAmbient (%4d) - LCD feedback (%4d) = Ambient (%d)\n",
				postambient, base->lcdfeedback, postambient - base->lcdfeedback);
#endif
		return TRUE;
	}

#if defined(DEBUG) && DISPLAYTIMER
	struct timeval endtime;
	double time;

	gettimeofday (&endtime, NULL);
	time = (endtime.tv_sec  - illu->start.tv_sec)  * 1000 +
		   (endtime.tv_usec - illu->start.tv_usec) / 1000;

	print_msg (PBB_INFO, "DBG: LCD Display total time: %fms, time per step: %fms\n",
			        time, time / illu->steps); 
	print_msg (PBB_INFO, "DBG: LCD Display Timer stopped.\n");
#endif
	illu->isrunning = 0;
	return FALSE;
}

/**
 * @brief  Keyboard illumination brightness timer
 *
 * This timer function will be called whenever the brightness
 * of the keyboard illumination should be changed. It is
 * sililar to display_lcdtimer() which does the same for the
 * LCD backlight.
 *
 * @see display_lcdtimer()
 *
 * @param  data standard callback parameter not used in this function
 * @return TRUE, if the timer should be called again or FALSE if the
 *         job has been finished.
 */ 
gboolean
display_kbdtimer (gpointer data)	
{
	struct moddata_display *base = &modbase_display;
	struct display_light *illu = &base->kbdlight;

	if (base->flags.lmu_enable) {
		if (illu->current != illu->target) {
			if (illu->fading && abs(illu->current - illu->target) >= illu->fadingstep) {
				if (illu->current > illu->target)  /* target is darker as current */
					illu->current -= illu->fadingstep; 
				else                               /* target is brighter as current */
					illu->current += illu->fadingstep;
			} else
				illu->current = illu->target;

#if defined(DEBUG) && FADING
			printf ("KBD Timer: Target=%3d, Current=%3d\n", illu->target, illu->current);
#endif
			process_queue_single (CONFIGQUEUE, TAG_KEYBLIGHTLEVEL, illu->current);
			return TRUE;
		}

#if defined(DEBUG) && DISPLAYTIMER
		struct timeval endtime;
		double time;

		gettimeofday (&endtime, NULL);
		time = (endtime.tv_sec  - illu->start.tv_sec)  * 1000 +
			   (endtime.tv_usec - illu->start.tv_usec) / 1000;

		print_msg (PBB_INFO, "DBG: KBD Display total time: %fms, time per step: %fms\n",
				        time, time / illu->steps); 
		print_msg (PBB_INFO, "DBG: KBD Display Timer stopped.\n");
#endif
	}
	illu->isrunning = 0;
	return FALSE;
}


/**
 * @brief  Automatic brightness adjustment timer
 *
 * This timer function is called every second and it performs
 * the automatic brighness adjustments for LCD and keyboard
 * illumination.
 *
 * The LCD brightness should change in parallel to the ambient
 * light. That means the brighter the environment the brighter
 * the LCD backlight. This way the display stays readable in
 * bright environments.
 *
 * The equation for the automatic brightness adjustment is:
 *    brightness level = s * a * ambient level + offset
 *
 *  @li <b>s</b> is the scaling factor that depends on the
 *      ambient light sensor and the brightness range of the
 *      LCD backlight.
 *  @li <b>a</b> is an additional amplification. This factor
 *      modifies the gradient of the curve so that for example
 *      the maximum brightness level will be reached before the
 *      ambient light sensor is at its maximum or vica versa.
 *      This factor is usually could be configured with the
 *      "LCD_Threshold" option: v = 100 / LCD_Threshold
 *  @li <b>offset</b> is the brightness shift caused by manual 
 *      level adjustment with the brightness up/down keys.
 *
 * The automatic brightness adjustment algorithm can be limited to
 * a certain working range. That means the user can configure
 * the controller not to use the full brightness range but a special
 * part of it. He can set a lower and upper brightness limits the
 * controller must stay between. Then It won't calculate brightness
 * levels behind this limits dependend on ambient light.
 *
 * The keyboard illumination should be switched on and become
 * brighter when the environmental light becomes too dark to
 * see the letters on the keys. The keyboard illumination works
 * antiparallel to the ambient light level.
 *
 * The equation for the keyboard brightness adjustment looks like this:
 * brightness level = -s * a * ambient level + offset + max keylight
 *
 * The equations to calculate the new values allow values
 * generated out of range. This doesn't matter as long as
 * the function display_set_lcdbrightness() clip them to the
 * valid range.
 */
void
display_ambienttimer (struct tagitem *taglist)
{
	struct moddata_display *base = &modbase_display;
	struct display_light *illu;
	int ambient, ambientmax, level, pwrsource;

	struct tagitem args[] = {{ TAG_AMBIENTLIGHT, -1 },
	                         { TAG_AMBIENTLIGHTMAX, 0 },
	                         { TAG_POWERSOURCE, 1 },
	                         { TAG_END, 0 }};

	if (base->flags.lmu_enable && base->flags.status == STATUS_NORMAL) {
		/* process this code only if an ambient light sensor was found
		 * and the display is not dimmed or switched off */
		process_queue (QUERYQUEUE, args);

		if ((ambient = tagfind (args, TAG_AMBIENTLIGHT, -1)) > -1) {
			ambient   -= base->lcdfeedback;
			ambientmax = tagfind (args, TAG_AMBIENTLIGHTMAX, 0);
			pwrsource  = tagfind (args, TAG_POWERSOURCE, 1);
	
			illu = &base->kbdlight;
			if (illu->autoadj_mode != AUTOADJ_OFF && !base->flags.kbd_switchedoff) {
				level = display_calc_brightness (illu, pwrsource, ambient, ambientmax);
				if (level + illu->offset > illu->max)
					illu->offset = illu->max - level;
				if (level + illu->offset < 1)
					illu->offset = -level;
				display_set_kbdbrightness(level + illu->offset);
			}
			
			/* skip the autoadjustment if the current brightness level is zero.
			 * It's users will and we don't want to interfere with it */
			illu = &base->lcdlight;
			if (illu->autoadj_mode != AUTOADJ_OFF && illu->current != 0) {
				level = display_calc_brightness (illu, pwrsource, ambient, ambientmax);
				if (level + illu->offset > illu->max)
					illu->offset = illu->max - level;
				if (level + illu->offset < 1)
					illu->offset = -level + 1;
				display_set_lcdbrightness(level + illu->offset);
			}
		}
	}
}

/**
 * @brief Query properties from the display module
 *
 * This function is a stub for display_handle_tags().
 *
 * It will be called whenever other modules or one of the
 * clients needs information from this module. The properties
 * are given in form of a taglist and the function
 * display_configure() fill in the data.
 *
 * @see display_configure(), display_handle_tags()
 *
 * @param  taglist  taglist containing the properties to be queried
 */
void
display_query (struct tagitem *taglist)
{
	display_handle_tags (MODE_QUERY, taglist);
}

/**
 * @brief Change properties of the display module
 *
 * This function is a stub for display_handle_tags().
 *
 * It will be called whenever other modules or one of the
 * clients wants to change one or more properties of this
 * module. The properties are given in form of a taglist.
 * The function display_configure() take the data from it,
 * do some sanity checks and change the module property.
 * The change become instantly active.
 *
 * @see display_query(), display_handle_tags()
 *
 * @param  taglist  taglist containing the properties to be changed
 */
void
display_configure (struct tagitem *taglist)
{
	display_handle_tags (MODE_CONFIG, taglist);
}

/**
 * @brief  Read or write of module properties
 *
 * This function handles external access to module properties. If
 * another module or a client wants to read or change a propertiy,
 * this function will do it. There is no other way to access module
 * properties from outside this module.
 *
 * Two groups of properties are known:
 *   @li <b>private</b> - properties can be used only inside pbbuttonsd,
 *   @li <b>public</b> - properties can be used also by clients.
 *
 * <b>Public Properties (tags)</b>
 *
 * <table class="pbb">
 * <tr><th>Property</th><th>Data</th><th>D</th><th>Description</th></tr>
 * <tr><td>LCDBRIGHTNESS    </td><td>level</td><td>&lt;&gt;</td>
 *     <td>This command will get or set the display brightness level.
 *         Level is a value in percent (100% = full display brightness).
 *         If an error occoured tag data is set to -1.</td></tr>
 * <tr><td>LCDFADINGSPEED   </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>KBDBRIGHTNESS    </td><td>level</td><td>&lt;&gt;</td>
 *     <td>This command will get or set the brightness level of the
 *         keyboard illumination (if the computer has one). Level is a
 *         value in percent (100% = full keyboard brightness). If an error
 *         occoured tag data is set to -1.</td></tr>
 * <tr><td>KBDONBRIGHTNESS  </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>KBDFADINGSPEED   </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>LCDILLUMUPKEY    </td><td>key code</td><td>&lt;&gt;</td>
 *     <td>Key code to increase the display brightness.</td></tr>
 * <tr><td>LCDILLUMUPMOD    </td><td>modifier flags</td><td>&lt;&gt;</td>
 *     <td>Modifier for the LCD-brightness-up key. The modifier shift
 *         is used internally and can't be used as key modifier.</td></tr>
 * <tr><td>LCDILLUMDOWNKEY  </td><td>key code</td><td>&lt;&gt;</td>
 *     <td>Key code to decrease the display brightness.</td></tr>
 * <tr><td>LCDILLUMDOWNMOD  </td><td>modifier flags</td><td>&lt;&gt;</td>
 *     <td>Modifier for the LCD-brightness-down key. The modifier shift
 *         is used internally and can't be used as key modifier.</td></tr>
 * <tr><td>KBDILLUMUPKEY    </td><td>key code</td><td>&lt;&gt;</td>
 *     <td>Key code to increase the keyboard illumination brightness.</td></tr>
 * <tr><td>KBDILLUMUPMOD    </td><td>modifier flags</td><td>&lt;&gt;</td>
 *     <td>Modifier for the keyboard-brightness-down key. The modifier shift
 *         is used internally and can't be used as key modifier.</td></tr>
 * <tr><td>KBDILLUMDOWNKEY  </td><td>key code</td><td>&lt;&gt;</td>
 *     <td>Key code to decrease the keyboard illumination brightness.</td></tr>
 * <tr><td>KBDILLUMDOWNMOD  </td><td>modifier flags</td><td>&lt;&gt;</td>
 *     <td>Modifier for the keyboard-brightness-down key. The modifier shift
 *         is used internally and can't be used as key modifier.</td></tr>
 * <tr><td>KBDILLUMONKEY    </td><td>key code</td><td>&lt;&gt;</td>
 *     <td>Key code to toggle the keyboard illumination on or off.</td></tr>
 * <tr><td>KBDILLUMONMOD    </td><td>modifier flags</td><td>&lt;&gt;</td>
 *     <td>Modifier for the keyboard-brightness-on/off key.</td></tr>
 * <tr><td>CRTMIRRORKEY     </td><td>key code</td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>CRTMIRRORMOD     </td><td>modifier flags</td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>FRAMEBUFFERDEVICE</td><td>device name</td><td>&lt;&gt;</td>
 *     <td>This property set the frame buffer device that should be used to
 *         switch the display on or off.</td></tr>
 * <tr><td>BLANKFRAMEBUFFER </td><td>bool</td><td>&lt;&gt;</td>
 *     <td>If this property is set to 1, the display will be additionally
 *         switched on or off by the frame buffer device. Without this only
 *         the backlight will be switched off. The display is then still
 *         working although nothing could be seen.</td></tr>
 * <tr><td>DIMFULLYDARK     </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>LCDAUTOADJUST    </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>KBDAUTOADJUST    </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>LCDAUTOADJBAT    </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>LCDAUTOADJAC     </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>KBDAUTOADJBAT    </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * <tr><td>KBDAUTOADJAC     </td><td></td><td>&lt;&gt;</td>
 *     <td></td></tr>
 * </table>
 *
 * <b>Private Properties (tags)</b>
 * 
 * <table class="pbb">
 * <tr><th>Property</th><th>Data</th><th>D</th><th>Description</th></tr>
 * <tr><td>BRIGHTNESSOP   </td><td>opcode</td><td>&gt;</td>
 *     <td>This command will change the display and keyboard brightness
 *         relative to its current value. The opcode defines the action and
 *         is processed by display_change_brightness(). If an error occoured
 *         tag data is set to -1.</td></tr>
 * <tr><td>COVERSTATUS    </td><td>status</td><td>&gt;</td>
 *     <td></td></tr>
 * <tr><td>PREPAREFORSLEEP</td><td>none  </td><td>&gt;</td>
 *     <td></td></tr>
 * <tr><td>WAKEUPFROMSLEEP</td><td>none  </td><td>&gt;</td>
 *     <td></td></tr>
 * </table>
 *
 * @see display_query(), display_configure()
 *
 * @param  cfgure  MODE_CONFIG or MODE_QUERY. Sets the operating
 *                 mode of display_handle_tags()
 * @param  taglist taglist containing the properties and data
 */
void
display_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_display *base = &modbase_display;
	int err, level;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_BRIGHTNESSOP:  /* private tag */
			if (cfgure)	display_change_brightness (taglist->data);
			break;
		case TAG_LCDBRIGHTNESS:
			if (cfgure) {
				level = taglist->data > 100 ? 100 : taglist->data;  /* level in percent */
				level = display_set_lcdbrightness (PERCENT2VAL(level, base->lcdlight.max));
				singletag_to_clients(CHANGEVALUE, TAG_LCDBRIGHTNESS,
						             VAL2PERCENT(level, base->lcdlight.max));
			} else
				taglist->data = base->flags.nobacklight ? -1 :
					VAL2PERCENT(base->lcdlight.current, base->lcdlight.max);
			break;
		case TAG_LCDFADINGSPEED:
			if (cfgure) {
				display_set_fading (&base->lcdlight, taglist->data);
				config_set_int (SECTION, "LCD_FadingSpeed", taglist->data);
			} else
				taglist->data = display_get_fading (&base->lcdlight);
			break;
		case TAG_KBDBRIGHTNESS:
			if (cfgure) {
				level = taglist->data > 100 ? 100 : taglist->data;  /* level in percent */
				if (level > 100) level = 100;
				if (level < 0)   level = 0;
				level = display_set_kbdbrightness (PERCENT2VAL(level, base->kbdlight.max));
				if (level > 0)   base->flags.kbd_switchedoff = 0;
				singletag_to_clients(CHANGEVALUE, TAG_KBDBRIGHTNESS,
						             VAL2PERCENT(level, base->kbdlight.max));
			} else
				taglist->data = VAL2PERCENT(base->kbdlight.current, base->kbdlight.max);
			break;
		case TAG_KBDONBRIGHTNESS:
			if (cfgure) {
				level = taglist->data > 100 ? 100 : taglist->data;  /* level in percent */
				config_set_percent (SECTION, "KBD_OnBrightness", level);
				base->kbdon_brightness = PERCENT2VAL(level, base->kbdlight.max);
				if (base->kbdon_brightness < KBD_BRIGHTNESS_MIN)
					base->kbdon_brightness = KBD_BRIGHTNESS_MIN;
			} else
				taglist->data = VAL2PERCENT(base->kbdon_brightness, base->kbdlight.max);
			break;
		case TAG_KBDFADINGSPEED:
			if (cfgure)	{
				display_set_fading (&base->kbdlight, taglist->data);
				config_set_int (SECTION, "KBD_FadingSpeed", taglist->data);
			} else
				taglist->data = display_get_fading (&base->kbdlight);
			break;
		case TAG_LCDILLUMUPKEY:
			if (cfgure)	{
				base->keylcdup = taglist->data;
				config_set_keymod (SECTION, "LCD_IllumUpKey", taglist->data, base->modlcdup);
			} else
				taglist->data = base->keylcdup;
			break;
		case TAG_LCDILLUMUPMOD:
			if (cfgure)	{
				base->modlcdup = taglist->data;
				config_set_keymod (SECTION, "LCD_IllumUpKey", base->keylcdup, taglist->data);
			} else
				taglist->data = base->modlcdup;
			break;
		case TAG_LCDILLUMDOWNKEY:
			if (cfgure)	{
				base->keylcddn = taglist->data;
				config_set_keymod (SECTION, "LCD_IllumDownKey", taglist->data, base->modlcddn);
			} else
				taglist->data = base->keylcddn;
			break;
		case TAG_LCDILLUMDOWNMOD:
			if (cfgure)	{
				base->modlcddn = taglist->data;
				config_set_keymod (SECTION, "LCD_IllumDownKey", base->keylcddn, taglist->data);
			} else
				taglist->data = base->modlcddn;
			break;
		case TAG_KBDILLUMUPKEY:
			if (cfgure)	{
				base->keykbdup = taglist->data;
				config_set_keymod (SECTION, "KBD_IllumUpKey", taglist->data, base->modkbdup);
			} else
				taglist->data = base->keykbdup;
			break;
		case TAG_KBDILLUMUPMOD:
			if (cfgure)	{
				base->modkbdup = taglist->data;
				config_set_keymod (SECTION, "KBD_IllumUpKey", base->keykbdup, taglist->data);
			} else
				taglist->data = base->modkbdup;
			break;
		case TAG_KBDILLUMDOWNKEY:
			if (cfgure)	{
				base->keykbddn = taglist->data;
				config_set_keymod (SECTION, "KBD_IllumDownKey", taglist->data, base->modkbddn);
			} else
				taglist->data = base->keykbddn;
			break;
		case TAG_KBDILLUMDOWNMOD:
			if (cfgure)	{
				base->modkbddn = taglist->data;
				config_set_keymod (SECTION, "KBD_IllumDownKey", base->keykbddn, taglist->data);
			} else
				taglist->data = base->modkbddn;
			break;
		case TAG_KBDILLUMONKEY:
			if (cfgure)	{
				base->keykbdon = taglist->data;
				config_set_keymod (SECTION, "KBD_IllumOnKey", taglist->data, base->modkbdon);
			} else
				taglist->data = base->keykbdon;
			break;
		case TAG_KBDILLUMONMOD:
			if (cfgure)	{
				base->modkbdon = taglist->data;
				config_set_keymod (SECTION, "KBD_IllumOnKey", base->keykbdon, taglist->data);
			} else
				taglist->data = base->modkbdon;
			break;
		case TAG_CRTMIRRORKEY:
			if (cfgure)	{
				base->keymirror = taglist->data;
				config_set_keymod (SECTION, "CRT_MirrorKey", taglist->data, base->modmirror);
			} else
				taglist->data = base->keymirror;
			break;
		case TAG_CRTMIRRORMOD:
			if (cfgure)	{
				base->modmirror = taglist->data;
				config_set_keymod (SECTION, "CRT_MirrorKey", base->keymirror, taglist->data);
			} else
				taglist->data = base->modmirror;
			break;
		case TAG_FRAMEBUFFERDEVICE:
			if (cfgure) {
				if ((err = check_path ((char *) taglist->data, TYPE_CHARDEV, CPFLG_NONE)))
					tagerror (taglist, err);
				else {
					replace_string (&base->fbdev, (char *) taglist->data);
					config_set_string (SECTION, "Device_FB", base->fbdev);
				}
			} else
				taglist->data = (long) base->fbdev;
			break;
		case TAG_BLANKFRAMEBUFFER:
			if (cfgure)	{
				base->flags.ctrl_fb = taglist->data & 1;
				config_set_boolean (SECTION, "UseFBBlank", base->flags.ctrl_fb); 
			} else
				taglist->data = base->flags.ctrl_fb;
			break;
		case TAG_DIMFULLYDARK:
			if (cfgure)	{
				base->flags.dimfullydark = taglist->data & 1;
				config_set_boolean (SECTION, "DimFullyDark", base->flags.dimfullydark); 
			} else
				taglist->data = base->flags.dimfullydark;
			break;
		case TAG_LCDAUTOADJUST:
			if (cfgure) {
				base->lcdlight.offset = 0;
				base->lcdlight.autoadj_mode = taglist->data > AUTOADJ_LAST ? 0 : taglist->data;
				config_set_option (SECTION, "LCD_AutoadjMode", AutoadjModeList, base->lcdlight.autoadj_mode);
			} else
				taglist->data = base->lcdlight.autoadj_mode;
			break;
		case TAG_KBDAUTOADJUST:
			if (cfgure) {
				base->kbdlight.offset = 0;
				base->kbdlight.autoadj_mode = taglist->data > AUTOADJ_LAST ? 0 : taglist->data;
				config_set_option (SECTION, "KBD_AutoadjMode", AutoadjModeList, base->kbdlight.autoadj_mode);
			} else
				taglist->data = base->kbdlight.autoadj_mode;
			break;
		case TAG_LCDAUTOADJBAT:
			if (cfgure)	display_set_autoadj_from_tag (&base->lcdlight, BATTERY, taglist->data);
			else		taglist->data = display_get_autoadj_for_tag (&base->lcdlight, BATTERY);
			break;
		case TAG_LCDAUTOADJAC:
			if (cfgure)	display_set_autoadj_from_tag (&base->lcdlight, AC, taglist->data);
			else		taglist->data = display_get_autoadj_for_tag (&base->lcdlight, AC);
			break;
		case TAG_KBDAUTOADJBAT:
			if (cfgure)	display_set_autoadj_from_tag (&base->kbdlight, BATTERY, taglist->data);
			else		taglist->data = display_get_autoadj_for_tag (&base->kbdlight, BATTERY);
			break;
		case TAG_KBDAUTOADJAC:
			if (cfgure)	display_set_autoadj_from_tag (&base->kbdlight, AC, taglist->data);
			else		taglist->data = display_get_autoadj_for_tag (&base->kbdlight, AC);
			break;
		case TAG_COVERSTATUS:      /* private tag */
			if (cfgure)	base->flags.coveropen = taglist->data & 1;
			break;
		case TAG_PREPAREFORSLEEP:  /* private tag */
			if ((cfgure) && base->flags.nobacklight) {
				display_framebuffer (0);
			}
			break;
		case TAG_WAKEUPFROMSLEEP:  /* private tag */
			display_sync_lcdbrightness ();
			if ((cfgure) && base->lcdlight.current) {
				display_framebuffer (1);
			}
			/* base->coveropen must be synchronized after sleep. This is done
			 * by the module_pmac. This module sends the WAKEUPFROMSLEEP tag
			 * and after a short delay the COVERSTATUS tag so that no special
			 * actions need to be taken here.
			 */
			break;
		}
		taglist++;
	}
}

/**
 * @brief  Calculates the brightness level for the current ambient light
 *
 * This function calculates the brightness level dependent on the
 * ambient light level. Two controller models are supported.
 *
 *  level = f(ambient)
 *
 * <pre>
 *     lvel     2                 lvel          2
 *      |         o------           |      +--<--o-----
 *      |        /                  |      |     |
 *      |       /                   |      |     |
 *      |------o                    |------o-->--+
 *      |       1                   |     1
 *      +-----------ambient->       +--------------ambient->
 *            linear                      hysteresis
 * </pre>
 *
 * The characteristic curve of the controller will be defined by two
 * ambient light / brightness level pairs. the brightness levels will
 * set the boundaries within the controller have to work. It is only
 * able to change the LCD brightness within this limits. Outside the
 * boundaries the brightness level will be kept constant.
 *
 * The ambient light values set ambient thresholds or characteristic
 * points where the brightness controller change its behaviour. What
 * happens in detail depends on the choosen model (see sketches above).
 *
 * @param  illu       struct display_light containing the parameter
 * @param  pwrsource  is the machine running on battery or on AC?
 * @param  ambient    current ambient light level (compensated)
 * @param  ambientmax maximum ambient light level
 * @return native brightness level for the backlight controller
 */
int
display_calc_brightness (struct display_light *illu, int pwrsource, int ambient, int ambientmax)
{
	struct autoadj_param *params = pwrsource ? &illu->ac_params : &illu->battery_params;
	double a,b,x,y;

	x = (double) ambient * 100 / (double) ambientmax;

	switch (illu->autoadj_mode) {
		case AUTOADJ_OFF:
			y = 0;
			break;
		case AUTOADJ_LIN:
			if (x < params->ambient1)
				y = params->level1;
			else if (x > params->ambient2)
				y = params->level2;
			else {
				a = (double)(params->level2 - params->level1) / (double)(params->ambient2 - params->ambient1);
				b = params->level2 - a * params->ambient2;
				y = a * x + b;
			}
			break;
		case AUTOADJ_HYS:
			if (illu->autoadj_hyst_out == params->level1) {
				if (x >= params->ambient2)
					illu->autoadj_hyst_out = params->level2;
			} else {
				if (x <= params->ambient1)
					illu->autoadj_hyst_out = params->level1;
			}
			y = (double) illu->autoadj_hyst_out;
			break;
	}
	return (int) nearbyint ((double) illu->max * y / 100);
}

/**
 * @brief  Clip brightness level and start LCD brightness timer
 *
 * This functions checks the limits of a given brightness level,
 * clip it if neseccary and set it as new target.
 *
 * If no brightness timer function is running, this function
 * will start one. The timer function call the backlight driver
 * to change the brightness value of the LCD backlight.
 *
 * @param  level new LCD brightness level
 * @return level clipped to a valid range
 */
int
display_set_lcdbrightness(int level)
{
	struct moddata_display *base = &modbase_display;
	struct display_light *illu = &base->lcdlight;
	int val;
	
	if (base->flags.nobacklight) {
		if (level > 0) level = illu->max;
		illu->target = level;
		return -1;
	} else {
		val = display_clip_brightness (illu, level);
		if (illu->isrunning == 0) {
			illu->isrunning = 1;

#if defined(DEBUG) && DISPLAYTIMER
			illu->steps = abs(illu->target - illu->current);
			gettimeofday (&illu->start, NULL);
			print_msg (PBB_INFO, "DBG: LCD Display Timer started.\n");
#endif
			/* The timer needs always some more time to react as given
			 * so we decrease the time by 1 milisecond here to compensate
			 * this behaviour.
			 */
			g_timeout_add (illu->fading ? illu->fading - 1 : 9, display_lcdtimer, NULL);
		}
#if defined(DEBUG) && BRIGHTNESS
		print_msg (PBB_INFO, "DBG: LCD Brightness: %d\n", val);
#endif
		return val;
	}
}

/**
 * @brief  Clip brightness level and start keyboard brightness timer
 *
 * This functions checks the limits of a given brightness level,
 * clip it if neseccary and set it as new target.
 *
 * If no brightness timer function is running, this function
 * will start one. The timer function call the backlight driver
 * to change the brightness level of the keyboard illumination.
 *
 * @param  level new keyboard brightness level
 * @return level clipped to a valid range
 */
int
display_set_kbdbrightness(int level)
{
	struct moddata_display *base = &modbase_display;
	struct display_light *illu = &base->kbdlight;
	int val;
	
	val = display_clip_brightness (illu, level);
	if (illu->isrunning == 0) {
		illu->isrunning = 1;

#if defined(DEBUG) && DISPLAYTIMER
		illu->steps = abs(illu->target - illu->current);
		gettimeofday (&illu->start, NULL);
		print_msg (PBB_INFO, "DBG: KBD Display Timer started.\n");
#endif
		/* The timer needs always some more time to react as given
		 * so we decrease the time by 1 milisecond here to compensate
		 * this behaviour.
		 */
		g_timeout_add (illu->fading ? illu->fading - 1 : 9, display_kbdtimer, NULL);
	}	
#if defined(DEBUG) && BRIGHTNESS
	print_msg (PBB_INFO, "DBG: KBD Brightness: %d\n", val);
#endif
	return val;
}

int
display_clip_brightness(struct display_light *illu, int level)
{
	if (level > illu->max)
		level = illu->max;
	if (level < 0)
		level = 0;
	illu->target = level;
	return level;
}

void
display_set_fading (struct display_light *illu, int fading)
{
	double timer, step;

	if (fading > 0) {
		timer = (double) fading / (illu->max + 1);
		step  = ceil (5.0 / timer);
		if (step < illu->max) {
			illu->fading = floor (timer * step + 0.5);
			illu->fadingstep = step;
			goto out;
		}
	}
	illu->fading = 0;
	illu->fadingstep = 1;
out:
#if defined(DEBUG) && DISPLAYTIMER
	print_msg (PBB_INFO, "DBG: Timer: %d, step: %d\n", illu->fading, illu->fadingstep);
#endif
	return;
}

int
display_get_fading (struct display_light *illu)
{
	return illu->fading * (illu->max + 1) / illu->fadingstep;
}

void
display_set_autoadj_parameter (struct display_light *illu, int pwrsource, int *params)
{
	struct autoadj_param *parm;

	parm = pwrsource ? &illu->ac_params : &illu->battery_params;
	parm->ambient1 = params[0] < 0 ? 0 : params[0] > 100 ? 100 : params[0];
	parm->level1   = params[1] < 0 ? 0 : params[1] > 100 ? 100 : params[1];
	parm->ambient2 = params[2] < 0 ? 0 : params[2] > 100 ? 100 : params[2];
	parm->level2   = params[3] < 0 ? 0 : params[3] > 100 ? 100 : params[3];
	g_free (params);
}

void
display_set_autoadj_from_tag (struct display_light *illu, int pwrsource, tag_t data)
{
	int *params = g_new0 (int, 4);

	printf("data = 0x%lx\n", data);
	params[0] = (data >> 24) & 0xff;
	params[1] = (data >> 16) & 0xff;
	params[2] = (data >>  8) & 0xff;
	params[3] = data & 0xff;
	display_set_autoadj_parameter (illu, pwrsource, params);
}

tag_t
display_get_autoadj_for_tag (struct display_light *illu, int pwrsource)
{
	struct autoadj_param *parm;
	tag_t data;

	parm = pwrsource ? &illu->ac_params : &illu->battery_params;
	data  = parm->ambient1 << 24 | parm->level1 << 16;
	data |= parm->ambient2 <<  8 | parm->level2;
	return data;
}

/**
 * @brief Change the brightness relative to the current level
 *
 * This function changes the display and keyboard brightness relative
 * to its current value. The opcode defines the action and can have
 * following values:
 *   @li <b>OP_DIM_LIGHT</b> - Dim display to lowest level, but
 *       remember the former value
 *   @li <b>OP_DIM_OFF</b> - Switch display off but remember the
 *       former value
 *   @li <b>OP_DIM_RECOVER</b> - Reset the display brightness back
 *       to the remembered level
 *
 * If someone else has changed the display or keyboard brighhtness,
 * this function also synchronize the internal brightness levels with
 * the real ones.
 *
 * This function is mainly used to dim the display when the user is idle.
 *
 * The brightness level will be set by display_lcdtimer() and
 * display_kbdtimer() functions.
 *
 * @see display_lcdtimer(), display_kbdtimer()
 *
 * @param  op  opcode, see list above
 */
void
display_change_brightness(int op)
{
	struct moddata_display *base = &modbase_display;

	/* somebody might have changed the brightness level directly so
	 * check if we are still synchron here
	 */
	display_sync_lcdbrightness ();
	
	if (((op == OP_DIM_RECOVER) && (base->flags.status == STATUS_NORMAL)) ||
	     ((op == OP_DIM_LIGHT) && (base->flags.status != STATUS_NORMAL)) ||
	     ((op == OP_DIM_OFF) && (base->flags.status == STATUS_OFF)))
		return;

	/* if a backlight controller is not available try to dim to
	 * complete darkness in hope the framebuffer device is able
	 * to switch the backlight off. The framebuffer option must
	 * be configured accordingly to make this work.
	 */
	if ((base->flags.dimfullydark || base->flags.nobacklight) && op == OP_DIM_LIGHT)
		op = OP_DIM_OFF;

	switch (op) {
	case OP_DIM_LIGHT:
		base->lcdlight.backup = base->lcdlight.current;
		/* if current LCD brightness level is 0 then let it
		 *  as it is because it is user's will */
		if (base->lcdlight.current > LCD_BRIGHTNESS_OFF)
			base->lcdlight.target = LCD_BRIGHTNESS_MIN;
		base->kbdlight.backup = base->kbdlight.current;
		base->kbdlight.target = KBD_BRIGHTNESS_OFF;
		base->flags.status = STATUS_DIMMED;
		break;
	case OP_DIM_OFF:
		if (base->flags.status != STATUS_DIMMED) {
			base->lcdlight.backup = base->lcdlight.current;
			base->kbdlight.backup = base->kbdlight.current;
		}
		base->lcdlight.target = LCD_BRIGHTNESS_OFF;
		base->kbdlight.target = KBD_BRIGHTNESS_OFF;
		base->flags.status = STATUS_OFF;
		break;
	case OP_DIM_RECOVER:
		base->lcdlight.target = base->lcdlight.backup;
		/* don't recover zero display brightness
		 * set to minimum brightness instead */
		if (base->lcdlight.target == LCD_BRIGHTNESS_OFF)
			base->lcdlight.target = LCD_BRIGHTNESS_MIN;
		base->kbdlight.target = base->kbdlight.backup;
		base->flags.status = STATUS_NORMAL;
		break;
	}
	display_set_lcdbrightness(base->lcdlight.target);
	display_set_kbdbrightness(base->kbdlight.target);
}

void
display_framebuffer (int on)
{
	struct moddata_display *base = &modbase_display;
	int fd = -1, err;

	if (base->flags.ctrl_fb) {
		if ((fd = open(base->fbdev, O_RDWR)) >= 0) {    /* open framebuffer device */
			err = ioctl(fd, FBIOBLANK, on ? 0 : VESA_POWERDOWN + 1);
			close (fd);
		} else
			print_msg (PBB_ERR, _("Can't open framebuffer device '%s'.\n"), base->fbdev);
	}
}

void
display_sync_lcdbrightness (void)
{
	struct moddata_display *base = &modbase_display;
	int level;

	level = process_queue_single (QUERYQUEUE, TAG_BACKLIGHTLEVEL, -1);
	if (level != base->lcdlight.current) {
		base->flags.status = STATUS_NORMAL;
		if (level != -1) {
			/* Somebody else has changed the LCD brightness
			 * so sync our system. */
			base->lcdlight.current  = level;
			base->lcdlight.target   = level;
			base->flags.nobacklight = 0;
		} else {
			/* no backlight controller available */
			base->lcdlight.current  = base->lcdlight.max;
			base->lcdlight.target   = base->lcdlight.max;
			base->flags.nobacklight = 1;
		}
	}
}

int
display_switchmirror ()
{
	struct moddata_display *base = &modbase_display;
	unsigned long value;
	int fd, rc = -1;
	
	if ((fd = open(base->fbdev, O_RDWR)) >= 0) {    /* open framebuffer device */
		if (ioctl(fd, FBIO_ATY128_GET_MIRROR, &value) == 0) {
			value ^= ATY_MIRROR_CRT_ON;
			ioctl(fd, FBIO_ATY128_SET_MIRROR, &value);
			rc = value & ATY_MIRROR_CRT_ON ? 1 : 0;
		} else if (ioctl(fd, FBIO_RADEON_GET_MIRROR, &value) == 0) {
			value ^= ATY_RADEON_CRT_ON;
			ioctl(fd, FBIO_RADEON_SET_MIRROR, &value);
			rc = value & ATY_RADEON_CRT_ON ? 1 : 0;
		}				
		close (fd);
	} else
		print_msg (PBB_ERR, _("Can't open framebuffer device '%s'.\n"), base->fbdev);
	return rc;
}

