/* --------------------------------------------------------------------------
 * driver_backlight_sysfs.c
 * This module contains the low level driver to control a LCD backlight via
 * /sys/class/backlight. The low level driver overloads methods of class
 * backlight.
 *
 * Copyright 2002-2006 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include <glib.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "class_backlight.h"
#include "driver_backlight_sysfs.h"
#include "debug.h"

/** Driver SysFS Backlight Interface **/

/* This driver uses the new kernel backlight interface introduced in kernel
 * version 2.6.18. It won't work on earlier kernels.
 *
 * It is also recommended to compile the kernel *wihtout* the kernel option
 * CONFIG_PMAC_BACKLIGHT_LEGACY to prevent conflicts with ancient kernel
 * features controlling the backlight.
 */

static GString *sysfs_backlight_path;

void
driver_backlight_sysfs_exit ()
{
	g_string_free (sysfs_backlight_path, TRUE);
}

int
sysfsbl_get_brightness ()
{
	GString *var;
	FILE *fd;
	int brightness = -1;
	char buffer[5];

	var = g_string_new (sysfs_backlight_path->str);
	var = g_string_append (var, "/brightness");

	if ((fd = fopen(var->str, "r")) >= 0) {
		fgets(buffer, sizeof(buffer), fd);
		buffer[4] = 0;  /* to be safe, terminate the buffer */
		brightness = atoi (buffer);
		fclose(fd);
	}
	
	g_string_free (var, TRUE);
	return brightness;
}

int
sysfsbl_get_brightness_max ()
{
	GString *var;
	FILE *fd;
	int max_brightness = -1;
	char buffer[5];

	var = g_string_new (sysfs_backlight_path->str);
	var = g_string_append (var, "/max_brightness");

	if ((fd = fopen(var->str, "r")) >= 0) {
		fgets(buffer, sizeof(buffer), fd);
		buffer[4] = 0;  /* to be safe, terminate the buffer */
		max_brightness = atoi (buffer);
		fclose(fd);
	}
	
	g_string_free (var, TRUE);
	return max_brightness;
}

void
sysfsbl_set_brightness (int val)
{
	GString *var;
	FILE *fd;
	char buffer[5];

	if (val > 255) return;

	var = g_string_new (sysfs_backlight_path->str);
	var = g_string_append (var, "/brightness");
	sprintf(buffer, "%d\n", val);

	if ((fd = fopen(var->str, "w")) >= 0) {
		fputs(buffer, fd);
		fclose(fd);
	}
	
	g_string_free (var, TRUE);
}

static struct driver_backlight driver_backlight_sysfs = {
	.name               = N_("SysFS Backlight Driver"),
	.get_brightness     = sysfsbl_get_brightness,
	.get_brightness_max = sysfsbl_get_brightness_max,
	.set_brightness     = sysfsbl_set_brightness,
	.driver_exit        = driver_backlight_sysfs_exit
};

struct driver_backlight *
driver_backlight_sysfs_init (struct tagitem *taglist)
{
	DIR *dh;
	struct dirent *dir;

	sysfs_backlight_path = g_string_new ("/sys/class/backlight");

	if ((dh = opendir(sysfs_backlight_path->str))) {
		while ((dir = readdir(dh))) {
			if (dir->d_name[0] == '.') continue;
			if (g_str_has_suffix(dir->d_name, "bl0") ||
				g_str_has_suffix(dir->d_name, "bl")) {
				g_string_append_printf (sysfs_backlight_path, "/%s", dir->d_name);
#if defined(DEBUG)
				print_msg (PBB_INFO, "DBG: SysFS Backlight Path: %s\n", sysfs_backlight_path->str);
#endif
				return &driver_backlight_sysfs;
			}
		}
	}

	return NULL;
}

