/*
 * unity-webapps-window-tracker-dbus-controllable.c
 * Copyright (C) Canonical LTD 2012
 *
 * unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <gio/gio.h>

#include "unity-webapps-window-tracker-dbus-controllable.h"
#include "../unity-webapps-debug.h"


#if defined(WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_PATH)
#error WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_PATH already defined
#endif

#if defined(WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_SIGNAL)
#error WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_SIGNAL already defined
#endif

#define WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_PATH \
  "com.canonical.Unity.Webapps.Tests.WindowTracker.DbusControllable"

#define WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_SIGNAL \
  "UpdateLatestWindowId"


struct _UnityWebappsWindowTrackerDbusControllablePrivate {
  GDBusConnection *connection;
  guint64 active_window_id;
};

G_DEFINE_TYPE(UnityWebappsWindowTrackerDbusControllable, unity_webapps_window_tracker_dbus_controllable, UNITY_WEBAPPS_TYPE_WINDOW_TRACKER)

#define UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE_GET_PRIVATE(object) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((object), UNITY_WEBAPPS_TYPE_WINDOW_TRACKER_DBUS_CONTROLLABLE, UnityWebappsWindowTrackerDbusControllablePrivate))

enum 
{
  PROP_0,
  PROP_DBUS_CONNECTION,
};


static void
_window_tracker_dbus_controllable_update_latest_window_id (GDBusConnection *connection,
                                                           const gchar *sender,
                                                           const gchar *object,
                                                           const gchar *interface,
                                                           const gchar *signal,
                                                           GVariant *params,
                                                           gpointer user_data)
{
  UnityWebappsWindowTrackerDbusControllable *tracker;
  guint64 window_id;

  tracker = (UnityWebappsWindowTrackerDbusControllable*) user_data;

  UNITY_WEBAPPS_NOTE (WINDOW_TRACKER,
                      "Received active window id update signal");

  if (!tracker)
    return;

  if (g_strcmp0(signal, WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_SIGNAL))
    return;

  window_id = 0;
  g_variant_get (params, "(t)", &window_id, NULL);

  UNITY_WEBAPPS_NOTE (WINDOW_TRACKER,
                      "Updating active window id changed signal: %" G_GUINT64_FORMAT,
                      window_id);

  tracker->priv->active_window_id = window_id;

  g_object_notify (G_OBJECT (tracker), "active-window-id");  
}

static void
unity_webapps_window_tracker_dbus_controllable_finalize (GObject *object)
{
  UnityWebappsWindowTrackerDbusControllable *tracker;
  tracker = UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE (object);
  (void)tracker;
}

static void
unity_webapps_window_tracker_dbus_controllable_dispose (GObject * object)
{
  UnityWebappsWindowTrackerDbusControllable *tracker;
  tracker = UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE (object);

  if (!tracker)
    return;

  g_object_unref(UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE_GET_PRIVATE(tracker)->connection);
}

static void
_setup_dbus_connection(UnityWebappsWindowTrackerDbusControllable *tracker)
{
  if (!tracker->priv->connection)
    return;

  UNITY_WEBAPPS_NOTE (WINDOW_TRACKER, "Setting up DBUS connection");

  //TODO(aau): a bit weak & risky to subscribe to all
  g_dbus_connection_signal_subscribe (tracker->priv->connection,
                                      NULL,
                                      WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_PATH,
                                      WINDOW_TRACKER_DBUS_CONTROLLABLE_DBUS_SIGNAL,
                                      NULL,
                                      NULL,
                                      G_DBUS_SIGNAL_FLAGS_NONE,
                                      _window_tracker_dbus_controllable_update_latest_window_id,
                                      tracker,
                                      NULL);
}

static void
_window_tracker_dbus_controllable_get_property (GObject *object,
                                                guint property_id,
                                                GValue *value,
                                                GParamSpec *pspec)
{
  UnityWebappsWindowTrackerDbusControllable *tracker;
  tracker = UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE (object);

  if (!tracker)
    return;

  switch (property_id)
    {
    case PROP_DBUS_CONNECTION:
      g_value_set_object (value, tracker->priv->connection);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
_window_tracker_dbus_controllable_set_property (GObject *object,
                                                guint property_id,
                                                const GValue *value,
                                                GParamSpec *pspec)
{
  UnityWebappsWindowTrackerDbusControllable *tracker;
  tracker = UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE (object);

  if (!tracker)
    return;

  switch (property_id)
    {
    case PROP_DBUS_CONNECTION:
      tracker->priv->connection = g_object_ref((GDBusConnection*) g_value_dup_object (value));
      _setup_dbus_connection(tracker);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static guint64
_window_tracker_dbus_controllable_get_active_window_id (UnityWebappsWindowTracker *tracker)
{
  UnityWebappsWindowTrackerDbusControllable *dbus_tracker;
  dbus_tracker = UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE (tracker);

  if (!dbus_tracker || !UNITY_WEBAPPS_IS_WINDOW_TRACKER_DBUS_CONTROLLABLE(dbus_tracker))
    return 0;

  return dbus_tracker->priv->active_window_id;
}

static void
unity_webapps_window_tracker_dbus_controllable_class_init (UnityWebappsWindowTrackerDbusControllableClass *klass)
{
  GParamSpec *pspec;
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  UnityWebappsWindowTrackerClass *window_tracker_class = UNITY_WEBAPPS_WINDOW_TRACKER_CLASS (klass);

  object_class->finalize = unity_webapps_window_tracker_dbus_controllable_finalize;
  object_class->dispose = unity_webapps_window_tracker_dbus_controllable_dispose;

  object_class->get_property = _window_tracker_dbus_controllable_get_property;
  object_class->set_property = _window_tracker_dbus_controllable_set_property;

  window_tracker_class->get_active_window_id =
    _window_tracker_dbus_controllable_get_active_window_id;

  g_type_class_add_private (object_class,
                            sizeof(UnityWebappsWindowTrackerDbusControllablePrivate));

  pspec = g_param_spec_object("connection",
                              "Connection",
                              "DBUS session connection",
			      G_TYPE_DBUS_CONNECTION,
			      G_PARAM_READWRITE |
			      G_PARAM_CONSTRUCT_ONLY |
			      G_PARAM_STATIC_STRINGS);

  g_object_class_install_property (object_class,
                                   PROP_DBUS_CONNECTION,
                                   pspec);
}

static void
unity_webapps_window_tracker_dbus_controllable_init (UnityWebappsWindowTrackerDbusControllable *tracker)
{
  tracker->priv = UNITY_WEBAPPS_WINDOW_TRACKER_DBUS_CONTROLLABLE_GET_PRIVATE (tracker);
  tracker->priv->active_window_id = 0;
}

UnityWebappsWindowTrackerDbusControllable *
unity_webapps_dbus_controllable_window_tracker_new (GDBusConnection *connection)
{
  return g_object_new (UNITY_WEBAPPS_TYPE_WINDOW_TRACKER_DBUS_CONTROLLABLE,
                       "dbus-connection",
                       connection,
                       NULL);
}

