/*
A library to allow applictions to provide simple indications of
information to be displayed to users of the application through the
interface shell.

Copyright 2009 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of either or both of the following licenses:

1) the GNU Lesser General Public License version 3, as published by the 
Free Software Foundation; and/or
2) the GNU Lesser General Public License version 2.1, as published by 
the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the applicable version of the GNU Lesser General Public 
License for more details.

You should have received a copy of both the GNU Lesser General Public 
License version 3 and version 2.1 along with this program.  If not, see 
<http://www.gnu.org/licenses/>
*/

#include "listener.h"
#include <dbus/dbus-glib-bindings.h>

#include "../libindicate/listener-private.h"

typedef enum _get_property_type get_property_type;
enum _get_property_type {
	PROPERTY_TYPE_ICON
};

typedef struct _get_property_t get_property_t;
struct _get_property_t {
	GCallback cb;
	gpointer data;
	IndicateListener * listener;
	IndicateListenerServer * server;
	IndicateListenerIndicator * indicator;
	gchar * property;
	get_property_type type;
};

static void
get_property_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	get_property_t * get_property_data = (get_property_t *)user_data;
	GError * error = NULL;
	GVariant * retvalue = g_dbus_proxy_call_finish(G_DBUS_PROXY(object), res, &error);

	if (error != NULL) {
		g_warning("Unable to get property data: %s", error->message);
		g_error_free(error);
		return;
	}

	GVariant * value = g_variant_get_child_value(retvalue, 0);

	switch (get_property_data->type) {
	case PROPERTY_TYPE_ICON: {
		indicate_gtk_listener_get_property_icon_cb cb = (indicate_gtk_listener_get_property_icon_cb)get_property_data->cb;
		const gchar * OUT_value = g_variant_get_string(value, NULL);

		/* There is no icon */
		if (OUT_value == NULL || OUT_value[0] == '\0') {
			break;
		}

		gsize length = 0;
		guchar * icondata = g_base64_decode(OUT_value, &length);
		
		GInputStream * input = g_memory_input_stream_new_from_data(icondata, length, NULL);
		if (input == NULL) {
			g_warning("Cound not create input stream from icon property data");
			g_free(icondata);
			break;
		}

		GError * error = NULL;
		GdkPixbuf * icon = gdk_pixbuf_new_from_stream(input, NULL, &error);
		if (icon != NULL) {
			cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, icon, get_property_data->data);
		}

		if (error != NULL) {
			g_warning("Unable to build Pixbuf from icon data: %s", error->message);
			g_error_free(error);
		}

		error = NULL;
		g_input_stream_close(input, NULL, &error);
		if (error != NULL) {
			g_warning("Unable to close input stream: %s", error->message);
			g_error_free(error);
		}
		g_free(icondata);
		break;
	}
	}

	g_variant_unref(value);
	g_free(get_property_data->property);
	g_free(get_property_data);

	return;
};

/* A small function to take the common list of parameters and
   build a callback structure to hold them all.  Eventually we
   get the data and unwind this structure. */
static void
get_property_helper (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, GCallback callback, gpointer data, get_property_type prop_type)
{
	/* g_debug("get_property_helper: %s %d", property, prop_type); */
	/* TODO: Do we need to somehow refcount the server/indicator while we're waiting on this? */
	get_property_t * get_property_data = g_new0(get_property_t, 1);
	get_property_data->cb = callback;
	get_property_data->data = data;
	get_property_data->listener = listener;
	get_property_data->server = server;
	get_property_data->indicator = indicator;
	get_property_data->property = g_strdup(property);
	get_property_data->type = prop_type;
	
	g_dbus_proxy_call(server->proxy,
	                  "GetIndicatorProperty",
	                  g_variant_new("(us)",
	                                INDICATE_LISTENER_INDICATOR_ID(indicator), property),
	                  G_DBUS_CALL_FLAGS_NONE,
	                  -1, /* timeout */
	                  NULL, /* cancel */
	                  get_property_cb,
	                  get_property_data);

	return;
}

/**
 * indicate_gtk_listener_get_property_icon:
 * @listener: The #IndicateListener representing the connection
 * @server: The server that the indicator is on
 * @indicator: Which indicator is being queried
 * @property: Name of the property to get
 * @callback: (scope async): The callback function to call with the data
 * @data: Arbitrary data to give the callback
 * 
 * A function to get a property from an indicator on a server
 * and bring it back locally.  This wraps all the hassle of using
 * the DBus API and makes it pretty easy to get properties.
 * 
 * Very similar to #indicate_listener_get_property but converts
 * the final value into an icon for easy (and type-safe)
 * usage by listeners.
 */
void
indicate_gtk_listener_get_property_icon (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_gtk_listener_get_property_icon_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_ICON);
}

