#include "Camellia.h"
#include "Common/Endian.h"
#include "Crypto/cpu.h"
#include "Crypto/misc.h"

#if CRYPTOPP_BOOL_X64

/* camellia.c ver 1.2.0-x86_64_asm1.1
 *
 * Copyright (c) 2006,2007
 * NTT (Nippon Telegraph and Telephone Corporation) . All rights reserved.
 *
 * SuperCop integration:
 * Copyright  2013 Jussi Kivilinna <jussi.kivilinna@iki.fi>
 *
 * VeraCrypt integration:
 * Copyright  2017 Mounir IDRASSI <mounir.idrassi@idrix.fr>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer as
 *   the first lines of this file unmodified.
 * 2. Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY NTT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL NTT BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * Algorithm Specification
 *  http://info.isl.ntt.co.jp/crypt/eng/camellia/specifications.html
 */

/* Adapted for VeraCrypt */

#include "Common/Crypto.h"

#ifndef _WIN32
extern int IsAesHwCpuSupported ();
#endif

void camellia_encrypt_asm(const byte *ctx, void *dst, const void *src);
void camellia_decrypt_asm(const byte *ctx, void *dst, const void *src);
void camellia_enc_blk2(const byte *ctx, byte *dst, const byte *src);
void camellia_dec_blk2(const byte *ctx, byte *dst, const byte *src);
void camellia_ecb_enc_16way(const byte *ctx, byte *dst, const byte *src);
void camellia_ecb_dec_16way(const byte *ctx, byte *dst, const byte *src);

/* key constants */

#define CAMELLIA_SIGMA1L (0xA09E667FL)
#define CAMELLIA_SIGMA1R (0x3BCC908BL)
#define CAMELLIA_SIGMA2L (0xB67AE858L)
#define CAMELLIA_SIGMA2R (0x4CAA73B2L)
#define CAMELLIA_SIGMA3L (0xC6EF372FL)
#define CAMELLIA_SIGMA3R (0xE94F82BEL)
#define CAMELLIA_SIGMA4L (0x54FF53A5L)
#define CAMELLIA_SIGMA4R (0xF1D36F1CL)
#define CAMELLIA_SIGMA5L (0x10E527FAL)
#define CAMELLIA_SIGMA5R (0xDE682D1DL)
#define CAMELLIA_SIGMA6L (0xB05688C2L)
#define CAMELLIA_SIGMA6R (0xB3E6C1FDL)

/*
 *  macros
 */


# define GETU32(p) bswap_32(*((uint32 *)(p)))
# define PUTU32(ct, st) {*((uint32 *)(ct)) = bswap_32((st));}

#define CamelliaSubkeyL(INDEX) (subkey[(INDEX)*2])
#define CamelliaSubkeyR(INDEX) (subkey[(INDEX)*2 + 1])

/* rotation right shift 1byte */
#define CAMELLIA_RR8(x) (((x) >> 8) + ((x) << 24))
/* rotation left shift 1bit */
#define CAMELLIA_RL1(x) (((x) << 1) + ((x) >> 31))
/* rotation left shift 1byte */
#define CAMELLIA_RL8(x) (((x) << 8) + ((x) >> 24))

#define CAMELLIA_ROLDQ(ll, lr, rl, rr, w0, w1, bits)	\
    do {						\
	w0 = ll;					\
	ll = (ll << bits) + (lr >> (32 - bits));	\
	lr = (lr << bits) + (rl >> (32 - bits));	\
	rl = (rl << bits) + (rr >> (32 - bits));	\
	rr = (rr << bits) + (w0 >> (32 - bits));	\
    } while(0)

#define CAMELLIA_ROLDQo32(ll, lr, rl, rr, w0, w1, bits)	\
    do {						\
	w0 = ll;					\
	w1 = lr;					\
	ll = (lr << (bits - 32)) + (rl >> (64 - bits));	\
	lr = (rl << (bits - 32)) + (rr >> (64 - bits));	\
	rl = (rr << (bits - 32)) + (w0 >> (64 - bits));	\
	rr = (w0 << (bits - 32)) + (w1 >> (64 - bits));	\
    } while(0)

#define CAMELLIA_SP11101110(INDEX) (camellia_sp11101110[(INDEX)])
#define CAMELLIA_SP02220222(INDEX) (camellia_sp02220222[(INDEX)])
#define CAMELLIA_SP30333033(INDEX) (camellia_sp30333033[(INDEX)])
#define CAMELLIA_SP44044404(INDEX) (camellia_sp44044404[(INDEX)])

#define CAMELLIA_SP10011110(INDEX) (camellia_sp10011110[(INDEX)])
#define CAMELLIA_SP22000222(INDEX) (camellia_sp22000222[(INDEX)])
#define CAMELLIA_SP03303033(INDEX) (camellia_sp03303033[(INDEX)])
#define CAMELLIA_SP00444404(INDEX) (camellia_sp00444404[(INDEX)])

#define CAMELLIA_F(xl, xr, kl, kr, yl, yr, il, ir, t0, t1)	\
    do {							\
	uint64 __tmp;						\
	il = xl ^ kl;						\
	ir = xr ^ kr;						\
	t0 = il >> 16;						\
	t1 = ir >> 16;						\
	__tmp = CAMELLIA_SP11101110(ir & 0xff)			\
	    ^ CAMELLIA_SP02220222((t1 >> 8) & 0xff)		\
	    ^ CAMELLIA_SP30333033(t1 & 0xff)			\
	    ^ CAMELLIA_SP44044404((ir >> 8) & 0xff)		\
	    ^ CAMELLIA_SP10011110((t0 >> 8) & 0xff)		\
	    ^ CAMELLIA_SP22000222(t0 & 0xff)			\
	    ^ CAMELLIA_SP03303033((il >> 8) & 0xff)		\
	    ^ CAMELLIA_SP00444404(il & 0xff);			\
	yl = (uint32) __tmp;						\
	yr = (uint32) (__tmp >> 32);					\
    } while(0)

const uint64 camellia_sp10011110[256] = {
    0x7000007070707000ULL, 0x8200008282828200ULL, 0x2c00002c2c2c2c00ULL,
    0xec0000ecececec00ULL, 0xb30000b3b3b3b300ULL, 0x2700002727272700ULL,
    0xc00000c0c0c0c000ULL, 0xe50000e5e5e5e500ULL, 0xe40000e4e4e4e400ULL,
    0x8500008585858500ULL, 0x5700005757575700ULL, 0x3500003535353500ULL,
    0xea0000eaeaeaea00ULL, 0x0c00000c0c0c0c00ULL, 0xae0000aeaeaeae00ULL,
    0x4100004141414100ULL, 0x2300002323232300ULL, 0xef0000efefefef00ULL,
    0x6b00006b6b6b6b00ULL, 0x9300009393939300ULL, 0x4500004545454500ULL,
    0x1900001919191900ULL, 0xa50000a5a5a5a500ULL, 0x2100002121212100ULL,
    0xed0000edededed00ULL, 0x0e00000e0e0e0e00ULL, 0x4f00004f4f4f4f00ULL,
    0x4e00004e4e4e4e00ULL, 0x1d00001d1d1d1d00ULL, 0x6500006565656500ULL,
    0x9200009292929200ULL, 0xbd0000bdbdbdbd00ULL, 0x8600008686868600ULL,
    0xb80000b8b8b8b800ULL, 0xaf0000afafafaf00ULL, 0x8f00008f8f8f8f00ULL,
    0x7c00007c7c7c7c00ULL, 0xeb0000ebebebeb00ULL, 0x1f00001f1f1f1f00ULL,
    0xce0000cececece00ULL, 0x3e00003e3e3e3e00ULL, 0x3000003030303000ULL,
    0xdc0000dcdcdcdc00ULL, 0x5f00005f5f5f5f00ULL, 0x5e00005e5e5e5e00ULL,
    0xc50000c5c5c5c500ULL, 0x0b00000b0b0b0b00ULL, 0x1a00001a1a1a1a00ULL,
    0xa60000a6a6a6a600ULL, 0xe10000e1e1e1e100ULL, 0x3900003939393900ULL,
    0xca0000cacacaca00ULL, 0xd50000d5d5d5d500ULL, 0x4700004747474700ULL,
    0x5d00005d5d5d5d00ULL, 0x3d00003d3d3d3d00ULL, 0xd90000d9d9d9d900ULL,
    0x0100000101010100ULL, 0x5a00005a5a5a5a00ULL, 0xd60000d6d6d6d600ULL,
    0x5100005151515100ULL, 0x5600005656565600ULL, 0x6c00006c6c6c6c00ULL,
    0x4d00004d4d4d4d00ULL, 0x8b00008b8b8b8b00ULL, 0x0d00000d0d0d0d00ULL,
    0x9a00009a9a9a9a00ULL, 0x6600006666666600ULL, 0xfb0000fbfbfbfb00ULL,
    0xcc0000cccccccc00ULL, 0xb00000b0b0b0b000ULL, 0x2d00002d2d2d2d00ULL,
    0x7400007474747400ULL, 0x1200001212121200ULL, 0x2b00002b2b2b2b00ULL,
    0x2000002020202000ULL, 0xf00000f0f0f0f000ULL, 0xb10000b1b1b1b100ULL,
    0x8400008484848400ULL, 0x9900009999999900ULL, 0xdf0000dfdfdfdf00ULL,
    0x4c00004c4c4c4c00ULL, 0xcb0000cbcbcbcb00ULL, 0xc20000c2c2c2c200ULL,
    0x3400003434343400ULL, 0x7e00007e7e7e7e00ULL, 0x7600007676767600ULL,
    0x0500000505050500ULL, 0x6d00006d6d6d6d00ULL, 0xb70000b7b7b7b700ULL,
    0xa90000a9a9a9a900ULL, 0x3100003131313100ULL, 0xd10000d1d1d1d100ULL,
    0x1700001717171700ULL, 0x0400000404040400ULL, 0xd70000d7d7d7d700ULL,
    0x1400001414141400ULL, 0x5800005858585800ULL, 0x3a00003a3a3a3a00ULL,
    0x6100006161616100ULL, 0xde0000dededede00ULL, 0x1b00001b1b1b1b00ULL,
    0x1100001111111100ULL, 0x1c00001c1c1c1c00ULL, 0x3200003232323200ULL,
    0x0f00000f0f0f0f00ULL, 0x9c00009c9c9c9c00ULL, 0x1600001616161600ULL,
    0x5300005353535300ULL, 0x1800001818181800ULL, 0xf20000f2f2f2f200ULL,
    0x2200002222222200ULL, 0xfe0000fefefefe00ULL, 0x4400004444444400ULL,
    0xcf0000cfcfcfcf00ULL, 0xb20000b2b2b2b200ULL, 0xc30000c3c3c3c300ULL,
    0xb50000b5b5b5b500ULL, 0x7a00007a7a7a7a00ULL, 0x9100009191919100ULL,
    0x2400002424242400ULL, 0x0800000808080800ULL, 0xe80000e8e8e8e800ULL,
    0xa80000a8a8a8a800ULL, 0x6000006060606000ULL, 0xfc0000fcfcfcfc00ULL,
    0x6900006969696900ULL, 0x5000005050505000ULL, 0xaa0000aaaaaaaa00ULL,
    0xd00000d0d0d0d000ULL, 0xa00000a0a0a0a000ULL, 0x7d00007d7d7d7d00ULL,
    0xa10000a1a1a1a100ULL, 0x8900008989898900ULL, 0x6200006262626200ULL,
    0x9700009797979700ULL, 0x5400005454545400ULL, 0x5b00005b5b5b5b00ULL,
    0x1e00001e1e1e1e00ULL, 0x9500009595959500ULL, 0xe00000e0e0e0e000ULL,
    0xff0000ffffffff00ULL, 0x6400006464646400ULL, 0xd20000d2d2d2d200ULL,
    0x1000001010101000ULL, 0xc40000c4c4c4c400ULL, 0x0000000000000000ULL,
    0x4800004848484800ULL, 0xa30000a3a3a3a300ULL, 0xf70000f7f7f7f700ULL,
    0x7500007575757500ULL, 0xdb0000dbdbdbdb00ULL, 0x8a00008a8a8a8a00ULL,
    0x0300000303030300ULL, 0xe60000e6e6e6e600ULL, 0xda0000dadadada00ULL,
    0x0900000909090900ULL, 0x3f00003f3f3f3f00ULL, 0xdd0000dddddddd00ULL,
    0x9400009494949400ULL, 0x8700008787878700ULL, 0x5c00005c5c5c5c00ULL,
    0x8300008383838300ULL, 0x0200000202020200ULL, 0xcd0000cdcdcdcd00ULL,
    0x4a00004a4a4a4a00ULL, 0x9000009090909000ULL, 0x3300003333333300ULL,
    0x7300007373737300ULL, 0x6700006767676700ULL, 0xf60000f6f6f6f600ULL,
    0xf30000f3f3f3f300ULL, 0x9d00009d9d9d9d00ULL, 0x7f00007f7f7f7f00ULL,
    0xbf0000bfbfbfbf00ULL, 0xe20000e2e2e2e200ULL, 0x5200005252525200ULL,
    0x9b00009b9b9b9b00ULL, 0xd80000d8d8d8d800ULL, 0x2600002626262600ULL,
    0xc80000c8c8c8c800ULL, 0x3700003737373700ULL, 0xc60000c6c6c6c600ULL,
    0x3b00003b3b3b3b00ULL, 0x8100008181818100ULL, 0x9600009696969600ULL,
    0x6f00006f6f6f6f00ULL, 0x4b00004b4b4b4b00ULL, 0x1300001313131300ULL,
    0xbe0000bebebebe00ULL, 0x6300006363636300ULL, 0x2e00002e2e2e2e00ULL,
    0xe90000e9e9e9e900ULL, 0x7900007979797900ULL, 0xa70000a7a7a7a700ULL,
    0x8c00008c8c8c8c00ULL, 0x9f00009f9f9f9f00ULL, 0x6e00006e6e6e6e00ULL,
    0xbc0000bcbcbcbc00ULL, 0x8e00008e8e8e8e00ULL, 0x2900002929292900ULL,
    0xf50000f5f5f5f500ULL, 0xf90000f9f9f9f900ULL, 0xb60000b6b6b6b600ULL,
    0x2f00002f2f2f2f00ULL, 0xfd0000fdfdfdfd00ULL, 0xb40000b4b4b4b400ULL,
    0x5900005959595900ULL, 0x7800007878787800ULL, 0x9800009898989800ULL,
    0x0600000606060600ULL, 0x6a00006a6a6a6a00ULL, 0xe70000e7e7e7e700ULL,
    0x4600004646464600ULL, 0x7100007171717100ULL, 0xba0000babababa00ULL,
    0xd40000d4d4d4d400ULL, 0x2500002525252500ULL, 0xab0000abababab00ULL,
    0x4200004242424200ULL, 0x8800008888888800ULL, 0xa20000a2a2a2a200ULL,
    0x8d00008d8d8d8d00ULL, 0xfa0000fafafafa00ULL, 0x7200007272727200ULL,
    0x0700000707070700ULL, 0xb90000b9b9b9b900ULL, 0x5500005555555500ULL,
    0xf80000f8f8f8f800ULL, 0xee0000eeeeeeee00ULL, 0xac0000acacacac00ULL,
    0x0a00000a0a0a0a00ULL, 0x3600003636363600ULL, 0x4900004949494900ULL,
    0x2a00002a2a2a2a00ULL, 0x6800006868686800ULL, 0x3c00003c3c3c3c00ULL,
    0x3800003838383800ULL, 0xf10000f1f1f1f100ULL, 0xa40000a4a4a4a400ULL,
    0x4000004040404000ULL, 0x2800002828282800ULL, 0xd30000d3d3d3d300ULL,
    0x7b00007b7b7b7b00ULL, 0xbb0000bbbbbbbb00ULL, 0xc90000c9c9c9c900ULL,
    0x4300004343434300ULL, 0xc10000c1c1c1c100ULL, 0x1500001515151500ULL,
    0xe30000e3e3e3e300ULL, 0xad0000adadadad00ULL, 0xf40000f4f4f4f400ULL,
    0x7700007777777700ULL, 0xc70000c7c7c7c700ULL, 0x8000008080808000ULL,
    0x9e00009e9e9e9e00ULL,
};

const uint64 camellia_sp22000222[256] = {
    0xe0e0000000e0e0e0ULL, 0x0505000000050505ULL, 0x5858000000585858ULL,
    0xd9d9000000d9d9d9ULL, 0x6767000000676767ULL, 0x4e4e0000004e4e4eULL,
    0x8181000000818181ULL, 0xcbcb000000cbcbcbULL, 0xc9c9000000c9c9c9ULL,
    0x0b0b0000000b0b0bULL, 0xaeae000000aeaeaeULL, 0x6a6a0000006a6a6aULL,
    0xd5d5000000d5d5d5ULL, 0x1818000000181818ULL, 0x5d5d0000005d5d5dULL,
    0x8282000000828282ULL, 0x4646000000464646ULL, 0xdfdf000000dfdfdfULL,
    0xd6d6000000d6d6d6ULL, 0x2727000000272727ULL, 0x8a8a0000008a8a8aULL,
    0x3232000000323232ULL, 0x4b4b0000004b4b4bULL, 0x4242000000424242ULL,
    0xdbdb000000dbdbdbULL, 0x1c1c0000001c1c1cULL, 0x9e9e0000009e9e9eULL,
    0x9c9c0000009c9c9cULL, 0x3a3a0000003a3a3aULL, 0xcaca000000cacacaULL,
    0x2525000000252525ULL, 0x7b7b0000007b7b7bULL, 0x0d0d0000000d0d0dULL,
    0x7171000000717171ULL, 0x5f5f0000005f5f5fULL, 0x1f1f0000001f1f1fULL,
    0xf8f8000000f8f8f8ULL, 0xd7d7000000d7d7d7ULL, 0x3e3e0000003e3e3eULL,
    0x9d9d0000009d9d9dULL, 0x7c7c0000007c7c7cULL, 0x6060000000606060ULL,
    0xb9b9000000b9b9b9ULL, 0xbebe000000bebebeULL, 0xbcbc000000bcbcbcULL,
    0x8b8b0000008b8b8bULL, 0x1616000000161616ULL, 0x3434000000343434ULL,
    0x4d4d0000004d4d4dULL, 0xc3c3000000c3c3c3ULL, 0x7272000000727272ULL,
    0x9595000000959595ULL, 0xabab000000abababULL, 0x8e8e0000008e8e8eULL,
    0xbaba000000bababaULL, 0x7a7a0000007a7a7aULL, 0xb3b3000000b3b3b3ULL,
    0x0202000000020202ULL, 0xb4b4000000b4b4b4ULL, 0xadad000000adadadULL,
    0xa2a2000000a2a2a2ULL, 0xacac000000acacacULL, 0xd8d8000000d8d8d8ULL,
    0x9a9a0000009a9a9aULL, 0x1717000000171717ULL, 0x1a1a0000001a1a1aULL,
    0x3535000000353535ULL, 0xcccc000000ccccccULL, 0xf7f7000000f7f7f7ULL,
    0x9999000000999999ULL, 0x6161000000616161ULL, 0x5a5a0000005a5a5aULL,
    0xe8e8000000e8e8e8ULL, 0x2424000000242424ULL, 0x5656000000565656ULL,
    0x4040000000404040ULL, 0xe1e1000000e1e1e1ULL, 0x6363000000636363ULL,
    0x0909000000090909ULL, 0x3333000000333333ULL, 0xbfbf000000bfbfbfULL,
    0x9898000000989898ULL, 0x9797000000979797ULL, 0x8585000000858585ULL,
    0x6868000000686868ULL, 0xfcfc000000fcfcfcULL, 0xecec000000ecececULL,
    0x0a0a0000000a0a0aULL, 0xdada000000dadadaULL, 0x6f6f0000006f6f6fULL,
    0x5353000000535353ULL, 0x6262000000626262ULL, 0xa3a3000000a3a3a3ULL,
    0x2e2e0000002e2e2eULL, 0x0808000000080808ULL, 0xafaf000000afafafULL,
    0x2828000000282828ULL, 0xb0b0000000b0b0b0ULL, 0x7474000000747474ULL,
    0xc2c2000000c2c2c2ULL, 0xbdbd000000bdbdbdULL, 0x3636000000363636ULL,
    0x2222000000222222ULL, 0x3838000000383838ULL, 0x6464000000646464ULL,
    0x1e1e0000001e1e1eULL, 0x3939000000393939ULL, 0x2c2c0000002c2c2cULL,
    0xa6a6000000a6a6a6ULL, 0x3030000000303030ULL, 0xe5e5000000e5e5e5ULL,
    0x4444000000444444ULL, 0xfdfd000000fdfdfdULL, 0x8888000000888888ULL,
    0x9f9f0000009f9f9fULL, 0x6565000000656565ULL, 0x8787000000878787ULL,
    0x6b6b0000006b6b6bULL, 0xf4f4000000f4f4f4ULL, 0x2323000000232323ULL,
    0x4848000000484848ULL, 0x1010000000101010ULL, 0xd1d1000000d1d1d1ULL,
    0x5151000000515151ULL, 0xc0c0000000c0c0c0ULL, 0xf9f9000000f9f9f9ULL,
    0xd2d2000000d2d2d2ULL, 0xa0a0000000a0a0a0ULL, 0x5555000000555555ULL,
    0xa1a1000000a1a1a1ULL, 0x4141000000414141ULL, 0xfafa000000fafafaULL,
    0x4343000000434343ULL, 0x1313000000131313ULL, 0xc4c4000000c4c4c4ULL,
    0x2f2f0000002f2f2fULL, 0xa8a8000000a8a8a8ULL, 0xb6b6000000b6b6b6ULL,
    0x3c3c0000003c3c3cULL, 0x2b2b0000002b2b2bULL, 0xc1c1000000c1c1c1ULL,
    0xffff000000ffffffULL, 0xc8c8000000c8c8c8ULL, 0xa5a5000000a5a5a5ULL,
    0x2020000000202020ULL, 0x8989000000898989ULL, 0x0000000000000000ULL,
    0x9090000000909090ULL, 0x4747000000474747ULL, 0xefef000000efefefULL,
    0xeaea000000eaeaeaULL, 0xb7b7000000b7b7b7ULL, 0x1515000000151515ULL,
    0x0606000000060606ULL, 0xcdcd000000cdcdcdULL, 0xb5b5000000b5b5b5ULL,
    0x1212000000121212ULL, 0x7e7e0000007e7e7eULL, 0xbbbb000000bbbbbbULL,
    0x2929000000292929ULL, 0x0f0f0000000f0f0fULL, 0xb8b8000000b8b8b8ULL,
    0x0707000000070707ULL, 0x0404000000040404ULL, 0x9b9b0000009b9b9bULL,
    0x9494000000949494ULL, 0x2121000000212121ULL, 0x6666000000666666ULL,
    0xe6e6000000e6e6e6ULL, 0xcece000000cececeULL, 0xeded000000edededULL,
    0xe7e7000000e7e7e7ULL, 0x3b3b0000003b3b3bULL, 0xfefe000000fefefeULL,
    0x7f7f0000007f7f7fULL, 0xc5c5000000c5c5c5ULL, 0xa4a4000000a4a4a4ULL,
    0x3737000000373737ULL, 0xb1b1000000b1b1b1ULL, 0x4c4c0000004c4c4cULL,
    0x9191000000919191ULL, 0x6e6e0000006e6e6eULL, 0x8d8d0000008d8d8dULL,
    0x7676000000767676ULL, 0x0303000000030303ULL, 0x2d2d0000002d2d2dULL,
    0xdede000000dededeULL, 0x9696000000969696ULL, 0x2626000000262626ULL,
    0x7d7d0000007d7d7dULL, 0xc6c6000000c6c6c6ULL, 0x5c5c0000005c5c5cULL,
    0xd3d3000000d3d3d3ULL, 0xf2f2000000f2f2f2ULL, 0x4f4f0000004f4f4fULL,
    0x1919000000191919ULL, 0x3f3f0000003f3f3fULL, 0xdcdc000000dcdcdcULL,
    0x7979000000797979ULL, 0x1d1d0000001d1d1dULL, 0x5252000000525252ULL,
    0xebeb000000ebebebULL, 0xf3f3000000f3f3f3ULL, 0x6d6d0000006d6d6dULL,
    0x5e5e0000005e5e5eULL, 0xfbfb000000fbfbfbULL, 0x6969000000696969ULL,
    0xb2b2000000b2b2b2ULL, 0xf0f0000000f0f0f0ULL, 0x3131000000313131ULL,
    0x0c0c0000000c0c0cULL, 0xd4d4000000d4d4d4ULL, 0xcfcf000000cfcfcfULL,
    0x8c8c0000008c8c8cULL, 0xe2e2000000e2e2e2ULL, 0x7575000000757575ULL,
    0xa9a9000000a9a9a9ULL, 0x4a4a0000004a4a4aULL, 0x5757000000575757ULL,
    0x8484000000848484ULL, 0x1111000000111111ULL, 0x4545000000454545ULL,
    0x1b1b0000001b1b1bULL, 0xf5f5000000f5f5f5ULL, 0xe4e4000000e4e4e4ULL,
    0x0e0e0000000e0e0eULL, 0x7373000000737373ULL, 0xaaaa000000aaaaaaULL,
    0xf1f1000000f1f1f1ULL, 0xdddd000000ddddddULL, 0x5959000000595959ULL,
    0x1414000000141414ULL, 0x6c6c0000006c6c6cULL, 0x9292000000929292ULL,
    0x5454000000545454ULL, 0xd0d0000000d0d0d0ULL, 0x7878000000787878ULL,
    0x7070000000707070ULL, 0xe3e3000000e3e3e3ULL, 0x4949000000494949ULL,
    0x8080000000808080ULL, 0x5050000000505050ULL, 0xa7a7000000a7a7a7ULL,
    0xf6f6000000f6f6f6ULL, 0x7777000000777777ULL, 0x9393000000939393ULL,
    0x8686000000868686ULL, 0x8383000000838383ULL, 0x2a2a0000002a2a2aULL,
    0xc7c7000000c7c7c7ULL, 0x5b5b0000005b5b5bULL, 0xe9e9000000e9e9e9ULL,
    0xeeee000000eeeeeeULL, 0x8f8f0000008f8f8fULL, 0x0101000000010101ULL,
    0x3d3d0000003d3d3dULL,
};

const uint64 camellia_sp03303033[256] = {
    0x0038380038003838ULL, 0x0041410041004141ULL, 0x0016160016001616ULL,
    0x0076760076007676ULL, 0x00d9d900d900d9d9ULL, 0x0093930093009393ULL,
    0x0060600060006060ULL, 0x00f2f200f200f2f2ULL, 0x0072720072007272ULL,
    0x00c2c200c200c2c2ULL, 0x00abab00ab00ababULL, 0x009a9a009a009a9aULL,
    0x0075750075007575ULL, 0x0006060006000606ULL, 0x0057570057005757ULL,
    0x00a0a000a000a0a0ULL, 0x0091910091009191ULL, 0x00f7f700f700f7f7ULL,
    0x00b5b500b500b5b5ULL, 0x00c9c900c900c9c9ULL, 0x00a2a200a200a2a2ULL,
    0x008c8c008c008c8cULL, 0x00d2d200d200d2d2ULL, 0x0090900090009090ULL,
    0x00f6f600f600f6f6ULL, 0x0007070007000707ULL, 0x00a7a700a700a7a7ULL,
    0x0027270027002727ULL, 0x008e8e008e008e8eULL, 0x00b2b200b200b2b2ULL,
    0x0049490049004949ULL, 0x00dede00de00dedeULL, 0x0043430043004343ULL,
    0x005c5c005c005c5cULL, 0x00d7d700d700d7d7ULL, 0x00c7c700c700c7c7ULL,
    0x003e3e003e003e3eULL, 0x00f5f500f500f5f5ULL, 0x008f8f008f008f8fULL,
    0x0067670067006767ULL, 0x001f1f001f001f1fULL, 0x0018180018001818ULL,
    0x006e6e006e006e6eULL, 0x00afaf00af00afafULL, 0x002f2f002f002f2fULL,
    0x00e2e200e200e2e2ULL, 0x0085850085008585ULL, 0x000d0d000d000d0dULL,
    0x0053530053005353ULL, 0x00f0f000f000f0f0ULL, 0x009c9c009c009c9cULL,
    0x0065650065006565ULL, 0x00eaea00ea00eaeaULL, 0x00a3a300a300a3a3ULL,
    0x00aeae00ae00aeaeULL, 0x009e9e009e009e9eULL, 0x00ecec00ec00ececULL,
    0x0080800080008080ULL, 0x002d2d002d002d2dULL, 0x006b6b006b006b6bULL,
    0x00a8a800a800a8a8ULL, 0x002b2b002b002b2bULL, 0x0036360036003636ULL,
    0x00a6a600a600a6a6ULL, 0x00c5c500c500c5c5ULL, 0x0086860086008686ULL,
    0x004d4d004d004d4dULL, 0x0033330033003333ULL, 0x00fdfd00fd00fdfdULL,
    0x0066660066006666ULL, 0x0058580058005858ULL, 0x0096960096009696ULL,
    0x003a3a003a003a3aULL, 0x0009090009000909ULL, 0x0095950095009595ULL,
    0x0010100010001010ULL, 0x0078780078007878ULL, 0x00d8d800d800d8d8ULL,
    0x0042420042004242ULL, 0x00cccc00cc00ccccULL, 0x00efef00ef00efefULL,
    0x0026260026002626ULL, 0x00e5e500e500e5e5ULL, 0x0061610061006161ULL,
    0x001a1a001a001a1aULL, 0x003f3f003f003f3fULL, 0x003b3b003b003b3bULL,
    0x0082820082008282ULL, 0x00b6b600b600b6b6ULL, 0x00dbdb00db00dbdbULL,
    0x00d4d400d400d4d4ULL, 0x0098980098009898ULL, 0x00e8e800e800e8e8ULL,
    0x008b8b008b008b8bULL, 0x0002020002000202ULL, 0x00ebeb00eb00ebebULL,
    0x000a0a000a000a0aULL, 0x002c2c002c002c2cULL, 0x001d1d001d001d1dULL,
    0x00b0b000b000b0b0ULL, 0x006f6f006f006f6fULL, 0x008d8d008d008d8dULL,
    0x0088880088008888ULL, 0x000e0e000e000e0eULL, 0x0019190019001919ULL,
    0x0087870087008787ULL, 0x004e4e004e004e4eULL, 0x000b0b000b000b0bULL,
    0x00a9a900a900a9a9ULL, 0x000c0c000c000c0cULL, 0x0079790079007979ULL,
    0x0011110011001111ULL, 0x007f7f007f007f7fULL, 0x0022220022002222ULL,
    0x00e7e700e700e7e7ULL, 0x0059590059005959ULL, 0x00e1e100e100e1e1ULL,
    0x00dada00da00dadaULL, 0x003d3d003d003d3dULL, 0x00c8c800c800c8c8ULL,
    0x0012120012001212ULL, 0x0004040004000404ULL, 0x0074740074007474ULL,
    0x0054540054005454ULL, 0x0030300030003030ULL, 0x007e7e007e007e7eULL,
    0x00b4b400b400b4b4ULL, 0x0028280028002828ULL, 0x0055550055005555ULL,
    0x0068680068006868ULL, 0x0050500050005050ULL, 0x00bebe00be00bebeULL,
    0x00d0d000d000d0d0ULL, 0x00c4c400c400c4c4ULL, 0x0031310031003131ULL,
    0x00cbcb00cb00cbcbULL, 0x002a2a002a002a2aULL, 0x00adad00ad00adadULL,
    0x000f0f000f000f0fULL, 0x00caca00ca00cacaULL, 0x0070700070007070ULL,
    0x00ffff00ff00ffffULL, 0x0032320032003232ULL, 0x0069690069006969ULL,
    0x0008080008000808ULL, 0x0062620062006262ULL, 0x0000000000000000ULL,
    0x0024240024002424ULL, 0x00d1d100d100d1d1ULL, 0x00fbfb00fb00fbfbULL,
    0x00baba00ba00babaULL, 0x00eded00ed00ededULL, 0x0045450045004545ULL,
    0x0081810081008181ULL, 0x0073730073007373ULL, 0x006d6d006d006d6dULL,
    0x0084840084008484ULL, 0x009f9f009f009f9fULL, 0x00eeee00ee00eeeeULL,
    0x004a4a004a004a4aULL, 0x00c3c300c300c3c3ULL, 0x002e2e002e002e2eULL,
    0x00c1c100c100c1c1ULL, 0x0001010001000101ULL, 0x00e6e600e600e6e6ULL,
    0x0025250025002525ULL, 0x0048480048004848ULL, 0x0099990099009999ULL,
    0x00b9b900b900b9b9ULL, 0x00b3b300b300b3b3ULL, 0x007b7b007b007b7bULL,
    0x00f9f900f900f9f9ULL, 0x00cece00ce00ceceULL, 0x00bfbf00bf00bfbfULL,
    0x00dfdf00df00dfdfULL, 0x0071710071007171ULL, 0x0029290029002929ULL,
    0x00cdcd00cd00cdcdULL, 0x006c6c006c006c6cULL, 0x0013130013001313ULL,
    0x0064640064006464ULL, 0x009b9b009b009b9bULL, 0x0063630063006363ULL,
    0x009d9d009d009d9dULL, 0x00c0c000c000c0c0ULL, 0x004b4b004b004b4bULL,
    0x00b7b700b700b7b7ULL, 0x00a5a500a500a5a5ULL, 0x0089890089008989ULL,
    0x005f5f005f005f5fULL, 0x00b1b100b100b1b1ULL, 0x0017170017001717ULL,
    0x00f4f400f400f4f4ULL, 0x00bcbc00bc00bcbcULL, 0x00d3d300d300d3d3ULL,
    0x0046460046004646ULL, 0x00cfcf00cf00cfcfULL, 0x0037370037003737ULL,
    0x005e5e005e005e5eULL, 0x0047470047004747ULL, 0x0094940094009494ULL,
    0x00fafa00fa00fafaULL, 0x00fcfc00fc00fcfcULL, 0x005b5b005b005b5bULL,
    0x0097970097009797ULL, 0x00fefe00fe00fefeULL, 0x005a5a005a005a5aULL,
    0x00acac00ac00acacULL, 0x003c3c003c003c3cULL, 0x004c4c004c004c4cULL,
    0x0003030003000303ULL, 0x0035350035003535ULL, 0x00f3f300f300f3f3ULL,
    0x0023230023002323ULL, 0x00b8b800b800b8b8ULL, 0x005d5d005d005d5dULL,
    0x006a6a006a006a6aULL, 0x0092920092009292ULL, 0x00d5d500d500d5d5ULL,
    0x0021210021002121ULL, 0x0044440044004444ULL, 0x0051510051005151ULL,
    0x00c6c600c600c6c6ULL, 0x007d7d007d007d7dULL, 0x0039390039003939ULL,
    0x0083830083008383ULL, 0x00dcdc00dc00dcdcULL, 0x00aaaa00aa00aaaaULL,
    0x007c7c007c007c7cULL, 0x0077770077007777ULL, 0x0056560056005656ULL,
    0x0005050005000505ULL, 0x001b1b001b001b1bULL, 0x00a4a400a400a4a4ULL,
    0x0015150015001515ULL, 0x0034340034003434ULL, 0x001e1e001e001e1eULL,
    0x001c1c001c001c1cULL, 0x00f8f800f800f8f8ULL, 0x0052520052005252ULL,
    0x0020200020002020ULL, 0x0014140014001414ULL, 0x00e9e900e900e9e9ULL,
    0x00bdbd00bd00bdbdULL, 0x00dddd00dd00ddddULL, 0x00e4e400e400e4e4ULL,
    0x00a1a100a100a1a1ULL, 0x00e0e000e000e0e0ULL, 0x008a8a008a008a8aULL,
    0x00f1f100f100f1f1ULL, 0x00d6d600d600d6d6ULL, 0x007a7a007a007a7aULL,
    0x00bbbb00bb00bbbbULL, 0x00e3e300e300e3e3ULL, 0x0040400040004040ULL,
    0x004f4f004f004f4fULL,
};

const uint64 camellia_sp00444404[256] = {
    0x0000707070700070ULL, 0x00002c2c2c2c002cULL, 0x0000b3b3b3b300b3ULL,
    0x0000c0c0c0c000c0ULL, 0x0000e4e4e4e400e4ULL, 0x0000575757570057ULL,
    0x0000eaeaeaea00eaULL, 0x0000aeaeaeae00aeULL, 0x0000232323230023ULL,
    0x00006b6b6b6b006bULL, 0x0000454545450045ULL, 0x0000a5a5a5a500a5ULL,
    0x0000edededed00edULL, 0x00004f4f4f4f004fULL, 0x00001d1d1d1d001dULL,
    0x0000929292920092ULL, 0x0000868686860086ULL, 0x0000afafafaf00afULL,
    0x00007c7c7c7c007cULL, 0x00001f1f1f1f001fULL, 0x00003e3e3e3e003eULL,
    0x0000dcdcdcdc00dcULL, 0x00005e5e5e5e005eULL, 0x00000b0b0b0b000bULL,
    0x0000a6a6a6a600a6ULL, 0x0000393939390039ULL, 0x0000d5d5d5d500d5ULL,
    0x00005d5d5d5d005dULL, 0x0000d9d9d9d900d9ULL, 0x00005a5a5a5a005aULL,
    0x0000515151510051ULL, 0x00006c6c6c6c006cULL, 0x00008b8b8b8b008bULL,
    0x00009a9a9a9a009aULL, 0x0000fbfbfbfb00fbULL, 0x0000b0b0b0b000b0ULL,
    0x0000747474740074ULL, 0x00002b2b2b2b002bULL, 0x0000f0f0f0f000f0ULL,
    0x0000848484840084ULL, 0x0000dfdfdfdf00dfULL, 0x0000cbcbcbcb00cbULL,
    0x0000343434340034ULL, 0x0000767676760076ULL, 0x00006d6d6d6d006dULL,
    0x0000a9a9a9a900a9ULL, 0x0000d1d1d1d100d1ULL, 0x0000040404040004ULL,
    0x0000141414140014ULL, 0x00003a3a3a3a003aULL, 0x0000dededede00deULL,
    0x0000111111110011ULL, 0x0000323232320032ULL, 0x00009c9c9c9c009cULL,
    0x0000535353530053ULL, 0x0000f2f2f2f200f2ULL, 0x0000fefefefe00feULL,
    0x0000cfcfcfcf00cfULL, 0x0000c3c3c3c300c3ULL, 0x00007a7a7a7a007aULL,
    0x0000242424240024ULL, 0x0000e8e8e8e800e8ULL, 0x0000606060600060ULL,
    0x0000696969690069ULL, 0x0000aaaaaaaa00aaULL, 0x0000a0a0a0a000a0ULL,
    0x0000a1a1a1a100a1ULL, 0x0000626262620062ULL, 0x0000545454540054ULL,
    0x00001e1e1e1e001eULL, 0x0000e0e0e0e000e0ULL, 0x0000646464640064ULL,
    0x0000101010100010ULL, 0x0000000000000000ULL, 0x0000a3a3a3a300a3ULL,
    0x0000757575750075ULL, 0x00008a8a8a8a008aULL, 0x0000e6e6e6e600e6ULL,
    0x0000090909090009ULL, 0x0000dddddddd00ddULL, 0x0000878787870087ULL,
    0x0000838383830083ULL, 0x0000cdcdcdcd00cdULL, 0x0000909090900090ULL,
    0x0000737373730073ULL, 0x0000f6f6f6f600f6ULL, 0x00009d9d9d9d009dULL,
    0x0000bfbfbfbf00bfULL, 0x0000525252520052ULL, 0x0000d8d8d8d800d8ULL,
    0x0000c8c8c8c800c8ULL, 0x0000c6c6c6c600c6ULL, 0x0000818181810081ULL,
    0x00006f6f6f6f006fULL, 0x0000131313130013ULL, 0x0000636363630063ULL,
    0x0000e9e9e9e900e9ULL, 0x0000a7a7a7a700a7ULL, 0x00009f9f9f9f009fULL,
    0x0000bcbcbcbc00bcULL, 0x0000292929290029ULL, 0x0000f9f9f9f900f9ULL,
    0x00002f2f2f2f002fULL, 0x0000b4b4b4b400b4ULL, 0x0000787878780078ULL,
    0x0000060606060006ULL, 0x0000e7e7e7e700e7ULL, 0x0000717171710071ULL,
    0x0000d4d4d4d400d4ULL, 0x0000abababab00abULL, 0x0000888888880088ULL,
    0x00008d8d8d8d008dULL, 0x0000727272720072ULL, 0x0000b9b9b9b900b9ULL,
    0x0000f8f8f8f800f8ULL, 0x0000acacacac00acULL, 0x0000363636360036ULL,
    0x00002a2a2a2a002aULL, 0x00003c3c3c3c003cULL, 0x0000f1f1f1f100f1ULL,
    0x0000404040400040ULL, 0x0000d3d3d3d300d3ULL, 0x0000bbbbbbbb00bbULL,
    0x0000434343430043ULL, 0x0000151515150015ULL, 0x0000adadadad00adULL,
    0x0000777777770077ULL, 0x0000808080800080ULL, 0x0000828282820082ULL,
    0x0000ecececec00ecULL, 0x0000272727270027ULL, 0x0000e5e5e5e500e5ULL,
    0x0000858585850085ULL, 0x0000353535350035ULL, 0x00000c0c0c0c000cULL,
    0x0000414141410041ULL, 0x0000efefefef00efULL, 0x0000939393930093ULL,
    0x0000191919190019ULL, 0x0000212121210021ULL, 0x00000e0e0e0e000eULL,
    0x00004e4e4e4e004eULL, 0x0000656565650065ULL, 0x0000bdbdbdbd00bdULL,
    0x0000b8b8b8b800b8ULL, 0x00008f8f8f8f008fULL, 0x0000ebebebeb00ebULL,
    0x0000cececece00ceULL, 0x0000303030300030ULL, 0x00005f5f5f5f005fULL,
    0x0000c5c5c5c500c5ULL, 0x00001a1a1a1a001aULL, 0x0000e1e1e1e100e1ULL,
    0x0000cacacaca00caULL, 0x0000474747470047ULL, 0x00003d3d3d3d003dULL,
    0x0000010101010001ULL, 0x0000d6d6d6d600d6ULL, 0x0000565656560056ULL,
    0x00004d4d4d4d004dULL, 0x00000d0d0d0d000dULL, 0x0000666666660066ULL,
    0x0000cccccccc00ccULL, 0x00002d2d2d2d002dULL, 0x0000121212120012ULL,
    0x0000202020200020ULL, 0x0000b1b1b1b100b1ULL, 0x0000999999990099ULL,
    0x00004c4c4c4c004cULL, 0x0000c2c2c2c200c2ULL, 0x00007e7e7e7e007eULL,
    0x0000050505050005ULL, 0x0000b7b7b7b700b7ULL, 0x0000313131310031ULL,
    0x0000171717170017ULL, 0x0000d7d7d7d700d7ULL, 0x0000585858580058ULL,
    0x0000616161610061ULL, 0x00001b1b1b1b001bULL, 0x00001c1c1c1c001cULL,
    0x00000f0f0f0f000fULL, 0x0000161616160016ULL, 0x0000181818180018ULL,
    0x0000222222220022ULL, 0x0000444444440044ULL, 0x0000b2b2b2b200b2ULL,
    0x0000b5b5b5b500b5ULL, 0x0000919191910091ULL, 0x0000080808080008ULL,
    0x0000a8a8a8a800a8ULL, 0x0000fcfcfcfc00fcULL, 0x0000505050500050ULL,
    0x0000d0d0d0d000d0ULL, 0x00007d7d7d7d007dULL, 0x0000898989890089ULL,
    0x0000979797970097ULL, 0x00005b5b5b5b005bULL, 0x0000959595950095ULL,
    0x0000ffffffff00ffULL, 0x0000d2d2d2d200d2ULL, 0x0000c4c4c4c400c4ULL,
    0x0000484848480048ULL, 0x0000f7f7f7f700f7ULL, 0x0000dbdbdbdb00dbULL,
    0x0000030303030003ULL, 0x0000dadadada00daULL, 0x00003f3f3f3f003fULL,
    0x0000949494940094ULL, 0x00005c5c5c5c005cULL, 0x0000020202020002ULL,
    0x00004a4a4a4a004aULL, 0x0000333333330033ULL, 0x0000676767670067ULL,
    0x0000f3f3f3f300f3ULL, 0x00007f7f7f7f007fULL, 0x0000e2e2e2e200e2ULL,
    0x00009b9b9b9b009bULL, 0x0000262626260026ULL, 0x0000373737370037ULL,
    0x00003b3b3b3b003bULL, 0x0000969696960096ULL, 0x00004b4b4b4b004bULL,
    0x0000bebebebe00beULL, 0x00002e2e2e2e002eULL, 0x0000797979790079ULL,
    0x00008c8c8c8c008cULL, 0x00006e6e6e6e006eULL, 0x00008e8e8e8e008eULL,
    0x0000f5f5f5f500f5ULL, 0x0000b6b6b6b600b6ULL, 0x0000fdfdfdfd00fdULL,
    0x0000595959590059ULL, 0x0000989898980098ULL, 0x00006a6a6a6a006aULL,
    0x0000464646460046ULL, 0x0000babababa00baULL, 0x0000252525250025ULL,
    0x0000424242420042ULL, 0x0000a2a2a2a200a2ULL, 0x0000fafafafa00faULL,
    0x0000070707070007ULL, 0x0000555555550055ULL, 0x0000eeeeeeee00eeULL,
    0x00000a0a0a0a000aULL, 0x0000494949490049ULL, 0x0000686868680068ULL,
    0x0000383838380038ULL, 0x0000a4a4a4a400a4ULL, 0x0000282828280028ULL,
    0x00007b7b7b7b007bULL, 0x0000c9c9c9c900c9ULL, 0x0000c1c1c1c100c1ULL,
    0x0000e3e3e3e300e3ULL, 0x0000f4f4f4f400f4ULL, 0x0000c7c7c7c700c7ULL,
    0x00009e9e9e9e009eULL,
};

const uint64 camellia_sp02220222[256] = {
    0x00e0e0e000e0e0e0ULL, 0x0005050500050505ULL, 0x0058585800585858ULL,
    0x00d9d9d900d9d9d9ULL, 0x0067676700676767ULL, 0x004e4e4e004e4e4eULL,
    0x0081818100818181ULL, 0x00cbcbcb00cbcbcbULL, 0x00c9c9c900c9c9c9ULL,
    0x000b0b0b000b0b0bULL, 0x00aeaeae00aeaeaeULL, 0x006a6a6a006a6a6aULL,
    0x00d5d5d500d5d5d5ULL, 0x0018181800181818ULL, 0x005d5d5d005d5d5dULL,
    0x0082828200828282ULL, 0x0046464600464646ULL, 0x00dfdfdf00dfdfdfULL,
    0x00d6d6d600d6d6d6ULL, 0x0027272700272727ULL, 0x008a8a8a008a8a8aULL,
    0x0032323200323232ULL, 0x004b4b4b004b4b4bULL, 0x0042424200424242ULL,
    0x00dbdbdb00dbdbdbULL, 0x001c1c1c001c1c1cULL, 0x009e9e9e009e9e9eULL,
    0x009c9c9c009c9c9cULL, 0x003a3a3a003a3a3aULL, 0x00cacaca00cacacaULL,
    0x0025252500252525ULL, 0x007b7b7b007b7b7bULL, 0x000d0d0d000d0d0dULL,
    0x0071717100717171ULL, 0x005f5f5f005f5f5fULL, 0x001f1f1f001f1f1fULL,
    0x00f8f8f800f8f8f8ULL, 0x00d7d7d700d7d7d7ULL, 0x003e3e3e003e3e3eULL,
    0x009d9d9d009d9d9dULL, 0x007c7c7c007c7c7cULL, 0x0060606000606060ULL,
    0x00b9b9b900b9b9b9ULL, 0x00bebebe00bebebeULL, 0x00bcbcbc00bcbcbcULL,
    0x008b8b8b008b8b8bULL, 0x0016161600161616ULL, 0x0034343400343434ULL,
    0x004d4d4d004d4d4dULL, 0x00c3c3c300c3c3c3ULL, 0x0072727200727272ULL,
    0x0095959500959595ULL, 0x00ababab00abababULL, 0x008e8e8e008e8e8eULL,
    0x00bababa00bababaULL, 0x007a7a7a007a7a7aULL, 0x00b3b3b300b3b3b3ULL,
    0x0002020200020202ULL, 0x00b4b4b400b4b4b4ULL, 0x00adadad00adadadULL,
    0x00a2a2a200a2a2a2ULL, 0x00acacac00acacacULL, 0x00d8d8d800d8d8d8ULL,
    0x009a9a9a009a9a9aULL, 0x0017171700171717ULL, 0x001a1a1a001a1a1aULL,
    0x0035353500353535ULL, 0x00cccccc00ccccccULL, 0x00f7f7f700f7f7f7ULL,
    0x0099999900999999ULL, 0x0061616100616161ULL, 0x005a5a5a005a5a5aULL,
    0x00e8e8e800e8e8e8ULL, 0x0024242400242424ULL, 0x0056565600565656ULL,
    0x0040404000404040ULL, 0x00e1e1e100e1e1e1ULL, 0x0063636300636363ULL,
    0x0009090900090909ULL, 0x0033333300333333ULL, 0x00bfbfbf00bfbfbfULL,
    0x0098989800989898ULL, 0x0097979700979797ULL, 0x0085858500858585ULL,
    0x0068686800686868ULL, 0x00fcfcfc00fcfcfcULL, 0x00ececec00ecececULL,
    0x000a0a0a000a0a0aULL, 0x00dadada00dadadaULL, 0x006f6f6f006f6f6fULL,
    0x0053535300535353ULL, 0x0062626200626262ULL, 0x00a3a3a300a3a3a3ULL,
    0x002e2e2e002e2e2eULL, 0x0008080800080808ULL, 0x00afafaf00afafafULL,
    0x0028282800282828ULL, 0x00b0b0b000b0b0b0ULL, 0x0074747400747474ULL,
    0x00c2c2c200c2c2c2ULL, 0x00bdbdbd00bdbdbdULL, 0x0036363600363636ULL,
    0x0022222200222222ULL, 0x0038383800383838ULL, 0x0064646400646464ULL,
    0x001e1e1e001e1e1eULL, 0x0039393900393939ULL, 0x002c2c2c002c2c2cULL,
    0x00a6a6a600a6a6a6ULL, 0x0030303000303030ULL, 0x00e5e5e500e5e5e5ULL,
    0x0044444400444444ULL, 0x00fdfdfd00fdfdfdULL, 0x0088888800888888ULL,
    0x009f9f9f009f9f9fULL, 0x0065656500656565ULL, 0x0087878700878787ULL,
    0x006b6b6b006b6b6bULL, 0x00f4f4f400f4f4f4ULL, 0x0023232300232323ULL,
    0x0048484800484848ULL, 0x0010101000101010ULL, 0x00d1d1d100d1d1d1ULL,
    0x0051515100515151ULL, 0x00c0c0c000c0c0c0ULL, 0x00f9f9f900f9f9f9ULL,
    0x00d2d2d200d2d2d2ULL, 0x00a0a0a000a0a0a0ULL, 0x0055555500555555ULL,
    0x00a1a1a100a1a1a1ULL, 0x0041414100414141ULL, 0x00fafafa00fafafaULL,
    0x0043434300434343ULL, 0x0013131300131313ULL, 0x00c4c4c400c4c4c4ULL,
    0x002f2f2f002f2f2fULL, 0x00a8a8a800a8a8a8ULL, 0x00b6b6b600b6b6b6ULL,
    0x003c3c3c003c3c3cULL, 0x002b2b2b002b2b2bULL, 0x00c1c1c100c1c1c1ULL,
    0x00ffffff00ffffffULL, 0x00c8c8c800c8c8c8ULL, 0x00a5a5a500a5a5a5ULL,
    0x0020202000202020ULL, 0x0089898900898989ULL, 0x0000000000000000ULL,
    0x0090909000909090ULL, 0x0047474700474747ULL, 0x00efefef00efefefULL,
    0x00eaeaea00eaeaeaULL, 0x00b7b7b700b7b7b7ULL, 0x0015151500151515ULL,
    0x0006060600060606ULL, 0x00cdcdcd00cdcdcdULL, 0x00b5b5b500b5b5b5ULL,
    0x0012121200121212ULL, 0x007e7e7e007e7e7eULL, 0x00bbbbbb00bbbbbbULL,
    0x0029292900292929ULL, 0x000f0f0f000f0f0fULL, 0x00b8b8b800b8b8b8ULL,
    0x0007070700070707ULL, 0x0004040400040404ULL, 0x009b9b9b009b9b9bULL,
    0x0094949400949494ULL, 0x0021212100212121ULL, 0x0066666600666666ULL,
    0x00e6e6e600e6e6e6ULL, 0x00cecece00cececeULL, 0x00ededed00edededULL,
    0x00e7e7e700e7e7e7ULL, 0x003b3b3b003b3b3bULL, 0x00fefefe00fefefeULL,
    0x007f7f7f007f7f7fULL, 0x00c5c5c500c5c5c5ULL, 0x00a4a4a400a4a4a4ULL,
    0x0037373700373737ULL, 0x00b1b1b100b1b1b1ULL, 0x004c4c4c004c4c4cULL,
    0x0091919100919191ULL, 0x006e6e6e006e6e6eULL, 0x008d8d8d008d8d8dULL,
    0x0076767600767676ULL, 0x0003030300030303ULL, 0x002d2d2d002d2d2dULL,
    0x00dedede00dededeULL, 0x0096969600969696ULL, 0x0026262600262626ULL,
    0x007d7d7d007d7d7dULL, 0x00c6c6c600c6c6c6ULL, 0x005c5c5c005c5c5cULL,
    0x00d3d3d300d3d3d3ULL, 0x00f2f2f200f2f2f2ULL, 0x004f4f4f004f4f4fULL,
    0x0019191900191919ULL, 0x003f3f3f003f3f3fULL, 0x00dcdcdc00dcdcdcULL,
    0x0079797900797979ULL, 0x001d1d1d001d1d1dULL, 0x0052525200525252ULL,
    0x00ebebeb00ebebebULL, 0x00f3f3f300f3f3f3ULL, 0x006d6d6d006d6d6dULL,
    0x005e5e5e005e5e5eULL, 0x00fbfbfb00fbfbfbULL, 0x0069696900696969ULL,
    0x00b2b2b200b2b2b2ULL, 0x00f0f0f000f0f0f0ULL, 0x0031313100313131ULL,
    0x000c0c0c000c0c0cULL, 0x00d4d4d400d4d4d4ULL, 0x00cfcfcf00cfcfcfULL,
    0x008c8c8c008c8c8cULL, 0x00e2e2e200e2e2e2ULL, 0x0075757500757575ULL,
    0x00a9a9a900a9a9a9ULL, 0x004a4a4a004a4a4aULL, 0x0057575700575757ULL,
    0x0084848400848484ULL, 0x0011111100111111ULL, 0x0045454500454545ULL,
    0x001b1b1b001b1b1bULL, 0x00f5f5f500f5f5f5ULL, 0x00e4e4e400e4e4e4ULL,
    0x000e0e0e000e0e0eULL, 0x0073737300737373ULL, 0x00aaaaaa00aaaaaaULL,
    0x00f1f1f100f1f1f1ULL, 0x00dddddd00ddddddULL, 0x0059595900595959ULL,
    0x0014141400141414ULL, 0x006c6c6c006c6c6cULL, 0x0092929200929292ULL,
    0x0054545400545454ULL, 0x00d0d0d000d0d0d0ULL, 0x0078787800787878ULL,
    0x0070707000707070ULL, 0x00e3e3e300e3e3e3ULL, 0x0049494900494949ULL,
    0x0080808000808080ULL, 0x0050505000505050ULL, 0x00a7a7a700a7a7a7ULL,
    0x00f6f6f600f6f6f6ULL, 0x0077777700777777ULL, 0x0093939300939393ULL,
    0x0086868600868686ULL, 0x0083838300838383ULL, 0x002a2a2a002a2a2aULL,
    0x00c7c7c700c7c7c7ULL, 0x005b5b5b005b5b5bULL, 0x00e9e9e900e9e9e9ULL,
    0x00eeeeee00eeeeeeULL, 0x008f8f8f008f8f8fULL, 0x0001010100010101ULL,
    0x003d3d3d003d3d3dULL,
};

const uint64 camellia_sp30333033[256] = {
    0x3800383838003838ULL, 0x4100414141004141ULL, 0x1600161616001616ULL,
    0x7600767676007676ULL, 0xd900d9d9d900d9d9ULL, 0x9300939393009393ULL,
    0x6000606060006060ULL, 0xf200f2f2f200f2f2ULL, 0x7200727272007272ULL,
    0xc200c2c2c200c2c2ULL, 0xab00ababab00ababULL, 0x9a009a9a9a009a9aULL,
    0x7500757575007575ULL, 0x0600060606000606ULL, 0x5700575757005757ULL,
    0xa000a0a0a000a0a0ULL, 0x9100919191009191ULL, 0xf700f7f7f700f7f7ULL,
    0xb500b5b5b500b5b5ULL, 0xc900c9c9c900c9c9ULL, 0xa200a2a2a200a2a2ULL,
    0x8c008c8c8c008c8cULL, 0xd200d2d2d200d2d2ULL, 0x9000909090009090ULL,
    0xf600f6f6f600f6f6ULL, 0x0700070707000707ULL, 0xa700a7a7a700a7a7ULL,
    0x2700272727002727ULL, 0x8e008e8e8e008e8eULL, 0xb200b2b2b200b2b2ULL,
    0x4900494949004949ULL, 0xde00dedede00dedeULL, 0x4300434343004343ULL,
    0x5c005c5c5c005c5cULL, 0xd700d7d7d700d7d7ULL, 0xc700c7c7c700c7c7ULL,
    0x3e003e3e3e003e3eULL, 0xf500f5f5f500f5f5ULL, 0x8f008f8f8f008f8fULL,
    0x6700676767006767ULL, 0x1f001f1f1f001f1fULL, 0x1800181818001818ULL,
    0x6e006e6e6e006e6eULL, 0xaf00afafaf00afafULL, 0x2f002f2f2f002f2fULL,
    0xe200e2e2e200e2e2ULL, 0x8500858585008585ULL, 0x0d000d0d0d000d0dULL,
    0x5300535353005353ULL, 0xf000f0f0f000f0f0ULL, 0x9c009c9c9c009c9cULL,
    0x6500656565006565ULL, 0xea00eaeaea00eaeaULL, 0xa300a3a3a300a3a3ULL,
    0xae00aeaeae00aeaeULL, 0x9e009e9e9e009e9eULL, 0xec00ececec00ececULL,
    0x8000808080008080ULL, 0x2d002d2d2d002d2dULL, 0x6b006b6b6b006b6bULL,
    0xa800a8a8a800a8a8ULL, 0x2b002b2b2b002b2bULL, 0x3600363636003636ULL,
    0xa600a6a6a600a6a6ULL, 0xc500c5c5c500c5c5ULL, 0x8600868686008686ULL,
    0x4d004d4d4d004d4dULL, 0x3300333333003333ULL, 0xfd00fdfdfd00fdfdULL,
    0x6600666666006666ULL, 0x5800585858005858ULL, 0x9600969696009696ULL,
    0x3a003a3a3a003a3aULL, 0x0900090909000909ULL, 0x9500959595009595ULL,
    0x1000101010001010ULL, 0x7800787878007878ULL, 0xd800d8d8d800d8d8ULL,
    0x4200424242004242ULL, 0xcc00cccccc00ccccULL, 0xef00efefef00efefULL,
    0x2600262626002626ULL, 0xe500e5e5e500e5e5ULL, 0x6100616161006161ULL,
    0x1a001a1a1a001a1aULL, 0x3f003f3f3f003f3fULL, 0x3b003b3b3b003b3bULL,
    0x8200828282008282ULL, 0xb600b6b6b600b6b6ULL, 0xdb00dbdbdb00dbdbULL,
    0xd400d4d4d400d4d4ULL, 0x9800989898009898ULL, 0xe800e8e8e800e8e8ULL,
    0x8b008b8b8b008b8bULL, 0x0200020202000202ULL, 0xeb00ebebeb00ebebULL,
    0x0a000a0a0a000a0aULL, 0x2c002c2c2c002c2cULL, 0x1d001d1d1d001d1dULL,
    0xb000b0b0b000b0b0ULL, 0x6f006f6f6f006f6fULL, 0x8d008d8d8d008d8dULL,
    0x8800888888008888ULL, 0x0e000e0e0e000e0eULL, 0x1900191919001919ULL,
    0x8700878787008787ULL, 0x4e004e4e4e004e4eULL, 0x0b000b0b0b000b0bULL,
    0xa900a9a9a900a9a9ULL, 0x0c000c0c0c000c0cULL, 0x7900797979007979ULL,
    0x1100111111001111ULL, 0x7f007f7f7f007f7fULL, 0x2200222222002222ULL,
    0xe700e7e7e700e7e7ULL, 0x5900595959005959ULL, 0xe100e1e1e100e1e1ULL,
    0xda00dadada00dadaULL, 0x3d003d3d3d003d3dULL, 0xc800c8c8c800c8c8ULL,
    0x1200121212001212ULL, 0x0400040404000404ULL, 0x7400747474007474ULL,
    0x5400545454005454ULL, 0x3000303030003030ULL, 0x7e007e7e7e007e7eULL,
    0xb400b4b4b400b4b4ULL, 0x2800282828002828ULL, 0x5500555555005555ULL,
    0x6800686868006868ULL, 0x5000505050005050ULL, 0xbe00bebebe00bebeULL,
    0xd000d0d0d000d0d0ULL, 0xc400c4c4c400c4c4ULL, 0x3100313131003131ULL,
    0xcb00cbcbcb00cbcbULL, 0x2a002a2a2a002a2aULL, 0xad00adadad00adadULL,
    0x0f000f0f0f000f0fULL, 0xca00cacaca00cacaULL, 0x7000707070007070ULL,
    0xff00ffffff00ffffULL, 0x3200323232003232ULL, 0x6900696969006969ULL,
    0x0800080808000808ULL, 0x6200626262006262ULL, 0x0000000000000000ULL,
    0x2400242424002424ULL, 0xd100d1d1d100d1d1ULL, 0xfb00fbfbfb00fbfbULL,
    0xba00bababa00babaULL, 0xed00ededed00ededULL, 0x4500454545004545ULL,
    0x8100818181008181ULL, 0x7300737373007373ULL, 0x6d006d6d6d006d6dULL,
    0x8400848484008484ULL, 0x9f009f9f9f009f9fULL, 0xee00eeeeee00eeeeULL,
    0x4a004a4a4a004a4aULL, 0xc300c3c3c300c3c3ULL, 0x2e002e2e2e002e2eULL,
    0xc100c1c1c100c1c1ULL, 0x0100010101000101ULL, 0xe600e6e6e600e6e6ULL,
    0x2500252525002525ULL, 0x4800484848004848ULL, 0x9900999999009999ULL,
    0xb900b9b9b900b9b9ULL, 0xb300b3b3b300b3b3ULL, 0x7b007b7b7b007b7bULL,
    0xf900f9f9f900f9f9ULL, 0xce00cecece00ceceULL, 0xbf00bfbfbf00bfbfULL,
    0xdf00dfdfdf00dfdfULL, 0x7100717171007171ULL, 0x2900292929002929ULL,
    0xcd00cdcdcd00cdcdULL, 0x6c006c6c6c006c6cULL, 0x1300131313001313ULL,
    0x6400646464006464ULL, 0x9b009b9b9b009b9bULL, 0x6300636363006363ULL,
    0x9d009d9d9d009d9dULL, 0xc000c0c0c000c0c0ULL, 0x4b004b4b4b004b4bULL,
    0xb700b7b7b700b7b7ULL, 0xa500a5a5a500a5a5ULL, 0x8900898989008989ULL,
    0x5f005f5f5f005f5fULL, 0xb100b1b1b100b1b1ULL, 0x1700171717001717ULL,
    0xf400f4f4f400f4f4ULL, 0xbc00bcbcbc00bcbcULL, 0xd300d3d3d300d3d3ULL,
    0x4600464646004646ULL, 0xcf00cfcfcf00cfcfULL, 0x3700373737003737ULL,
    0x5e005e5e5e005e5eULL, 0x4700474747004747ULL, 0x9400949494009494ULL,
    0xfa00fafafa00fafaULL, 0xfc00fcfcfc00fcfcULL, 0x5b005b5b5b005b5bULL,
    0x9700979797009797ULL, 0xfe00fefefe00fefeULL, 0x5a005a5a5a005a5aULL,
    0xac00acacac00acacULL, 0x3c003c3c3c003c3cULL, 0x4c004c4c4c004c4cULL,
    0x0300030303000303ULL, 0x3500353535003535ULL, 0xf300f3f3f300f3f3ULL,
    0x2300232323002323ULL, 0xb800b8b8b800b8b8ULL, 0x5d005d5d5d005d5dULL,
    0x6a006a6a6a006a6aULL, 0x9200929292009292ULL, 0xd500d5d5d500d5d5ULL,
    0x2100212121002121ULL, 0x4400444444004444ULL, 0x5100515151005151ULL,
    0xc600c6c6c600c6c6ULL, 0x7d007d7d7d007d7dULL, 0x3900393939003939ULL,
    0x8300838383008383ULL, 0xdc00dcdcdc00dcdcULL, 0xaa00aaaaaa00aaaaULL,
    0x7c007c7c7c007c7cULL, 0x7700777777007777ULL, 0x5600565656005656ULL,
    0x0500050505000505ULL, 0x1b001b1b1b001b1bULL, 0xa400a4a4a400a4a4ULL,
    0x1500151515001515ULL, 0x3400343434003434ULL, 0x1e001e1e1e001e1eULL,
    0x1c001c1c1c001c1cULL, 0xf800f8f8f800f8f8ULL, 0x5200525252005252ULL,
    0x2000202020002020ULL, 0x1400141414001414ULL, 0xe900e9e9e900e9e9ULL,
    0xbd00bdbdbd00bdbdULL, 0xdd00dddddd00ddddULL, 0xe400e4e4e400e4e4ULL,
    0xa100a1a1a100a1a1ULL, 0xe000e0e0e000e0e0ULL, 0x8a008a8a8a008a8aULL,
    0xf100f1f1f100f1f1ULL, 0xd600d6d6d600d6d6ULL, 0x7a007a7a7a007a7aULL,
    0xbb00bbbbbb00bbbbULL, 0xe300e3e3e300e3e3ULL, 0x4000404040004040ULL,
    0x4f004f4f4f004f4fULL,
};

const uint64 camellia_sp44044404[256] = {
    0x7070007070700070ULL, 0x2c2c002c2c2c002cULL, 0xb3b300b3b3b300b3ULL,
    0xc0c000c0c0c000c0ULL, 0xe4e400e4e4e400e4ULL, 0x5757005757570057ULL,
    0xeaea00eaeaea00eaULL, 0xaeae00aeaeae00aeULL, 0x2323002323230023ULL,
    0x6b6b006b6b6b006bULL, 0x4545004545450045ULL, 0xa5a500a5a5a500a5ULL,
    0xeded00ededed00edULL, 0x4f4f004f4f4f004fULL, 0x1d1d001d1d1d001dULL,
    0x9292009292920092ULL, 0x8686008686860086ULL, 0xafaf00afafaf00afULL,
    0x7c7c007c7c7c007cULL, 0x1f1f001f1f1f001fULL, 0x3e3e003e3e3e003eULL,
    0xdcdc00dcdcdc00dcULL, 0x5e5e005e5e5e005eULL, 0x0b0b000b0b0b000bULL,
    0xa6a600a6a6a600a6ULL, 0x3939003939390039ULL, 0xd5d500d5d5d500d5ULL,
    0x5d5d005d5d5d005dULL, 0xd9d900d9d9d900d9ULL, 0x5a5a005a5a5a005aULL,
    0x5151005151510051ULL, 0x6c6c006c6c6c006cULL, 0x8b8b008b8b8b008bULL,
    0x9a9a009a9a9a009aULL, 0xfbfb00fbfbfb00fbULL, 0xb0b000b0b0b000b0ULL,
    0x7474007474740074ULL, 0x2b2b002b2b2b002bULL, 0xf0f000f0f0f000f0ULL,
    0x8484008484840084ULL, 0xdfdf00dfdfdf00dfULL, 0xcbcb00cbcbcb00cbULL,
    0x3434003434340034ULL, 0x7676007676760076ULL, 0x6d6d006d6d6d006dULL,
    0xa9a900a9a9a900a9ULL, 0xd1d100d1d1d100d1ULL, 0x0404000404040004ULL,
    0x1414001414140014ULL, 0x3a3a003a3a3a003aULL, 0xdede00dedede00deULL,
    0x1111001111110011ULL, 0x3232003232320032ULL, 0x9c9c009c9c9c009cULL,
    0x5353005353530053ULL, 0xf2f200f2f2f200f2ULL, 0xfefe00fefefe00feULL,
    0xcfcf00cfcfcf00cfULL, 0xc3c300c3c3c300c3ULL, 0x7a7a007a7a7a007aULL,
    0x2424002424240024ULL, 0xe8e800e8e8e800e8ULL, 0x6060006060600060ULL,
    0x6969006969690069ULL, 0xaaaa00aaaaaa00aaULL, 0xa0a000a0a0a000a0ULL,
    0xa1a100a1a1a100a1ULL, 0x6262006262620062ULL, 0x5454005454540054ULL,
    0x1e1e001e1e1e001eULL, 0xe0e000e0e0e000e0ULL, 0x6464006464640064ULL,
    0x1010001010100010ULL, 0x0000000000000000ULL, 0xa3a300a3a3a300a3ULL,
    0x7575007575750075ULL, 0x8a8a008a8a8a008aULL, 0xe6e600e6e6e600e6ULL,
    0x0909000909090009ULL, 0xdddd00dddddd00ddULL, 0x8787008787870087ULL,
    0x8383008383830083ULL, 0xcdcd00cdcdcd00cdULL, 0x9090009090900090ULL,
    0x7373007373730073ULL, 0xf6f600f6f6f600f6ULL, 0x9d9d009d9d9d009dULL,
    0xbfbf00bfbfbf00bfULL, 0x5252005252520052ULL, 0xd8d800d8d8d800d8ULL,
    0xc8c800c8c8c800c8ULL, 0xc6c600c6c6c600c6ULL, 0x8181008181810081ULL,
    0x6f6f006f6f6f006fULL, 0x1313001313130013ULL, 0x6363006363630063ULL,
    0xe9e900e9e9e900e9ULL, 0xa7a700a7a7a700a7ULL, 0x9f9f009f9f9f009fULL,
    0xbcbc00bcbcbc00bcULL, 0x2929002929290029ULL, 0xf9f900f9f9f900f9ULL,
    0x2f2f002f2f2f002fULL, 0xb4b400b4b4b400b4ULL, 0x7878007878780078ULL,
    0x0606000606060006ULL, 0xe7e700e7e7e700e7ULL, 0x7171007171710071ULL,
    0xd4d400d4d4d400d4ULL, 0xabab00ababab00abULL, 0x8888008888880088ULL,
    0x8d8d008d8d8d008dULL, 0x7272007272720072ULL, 0xb9b900b9b9b900b9ULL,
    0xf8f800f8f8f800f8ULL, 0xacac00acacac00acULL, 0x3636003636360036ULL,
    0x2a2a002a2a2a002aULL, 0x3c3c003c3c3c003cULL, 0xf1f100f1f1f100f1ULL,
    0x4040004040400040ULL, 0xd3d300d3d3d300d3ULL, 0xbbbb00bbbbbb00bbULL,
    0x4343004343430043ULL, 0x1515001515150015ULL, 0xadad00adadad00adULL,
    0x7777007777770077ULL, 0x8080008080800080ULL, 0x8282008282820082ULL,
    0xecec00ececec00ecULL, 0x2727002727270027ULL, 0xe5e500e5e5e500e5ULL,
    0x8585008585850085ULL, 0x3535003535350035ULL, 0x0c0c000c0c0c000cULL,
    0x4141004141410041ULL, 0xefef00efefef00efULL, 0x9393009393930093ULL,
    0x1919001919190019ULL, 0x2121002121210021ULL, 0x0e0e000e0e0e000eULL,
    0x4e4e004e4e4e004eULL, 0x6565006565650065ULL, 0xbdbd00bdbdbd00bdULL,
    0xb8b800b8b8b800b8ULL, 0x8f8f008f8f8f008fULL, 0xebeb00ebebeb00ebULL,
    0xcece00cecece00ceULL, 0x3030003030300030ULL, 0x5f5f005f5f5f005fULL,
    0xc5c500c5c5c500c5ULL, 0x1a1a001a1a1a001aULL, 0xe1e100e1e1e100e1ULL,
    0xcaca00cacaca00caULL, 0x4747004747470047ULL, 0x3d3d003d3d3d003dULL,
    0x0101000101010001ULL, 0xd6d600d6d6d600d6ULL, 0x5656005656560056ULL,
    0x4d4d004d4d4d004dULL, 0x0d0d000d0d0d000dULL, 0x6666006666660066ULL,
    0xcccc00cccccc00ccULL, 0x2d2d002d2d2d002dULL, 0x1212001212120012ULL,
    0x2020002020200020ULL, 0xb1b100b1b1b100b1ULL, 0x9999009999990099ULL,
    0x4c4c004c4c4c004cULL, 0xc2c200c2c2c200c2ULL, 0x7e7e007e7e7e007eULL,
    0x0505000505050005ULL, 0xb7b700b7b7b700b7ULL, 0x3131003131310031ULL,
    0x1717001717170017ULL, 0xd7d700d7d7d700d7ULL, 0x5858005858580058ULL,
    0x6161006161610061ULL, 0x1b1b001b1b1b001bULL, 0x1c1c001c1c1c001cULL,
    0x0f0f000f0f0f000fULL, 0x1616001616160016ULL, 0x1818001818180018ULL,
    0x2222002222220022ULL, 0x4444004444440044ULL, 0xb2b200b2b2b200b2ULL,
    0xb5b500b5b5b500b5ULL, 0x9191009191910091ULL, 0x0808000808080008ULL,
    0xa8a800a8a8a800a8ULL, 0xfcfc00fcfcfc00fcULL, 0x5050005050500050ULL,
    0xd0d000d0d0d000d0ULL, 0x7d7d007d7d7d007dULL, 0x8989008989890089ULL,
    0x9797009797970097ULL, 0x5b5b005b5b5b005bULL, 0x9595009595950095ULL,
    0xffff00ffffff00ffULL, 0xd2d200d2d2d200d2ULL, 0xc4c400c4c4c400c4ULL,
    0x4848004848480048ULL, 0xf7f700f7f7f700f7ULL, 0xdbdb00dbdbdb00dbULL,
    0x0303000303030003ULL, 0xdada00dadada00daULL, 0x3f3f003f3f3f003fULL,
    0x9494009494940094ULL, 0x5c5c005c5c5c005cULL, 0x0202000202020002ULL,
    0x4a4a004a4a4a004aULL, 0x3333003333330033ULL, 0x6767006767670067ULL,
    0xf3f300f3f3f300f3ULL, 0x7f7f007f7f7f007fULL, 0xe2e200e2e2e200e2ULL,
    0x9b9b009b9b9b009bULL, 0x2626002626260026ULL, 0x3737003737370037ULL,
    0x3b3b003b3b3b003bULL, 0x9696009696960096ULL, 0x4b4b004b4b4b004bULL,
    0xbebe00bebebe00beULL, 0x2e2e002e2e2e002eULL, 0x7979007979790079ULL,
    0x8c8c008c8c8c008cULL, 0x6e6e006e6e6e006eULL, 0x8e8e008e8e8e008eULL,
    0xf5f500f5f5f500f5ULL, 0xb6b600b6b6b600b6ULL, 0xfdfd00fdfdfd00fdULL,
    0x5959005959590059ULL, 0x9898009898980098ULL, 0x6a6a006a6a6a006aULL,
    0x4646004646460046ULL, 0xbaba00bababa00baULL, 0x2525002525250025ULL,
    0x4242004242420042ULL, 0xa2a200a2a2a200a2ULL, 0xfafa00fafafa00faULL,
    0x0707000707070007ULL, 0x5555005555550055ULL, 0xeeee00eeeeee00eeULL,
    0x0a0a000a0a0a000aULL, 0x4949004949490049ULL, 0x6868006868680068ULL,
    0x3838003838380038ULL, 0xa4a400a4a4a400a4ULL, 0x2828002828280028ULL,
    0x7b7b007b7b7b007bULL, 0xc9c900c9c9c900c9ULL, 0xc1c100c1c1c100c1ULL,
    0xe3e300e3e3e300e3ULL, 0xf4f400f4f4f400f4ULL, 0xc7c700c7c7c700c7ULL,
    0x9e9e009e9e9e009eULL,
};

const uint64 camellia_sp11101110[256] = {
    0x7070700070707000ULL, 0x8282820082828200ULL, 0x2c2c2c002c2c2c00ULL,
    0xececec00ececec00ULL, 0xb3b3b300b3b3b300ULL, 0x2727270027272700ULL,
    0xc0c0c000c0c0c000ULL, 0xe5e5e500e5e5e500ULL, 0xe4e4e400e4e4e400ULL,
    0x8585850085858500ULL, 0x5757570057575700ULL, 0x3535350035353500ULL,
    0xeaeaea00eaeaea00ULL, 0x0c0c0c000c0c0c00ULL, 0xaeaeae00aeaeae00ULL,
    0x4141410041414100ULL, 0x2323230023232300ULL, 0xefefef00efefef00ULL,
    0x6b6b6b006b6b6b00ULL, 0x9393930093939300ULL, 0x4545450045454500ULL,
    0x1919190019191900ULL, 0xa5a5a500a5a5a500ULL, 0x2121210021212100ULL,
    0xededed00ededed00ULL, 0x0e0e0e000e0e0e00ULL, 0x4f4f4f004f4f4f00ULL,
    0x4e4e4e004e4e4e00ULL, 0x1d1d1d001d1d1d00ULL, 0x6565650065656500ULL,
    0x9292920092929200ULL, 0xbdbdbd00bdbdbd00ULL, 0x8686860086868600ULL,
    0xb8b8b800b8b8b800ULL, 0xafafaf00afafaf00ULL, 0x8f8f8f008f8f8f00ULL,
    0x7c7c7c007c7c7c00ULL, 0xebebeb00ebebeb00ULL, 0x1f1f1f001f1f1f00ULL,
    0xcecece00cecece00ULL, 0x3e3e3e003e3e3e00ULL, 0x3030300030303000ULL,
    0xdcdcdc00dcdcdc00ULL, 0x5f5f5f005f5f5f00ULL, 0x5e5e5e005e5e5e00ULL,
    0xc5c5c500c5c5c500ULL, 0x0b0b0b000b0b0b00ULL, 0x1a1a1a001a1a1a00ULL,
    0xa6a6a600a6a6a600ULL, 0xe1e1e100e1e1e100ULL, 0x3939390039393900ULL,
    0xcacaca00cacaca00ULL, 0xd5d5d500d5d5d500ULL, 0x4747470047474700ULL,
    0x5d5d5d005d5d5d00ULL, 0x3d3d3d003d3d3d00ULL, 0xd9d9d900d9d9d900ULL,
    0x0101010001010100ULL, 0x5a5a5a005a5a5a00ULL, 0xd6d6d600d6d6d600ULL,
    0x5151510051515100ULL, 0x5656560056565600ULL, 0x6c6c6c006c6c6c00ULL,
    0x4d4d4d004d4d4d00ULL, 0x8b8b8b008b8b8b00ULL, 0x0d0d0d000d0d0d00ULL,
    0x9a9a9a009a9a9a00ULL, 0x6666660066666600ULL, 0xfbfbfb00fbfbfb00ULL,
    0xcccccc00cccccc00ULL, 0xb0b0b000b0b0b000ULL, 0x2d2d2d002d2d2d00ULL,
    0x7474740074747400ULL, 0x1212120012121200ULL, 0x2b2b2b002b2b2b00ULL,
    0x2020200020202000ULL, 0xf0f0f000f0f0f000ULL, 0xb1b1b100b1b1b100ULL,
    0x8484840084848400ULL, 0x9999990099999900ULL, 0xdfdfdf00dfdfdf00ULL,
    0x4c4c4c004c4c4c00ULL, 0xcbcbcb00cbcbcb00ULL, 0xc2c2c200c2c2c200ULL,
    0x3434340034343400ULL, 0x7e7e7e007e7e7e00ULL, 0x7676760076767600ULL,
    0x0505050005050500ULL, 0x6d6d6d006d6d6d00ULL, 0xb7b7b700b7b7b700ULL,
    0xa9a9a900a9a9a900ULL, 0x3131310031313100ULL, 0xd1d1d100d1d1d100ULL,
    0x1717170017171700ULL, 0x0404040004040400ULL, 0xd7d7d700d7d7d700ULL,
    0x1414140014141400ULL, 0x5858580058585800ULL, 0x3a3a3a003a3a3a00ULL,
    0x6161610061616100ULL, 0xdedede00dedede00ULL, 0x1b1b1b001b1b1b00ULL,
    0x1111110011111100ULL, 0x1c1c1c001c1c1c00ULL, 0x3232320032323200ULL,
    0x0f0f0f000f0f0f00ULL, 0x9c9c9c009c9c9c00ULL, 0x1616160016161600ULL,
    0x5353530053535300ULL, 0x1818180018181800ULL, 0xf2f2f200f2f2f200ULL,
    0x2222220022222200ULL, 0xfefefe00fefefe00ULL, 0x4444440044444400ULL,
    0xcfcfcf00cfcfcf00ULL, 0xb2b2b200b2b2b200ULL, 0xc3c3c300c3c3c300ULL,
    0xb5b5b500b5b5b500ULL, 0x7a7a7a007a7a7a00ULL, 0x9191910091919100ULL,
    0x2424240024242400ULL, 0x0808080008080800ULL, 0xe8e8e800e8e8e800ULL,
    0xa8a8a800a8a8a800ULL, 0x6060600060606000ULL, 0xfcfcfc00fcfcfc00ULL,
    0x6969690069696900ULL, 0x5050500050505000ULL, 0xaaaaaa00aaaaaa00ULL,
    0xd0d0d000d0d0d000ULL, 0xa0a0a000a0a0a000ULL, 0x7d7d7d007d7d7d00ULL,
    0xa1a1a100a1a1a100ULL, 0x8989890089898900ULL, 0x6262620062626200ULL,
    0x9797970097979700ULL, 0x5454540054545400ULL, 0x5b5b5b005b5b5b00ULL,
    0x1e1e1e001e1e1e00ULL, 0x9595950095959500ULL, 0xe0e0e000e0e0e000ULL,
    0xffffff00ffffff00ULL, 0x6464640064646400ULL, 0xd2d2d200d2d2d200ULL,
    0x1010100010101000ULL, 0xc4c4c400c4c4c400ULL, 0x0000000000000000ULL,
    0x4848480048484800ULL, 0xa3a3a300a3a3a300ULL, 0xf7f7f700f7f7f700ULL,
    0x7575750075757500ULL, 0xdbdbdb00dbdbdb00ULL, 0x8a8a8a008a8a8a00ULL,
    0x0303030003030300ULL, 0xe6e6e600e6e6e600ULL, 0xdadada00dadada00ULL,
    0x0909090009090900ULL, 0x3f3f3f003f3f3f00ULL, 0xdddddd00dddddd00ULL,
    0x9494940094949400ULL, 0x8787870087878700ULL, 0x5c5c5c005c5c5c00ULL,
    0x8383830083838300ULL, 0x0202020002020200ULL, 0xcdcdcd00cdcdcd00ULL,
    0x4a4a4a004a4a4a00ULL, 0x9090900090909000ULL, 0x3333330033333300ULL,
    0x7373730073737300ULL, 0x6767670067676700ULL, 0xf6f6f600f6f6f600ULL,
    0xf3f3f300f3f3f300ULL, 0x9d9d9d009d9d9d00ULL, 0x7f7f7f007f7f7f00ULL,
    0xbfbfbf00bfbfbf00ULL, 0xe2e2e200e2e2e200ULL, 0x5252520052525200ULL,
    0x9b9b9b009b9b9b00ULL, 0xd8d8d800d8d8d800ULL, 0x2626260026262600ULL,
    0xc8c8c800c8c8c800ULL, 0x3737370037373700ULL, 0xc6c6c600c6c6c600ULL,
    0x3b3b3b003b3b3b00ULL, 0x8181810081818100ULL, 0x9696960096969600ULL,
    0x6f6f6f006f6f6f00ULL, 0x4b4b4b004b4b4b00ULL, 0x1313130013131300ULL,
    0xbebebe00bebebe00ULL, 0x6363630063636300ULL, 0x2e2e2e002e2e2e00ULL,
    0xe9e9e900e9e9e900ULL, 0x7979790079797900ULL, 0xa7a7a700a7a7a700ULL,
    0x8c8c8c008c8c8c00ULL, 0x9f9f9f009f9f9f00ULL, 0x6e6e6e006e6e6e00ULL,
    0xbcbcbc00bcbcbc00ULL, 0x8e8e8e008e8e8e00ULL, 0x2929290029292900ULL,
    0xf5f5f500f5f5f500ULL, 0xf9f9f900f9f9f900ULL, 0xb6b6b600b6b6b600ULL,
    0x2f2f2f002f2f2f00ULL, 0xfdfdfd00fdfdfd00ULL, 0xb4b4b400b4b4b400ULL,
    0x5959590059595900ULL, 0x7878780078787800ULL, 0x9898980098989800ULL,
    0x0606060006060600ULL, 0x6a6a6a006a6a6a00ULL, 0xe7e7e700e7e7e700ULL,
    0x4646460046464600ULL, 0x7171710071717100ULL, 0xbababa00bababa00ULL,
    0xd4d4d400d4d4d400ULL, 0x2525250025252500ULL, 0xababab00ababab00ULL,
    0x4242420042424200ULL, 0x8888880088888800ULL, 0xa2a2a200a2a2a200ULL,
    0x8d8d8d008d8d8d00ULL, 0xfafafa00fafafa00ULL, 0x7272720072727200ULL,
    0x0707070007070700ULL, 0xb9b9b900b9b9b900ULL, 0x5555550055555500ULL,
    0xf8f8f800f8f8f800ULL, 0xeeeeee00eeeeee00ULL, 0xacacac00acacac00ULL,
    0x0a0a0a000a0a0a00ULL, 0x3636360036363600ULL, 0x4949490049494900ULL,
    0x2a2a2a002a2a2a00ULL, 0x6868680068686800ULL, 0x3c3c3c003c3c3c00ULL,
    0x3838380038383800ULL, 0xf1f1f100f1f1f100ULL, 0xa4a4a400a4a4a400ULL,
    0x4040400040404000ULL, 0x2828280028282800ULL, 0xd3d3d300d3d3d300ULL,
    0x7b7b7b007b7b7b00ULL, 0xbbbbbb00bbbbbb00ULL, 0xc9c9c900c9c9c900ULL,
    0x4343430043434300ULL, 0xc1c1c100c1c1c100ULL, 0x1515150015151500ULL,
    0xe3e3e300e3e3e300ULL, 0xadadad00adadad00ULL, 0xf4f4f400f4f4f400ULL,
    0x7777770077777700ULL, 0xc7c7c700c7c7c700ULL, 0x8080800080808000ULL,
    0x9e9e9e009e9e9e00ULL,
};


/**
 * Stuff related to the Camellia key schedule
 */
#define subl(x) subL[(x)]
#define subr(x) subR[(x)]

void camellia_set_key(const unsigned __int8 key[], unsigned __int8 *ks)
{
    uint32 kll,klr,krl,krr;           /* left half of key */
    uint32 krll,krlr,krrl,krrr;       /* right half of key */
    uint32 il, ir, t0, t1, w0, w1;    /* temporary variables */
    uint32 kw4l, kw4r, dw, tl, tr;
    uint32 subL[34];
    uint32 subR[34];
	 uint32 *subkey = (uint32*) ks;
    /**
     *  key = (kll || klr || krl || krr || krll || krlr || krrl || krrr)
     *  (|| is concatination)
     */
    kll  = GETU32(key     );
    klr  = GETU32(key +  4);
    krl  = GETU32(key +  8);
    krr  = GETU32(key + 12);
    krll = GETU32(key + 16);
    krlr = GETU32(key + 20);
    krrl = GETU32(key + 24);
    krrr = GETU32(key + 28);
    /* generate KL dependent subkeys */
    subl(0) = kll; subr(0) = klr;
    subl(1) = krl; subr(1) = krr;
    CAMELLIA_ROLDQo32(kll, klr, krl, krr, w0, w1, 45);
    subl(12) = kll; subr(12) = klr;
    subl(13) = krl; subr(13) = krr;
    CAMELLIA_ROLDQ(kll, klr, krl, krr, w0, w1, 15);
    subl(16) = kll; subr(16) = klr;
    subl(17) = krl; subr(17) = krr;
    CAMELLIA_ROLDQ(kll, klr, krl, krr, w0, w1, 17);
    subl(22) = kll; subr(22) = klr;
    subl(23) = krl; subr(23) = krr;
    CAMELLIA_ROLDQo32(kll, klr, krl, krr, w0, w1, 34);
    subl(30) = kll; subr(30) = klr;
    subl(31) = krl; subr(31) = krr;
    /* generate KR dependent subkeys */
    CAMELLIA_ROLDQ(krll, krlr, krrl, krrr, w0, w1, 15);
    subl(4) = krll; subr(4) = krlr;
    subl(5) = krrl; subr(5) = krrr;
    CAMELLIA_ROLDQ(krll, krlr, krrl, krrr, w0, w1, 15);
    subl(8) = krll; subr(8) = krlr;
    subl(9) = krrl; subr(9) = krrr;
    CAMELLIA_ROLDQ(krll, krlr, krrl, krrr, w0, w1, 30);
    subl(18) = krll; subr(18) = krlr;
    subl(19) = krrl; subr(19) = krrr;
    CAMELLIA_ROLDQo32(krll, krlr, krrl, krrr, w0, w1, 34);
    subl(26) = krll; subr(26) = krlr;
    subl(27) = krrl; subr(27) = krrr;
    CAMELLIA_ROLDQo32(krll, krlr, krrl, krrr, w0, w1, 34);
    /* generate KA */
    kll = subl(0) ^ krll; klr = subr(0) ^ krlr;
    krl = subl(1) ^ krrl; krr = subr(1) ^ krrr;
    CAMELLIA_F(kll, klr,
	       CAMELLIA_SIGMA1L, CAMELLIA_SIGMA1R,
	       w0, w1, il, ir, t0, t1);
    krl ^= w0; krr ^= w1;
    CAMELLIA_F(krl, krr,
	       CAMELLIA_SIGMA2L, CAMELLIA_SIGMA2R,
	       kll, klr, il, ir, t0, t1);
    kll ^= krll; klr ^= krlr;
    CAMELLIA_F(kll, klr,
	       CAMELLIA_SIGMA3L, CAMELLIA_SIGMA3R,
	       krl, krr, il, ir, t0, t1);
    krl ^= w0 ^ krrl; krr ^= w1 ^ krrr;
    CAMELLIA_F(krl, krr,
	       CAMELLIA_SIGMA4L, CAMELLIA_SIGMA4R,
	       w0, w1, il, ir, t0, t1);
    kll ^= w0; klr ^= w1;
    /* generate KB */
    krll ^= kll; krlr ^= klr;
    krrl ^= krl; krrr ^= krr;
    CAMELLIA_F(krll, krlr,
	       CAMELLIA_SIGMA5L, CAMELLIA_SIGMA5R,
	       w0, w1, il, ir, t0, t1);
    krrl ^= w0; krrr ^= w1;
    CAMELLIA_F(krrl, krrr,
	       CAMELLIA_SIGMA6L, CAMELLIA_SIGMA6R,
	       w0, w1, il, ir, t0, t1);
    krll ^= w0; krlr ^= w1;
    /* generate KA dependent subkeys */
    CAMELLIA_ROLDQ(kll, klr, krl, krr, w0, w1, 15);
    subl(6) = kll; subr(6) = klr;
    subl(7) = krl; subr(7) = krr;
    CAMELLIA_ROLDQ(kll, klr, krl, krr, w0, w1, 30);
    subl(14) = kll; subr(14) = klr;
    subl(15) = krl; subr(15) = krr;
    subl(24) = klr; subr(24) = krl;
    subl(25) = krr; subr(25) = kll;
    CAMELLIA_ROLDQo32(kll, klr, krl, krr, w0, w1, 49);
    subl(28) = kll; subr(28) = klr;
    subl(29) = krl; subr(29) = krr;
    /* generate KB dependent subkeys */
    subl(2) = krll; subr(2) = krlr;
    subl(3) = krrl; subr(3) = krrr;
    CAMELLIA_ROLDQ(krll, krlr, krrl, krrr, w0, w1, 30);
    subl(10) = krll; subr(10) = krlr;
    subl(11) = krrl; subr(11) = krrr;
    CAMELLIA_ROLDQ(krll, krlr, krrl, krrr, w0, w1, 30);
    subl(20) = krll; subr(20) = krlr;
    subl(21) = krrl; subr(21) = krrr;
    CAMELLIA_ROLDQo32(krll, krlr, krrl, krrr, w0, w1, 51);
    subl(32) = krll; subr(32) = krlr;
    subl(33) = krrl; subr(33) = krrr;
    /* absorb kw2 to other subkeys */
    subl(3) ^= subl(1); subr(3) ^= subr(1);
    subl(5) ^= subl(1); subr(5) ^= subr(1);
    subl(7) ^= subl(1); subr(7) ^= subr(1);
    subl(1) ^= subr(1) & ~subr(9);
    dw = subl(1) & subl(9), subr(1) ^= CAMELLIA_RL1(dw);
    subl(11) ^= subl(1); subr(11) ^= subr(1);
    subl(13) ^= subl(1); subr(13) ^= subr(1);
    subl(15) ^= subl(1); subr(15) ^= subr(1);
    subl(1) ^= subr(1) & ~subr(17);
    dw = subl(1) & subl(17), subr(1) ^= CAMELLIA_RL1(dw);
    subl(19) ^= subl(1); subr(19) ^= subr(1);
    subl(21) ^= subl(1); subr(21) ^= subr(1);
    subl(23) ^= subl(1); subr(23) ^= subr(1);
    subl(1) ^= subr(1) & ~subr(25);
    dw = subl(1) & subl(25), subr(1) ^= CAMELLIA_RL1(dw);
    subl(27) ^= subl(1); subr(27) ^= subr(1);
    subl(29) ^= subl(1); subr(29) ^= subr(1);
    subl(31) ^= subl(1); subr(31) ^= subr(1);
    subl(32) ^= subl(1); subr(32) ^= subr(1);
    /* absorb kw4 to other subkeys */
    kw4l = subl(33); kw4r = subr(33);
    subl(30) ^= kw4l; subr(30) ^= kw4r;
    subl(28) ^= kw4l; subr(28) ^= kw4r;
    subl(26) ^= kw4l; subr(26) ^= kw4r;
    kw4l ^= kw4r & ~subr(24);
    dw = kw4l & subl(24), kw4r ^= CAMELLIA_RL1(dw);
    subl(22) ^= kw4l; subr(22) ^= kw4r;
    subl(20) ^= kw4l; subr(20) ^= kw4r;
    subl(18) ^= kw4l; subr(18) ^= kw4r;
    kw4l ^= kw4r & ~subr(16);
    dw = kw4l & subl(16), kw4r ^= CAMELLIA_RL1(dw);
    subl(14) ^= kw4l; subr(14) ^= kw4r;
    subl(12) ^= kw4l; subr(12) ^= kw4r;
    subl(10) ^= kw4l; subr(10) ^= kw4r;
    kw4l ^= kw4r & ~subr(8);
    dw = kw4l & subl(8), kw4r ^= CAMELLIA_RL1(dw);
    subl(6) ^= kw4l; subr(6) ^= kw4r;
    subl(4) ^= kw4l; subr(4) ^= kw4r;
    subl(2) ^= kw4l; subr(2) ^= kw4r;
    subl(0) ^= kw4l; subr(0) ^= kw4r;
    /* key XOR is end of F-function */
    CamelliaSubkeyL(0) = subl(0) ^ subl(2);
    CamelliaSubkeyR(0) = subr(0) ^ subr(2);
    CamelliaSubkeyL(2) = subl(3);
    CamelliaSubkeyR(2) = subr(3);
    CamelliaSubkeyL(3) = subl(2) ^ subl(4);
    CamelliaSubkeyR(3) = subr(2) ^ subr(4);
    CamelliaSubkeyL(4) = subl(3) ^ subl(5);
    CamelliaSubkeyR(4) = subr(3) ^ subr(5);
    CamelliaSubkeyL(5) = subl(4) ^ subl(6);
    CamelliaSubkeyR(5) = subr(4) ^ subr(6);
    CamelliaSubkeyL(6) = subl(5) ^ subl(7);
    CamelliaSubkeyR(6) = subr(5) ^ subr(7);
    tl = subl(10) ^ (subr(10) & ~subr(8));
    dw = tl & subl(8), tr = subr(10) ^ CAMELLIA_RL1(dw);
    CamelliaSubkeyL(7) = subl(6) ^ tl;
    CamelliaSubkeyR(7) = subr(6) ^ tr;
    CamelliaSubkeyL(8) = subl(8);
    CamelliaSubkeyR(8) = subr(8);
    CamelliaSubkeyL(9) = subl(9);
    CamelliaSubkeyR(9) = subr(9);
    tl = subl(7) ^ (subr(7) & ~subr(9));
    dw = tl & subl(9), tr = subr(7) ^ CAMELLIA_RL1(dw);
    CamelliaSubkeyL(10) = tl ^ subl(11);
    CamelliaSubkeyR(10) = tr ^ subr(11);
    CamelliaSubkeyL(11) = subl(10) ^ subl(12);
    CamelliaSubkeyR(11) = subr(10) ^ subr(12);
    CamelliaSubkeyL(12) = subl(11) ^ subl(13);
    CamelliaSubkeyR(12) = subr(11) ^ subr(13);
    CamelliaSubkeyL(13) = subl(12) ^ subl(14);
    CamelliaSubkeyR(13) = subr(12) ^ subr(14);
    CamelliaSubkeyL(14) = subl(13) ^ subl(15);
    CamelliaSubkeyR(14) = subr(13) ^ subr(15);
    tl = subl(18) ^ (subr(18) & ~subr(16));
    dw = tl & subl(16), tr = subr(18) ^ CAMELLIA_RL1(dw);
    CamelliaSubkeyL(15) = subl(14) ^ tl;
    CamelliaSubkeyR(15) = subr(14) ^ tr;
    CamelliaSubkeyL(16) = subl(16);
    CamelliaSubkeyR(16) = subr(16);
    CamelliaSubkeyL(17) = subl(17);
    CamelliaSubkeyR(17) = subr(17);
    tl = subl(15) ^ (subr(15) & ~subr(17));
    dw = tl & subl(17), tr = subr(15) ^ CAMELLIA_RL1(dw);
    CamelliaSubkeyL(18) = tl ^ subl(19);
    CamelliaSubkeyR(18) = tr ^ subr(19);
    CamelliaSubkeyL(19) = subl(18) ^ subl(20);
    CamelliaSubkeyR(19) = subr(18) ^ subr(20);
    CamelliaSubkeyL(20) = subl(19) ^ subl(21);
    CamelliaSubkeyR(20) = subr(19) ^ subr(21);
    CamelliaSubkeyL(21) = subl(20) ^ subl(22);
    CamelliaSubkeyR(21) = subr(20) ^ subr(22);
    CamelliaSubkeyL(22) = subl(21) ^ subl(23);
    CamelliaSubkeyR(22) = subr(21) ^ subr(23);
    tl = subl(26) ^ (subr(26) & ~subr(24));
    dw = tl & subl(24), tr = subr(26) ^ CAMELLIA_RL1(dw);
    CamelliaSubkeyL(23) = subl(22) ^ tl;
    CamelliaSubkeyR(23) = subr(22) ^ tr;
    CamelliaSubkeyL(24) = subl(24);
    CamelliaSubkeyR(24) = subr(24);
    CamelliaSubkeyL(25) = subl(25);
    CamelliaSubkeyR(25) = subr(25);
    tl = subl(23) ^ (subr(23) &  ~subr(25));
    dw = tl & subl(25), tr = subr(23) ^ CAMELLIA_RL1(dw);
    CamelliaSubkeyL(26) = tl ^ subl(27);
    CamelliaSubkeyR(26) = tr ^ subr(27);
    CamelliaSubkeyL(27) = subl(26) ^ subl(28);
    CamelliaSubkeyR(27) = subr(26) ^ subr(28);
    CamelliaSubkeyL(28) = subl(27) ^ subl(29);
    CamelliaSubkeyR(28) = subr(27) ^ subr(29);
    CamelliaSubkeyL(29) = subl(28) ^ subl(30);
    CamelliaSubkeyR(29) = subr(28) ^ subr(30);
    CamelliaSubkeyL(30) = subl(29) ^ subl(31);
    CamelliaSubkeyR(30) = subr(29) ^ subr(31);
    CamelliaSubkeyL(31) = subl(30);
    CamelliaSubkeyR(31) = subr(30);
    CamelliaSubkeyL(32) = subl(32) ^ subl(31);
    CamelliaSubkeyR(32) = subr(32) ^ subr(31);
    return;
}

void camellia_encrypt(const unsigned __int8 *inBlock, unsigned __int8 *outBlock, unsigned __int8 *ks)
{
	camellia_encrypt_asm (ks, outBlock, inBlock);
}

void camellia_decrypt(const unsigned __int8 *inBlock,  unsigned __int8 *outBlock, unsigned __int8 *ks)
{
	camellia_decrypt_asm (ks, outBlock, inBlock);
}

void camellia_encrypt_blocks(unsigned __int8 *instance, const byte* in_blk, byte* out_blk, uint32 blockCount)
{
#if !defined (_UEFI)
	if ((blockCount >= 16) && IsCpuIntel() && IsAesHwCpuSupported () && HasSAVX()) /* on AMD cpu, AVX is too slow */
	{
#if defined (TC_WINDOWS_DRIVER)
		XSTATE_SAVE SaveState;
		if (NT_SUCCESS (KeSaveExtendedProcessorState(XSTATE_MASK_GSSE, &SaveState)))
		{
#endif
			while (blockCount >= 16)
			{
				camellia_ecb_enc_16way (instance, out_blk, in_blk);
				out_blk += 16 * 16;
				in_blk += 16 * 16;
				blockCount -= 16;
			}
#if defined (TC_WINDOWS_DRIVER)
			KeRestoreExtendedProcessorState(&SaveState);
		}
#endif
	}
#endif

	while (blockCount >= 2)
	{
		camellia_enc_blk2 (instance, out_blk, in_blk);
		out_blk += 2 * 16;
		in_blk += 2 * 16;
		blockCount -= 2;
	}
	
	if (blockCount)
		camellia_encrypt (in_blk, out_blk, instance);		
}

void camellia_decrypt_blocks(unsigned __int8 *instance, const byte* in_blk, byte* out_blk, uint32 blockCount)
{
#if !defined (_UEFI)
	if ((blockCount >= 16) && IsCpuIntel() && IsAesHwCpuSupported () && HasSAVX()) /* on AMD cpu, AVX is too slow */
	{
#if defined (TC_WINDOWS_DRIVER)
		XSTATE_SAVE SaveState;
		if (NT_SUCCESS (KeSaveExtendedProcessorState(XSTATE_MASK_GSSE, &SaveState)))
		{
#endif
		while (blockCount >= 16)
		{
			camellia_ecb_dec_16way (instance, out_blk, in_blk);
			out_blk += 16 * 16;
			in_blk += 16 * 16;
			blockCount -= 16;
		}
#if defined (TC_WINDOWS_DRIVER)
			KeRestoreExtendedProcessorState(&SaveState);
		}
#endif
	}
#endif

	while (blockCount >= 2)
	{
		camellia_dec_blk2 (instance, out_blk, in_blk);
		out_blk += 2 * 16;
		in_blk += 2 * 16;
		blockCount -= 2;
	}
	
	if (blockCount)
		camellia_decrypt (in_blk, out_blk, instance);		
}

#else

/*
This code is written by kerukuro for cppcrypto library (http://cppcrypto.sourceforge.net/)
and released into public domain.
*/

/* Adapted for VeraCrypt */

//#define CPPCRYPTO_DEBUG

static const uint64 S[8][256] = {
		{
			LL(0x7070700070000070), LL(0x8282820082000082), LL(0x2c2c2c002c00002c), LL(0xececec00ec0000ec), LL(0xb3b3b300b30000b3), LL(0x2727270027000027), LL(0xc0c0c000c00000c0), LL(0xe5e5e500e50000e5), 
			LL(0xe4e4e400e40000e4), LL(0x8585850085000085), LL(0x5757570057000057), LL(0x3535350035000035), LL(0xeaeaea00ea0000ea), LL(0x0c0c0c000c00000c), LL(0xaeaeae00ae0000ae), LL(0x4141410041000041), 
			LL(0x2323230023000023), LL(0xefefef00ef0000ef), LL(0x6b6b6b006b00006b), LL(0x9393930093000093), LL(0x4545450045000045), LL(0x1919190019000019), LL(0xa5a5a500a50000a5), LL(0x2121210021000021), 
			LL(0xededed00ed0000ed), LL(0x0e0e0e000e00000e), LL(0x4f4f4f004f00004f), LL(0x4e4e4e004e00004e), LL(0x1d1d1d001d00001d), LL(0x6565650065000065), LL(0x9292920092000092), LL(0xbdbdbd00bd0000bd), 
			LL(0x8686860086000086), LL(0xb8b8b800b80000b8), LL(0xafafaf00af0000af), LL(0x8f8f8f008f00008f), LL(0x7c7c7c007c00007c), LL(0xebebeb00eb0000eb), LL(0x1f1f1f001f00001f), LL(0xcecece00ce0000ce), 
			LL(0x3e3e3e003e00003e), LL(0x3030300030000030), LL(0xdcdcdc00dc0000dc), LL(0x5f5f5f005f00005f), LL(0x5e5e5e005e00005e), LL(0xc5c5c500c50000c5), LL(0x0b0b0b000b00000b), LL(0x1a1a1a001a00001a), 
			LL(0xa6a6a600a60000a6), LL(0xe1e1e100e10000e1), LL(0x3939390039000039), LL(0xcacaca00ca0000ca), LL(0xd5d5d500d50000d5), LL(0x4747470047000047), LL(0x5d5d5d005d00005d), LL(0x3d3d3d003d00003d), 
			LL(0xd9d9d900d90000d9), LL(0x0101010001000001), LL(0x5a5a5a005a00005a), LL(0xd6d6d600d60000d6), LL(0x5151510051000051), LL(0x5656560056000056), LL(0x6c6c6c006c00006c), LL(0x4d4d4d004d00004d), 
			LL(0x8b8b8b008b00008b), LL(0x0d0d0d000d00000d), LL(0x9a9a9a009a00009a), LL(0x6666660066000066), LL(0xfbfbfb00fb0000fb), LL(0xcccccc00cc0000cc), LL(0xb0b0b000b00000b0), LL(0x2d2d2d002d00002d), 
			LL(0x7474740074000074), LL(0x1212120012000012), LL(0x2b2b2b002b00002b), LL(0x2020200020000020), LL(0xf0f0f000f00000f0), LL(0xb1b1b100b10000b1), LL(0x8484840084000084), LL(0x9999990099000099), 
			LL(0xdfdfdf00df0000df), LL(0x4c4c4c004c00004c), LL(0xcbcbcb00cb0000cb), LL(0xc2c2c200c20000c2), LL(0x3434340034000034), LL(0x7e7e7e007e00007e), LL(0x7676760076000076), LL(0x0505050005000005), 
			LL(0x6d6d6d006d00006d), LL(0xb7b7b700b70000b7), LL(0xa9a9a900a90000a9), LL(0x3131310031000031), LL(0xd1d1d100d10000d1), LL(0x1717170017000017), LL(0x0404040004000004), LL(0xd7d7d700d70000d7), 
			LL(0x1414140014000014), LL(0x5858580058000058), LL(0x3a3a3a003a00003a), LL(0x6161610061000061), LL(0xdedede00de0000de), LL(0x1b1b1b001b00001b), LL(0x1111110011000011), LL(0x1c1c1c001c00001c), 
			LL(0x3232320032000032), LL(0x0f0f0f000f00000f), LL(0x9c9c9c009c00009c), LL(0x1616160016000016), LL(0x5353530053000053), LL(0x1818180018000018), LL(0xf2f2f200f20000f2), LL(0x2222220022000022), 
			LL(0xfefefe00fe0000fe), LL(0x4444440044000044), LL(0xcfcfcf00cf0000cf), LL(0xb2b2b200b20000b2), LL(0xc3c3c300c30000c3), LL(0xb5b5b500b50000b5), LL(0x7a7a7a007a00007a), LL(0x9191910091000091), 
			LL(0x2424240024000024), LL(0x0808080008000008), LL(0xe8e8e800e80000e8), LL(0xa8a8a800a80000a8), LL(0x6060600060000060), LL(0xfcfcfc00fc0000fc), LL(0x6969690069000069), LL(0x5050500050000050), 
			LL(0xaaaaaa00aa0000aa), LL(0xd0d0d000d00000d0), LL(0xa0a0a000a00000a0), LL(0x7d7d7d007d00007d), LL(0xa1a1a100a10000a1), LL(0x8989890089000089), LL(0x6262620062000062), LL(0x9797970097000097), 
			LL(0x5454540054000054), LL(0x5b5b5b005b00005b), LL(0x1e1e1e001e00001e), LL(0x9595950095000095), LL(0xe0e0e000e00000e0), LL(0xffffff00ff0000ff), LL(0x6464640064000064), LL(0xd2d2d200d20000d2), 
			LL(0x1010100010000010), LL(0xc4c4c400c40000c4), LL(0x0000000000000000), LL(0x4848480048000048), LL(0xa3a3a300a30000a3), LL(0xf7f7f700f70000f7), LL(0x7575750075000075), LL(0xdbdbdb00db0000db), 
			LL(0x8a8a8a008a00008a), LL(0x0303030003000003), LL(0xe6e6e600e60000e6), LL(0xdadada00da0000da), LL(0x0909090009000009), LL(0x3f3f3f003f00003f), LL(0xdddddd00dd0000dd), LL(0x9494940094000094), 
			LL(0x8787870087000087), LL(0x5c5c5c005c00005c), LL(0x8383830083000083), LL(0x0202020002000002), LL(0xcdcdcd00cd0000cd), LL(0x4a4a4a004a00004a), LL(0x9090900090000090), LL(0x3333330033000033), 
			LL(0x7373730073000073), LL(0x6767670067000067), LL(0xf6f6f600f60000f6), LL(0xf3f3f300f30000f3), LL(0x9d9d9d009d00009d), LL(0x7f7f7f007f00007f), LL(0xbfbfbf00bf0000bf), LL(0xe2e2e200e20000e2), 
			LL(0x5252520052000052), LL(0x9b9b9b009b00009b), LL(0xd8d8d800d80000d8), LL(0x2626260026000026), LL(0xc8c8c800c80000c8), LL(0x3737370037000037), LL(0xc6c6c600c60000c6), LL(0x3b3b3b003b00003b), 
			LL(0x8181810081000081), LL(0x9696960096000096), LL(0x6f6f6f006f00006f), LL(0x4b4b4b004b00004b), LL(0x1313130013000013), LL(0xbebebe00be0000be), LL(0x6363630063000063), LL(0x2e2e2e002e00002e), 
			LL(0xe9e9e900e90000e9), LL(0x7979790079000079), LL(0xa7a7a700a70000a7), LL(0x8c8c8c008c00008c), LL(0x9f9f9f009f00009f), LL(0x6e6e6e006e00006e), LL(0xbcbcbc00bc0000bc), LL(0x8e8e8e008e00008e), 
			LL(0x2929290029000029), LL(0xf5f5f500f50000f5), LL(0xf9f9f900f90000f9), LL(0xb6b6b600b60000b6), LL(0x2f2f2f002f00002f), LL(0xfdfdfd00fd0000fd), LL(0xb4b4b400b40000b4), LL(0x5959590059000059), 
			LL(0x7878780078000078), LL(0x9898980098000098), LL(0x0606060006000006), LL(0x6a6a6a006a00006a), LL(0xe7e7e700e70000e7), LL(0x4646460046000046), LL(0x7171710071000071), LL(0xbababa00ba0000ba), 
			LL(0xd4d4d400d40000d4), LL(0x2525250025000025), LL(0xababab00ab0000ab), LL(0x4242420042000042), LL(0x8888880088000088), LL(0xa2a2a200a20000a2), LL(0x8d8d8d008d00008d), LL(0xfafafa00fa0000fa), 
			LL(0x7272720072000072), LL(0x0707070007000007), LL(0xb9b9b900b90000b9), LL(0x5555550055000055), LL(0xf8f8f800f80000f8), LL(0xeeeeee00ee0000ee), LL(0xacacac00ac0000ac), LL(0x0a0a0a000a00000a), 
			LL(0x3636360036000036), LL(0x4949490049000049), LL(0x2a2a2a002a00002a), LL(0x6868680068000068), LL(0x3c3c3c003c00003c), LL(0x3838380038000038), LL(0xf1f1f100f10000f1), LL(0xa4a4a400a40000a4), 
			LL(0x4040400040000040), LL(0x2828280028000028), LL(0xd3d3d300d30000d3), LL(0x7b7b7b007b00007b), LL(0xbbbbbb00bb0000bb), LL(0xc9c9c900c90000c9), LL(0x4343430043000043), LL(0xc1c1c100c10000c1), 
			LL(0x1515150015000015), LL(0xe3e3e300e30000e3), LL(0xadadad00ad0000ad), LL(0xf4f4f400f40000f4), LL(0x7777770077000077), LL(0xc7c7c700c70000c7), LL(0x8080800080000080), LL(0x9e9e9e009e00009e)
		},
		{
			LL(0x00e0e0e0e0e00000), LL(0x0005050505050000), LL(0x0058585858580000), LL(0x00d9d9d9d9d90000), LL(0x0067676767670000), LL(0x004e4e4e4e4e0000), LL(0x0081818181810000), LL(0x00cbcbcbcbcb0000), 
			LL(0x00c9c9c9c9c90000), LL(0x000b0b0b0b0b0000), LL(0x00aeaeaeaeae0000), LL(0x006a6a6a6a6a0000), LL(0x00d5d5d5d5d50000), LL(0x0018181818180000), LL(0x005d5d5d5d5d0000), LL(0x0082828282820000), 
			LL(0x0046464646460000), LL(0x00dfdfdfdfdf0000), LL(0x00d6d6d6d6d60000), LL(0x0027272727270000), LL(0x008a8a8a8a8a0000), LL(0x0032323232320000), LL(0x004b4b4b4b4b0000), LL(0x0042424242420000), 
			LL(0x00dbdbdbdbdb0000), LL(0x001c1c1c1c1c0000), LL(0x009e9e9e9e9e0000), LL(0x009c9c9c9c9c0000), LL(0x003a3a3a3a3a0000), LL(0x00cacacacaca0000), LL(0x0025252525250000), LL(0x007b7b7b7b7b0000), 
			LL(0x000d0d0d0d0d0000), LL(0x0071717171710000), LL(0x005f5f5f5f5f0000), LL(0x001f1f1f1f1f0000), LL(0x00f8f8f8f8f80000), LL(0x00d7d7d7d7d70000), LL(0x003e3e3e3e3e0000), LL(0x009d9d9d9d9d0000), 
			LL(0x007c7c7c7c7c0000), LL(0x0060606060600000), LL(0x00b9b9b9b9b90000), LL(0x00bebebebebe0000), LL(0x00bcbcbcbcbc0000), LL(0x008b8b8b8b8b0000), LL(0x0016161616160000), LL(0x0034343434340000), 
			LL(0x004d4d4d4d4d0000), LL(0x00c3c3c3c3c30000), LL(0x0072727272720000), LL(0x0095959595950000), LL(0x00ababababab0000), LL(0x008e8e8e8e8e0000), LL(0x00bababababa0000), LL(0x007a7a7a7a7a0000), 
			LL(0x00b3b3b3b3b30000), LL(0x0002020202020000), LL(0x00b4b4b4b4b40000), LL(0x00adadadadad0000), LL(0x00a2a2a2a2a20000), LL(0x00acacacacac0000), LL(0x00d8d8d8d8d80000), LL(0x009a9a9a9a9a0000), 
			LL(0x0017171717170000), LL(0x001a1a1a1a1a0000), LL(0x0035353535350000), LL(0x00cccccccccc0000), LL(0x00f7f7f7f7f70000), LL(0x0099999999990000), LL(0x0061616161610000), LL(0x005a5a5a5a5a0000), 
			LL(0x00e8e8e8e8e80000), LL(0x0024242424240000), LL(0x0056565656560000), LL(0x0040404040400000), LL(0x00e1e1e1e1e10000), LL(0x0063636363630000), LL(0x0009090909090000), LL(0x0033333333330000), 
			LL(0x00bfbfbfbfbf0000), LL(0x0098989898980000), LL(0x0097979797970000), LL(0x0085858585850000), LL(0x0068686868680000), LL(0x00fcfcfcfcfc0000), LL(0x00ececececec0000), LL(0x000a0a0a0a0a0000), 
			LL(0x00dadadadada0000), LL(0x006f6f6f6f6f0000), LL(0x0053535353530000), LL(0x0062626262620000), LL(0x00a3a3a3a3a30000), LL(0x002e2e2e2e2e0000), LL(0x0008080808080000), LL(0x00afafafafaf0000), 
			LL(0x0028282828280000), LL(0x00b0b0b0b0b00000), LL(0x0074747474740000), LL(0x00c2c2c2c2c20000), LL(0x00bdbdbdbdbd0000), LL(0x0036363636360000), LL(0x0022222222220000), LL(0x0038383838380000), 
			LL(0x0064646464640000), LL(0x001e1e1e1e1e0000), LL(0x0039393939390000), LL(0x002c2c2c2c2c0000), LL(0x00a6a6a6a6a60000), LL(0x0030303030300000), LL(0x00e5e5e5e5e50000), LL(0x0044444444440000), 
			LL(0x00fdfdfdfdfd0000), LL(0x0088888888880000), LL(0x009f9f9f9f9f0000), LL(0x0065656565650000), LL(0x0087878787870000), LL(0x006b6b6b6b6b0000), LL(0x00f4f4f4f4f40000), LL(0x0023232323230000), 
			LL(0x0048484848480000), LL(0x0010101010100000), LL(0x00d1d1d1d1d10000), LL(0x0051515151510000), LL(0x00c0c0c0c0c00000), LL(0x00f9f9f9f9f90000), LL(0x00d2d2d2d2d20000), LL(0x00a0a0a0a0a00000), 
			LL(0x0055555555550000), LL(0x00a1a1a1a1a10000), LL(0x0041414141410000), LL(0x00fafafafafa0000), LL(0x0043434343430000), LL(0x0013131313130000), LL(0x00c4c4c4c4c40000), LL(0x002f2f2f2f2f0000), 
			LL(0x00a8a8a8a8a80000), LL(0x00b6b6b6b6b60000), LL(0x003c3c3c3c3c0000), LL(0x002b2b2b2b2b0000), LL(0x00c1c1c1c1c10000), LL(0x00ffffffffff0000), LL(0x00c8c8c8c8c80000), LL(0x00a5a5a5a5a50000), 
			LL(0x0020202020200000), LL(0x0089898989890000), LL(0x0000000000000000), LL(0x0090909090900000), LL(0x0047474747470000), LL(0x00efefefefef0000), LL(0x00eaeaeaeaea0000), LL(0x00b7b7b7b7b70000), 
			LL(0x0015151515150000), LL(0x0006060606060000), LL(0x00cdcdcdcdcd0000), LL(0x00b5b5b5b5b50000), LL(0x0012121212120000), LL(0x007e7e7e7e7e0000), LL(0x00bbbbbbbbbb0000), LL(0x0029292929290000), 
			LL(0x000f0f0f0f0f0000), LL(0x00b8b8b8b8b80000), LL(0x0007070707070000), LL(0x0004040404040000), LL(0x009b9b9b9b9b0000), LL(0x0094949494940000), LL(0x0021212121210000), LL(0x0066666666660000), 
			LL(0x00e6e6e6e6e60000), LL(0x00cecececece0000), LL(0x00ededededed0000), LL(0x00e7e7e7e7e70000), LL(0x003b3b3b3b3b0000), LL(0x00fefefefefe0000), LL(0x007f7f7f7f7f0000), LL(0x00c5c5c5c5c50000), 
			LL(0x00a4a4a4a4a40000), LL(0x0037373737370000), LL(0x00b1b1b1b1b10000), LL(0x004c4c4c4c4c0000), LL(0x0091919191910000), LL(0x006e6e6e6e6e0000), LL(0x008d8d8d8d8d0000), LL(0x0076767676760000), 
			LL(0x0003030303030000), LL(0x002d2d2d2d2d0000), LL(0x00dedededede0000), LL(0x0096969696960000), LL(0x0026262626260000), LL(0x007d7d7d7d7d0000), LL(0x00c6c6c6c6c60000), LL(0x005c5c5c5c5c0000), 
			LL(0x00d3d3d3d3d30000), LL(0x00f2f2f2f2f20000), LL(0x004f4f4f4f4f0000), LL(0x0019191919190000), LL(0x003f3f3f3f3f0000), LL(0x00dcdcdcdcdc0000), LL(0x0079797979790000), LL(0x001d1d1d1d1d0000), 
			LL(0x0052525252520000), LL(0x00ebebebebeb0000), LL(0x00f3f3f3f3f30000), LL(0x006d6d6d6d6d0000), LL(0x005e5e5e5e5e0000), LL(0x00fbfbfbfbfb0000), LL(0x0069696969690000), LL(0x00b2b2b2b2b20000), 
			LL(0x00f0f0f0f0f00000), LL(0x0031313131310000), LL(0x000c0c0c0c0c0000), LL(0x00d4d4d4d4d40000), LL(0x00cfcfcfcfcf0000), LL(0x008c8c8c8c8c0000), LL(0x00e2e2e2e2e20000), LL(0x0075757575750000), 
			LL(0x00a9a9a9a9a90000), LL(0x004a4a4a4a4a0000), LL(0x0057575757570000), LL(0x0084848484840000), LL(0x0011111111110000), LL(0x0045454545450000), LL(0x001b1b1b1b1b0000), LL(0x00f5f5f5f5f50000), 
			LL(0x00e4e4e4e4e40000), LL(0x000e0e0e0e0e0000), LL(0x0073737373730000), LL(0x00aaaaaaaaaa0000), LL(0x00f1f1f1f1f10000), LL(0x00dddddddddd0000), LL(0x0059595959590000), LL(0x0014141414140000), 
			LL(0x006c6c6c6c6c0000), LL(0x0092929292920000), LL(0x0054545454540000), LL(0x00d0d0d0d0d00000), LL(0x0078787878780000), LL(0x0070707070700000), LL(0x00e3e3e3e3e30000), LL(0x0049494949490000), 
			LL(0x0080808080800000), LL(0x0050505050500000), LL(0x00a7a7a7a7a70000), LL(0x00f6f6f6f6f60000), LL(0x0077777777770000), LL(0x0093939393930000), LL(0x0086868686860000), LL(0x0083838383830000), 
			LL(0x002a2a2a2a2a0000), LL(0x00c7c7c7c7c70000), LL(0x005b5b5b5b5b0000), LL(0x00e9e9e9e9e90000), LL(0x00eeeeeeeeee0000), LL(0x008f8f8f8f8f0000), LL(0x0001010101010000), LL(0x003d3d3d3d3d0000)
		},
		{
			LL(0x3800383800383800), LL(0x4100414100414100), LL(0x1600161600161600), LL(0x7600767600767600), LL(0xd900d9d900d9d900), LL(0x9300939300939300), LL(0x6000606000606000), LL(0xf200f2f200f2f200), 
			LL(0x7200727200727200), LL(0xc200c2c200c2c200), LL(0xab00abab00abab00), LL(0x9a009a9a009a9a00), LL(0x7500757500757500), LL(0x0600060600060600), LL(0x5700575700575700), LL(0xa000a0a000a0a000), 
			LL(0x9100919100919100), LL(0xf700f7f700f7f700), LL(0xb500b5b500b5b500), LL(0xc900c9c900c9c900), LL(0xa200a2a200a2a200), LL(0x8c008c8c008c8c00), LL(0xd200d2d200d2d200), LL(0x9000909000909000), 
			LL(0xf600f6f600f6f600), LL(0x0700070700070700), LL(0xa700a7a700a7a700), LL(0x2700272700272700), LL(0x8e008e8e008e8e00), LL(0xb200b2b200b2b200), LL(0x4900494900494900), LL(0xde00dede00dede00), 
			LL(0x4300434300434300), LL(0x5c005c5c005c5c00), LL(0xd700d7d700d7d700), LL(0xc700c7c700c7c700), LL(0x3e003e3e003e3e00), LL(0xf500f5f500f5f500), LL(0x8f008f8f008f8f00), LL(0x6700676700676700), 
			LL(0x1f001f1f001f1f00), LL(0x1800181800181800), LL(0x6e006e6e006e6e00), LL(0xaf00afaf00afaf00), LL(0x2f002f2f002f2f00), LL(0xe200e2e200e2e200), LL(0x8500858500858500), LL(0x0d000d0d000d0d00), 
			LL(0x5300535300535300), LL(0xf000f0f000f0f000), LL(0x9c009c9c009c9c00), LL(0x6500656500656500), LL(0xea00eaea00eaea00), LL(0xa300a3a300a3a300), LL(0xae00aeae00aeae00), LL(0x9e009e9e009e9e00), 
			LL(0xec00ecec00ecec00), LL(0x8000808000808000), LL(0x2d002d2d002d2d00), LL(0x6b006b6b006b6b00), LL(0xa800a8a800a8a800), LL(0x2b002b2b002b2b00), LL(0x3600363600363600), LL(0xa600a6a600a6a600), 
			LL(0xc500c5c500c5c500), LL(0x8600868600868600), LL(0x4d004d4d004d4d00), LL(0x3300333300333300), LL(0xfd00fdfd00fdfd00), LL(0x6600666600666600), LL(0x5800585800585800), LL(0x9600969600969600), 
			LL(0x3a003a3a003a3a00), LL(0x0900090900090900), LL(0x9500959500959500), LL(0x1000101000101000), LL(0x7800787800787800), LL(0xd800d8d800d8d800), LL(0x4200424200424200), LL(0xcc00cccc00cccc00), 
			LL(0xef00efef00efef00), LL(0x2600262600262600), LL(0xe500e5e500e5e500), LL(0x6100616100616100), LL(0x1a001a1a001a1a00), LL(0x3f003f3f003f3f00), LL(0x3b003b3b003b3b00), LL(0x8200828200828200), 
			LL(0xb600b6b600b6b600), LL(0xdb00dbdb00dbdb00), LL(0xd400d4d400d4d400), LL(0x9800989800989800), LL(0xe800e8e800e8e800), LL(0x8b008b8b008b8b00), LL(0x0200020200020200), LL(0xeb00ebeb00ebeb00), 
			LL(0x0a000a0a000a0a00), LL(0x2c002c2c002c2c00), LL(0x1d001d1d001d1d00), LL(0xb000b0b000b0b000), LL(0x6f006f6f006f6f00), LL(0x8d008d8d008d8d00), LL(0x8800888800888800), LL(0x0e000e0e000e0e00), 
			LL(0x1900191900191900), LL(0x8700878700878700), LL(0x4e004e4e004e4e00), LL(0x0b000b0b000b0b00), LL(0xa900a9a900a9a900), LL(0x0c000c0c000c0c00), LL(0x7900797900797900), LL(0x1100111100111100), 
			LL(0x7f007f7f007f7f00), LL(0x2200222200222200), LL(0xe700e7e700e7e700), LL(0x5900595900595900), LL(0xe100e1e100e1e100), LL(0xda00dada00dada00), LL(0x3d003d3d003d3d00), LL(0xc800c8c800c8c800), 
			LL(0x1200121200121200), LL(0x0400040400040400), LL(0x7400747400747400), LL(0x5400545400545400), LL(0x3000303000303000), LL(0x7e007e7e007e7e00), LL(0xb400b4b400b4b400), LL(0x2800282800282800), 
			LL(0x5500555500555500), LL(0x6800686800686800), LL(0x5000505000505000), LL(0xbe00bebe00bebe00), LL(0xd000d0d000d0d000), LL(0xc400c4c400c4c400), LL(0x3100313100313100), LL(0xcb00cbcb00cbcb00), 
			LL(0x2a002a2a002a2a00), LL(0xad00adad00adad00), LL(0x0f000f0f000f0f00), LL(0xca00caca00caca00), LL(0x7000707000707000), LL(0xff00ffff00ffff00), LL(0x3200323200323200), LL(0x6900696900696900), 
			LL(0x0800080800080800), LL(0x6200626200626200), LL(0x0000000000000000), LL(0x2400242400242400), LL(0xd100d1d100d1d100), LL(0xfb00fbfb00fbfb00), LL(0xba00baba00baba00), LL(0xed00eded00eded00), 
			LL(0x4500454500454500), LL(0x8100818100818100), LL(0x7300737300737300), LL(0x6d006d6d006d6d00), LL(0x8400848400848400), LL(0x9f009f9f009f9f00), LL(0xee00eeee00eeee00), LL(0x4a004a4a004a4a00), 
			LL(0xc300c3c300c3c300), LL(0x2e002e2e002e2e00), LL(0xc100c1c100c1c100), LL(0x0100010100010100), LL(0xe600e6e600e6e600), LL(0x2500252500252500), LL(0x4800484800484800), LL(0x9900999900999900), 
			LL(0xb900b9b900b9b900), LL(0xb300b3b300b3b300), LL(0x7b007b7b007b7b00), LL(0xf900f9f900f9f900), LL(0xce00cece00cece00), LL(0xbf00bfbf00bfbf00), LL(0xdf00dfdf00dfdf00), LL(0x7100717100717100), 
			LL(0x2900292900292900), LL(0xcd00cdcd00cdcd00), LL(0x6c006c6c006c6c00), LL(0x1300131300131300), LL(0x6400646400646400), LL(0x9b009b9b009b9b00), LL(0x6300636300636300), LL(0x9d009d9d009d9d00), 
			LL(0xc000c0c000c0c000), LL(0x4b004b4b004b4b00), LL(0xb700b7b700b7b700), LL(0xa500a5a500a5a500), LL(0x8900898900898900), LL(0x5f005f5f005f5f00), LL(0xb100b1b100b1b100), LL(0x1700171700171700), 
			LL(0xf400f4f400f4f400), LL(0xbc00bcbc00bcbc00), LL(0xd300d3d300d3d300), LL(0x4600464600464600), LL(0xcf00cfcf00cfcf00), LL(0x3700373700373700), LL(0x5e005e5e005e5e00), LL(0x4700474700474700), 
			LL(0x9400949400949400), LL(0xfa00fafa00fafa00), LL(0xfc00fcfc00fcfc00), LL(0x5b005b5b005b5b00), LL(0x9700979700979700), LL(0xfe00fefe00fefe00), LL(0x5a005a5a005a5a00), LL(0xac00acac00acac00), 
			LL(0x3c003c3c003c3c00), LL(0x4c004c4c004c4c00), LL(0x0300030300030300), LL(0x3500353500353500), LL(0xf300f3f300f3f300), LL(0x2300232300232300), LL(0xb800b8b800b8b800), LL(0x5d005d5d005d5d00), 
			LL(0x6a006a6a006a6a00), LL(0x9200929200929200), LL(0xd500d5d500d5d500), LL(0x2100212100212100), LL(0x4400444400444400), LL(0x5100515100515100), LL(0xc600c6c600c6c600), LL(0x7d007d7d007d7d00), 
			LL(0x3900393900393900), LL(0x8300838300838300), LL(0xdc00dcdc00dcdc00), LL(0xaa00aaaa00aaaa00), LL(0x7c007c7c007c7c00), LL(0x7700777700777700), LL(0x5600565600565600), LL(0x0500050500050500), 
			LL(0x1b001b1b001b1b00), LL(0xa400a4a400a4a400), LL(0x1500151500151500), LL(0x3400343400343400), LL(0x1e001e1e001e1e00), LL(0x1c001c1c001c1c00), LL(0xf800f8f800f8f800), LL(0x5200525200525200), 
			LL(0x2000202000202000), LL(0x1400141400141400), LL(0xe900e9e900e9e900), LL(0xbd00bdbd00bdbd00), LL(0xdd00dddd00dddd00), LL(0xe400e4e400e4e400), LL(0xa100a1a100a1a100), LL(0xe000e0e000e0e000), 
			LL(0x8a008a8a008a8a00), LL(0xf100f1f100f1f100), LL(0xd600d6d600d6d600), LL(0x7a007a7a007a7a00), LL(0xbb00bbbb00bbbb00), LL(0xe300e3e300e3e300), LL(0x4000404000404000), LL(0x4f004f4f004f4f00)
		},
		{
			LL(0x7070007000007070), LL(0x2c2c002c00002c2c), LL(0xb3b300b30000b3b3), LL(0xc0c000c00000c0c0), LL(0xe4e400e40000e4e4), LL(0x5757005700005757), LL(0xeaea00ea0000eaea), LL(0xaeae00ae0000aeae), 
			LL(0x2323002300002323), LL(0x6b6b006b00006b6b), LL(0x4545004500004545), LL(0xa5a500a50000a5a5), LL(0xeded00ed0000eded), LL(0x4f4f004f00004f4f), LL(0x1d1d001d00001d1d), LL(0x9292009200009292), 
			LL(0x8686008600008686), LL(0xafaf00af0000afaf), LL(0x7c7c007c00007c7c), LL(0x1f1f001f00001f1f), LL(0x3e3e003e00003e3e), LL(0xdcdc00dc0000dcdc), LL(0x5e5e005e00005e5e), LL(0x0b0b000b00000b0b), 
			LL(0xa6a600a60000a6a6), LL(0x3939003900003939), LL(0xd5d500d50000d5d5), LL(0x5d5d005d00005d5d), LL(0xd9d900d90000d9d9), LL(0x5a5a005a00005a5a), LL(0x5151005100005151), LL(0x6c6c006c00006c6c), 
			LL(0x8b8b008b00008b8b), LL(0x9a9a009a00009a9a), LL(0xfbfb00fb0000fbfb), LL(0xb0b000b00000b0b0), LL(0x7474007400007474), LL(0x2b2b002b00002b2b), LL(0xf0f000f00000f0f0), LL(0x8484008400008484), 
			LL(0xdfdf00df0000dfdf), LL(0xcbcb00cb0000cbcb), LL(0x3434003400003434), LL(0x7676007600007676), LL(0x6d6d006d00006d6d), LL(0xa9a900a90000a9a9), LL(0xd1d100d10000d1d1), LL(0x0404000400000404), 
			LL(0x1414001400001414), LL(0x3a3a003a00003a3a), LL(0xdede00de0000dede), LL(0x1111001100001111), LL(0x3232003200003232), LL(0x9c9c009c00009c9c), LL(0x5353005300005353), LL(0xf2f200f20000f2f2), 
			LL(0xfefe00fe0000fefe), LL(0xcfcf00cf0000cfcf), LL(0xc3c300c30000c3c3), LL(0x7a7a007a00007a7a), LL(0x2424002400002424), LL(0xe8e800e80000e8e8), LL(0x6060006000006060), LL(0x6969006900006969), 
			LL(0xaaaa00aa0000aaaa), LL(0xa0a000a00000a0a0), LL(0xa1a100a10000a1a1), LL(0x6262006200006262), LL(0x5454005400005454), LL(0x1e1e001e00001e1e), LL(0xe0e000e00000e0e0), LL(0x6464006400006464), 
			LL(0x1010001000001010), LL(0x0000000000000000), LL(0xa3a300a30000a3a3), LL(0x7575007500007575), LL(0x8a8a008a00008a8a), LL(0xe6e600e60000e6e6), LL(0x0909000900000909), LL(0xdddd00dd0000dddd), 
			LL(0x8787008700008787), LL(0x8383008300008383), LL(0xcdcd00cd0000cdcd), LL(0x9090009000009090), LL(0x7373007300007373), LL(0xf6f600f60000f6f6), LL(0x9d9d009d00009d9d), LL(0xbfbf00bf0000bfbf), 
			LL(0x5252005200005252), LL(0xd8d800d80000d8d8), LL(0xc8c800c80000c8c8), LL(0xc6c600c60000c6c6), LL(0x8181008100008181), LL(0x6f6f006f00006f6f), LL(0x1313001300001313), LL(0x6363006300006363), 
			LL(0xe9e900e90000e9e9), LL(0xa7a700a70000a7a7), LL(0x9f9f009f00009f9f), LL(0xbcbc00bc0000bcbc), LL(0x2929002900002929), LL(0xf9f900f90000f9f9), LL(0x2f2f002f00002f2f), LL(0xb4b400b40000b4b4), 
			LL(0x7878007800007878), LL(0x0606000600000606), LL(0xe7e700e70000e7e7), LL(0x7171007100007171), LL(0xd4d400d40000d4d4), LL(0xabab00ab0000abab), LL(0x8888008800008888), LL(0x8d8d008d00008d8d), 
			LL(0x7272007200007272), LL(0xb9b900b90000b9b9), LL(0xf8f800f80000f8f8), LL(0xacac00ac0000acac), LL(0x3636003600003636), LL(0x2a2a002a00002a2a), LL(0x3c3c003c00003c3c), LL(0xf1f100f10000f1f1), 
			LL(0x4040004000004040), LL(0xd3d300d30000d3d3), LL(0xbbbb00bb0000bbbb), LL(0x4343004300004343), LL(0x1515001500001515), LL(0xadad00ad0000adad), LL(0x7777007700007777), LL(0x8080008000008080), 
			LL(0x8282008200008282), LL(0xecec00ec0000ecec), LL(0x2727002700002727), LL(0xe5e500e50000e5e5), LL(0x8585008500008585), LL(0x3535003500003535), LL(0x0c0c000c00000c0c), LL(0x4141004100004141), 
			LL(0xefef00ef0000efef), LL(0x9393009300009393), LL(0x1919001900001919), LL(0x2121002100002121), LL(0x0e0e000e00000e0e), LL(0x4e4e004e00004e4e), LL(0x6565006500006565), LL(0xbdbd00bd0000bdbd), 
			LL(0xb8b800b80000b8b8), LL(0x8f8f008f00008f8f), LL(0xebeb00eb0000ebeb), LL(0xcece00ce0000cece), LL(0x3030003000003030), LL(0x5f5f005f00005f5f), LL(0xc5c500c50000c5c5), LL(0x1a1a001a00001a1a), 
			LL(0xe1e100e10000e1e1), LL(0xcaca00ca0000caca), LL(0x4747004700004747), LL(0x3d3d003d00003d3d), LL(0x0101000100000101), LL(0xd6d600d60000d6d6), LL(0x5656005600005656), LL(0x4d4d004d00004d4d), 
			LL(0x0d0d000d00000d0d), LL(0x6666006600006666), LL(0xcccc00cc0000cccc), LL(0x2d2d002d00002d2d), LL(0x1212001200001212), LL(0x2020002000002020), LL(0xb1b100b10000b1b1), LL(0x9999009900009999), 
			LL(0x4c4c004c00004c4c), LL(0xc2c200c20000c2c2), LL(0x7e7e007e00007e7e), LL(0x0505000500000505), LL(0xb7b700b70000b7b7), LL(0x3131003100003131), LL(0x1717001700001717), LL(0xd7d700d70000d7d7), 
			LL(0x5858005800005858), LL(0x6161006100006161), LL(0x1b1b001b00001b1b), LL(0x1c1c001c00001c1c), LL(0x0f0f000f00000f0f), LL(0x1616001600001616), LL(0x1818001800001818), LL(0x2222002200002222), 
			LL(0x4444004400004444), LL(0xb2b200b20000b2b2), LL(0xb5b500b50000b5b5), LL(0x9191009100009191), LL(0x0808000800000808), LL(0xa8a800a80000a8a8), LL(0xfcfc00fc0000fcfc), LL(0x5050005000005050), 
			LL(0xd0d000d00000d0d0), LL(0x7d7d007d00007d7d), LL(0x8989008900008989), LL(0x9797009700009797), LL(0x5b5b005b00005b5b), LL(0x9595009500009595), LL(0xffff00ff0000ffff), LL(0xd2d200d20000d2d2), 
			LL(0xc4c400c40000c4c4), LL(0x4848004800004848), LL(0xf7f700f70000f7f7), LL(0xdbdb00db0000dbdb), LL(0x0303000300000303), LL(0xdada00da0000dada), LL(0x3f3f003f00003f3f), LL(0x9494009400009494), 
			LL(0x5c5c005c00005c5c), LL(0x0202000200000202), LL(0x4a4a004a00004a4a), LL(0x3333003300003333), LL(0x6767006700006767), LL(0xf3f300f30000f3f3), LL(0x7f7f007f00007f7f), LL(0xe2e200e20000e2e2), 
			LL(0x9b9b009b00009b9b), LL(0x2626002600002626), LL(0x3737003700003737), LL(0x3b3b003b00003b3b), LL(0x9696009600009696), LL(0x4b4b004b00004b4b), LL(0xbebe00be0000bebe), LL(0x2e2e002e00002e2e), 
			LL(0x7979007900007979), LL(0x8c8c008c00008c8c), LL(0x6e6e006e00006e6e), LL(0x8e8e008e00008e8e), LL(0xf5f500f50000f5f5), LL(0xb6b600b60000b6b6), LL(0xfdfd00fd0000fdfd), LL(0x5959005900005959), 
			LL(0x9898009800009898), LL(0x6a6a006a00006a6a), LL(0x4646004600004646), LL(0xbaba00ba0000baba), LL(0x2525002500002525), LL(0x4242004200004242), LL(0xa2a200a20000a2a2), LL(0xfafa00fa0000fafa), 
			LL(0x0707000700000707), LL(0x5555005500005555), LL(0xeeee00ee0000eeee), LL(0x0a0a000a00000a0a), LL(0x4949004900004949), LL(0x6868006800006868), LL(0x3838003800003838), LL(0xa4a400a40000a4a4), 
			LL(0x2828002800002828), LL(0x7b7b007b00007b7b), LL(0xc9c900c90000c9c9), LL(0xc1c100c10000c1c1), LL(0xe3e300e30000e3e3), LL(0xf4f400f40000f4f4), LL(0xc7c700c70000c7c7), LL(0x9e9e009e00009e9e)
		},
		{
			LL(0x00e0e0e000e0e0e0), LL(0x0005050500050505), LL(0x0058585800585858), LL(0x00d9d9d900d9d9d9), LL(0x0067676700676767), LL(0x004e4e4e004e4e4e), LL(0x0081818100818181), LL(0x00cbcbcb00cbcbcb), 
			LL(0x00c9c9c900c9c9c9), LL(0x000b0b0b000b0b0b), LL(0x00aeaeae00aeaeae), LL(0x006a6a6a006a6a6a), LL(0x00d5d5d500d5d5d5), LL(0x0018181800181818), LL(0x005d5d5d005d5d5d), LL(0x0082828200828282), 
			LL(0x0046464600464646), LL(0x00dfdfdf00dfdfdf), LL(0x00d6d6d600d6d6d6), LL(0x0027272700272727), LL(0x008a8a8a008a8a8a), LL(0x0032323200323232), LL(0x004b4b4b004b4b4b), LL(0x0042424200424242), 
			LL(0x00dbdbdb00dbdbdb), LL(0x001c1c1c001c1c1c), LL(0x009e9e9e009e9e9e), LL(0x009c9c9c009c9c9c), LL(0x003a3a3a003a3a3a), LL(0x00cacaca00cacaca), LL(0x0025252500252525), LL(0x007b7b7b007b7b7b), 
			LL(0x000d0d0d000d0d0d), LL(0x0071717100717171), LL(0x005f5f5f005f5f5f), LL(0x001f1f1f001f1f1f), LL(0x00f8f8f800f8f8f8), LL(0x00d7d7d700d7d7d7), LL(0x003e3e3e003e3e3e), LL(0x009d9d9d009d9d9d), 
			LL(0x007c7c7c007c7c7c), LL(0x0060606000606060), LL(0x00b9b9b900b9b9b9), LL(0x00bebebe00bebebe), LL(0x00bcbcbc00bcbcbc), LL(0x008b8b8b008b8b8b), LL(0x0016161600161616), LL(0x0034343400343434), 
			LL(0x004d4d4d004d4d4d), LL(0x00c3c3c300c3c3c3), LL(0x0072727200727272), LL(0x0095959500959595), LL(0x00ababab00ababab), LL(0x008e8e8e008e8e8e), LL(0x00bababa00bababa), LL(0x007a7a7a007a7a7a), 
			LL(0x00b3b3b300b3b3b3), LL(0x0002020200020202), LL(0x00b4b4b400b4b4b4), LL(0x00adadad00adadad), LL(0x00a2a2a200a2a2a2), LL(0x00acacac00acacac), LL(0x00d8d8d800d8d8d8), LL(0x009a9a9a009a9a9a), 
			LL(0x0017171700171717), LL(0x001a1a1a001a1a1a), LL(0x0035353500353535), LL(0x00cccccc00cccccc), LL(0x00f7f7f700f7f7f7), LL(0x0099999900999999), LL(0x0061616100616161), LL(0x005a5a5a005a5a5a), 
			LL(0x00e8e8e800e8e8e8), LL(0x0024242400242424), LL(0x0056565600565656), LL(0x0040404000404040), LL(0x00e1e1e100e1e1e1), LL(0x0063636300636363), LL(0x0009090900090909), LL(0x0033333300333333), 
			LL(0x00bfbfbf00bfbfbf), LL(0x0098989800989898), LL(0x0097979700979797), LL(0x0085858500858585), LL(0x0068686800686868), LL(0x00fcfcfc00fcfcfc), LL(0x00ececec00ececec), LL(0x000a0a0a000a0a0a), 
			LL(0x00dadada00dadada), LL(0x006f6f6f006f6f6f), LL(0x0053535300535353), LL(0x0062626200626262), LL(0x00a3a3a300a3a3a3), LL(0x002e2e2e002e2e2e), LL(0x0008080800080808), LL(0x00afafaf00afafaf), 
			LL(0x0028282800282828), LL(0x00b0b0b000b0b0b0), LL(0x0074747400747474), LL(0x00c2c2c200c2c2c2), LL(0x00bdbdbd00bdbdbd), LL(0x0036363600363636), LL(0x0022222200222222), LL(0x0038383800383838), 
			LL(0x0064646400646464), LL(0x001e1e1e001e1e1e), LL(0x0039393900393939), LL(0x002c2c2c002c2c2c), LL(0x00a6a6a600a6a6a6), LL(0x0030303000303030), LL(0x00e5e5e500e5e5e5), LL(0x0044444400444444), 
			LL(0x00fdfdfd00fdfdfd), LL(0x0088888800888888), LL(0x009f9f9f009f9f9f), LL(0x0065656500656565), LL(0x0087878700878787), LL(0x006b6b6b006b6b6b), LL(0x00f4f4f400f4f4f4), LL(0x0023232300232323), 
			LL(0x0048484800484848), LL(0x0010101000101010), LL(0x00d1d1d100d1d1d1), LL(0x0051515100515151), LL(0x00c0c0c000c0c0c0), LL(0x00f9f9f900f9f9f9), LL(0x00d2d2d200d2d2d2), LL(0x00a0a0a000a0a0a0), 
			LL(0x0055555500555555), LL(0x00a1a1a100a1a1a1), LL(0x0041414100414141), LL(0x00fafafa00fafafa), LL(0x0043434300434343), LL(0x0013131300131313), LL(0x00c4c4c400c4c4c4), LL(0x002f2f2f002f2f2f), 
			LL(0x00a8a8a800a8a8a8), LL(0x00b6b6b600b6b6b6), LL(0x003c3c3c003c3c3c), LL(0x002b2b2b002b2b2b), LL(0x00c1c1c100c1c1c1), LL(0x00ffffff00ffffff), LL(0x00c8c8c800c8c8c8), LL(0x00a5a5a500a5a5a5), 
			LL(0x0020202000202020), LL(0x0089898900898989), LL(0x0000000000000000), LL(0x0090909000909090), LL(0x0047474700474747), LL(0x00efefef00efefef), LL(0x00eaeaea00eaeaea), LL(0x00b7b7b700b7b7b7), 
			LL(0x0015151500151515), LL(0x0006060600060606), LL(0x00cdcdcd00cdcdcd), LL(0x00b5b5b500b5b5b5), LL(0x0012121200121212), LL(0x007e7e7e007e7e7e), LL(0x00bbbbbb00bbbbbb), LL(0x0029292900292929), 
			LL(0x000f0f0f000f0f0f), LL(0x00b8b8b800b8b8b8), LL(0x0007070700070707), LL(0x0004040400040404), LL(0x009b9b9b009b9b9b), LL(0x0094949400949494), LL(0x0021212100212121), LL(0x0066666600666666), 
			LL(0x00e6e6e600e6e6e6), LL(0x00cecece00cecece), LL(0x00ededed00ededed), LL(0x00e7e7e700e7e7e7), LL(0x003b3b3b003b3b3b), LL(0x00fefefe00fefefe), LL(0x007f7f7f007f7f7f), LL(0x00c5c5c500c5c5c5), 
			LL(0x00a4a4a400a4a4a4), LL(0x0037373700373737), LL(0x00b1b1b100b1b1b1), LL(0x004c4c4c004c4c4c), LL(0x0091919100919191), LL(0x006e6e6e006e6e6e), LL(0x008d8d8d008d8d8d), LL(0x0076767600767676), 
			LL(0x0003030300030303), LL(0x002d2d2d002d2d2d), LL(0x00dedede00dedede), LL(0x0096969600969696), LL(0x0026262600262626), LL(0x007d7d7d007d7d7d), LL(0x00c6c6c600c6c6c6), LL(0x005c5c5c005c5c5c), 
			LL(0x00d3d3d300d3d3d3), LL(0x00f2f2f200f2f2f2), LL(0x004f4f4f004f4f4f), LL(0x0019191900191919), LL(0x003f3f3f003f3f3f), LL(0x00dcdcdc00dcdcdc), LL(0x0079797900797979), LL(0x001d1d1d001d1d1d), 
			LL(0x0052525200525252), LL(0x00ebebeb00ebebeb), LL(0x00f3f3f300f3f3f3), LL(0x006d6d6d006d6d6d), LL(0x005e5e5e005e5e5e), LL(0x00fbfbfb00fbfbfb), LL(0x0069696900696969), LL(0x00b2b2b200b2b2b2), 
			LL(0x00f0f0f000f0f0f0), LL(0x0031313100313131), LL(0x000c0c0c000c0c0c), LL(0x00d4d4d400d4d4d4), LL(0x00cfcfcf00cfcfcf), LL(0x008c8c8c008c8c8c), LL(0x00e2e2e200e2e2e2), LL(0x0075757500757575), 
			LL(0x00a9a9a900a9a9a9), LL(0x004a4a4a004a4a4a), LL(0x0057575700575757), LL(0x0084848400848484), LL(0x0011111100111111), LL(0x0045454500454545), LL(0x001b1b1b001b1b1b), LL(0x00f5f5f500f5f5f5), 
			LL(0x00e4e4e400e4e4e4), LL(0x000e0e0e000e0e0e), LL(0x0073737300737373), LL(0x00aaaaaa00aaaaaa), LL(0x00f1f1f100f1f1f1), LL(0x00dddddd00dddddd), LL(0x0059595900595959), LL(0x0014141400141414), 
			LL(0x006c6c6c006c6c6c), LL(0x0092929200929292), LL(0x0054545400545454), LL(0x00d0d0d000d0d0d0), LL(0x0078787800787878), LL(0x0070707000707070), LL(0x00e3e3e300e3e3e3), LL(0x0049494900494949), 
			LL(0x0080808000808080), LL(0x0050505000505050), LL(0x00a7a7a700a7a7a7), LL(0x00f6f6f600f6f6f6), LL(0x0077777700777777), LL(0x0093939300939393), LL(0x0086868600868686), LL(0x0083838300838383), 
			LL(0x002a2a2a002a2a2a), LL(0x00c7c7c700c7c7c7), LL(0x005b5b5b005b5b5b), LL(0x00e9e9e900e9e9e9), LL(0x00eeeeee00eeeeee), LL(0x008f8f8f008f8f8f), LL(0x0001010100010101), LL(0x003d3d3d003d3d3d)
		},
		{
			LL(0x3800383838003838), LL(0x4100414141004141), LL(0x1600161616001616), LL(0x7600767676007676), LL(0xd900d9d9d900d9d9), LL(0x9300939393009393), LL(0x6000606060006060), LL(0xf200f2f2f200f2f2), 
			LL(0x7200727272007272), LL(0xc200c2c2c200c2c2), LL(0xab00ababab00abab), LL(0x9a009a9a9a009a9a), LL(0x7500757575007575), LL(0x0600060606000606), LL(0x5700575757005757), LL(0xa000a0a0a000a0a0), 
			LL(0x9100919191009191), LL(0xf700f7f7f700f7f7), LL(0xb500b5b5b500b5b5), LL(0xc900c9c9c900c9c9), LL(0xa200a2a2a200a2a2), LL(0x8c008c8c8c008c8c), LL(0xd200d2d2d200d2d2), LL(0x9000909090009090), 
			LL(0xf600f6f6f600f6f6), LL(0x0700070707000707), LL(0xa700a7a7a700a7a7), LL(0x2700272727002727), LL(0x8e008e8e8e008e8e), LL(0xb200b2b2b200b2b2), LL(0x4900494949004949), LL(0xde00dedede00dede), 
			LL(0x4300434343004343), LL(0x5c005c5c5c005c5c), LL(0xd700d7d7d700d7d7), LL(0xc700c7c7c700c7c7), LL(0x3e003e3e3e003e3e), LL(0xf500f5f5f500f5f5), LL(0x8f008f8f8f008f8f), LL(0x6700676767006767), 
			LL(0x1f001f1f1f001f1f), LL(0x1800181818001818), LL(0x6e006e6e6e006e6e), LL(0xaf00afafaf00afaf), LL(0x2f002f2f2f002f2f), LL(0xe200e2e2e200e2e2), LL(0x8500858585008585), LL(0x0d000d0d0d000d0d), 
			LL(0x5300535353005353), LL(0xf000f0f0f000f0f0), LL(0x9c009c9c9c009c9c), LL(0x6500656565006565), LL(0xea00eaeaea00eaea), LL(0xa300a3a3a300a3a3), LL(0xae00aeaeae00aeae), LL(0x9e009e9e9e009e9e), 
			LL(0xec00ececec00ecec), LL(0x8000808080008080), LL(0x2d002d2d2d002d2d), LL(0x6b006b6b6b006b6b), LL(0xa800a8a8a800a8a8), LL(0x2b002b2b2b002b2b), LL(0x3600363636003636), LL(0xa600a6a6a600a6a6), 
			LL(0xc500c5c5c500c5c5), LL(0x8600868686008686), LL(0x4d004d4d4d004d4d), LL(0x3300333333003333), LL(0xfd00fdfdfd00fdfd), LL(0x6600666666006666), LL(0x5800585858005858), LL(0x9600969696009696), 
			LL(0x3a003a3a3a003a3a), LL(0x0900090909000909), LL(0x9500959595009595), LL(0x1000101010001010), LL(0x7800787878007878), LL(0xd800d8d8d800d8d8), LL(0x4200424242004242), LL(0xcc00cccccc00cccc), 
			LL(0xef00efefef00efef), LL(0x2600262626002626), LL(0xe500e5e5e500e5e5), LL(0x6100616161006161), LL(0x1a001a1a1a001a1a), LL(0x3f003f3f3f003f3f), LL(0x3b003b3b3b003b3b), LL(0x8200828282008282), 
			LL(0xb600b6b6b600b6b6), LL(0xdb00dbdbdb00dbdb), LL(0xd400d4d4d400d4d4), LL(0x9800989898009898), LL(0xe800e8e8e800e8e8), LL(0x8b008b8b8b008b8b), LL(0x0200020202000202), LL(0xeb00ebebeb00ebeb), 
			LL(0x0a000a0a0a000a0a), LL(0x2c002c2c2c002c2c), LL(0x1d001d1d1d001d1d), LL(0xb000b0b0b000b0b0), LL(0x6f006f6f6f006f6f), LL(0x8d008d8d8d008d8d), LL(0x8800888888008888), LL(0x0e000e0e0e000e0e), 
			LL(0x1900191919001919), LL(0x8700878787008787), LL(0x4e004e4e4e004e4e), LL(0x0b000b0b0b000b0b), LL(0xa900a9a9a900a9a9), LL(0x0c000c0c0c000c0c), LL(0x7900797979007979), LL(0x1100111111001111), 
			LL(0x7f007f7f7f007f7f), LL(0x2200222222002222), LL(0xe700e7e7e700e7e7), LL(0x5900595959005959), LL(0xe100e1e1e100e1e1), LL(0xda00dadada00dada), LL(0x3d003d3d3d003d3d), LL(0xc800c8c8c800c8c8), 
			LL(0x1200121212001212), LL(0x0400040404000404), LL(0x7400747474007474), LL(0x5400545454005454), LL(0x3000303030003030), LL(0x7e007e7e7e007e7e), LL(0xb400b4b4b400b4b4), LL(0x2800282828002828), 
			LL(0x5500555555005555), LL(0x6800686868006868), LL(0x5000505050005050), LL(0xbe00bebebe00bebe), LL(0xd000d0d0d000d0d0), LL(0xc400c4c4c400c4c4), LL(0x3100313131003131), LL(0xcb00cbcbcb00cbcb), 
			LL(0x2a002a2a2a002a2a), LL(0xad00adadad00adad), LL(0x0f000f0f0f000f0f), LL(0xca00cacaca00caca), LL(0x7000707070007070), LL(0xff00ffffff00ffff), LL(0x3200323232003232), LL(0x6900696969006969), 
			LL(0x0800080808000808), LL(0x6200626262006262), LL(0x0000000000000000), LL(0x2400242424002424), LL(0xd100d1d1d100d1d1), LL(0xfb00fbfbfb00fbfb), LL(0xba00bababa00baba), LL(0xed00ededed00eded), 
			LL(0x4500454545004545), LL(0x8100818181008181), LL(0x7300737373007373), LL(0x6d006d6d6d006d6d), LL(0x8400848484008484), LL(0x9f009f9f9f009f9f), LL(0xee00eeeeee00eeee), LL(0x4a004a4a4a004a4a), 
			LL(0xc300c3c3c300c3c3), LL(0x2e002e2e2e002e2e), LL(0xc100c1c1c100c1c1), LL(0x0100010101000101), LL(0xe600e6e6e600e6e6), LL(0x2500252525002525), LL(0x4800484848004848), LL(0x9900999999009999), 
			LL(0xb900b9b9b900b9b9), LL(0xb300b3b3b300b3b3), LL(0x7b007b7b7b007b7b), LL(0xf900f9f9f900f9f9), LL(0xce00cecece00cece), LL(0xbf00bfbfbf00bfbf), LL(0xdf00dfdfdf00dfdf), LL(0x7100717171007171), 
			LL(0x2900292929002929), LL(0xcd00cdcdcd00cdcd), LL(0x6c006c6c6c006c6c), LL(0x1300131313001313), LL(0x6400646464006464), LL(0x9b009b9b9b009b9b), LL(0x6300636363006363), LL(0x9d009d9d9d009d9d), 
			LL(0xc000c0c0c000c0c0), LL(0x4b004b4b4b004b4b), LL(0xb700b7b7b700b7b7), LL(0xa500a5a5a500a5a5), LL(0x8900898989008989), LL(0x5f005f5f5f005f5f), LL(0xb100b1b1b100b1b1), LL(0x1700171717001717), 
			LL(0xf400f4f4f400f4f4), LL(0xbc00bcbcbc00bcbc), LL(0xd300d3d3d300d3d3), LL(0x4600464646004646), LL(0xcf00cfcfcf00cfcf), LL(0x3700373737003737), LL(0x5e005e5e5e005e5e), LL(0x4700474747004747), 
			LL(0x9400949494009494), LL(0xfa00fafafa00fafa), LL(0xfc00fcfcfc00fcfc), LL(0x5b005b5b5b005b5b), LL(0x9700979797009797), LL(0xfe00fefefe00fefe), LL(0x5a005a5a5a005a5a), LL(0xac00acacac00acac), 
			LL(0x3c003c3c3c003c3c), LL(0x4c004c4c4c004c4c), LL(0x0300030303000303), LL(0x3500353535003535), LL(0xf300f3f3f300f3f3), LL(0x2300232323002323), LL(0xb800b8b8b800b8b8), LL(0x5d005d5d5d005d5d), 
			LL(0x6a006a6a6a006a6a), LL(0x9200929292009292), LL(0xd500d5d5d500d5d5), LL(0x2100212121002121), LL(0x4400444444004444), LL(0x5100515151005151), LL(0xc600c6c6c600c6c6), LL(0x7d007d7d7d007d7d), 
			LL(0x3900393939003939), LL(0x8300838383008383), LL(0xdc00dcdcdc00dcdc), LL(0xaa00aaaaaa00aaaa), LL(0x7c007c7c7c007c7c), LL(0x7700777777007777), LL(0x5600565656005656), LL(0x0500050505000505), 
			LL(0x1b001b1b1b001b1b), LL(0xa400a4a4a400a4a4), LL(0x1500151515001515), LL(0x3400343434003434), LL(0x1e001e1e1e001e1e), LL(0x1c001c1c1c001c1c), LL(0xf800f8f8f800f8f8), LL(0x5200525252005252), 
			LL(0x2000202020002020), LL(0x1400141414001414), LL(0xe900e9e9e900e9e9), LL(0xbd00bdbdbd00bdbd), LL(0xdd00dddddd00dddd), LL(0xe400e4e4e400e4e4), LL(0xa100a1a1a100a1a1), LL(0xe000e0e0e000e0e0), 
			LL(0x8a008a8a8a008a8a), LL(0xf100f1f1f100f1f1), LL(0xd600d6d6d600d6d6), LL(0x7a007a7a7a007a7a), LL(0xbb00bbbbbb00bbbb), LL(0xe300e3e3e300e3e3), LL(0x4000404040004040), LL(0x4f004f4f4f004f4f)
		},
		{
			LL(0x7070007070700070), LL(0x2c2c002c2c2c002c), LL(0xb3b300b3b3b300b3), LL(0xc0c000c0c0c000c0), LL(0xe4e400e4e4e400e4), LL(0x5757005757570057), LL(0xeaea00eaeaea00ea), LL(0xaeae00aeaeae00ae), 
			LL(0x2323002323230023), LL(0x6b6b006b6b6b006b), LL(0x4545004545450045), LL(0xa5a500a5a5a500a5), LL(0xeded00ededed00ed), LL(0x4f4f004f4f4f004f), LL(0x1d1d001d1d1d001d), LL(0x9292009292920092), 
			LL(0x8686008686860086), LL(0xafaf00afafaf00af), LL(0x7c7c007c7c7c007c), LL(0x1f1f001f1f1f001f), LL(0x3e3e003e3e3e003e), LL(0xdcdc00dcdcdc00dc), LL(0x5e5e005e5e5e005e), LL(0x0b0b000b0b0b000b), 
			LL(0xa6a600a6a6a600a6), LL(0x3939003939390039), LL(0xd5d500d5d5d500d5), LL(0x5d5d005d5d5d005d), LL(0xd9d900d9d9d900d9), LL(0x5a5a005a5a5a005a), LL(0x5151005151510051), LL(0x6c6c006c6c6c006c), 
			LL(0x8b8b008b8b8b008b), LL(0x9a9a009a9a9a009a), LL(0xfbfb00fbfbfb00fb), LL(0xb0b000b0b0b000b0), LL(0x7474007474740074), LL(0x2b2b002b2b2b002b), LL(0xf0f000f0f0f000f0), LL(0x8484008484840084), 
			LL(0xdfdf00dfdfdf00df), LL(0xcbcb00cbcbcb00cb), LL(0x3434003434340034), LL(0x7676007676760076), LL(0x6d6d006d6d6d006d), LL(0xa9a900a9a9a900a9), LL(0xd1d100d1d1d100d1), LL(0x0404000404040004), 
			LL(0x1414001414140014), LL(0x3a3a003a3a3a003a), LL(0xdede00dedede00de), LL(0x1111001111110011), LL(0x3232003232320032), LL(0x9c9c009c9c9c009c), LL(0x5353005353530053), LL(0xf2f200f2f2f200f2), 
			LL(0xfefe00fefefe00fe), LL(0xcfcf00cfcfcf00cf), LL(0xc3c300c3c3c300c3), LL(0x7a7a007a7a7a007a), LL(0x2424002424240024), LL(0xe8e800e8e8e800e8), LL(0x6060006060600060), LL(0x6969006969690069), 
			LL(0xaaaa00aaaaaa00aa), LL(0xa0a000a0a0a000a0), LL(0xa1a100a1a1a100a1), LL(0x6262006262620062), LL(0x5454005454540054), LL(0x1e1e001e1e1e001e), LL(0xe0e000e0e0e000e0), LL(0x6464006464640064), 
			LL(0x1010001010100010), LL(0x0000000000000000), LL(0xa3a300a3a3a300a3), LL(0x7575007575750075), LL(0x8a8a008a8a8a008a), LL(0xe6e600e6e6e600e6), LL(0x0909000909090009), LL(0xdddd00dddddd00dd), 
			LL(0x8787008787870087), LL(0x8383008383830083), LL(0xcdcd00cdcdcd00cd), LL(0x9090009090900090), LL(0x7373007373730073), LL(0xf6f600f6f6f600f6), LL(0x9d9d009d9d9d009d), LL(0xbfbf00bfbfbf00bf), 
			LL(0x5252005252520052), LL(0xd8d800d8d8d800d8), LL(0xc8c800c8c8c800c8), LL(0xc6c600c6c6c600c6), LL(0x8181008181810081), LL(0x6f6f006f6f6f006f), LL(0x1313001313130013), LL(0x6363006363630063), 
			LL(0xe9e900e9e9e900e9), LL(0xa7a700a7a7a700a7), LL(0x9f9f009f9f9f009f), LL(0xbcbc00bcbcbc00bc), LL(0x2929002929290029), LL(0xf9f900f9f9f900f9), LL(0x2f2f002f2f2f002f), LL(0xb4b400b4b4b400b4), 
			LL(0x7878007878780078), LL(0x0606000606060006), LL(0xe7e700e7e7e700e7), LL(0x7171007171710071), LL(0xd4d400d4d4d400d4), LL(0xabab00ababab00ab), LL(0x8888008888880088), LL(0x8d8d008d8d8d008d), 
			LL(0x7272007272720072), LL(0xb9b900b9b9b900b9), LL(0xf8f800f8f8f800f8), LL(0xacac00acacac00ac), LL(0x3636003636360036), LL(0x2a2a002a2a2a002a), LL(0x3c3c003c3c3c003c), LL(0xf1f100f1f1f100f1), 
			LL(0x4040004040400040), LL(0xd3d300d3d3d300d3), LL(0xbbbb00bbbbbb00bb), LL(0x4343004343430043), LL(0x1515001515150015), LL(0xadad00adadad00ad), LL(0x7777007777770077), LL(0x8080008080800080), 
			LL(0x8282008282820082), LL(0xecec00ececec00ec), LL(0x2727002727270027), LL(0xe5e500e5e5e500e5), LL(0x8585008585850085), LL(0x3535003535350035), LL(0x0c0c000c0c0c000c), LL(0x4141004141410041), 
			LL(0xefef00efefef00ef), LL(0x9393009393930093), LL(0x1919001919190019), LL(0x2121002121210021), LL(0x0e0e000e0e0e000e), LL(0x4e4e004e4e4e004e), LL(0x6565006565650065), LL(0xbdbd00bdbdbd00bd), 
			LL(0xb8b800b8b8b800b8), LL(0x8f8f008f8f8f008f), LL(0xebeb00ebebeb00eb), LL(0xcece00cecece00ce), LL(0x3030003030300030), LL(0x5f5f005f5f5f005f), LL(0xc5c500c5c5c500c5), LL(0x1a1a001a1a1a001a), 
			LL(0xe1e100e1e1e100e1), LL(0xcaca00cacaca00ca), LL(0x4747004747470047), LL(0x3d3d003d3d3d003d), LL(0x0101000101010001), LL(0xd6d600d6d6d600d6), LL(0x5656005656560056), LL(0x4d4d004d4d4d004d), 
			LL(0x0d0d000d0d0d000d), LL(0x6666006666660066), LL(0xcccc00cccccc00cc), LL(0x2d2d002d2d2d002d), LL(0x1212001212120012), LL(0x2020002020200020), LL(0xb1b100b1b1b100b1), LL(0x9999009999990099), 
			LL(0x4c4c004c4c4c004c), LL(0xc2c200c2c2c200c2), LL(0x7e7e007e7e7e007e), LL(0x0505000505050005), LL(0xb7b700b7b7b700b7), LL(0x3131003131310031), LL(0x1717001717170017), LL(0xd7d700d7d7d700d7), 
			LL(0x5858005858580058), LL(0x6161006161610061), LL(0x1b1b001b1b1b001b), LL(0x1c1c001c1c1c001c), LL(0x0f0f000f0f0f000f), LL(0x1616001616160016), LL(0x1818001818180018), LL(0x2222002222220022), 
			LL(0x4444004444440044), LL(0xb2b200b2b2b200b2), LL(0xb5b500b5b5b500b5), LL(0x9191009191910091), LL(0x0808000808080008), LL(0xa8a800a8a8a800a8), LL(0xfcfc00fcfcfc00fc), LL(0x5050005050500050), 
			LL(0xd0d000d0d0d000d0), LL(0x7d7d007d7d7d007d), LL(0x8989008989890089), LL(0x9797009797970097), LL(0x5b5b005b5b5b005b), LL(0x9595009595950095), LL(0xffff00ffffff00ff), LL(0xd2d200d2d2d200d2), 
			LL(0xc4c400c4c4c400c4), LL(0x4848004848480048), LL(0xf7f700f7f7f700f7), LL(0xdbdb00dbdbdb00db), LL(0x0303000303030003), LL(0xdada00dadada00da), LL(0x3f3f003f3f3f003f), LL(0x9494009494940094), 
			LL(0x5c5c005c5c5c005c), LL(0x0202000202020002), LL(0x4a4a004a4a4a004a), LL(0x3333003333330033), LL(0x6767006767670067), LL(0xf3f300f3f3f300f3), LL(0x7f7f007f7f7f007f), LL(0xe2e200e2e2e200e2), 
			LL(0x9b9b009b9b9b009b), LL(0x2626002626260026), LL(0x3737003737370037), LL(0x3b3b003b3b3b003b), LL(0x9696009696960096), LL(0x4b4b004b4b4b004b), LL(0xbebe00bebebe00be), LL(0x2e2e002e2e2e002e), 
			LL(0x7979007979790079), LL(0x8c8c008c8c8c008c), LL(0x6e6e006e6e6e006e), LL(0x8e8e008e8e8e008e), LL(0xf5f500f5f5f500f5), LL(0xb6b600b6b6b600b6), LL(0xfdfd00fdfdfd00fd), LL(0x5959005959590059), 
			LL(0x9898009898980098), LL(0x6a6a006a6a6a006a), LL(0x4646004646460046), LL(0xbaba00bababa00ba), LL(0x2525002525250025), LL(0x4242004242420042), LL(0xa2a200a2a2a200a2), LL(0xfafa00fafafa00fa), 
			LL(0x0707000707070007), LL(0x5555005555550055), LL(0xeeee00eeeeee00ee), LL(0x0a0a000a0a0a000a), LL(0x4949004949490049), LL(0x6868006868680068), LL(0x3838003838380038), LL(0xa4a400a4a4a400a4), 
			LL(0x2828002828280028), LL(0x7b7b007b7b7b007b), LL(0xc9c900c9c9c900c9), LL(0xc1c100c1c1c100c1), LL(0xe3e300e3e3e300e3), LL(0xf4f400f4f4f400f4), LL(0xc7c700c7c7c700c7), LL(0x9e9e009e9e9e009e)
		},
		{
			LL(0x7070700070707000), LL(0x8282820082828200), LL(0x2c2c2c002c2c2c00), LL(0xececec00ececec00), LL(0xb3b3b300b3b3b300), LL(0x2727270027272700), LL(0xc0c0c000c0c0c000), LL(0xe5e5e500e5e5e500), 
			LL(0xe4e4e400e4e4e400), LL(0x8585850085858500), LL(0x5757570057575700), LL(0x3535350035353500), LL(0xeaeaea00eaeaea00), LL(0x0c0c0c000c0c0c00), LL(0xaeaeae00aeaeae00), LL(0x4141410041414100), 
			LL(0x2323230023232300), LL(0xefefef00efefef00), LL(0x6b6b6b006b6b6b00), LL(0x9393930093939300), LL(0x4545450045454500), LL(0x1919190019191900), LL(0xa5a5a500a5a5a500), LL(0x2121210021212100), 
			LL(0xededed00ededed00), LL(0x0e0e0e000e0e0e00), LL(0x4f4f4f004f4f4f00), LL(0x4e4e4e004e4e4e00), LL(0x1d1d1d001d1d1d00), LL(0x6565650065656500), LL(0x9292920092929200), LL(0xbdbdbd00bdbdbd00), 
			LL(0x8686860086868600), LL(0xb8b8b800b8b8b800), LL(0xafafaf00afafaf00), LL(0x8f8f8f008f8f8f00), LL(0x7c7c7c007c7c7c00), LL(0xebebeb00ebebeb00), LL(0x1f1f1f001f1f1f00), LL(0xcecece00cecece00), 
			LL(0x3e3e3e003e3e3e00), LL(0x3030300030303000), LL(0xdcdcdc00dcdcdc00), LL(0x5f5f5f005f5f5f00), LL(0x5e5e5e005e5e5e00), LL(0xc5c5c500c5c5c500), LL(0x0b0b0b000b0b0b00), LL(0x1a1a1a001a1a1a00), 
			LL(0xa6a6a600a6a6a600), LL(0xe1e1e100e1e1e100), LL(0x3939390039393900), LL(0xcacaca00cacaca00), LL(0xd5d5d500d5d5d500), LL(0x4747470047474700), LL(0x5d5d5d005d5d5d00), LL(0x3d3d3d003d3d3d00), 
			LL(0xd9d9d900d9d9d900), LL(0x0101010001010100), LL(0x5a5a5a005a5a5a00), LL(0xd6d6d600d6d6d600), LL(0x5151510051515100), LL(0x5656560056565600), LL(0x6c6c6c006c6c6c00), LL(0x4d4d4d004d4d4d00), 
			LL(0x8b8b8b008b8b8b00), LL(0x0d0d0d000d0d0d00), LL(0x9a9a9a009a9a9a00), LL(0x6666660066666600), LL(0xfbfbfb00fbfbfb00), LL(0xcccccc00cccccc00), LL(0xb0b0b000b0b0b000), LL(0x2d2d2d002d2d2d00), 
			LL(0x7474740074747400), LL(0x1212120012121200), LL(0x2b2b2b002b2b2b00), LL(0x2020200020202000), LL(0xf0f0f000f0f0f000), LL(0xb1b1b100b1b1b100), LL(0x8484840084848400), LL(0x9999990099999900), 
			LL(0xdfdfdf00dfdfdf00), LL(0x4c4c4c004c4c4c00), LL(0xcbcbcb00cbcbcb00), LL(0xc2c2c200c2c2c200), LL(0x3434340034343400), LL(0x7e7e7e007e7e7e00), LL(0x7676760076767600), LL(0x0505050005050500), 
			LL(0x6d6d6d006d6d6d00), LL(0xb7b7b700b7b7b700), LL(0xa9a9a900a9a9a900), LL(0x3131310031313100), LL(0xd1d1d100d1d1d100), LL(0x1717170017171700), LL(0x0404040004040400), LL(0xd7d7d700d7d7d700), 
			LL(0x1414140014141400), LL(0x5858580058585800), LL(0x3a3a3a003a3a3a00), LL(0x6161610061616100), LL(0xdedede00dedede00), LL(0x1b1b1b001b1b1b00), LL(0x1111110011111100), LL(0x1c1c1c001c1c1c00), 
			LL(0x3232320032323200), LL(0x0f0f0f000f0f0f00), LL(0x9c9c9c009c9c9c00), LL(0x1616160016161600), LL(0x5353530053535300), LL(0x1818180018181800), LL(0xf2f2f200f2f2f200), LL(0x2222220022222200), 
			LL(0xfefefe00fefefe00), LL(0x4444440044444400), LL(0xcfcfcf00cfcfcf00), LL(0xb2b2b200b2b2b200), LL(0xc3c3c300c3c3c300), LL(0xb5b5b500b5b5b500), LL(0x7a7a7a007a7a7a00), LL(0x9191910091919100), 
			LL(0x2424240024242400), LL(0x0808080008080800), LL(0xe8e8e800e8e8e800), LL(0xa8a8a800a8a8a800), LL(0x6060600060606000), LL(0xfcfcfc00fcfcfc00), LL(0x6969690069696900), LL(0x5050500050505000), 
			LL(0xaaaaaa00aaaaaa00), LL(0xd0d0d000d0d0d000), LL(0xa0a0a000a0a0a000), LL(0x7d7d7d007d7d7d00), LL(0xa1a1a100a1a1a100), LL(0x8989890089898900), LL(0x6262620062626200), LL(0x9797970097979700), 
			LL(0x5454540054545400), LL(0x5b5b5b005b5b5b00), LL(0x1e1e1e001e1e1e00), LL(0x9595950095959500), LL(0xe0e0e000e0e0e000), LL(0xffffff00ffffff00), LL(0x6464640064646400), LL(0xd2d2d200d2d2d200), 
			LL(0x1010100010101000), LL(0xc4c4c400c4c4c400), LL(0x0000000000000000), LL(0x4848480048484800), LL(0xa3a3a300a3a3a300), LL(0xf7f7f700f7f7f700), LL(0x7575750075757500), LL(0xdbdbdb00dbdbdb00), 
			LL(0x8a8a8a008a8a8a00), LL(0x0303030003030300), LL(0xe6e6e600e6e6e600), LL(0xdadada00dadada00), LL(0x0909090009090900), LL(0x3f3f3f003f3f3f00), LL(0xdddddd00dddddd00), LL(0x9494940094949400), 
			LL(0x8787870087878700), LL(0x5c5c5c005c5c5c00), LL(0x8383830083838300), LL(0x0202020002020200), LL(0xcdcdcd00cdcdcd00), LL(0x4a4a4a004a4a4a00), LL(0x9090900090909000), LL(0x3333330033333300), 
			LL(0x7373730073737300), LL(0x6767670067676700), LL(0xf6f6f600f6f6f600), LL(0xf3f3f300f3f3f300), LL(0x9d9d9d009d9d9d00), LL(0x7f7f7f007f7f7f00), LL(0xbfbfbf00bfbfbf00), LL(0xe2e2e200e2e2e200), 
			LL(0x5252520052525200), LL(0x9b9b9b009b9b9b00), LL(0xd8d8d800d8d8d800), LL(0x2626260026262600), LL(0xc8c8c800c8c8c800), LL(0x3737370037373700), LL(0xc6c6c600c6c6c600), LL(0x3b3b3b003b3b3b00), 
			LL(0x8181810081818100), LL(0x9696960096969600), LL(0x6f6f6f006f6f6f00), LL(0x4b4b4b004b4b4b00), LL(0x1313130013131300), LL(0xbebebe00bebebe00), LL(0x6363630063636300), LL(0x2e2e2e002e2e2e00), 
			LL(0xe9e9e900e9e9e900), LL(0x7979790079797900), LL(0xa7a7a700a7a7a700), LL(0x8c8c8c008c8c8c00), LL(0x9f9f9f009f9f9f00), LL(0x6e6e6e006e6e6e00), LL(0xbcbcbc00bcbcbc00), LL(0x8e8e8e008e8e8e00), 
			LL(0x2929290029292900), LL(0xf5f5f500f5f5f500), LL(0xf9f9f900f9f9f900), LL(0xb6b6b600b6b6b600), LL(0x2f2f2f002f2f2f00), LL(0xfdfdfd00fdfdfd00), LL(0xb4b4b400b4b4b400), LL(0x5959590059595900), 
			LL(0x7878780078787800), LL(0x9898980098989800), LL(0x0606060006060600), LL(0x6a6a6a006a6a6a00), LL(0xe7e7e700e7e7e700), LL(0x4646460046464600), LL(0x7171710071717100), LL(0xbababa00bababa00), 
			LL(0xd4d4d400d4d4d400), LL(0x2525250025252500), LL(0xababab00ababab00), LL(0x4242420042424200), LL(0x8888880088888800), LL(0xa2a2a200a2a2a200), LL(0x8d8d8d008d8d8d00), LL(0xfafafa00fafafa00), 
			LL(0x7272720072727200), LL(0x0707070007070700), LL(0xb9b9b900b9b9b900), LL(0x5555550055555500), LL(0xf8f8f800f8f8f800), LL(0xeeeeee00eeeeee00), LL(0xacacac00acacac00), LL(0x0a0a0a000a0a0a00), 
			LL(0x3636360036363600), LL(0x4949490049494900), LL(0x2a2a2a002a2a2a00), LL(0x6868680068686800), LL(0x3c3c3c003c3c3c00), LL(0x3838380038383800), LL(0xf1f1f100f1f1f100), LL(0xa4a4a400a4a4a400), 
			LL(0x4040400040404000), LL(0x2828280028282800), LL(0xd3d3d300d3d3d300), LL(0x7b7b7b007b7b7b00), LL(0xbbbbbb00bbbbbb00), LL(0xc9c9c900c9c9c900), LL(0x4343430043434300), LL(0xc1c1c100c1c1c100), 
			LL(0x1515150015151500), LL(0xe3e3e300e3e3e300), LL(0xadadad00adadad00), LL(0xf4f4f400f4f4f400), LL(0x7777770077777700), LL(0xc7c7c700c7c7c700), LL(0x8080800080808000), LL(0x9e9e9e009e9e9e00)
		}
	};

	const uint64 KS[] = { LL(0xA09E667F3BCC908B), LL(0xB67AE8584CAA73B2), LL(0xC6EF372FE94F82BE), LL(0x54FF53A5F1D36F1C), LL(0x10E527FADE682D1D), LL(0xB05688C2B3E6C1FD) };

	VC_INLINE uint64 F(uint64 x, uint64 k)
	{
		uint64 t = x ^ k;
		return S[0][(t >> 56) & 0xFF] ^ S[1][(t >> 48) & 0xFF] ^ S[2][(t >> 40) & 0xFF] ^ S[3][(t >> 32) & 0xFF] ^ S[4][(t >> 24) & 0xFF] ^ S[5][(t >> 16) & 0xFF] ^ S[6][(t >> 8) & 0xFF] ^ S[7][t & 0xFF];
	}

	VC_INLINE uint64 FL(uint64 x, uint64 k)
	{
		uint32 xl = (uint32)(x >> 32);
		uint32 xr = (uint32)(x);
		uint32 kl = (uint32)(k >> 32);
		uint32 kr = (uint32)(k);

		uint32 yr = rotl32(xl & kl, 1) ^ xr;
		uint32 yl = (yr | kr) ^ xl;
		return (((uint64)(yl)) << 32) | yr;
	}

	VC_INLINE uint64 FL1(uint64 y, uint64 k)
	{
		uint32 yl = (uint32)(y >> 32);
		uint32 yr = (uint32)(y);
		uint32 kl = (uint32)(k >> 32);
		uint32 kr = (uint32)(k);

		uint32 xl = (yr | kr) ^ yl;
		uint32 xr = rotl32(xl & kl, 1) ^ yr;
		return (((uint64)(xl)) << 32) | xr;
	}

	VC_INLINE uint64 rotl128l(uint64 l, uint64 r, int n)
	{
		return (l << n) | (r >> (64 - n));
	}

	VC_INLINE uint64 rotl128r(uint64 l, uint64 r, int n)
	{
		return (r << n) | (l >> (64 - n));
	}

	
void camellia_set_key(const unsigned __int8 key[], unsigned __int8 *ksPtr)
{
	uint64 *ks = (uint64 *) ksPtr;
	uint64 kll = bswap_64(*((uint64*)key));
	uint64 klr = bswap_64(*((uint64*)(key + 8)));
	uint64 krl = bswap_64(*((uint64*)(key + 16)));
	uint64 krr = bswap_64(*((uint64*)(key + 24)));

#ifdef CPPCRYPTO_DEBUG
	printf("kl: %016I64x %016I64x\n", kll, klr);
	printf("kr: %016I64x %016I64x\n", krl, krr);
#endif

	uint64 kal = kll ^ krl;
	uint64 kar = klr ^ krr;
	uint64 lp = kal;
	uint64 kbl, kbr;

	kal = kar ^ F(kal, KS[0]);
	kar = kal;
	kal = lp ^ F(kal, KS[1]);
	kal ^= kll;
	kar ^= klr;

	lp = kal;
	kal = kar ^ F(kal, KS[2]);
	kar = kal;
	kal = lp ^ F(kal, KS[3]);

#ifdef CPPCRYPTO_DEBUG
	printf("ka: %016I64x %016I64x\n", kal, kar);
#endif
	kbl = kal ^ krl;
	kbr = kar ^ krr;
	lp = kbl;
	kbl = kbr ^ F(kbl, KS[4]);
	kbr = kbl;
	kbl = lp ^ F(kbl, KS[5]);
#ifdef CPPCRYPTO_DEBUG
	printf("kb: %016I64x %016I64x\n", kbl, kbr);
#endif

	ks[0] = kll;
	ks[1] = klr;
	ks[2] = kbl;
	ks[3] = kbr;
	ks[4] = rotl128l(krl, krr, 15);
	ks[5] = rotl128r(krl, krr, 15);
	ks[6] = rotl128l(kal, kar, 15);
	ks[7] = rotl128r(kal, kar, 15);
	ks[8] = rotl128l(krl, krr, 30);
	ks[9] = rotl128r(krl, krr, 30);

	ks[10] = rotl128l(kbl, kbr, 30);
	ks[11] = rotl128r(kbl, kbr, 30);
	ks[12] = rotl128l(kll, klr, 45);
	ks[13] = rotl128r(kll, klr, 45);
	ks[14] = rotl128l(kal, kar, 45);
	ks[15] = rotl128r(kal, kar, 45);
	ks[16] = rotl128l(kll, klr, 60);
	ks[17] = rotl128r(kll, klr, 60);

	ks[18] = rotl128l(krl, krr, 60);
	ks[19] = rotl128r(krl, krr, 60);
	ks[20] = rotl128l(kbl, kbr, 60);
	ks[21] = rotl128r(kbl, kbr, 60);
	ks[22] = rotl128r(kll, klr, 13);
	ks[23] = rotl128l(kll, klr, 13);
	ks[24] = rotl128r(kal, kar, 13);
	ks[25] = rotl128l(kal, kar, 13);

	ks[26] = ks[9]; // rotl128r(krl, krr, 30);
	ks[27] = ks[8]; // rotl128l(krl, krr, 30);
	ks[28] = rotl128r(kal, kar, 30);
	ks[29] = rotl128l(kal, kar, 30);
	ks[30] = rotl128r(kll, klr, 47);
	ks[31] = rotl128l(kll, klr, 47);
	ks[32] = rotl128r(kbl, kbr, 47);
	ks[33] = rotl128l(kbl, kbr, 47);

#ifdef CPPCRYPTO_DEBUG
	for (int i = 0; i < 34; i++)
		printf("ks[%d]: %016I64x\n", i, ks[i]);
#endif
}

void camellia_encrypt(const unsigned __int8 *in, unsigned __int8 *out, unsigned __int8 *ksPtr)
{
	uint64 *ks = (uint64 *) ksPtr;
	uint64 l = bswap_64(*((uint64*)in)) ^ ks[0];
	uint64 r = bswap_64(*((uint64*)(in + 8))) ^ ks[1];

#ifdef CPPCRYPTO_DEBUG
	printf("r0: %016I64x %016I64x\n", l, r);
#endif
	// r = 1
	uint64 lp = l;
	l = r ^ F(l, ks[2]);
	// r = lp;
#ifdef CPPCRYPTO_DEBUG
	printf("r1: %016I64x %016I64x\n", l, lp);
#endif

	// r = 2
	r = l;
	l = lp ^ F(l, ks[3]);
#ifdef CPPCRYPTO_DEBUG
	printf("r2: %016I64x %016I64x\n", l, r);
#endif

	// r = 3
	lp = l;
	l = r ^ F(l, ks[4]);
#ifdef CPPCRYPTO_DEBUG
	printf("r3: %016I64x %016I64x\n", l, lp);
#endif

	// r = 4
	r = l;
	l = lp ^ F(l, ks[5]);
#ifdef CPPCRYPTO_DEBUG
	printf("r4: %016I64x %016I64x\n", l, r);
#endif

	// r = 5
	lp = l;
	l = r ^ F(l, ks[6]);
#ifdef CPPCRYPTO_DEBUG
	printf("r5: %016I64x %016I64x\n", l, lp);
#endif

	// r = 6
	r = l;
	l = lp ^ F(l, ks[7]);
	l = FL(l, ks[8]);
	r = FL1(r, ks[9]);
#ifdef CPPCRYPTO_DEBUG
	printf("r6: %016I64x %016I64x\n", l, r);
#endif

	// r = 7
	lp = l;
	l = r ^ F(l, ks[10]);

	// r = 8
	r = l;
	l = lp ^ F(l, ks[11]);

	// r = 9
	lp = l;
	l = r ^ F(l, ks[12]);

	// r = 10
	r = l;
	l = lp ^ F(l, ks[13]);

	// r = 11
	lp = l;
	l = r ^ F(l, ks[14]);

	// r = 12
	r = l;
	l = lp ^ F(l, ks[15]);
	l = FL(l, ks[16]);
	r = FL1(r, ks[17]);

	// r = 13
	lp = l;
	l = r ^ F(l, ks[18]);

	// r = 14
	r = l;
	l = lp ^ F(l, ks[19]);

	// r = 15
	lp = l;
	l = r ^ F(l, ks[20]);

	// r = 16
	r = l;
	l = lp ^ F(l, ks[21]);

	// r = 17
	lp = l;
	l = r ^ F(l, ks[22]);

	// r = 18
	r = l;
	l = lp ^ F(l, ks[23]);
	l = FL(l, ks[24]);
	r = FL1(r, ks[25]);

	// r = 19
	lp = l;
	l = r ^ F(l, ks[26]);

	// r = 20
	r = l;
	l = lp ^ F(l, ks[27]);

	// r = 21
	lp = l;
	l = r ^ F(l, ks[28]);

	// r = 22
	r = l;
	l = lp ^ F(l, ks[29]);

	// r = 23
	lp = l;
	l = r ^ F(l, ks[30]);

	// r = 24
	r = l;
	l = lp ^ F(l, ks[31]);

	r ^= ks[32];
	l ^= ks[33];

	*(uint64*)out = bswap_64(r);
	*(uint64*)(out + 8) = bswap_64(l);
}


void camellia_decrypt(const unsigned __int8 *in,  unsigned __int8 *out, unsigned __int8 *ksPtr)
{
	uint64 *ks = (uint64 *) ksPtr;
	uint64 r = bswap_64(*((uint64*)in)) ^ ks[32];
	uint64 l = bswap_64(*((uint64*)(in + 8))) ^ ks[33];

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", l, r);
#endif

	// r = 24
	uint64 rp = r;
	r = l ^ F(r, ks[31]);

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", rp, r);
#endif

	// r = 23
	l = r;
	r = rp ^ F(r, ks[30]);
#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", l, r);
#endif

	// r = 22
	rp = r;
	r = l ^ F(r, ks[29]);
#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", rp, r);
#endif

	// r = 21
	l = r;
	r = rp ^ F(r, ks[28]);
#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", l, r);
#endif

	// r = 20
	rp = r;
	r = l ^ F(r, ks[27]);
#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", rp, r);
#endif

	// r = 19
	l = r;
	r = rp ^ F(r, ks[26]);
#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", l, r);
#endif
	l = FL1(l, ks[24]);
	r = FL(r, ks[25]);

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPTFL: %016I64x %016I64x\n", l, r);
#endif

	// r = 18
	rp = r;
	r = l ^ F(r, ks[23]);
#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT18: %016I64x %016I64x\n", rp, r);
#endif

	// r = 17
	l = r;
	r = rp ^ F(r, ks[22]);

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", l, r);
#endif

	// r = 16
	rp = r;
	r = l ^ F(r, ks[21]);

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", rp, r);
#endif

	// r = 15
	l = r;
	r = rp ^ F(r, ks[20]);

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", l, r);
#endif

	// r = 14
	rp = r;
	r = l ^ F(r, ks[19]);

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", rp, r);
#endif

	// r = 13
	l = r;
	r = rp ^ F(r, ks[18]);
	l = FL1(l, ks[16]);
	r = FL(r, ks[17]);

#ifdef CPPCRYPTO_DEBUG
	printf("DECRYPT: %016I64x %016I64x\n", l, r);
#endif

	// r = 12
	rp = r;
	r = l ^ F(r, ks[15]);

	// r = 11
	l = r;
	r = rp ^ F(r, ks[14]);

	// r = 10
	rp = r;
	r = l ^ F(r, ks[13]);

	// r = 9
	l = r;
	r = rp ^ F(r, ks[12]);

	// r = 8
	rp = r;
	r = l ^ F(r, ks[11]);

	// r = 7
	l = r;
	r = rp ^ F(r, ks[10]);
	l = FL1(l, ks[8]);
	r = FL(r, ks[9]);

	// r = 6
	rp = r;
	r = l ^ F(r, ks[7]);

	// r = 5
	l = r;
	r = rp ^ F(r, ks[6]);

	// r = 4
	rp = r;
	r = l ^ F(r, ks[5]);

	// r = 3
	l = r;
	r = rp ^ F(r, ks[4]);

	// r = 2
	rp = r;
	r = l ^ F(r, ks[3]);

	// r = 1
	l = r;
	r = rp ^ F(r, ks[2]);

	// r = 0
	l ^= ks[0];
	r ^= ks[1];

	*(uint64*)out = bswap_64(l);
	*(uint64*)(out + 8) = bswap_64(r);
}

#endif
