/*
 * Copyright (c) 2017, 2019, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * The certificates and corresponding private keys used by the test.
 * All of certificates uses relative weak key size and hash algorithm, then
 * all JDK releases can load them. Accordingly, a custom java.security file is
 * provided to make sure such weak key sizes and algorithms are not blocked by
 * any JDK build.
 */
public enum Cert {

    // This certificate is generated by the below command:
    // openssl genpkey -algorithm rsa -pkeyopt rsa_keygen_bits:2048 \
    //     -pkeyopt rsa_keygen_pubexp:65537 -out key.pem
    // openssl req -x509 -new -key key.pem \
    //     -subj "/CN=RSA-2048-SHA256" -sha256 -out cer.pem
    RSA_2048_SHA256(
            KeyAlgorithm.RSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIDFTCCAf2gAwIBAgIUcCwtPduMIU144++G82mUEVNNK9kwDQYJKoZIhvcNAQEL\n" +
            "BQAwGjEYMBYGA1UEAwwPUlNBLTIwNDgtU0hBMjU2MB4XDTE5MDExNjA2MDgzNVoX\n" +
            "DTE5MDIxNTA2MDgzNVowGjEYMBYGA1UEAwwPUlNBLTIwNDgtU0hBMjU2MIIBIjAN\n" +
            "BgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAycT4Kd742lTlULVh5BcsZIG/AvVl\n" +
            "4IVnCoKoE8EyAf+YB2f+pYTDtyPzxnIUUqJ1/1dRY1EyHKQWRv1/J6H9qrKl48Sx\n" +
            "zgctOMN6zrCjPGx85MWRW7jOTi9/FNjCfmmGDzo7jjfhEeSzU56zyOMMka2UvKYa\n" +
            "P7YSTfC6nT79uaQNj/fqSK98FDLypDcrMiwzJZ7UX4M4Yo33EtqT0wFJfHl/LHJT\n" +
            "lmpQdn7dDCqZGviP59tfuGPO7/la18OiN8hU8cttEkAcW3k19kYNhhtfxqs1MtAZ\n" +
            "xGlN3eeW4IfjitMerEUd5wHrACyC4TKuj5NO6Wk1vl8pINsdkUttv5pHbQIDAQAB\n" +
            "o1MwUTAdBgNVHQ4EFgQUT6UTyQ2i4qOkx3AAPwWS6wdmgncwHwYDVR0jBBgwFoAU\n" +
            "T6UTyQ2i4qOkx3AAPwWS6wdmgncwDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0B\n" +
            "AQsFAAOCAQEAPa4ib8oo7vgOh1c/HBvynkzoZ/ci3hqQCArqHkTKQEFSpHeUz46j\n" +
            "u+GBRV/bFvZWc+GR9BPedKZUyRzCy3lmWclsxXEnv1uz/PTGBRMtZpjaaveTHzXm\n" +
            "VVIkMH+wTZsZ/EQiz2pHgPuAJdPTHlwIYOYo5jk/eZoTKGupBuce+gsn0ctSZQc/\n" +
            "TyayRCvnbQQ9Q6VbcfbrWGAmnCa4VANGuk3uZFj2Hc87udJ+2R8WkyfgXtwypNtb\n" +
            "1SrRuCqthfCFa4s+P0LlddVqp18gSvsiB+yA1RVZSlSD4GfJfrgtSsJu/ovqThr7\n" +
            "+YTfrHCVl4gliXaVujl6tQHaFk23WbAMwg==\n" +
            "-----END CERTIFICATE-----",
            "308204be020100300d06092a864886f70d0101010500048204a8308204a40201" +
            "000282010100c9c4f829def8da54e550b561e4172c6481bf02f565e085670a82" +
            "a813c13201ff980767fea584c3b723f3c6721452a275ff57516351321ca41646" +
            "fd7f27a1fdaab2a5e3c4b1ce072d38c37aceb0a33c6c7ce4c5915bb8ce4e2f7f" +
            "14d8c27e69860f3a3b8e37e111e4b3539eb3c8e30c91ad94bca61a3fb6124df0" +
            "ba9d3efdb9a40d8ff7ea48af7c1432f2a4372b322c33259ed45f8338628df712" +
            "da93d301497c797f2c7253966a50767edd0c2a991af88fe7db5fb863ceeff95a" +
            "d7c3a237c854f1cb6d12401c5b7935f6460d861b5fc6ab3532d019c4694ddde7" +
            "96e087e38ad31eac451de701eb002c82e132ae8f934ee96935be5f2920db1d91" +
            "4b6dbf9a476d0203010001028201006dba71df8b8438707cf9647b2529391a3b" +
            "b95e69888b0ee197c4b09575b6b58183f35b2a1067e06c23e03a26e6487e53bf" +
            "96840b8827c18db713ca5eb176165713aac5f0bd65b75f6f8457b03a3dbbe9a0" +
            "0e662784034027230b7091e54c0c253cf8c554b5acf02739231ba6d87429ecbb" +
            "c2acc98472eb988ecc81206d165d33147e03279e60f7fbf73d8f199895f627a3" +
            "3cf0c2ef2bcbd096f2e08b2684ea675956da0d95e941afe081e8c79ddb003b50" +
            "0f3b340978bce6821438ef25ddbf4fc9dba3f421dbf576f3099dbd4463dbcd2f" +
            "da5a987067d00c5af85faa7aea6427f12a1c03c9b5155fc5b5d4da51b4e9f5bf" +
            "34087e582728bcaf40b39b0938163d02818100e379b3e110ca339eb1491b95ca" +
            "0e73359a167722250f161ff78fef35e22282af28e02da454d0dca6af65b9118b" +
            "6e3efe9cabae5d85746b0c336e3d9002c0575afe370ae7a0294c01988d6fa098" +
            "9c4a6fc0816addcef3e891f2e56289da5b1b7a1c743664bb8b30ed6028942f72" +
            "74f25c075b0646b47fae6c3fc31b4bfd05b02302818100e31210ff848f5a73c6" +
            "1a508be415845bb18dcf09688ad543e8b040d9399850801a322e631dc605ec3e" +
            "d25695b4f66cb6a728a4e11ff211122c7d0de7de356337b344fca03176c2c370" +
            "7fbcdec2433a6c75d7a7d57b761ad6a8c1c8faaf316e0f410643f667958fcfac" +
            "c9960d860c73cec45d118517fe72c5939730d8482bdb2f0281807e1a5ab0bb29" +
            "0ce2bd6f44de8debe2cc65466cf6bdca963e5474336d10968711e93c15b152df" +
            "9262c93b40144cd26a13a5f0bab1d7a8c92b335bbabf19f75cb5f1d5bbb2da23" +
            "eaa1bbdb3475b80474736d29917fb3199de542dd0cfa54d54aef2fd4f0ce78f5" +
            "59c34e1a50c3d8d4a20288855a7e59d3aa731209ec18fd04693702818100c384" +
            "54da8edb9878c384f901db1ca3e1081b20bfeb224fcbaf59c41cc6b8dde7cfa6" +
            "91c68a666dc723b89d113ec6488965995b8ef4a0cc0e27fc6db2cee48d4ff2ae" +
            "5e0fd94777202d87efaaa6fe9819b7c63f1f54b5371aca2841d3887239602d0f" +
            "2609cedb3aff08ba72d7a62aa6b4cce38e2859a6a0507b6add85fd6eb5c50281" +
            "8100c07809a53e9ec6769bf2d31ed767766078ff7020199869473c26a211a4b2" +
            "9828618b69e6e29ed0f1d8d8cac2c61490c3f20d19c1ff067c5d291080c8e15b" +
            "2ce267cd37f8b0c1afe9c070418b310b5641640805d3a75890ccf6d2bf0bea11" +
            "d327323db5452cadc1a3bd17c20ab92b6f09d4216a03c7a03c6ffc72e51f51eb" +
            "dfa4"),

    // This certificate is generated by the below command:
    // openssl genpkey -algorithm rsa -pkeyopt rsa_keygen_bits:2048 \
    //     -pkeyopt rsa_keygen_pubexp:65537 -out key.pem
    // openssl req -x509 -new -key key.pem \
    //     -subj "/CN=EXAMPLE" -sha256 -out cer.pem
    EXAMPLE_RSA_2048_SHA256(
            KeyAlgorithm.RSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIDBTCCAe2gAwIBAgIUfmLJ5eIbVUGXAzlZXtw08GQ6ppMwDQYJKoZIhvcNAQEL\n" +
            "BQAwEjEQMA4GA1UEAwwHRVhBTVBMRTAeFw0xOTAxMTYwNjA4MzVaFw0xOTAyMTUw\n" +
            "NjA4MzVaMBIxEDAOBgNVBAMMB0VYQU1QTEUwggEiMA0GCSqGSIb3DQEBAQUAA4IB\n" +
            "DwAwggEKAoIBAQCp7IDXT8J9iDMVAuLMTZS9oDD83BDkL5INGdXk1esTzyqXFmV2\n" +
            "d5zNTr4A8w+YstkR081zL4MyEvHyQF1IlWoniRMXTZNMYYtU8dI8h2Fl5etSIEsc\n" +
            "ArsAp3QMcoqEu4F4T68KPU7z1M5kYKevGqPsO4GJwjoydSZMLZMrq09yEyXlnE9l\n" +
            "pNhyfFbQIp86mtXkY9nP3hn7JX6KMZSwAHbp7FtFkGfMx+usMnsMan+Z7UyWJE3o\n" +
            "2cf29Fr9lBdV24gWAymyJA3BAW60wEI2JPYzIZVNn4zxmlkWk5sr+m5rUCXMzsyp\n" +
            "G+rPk7YSpPutmczPe1BEiFwkgk+E5gZsNESbAgMBAAGjUzBRMB0GA1UdDgQWBBRm\n" +
            "mZ3V6rNvJyG5DdYt1yo/Eiz+AjAfBgNVHSMEGDAWgBRmmZ3V6rNvJyG5DdYt1yo/\n" +
            "Eiz+AjAPBgNVHRMBAf8EBTADAQH/MA0GCSqGSIb3DQEBCwUAA4IBAQBNHOWMa7f8\n" +
            "iXwhET/6Rpu0JILusRrJVGXl3JOHxDsibUPAeZ4KI7VEEBw6ln0bKAFaYg+Nh9Xh\n" +
            "dMuPC6e5wrX2Z912ncexa7BXDQDS4VyuvEGpp7rWFNtKRfc75hKmcRz+blgdhw9m\n" +
            "gF1VcW3vBIjAUjMllRuyPlyXThgOPwcBXGEIewvyLXWbkNDFIeqycwsQsw5JJcbA\n" +
            "Fh4alzjapSvSM84VS79u/MxaNZAtKpaymMaM05A8vIp8iHDm4N4AhIwHLT1mrtFt\n" +
            "8y+3p4W6vtA+SlFGz8fQw5ppoxvPeJyHZmSmGeorcBv9XXWHhJ0rGz8UbE76xE0B\n" +
            "EwC7yAE/SiA7\n" +
            "-----END CERTIFICATE-----",
            "308204be020100300d06092a864886f70d0101010500048204a8308204a40201" +
            "000282010100a9ec80d74fc27d88331502e2cc4d94bda030fcdc10e42f920d19" +
            "d5e4d5eb13cf2a97166576779ccd4ebe00f30f98b2d911d3cd732f833212f1f2" +
            "405d48956a278913174d934c618b54f1d23c876165e5eb52204b1c02bb00a774" +
            "0c728a84bb81784faf0a3d4ef3d4ce6460a7af1aa3ec3b8189c23a3275264c2d" +
            "932bab4f721325e59c4f65a4d8727c56d0229f3a9ad5e463d9cfde19fb257e8a" +
            "3194b00076e9ec5b459067ccc7ebac327b0c6a7f99ed4c96244de8d9c7f6f45a" +
            "fd941755db88160329b2240dc1016eb4c0423624f63321954d9f8cf19a591693" +
            "9b2bfa6e6b5025cccecca91beacf93b612a4fbad99cccf7b5044885c24824f84" +
            "e6066c34449b0203010001028201005842e1357557678eec4198ab274590e1e2" +
            "282fdf3ae2db96b8395831b1af962d8048d438458da1a3bea7d0a46fd077ed6a" +
            "66228c16fcc570b3bd8a132a1579fb2927026ea7f8ff9db8b496e81bc5ca80df" +
            "775c10c76edfa55a555bf5cedf5ce9c60d55b532dd24a7bfc0c1b7b7ab55c3e9" +
            "b0c25661963de573a22494853a11dce95ea31417d3c87c806ef74cb6c8b7190c" +
            "cfcdc2d21e8a756061c9e6cf40bca95d5aa43fb990b9492250ec9a752151320c" +
            "b30a64beb0e17d83ad9ea702afcd5d8d6b7bfe11031aa27d83652e8db864bdbc" +
            "447aee6e973018f77155aa24e05d3b7f9e232096ff93e8e2b1361b6cdbd4edf8" +
            "dd88b46b178b38c34fe0ea5fc40f9102818100db14d91a4e8a89cc95a7cc687f" +
            "2f2f295bc8a945826b3da840f871161ce2062f980d45b50172e744a308c0972e" +
            "6f3f028465e6a59b75c0687bc3db4b69f5a931b73e9dedc6b05d41f4c1dd0575" +
            "8e0460efba9bbb98f1c6ae3f018e2fb967a085dc5290ba8487703e1aee05fd90" +
            "3c17867130c5676b7b9567a6fd61e9be6d660902818100c68f0053c621d52d65" +
            "4ec381775e1b8fbf15ad391a0ad478d5b46374ad3f7c417a2f425f285999b4e6" +
            "1672614ec6b4bad54400ecbc9e92f35cdab590a8cff4c029146a9b101d53950f" +
            "7dddaa45946bfcf2a4e86bcddfc141a2cc49969519a326c7e5b001912e151d86" +
            "b17f5789f39513c3f660c3e41f169a7668b22c17241e8302818100b86d0a7e4c" +
            "d3ef40dc530f8e8052b63ef8d729382c9c1ea17f6025c2d9b9a43f789ee3a986" +
            "78b61b5fabc485004002291a4fb6247f8456df1e21388079c8a61006149e5a46" +
            "42bd9f026e18a3b9dc3def64a010ed91c926da148c38a8104a1e25d1dd679cbc" +
            "684fa2d884bb62438372c2689307fb11ce4d6d9e73fb730c2d8811028181008d" +
            "fda14c4739d68a9a11d3397835321c7f976ec290df01c64f7caa4abbc1d487b6" +
            "6aa95a072edbfe4333f623a403f1265270490102799bb8b0c42e66fe7188230a" +
            "bd70e6e685324a3c43d40a79ab83f5e5470c765b49119870650a92c69908d528" +
            "ca162d68b6bd9ed9bd80c506ffcbb1d0c715b7c02083377e49ac705f34132502" +
            "8180100e20febd1a5af92fdfc36aaf47d1a771cb1484d79e2389d5e6f47dc773" +
            "512edef82676f9f9d5a77aac2f01d66fe864d85abcce47e3d22491421f959c1e" +
            "5545c16fc5c5f5550ced81485dc237d9df8753cd6031e431bd34425e81b1e193" +
            "c51a6d2c8a7cc01028f47b7fb7d79b481facb76c4775ff997f2e63acb3ff429c" +
            "47b3"),

    // This certificate is generated by the below commands:
    // openssl genpkey -genparam -algorithm dsa -pkeyopt dsa_paramgen_bits:2048 \
    //     -pkeyopt dsa_paramgen_q_bits:256 -out param.pem
    // openssl genpkey -paramfile param.pem -out key.pem
    // openssl req -x509 -new -key key.pem -days 7300 \
    //     -subj "/CN=DSA-2048-SHA256" -sha256 -out cer.pem
    DSA_2048_SHA256(
            KeyAlgorithm.DSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIEezCCBCCgAwIBAgIUYbBHFrJkO9EokOROEScK0dr3aVowCwYJYIZIAWUDBAMC\n" +
            "MBoxGDAWBgNVBAMMD0RTQS0yMDQ4LVNIQTI1NjAeFw0xOTAxMTYwNjA4NTJaFw0z\n" +
            "OTAxMTEwNjA4NTJaMBoxGDAWBgNVBAMMD0RTQS0yMDQ4LVNIQTI1NjCCA0cwggI5\n" +
            "BgcqhkjOOAQBMIICLAKCAQEAnGTeaC+MitQykXl2UcGtJhkelRhBalZuzR9fh8Xf\n" +
            "swSj2auVzhjkQfns6E14X72XLYDwHPQilh4ZSiC1cX7bpvohJ7CjbouGnT78Gwf0\n" +
            "pfVMRBRfbNiwBQzFtzehAqlLPXyHWCPxAzW0EgzZh8asIofBEGmGh6sxlpXbEcCS\n" +
            "Ra5Q2vdDRVksm4APHKwbS1F6L/0QDvjDAudWyLFdFMSBuoyeFwCf7Gr5lcVi538h\n" +
            "MliXSAhJ7SXe4B9K/5/VicuzAffE+EZsZZuJKeQ4oHz8xtycdMcW86Iag1i/POzD\n" +
            "0Z7c5p9j1zDgms1ZRSz4fd7YgNGjmmf6dYJlZmCX4w9YiwIhALnVATHxZmk7G1kf\n" +
            "LGX7SMeflAhA/D3EPA5QWdgTc0orAoIBAGYTWMnKHBF4oMKUsx6lF6KvSrqFQ0+j\n" +
            "mWN1RNAGiVYm/Js9sc7jolZCbVTWM7GblO2DxiKed3FtcL3xw4P6l3GU7kFthsjh\n" +
            "bHbMG58s5JVboLX50wZo1uBOb0kRcZYjYTfUwbBYpGNnl7xfQsZ/Bxq/wzyn4gxb\n" +
            "+C0pu/vzmko+opKfFr9a2EL+ADvQhPd6y/L0YcpiTihvfXDWu+n3bNlwhUZYevux\n" +
            "SPVkQH3h5YEqXePF7UeY506/2sr41/xCbCkuH+Ob77Cy1IjMqr4OpXzj6wCSjlFy\n" +
            "Re66yqsjGpuBeTtsn9lKmlFVl4QUdw7XYbRubafNFEdd5IazMflCcgMDggEGAAKC\n" +
            "AQEAnF3/TT0nYzfwY6/+lmoqoBtGU4of7RK4U3Jdhf6dj755GwgsPGAz3Rzh/hq/\n" +
            "B3fuyrtnE/aU0EK1dtRNqnKeFsMCURBi3sWYYe7vamw5TukFMSUOuH76v75b26c+\n" +
            "nTfZF30iDz+66Z5gcOSwhz9aLCnF/wYwijnzagnCF4m0Nhwsjtz4qECMPfL/Rkx8\n" +
            "s83tUF53mx5o8WCYMNr4yJGcHEkautvZN05zwoxZTPKZNngWeCUAY07XwOYHTUvx\n" +
            "C9Sxiom9k1pZPbHZnGpUx96at2dMoeTeHR6xF/0QpiIb4wLpOg/CRxsopBmhEl0S\n" +
            "BEXYNIcrXLwVBLvJD6H44DxRiqNTMFEwHQYDVR0OBBYEFCr6C8Bl7wjz5L3cYMG3\n" +
            "/ZFe7Ee0MB8GA1UdIwQYMBaAFCr6C8Bl7wjz5L3cYMG3/ZFe7Ee0MA8GA1UdEwEB\n" +
            "/wQFMAMBAf8wCwYJYIZIAWUDBAMCA0gAMEUCIArByiqjCG1ZuOVY91OPa9g8q60y\n" +
            "/BSRWRgikEuq3AbgAiEAoZoKXW80FTMxv/9tFy0N7OrUV4rc9+AUBSTuRCuTAk4=\n" +
            "-----END CERTIFICATE-----",
            "308202640201003082023906072a8648ce3804013082022c02820101009c64de" +
            "682f8c8ad43291797651c1ad26191e9518416a566ecd1f5f87c5dfb304a3d9ab" +
            "95ce18e441f9ece84d785fbd972d80f01cf422961e194a20b5717edba6fa2127" +
            "b0a36e8b869d3efc1b07f4a5f54c44145f6cd8b0050cc5b737a102a94b3d7c87" +
            "5823f10335b4120cd987c6ac2287c110698687ab319695db11c09245ae50daf7" +
            "4345592c9b800f1cac1b4b517a2ffd100ef8c302e756c8b15d14c481ba8c9e17" +
            "009fec6af995c562e77f21325897480849ed25dee01f4aff9fd589cbb301f7c4" +
            "f8466c659b8929e438a07cfcc6dc9c74c716f3a21a8358bf3cecc3d19edce69f" +
            "63d730e09acd59452cf87dded880d1a39a67fa758265666097e30f588b022100" +
            "b9d50131f166693b1b591f2c65fb48c79f940840fc3dc43c0e5059d813734a2b" +
            "02820100661358c9ca1c1178a0c294b31ea517a2af4aba85434fa399637544d0" +
            "06895626fc9b3db1cee3a256426d54d633b19b94ed83c6229e77716d70bdf1c3" +
            "83fa977194ee416d86c8e16c76cc1b9f2ce4955ba0b5f9d30668d6e04e6f4911" +
            "7196236137d4c1b058a4636797bc5f42c67f071abfc33ca7e20c5bf82d29bbfb" +
            "f39a4a3ea2929f16bf5ad842fe003bd084f77acbf2f461ca624e286f7d70d6bb" +
            "e9f76cd9708546587afbb148f564407de1e5812a5de3c5ed4798e74ebfdacaf8" +
            "d7fc426c292e1fe39befb0b2d488ccaabe0ea57ce3eb00928e517245eebacaab" +
            "231a9b81793b6c9fd94a9a5155978414770ed761b46e6da7cd14475de486b331" +
            "f942720304220220509dd213cc9769e93825063a4a60500f67c4b979f6504b2f" +
            "ccdbefb3ab8fe6da"),

    // This certificate is generated by the below commands:
    // openssl genpkey -genparam -algorithm dsa -pkeyopt dsa_paramgen_bits:2048 \
    //     -pkeyopt dsa_paramgen_q_bits:256 -out param.pem
    // openssl genpkey -paramfile param.pem -out key.pem
    // openssl req -x509 -new -key key.pem -days 7300 \
    //     -subj "/CN=EXAMPLE" -sha256 -out cer.pem
    EXAMPLE_DSA_2048_SHA256(
            KeyAlgorithm.DSA,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIEaTCCBA+gAwIBAgIUckU2myqkWak+Svv0FFJX91vv1jMwCwYJYIZIAWUDBAMC\n" +
            "MBIxEDAOBgNVBAMMB0VYQU1QTEUwHhcNMTkwMTE2MDYwODUzWhcNMzkwMTExMDYw\n" +
            "ODUzWjASMRAwDgYDVQQDDAdFWEFNUExFMIIDRjCCAjkGByqGSM44BAEwggIsAoIB\n" +
            "AQDGmXUxMKzLIVUQUy4lu4fbnv8fRXtWIbv1jxWATCNFiFt4cUxrr7GVC16MGrgm\n" +
            "FtMIHXWwFlq2wTL0RhVp2WPIPQzxCwQFivQjpdewVCB1BoJ1ozlvQjU4hri5Ymdt\n" +
            "ebe90uT8NsvrQrSKYCl+/pPNiZKT8oX1aKzRYPQLn0bVXUicWlACoLfHGM4irjEC\n" +
            "4JezC/tdLleeNYNpy2/dKYu/atyN/u0d+dPRLWBCw6/qCRn1yRAv04GC3WYBlSXz\n" +
            "f9OKlCH5kfm9sLyatz/RWDqOb/YWW2Rk7UTKAnoTAyB+I9yUXg6Gad2csNkxXv55\n" +
            "9oJAhdhsOS5cdBoqlZEahIFBAiEAjiMCVBGpnAxjb2dXM/Eec7EfThflQXbl33Zn\n" +
            "Uq3AAsUCggEAaBNP9Uttfs1eV/38aurLd3T1OiM0CF7DPxE0qpSM9dQz9cKZajIE\n" +
            "lsVTGOLBC5/+VSat9t1VG+JoyvSspkvk97/mxx0WCz/QAYTdwCXVuMuSv+EqeOMP\n" +
            "lCxEdbOS8pfD8shOK+pnDSHMJvURYxB+fJkHHeXfwesH3632Vq0GlJ8PgXH5NLHM\n" +
            "MWv7oZjyZMnGWDq2taJcZZG5ETNda2fATNCF9Al430MUxie2Sp50vA1KEtyUqMu+\n" +
            "CLpyOynPHi96TWHNfD23TmKFVN9Uh2nUNIpUk8NMKBwg2O7FvvNnKfbl44ikuCnc\n" +
            "06U7SdF3y8NRdwyayMI3BkOsV8mkoMwUgAOCAQUAAoIBADK2c1Gl3+6zrFU8Uhro\n" +
            "5lFnDy3UYXINpdtWR/8T0FJ6YMax70VCkyxucq3AUnmq9AQtFijjmXDnxzD5g7IS\n" +
            "zOU/28Kg1Mmw26uzrpUZPiixNU7v/xzE37M0Ig3VCEQ9mw57/yv8gwZurHtsokGt\n" +
            "k0lzH/krhYPCOpskg6NulGq5lGsnNVdPkSkiAWZFHTysgKgxvMUxXj0bUm4ZyNw6\n" +
            "Lp2bFHKbmSeTy3OLo/Kvd7BIjhV+Bi5LFh7h8spf6SC7qyEm0X7Keu+61xkFHU8a\n" +
            "aghnGQYwuzpTp+hsbAJdHwVLw9ElqEoaVp3rAmxtsdzqhZSxcMZoypbjEJpiZ1/v\n" +
            "fQCjUzBRMB0GA1UdDgQWBBRTPXqHl7VKr5U94p9c882ge9DOXzAfBgNVHSMEGDAW\n" +
            "gBRTPXqHl7VKr5U94p9c882ge9DOXzAPBgNVHRMBAf8EBTADAQH/MAsGCWCGSAFl\n" +
            "AwQDAgNHADBEAiAOQxtKiMImXKWQvUKZxoUc+chXRxAj3UD3Zj7RksPF2AIgf8HG\n" +
            "hOnr3hzIej3Da2Ty9RTittcgZ14boraO/Vlx9Jo=\n" +
            "-----END CERTIFICATE-----",
            "308202640201003082023906072a8648ce3804013082022c0282010100c69975" +
            "3130accb215510532e25bb87db9eff1f457b5621bbf58f15804c2345885b7871" +
            "4c6bafb1950b5e8c1ab82616d3081d75b0165ab6c132f4461569d963c83d0cf1" +
            "0b04058af423a5d7b0542075068275a3396f42353886b8b962676d79b7bdd2e4" +
            "fc36cbeb42b48a60297efe93cd899293f285f568acd160f40b9f46d55d489c5a" +
            "5002a0b7c718ce22ae3102e097b30bfb5d2e579e358369cb6fdd298bbf6adc8d" +
            "feed1df9d3d12d6042c3afea0919f5c9102fd38182dd66019525f37fd38a9421" +
            "f991f9bdb0bc9ab73fd1583a8e6ff6165b6464ed44ca027a1303207e23dc945e" +
            "0e8669dd9cb0d9315efe79f6824085d86c392e5c741a2a95911a848141022100" +
            "8e23025411a99c0c636f675733f11e73b11f4e17e54176e5df766752adc002c5" +
            "0282010068134ff54b6d7ecd5e57fdfc6aeacb7774f53a2334085ec33f1134aa" +
            "948cf5d433f5c2996a320496c55318e2c10b9ffe5526adf6dd551be268caf4ac" +
            "a64be4f7bfe6c71d160b3fd00184ddc025d5b8cb92bfe12a78e30f942c4475b3" +
            "92f297c3f2c84e2bea670d21cc26f51163107e7c99071de5dfc1eb07dfadf656" +
            "ad06949f0f8171f934b1cc316bfba198f264c9c6583ab6b5a25c6591b911335d" +
            "6b67c04cd085f40978df4314c627b64a9e74bc0d4a12dc94a8cbbe08ba723b29" +
            "cf1e2f7a4d61cd7c3db74e628554df548769d4348a5493c34c281c20d8eec5be" +
            "f36729f6e5e388a4b829dcd3a53b49d177cbc351770c9ac8c2370643ac57c9a4" +
            "a0cc1480042202201ba738489c54dddd5ffbf0009ef9d11851182235a251a410" +
            "4a6354181ecd0348"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -x509 -new -key key.pem \
    //     -subj "/CN=ECDSA-SECP256-SHA256" -sha256 -out cer.pem
    ECDSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIBkzCCATmgAwIBAgIUXebzNfSMvdkNrc5175FM6dE/gxwwCgYIKoZIzj0EAwIw\n" +
            "HzEdMBsGA1UEAwwURUNEU0EtU0VDUDI1Ni1TSEEyNTYwHhcNMTkwMTE2MDYwODQx\n" +
            "WhcNMTkwMjE1MDYwODQxWjAfMR0wGwYDVQQDDBRFQ0RTQS1TRUNQMjU2LVNIQTI1\n" +
            "NjBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABOp7VcLfpVc5ghO0HlYjwH+YSAf7\n" +
            "zTShLeoY35PwqcoUgg9DBR4g7rM3xovOKxGZ5uORD8vo5l9L0+53f2+7YH+jUzBR\n" +
            "MB0GA1UdDgQWBBSmW8/6KP8EJKBVlfmFkBSbVWTEWzAfBgNVHSMEGDAWgBSmW8/6\n" +
            "KP8EJKBVlfmFkBSbVWTEWzAPBgNVHRMBAf8EBTADAQH/MAoGCCqGSM49BAMCA0gA\n" +
            "MEUCIQCtGdW3Xl5OzX7QiwtiT6pbIrm6eCUwN/vVoMfs3Yn5rgIgCbLidpdMpFrd\n" +
            "HWB2/mVxQegLBCOIGMVPXrTat4A76As=\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "01010420aa9602d20de7ebfc2787a6e5ae20166d1e8d8bd29b4e5f046414bae8" +
            "9a8e7eb9a14403420004ea7b55c2dfa557398213b41e5623c07f984807fbcd34" +
            "a12dea18df93f0a9ca14820f43051e20eeb337c68bce2b1199e6e3910fcbe8e6" +
            "5f4bd3ee777f6fbb607f"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -x509 -new -key key.pem \
    //     -subj "/CN=EXAMPLE" -sha256 -out cer.pem
    EXAMPLE_ECDSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIBeTCCAR+gAwIBAgIUWZkzN4WOoj7HUYLoWtgy+ad/zjswCgYIKoZIzj0EAwIw\n" +
            "EjEQMA4GA1UEAwwHRVhBTVBMRTAeFw0xOTAxMTYwNjA4NDFaFw0xOTAyMTUwNjA4\n" +
            "NDFaMBIxEDAOBgNVBAMMB0VYQU1QTEUwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNC\n" +
            "AAQMzWjRuN9r2/kcVYyHFpKduuYj2VFRXo81qd+EcyjnLZab5m9RqIYy6iDUvZk5\n" +
            "w8wHeHGpMybPzNSEQ2mVY5Yvo1MwUTAdBgNVHQ4EFgQUUhVK37nQDP6OEW5w7XEb\n" +
            "p8/sCxMwHwYDVR0jBBgwFoAUUhVK37nQDP6OEW5w7XEbp8/sCxMwDwYDVR0TAQH/\n" +
            "BAUwAwEB/zAKBggqhkjOPQQDAgNIADBFAiEA9B2e4TWZKMxS8/uyVjl6D+cyqsPU\n" +
            "7/qATL5cTK1AIUACID8BPfcPE46ARCretiGWywVuQzAVKz1fkjwEwLmAo+2x\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "01010420ed3600c6b62b351b48b4c61f37c46c45460ff599aa7f9c2d79635d0e" +
            "147c23a3a144034200040ccd68d1b8df6bdbf91c558c8716929dbae623d95151" +
            "5e8f35a9df847328e72d969be66f51a88632ea20d4bd9939c3cc077871a93326" +
            "cfccd48443699563962f"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -new -key key.pem \
    //     -subj "/CN=EC-RSA-SECP256-SHA256" -sha256 -out csr.pem
    // openssl x509 -req -CAcreateserial -in csr.pem -sha256 \
    //     -CA CA.cer -CAkey CA.key -out cer.pem
    // Actually the CA is RSA_2048_SHA256
    EC_RSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIB9TCB3gIUIpSnxoBfFOKeRdx52FvEKuk1uvYwDQYJKoZIhvcNAQELBQAwGjEY\n" +
            "MBYGA1UEAwwPUlNBLTIwNDgtU0hBMjU2MB4XDTE5MDExNjA2MDg0NloXDTE5MDIx\n" +
            "NTA2MDg0NlowIDEeMBwGA1UEAwwVRUMtUlNBLVNFQ1AyNTYtU0hBMjU2MFkwEwYH\n" +
            "KoZIzj0CAQYIKoZIzj0DAQcDQgAETcLM10u7ehsxJQbX3ypbSRMz7ZDkFS/QsmPF\n" +
            "04NSwiBncQPjtpaPSshVDJzigEDfACcwIdO0BH4Eh2oHcB6hDzANBgkqhkiG9w0B\n" +
            "AQsFAAOCAQEAHg06hhIec3ctUh7pao53ZF3SVJ/Pty4rgQ3Hb5gNhZHrmYWldj6J\n" +
            "UagMRtfX0fJwzIdY0kNTol38es+6XDTCdYsaDDw4Ix1yF/xoExu6PqJ49npMVxqB\n" +
            "yeXwg8aDB9sbmeczZp0kWa1DiN3HgJGoA8HbPOUZbuetCVl2ME82ZPdKdLaHgjO/\n" +
            "Af3/gjYGVR27YB5sVIXkq3wJ5wEF+EvePKQZqnHFLhjz0xIIyp7mU6NFr26TsNh0\n" +
            "JYecs5S0ydhf41x9GS4p8KpqRcfAOX4z5DEBe+BjgSuprGZabflFCbZ/PQrtBmdp\n" +
            "5+cg/cNcA3zEXnsAzLu2R/+73/h9v75g/Q==\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "010104200ab17f2b70ba744b08564635ebfc535c60d43139e1c722cfb9c7152e" +
            "9faad5b3a144034200044dc2ccd74bbb7a1b312506d7df2a5b491333ed90e415" +
            "2fd0b263c5d38352c220677103e3b6968f4ac8550c9ce28040df00273021d3b4" +
            "047e04876a07701ea10f"),

    // This certificate is generated by the below commands:
    // openssl genpkey -algorithm ec -pkeyopt ec_paramgen_curve:P-256 \
    //     -pkeyopt ec_param_enc:named_curve -out key.pem
    // openssl req -new -key key.pem -subj "/CN=EXAMPLE" -sha256 -out csr.pem
    // openssl x509 -req -CAcreateserial -in csr.pem -sha256 \
    //     -CA CA.cer -CAkey CA.key -out cer.pem
    // Actually the CA is EXAMPLE_RSA_2048_SHA256
    EXAMPLE_EC_RSA_PRIME256V1_SHA256(
            KeyAlgorithm.EC,
            "-----BEGIN CERTIFICATE-----\n" +
            "MIIB3zCByAIUbogHun+2LFxbX9B5fImITzFQVfIwDQYJKoZIhvcNAQELBQAwEjEQ\n" +
            "MA4GA1UEAwwHRVhBTVBMRTAeFw0xOTAxMTYwNjA4NDZaFw0xOTAyMTUwNjA4NDZa\n" +
            "MBIxEDAOBgNVBAMMB0VYQU1QTEUwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAASA\n" +
            "pPapPXO0qJO7mDm9dTuSKdsxWatAaloJZnwenBFIY8krulHd8VPSGbERmxRBY/z2\n" +
            "+MH8dwaC1t9DAFb+1qdWMA0GCSqGSIb3DQEBCwUAA4IBAQA3SfXMUFj7xCl7nR8L\n" +
            "/6mf0La/q4O1pGKmVpy55aP7dZmY3xzaPtDFevdnMqb7tBTiLl0Y8OehMQW/8usb\n" +
            "qtcYPekZJV5g1ezMhyB/AHecotfsrBS7l9r+IWYf/GUoQ8izC1srNXVrqDCt0cbB\n" +
            "o7bc0lQFhI+rcMt1AzQtrNkVhX0dcBbLyhNJzgyAXatSB5R0/R3kTddUZfrOtOoC\n" +
            "IXUZJRQ7hZKx7qi/U4+q246IuKSSp2SjFTU1QpeO4/Q06eJ3sbtx5Nd1Rfzlo2Jq\n" +
            "uYi8szOupFC8xKCB+odMndHvh1QO/8E4e4r0mShkrnK3M/lKiwi67yl3Jk9uY9ls\n" +
            "X5Q6\n" +
            "-----END CERTIFICATE-----",
            "308187020100301306072a8648ce3d020106082a8648ce3d030107046d306b02" +
            "01010420618ff6a884185b97322aaeab5f9fdf8a14ab18d0478565317b483db0" +
            "0f3b5bf6a1440342000480a4f6a93d73b4a893bb9839bd753b9229db3159ab40" +
            "6a5a09667c1e9c114863c92bba51ddf153d219b1119b144163fcf6f8c1fc7706" +
            "82d6df430056fed6a756");

    public final KeyAlgorithm keyAlgorithm;
    public final String certMaterials;
    public final String privKeyMaterials;

    private Cert(
            KeyAlgorithm keyAlgorithm,
            String certMaterials,
            String privKeyMaterials) {
        this.keyAlgorithm = keyAlgorithm;
        this.certMaterials = certMaterials;
        this.privKeyMaterials = privKeyMaterials;
    }

    // Two certificates (mainCert and exampleCert) are selected to respect the
    // specified cipher suite. SNI-associated cases specify exampleCert as desired.
    public static Cert[] getCerts(CipherSuite cipherSuite) {
        Cert mainCert = Cert.ECDSA_PRIME256V1_SHA256;
        Cert exampleCert = Cert.EXAMPLE_ECDSA_PRIME256V1_SHA256;
        if (cipherSuite.keyExAlgorithm == KeyExAlgorithm.ECDHE_RSA
                || cipherSuite.keyExAlgorithm == KeyExAlgorithm.DHE_RSA
                || cipherSuite.keyExAlgorithm == KeyExAlgorithm.RSA) {
            mainCert = Cert.RSA_2048_SHA256;
            exampleCert = Cert.EXAMPLE_RSA_2048_SHA256;
        } else if (cipherSuite.keyExAlgorithm == KeyExAlgorithm.ECDH_RSA) {
            mainCert = Cert.EC_RSA_PRIME256V1_SHA256;
            exampleCert = Cert.EXAMPLE_EC_RSA_PRIME256V1_SHA256;
        } else if (cipherSuite.keyExAlgorithm == KeyExAlgorithm.DHE_DSS) {
            mainCert = Cert.DSA_2048_SHA256;
            exampleCert = Cert.EXAMPLE_DSA_2048_SHA256;
        }
        System.out.printf("mainCert=%s, exampleCert=%s%n",
                mainCert, exampleCert);
        return new Cert[] { mainCert, exampleCert };
    }
}
