// Portions of code in this module came from the examples directory in mpv's
// git repo.  Reworked by me.

#include <QtGlobal>
#if defined(Q_OS_UNIX) && !defined(Q_OS_DARWIN)
#include <QtX11Extras/QX11Info>
#endif
#include <QLayout>
#include <QMainWindow>
#include <QThread>
#include <QTimer>
#include <QOpenGLContext>
#include <QMouseEvent>
#include <QMetaObject>
#include <QDir>
#include <QDebug>
#include <cmath>
#include <stdexcept>
#include <mpv/qthelper.hpp>
#include "mpvwidget.h"
#include "helpers.h"
#include "platform/unify.h"

#ifndef Q_PROCESSOR_ARM
    #ifndef GLAPIENTRY
    // On Windows, GLAPIENTRY may sometimes conveniently go missing
    #define GLAPIENTRY __stdcall
    #endif
#endif
#ifndef GLAPIENTRY
#define GLAPIENTRY
#endif



static const int HOOK_UNLOAD_CALLBACK_ID = 0xbeefdab;



MpvObject::MpvObject(QObject *owner, const QString &clientName) : QObject(owner)
{
    // Setup threads
    worker = new QThread();
    worker->start();

    // setup controller
    ctrl = new MpvController();
    ctrl->moveToThread(worker);

    // setup timer
    hideTimer = new QTimer(this);
    hideTimer->setSingleShot(true);
    hideTimer->setInterval(1000);

    // Wire the basic mpv functions to avoid littering the codebase with
    // QMetaObject::invokeMethod.  This way the compiler will catch our
    // typos rather than a runtime error.
    connect(this, &MpvObject::ctrlCommand,
            ctrl, &MpvController::command, Qt::QueuedConnection);
    connect(this, &MpvObject::ctrlSetOptionVariant,
            ctrl, &MpvController::setOptionVariant, Qt::QueuedConnection);
    connect(this, &MpvObject::ctrlSetPropertyVariant,
            ctrl, &MpvController::setPropertyVariant, Qt::QueuedConnection);
    connect(this, &MpvObject::ctrlSetLogLevel,
            ctrl, &MpvController::setLogLevel);

    // Wire up the event-handling callbacks
    connect(ctrl, &MpvController::mpvPropertyChanged,
            this, &MpvObject::ctrl_mpvPropertyChanged, Qt::QueuedConnection);
    connect(ctrl, &MpvController::logMessage,
            this, &MpvObject::ctrl_logMessage, Qt::QueuedConnection);
    connect(ctrl, &MpvController::clientMessage,
            this, &MpvObject::ctrl_clientMessage, Qt::QueuedConnection);
    connect(ctrl, &MpvController::unhandledMpvEvent,
            this, &MpvObject::ctrl_unhandledMpvEvent, Qt::QueuedConnection);
    connect(ctrl, &MpvController::videoSizeChanged,
            this, &MpvObject::ctrl_videoSizeChanged, Qt::QueuedConnection);

    // Wire up the mouse and timer-related callbacks
    connect(this, &MpvObject::mouseMoved,
            this, &MpvObject::self_mouseMoved);
    connect(hideTimer, &QTimer::timeout,
            this, &MpvObject::hideTimer_timeout);

    // Initialize mpv
    QMetaObject::invokeMethod(ctrl, "create", Qt::BlockingQueuedConnection);

    // clean up objects when the worker thread is deleted
    connect(worker, &QThread::finished, ctrl, &MpvController::deleteLater);

    // Observe some properties
    MpvController::PropertyList options = {
        { "time-pos", 0, MPV_FORMAT_DOUBLE },
        { "pause", 0, MPV_FORMAT_FLAG },
        { "media-title", 0, MPV_FORMAT_STRING },
        { "chapter-metadata", 0, MPV_FORMAT_NODE },
        { "track-list", 0, MPV_FORMAT_NODE },
        { "chapter-list", 0, MPV_FORMAT_NODE },
        { "duration", 0, MPV_FORMAT_DOUBLE },
        { "estimated-vf-fps", 0, MPV_FORMAT_DOUBLE },
        { "avsync", 0, MPV_FORMAT_DOUBLE },
        { "frame-drop-count", 0, MPV_FORMAT_INT64 },
        { "decoder-frame-drop-count", 0, MPV_FORMAT_INT64 },
        { "audio-bitrate", 0, MPV_FORMAT_DOUBLE },
        { "video-bitrate", 0, MPV_FORMAT_DOUBLE },
        { "paused-for-cache", 0, MPV_FORMAT_FLAG },
        { "metadata", 0, MPV_FORMAT_NODE },
        { "audio-device-list", 0, MPV_FORMAT_NODE },
        { "filename", 0, MPV_FORMAT_STRING },
        { "file-format", 0, MPV_FORMAT_STRING },
        { "file-size", 0, MPV_FORMAT_STRING },
        { "file-date-created", 0, MPV_FORMAT_NODE },
        { "format", 0, MPV_FORMAT_STRING },
        { "path", 0, MPV_FORMAT_STRING },
        { "seekable", 0, MPV_FORMAT_FLAG }
    };
    QSet<QString> throttled = {
        "time-pos", "avsync", "estimated-vf-fps", "frame-drop-count",
        "decoder-frame-drop-count", "audio-bitrate", "video-bitrate"
    };
    QMetaObject::invokeMethod(ctrl, "observeProperties",
                              Qt::BlockingQueuedConnection,
                              Q_ARG(const MpvController::PropertyList &, options),
                              Q_ARG(const QSet<QString> &, throttled));

    // Add hooks
    QMetaObject::invokeMethod(ctrl, "addHook",
                              Qt::BlockingQueuedConnection,
                              Q_ARG(QString, "on_unload"),
                              Q_ARG(int, HOOK_UNLOAD_CALLBACK_ID));

    blockingSetMpvOptionVariant("ytdl", "yes");
    blockingSetMpvOptionVariant("audio-client-name", clientName);
    ctrl->setLogLevel("info");
}

MpvObject::~MpvObject()
{
    if (widget)
        delete widget;
    if (hideTimer) {
        delete hideTimer;
        hideTimer = nullptr;
    }
    worker->deleteLater();
}

void MpvObject::setHostLayout(QLayout *hostLayout)
{
    if (!this->hostLayout)
        this->hostLayout = hostLayout;
}

void MpvObject::setHostWindow(QMainWindow *hostWindow)
{
    if (!this->hostWindow)
        this->hostWindow = hostWindow;
}

void MpvObject::setWidgetType(Helpers::MpvWidgetType widgetType)
{
    if (!hostLayout && !hostWindow)
        return;

    if (this->widgetType == widgetType)
        return;
    this->widgetType = widgetType;

    if (widget) {
        delete widget;
        widget = nullptr;
    }

    switch(widgetType) {
    case Helpers::NullWidget:
        widget = nullptr;
        break;
    case Helpers::EmbedWidget:
        widget = new MpvEmbedWidget(this);
        break;
    case Helpers::GlCbWidget:
        widget = new MpvGlCbWidget(this);
        break;
    case Helpers::VulkanCbWidget:
        widget = new MpvVulkanCbWidget(this);
        break;
    }
    if (!widget)
        return;

    if (hostLayout)
        hostLayout->addWidget(widget->self());
    else if (hostWindow)
        hostWindow->setCentralWidget(widget->self());
    widget->setController(ctrl);
    widget->initMpv();
}

QString MpvObject::mpvVersion()
{
    return getMpvPropertyVariant("mpv-version").toString();
}

MpvController *MpvObject::controller()
{
    return ctrl;
}

QWidget *MpvObject::mpvWidget()
{
    return widget->self();
}

QList<AudioDevice> MpvObject::audioDevices()
{
    return AudioDevice::listFromVList(getMpvPropertyVariant("audio-device-list").toList());
}

QStringList MpvObject::supportedProtocols()
{
    return ctrl->protocolList();
}

void MpvObject::showMessage(QString message)
{
    if (shownStatsPage <= 0 || shownStatsPage >= 3)
        emit ctrlCommand(QVariantList({"show_text", message, "1000"}));
}

void MpvObject::showStatsPage(int page)
{
    bool statsVisible = (shownStatsPage > 0 && shownStatsPage < 3);
    bool wantVisible = (page > 0 && page < 3);

    if (wantVisible ^ statsVisible) {
        qDebug() << "toggling stats page";
        ctrlCommand(QStringList({"script-binding",
                                 "stats/display-stats-toggle"}));
    }
    if (wantVisible) {
        qDebug() << "setting page to " << page;
        QStringList cmd { "script-binding",
                          QString("stats/display-page-%1").arg(QString::number(page)) };
        ctrlCommand(cmd);
    }
    shownStatsPage = page;
}

int MpvObject::cycleStatsPage()
{
    showStatsPage(shownStatsPage < 2 ? shownStatsPage+1 : 0);
    return shownStatsPage;
}

void MpvObject::fileOpen(QString filename)
{
    setSubFile("\n");
    //setStartTime(0.0);
    emit ctrlCommand(QStringList({"loadfile", filename}));
    setMouseHideTime(hideTimer->interval());
}

void MpvObject::discFilesOpen(QString path) {
    QStringList entryList = QDir(path).entryList();
    if (entryList.contains("VIDEO_TS") || entryList.contains("AUDIO_TS")) {
        fileOpen(path + "/VIDEO_TS/VIDEO_TS.IFO");
    } else if (entryList.contains("BDMV") || entryList.contains("AACS")) {
        fileOpen("bluray://" + path);
    }
}

void MpvObject::stopPlayback()
{
    emit ctrlCommand("stop");
}

void MpvObject::stepBackward()
{
    emit ctrlCommand("frame_back_step");
}

void MpvObject::stepForward()
{
    emit ctrlCommand("frame_step");
}

void MpvObject::seek(double amount, bool exact)
{
    QVariantList payload({"seek", amount});
    if (exact)
        payload.append("exact");
    emit ctrlCommand(payload);
}

void MpvObject::screenshot(const QString &fileName, Helpers::ScreenshotRender render)
{
    static QMap <Helpers::ScreenshotRender,const char*> methods {
        { Helpers::VideoRender, "video" },
        { Helpers::SubsRender, "subtitles" },
        { Helpers::WindowRender, "window" }
    };
    if (render == Helpers::WindowRender) {
        widget->self()->grab().save(fileName);
        return;
    }
    emit ctrlCommand(QStringList({"screenshot-to-file", fileName,
                                  methods.value(render, "video")}));
}

void MpvObject::setMouseHideTime(int msec)
{
    hideTimer->stop();
    hideTimer->setInterval(msec);
    showCursor();
    if (msec > 0)
        hideTimer->start();
}

void MpvObject::setLogoUrl(const QString &filename)
{
    widget->setLogoUrl(filename);
}

void MpvObject::setLogoBackground(const QColor &color)
{
    widget->setLogoBackground(color);
}

void MpvObject::setSubFile(QString filename)
{
    emit ctrlSetOptionVariant("sub-files", filename);
}

void MpvObject::addSubFile(QString filename)
{
    emit ctrlCommand(QStringList({"sub-add", filename}));
}

int64_t MpvObject::chapter()
{
    return getMpvPropertyVariant("chapter").toLongLong();
}

bool MpvObject::setChapter(int64_t chapter)
{
    // As this requires knowledge of mpv's return value, it cannot be
    // queued as a simple message.  The usual return values are:
    // MPV_ERROR_PROPERTY_UNAVAILABLE: unchaptered file
    // MPV_ERROR_PROPERTY_FORMAT: past-the-end value requested
    // MPV_ERROR_SUCCESS: success
    int r;
    QMetaObject::invokeMethod(ctrl, "setPropertyVariant",
                              Qt::BlockingQueuedConnection,
                              Q_RETURN_ARG(int, r),
                              Q_ARG(QString, "chapter"),
                              Q_ARG(QVariant, QVariant(qlonglong(chapter))));
    return r == MPV_ERROR_SUCCESS;
}

QString MpvObject::mediaTitle()
{
    return getMpvPropertyVariant("media-title").toString();
}

void MpvObject::setMute(bool yes)
{
    setMpvPropertyVariant("mute", yes);
}

void MpvObject::setPaused(bool yes)
{
    setMpvPropertyVariant("pause", yes);
}

void MpvObject::setSpeed(double speed)
{
    setMpvPropertyVariant("speed", speed);
}

void MpvObject::setTime(double position)
{
    setMpvPropertyVariant("time-pos", position);
}

void MpvObject::setTimeSync(double position)
{
    ctrl->command(QVariantList() << "seek" << position << "absolute");
}

void MpvObject::setLoopPoints(double first, double end)
{
    setMpvPropertyVariant("ab-loop-a",
                          first < 0 ? QVariant("no") : QVariant(first));
    setMpvPropertyVariant("ab-loop-b",
                          end < 0 ? QVariant("no") : QVariant(end));
}

void MpvObject::setAudioTrack(int64_t id)
{
    setMpvPropertyVariant("aid", qlonglong(id));
}

void MpvObject::setSubtitleTrack(int64_t id)
{
    setMpvPropertyVariant("sid", qlonglong(id));
}

void MpvObject::setVideoTrack(int64_t id)
{
    setMpvPropertyVariant("vid", qlonglong(id));
}

void MpvObject::setDrawLogo(bool yes)
{
    widget->setDrawLogo(yes);
}

void MpvObject::setVolume(int64_t volume)
{
    setMpvPropertyVariant("volume", qlonglong(volume));
}

bool MpvObject::eofReached()
{
    return getMpvPropertyVariant("eof-reached").toBool();
}

void MpvObject::setClientDebuggingMessages(bool yes)
{
    debugMessages = yes;
}

void MpvObject::setMpvLogLevel(QString logLevel)
{
    emit ctrlSetLogLevel(logLevel);
}

double MpvObject::playLength()
{
    return playLength_;
}

double MpvObject::playTime()
{
    return playTime_;
}

QSize MpvObject::videoSize()
{
    return videoSize_;
}

bool MpvObject::clientDebuggingMessages()
{
    return debugMessages;
}

void MpvObject::setCachedMpvOption(const QString &option, const QVariant &value)
{
    if (cachedState.contains(option) && cachedState.value(option) == value)
        return;
    cachedState.insert(option, value);
    setMpvOptionVariant(option, value);
}

QVariant MpvObject::blockingMpvCommand(QVariant params)
{
    QVariant v;
    QMetaObject::invokeMethod(ctrl, "command",
                              Qt::BlockingQueuedConnection,
                              Q_RETURN_ARG(QVariant, v),
                              Q_ARG(QVariant, params));
    return v;
}

QVariant MpvObject::blockingSetMpvPropertyVariant(QString name, QVariant value)
{
    int v;
    QMetaObject::invokeMethod(ctrl, "setPropertyVariant",
                              Qt::BlockingQueuedConnection,
                              Q_RETURN_ARG(int, v),
                              Q_ARG(QString, name),
                              Q_ARG(QVariant, value));
    return v == MPV_ERROR_SUCCESS ? QVariant()
                                  : QVariant::fromValue(MpvErrorCode(v));
}

QVariant MpvObject::blockingSetMpvOptionVariant(QString name, QVariant value)
{
    int v;
    QMetaObject::invokeMethod(ctrl, "setOptionVariant",
                              Qt::BlockingQueuedConnection,
                              Q_RETURN_ARG(int, v),
                              Q_ARG(QString, name),
                              Q_ARG(QVariant, value));
    return v == MPV_ERROR_SUCCESS ? QVariant()
                                  : QVariant::fromValue(MpvErrorCode(v));
}

QVariant MpvObject::getMpvPropertyVariant(QString name)
{
    QVariant v;
    QMetaObject::invokeMethod(ctrl, "getPropertyVariant",
                              Qt::BlockingQueuedConnection,
                              Q_RETURN_ARG(QVariant, v),
                              Q_ARG(QString, name));
    return v;
}


void MpvObject::setMpvPropertyVariant(QString name, QVariant value)
{
    if (debugMessages)
        qDebug() << "property set " << name << value;
    emit ctrlSetPropertyVariant(name, value);
}

void MpvObject::setMpvOptionVariant(QString name, QVariant value)
{
    if (debugMessages)
        qDebug() << "option set " << name << value;
    emit ctrlSetOptionVariant(name, value);
}

void MpvObject::showCursor()
{
    widget->self()->setCursor(Qt::ArrowCursor);
}

void MpvObject::hideCursor()
{
    widget->self()->setCursor(Qt::BlankCursor);
}


#define HANDLE_PROP(p, method, converter, dflt) \
    if (name == p) { \
        if (ok && v.canConvert<decltype(dflt)>()) \
            method(v.converter()); \
        else \
            method(dflt); \
        return; \
    }

void MpvObject::ctrl_mpvPropertyChanged(QString name, QVariant v)
{
    if (debugMessages)
        qDebug() << "property changed " << name << v;

    bool ok = v.type() < QVariant::UserType;
    //FIXME: use constant-time map to function lookup
    HANDLE_PROP("time-pos", emit self_playTimeChanged, toDouble, -1.0);
    HANDLE_PROP("duration", emit self_playLengthChanged, toDouble, -1.0);
    HANDLE_PROP("seekable", emit seekableChanged, toBool, false);
    HANDLE_PROP("pause", emit pausedChanged, toBool, true);
    HANDLE_PROP("media-title", emit mediaTitleChanged, toString, QString());
    HANDLE_PROP("chapter-metadata", emit chapterDataChanged, toMap, QVariantMap());
    HANDLE_PROP("chapter-list", emit chaptersChanged, toList, QVariantList());
    HANDLE_PROP("track-list", emit tracksChanged, toList, QVariantList());
    HANDLE_PROP("estimated-vf-fps", emit fpsChanged, toDouble, 0.0);
    HANDLE_PROP("avsync", emit avsyncChanged, toDouble, 0.0);
    HANDLE_PROP("frame-drop-count", emit displayFramedropsChanged, toLongLong, 0ll);
    HANDLE_PROP("decoder-frame-drop-count", emit decoderFramedropsChanged, toLongLong, 0ll);
    HANDLE_PROP("audio-bitrate", emit audioBitrateChanged, toDouble, 0.0);
    HANDLE_PROP("video-bitrate", emit videoBitrateChanged, toDouble, 0.0);
    HANDLE_PROP("metadata", emit self_metadata, toMap, QVariantMap());
    HANDLE_PROP("audio-device-list", emit self_audioDeviceList, toList, QVariantList());
    HANDLE_PROP("filename", emit fileNameChanged, toString, QString());
    HANDLE_PROP("file-format", emit fileFormatChanged, toString, QString());
    HANDLE_PROP("file-date-created", emit fileCreationTimeChanged, toLongLong, 0ll);
    HANDLE_PROP("file-size", emit fileSizeChanged, toLongLong, 0ll);
    HANDLE_PROP("path", emit filePathChanged, toString, QString());
}

void MpvObject::ctrl_logMessage(QString message)
{
    qDebug() << message;
}

void MpvObject::ctrl_clientMessage(uint64_t id, const QStringList &args)
{
    Q_UNUSED(id);
    if (args[1] == QString::number(HOOK_UNLOAD_CALLBACK_ID)) {
        QVariantList playlist = getMpvPropertyVariant("playlist").toList();
        if (playlist.count() > 1)
            emit playlistChanged(playlist);
        emit ctrlCommand(QStringList({"hook-ack", args[2]}));
    }
}

void MpvObject::ctrl_unhandledMpvEvent(int eventLevel)
{
    switch(eventLevel) {
    case MPV_EVENT_START_FILE: {
        if (debugMessages)
            qDebug() << "start file";
        emit playbackLoading();
        break;
    }
    case MPV_EVENT_FILE_LOADED: {
        if (debugMessages)
            qDebug() << "file loaded";
        emit playbackStarted();
        break;
    }
    case MPV_EVENT_END_FILE: {
        if (debugMessages)
            qDebug() << "end file";
        emit playbackFinished();
        break;
    }
    case MPV_EVENT_IDLE: {
        if (debugMessages)
            qDebug() << "idling";
        emit playbackIdling();
        break;
    }
    case MPV_EVENT_SHUTDOWN: {
        if (debugMessages)
            qDebug() << "event shutdown";
        emit playbackFinished();
        break;
    }
    }
}

void MpvObject::ctrl_videoSizeChanged(QSize size)
{
    videoSize_ = size;
    emit videoSizeChanged(videoSize_);
}

void MpvObject::self_playTimeChanged(double playTime)
{
    playTime_ = playTime;
    emit playTimeChanged(playTime);
}

void MpvObject::self_playLengthChanged(double playLength)
{
    playLength_ = playLength;
    emit playLengthChanged(playLength);
}

void MpvObject::self_metadata(QVariantMap metadata)
{
    QVariantMap map;
    for (auto it = metadata.begin(); it != metadata.end(); it++)
        map.insert(it.key().toLower(), it.value());
    emit metaDataChanged(map);
}

void MpvObject::self_audioDeviceList(const QVariantList &list)
{
    emit audioDeviceList(AudioDevice::listFromVList(list));
}

void MpvObject::self_mouseMoved()
{
    if (hideTimer->interval() > 0)
        hideTimer->start();
    showCursor();
}

void MpvObject::hideTimer_timeout()
{
    hideCursor();
}

//----------------------------------------------------------------------------

MpvWidgetInterface::MpvWidgetInterface(MpvObject *object)
    : mpvObject(object)
{
}

MpvWidgetInterface::~MpvWidgetInterface()
{
    ctrl = nullptr;
}

void MpvWidgetInterface::setController(MpvController *controller)
{
    ctrl = controller;
}

void MpvWidgetInterface::setLogoUrl(const QString &filename)
{
    Q_UNUSED(filename);
}

void MpvWidgetInterface::setLogoBackground(const QColor &color)
{
    Q_UNUSED(color);
}

void MpvWidgetInterface::setDrawLogo(bool yes)
{
    Q_UNUSED(yes);
}


//----------------------------------------------------------------------------

static void* GLAPIENTRY glMPGetNativeDisplay(const char* name) {
#if defined(Q_OS_UNIX) && !defined(Q_OS_DARWIN)
    if (!strcmp(name, "x11")) {
        return QX11Info::display();
    }
#elif defined(Q_OS_WIN)
    if (!strcmp(name, "IDirect3DDevice9")) {
        // Do something here ?
    }
#else
    Q_UNUSED(name);
#endif
    return nullptr;
}

static void *get_proc_address(void *ctx, const char *name) {
    (void)ctx;
    auto glctx = QOpenGLContext::currentContext();
    if (!strcmp(name, "glMPGetNativeDisplay"))
        return (void*)glMPGetNativeDisplay;
    void *res = glctx ? (void*)glctx->getProcAddress(QByteArray(name)) : nullptr;

#ifdef Q_OS_WIN32
    // QOpenGLContext::getProcAddress() in Qt 5.6 and below doesn't resolve all
    // core OpenGL functions, so fall back to Windows' GetProcAddress().
    if (!res) {
        HMODULE module = (HMODULE)QOpenGLContext::openGLModuleHandle();
        if (!module) {
            // QOpenGLContext::openGLModuleHandle() returns NULL when Qt isn't
            // using dynamic OpenGL. In this case, openGLModuleType() can be
            // used to determine which module to query.
            switch (QOpenGLContext::openGLModuleType()) {
            case QOpenGLContext::LibGL:
                module = GetModuleHandleW(L"opengl32.dll");
                break;
            case QOpenGLContext::LibGLES:
                module = GetModuleHandleW(L"libGLESv2.dll");
                break;
            }
        }
        if (module)
            res = (void*)GetProcAddress(module, name);
    }
#endif

    return res;
}

MpvGlCbWidget::MpvGlCbWidget(MpvObject *object, QWidget *parent) :
    QOpenGLWidget(parent), MpvWidgetInterface(object)
{
    connect(this, &QOpenGLWidget::frameSwapped,
            this, &MpvGlCbWidget::self_frameSwapped);
    connect(mpvObject, &MpvObject::playbackStarted,
            this, &MpvGlCbWidget::self_playbackStarted);
    connect(mpvObject, &MpvObject::playbackFinished,
            this, &MpvGlCbWidget::self_playbackFinished);
    setContextMenuPolicy(Qt::CustomContextMenu);
}

MpvGlCbWidget::~MpvGlCbWidget()
{
    if (glMpv) {
        makeCurrent();
        mpv_opengl_cb_set_update_callback(glMpv, nullptr, nullptr);
        mpv_opengl_cb_uninit_gl(glMpv);
    }
    if (logo) {
        delete logo;
        logo = nullptr;
    }
}

QWidget *MpvGlCbWidget::self()
{
    return this;
}

void MpvGlCbWidget::initMpv()
{
    // grab a copy of the mpvGl draw context
    QMetaObject::invokeMethod(ctrl, "mpvDrawContext",
                              Qt::BlockingQueuedConnection,
                              Q_RETURN_ARG(mpv_opengl_cb_context *, glMpv));

    // ask mpv to make draw requests to us
    mpv_opengl_cb_set_update_callback(glMpv, MpvGlCbWidget::ctrl_update,
                                      (void *)this);
}

void MpvGlCbWidget::setLogoUrl(const QString &filename)
{
    makeCurrent();
    if (!logo) {
        logo = new LogoDrawer(this);
        connect(logo, &LogoDrawer::logoSize,
                mpvObject, &MpvObject::logoSizeChanged);
    }
    logo->setLogoUrl(filename);
    logo->resizeGL(width(), height());
    if (drawLogo)
        update();
    doneCurrent();
}

void MpvGlCbWidget::setLogoBackground(const QColor &color)
{
    logo->setLogoBackground(color);
}

void MpvGlCbWidget::setDrawLogo(bool yes)
{
    drawLogo = yes;
    update();
}

void MpvGlCbWidget::initializeGL()
{
    if (mpv_opengl_cb_init_gl(glMpv, nullptr, get_proc_address, nullptr) < 0)
        throw std::runtime_error("[MpvWidget] cb init gl failed.");

    if (!logo)
        logo = new LogoDrawer(this);
}

void MpvGlCbWidget::paintGL()
{
    //FIXME: use log()
    if (mpvObject->clientDebuggingMessages())
        qDebug() << "paintGL";
    if (!drawLogo) {
        mpv_opengl_cb_draw(glMpv, defaultFramebufferObject(),
                           glWidth, -glHeight);
    } else {
        logo->paintGL(this);
    }
}

void MpvGlCbWidget::resizeGL(int w, int h)
{
    qreal r = devicePixelRatio();
    glWidth = int(w * r);
    glHeight = int(h * r);
    logo->resizeGL(width(),height());
}

void MpvGlCbWidget::mouseMoveEvent(QMouseEvent *event)
{
    emit mpvObject->mouseMoved(event->x(), event->y());
    QOpenGLWidget::mouseMoveEvent(event);
}

void MpvGlCbWidget::mousePressEvent(QMouseEvent *event)
{
    emit mpvObject->mousePress(event->x(), event->y());
    QOpenGLWidget::mousePressEvent(event);
}

void MpvGlCbWidget::ctrl_update(void *ctx)
{
    QMetaObject::invokeMethod(reinterpret_cast<MpvGlCbWidget*>(ctx), "maybeUpdate");
}

void MpvGlCbWidget::maybeUpdate()
{
    if (window()->isMinimized()) {
        makeCurrent();
        paintGL();
        context()->swapBuffers(context()->surface());
        self_frameSwapped();
        doneCurrent();
    } else {
        update();
    }
}

void MpvGlCbWidget::self_frameSwapped()
{
    if (!drawLogo)
        mpv_opengl_cb_report_flip(glMpv, 0);
}

void MpvGlCbWidget::self_playbackStarted()
{
    drawLogo = false;
}

void MpvGlCbWidget::self_playbackFinished()
{
    drawLogo = true;
    update();
}



MpvCallback::MpvCallback(const Callback &callback,
                         QObject *owner)
    : QObject(owner)
{
    this->callback = callback;
}

void MpvCallback::reply(QVariant value)
{
    callback(value);
    deleteLater();
}



MpvController::MpvController(QObject *parent) : QObject(parent),
    glMpv(nullptr), lastVideoSize(0,0)
{
    throttler = new QTimer(this);
    connect(throttler, &QTimer::timeout,
            this, &MpvController::flushProperties);
    throttler->setInterval(1000/12);
    throttler->start();
}

MpvController::~MpvController()
{
    mpv_set_wakeup_callback(mpv, nullptr, nullptr);
    throttler->deleteLater();
}

void MpvController::create(bool video, bool audio)
{
    mpv = mpv::qt::Handle::FromRawHandle(mpv_create());
    if (!mpv)
        throw std::runtime_error("could not create mpv context");

    setOptionVariant("scripts", Platform::resourcesPath() + "/scripts/stats.lua");

    if (mpv_initialize(mpv) < 0)
        throw std::runtime_error("could not initialize mpv context");

    if (!audio) {
        setOptionVariant("ao", "null");
        setOptionVariant("no-audio", true);
    }
    if (!video) {
        // NOTE: this completely skips setting up the gl interface.
        setOptionVariant("vo", "null");
        setOptionVariant("no-video", true);
    } else {
        setOptionVariant("vo", "opengl-cb");
        glMpv = (mpv_opengl_cb_context *)mpv_get_sub_api(mpv, MPV_SUB_API_OPENGL_CB);
        if (!glMpv)
            throw std::runtime_error("OpenGL not compiled in");
    }
    mpv_set_wakeup_callback(mpv, MpvController::mpvWakeup, this);

    protocolList_ = getPropertyVariant("protocol-list").toStringList();
}

void MpvController::addHook(const QString &name, int id)
{
    command(QStringList({"hook-add", name, QString::number(id), "0"}));
}

int MpvController::observeProperties(const MpvController::PropertyList &properties,
                                      const QSet<QString> &throttled)
{
    int rval = 0;
    foreach (const MpvProperty &item, properties)
        rval  = std::min(rval, mpv_observe_property(mpv, item.userData, item.name.toUtf8().data(), item.format));
    throttledProperties.unite(throttled);
    return rval;
}

int MpvController::unobservePropertiesById(const QSet<uint64_t> &ids)
{
    int rval = 0;
    foreach (uint64_t id, ids)
        rval = std::min(rval, mpv_unobserve_property(mpv, id));
    return rval;
}

void MpvController::setThrottleTime(int msec)
{
    throttler->setInterval(msec);
}

QString MpvController::clientName()
{
    return QString::fromUtf8(mpv_client_name(mpv));
}

QStringList MpvController::protocolList()
{
    return protocolList_;
}

int64_t MpvController::timeMicroseconds()
{
    return mpv_get_time_us(mpv);
}

unsigned long MpvController::apiVersion()
{
    return mpv_client_api_version();
}

void MpvController::setLogLevel(QString logLevel)
{
    mpv_request_log_messages(mpv, logLevel.toUtf8().data());
}

mpv_opengl_cb_context* MpvController::mpvDrawContext()
{
    return glMpv;
}

int MpvController::setOptionVariant(QString name, const QVariant &value)
{
    return mpv::qt::set_option_variant(mpv, name, value);
}

QVariant MpvController::command(const QVariant &params)
{
    if (params.canConvert<QString>()) {
        int value = mpv_command_string(mpv, params.toString().toUtf8().data());
        if (value < 0)
            return QVariant::fromValue(MpvErrorCode(value));
        return QVariant();
    }

    mpv::qt::node_builder node(params);
    mpv_node res;
    int value = mpv_command_node(mpv, node.node(), &res);
    if (value < 0)
        return QVariant::fromValue(MpvErrorCode(value));
    mpv::qt::node_autofree f(&res);
    QVariant v = mpv::qt::node_to_variant(&res);
    return v;
}

int MpvController::setPropertyVariant(const QString &name, const QVariant &value)
{
    return mpv::qt::set_property_variant(mpv, name, value);
}

QVariant MpvController::getPropertyVariant(const QString &name)
{
    mpv_node node;
    int r = mpv_get_property(mpv, name.toUtf8().data(), MPV_FORMAT_NODE, &node);
    if (r < 0)
        return QVariant::fromValue<MpvErrorCode>(MpvErrorCode(r));
    QVariant v = mpv::qt::node_to_variant(&node);
    mpv_free_node_contents(&node);
    return v;
}

int MpvController::setPropertyString(const QString &name, const QString &value)
{
    return mpv_set_property_string(mpv, name.toUtf8().data(), value.toUtf8().data());
}

QString MpvController::getPropertyString(const QString &name)
{
    char *c = mpv_get_property_string(mpv, name.toUtf8().data());
    if (!c)
        return QString();
    QByteArray b(c);
    mpv_free(c);
    return QString::fromUtf8(b);
}

void MpvController::commandAsync(const QVariant &params, MpvCallback *callback)
{
    mpv::qt::node_builder node(params);
    mpv_command_node_async(mpv, reinterpret_cast<uint64_t>(callback),
                           node.node());
}

void MpvController::setPropertyVariantAsync(const QString &name,
                                            const QVariant &value,
                                            MpvCallback *callback)
{
    mpv::qt::node_builder node(value);
    mpv_set_property_async(mpv, reinterpret_cast<uint64_t>(callback),
                           name.toUtf8().data(), MPV_FORMAT_NODE, node.node());
}

void MpvController::getPropertyVariantAsync(const QString &name,
                                            MpvCallback *callback)
{
    mpv_get_property_async(mpv, reinterpret_cast<uint64_t>(callback),
                           name.toUtf8().data(), MPV_FORMAT_NODE);
}

void MpvController::parseMpvEvents()
{
    // Process all events, until the event queue is empty.
    while (mpv) {
        mpv_event *event = mpv_wait_event(mpv, 0);
        if (event->event_id == MPV_EVENT_NONE) {
            break;
        }
        handleMpvEvent(event);
    }
}

void MpvController::setThrottledProperty(const QString &name, const QVariant &v, uint64_t userData)
{
    throttledValues.insert(name, QPair<QVariant,uint64_t>(v,userData));
}

void MpvController::flushProperties()
{
    for (auto it = throttledValues.begin(); it != throttledValues.end(); it++)
        emit mpvPropertyChanged(it.key(), it.value().first, it.value().second);
    throttledValues.clear();
}

void MpvController::handleMpvEvent(mpv_event *event)
{
    auto propertyToVariant = [event](mpv_event_property *prop) -> QVariant {
        auto asBool = [&](bool dflt = false) {
            return (prop->format != MPV_FORMAT_FLAG || prop->data == nullptr) ?
                        dflt : *reinterpret_cast<bool*>(prop->data);
        };
        auto asDouble = [&](double dflt = nan("")) {
            return (prop->format != MPV_FORMAT_DOUBLE || prop->data == nullptr) ?
                        dflt : *reinterpret_cast<double*>(prop->data);
        };
        auto asInt64 = [&](int64_t dflt = -1) {
            return (prop->format != MPV_FORMAT_INT64 || prop->data == nullptr) ?
                        dflt : *reinterpret_cast<int64_t*>(prop->data);
        };
        auto asString = [&](QString dflt = QString()) {
            return (!(prop->format == MPV_FORMAT_STRING ||
                      prop->format == MPV_FORMAT_OSD_STRING) ||
                    prop->data == nullptr) ?
                        dflt : QString(*reinterpret_cast<char**>(prop->data));
        };
        auto asNode = [&](QVariant dflt = QVariant()) {
            return (prop->format != MPV_FORMAT_NODE || prop->data == nullptr) ?
                        dflt : mpv::qt::node_to_variant(
                            reinterpret_cast<mpv_node*>(prop->data));
        };
        if (prop->data == nullptr) {
            return QVariant::fromValue<MpvErrorCode>(MpvErrorCode(event->error));
        } else if (prop->format == MPV_FORMAT_NODE) {
            return asNode();
        } else if (prop->format == MPV_FORMAT_INT64) {
            return qlonglong(asInt64());
        } else if (prop->format == MPV_FORMAT_DOUBLE) {
            return asDouble();
        } else if (prop->format == MPV_FORMAT_STRING ||
                   prop->format == MPV_FORMAT_OSD_STRING) {
            return asString();
        } else if (prop->format == MPV_FORMAT_FLAG) {
            return asBool();
        }
        return QVariant();
    };

    switch (event->event_id) {
    case MPV_EVENT_GET_PROPERTY_REPLY: {
        QVariant v = propertyToVariant(reinterpret_cast<mpv_event_property*>(event->data));
        if (!event->reply_userdata)
            return;
        QMetaObject::invokeMethod(reinterpret_cast<MpvCallback*>(event->reply_userdata),
                                  "reply", Qt::QueuedConnection,
                                  Q_ARG(QVariant, v));
        break;
    }
    case MPV_EVENT_COMMAND_REPLY:
    case MPV_EVENT_SET_PROPERTY_REPLY: {
        QVariant v = QVariant::fromValue<MpvErrorCode>(MpvErrorCode(event->error));
        if (!event->reply_userdata)
            return;
        QMetaObject::invokeMethod(reinterpret_cast<MpvCallback*>(event->reply_userdata),
                                  "reply", Qt::QueuedConnection,
                                  Q_ARG(QVariant, v));
        break;
    }
    case MPV_EVENT_PROPERTY_CHANGE: {
        QVariant v = propertyToVariant(reinterpret_cast<mpv_event_property*>(event->data));
        QString propname = QString::fromUtf8(reinterpret_cast<mpv_event_property*>(event->data)->name);
        if (throttledProperties.contains(propname))
            setThrottledProperty(propname, v, event->reply_userdata);
        else
            emit mpvPropertyChanged(propname, v, event->reply_userdata);
        break;
    }
    case MPV_EVENT_LOG_MESSAGE: {
        mpv_event_log_message *msg =
                reinterpret_cast<mpv_event_log_message*>(event->data);
        emit logMessage(QString("[%1] %2: %3").arg(msg->prefix, msg->level,
                                                   msg->text));
        break;
    }
    case MPV_EVENT_CLIENT_MESSAGE: {
        mpv_event_client_message *msg =
                reinterpret_cast<mpv_event_client_message*>(event->data);
        QStringList list;
        for (int i = 0; i < msg->num_args; i++)
            list.append(msg->args[i]);
        emit clientMessage(event->reply_userdata, list);
        break;
    }
    case MPV_EVENT_VIDEO_RECONFIG: {
        // Retrieve the new video size.
        QVariant vw, vh;
        vw = getPropertyVariant("width");
        vh = getPropertyVariant("height");
        int w, h;
        if (!vw.canConvert<MpvErrorCode>() && !vh.canConvert<MpvErrorCode>()
                && (w = vw.toInt()) > 0 && (h = vh.toInt()) > 0) {
            QSize videoSize(w, h);
            if (lastVideoSize != videoSize) {
                emit videoSizeChanged(videoSize);
                lastVideoSize = videoSize;
            }
        } else if (!lastVideoSize.isEmpty()) {
            lastVideoSize = QSize();
            emit videoSizeChanged(QSize());
        }
        break;
    }
    default:
        emit unhandledMpvEvent(event->event_id);
    }
}

void MpvController::mpvWakeup(void *ctx)
{
    QMetaObject::invokeMethod((MpvController*)ctx, "parseMpvEvents",
                              Qt::QueuedConnection);
}
