/**
*** \file preferenceswidget.cpp
*** \brief preferenceswidget.h implementation.
*** \details Implementation file for preferenceswidget.h.
**/

/*
** This file is part of moonphase.
** Copyright (C) 2014-2018 by Alan Wise <awisesoftware@gmail.com>.
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


/**
*** \brief preferenceswidget.cpp identifier.
*** \details Identifier for preferenceswidget.cpp.
**/
#define   PREFERENCESWIDGET_CPP


/****
*****
***** INCLUDES
*****
****/

#include  "preferenceswidget.h"
#ifdef    DEBUG_PREFERENCESWIDGET_CPP
#ifndef   USE_DEBUGLOG
#define   USE_DEBUGLOG
#endif    /* USE_DEBUGLOG */
#endif    /* DEBUG_PREFERENCESWIDGET_CPP */
#include  "debuglog.h"
#include  "messagelog.h"

#include  "drawframe.h"

#include  <QShowEvent>


/****
*****
***** DEFINES
*****
****/

/**
*** \brief Animation update rate.
*** \details Time between animation updates.
**/
#define   ANIMATIONTIMER_RATE     (200)         /* in milliseconds */


/****
*****
***** DATA TYPES
*****
****/


/****
*****
***** PROTOTYPES
*****
****/


/****
*****
***** DATA
*****
****/


/****
*****
***** VARIABLES
*****
****/


/****
*****
***** FUNCTIONS
*****
****/

PREFERENCESWIDGET_C::PREFERENCESWIDGET_C(QWidget *pParent) : QWidget(pParent)
{
  DEBUGLOG_Printf1("PREFERENCESWIDGET_C::PREFERENCESWIDGET_C(%p)",pParent);
  DEBUGLOG_LogIn();

  m_OriginalBackgroundColor=QColor(0,0,0);
  MoonAnimation_Initialize(&m_MoonPreviewImages);
  /* Check return. */
  m_PreviewPercentCounter=0;
  m_OriginalRemindOncePerSessionFlag=false;

  /* Set up the user interface. */
  setupUi(this);

  /* Create and set up the animation timer. */
  connect(&m_AnimationTimer,SIGNAL(timeout()),
      this,SLOT(AnimationTimerTriggeredSlot()));

  DEBUGLOG_LogOut();
  return;
}

PREFERENCESWIDGET_C::~PREFERENCESWIDGET_C(void)
{
  ERRORCODE_T ErrorCode;


  DEBUGLOG_Printf0("PREFERENCESWIDGET_C::~PREFERENCESWIDGET_C()");
  DEBUGLOG_LogIn();

  ErrorCode=MoonAnimation_Uninitialize(&m_MoonPreviewImages);
  MESSAGELOG_LogError(ErrorCode);

  DEBUGLOG_LogOut();
  return;
}

SETTINGS_C::APPLICATIONPREFERENCES_C PREFERENCESWIDGET_C::GetPreferences(void)
{
  SETTINGS_C::APPLICATIONPREFERENCES_C Preferences;


  DEBUGLOG_Printf0("PREFERENCESWIDGET_C::GetPreferences()");
  DEBUGLOG_LogIn();

  /* Get the widget data. */
  Preferences.SetAllowMultipleInstancesFlag(
      m_pAllowMultipleInstancesCheckBox->isChecked());
  Preferences.SetAnimationPathname(m_pAnimationFilenameChooser->text());
  Preferences.SetBackgroundColor(m_pBackgroundColorButton->currentColor());
  Preferences.SetConfirmDiscardFlag(m_pConfirmDiscardCheckBox->isChecked());
  Preferences.SetConfirmQuitFlag(m_pConfirmQuitCheckBox->isChecked());
  Preferences.SetRemindOncePerSessionFlag(
      m_pRemindOncePerSessionCheckBox->isChecked());
  Preferences.SetStillRunningReminderFlag(m_pStillRunningReminder->isChecked());

  Preferences.SetUpdateInterval(m_pUpdateIntervalSpinBox->value());
  Preferences.SetUseOpaqueBackgroundFlag(
      m_pUseOpaqueBackgroundGroupBox->isChecked());

  DEBUGLOG_LogOut();
  return(Preferences);
}

void PREFERENCESWIDGET_C::hideEvent(QShowEvent *pEvent)
{
  DEBUGLOG_Printf1("PREFERENCESWIDGET_C::hideEvent(%p)",pEvent);
  DEBUGLOG_LogIn();

  SetVisible(false);
  pEvent->accept();

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::SetPreferences(
    SETTINGS_C::APPLICATIONPREFERENCES_C const *pPreferences)
{
  DEBUGLOG_Printf1("PREFERENCESWIDGET_C::SetPreferences(%p)",pPreferences);
  DEBUGLOG_LogIn();

  /* Set the widget data. */
  m_pAllowMultipleInstancesCheckBox->setChecked(
      pPreferences->GetAllowMultipleInstancesFlag());
  m_pAnimationFilenameChooser->setText(pPreferences->GetAnimationPathname());
  m_OriginalBackgroundColor=pPreferences->GetBackgroundColor();
  m_pBackgroundColorButton->setCurrentColor(m_OriginalBackgroundColor);
  m_pConfirmDiscardCheckBox->setChecked(pPreferences->GetConfirmDiscardFlag());
  m_pConfirmQuitCheckBox->setChecked(pPreferences->GetConfirmQuitFlag());
  m_OriginalRemindOncePerSessionFlag=pPreferences->GetRemindOncePerSessionFlag();
  m_pRemindOncePerSessionCheckBox->setChecked(
      m_OriginalRemindOncePerSessionFlag);
  m_pStillRunningReminder->setChecked(
      pPreferences->GetStillRunningReminderFlag());
  m_pUpdateIntervalSpinBox->setValue(pPreferences->GetUpdateInterval());
  m_pUseOpaqueBackgroundGroupBox->setChecked(
      pPreferences->GetUseOpaqueBackgroundFlag());

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::SetVisible(bool VisibleFlag)
{
  DEBUGLOG_Printf1("PREFERENCESWIDGET_C::SetVisible(%u)",VisibleFlag);
  DEBUGLOG_LogIn();

  if (VisibleFlag==true)
  {
    m_PreviewPercentCounter=0;
    m_AnimationTimer.start(ANIMATIONTIMER_RATE);
  }
  else
    m_AnimationTimer.stop();

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::showEvent(QShowEvent *pEvent)
{
  DEBUGLOG_Printf1("PREFERENCESWIDGET_C::showEvent(%p)",pEvent);
  DEBUGLOG_LogIn();

  SetVisible(true);
  pEvent->accept();

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::AnimationPathnameChangedSlot(void)
{
  DEBUGLOG_Printf0("PREFERENCESWIDGET_C::AnimationPathnameChangedSlot()");
  DEBUGLOG_LogIn();

  /* Try to load the file. Don't care if it fails. */
  MoonAnimation_ReadFile(
      &m_MoonPreviewImages,qPrintable(m_pAnimationFilenameChooser->text()));
  PreferencesChangedSlot();

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::AnimationTimerTriggeredSlot(void)
{
  QString PercentString;
  QPixmap Pixmap;


  DEBUGLOG_Printf0("PREFERENCESWIDGET_C::AnimationTimerTriggeredSlot()");
  DEBUGLOG_LogIn();

  /* Update the percent label. */
  PercentString.sprintf("%d%%",m_PreviewPercentCounter);
  m_pPreviewPercentLabel->setText(PercentString);

  /* Update the preview image. */
  Pixmap=DrawFrame(&m_MoonPreviewImages,m_PreviewPercentCounter,
      m_pUseOpaqueBackgroundGroupBox->isChecked(),
      m_pBackgroundColorButton->currentColor());
  m_pPreviewViewLabel->setPixmap(
      Pixmap.scaled(
      m_pPreviewViewLabel->width(),m_pPreviewViewLabel->height(),
      Qt::KeepAspectRatio,Qt::SmoothTransformation));

  /* Next percent. */
  m_PreviewPercentCounter=(m_PreviewPercentCounter+1)%101;

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::PreferencesChangedSlot(void)
{
  DEBUGLOG_Printf0("PREFERENCESWIDGET_C::PreferencesChangedSlot()");
  DEBUGLOG_LogIn();

  /* Update a sub-option. */
  m_pRemindOncePerSessionCheckBox->setEnabled(
      m_pStillRunningReminder->isChecked()!=false);

  /* Send a signal to update the "Apply" button. */
  emit PreferencesChangedSignal();

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::StillRunningReminderClickedSlot(void)
{
  DEBUGLOG_Printf0("PREFERENCESWIDGET_C::StillRunningReminderClickedSlot()");
  DEBUGLOG_LogIn();

  /* If the "Still Running Reminder" checkbox is checked, then the
      "Once Per Session" checkbox is toggled, and then the "Still..." is
      unchecked, the "Apply" button should disable, but doesn't. So, when the
      "Still..." checkbox is cleared, the previous "Once..." state is restored
      to its prior value. */
  if (m_pStillRunningReminder->isChecked()==false)
    m_pRemindOncePerSessionCheckBox->setChecked(
        m_OriginalRemindOncePerSessionFlag);

  DEBUGLOG_LogOut();
  return;
}

void PREFERENCESWIDGET_C::UseOpaqueBackgroundClickedSlot(void)
{
  DEBUGLOG_Printf0("PREFERENCESWIDGET_C::UseOpaqueBackgroundClickedSlot()");
  DEBUGLOG_LogIn();

  /* If the "Use Opaque Background" checkbox is checked, then a color is
      selected from the color button, and then the "Use Opaque Background" is
      unchecked, the "Apply" button should disable, but doesn't. When the
      "Use..." checkbox is cleared, the previous background color is restored
      to its prior value. */
  if (m_pUseOpaqueBackgroundGroupBox->isChecked()==false)
    m_pBackgroundColorButton->setCurrentColor(m_OriginalBackgroundColor);

  DEBUGLOG_LogOut();
  return;
}


#undef    PREFERENCESWIDGET_CPP


/**
*** preferenceswidget.cpp
**/
