/**
*** \file controlpaneldialog.cpp
*** \brief controlpaneldialog.h implementation.
*** \details Implementation file for controlpaneldialog.h.
**/

/*
** This file is part of moonphase.
** Copyright (C) 2014-2018 by Alan Wise <awisesoftware@gmail.com>.
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


/**
*** \brief controlpaneldialog.cpp identifier.
*** \details Identifier for controlpaneldialog.cpp.
**/
#define   CONTROLPANELDIALOG_CPP


/****
*****
***** INCLUDES
*****
****/

#include  "controlpaneldialog.h"
#ifdef    DEBUG_CONTROLPANELDIALOG_CPP
#ifndef   USE_DEBUGLOG
#define   USE_DEBUGLOG
#endif    /* USE_DEBUGLOG */
#endif    /* DEBUG_CONTROLPANELDIALOG_CPP */
#include  "debuglog.h"
#include  "messagelog.h"

#include  "config.h"
#include  "licenseagreement.h"
#include  "changelog.h"
#include  "informationitem.h"
#include  "informationitemoptions.h"
#include  "informationpaneldialog.h"
#include  "settings.h"
#include  "information.h"
#include  "informationoptionsdialog.h"
#include  "versionstring.h"

#include  <QDesktopWidget>
#include  <QMenu>
#include  <QTimer>
#include  <QMessageBox>
#include  <QPainter>


/****
*****
***** DEFINES
*****
****/

/**
*** \brief Icon update rate.
*** \details Time between checks for any icon updates.
**/
#define   UPDATETIMER_RATE        (60*60*1000)  /* 1 hour, in milliseconds */

/**
*** \brief Information panel update rate.
*** \details Time between checks for information panel updates.
**/
#define   INFORMATIONPANELTIMER_RATE  (1000)    /* 1 second, in milliseconds */

/**
*** \brief Binary subdirectory.
*** \details Subdirectory containing the executable.
**/
#define   DIRECTORY_BINARY  "/bin"

/**
*** \brief Image subdirectory.
*** \details Subdirectory containing the moonphase images.
**/
#define   DIRECTORY_IMAGES  "/share/" MOONPHASEQT_EXECUTABLENAME

/**
*** \brief Default animation.
*** \details Filename of the default animation.
**/
#define   FILENAME_IMAGE    "moon_56frames.png"

/**
*** \brief Read animation warning.
*** \details Displays a message box informing the user that there was an error
***   reading the moon animation file.
**/
#define   READANIMATIONWARNING(p) \
    QMessageBox::warning(p,tr(MOONPHASEQT_DISPLAYNAME), \
        tr("Unable to read moon animation file. Please select a new file."));

/**
*** \brief Information item role.
*** \details Role to use to store the information item associated with the list
***   widget item.
**/
#define   ROLE_INFORMATIONTYPE    (Qt::UserRole)

/**
*** \brief Convert QVariant into INFORMATIONPANELITEM_C.
*** \details Retrieves a QVariant pointer from the list widget item and converts
***   it into an INFORMATIONPANELITEM_C pointer.
**/
#define   LWI2IPI(lwi)  \
              (lwi)->data(ROLE_INFORMATIONTYPE).value<INFORMATIONPANELITEM_C*>()


/****
*****
***** DATA TYPES
*****
****/

/**
*** \brief Make INFORMATIONPANELITEM_C known to QMetaType.
*** \details Make INFORMATIONPANELITEM_C known to QMetaType.
**/
Q_DECLARE_METATYPE(INFORMATIONPANELITEM_C*);


/****
*****
***** PROTOTYPES
*****
****/


/****
*****
***** DATA
*****
****/


/****
*****
***** VARIABLES
*****
****/


/****
*****
***** FUNCTIONS
*****
****/

CONTROLPANELDIALOG_C::CONTROLPANELDIALOG_C(QWidget *pParent) : QDialog(pParent)
{
  ERRORCODE_T ErrorCode;
  QString BaseDirectory;
  QFileInfo FileInfo;
  bool FoundFlag;
  int Index;
  char const *pLabel;


  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::CONTROLPANELDIALOG_C(%p)",pParent);
  DEBUGLOG_LogIn();

  m_pTrayIconMenu=NULL;
  m_pTrayIcon=NULL;
  m_pInformationPanelDialog=NULL;
  m_pUpdateTimer=NULL;
  m_pInformationPanelTimer=NULL;
  m_UpdateIntervalCounter=0;
  m_CloseReminderIssued=false;
  m_StartUpFlag=true;
  m_FirstUpdateFlag=true;
  m_NextUpdateCheck=QDate::currentDate();
  m_pDoubleClickTimeoutTimer=NULL;

  /* Set up the user interface. */
  setupUi(this);

  /* Set window title. */
  setWindowTitle(windowTitle()+tr(" - ")+tr(MOONPHASEQT_DISPLAYNAME));

  /* Add minimize button. */
  setWindowFlags(Qt::Window);

  /* Because of above call, window starts in odd places. Center it instead. */
  move(QPoint(
    (QApplication::desktop()->width()-frameGeometry().width())/2,
    (QApplication::desktop()->height()-frameGeometry().height())/2));

  /* Create and set up the tray icon context menu. */
  m_pTrayIconMenu=new QMenu(this);
  m_pTrayIconMenu->addAction(m_pShowControlPanelAction);
  m_pTrayIconMenu->addSeparator();
  m_pTrayIconMenu->addAction(m_pShowInformationPanelAction);
  m_pTrayIconMenu->addSeparator();
  m_pTrayIconMenu->addAction(m_pQuitAction);

  /* Create and set up the tray icon. */
  m_pTrayIcon=new QSystemTrayIcon(this);
  m_pTrayIcon->setContextMenu(m_pTrayIconMenu);
  connect(m_pTrayIcon,SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
      this,SLOT(ControlPanelActivatedSlot(QSystemTrayIcon::ActivationReason)));

  /* Connect the quit action to the quit slot. */
  connect(m_pQuitAction,SIGNAL(triggered()),this,SLOT(QuitSlot()));

  /* Create the information panel dialog. */
  m_pInformationPanelDialog=new INFORMATIONPANELDIALOG_C(this);

  /* Create, set up, and start the update timer. */
  m_pUpdateTimer=new QTimer(this);
  connect(m_pUpdateTimer,SIGNAL(timeout()),
      this,SLOT(UpdateTimerTriggeredSlot()));
  m_pUpdateTimer->start(UPDATETIMER_RATE);
  m_pAboutWidget->SetCheckForUpdateWidgetVisible(true);

  /* Create and set up the information panel timer. */
  m_pInformationPanelTimer=new QTimer(this);
  connect(m_pInformationPanelTimer,SIGNAL(timeout()),
      this,SLOT(InformationPanelTimerTriggeredSlot()));

#if       !defined(__APPLE__)
  m_pDoubleClickTimeoutTimer=new QTimer(this);
  m_pDoubleClickTimeoutTimer->setSingleShot(true);
  connect(m_pDoubleClickTimeoutTimer,SIGNAL(timeout()),
      this,SLOT(DoubleClickTimeoutTimerTriggered()));
#endif    /* !defined(__APPLE__) */

  /* Initialize member variables. */
  ErrorCode=MoonData_Initialize(&m_MoonData);
  MESSAGELOG_LogError(ErrorCode);
  if (ErrorCode>0)
  {
    ErrorCode=MoonAnimation_Initialize(&m_MoonTrayImages);
    MESSAGELOG_LogError(ErrorCode);
    if (ErrorCode<0)
      MoonData_Uninitialize(&m_MoonData);
  }
  if (ErrorCode<0)
    throw(ErrorCode);

  /* Read configuration. */
  m_Settings.Load();

  /* The animation pathname hasn't been set. Let's try to find the default. */
  if (m_Settings.GetApplicationPreferencesPointer()->GetAnimationPathname()=="")
  {
    /* Under Unix, the executable is in a base directory + "/bin".
        Under Windows, the executable is in the base directory. */
    BaseDirectory=QCoreApplication::applicationDirPath();

    /* Strip off any "/bin". */
    if (BaseDirectory.endsWith(DIRECTORY_BINARY,Qt::CaseInsensitive)==true)
      BaseDirectory.chop(strlen(DIRECTORY_BINARY));

    /* Check the base directory. Should work on Windows, not Unix. */
    FoundFlag=0;
    FileInfo.setFile(BaseDirectory,FILENAME_IMAGE);
    FoundFlag=FileInfo.exists();
    if (FoundFlag==false)
    {
      /* Check the "images" directory. Should work on Unix, not Windows. */
      FileInfo.setFile(BaseDirectory+QString(DIRECTORY_IMAGES),FILENAME_IMAGE);
      FoundFlag=FileInfo.exists();
    }

    /* Found something, set it. */
    if (FoundFlag==true)
      m_Settings.GetApplicationPreferencesPointer()->SetAnimationPathname(FileInfo.filePath());

    /* Save the defaults to the config file. */
    m_Settings.GetApplicationPreferencesPointer()->Save();
  }

LoadSettings();

  m_pPreferencesWidget->SetPreferences(m_Settings.GetApplicationPreferencesPointer());
  connect(m_pPreferencesWidget,SIGNAL(PreferencesChangedSignal()),
      this,SLOT(PreferencesChangedSlot()));
//  m_pMoonInformationWidget->SetSettings(m_Settings.GetInformationPanelAndToolTipSettings());
//  connect(m_pMoonInformationWidget,SIGNAL(InformationPanelPreferencesAcceptedSignal(SETTINGS_C::IPPREFERENCES_C const &)),
//      this,SLOT(InformationPanelPreferencesAcceptedSlot(SETTINGS_C::IPPREFERENCES_C const &)));

  /* Initialize tabs. */
  InitializeAboutTab();

  /* Populate the information list widget. */
  Index=0;
  goto IntoLoop;
  while(pLabel!=NULL)
  {
    m_pDataListWidget->addItem(pLabel);
    Index++;
IntoLoop:
    MESSAGELOG_Todo("Check ErrorCode.");
    Information_GetLabel(Index,&pLabel);
  }

  /* Attempt to load the animation. */
  ErrorCode=MoonAnimation_ReadFile(
      &m_MoonTrayImages,qPrintable(m_Settings.GetApplicationPreferencesPointer()->GetAnimationPathname()));
  MESSAGELOG_LogError(ErrorCode);

  ForceUpdate();
  SetVisible(ErrorCode<0);

  /* Force Apply button to update. */
  PreferencesChangedSlot();

  /* Show the tray icon. */
  m_pTrayIcon->show();

  /* Any error here is because of an invalid animation file. Show the control
      panel, and display a messsage. */
  if (ErrorCode<0)
  {
    ControlPanelActivatedSlot(QSystemTrayIcon::DoubleClick);
    READANIMATIONWARNING(this);
    ErrorCode=ERRORCODE_SUCCESS;  /* "Error" handled. */
  }

  /* Resize the control panel to minimum size. */
  resize(minimumSizeHint());

  connect(&m_UpdateNotifier,SIGNAL(VersionSignal(QString)),
      this,SLOT(VersionSlot(QString)));
  m_UpdateNotifier.SetURL(
      "https://downloads.sourceforge.net/project/moonphase/current_release");

  m_StartUpFlag=false;

  DEBUGLOG_LogOut();
  return;
}

CONTROLPANELDIALOG_C::~CONTROLPANELDIALOG_C(void)
{
  ERRORCODE_T ErrorCode;
#ifndef   ENABLE_MESSAGELOG
  Q_UNUSED(ErrorCode);
#endif    /* ENABLE_MESSAGELOG */


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::~CONTROLPANELDIALOG_C()");
  DEBUGLOG_LogIn();

  if (m_pUpdateTimer==NULL)
  {
    MESSAGELOG_Warning("m_pUpdateTimer==NULL");
  }
  delete m_pUpdateTimer;
  if (m_pTrayIcon==NULL)
  {
    MESSAGELOG_Warning("m_pTrayIcon==NULL");
  }
  delete m_pTrayIcon;
  if (m_pTrayIconMenu==NULL)
  {
    MESSAGELOG_Warning("m_pTrayIconMenu==NULL");
  }
  delete m_pTrayIconMenu;
  ErrorCode=MoonAnimation_Uninitialize(&m_MoonTrayImages);
  MESSAGELOG_LogError(ErrorCode);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::CheckSavePreferences(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::CheckSavePreferences()");
  DEBUGLOG_LogIn();

  /* If the Apply button is enabled, there are changes to the preferences. */
  if ( (m_pButtonBox->button(QDialogButtonBox::Apply)->isEnabled()==true) &&
      (m_Settings.GetApplicationPreferencesPointer()->GetConfirmDiscardFlag()==true) )
  {
    if (QMessageBox::question(this,MOONPHASEQT_DISPLAYNAME,
        tr("Your preferences have changed. Do you want to save them?"),
        QMessageBox::Yes|QMessageBox::No,QMessageBox::Yes)==QMessageBox::Yes)
    {
      /* Fake an Apply button click. */
      ButtonBoxButtonClickedSlot(m_pButtonBox->button(QDialogButtonBox::Apply));
    }
  }

  /* Whether saved or discarded, reload them from the configuration file. */
  LoadSettings();
  PreferencesChangedSlot();   // Force a dialog update.

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::closeEvent(QCloseEvent *pEvent)
{
  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::closeEvent(%p)",pEvent);
  DEBUGLOG_LogIn();

  /* Check if preferences have been changed.
      If so, ask if user wants to save them, and if yes, save them. */
  CheckSavePreferences();

  /* DialogVisible && ReminderFlag && (!OnceFlag || (OnceFlag && !First)) */
  if ( (isVisible()==true) &&
      (m_Settings.GetApplicationPreferencesPointer()->GetStillRunningReminderFlag()==true) &&
      ((m_Settings.GetApplicationPreferencesPointer()->GetRemindOncePerSessionFlag()==false) ||
      ((m_Settings.GetApplicationPreferencesPointer()->GetRemindOncePerSessionFlag()==true) &&
      (m_CloseReminderIssued==false))) )
  {
    QMessageBox::information(this,MOONPHASEQT_DISPLAYNAME,
        tr("This program will continue to run in the system tray. To stop it, "
        "right click the system tray icon and select <b>Quit</b>."));
    if (m_Settings.GetApplicationPreferencesPointer()->GetRemindOncePerSessionFlag()==true)
      m_CloseReminderIssued=true;
  }
  SetVisible(false);
  pEvent->ignore();

  DEBUGLOG_LogOut();
  return;
}

QPixmap CONTROLPANELDIALOG_C::DrawFrame(MOONANIMATION_T *pMoonAnimation,
    unsigned int Percent,bool UseBackgroundColorFlag,QColor BackgroundColor)
{
  ERRORCODE_T ErrorCode;
  unsigned int Index;
  unsigned int FrameCount;
  QPixmap const *pConstPixmap;
  QPixmap Pixmap;


  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::GetFrame(%u)",Percent);
  DEBUGLOG_LogIn();

  /* Parameter checking. */
  if (pMoonAnimation==NULL)
  {
    ErrorCode=ERRORCODE_NULLPARAMETER;
    MESSAGELOG_LogError(ErrorCode);
  }
  else if (Percent>100)
  {
    ErrorCode=ERRORCODE_INVALIDPARAMETER;
    MESSAGELOG_LogError(ErrorCode);
  }
  else
  {
    ErrorCode=MoonAnimation_GetFrameCount(pMoonAnimation);
    MESSAGELOG_LogError(ErrorCode);
    if (ErrorCode>0)
    {
      /* ErrorCode has a frame count > 0. */
      FrameCount=ErrorCode;
      Index=(unsigned int)(((Percent/100.001)*FrameCount)+.5);
      if (Index>=FrameCount)
        Index-=FrameCount;
      ErrorCode=MoonAnimation_GetFrame(pMoonAnimation,Index,&pConstPixmap);
      MESSAGELOG_LogError(ErrorCode);
      if (ErrorCode>0)
        Pixmap=*pConstPixmap;
    }
  }
  if (ErrorCode<=0)
  {
    Pixmap.load(":/ErrorIcon");
    ErrorCode=ERRORCODE_SUCCESS;
  }
  if (UseBackgroundColorFlag==true)
  {
    QImage Background(Pixmap.width(),Pixmap.height(),QImage::Format_ARGB32);
    Background.fill(BackgroundColor);
    QPainter Painter(&Background);
    Painter.drawPixmap(0,0,Pixmap);
    Painter.end();
    Pixmap=QPixmap::fromImage(Background);
  }

  DEBUGLOG_LogOut();
  return(Pixmap);
}

void CONTROLPANELDIALOG_C::ForceUpdate(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::ForceUpdate()");
  DEBUGLOG_LogIn();

  m_UpdateIntervalCounter=UINT_MAX-1;
  UpdateTimerTriggeredSlot();

  DEBUGLOG_LogOut();
  return;
}

bool CONTROLPANELDIALOG_C::GetAllowMultipleInstancesFlag(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::GetAllowMultipleInstancesFlag()");
  DEBUGLOG_LogIn();

  DEBUGLOG_LogOut();
  return(m_Settings.GetApplicationPreferencesPointer()->GetAllowMultipleInstancesFlag());
}

void CONTROLPANELDIALOG_C::InitializeAboutTab(void)
{
  QString Information;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::InitializeAboutTab()");
  DEBUGLOG_LogIn();

  /* Set the data in the various widgets. */
  m_pAboutWidget->SetDisplayName(MOONPHASEQT_DISPLAYNAME);
  m_pAboutWidget->SetExecutableName(MOONPHASEQT_EXECUTABLENAME);
  m_pAboutWidget->SetVersion(MOONPHASEQT_VERSION);
  Information=MOONPHASEQT_DESCRIPTION;
  Information+="<br><br>" MOONPHASEQT_COPYRIGHTNOTICE;
#ifdef    _WIN32
  Information+="<br><br>"+QString(tr("Statically linked with Qt (Version "))
      +qVersion()+QString(tr(")\n"));
#endif    /* _WIN32 */

  /* Show moon phase web site. */
  Information+="<br><br><a href=\"" MOONPHASEQT_WEBSITE "\">"
      MOONPHASE_DISPLAYNAME " "+tr("Web Site")+QString("</a>\n");

  /* Show moon phase facebook page. */
  Information+="<br><br><a href=\"https://www.facebook.com/MoonPhaseApplication\">"
      MOONPHASE_DISPLAYNAME " "+tr("Facebook Page")+QString("</a>\n");
  m_pAboutWidget->SetInformationText(Information);

#if       defined(MOONPHASEQT_NOCHECKFORUPDATES)
  m_pAboutWidget->SetCheckForUpdateWidgetVisible(false);
#endif    /* defined(MOONPHASEQT_NOCHECKFORUPDATES) */

  m_pAboutWidget->SetLicenseText(f_pLicenseAgreement);

  m_pAboutWidget->SetChangeLogText(f_pChangeLog);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::LoadSettings(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::LoadSettings()");
  DEBUGLOG_LogIn();

  /* Load from the config file. */
  m_Settings.Load();

  /* Set the widget data. */
  m_pImperialUnitsRadioButton->setChecked(
      !m_Settings.GetInformationSettingsPointer()->GetInformationPreferencesPointer()->GetDefaultToMetricUnitsFlag());
  m_pLatitudeSpinBox->setValue(m_Settings.GetInformationSettingsPointer()->GetInformationPreferencesPointer()->GetLatitude());
  m_pLongitudeSpinBox->setValue(m_Settings.GetInformationSettingsPointer()->GetInformationPreferencesPointer()->GetLongitude());
  m_pMetricUnitsRadioButton->setChecked(
      m_Settings.GetInformationSettingsPointer()->GetInformationPreferencesPointer()->GetDefaultToMetricUnitsFlag());

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::MoveItem(int Direction)
{
  QListWidgetItem *pSelectedItem;
  int Row;
  QListWidgetItem *pItem;


  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::MoveItem(%d)",Direction);
  DEBUGLOG_LogIn();

  /* Check the direction. */
  if ( (Direction!=1) && (Direction!=-1) )
  {
    MESSAGELOG_Error("Invalid parameter.");
  }
  else
  {
    /* Get the current item. */
    pSelectedItem=m_pDisplayListWidget->selectedItems()[0];
    if (pSelectedItem==NULL)
    {
      MESSAGELOG_Error("NULL selected item pointer.");
    }
    else
    {
      /* Extract it, then re-insert it using the delta. */
      Row=m_pDisplayListWidget->row(pSelectedItem);
      pItem=m_pDisplayListWidget->takeItem(Row);
      if (pItem==NULL)
      {
        MESSAGELOG_Error("NULL item pointer.");
      }
      else
      {
        m_pDisplayListWidget->insertItem(Row+Direction,pItem);
        m_pDisplayListWidget->setCurrentItem(pItem);
      }

      /* Update the display. */
      InformationPanelTimerTriggeredSlot();
      UpdateControls();
    }
  }

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::ReadPreferences(SETTINGS_C *pSettings)
{
  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::ReadPreferences(%p)",pSettings);
  DEBUGLOG_LogIn();

  /* Get the widget data. */
  *pSettings->GetApplicationPreferencesPointer()=m_pPreferencesWidget->GetPreferences();
  pSettings->GetInformationSettingsPointer()->GetInformationPreferencesPointer()->SetDefaultToMetricUnitsFlag(
      m_pMetricUnitsRadioButton->isChecked());
  pSettings->GetInformationSettingsPointer()->GetInformationPreferencesPointer()->SetLatitude(m_pLatitudeSpinBox->value());
  pSettings->GetInformationSettingsPointer()->GetInformationPreferencesPointer()->SetLongitude(m_pLongitudeSpinBox->value());

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::RecalculateMoonData(time_t Time)
{
  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::RecalculateMoonData(%1)",time);
  DEBUGLOG_LogIn();

  /* Recalculate the astronomical data. */
  MoonData_Recalculate(&m_MoonData,Time);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::SaveSettings(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::SaveSettings()");
  DEBUGLOG_LogIn();

  /* Read the widgets. */
  ReadPreferences(&m_Settings);

  /* Save to the config file. */
  m_Settings.GetApplicationPreferencesPointer()->Save();
  m_Settings.GetInformationSettingsPointer()->GetInformationPreferencesPointer()->Save();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::SetVisible(bool VisibleFlag)
{
  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::SetVisible(%u)",VisibleFlag);
  DEBUGLOG_LogIn();

  QDialog::setVisible(VisibleFlag);
  if ( (VisibleFlag==true) && (isVisible()==true) )
  {
    /* Window already visible, bring to front. */
    raise();
    activateWindow();
  }

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::UpdateControls(void)
{
  bool EnableAddFlag;
  bool EnableRemoveFlag;
  bool EnableUpFlag;
  bool EnableDownFlag;
  bool EnableOptionsFlag;
  QListWidgetItem *pLWItem;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::UpdateControls()");
  DEBUGLOG_LogIn();

  EnableAddFlag=false;
  EnableRemoveFlag=false;
  EnableUpFlag=false;
  EnableDownFlag=false;
  EnableOptionsFlag=false;

  if (m_pDataListWidget->selectedItems().count()!=0)
    EnableAddFlag=true;

  if (m_pDisplayListWidget->selectedItems().count()!=0)
    EnableRemoveFlag=true;

  if (m_pDisplayListWidget->selectedItems().count()!=0)
  {
    pLWItem=m_pDisplayListWidget->selectedItems()[0];
    if (pLWItem==NULL)
    {
      MESSAGELOG_Error("NULL selected item pointer.");
    }
    else
    {
      if (m_pDisplayListWidget->row(pLWItem)!=0)
        EnableUpFlag=true;

      if ((m_pDisplayListWidget->row(pLWItem)+1)!=
            m_pDisplayListWidget->count())
        EnableDownFlag=true;

      if (m_pDisplayListWidget->selectedItems().count()==1)
        EnableOptionsFlag=true;
    }
  }

  m_pAddDisplayItemToolButton->setEnabled(EnableAddFlag);
  m_pRemoveDisplayItemToolButton->setEnabled(EnableRemoveFlag);
  m_pMoveUpDisplayItemToolButton->setEnabled(EnableUpFlag);
  m_pMoveDownDisplayItemToolButton->setEnabled(EnableDownFlag);
  m_pOptionsDisplayItemToolButton->setEnabled(EnableOptionsFlag);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::AddDataItemButtonClickedSlot(void)
{
  QListWidgetItem *pSelectedItem;
  int InfoItemIndex;
  char const *pLabel;
  QListWidgetItem *pLWItem;
  INFORMATIONPANELITEM_C *pIItem;
  int Index;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::AddDataItemButtonClickedSlot()");
  DEBUGLOG_LogIn();

  foreach(pSelectedItem,m_pDataListWidget->selectedItems())
  {
    /* Add a row at the end of the display list widget. */
    InfoItemIndex=m_pDataListWidget->row(pSelectedItem);
    /*pLabel=*/Information_GetLabel(InfoItemIndex,&pLabel);
    if (pLabel==NULL)
    {
      MESSAGELOG_Error("Invalid item label index.");
    }
    else
      m_pDisplayListWidget->addItem(pLabel);

    /* Get the new list widget item. */
    pLWItem=m_pDisplayListWidget->item(m_pDisplayListWidget->count()-1);
    if (pLWItem==NULL)
    {
      MESSAGELOG_Error("NULL list widget item pointer.");
    }
    else
    {
      /* Create a new information item. */
      pIItem=new INFORMATIONPANELITEM_C;

      /* Save the information item in the list widget item data. */
      pLWItem->setData(ROLE_INFORMATIONTYPE,QVariant::fromValue(pIItem));

      /* The information item index is the same as the row index. */
      pIItem->SetInformationIndex(InfoItemIndex);

      /* Get the default unit/format index. */
      Information_GetDefaultUnitFormatIndex(
          InfoItemIndex,m_pImperialUnitsRadioButton->isChecked()==true,&Index);
      pIItem->SetUnitFormatIndex(Index);
    }
  }

  /* Update the line count in the information panel. */
  m_pInformationPanelDialog->SetLineCount(m_pDisplayListWidget->count());

  /* Update the display. */
  InformationPanelTimerTriggeredSlot();
  UpdateControls();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::ButtonBoxButtonClickedSlot(QAbstractButton *pButton)
{
  SETTINGS_C Settings;
  bool ForceUpdateFlag;


  DEBUGLOG_Printf1(
      "CONTROLPANELDIALOG_C::ButtonBoxButtonClickedSlot(%p)",pButton);
  DEBUGLOG_LogIn();

  switch(m_pButtonBox->standardButton(pButton))
  {
    case QDialogButtonBox::Apply:
      /* Force an update if the file name changed, the use opaque background
          flag changed, or the background color changed. */
      ForceUpdateFlag=false;
      ReadPreferences(&Settings);
      if ( (Settings.GetApplicationPreferencesPointer()->GetAnimationPathname()!=
          m_Settings.GetApplicationPreferencesPointer()->GetAnimationPathname()) ||
          (Settings.GetApplicationPreferencesPointer()->GetUseOpaqueBackgroundFlag()!=
          m_Settings.GetApplicationPreferencesPointer()->GetUseOpaqueBackgroundFlag()) ||
          (Settings.GetApplicationPreferencesPointer()->GetBackgroundColor()!=
          m_Settings.GetApplicationPreferencesPointer()->GetBackgroundColor()) ||
          (Settings.GetApplicationPreferencesPointer()->GetUpdateInterval()!=
          m_Settings.GetApplicationPreferencesPointer()->GetUpdateInterval()) )
      {
        if (Settings.GetApplicationPreferencesPointer()->GetAnimationPathname()!=
            m_Settings.GetApplicationPreferencesPointer()->GetAnimationPathname())
        {
          /* Ignore return. If any error, the user has already been informed. */
          MoonAnimation_ReadFile(
            &m_MoonTrayImages,qPrintable(Settings.GetApplicationPreferencesPointer()->GetAnimationPathname()));
        }
        ForceUpdateFlag=true;
      }
      if (Settings.GetApplicationPreferencesPointer()->GetRemindOncePerSessionFlag()==false)
        m_CloseReminderIssued=false;
      SaveSettings();
      PreferencesChangedSlot();     // Force a dialog update.
      if (ForceUpdateFlag==true)
        ForceUpdate();
      break;

    case QDialogButtonBox::Close:
      close();
      break;

    default:
      MESSAGELOG_Error("Invalid button.");
      break;
  }

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::CheckButtonClickedSlot(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::CheckButtonClickedSlot()");
  DEBUGLOG_LogIn();

  m_pAboutWidget->SetUpdateText(tr("Checking for an update!"));
  m_UpdateNotifier.CheckForUpdate();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::
    ControlPanelActivatedSlot(QSystemTrayIcon::ActivationReason Reason)
{
  DEBUGLOG_Printf1(
      "CONTROLPANELDIALOG_C::ControlPanelActivatedSlot(%d)",Reason);
  DEBUGLOG_LogIn();

  switch(Reason)
  {
    case QSystemTrayIcon::DoubleClick:
#if       !defined(__APPLE__)
      m_pDoubleClickTimeoutTimer->stop();
#endif    /* !defined(__APPLE__) */
      SetVisible(true);
      break;

#if       !defined(__APPLE__)
    case QSystemTrayIcon::Trigger:
      if (m_pDoubleClickTimeoutTimer->isActive()==true)
        return;
      m_pDoubleClickTimeoutTimer->setInterval(
          QApplication::doubleClickInterval());
      m_pDoubleClickTimeoutTimer->start();
      break;
#endif    /* !defined(__APPLE__) */

    default:
      break;
  }

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::DataItemSelectionChangedSlot(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::DataItemSelectionChangedSlot()");
  DEBUGLOG_LogIn();

  /* Allow only one list widget to have a selected item. */
  if (m_pDataListWidget->selectedItems().count()!=0)
    m_pDisplayListWidget->clearSelection();

  UpdateControls();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::DisplayItemSelectionChangedSlot(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::DisplayItemSelectionChangedSlot()");
  DEBUGLOG_LogIn();

  /* Allow only one list widget to have a selected item. */
  if (m_pDisplayListWidget->selectedItems().count()!=0)
    m_pDataListWidget->clearSelection();

  UpdateControls();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::DoubleClickTimeoutTimerTriggered(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::DoubleClickTimeoutTimerTriggered()");
  DEBUGLOG_LogIn();

  ShowInformationPanelSlot();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::InformationPanelTimerTriggeredSlot(void)
{
  QListWidgetItem *pLWItem;
  QString Text;
  INFORMATIONPANELITEM_C *pIItem;


  DEBUGLOG_Printf0(
        "CONTROLPANELDIALOG_C::InformationPanelTimerTriggeredSlot()");
  DEBUGLOG_LogIn();

  /* Is the information panel visible? */
  if (m_pInformationPanelDialog->isVisible()==false)
  {
    /* No, stop the timer (no need to update the information panel). */
    m_pInformationPanelTimer->stop();
  }
  else
  {
    /* Need to update the information panel. */
    RecalculateMoonData(time(NULL));

    /* Update each line. */
    for(int Index=0;Index<m_pDisplayListWidget->count();Index++)
    {
      /* Get the list widget item. */
      pLWItem=m_pDisplayListWidget->item(Index);
      if (pLWItem==NULL)
      {
        MESSAGELOG_Error("NULL list widget item pointer.");
      }
      else
      {
        /* Get the information item pointer. */
        pIItem=LWI2IPI(pLWItem);
        if (pIItem==NULL)
        {
          MESSAGELOG_Error("NULL information item pointer.");
        }
        else
        {
          INFORMATIONPANELITEMOPTIONS_C Options;
          DATETIMEOPTIONS_T DTOptions;
          MOONDATAPRINTOPTIONS_T MDPOptions;
          char *pPtr;

          Options=pIItem->GetOptions();
          Options.Convert(DTOptions);
          memcpy(&MDPOptions,&DTOptions,sizeof(MDPOptions));
          Information_Print(&m_MoonData,pIItem->GetInformationIndex(),
              pIItem->GetUnitFormatIndex(),&MDPOptions,&pPtr);

          m_pInformationPanelDialog->SetLine(Index,QString::fromUtf8(pPtr),Options);
          if (pPtr!=NULL)
            free(pPtr);
        }
      }
    }
  }

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::InstanceMessageSlot(QString const &Message)
{
  DEBUGLOG_Printf2("CONTROLPANELDIALOG_C::InstanceMessageSlot(%p(%s))",
      &Message,qPrintable(Message));
  DEBUGLOG_LogIn();

  if (QString::localeAwareCompare(Message,"Activate")==0)
    ControlPanelActivatedSlot(QSystemTrayIcon::DoubleClick);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::LatitudeChangedSlot(double Latitude)
{
  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::LatitudeChangedSlot(%f)",Latitude);
  DEBUGLOG_LogIn();

  m_MoonData.CTransData.Glat=Latitude;
  MoonData_Recalculate(&m_MoonData,time(NULL));
  PreferencesChangedSlot();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::LongitudeChangedSlot(double Longitude)
{
  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::LongitudeChangedSlot(%f)",Longitude);
  DEBUGLOG_LogIn();

  m_MoonData.CTransData.Glon=-Longitude;
  MoonData_Recalculate(&m_MoonData,time(NULL));
  PreferencesChangedSlot();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::MoveDownDisplayItemButtonClickedSlot(void)
{
  DEBUGLOG_Printf0(
      "CONTROLPANELDIALOG_C::MoveDownDisplayItemButtonClickedSlot()");
  DEBUGLOG_LogIn();

  MoveItem(1);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::MoveUpDisplayItemButtonClickedSlot(void)
{
  DEBUGLOG_Printf0(
      "CONTROLPANELDIALOG_C::MoveUpDisplayItemButtonClickedSlot()");
  DEBUGLOG_LogIn();

  MoveItem(-1);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::OptionsAppliedSlot(INFORMATIONPANELITEMOPTIONS_C const &Options)
{
  QListWidgetItem *pLWItem;
  INFORMATIONPANELITEM_C *pIItem;


  DEBUGLOG_Printf1("CONTROLPANELDIALOG_C::OptionsAppliedSlot(%p)",&Options);
  DEBUGLOG_LogIn();

  pLWItem=m_pDisplayListWidget->selectedItems()[0];
  if (pLWItem==NULL)
  {
    MESSAGELOG_Error("NULL list widget item pointer");
  }
  else
  {
    pIItem=LWI2IPI(pLWItem);
    if (pIItem==NULL)
    {
      MESSAGELOG_Error("NULL list widget item pointer");
    }
    else
    {
      /* Copy the options. */
      pIItem->SetOptions(Options);
    }
  }

  /* Update the display. */
  InformationPanelTimerTriggeredSlot();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::OptionsDisplayItemButtonClickedSlot(void)
{
  QListWidgetItem *pLWItem;
  INFORMATIONPANELITEM_C *pIItem;
  int Index;
  char *pPtr;
  QList<QString> List;
  INFORMATIONOPTIONSDIALOG_C::DIALOGDATA_T DialogData;
  INFORMATIONPANELITEMOPTIONS_C Options;
  INFORMATIONOPTIONSDIALOG_C OptionsDialog(this);


  DEBUGLOG_Printf0(
      "CONTROLPANELDIALOG_C::OptionsDisplayItemButtonClickedSlot()");
  DEBUGLOG_LogIn();

  pLWItem=m_pDisplayListWidget->selectedItems()[0];
  if (pLWItem==NULL)
  {
    MESSAGELOG_Error("NULL list widget item pointer");
  }
  else
  {
    pIItem=LWI2IPI(pLWItem);
    if (pIItem==NULL)
    {
      MESSAGELOG_Error("NULL list widget item pointer");
    }
    else
    {
      Index=0;
      goto Start0;
      while(pPtr!=NULL)
      {
        List.append(QString::fromUtf8(pPtr));
        free(pPtr);
        Index++;
Start0:
        Information_GetUnitFormatDescription(
            pIItem->GetInformationIndex(),Index,&pPtr);
      }

      DialogData.DescriptionList=List;

      List.clear();
      Index=0;
      goto Start1;
      while(pPtr!=NULL)
      {
        List.append(QString::fromUtf8(pPtr));
        free(pPtr);
        Index++;
Start1:
        Information_GetFormat(pIItem->GetInformationIndex(),Index,&pPtr);
      }

      DialogData.FormatsList=List;

      /* Set the mode. */
      EDITMODE_T EditMode;
      Information_GetEditMode(pIItem->GetInformationIndex(),&EditMode);
      DialogData.EditMode=EditMode;

      /* Set the options in the dialog box. */
      Options=pIItem->GetOptions();
      DialogData.Options=Options;

      /* Set the dialog data. */
      OptionsDialog.SetData(DialogData);

      /* Catch the signal from the Apply button. */
      connect(&OptionsDialog,SIGNAL(OptionsAppliedSignal(INFORMATIONPANELITEMOPTIONS_C const &)),
          this,SLOT(OptionsAppliedSlot(INFORMATIONPANELITEMOPTIONS_C const &)));

      /* Show the dialog box. */
      if (OptionsDialog.exec()==QDialog::Accepted)
      {
        /* Accepted. */

        /* Copy the options. */
        OptionsDialog.GetOptions(Options);
        pIItem->SetOptions(Options);

        /* Update the display. */
        InformationPanelTimerTriggeredSlot();
      }
    }
  }

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::PreferencesChangedSlot(void)
{
  SETTINGS_C Settings;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::PreferencesChangedSlot()");
  DEBUGLOG_LogIn();

  /* Get widget preferences. */
  ReadPreferences(&Settings);

  /* Update Apply button. */
  m_pButtonBox->button(QDialogButtonBox::Apply)->
      setEnabled((m_pPreferencesWidget->GetPreferences()!=*m_Settings.GetApplicationPreferencesPointer()) ||
      (*Settings.GetInformationSettingsPointer()->GetInformationPreferencesPointer()!=*m_Settings.GetInformationSettingsPointer()->GetInformationPreferencesPointer()));

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::QuitSlot(void)
{
  bool QuitFlag;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::QuitSlot()");
  DEBUGLOG_LogIn();

  /* Check if preferences have been changed.
      If so, ask if user wants to save them, and if yes, save them. */
  CheckSavePreferences();

  /* Need to confirm quit? */
  if (m_Settings.GetApplicationPreferencesPointer()->GetConfirmQuitFlag()==true)
  {
    /* Yes. */
    QuitFlag=false;
    if (QMessageBox::question(this,MOONPHASEQT_DISPLAYNAME,
        tr("Are you sure you want to quit?"),
        QMessageBox::Yes|QMessageBox::No,QMessageBox::No)==QMessageBox::Yes)
      QuitFlag=true;
  }
  else
  {
    /* No. */
    QuitFlag=true;
  }
  if (QuitFlag==true)
    qApp->quit();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::RemoveDisplayItemButtonClickedSlot(void)
{
  QListWidgetItem *pSelectedItem;
  QListWidgetItem *pItem;
  int Row;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::RemoveDisplayItemButtonClickedSlot()");
  DEBUGLOG_LogIn();

  foreach(pSelectedItem,m_pDisplayListWidget->selectedItems())
  {
    Row=m_pDisplayListWidget->row(pSelectedItem);
    pItem=m_pDisplayListWidget->takeItem(Row);
    if (pItem==NULL)
    {
      MESSAGELOG_Error("NULL item pointer.");
    }
    delete pItem;
  }
  m_pInformationPanelDialog->SetLineCount(m_pDisplayListWidget->count());

  /* Update the display. */
  InformationPanelTimerTriggeredSlot();
  UpdateControls();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::ShowControlPanelSlot(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::ShowControlPanelSlot()");
  DEBUGLOG_LogIn();

  ControlPanelActivatedSlot(QSystemTrayIcon::DoubleClick);

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::ShowInformationPanelSlot(void)
{
  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::ShowInformationPanelSlot()");
  DEBUGLOG_LogIn();

  if (m_pInformationPanelDialog->isVisible()==false)
    m_pInformationPanelTimer->start(INFORMATIONPANELTIMER_RATE);
  m_pInformationPanelDialog->show();
  m_pInformationPanelDialog->raise();
  m_pInformationPanelDialog->activateWindow();

  /* Update the display. */
  InformationPanelTimerTriggeredSlot();

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::UpdateTimerTriggeredSlot(void)
{
  QPixmap Pixmap;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::UpdateTimerTriggeredSlot()");
  DEBUGLOG_LogIn();

  /* Check weekly for a program update. */
  if (QDate::currentDate()>=m_NextUpdateCheck)
  {
#ifdef    DEBUG
    {
      time_t TimeSeconds1970;
      struct tm *pTimeInfo;
#define   BUFFER_SIZE   (1024)
      char pPtr[BUFFER_SIZE];


      time(&TimeSeconds1970);
      pTimeInfo=localtime(&TimeSeconds1970);
      strftime(pPtr,BUFFER_SIZE,"Check for update: %H:%M:%S",pTimeInfo);
#undef    BUFFER_SIZE
      MESSAGELOG_Info(pPtr);
    }
#endif    /* DEBUG */

#ifndef   DEBUG
    /* Force a check for an update. */
    CheckButtonClickedSlot();
#endif    /* DEBUG */
    m_NextUpdateCheck=m_NextUpdateCheck.addDays(7);
  }

  /* Check if calculations need to be run, and (possibly) the icon updated. */
  m_UpdateIntervalCounter++;
  if (m_UpdateIntervalCounter>=(m_Settings.GetApplicationPreferencesPointer()->GetUpdateInterval()))
  {
#ifdef    DEBUG
    {
      time_t TimeSeconds1970;
      struct tm *pTimeInfo;
#define   BUFFER_SIZE   (1024)
      char pPtr[BUFFER_SIZE];


      time(&TimeSeconds1970);
      pTimeInfo=localtime(&TimeSeconds1970);
      strftime(pPtr,BUFFER_SIZE,"Update tray: %H:%M:%S",pTimeInfo);
#undef    BUFFER_SIZE
      MESSAGELOG_Info(pPtr);
    }
#endif    /* DEBUG */

    /* Recalculate the astronomical data. */
    RecalculateMoonData(time(NULL));

    /* Update the tray icon. */
    UpdateTrayIcon();

    /* Reset the counter. */
    m_UpdateIntervalCounter=0;
  }

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::UpdateTrayIcon(void)
{
  QString PercentString;
  QPixmap Pixmap;
  float Percent;


  DEBUGLOG_Printf0("CONTROLPANELDIALOG_C::UpdateTrayIcon()");
  DEBUGLOG_LogIn();

  /* Update the icon. */
  Percent=MoonData_GetMoonPhasePercent(&m_MoonData);
  Pixmap=DrawFrame(&m_MoonTrayImages,(unsigned int)Percent,
      m_Settings.GetApplicationPreferencesPointer()->GetUseOpaqueBackgroundFlag(),
      m_Settings.GetApplicationPreferencesPointer()->GetBackgroundColor());
  m_pTrayIcon->setIcon(Pixmap);

  /* Update the tool tip */
  m_pTrayIcon->setToolTip(
      MOONPHASEQT_DISPLAYNAME+QString(" - %1%").arg((int)(Percent+0.5)));

  DEBUGLOG_LogOut();
  return;
}

void CONTROLPANELDIALOG_C::VersionSlot(QString Version)
{
  VERSIONSTRING_C Current(MOONPHASEQT_VERSION);
  VERSIONSTRING_C Internet;//(qPrintable(Version));


  DEBUGLOG_Printf2(
      "CONTROLPANELDIALOG_C::VersionSlot(%p(%s))",&Version,qPrintable(Version));
  DEBUGLOG_LogIn();

  if (Version.isEmpty()==true)
    m_pAboutWidget->SetUpdateText(tr("Error accessing the internet!"));
  else
  {
    Internet.Set(qPrintable(Version));
    if (Current>Internet)
      m_pAboutWidget->SetUpdateText(tr("You have a pre-release version!"));
    else if (Current==Internet)
      m_pAboutWidget->SetUpdateText(tr("This program is up to date!"));
    else
    {
      m_pAboutWidget->SetUpdateText(tr("An update is available!"));

      if (m_FirstUpdateFlag==true)
      {
        QMessageBox UpdateMessageBox(this);


        UpdateMessageBox.setWindowTitle(MOONPHASEQT_DISPLAYNAME);
        UpdateMessageBox.setTextFormat(Qt::RichText); // Makes links clickable
        UpdateMessageBox.setText(tr("An update to this program is available!"
            "<br><br>Visit the <a href='" MOONPHASEQT_WEBSITE "'>"
            MOONPHASEQT_DISPLAYNAME " Web Site</a>"));
        UpdateMessageBox.exec();
      }
    }
    m_FirstUpdateFlag=false;
  }

  DEBUGLOG_LogOut();
  return;
}


#undef    CONTROLPANELDIALOG_CPP


/**
*** controlpaneldialog.cpp
**/
