/*
 * Copyright (C) 2023 Linux Studio Plugins Project <https://lsp-plug.in/>
 *           (C) 2023 Vladimir Sadovnikov <sadko4u@gmail.com>
 *
 * This file is part of lsp-dsp-lib
 * Created on: 22 апр. 2023 г.
 *
 * lsp-dsp-lib is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * lsp-dsp-lib is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with lsp-dsp-lib. If not, see <https://www.gnu.org/licenses/>.
 */

#include <lsp-plug.in/common/finally.h>
#include <lsp-plug.in/dsp/dsp.h>
#include <lsp-plug.in/test-fw/ByteBuffer.h>
#include <lsp-plug.in/test-fw/utest.h>

#include <private/utest/dsp/bitmap/util.h>

namespace lsp
{
    namespace generic
    {
        void bitmap_put_b2b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_add_b2b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_sub_b2b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_max_b2b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
        void bitmap_min_b2b8(dsp::bitmap_t *dst, const dsp::bitmap_t *src, ssize_t x, ssize_t y);
    }
} /* namespace lsp */

namespace lsp
{
    namespace test
    {
        static uint8_t bitmap_data[] =
        {
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x0f, 0x40, 0x00, 0x01, 0xbf, 0xe4, 0x0f, 0xff, 0xe0, 0x00,
            0x0f, 0x40, 0x00, 0x0b, 0xff, 0xfc, 0x0f, 0xff, 0xfd, 0x00,
            0x0f, 0x40, 0x00, 0x1f, 0x40, 0x14, 0x0f, 0x40, 0x2f, 0x00,
            0x0f, 0x40, 0x00, 0x2e, 0x00, 0x00, 0x0f, 0x40, 0x0f, 0x40,
            0x0f, 0x40, 0x00, 0x2e, 0x00, 0x00, 0x0f, 0x40, 0x0b, 0x80,
            0x0f, 0x40, 0x00, 0x2f, 0x00, 0x00, 0x0f, 0x40, 0x0b, 0x40,
            0x0f, 0x40, 0x00, 0x0f, 0x80, 0x00, 0x0f, 0x40, 0x0f, 0x40,
            0x0f, 0x40, 0x00, 0x07, 0xf8, 0x00, 0x0f, 0x40, 0x2f, 0x00,
            0x0f, 0x40, 0x00, 0x00, 0xbf, 0xc0, 0x0f, 0xff, 0xfc, 0x00,
            0x0f, 0x40, 0x00, 0x00, 0x0b, 0xf4, 0x0f, 0xff, 0x90, 0x00,
            0x0f, 0x40, 0x00, 0x00, 0x00, 0xfc, 0x0f, 0x40, 0x00, 0x00,
            0x0f, 0x40, 0x00, 0x00, 0x00, 0x3d, 0x0f, 0x40, 0x00, 0x00,
            0x0f, 0x40, 0x00, 0x00, 0x00, 0x3d, 0x0f, 0x40, 0x00, 0x00,
            0x0f, 0x40, 0x00, 0x00, 0x00, 0x3c, 0x0f, 0x40, 0x00, 0x00,
            0x0f, 0x40, 0x00, 0x10, 0x00, 0xbc, 0x0f, 0x40, 0x00, 0x00,
            0x0f, 0xff, 0xfd, 0x3f, 0xff, 0xf0, 0x0f, 0x40, 0x00, 0x00,
            0x0f, 0xff, 0xfd, 0x1b, 0xff, 0x40, 0x0f, 0x40, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        };

        static const lsp::dsp::bitmap_t bitmap =
        {
            38, // width
            33, // height
            10, // stride
            bitmap_data // data
        };

        typedef void (* bitmap_func_b1b8_t)(lsp::dsp::bitmap_t *dst, const lsp::dsp::bitmap_t *src, ssize_t x, ssize_t y);

        static const bitmap_pattern_t put_patterns[] =
        {
            {
                0, 0,
                0x88,
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000ffff550000000000000000000055aaffffffffaa55000000ffffffffffffffaa00000000\n"
                "0000ffff55000000000000000000aaffffffffffffffff000000ffffffffffffffffff550000\n"
                "0000ffff55000000000000000055ffff55000000005555000000ffff5500000000aaffff0000\n"
                "0000ffff550000000000000000aaffaa00000000000000000000ffff550000000000ffff5500\n"
                "0000ffff550000000000000000aaffaa00000000000000000000ffff550000000000aaffaa00\n"
                "0000ffff550000000000000000aaffff00000000000000000000ffff550000000000aaff5500\n"
                "0000ffff55000000000000000000ffffaa000000000000000000ffff550000000000ffff5500\n"
                "0000ffff5500000000000000000055ffffffaa00000000000000ffff5500000000aaffff0000\n"
                "0000ffff550000000000000000000000aaffffffff0000000000ffffffffffffffffff000000\n"
                "0000ffff5500000000000000000000000000aaffffff55000000ffffffffffffaa5500000000\n"
                "0000ffff55000000000000000000000000000000ffffff000000ffff55000000000000000000\n"
                "0000ffff5500000000000000000000000000000000ffff550000ffff55000000000000000000\n"
                "0000ffff5500000000000000000000000000000000ffff550000ffff55000000000000000000\n"
                "0000ffff5500000000000000000000000000000000ffff000000ffff55000000000000000000\n"
                "0000ffff55000000000000000055000000000000aaffff000000ffff55000000000000000000\n"
                "0000ffffffffffffffffff5500ffffffffffffffffff00000000ffff55000000000000000000\n"
                "0000ffffffffffffffffff550055aaffffffffff550000000000ffff55000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
            },
            {
                -8, -8,
                0x88,
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000055aaffffffffaa55000000ffffffffffffffaa000000008888888888888888\n"
                "000000000000aaffffffffffffffff000000ffffffffffffffffff5500008888888888888888\n"
                "000000000055ffff55000000005555000000ffff5500000000aaffff00008888888888888888\n"
                "0000000000aaffaa00000000000000000000ffff550000000000ffff55008888888888888888\n"
                "0000000000aaffaa00000000000000000000ffff550000000000aaffaa008888888888888888\n"
                "0000000000aaffff00000000000000000000ffff550000000000aaff55008888888888888888\n"
                "000000000000ffffaa000000000000000000ffff550000000000ffff55008888888888888888\n"
                "00000000000055ffffffaa00000000000000ffff5500000000aaffff00008888888888888888\n"
                "0000000000000000aaffffffff0000000000ffffffffffffffffff0000008888888888888888\n"
                "00000000000000000000aaffffff55000000ffffffffffffaa55000000008888888888888888\n"
                "000000000000000000000000ffffff000000ffff550000000000000000008888888888888888\n"
                "00000000000000000000000000ffff550000ffff550000000000000000008888888888888888\n"
                "00000000000000000000000000ffff550000ffff550000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff550000000000000000008888888888888888\n"
                "000000000055000000000000aaffff000000ffff550000000000000000008888888888888888\n"
                "ffffff5500ffffffffffffffffff00000000ffff550000000000000000008888888888888888\n"
                "ffffff550055aaffffffffff550000000000ffff550000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, -8,
                0x88,
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "88888888888888880000ffff550000000000000000000055aaffffffffaa55000000ffffffff\n"
                "88888888888888880000ffff55000000000000000000aaffffffffffffffff000000ffffffff\n"
                "88888888888888880000ffff55000000000000000055ffff55000000005555000000ffff5500\n"
                "88888888888888880000ffff550000000000000000aaffaa00000000000000000000ffff5500\n"
                "88888888888888880000ffff550000000000000000aaffaa00000000000000000000ffff5500\n"
                "88888888888888880000ffff550000000000000000aaffff00000000000000000000ffff5500\n"
                "88888888888888880000ffff55000000000000000000ffffaa000000000000000000ffff5500\n"
                "88888888888888880000ffff5500000000000000000055ffffffaa00000000000000ffff5500\n"
                "88888888888888880000ffff550000000000000000000000aaffffffff0000000000ffffffff\n"
                "88888888888888880000ffff5500000000000000000000000000aaffffff55000000ffffffff\n"
                "88888888888888880000ffff55000000000000000000000000000000ffffff000000ffff5500\n"
                "88888888888888880000ffff5500000000000000000000000000000000ffff550000ffff5500\n"
                "88888888888888880000ffff5500000000000000000000000000000000ffff550000ffff5500\n"
                "88888888888888880000ffff5500000000000000000000000000000000ffff000000ffff5500\n"
                "88888888888888880000ffff55000000000000000055000000000000aaffff000000ffff5500\n"
                "88888888888888880000ffffffffffffffffff5500ffffffffffffffffff00000000ffff5500\n"
                "88888888888888880000ffffffffffffffffff550055aaffffffffff550000000000ffff5500\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "8888888888888888000000000000000000000000000000000000000000000000000000000000\n"
                "88888888888888880000ffff550000000000000000000055aaffffffffaa55000000ffffffff\n"
                "88888888888888880000ffff55000000000000000000aaffffffffffffffff000000ffffffff\n"
                "88888888888888880000ffff55000000000000000055ffff55000000005555000000ffff5500\n"
                "88888888888888880000ffff550000000000000000aaffaa00000000000000000000ffff5500\n"
                "88888888888888880000ffff550000000000000000aaffaa00000000000000000000ffff5500\n"
                "88888888888888880000ffff550000000000000000aaffff00000000000000000000ffff5500\n"
                "88888888888888880000ffff55000000000000000000ffffaa000000000000000000ffff5500\n"
                "88888888888888880000ffff5500000000000000000055ffffffaa00000000000000ffff5500\n"
                "88888888888888880000ffff550000000000000000000000aaffffffff0000000000ffffffff\n"
                "88888888888888880000ffff5500000000000000000000000000aaffffff55000000ffffffff\n"
                "88888888888888880000ffff55000000000000000000000000000000ffffff000000ffff5500\n"
                "88888888888888880000ffff5500000000000000000000000000000000ffff550000ffff5500\n"
                "88888888888888880000ffff5500000000000000000000000000000000ffff550000ffff5500\n"
                "88888888888888880000ffff5500000000000000000000000000000000ffff000000ffff5500\n"
                "88888888888888880000ffff55000000000000000055000000000000aaffff000000ffff5500\n"
                "88888888888888880000ffffffffffffffffff5500ffffffffffffffffff00000000ffff5500\n"
            },
            {
                -8, 8,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000000000000000000000000000000000000000000000000008888888888888888\n"
                "0000000000000055aaffffffffaa55000000ffffffffffffffaa000000008888888888888888\n"
                "000000000000aaffffffffffffffff000000ffffffffffffffffff5500008888888888888888\n"
                "000000000055ffff55000000005555000000ffff5500000000aaffff00008888888888888888\n"
                "0000000000aaffaa00000000000000000000ffff550000000000ffff55008888888888888888\n"
                "0000000000aaffaa00000000000000000000ffff550000000000aaffaa008888888888888888\n"
                "0000000000aaffff00000000000000000000ffff550000000000aaff55008888888888888888\n"
                "000000000000ffffaa000000000000000000ffff550000000000ffff55008888888888888888\n"
                "00000000000055ffffffaa00000000000000ffff5500000000aaffff00008888888888888888\n"
                "0000000000000000aaffffffff0000000000ffffffffffffffffff0000008888888888888888\n"
                "00000000000000000000aaffffff55000000ffffffffffffaa55000000008888888888888888\n"
                "000000000000000000000000ffffff000000ffff550000000000000000008888888888888888\n"
                "00000000000000000000000000ffff550000ffff550000000000000000008888888888888888\n"
                "00000000000000000000000000ffff550000ffff550000000000000000008888888888888888\n"
                "00000000000000000000000000ffff000000ffff550000000000000000008888888888888888\n"
                "000000000055000000000000aaffff000000ffff550000000000000000008888888888888888\n"
                "ffffff5500ffffffffffffffffff00000000ffff550000000000000000008888888888888888\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t add_patterns[] =
        {
            {
                0, 0,
                0x11,
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111ffff661111111111111111111166bbffffffffbb66111111ffffffffffffffbb11111111\n"
                "1111ffff66111111111111111111bbffffffffffffffff111111ffffffffffffffffff661111\n"
                "1111ffff66111111111111111166ffff66111111116666111111ffff6611111111bbffff1111\n"
                "1111ffff661111111111111111bbffbb11111111111111111111ffff661111111111ffff6611\n"
                "1111ffff661111111111111111bbffbb11111111111111111111ffff661111111111bbffbb11\n"
                "1111ffff661111111111111111bbffff11111111111111111111ffff661111111111bbff6611\n"
                "1111ffff66111111111111111111ffffbb111111111111111111ffff661111111111ffff6611\n"
                "1111ffff6611111111111111111166ffffffbb11111111111111ffff6611111111bbffff1111\n"
                "1111ffff661111111111111111111111bbffffffff1111111111ffffffffffffffffff111111\n"
                "1111ffff6611111111111111111111111111bbffffff66111111ffffffffffffbb6611111111\n"
                "1111ffff66111111111111111111111111111111ffffff111111ffff66111111111111111111\n"
                "1111ffff6611111111111111111111111111111111ffff661111ffff66111111111111111111\n"
                "1111ffff6611111111111111111111111111111111ffff661111ffff66111111111111111111\n"
                "1111ffff6611111111111111111111111111111111ffff111111ffff66111111111111111111\n"
                "1111ffff66111111111111111166111111111111bbffff111111ffff66111111111111111111\n"
                "1111ffffffffffffffffff6611ffffffffffffffffff11111111ffff66111111111111111111\n"
                "1111ffffffffffffffffff661166bbffffffffff661111111111ffff66111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
            },
            {
                -9, -9,
                0x11,
                "11111111111166bbffffffffbb66111111ffffffffffffffbb11111111111111111111111111\n"
                "1111111111bbffffffffffffffff111111ffffffffffffffffff661111111111111111111111\n"
                "1111111166ffff66111111116666111111ffff6611111111bbffff1111111111111111111111\n"
                "11111111bbffbb11111111111111111111ffff661111111111ffff6611111111111111111111\n"
                "11111111bbffbb11111111111111111111ffff661111111111bbffbb11111111111111111111\n"
                "11111111bbffff11111111111111111111ffff661111111111bbff6611111111111111111111\n"
                "1111111111ffffbb111111111111111111ffff661111111111ffff6611111111111111111111\n"
                "111111111166ffffffbb11111111111111ffff6611111111bbffff1111111111111111111111\n"
                "11111111111111bbffffffff1111111111ffffffffffffffffff111111111111111111111111\n"
                "111111111111111111bbffffff66111111ffffffffffffbb6611111111111111111111111111\n"
                "1111111111111111111111ffffff111111ffff66111111111111111111111111111111111111\n"
                "111111111111111111111111ffff661111ffff66111111111111111111111111111111111111\n"
                "111111111111111111111111ffff661111ffff66111111111111111111111111111111111111\n"
                "111111111111111111111111ffff111111ffff66111111111111111111111111111111111111\n"
                "1111111166111111111111bbffff111111ffff66111111111111111111111111111111111111\n"
                "ffff6611ffffffffffffffffff11111111ffff66111111111111111111111111111111111111\n"
                "ffff661166bbffffffffff661111111111ffff66111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
            },
            {
                10, -10,
                0x11,
                "111111111111111111111111ffff66111111111111111111bbffffffffffffffff111111ffff\n"
                "111111111111111111111111ffff66111111111111111166ffff66111111116666111111ffff\n"
                "111111111111111111111111ffff661111111111111111bbffbb11111111111111111111ffff\n"
                "111111111111111111111111ffff661111111111111111bbffbb11111111111111111111ffff\n"
                "111111111111111111111111ffff661111111111111111bbffff11111111111111111111ffff\n"
                "111111111111111111111111ffff66111111111111111111ffffbb111111111111111111ffff\n"
                "111111111111111111111111ffff6611111111111111111166ffffffbb11111111111111ffff\n"
                "111111111111111111111111ffff661111111111111111111111bbffffffff1111111111ffff\n"
                "111111111111111111111111ffff6611111111111111111111111111bbffffff66111111ffff\n"
                "111111111111111111111111ffff66111111111111111111111111111111ffffff111111ffff\n"
                "111111111111111111111111ffff6611111111111111111111111111111111ffff661111ffff\n"
                "111111111111111111111111ffff6611111111111111111111111111111111ffff661111ffff\n"
                "111111111111111111111111ffff6611111111111111111111111111111111ffff111111ffff\n"
                "111111111111111111111111ffff66111111111111111166111111111111bbffff111111ffff\n"
                "111111111111111111111111ffffffffffffffffff6611ffffffffffffffffff11111111ffff\n"
                "111111111111111111111111ffffffffffffffffff661166bbffffffffff661111111111ffff\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
            },
            {
                9, 11,
                0x11,
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111ffff661111111111111111111166bbffffffffbb66111111ffffff\n"
                "1111111111111111111111ffff66111111111111111111bbffffffffffffffff111111ffffff\n"
                "1111111111111111111111ffff66111111111111111166ffff66111111116666111111ffff66\n"
                "1111111111111111111111ffff661111111111111111bbffbb11111111111111111111ffff66\n"
                "1111111111111111111111ffff661111111111111111bbffbb11111111111111111111ffff66\n"
                "1111111111111111111111ffff661111111111111111bbffff11111111111111111111ffff66\n"
                "1111111111111111111111ffff66111111111111111111ffffbb111111111111111111ffff66\n"
                "1111111111111111111111ffff6611111111111111111166ffffffbb11111111111111ffff66\n"
                "1111111111111111111111ffff661111111111111111111111bbffffffff1111111111ffffff\n"
                "1111111111111111111111ffff6611111111111111111111111111bbffffff66111111ffffff\n"
                "1111111111111111111111ffff66111111111111111111111111111111ffffff111111ffff66\n"
                "1111111111111111111111ffff6611111111111111111111111111111111ffff661111ffff66\n"
                "1111111111111111111111ffff6611111111111111111111111111111111ffff661111ffff66\n"
            },
            {
                -11, 7,
                0x11,
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111166bbffffffffbb66111111ffffffffffffffbb111111111111111111111111111111\n"
                "111111bbffffffffffffffff111111ffffffffffffffffff6611111111111111111111111111\n"
                "111166ffff66111111116666111111ffff6611111111bbffff11111111111111111111111111\n"
                "1111bbffbb11111111111111111111ffff661111111111ffff66111111111111111111111111\n"
                "1111bbffbb11111111111111111111ffff661111111111bbffbb111111111111111111111111\n"
                "1111bbffff11111111111111111111ffff661111111111bbff66111111111111111111111111\n"
                "111111ffffbb111111111111111111ffff661111111111ffff66111111111111111111111111\n"
                "11111166ffffffbb11111111111111ffff6611111111bbffff11111111111111111111111111\n"
                "1111111111bbffffffff1111111111ffffffffffffffffff1111111111111111111111111111\n"
                "11111111111111bbffffff66111111ffffffffffffbb66111111111111111111111111111111\n"
                "111111111111111111ffffff111111ffff661111111111111111111111111111111111111111\n"
                "11111111111111111111ffff661111ffff661111111111111111111111111111111111111111\n"
                "11111111111111111111ffff661111ffff661111111111111111111111111111111111111111\n"
                "11111111111111111111ffff111111ffff661111111111111111111111111111111111111111\n"
                "111166111111111111bbffff111111ffff661111111111111111111111111111111111111111\n"
                "6611ffffffffffffffffff11111111ffff661111111111111111111111111111111111111111\n"
                "661166bbffffffffff661111111111ffff661111111111111111111111111111111111111111\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t sub_patterns[] =
        {
            {
                0, 0,
                0xee,
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeee000099eeeeeeeeeeeeeeeeeeee9944000000004499eeeeee0000000000000044eeeeeeee\n"
                "eeee000099eeeeeeeeeeeeeeeeee440000000000000000eeeeee00000000000000000099eeee\n"
                "eeee000099eeeeeeeeeeeeeeee99000099eeeeeeee9999eeeeee000099eeeeeeee440000eeee\n"
                "eeee000099eeeeeeeeeeeeeeee440044eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee000099ee\n"
                "eeee000099eeeeeeeeeeeeeeee440044eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee440044ee\n"
                "eeee000099eeeeeeeeeeeeeeee440000eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee440099ee\n"
                "eeee000099eeeeeeeeeeeeeeeeee000044eeeeeeeeeeeeeeeeee000099eeeeeeeeee000099ee\n"
                "eeee000099eeeeeeeeeeeeeeeeee9900000044eeeeeeeeeeeeee000099eeeeeeee440000eeee\n"
                "eeee000099eeeeeeeeeeeeeeeeeeeeee4400000000eeeeeeeeee000000000000000000eeeeee\n"
                "eeee000099eeeeeeeeeeeeeeeeeeeeeeeeee4400000099eeeeee0000000000004499eeeeeeee\n"
                "eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeee000000eeeeee000099eeeeeeeeeeeeeeeeee\n"
                "eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee000099eeee000099eeeeeeeeeeeeeeeeee\n"
                "eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee000099eeee000099eeeeeeeeeeeeeeeeee\n"
                "eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee0000eeeeee000099eeeeeeeeeeeeeeeeee\n"
                "eeee000099eeeeeeeeeeeeeeee99eeeeeeeeeeee440000eeeeee000099eeeeeeeeeeeeeeeeee\n"
                "eeee00000000000000000099ee000000000000000000eeeeeeee000099eeeeeeeeeeeeeeeeee\n"
                "eeee00000000000000000099ee9944000000000099eeeeeeeeee000099eeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
            },
            {
                -9, -9,
                0xee,
                "eeeeeeeeeeee9944000000004499eeeeee0000000000000044eeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeee440000000000000000eeeeee00000000000000000099eeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeee99000099eeeeeeee9999eeeeee000099eeeeeeee440000eeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeee440044eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee000099eeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeee440044eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee440044eeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeee440000eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee440099eeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeee000044eeeeeeeeeeeeeeeeee000099eeeeeeeeee000099eeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeee9900000044eeeeeeeeeeeeee000099eeeeeeee440000eeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeee4400000000eeeeeeeeee000000000000000000eeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeee4400000099eeeeee0000000000004499eeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeee000000eeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeee0000eeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeee99eeeeeeeeeeee440000eeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "000099ee000000000000000000eeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "000099ee9944000000000099eeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
            },
            {
                10, -10,
                0xee,
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeee440000000000000000eeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee99000099eeeeeeee9999eeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee440044eeeeeeeeeeeeeeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee440044eeeeeeeeeeeeeeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee440000eeeeeeeeeeeeeeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeee000044eeeeeeeeeeeeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeee9900000044eeeeeeeeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeee4400000000eeeeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeee4400000099eeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeee000000eeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee000099eeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee000099eeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee0000eeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee99eeeeeeeeeeee440000eeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee00000000000000000099ee000000000000000000eeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeee00000000000000000099ee9944000000000099eeeeeeeeee0000\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
            },
            {
                9, 11,
                0xee,
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeee9944000000004499eeeeee000000\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeee440000000000000000eeeeee000000\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee99000099eeeeeeee9999eeeeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee440044eeeeeeeeeeeeeeeeeeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee440044eeeeeeeeeeeeeeeeeeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeee440000eeeeeeeeeeeeeeeeeeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeee000044eeeeeeeeeeeeeeeeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeee9900000044eeeeeeeeeeeeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeee4400000000eeeeeeeeee000000\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeee4400000099eeeeee000000\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeee000000eeeeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee000099eeee000099\n"
                "eeeeeeeeeeeeeeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee000099eeee000099\n"
            },
            {
                -11, 7,
                0xee,
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeee9944000000004499eeeeee0000000000000044eeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeee440000000000000000eeeeee00000000000000000099eeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeee99000099eeeeeeee9999eeeeee000099eeeeeeee440000eeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeee440044eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeee440044eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee440044eeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeee440000eeeeeeeeeeeeeeeeeeee000099eeeeeeeeee440099eeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeee000044eeeeeeeeeeeeeeeeee000099eeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeee9900000044eeeeeeeeeeeeee000099eeeeeeee440000eeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeee4400000000eeeeeeeeee000000000000000000eeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeee4400000099eeeeee0000000000004499eeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeee000000eeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeee000099eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeee000099eeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeee0000eeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeee99eeeeeeeeeeee440000eeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "99ee000000000000000000eeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "99ee9944000000000099eeeeeeeeee000099eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t max_patterns[] =
        {
            {
                0, 0,
                0x88,
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888ffff888888888888888888888888aaffffffffaa88888888ffffffffffffffaa88888888\n"
                "8888ffff88888888888888888888aaffffffffffffffff888888ffffffffffffffffff888888\n"
                "8888ffff88888888888888888888ffff88888888888888888888ffff8888888888aaffff8888\n"
                "8888ffff888888888888888888aaffaa88888888888888888888ffff888888888888ffff8888\n"
                "8888ffff888888888888888888aaffaa88888888888888888888ffff888888888888aaffaa88\n"
                "8888ffff888888888888888888aaffff88888888888888888888ffff888888888888aaff8888\n"
                "8888ffff88888888888888888888ffffaa888888888888888888ffff888888888888ffff8888\n"
                "8888ffff8888888888888888888888ffffffaa88888888888888ffff8888888888aaffff8888\n"
                "8888ffff888888888888888888888888aaffffffff8888888888ffffffffffffffffff888888\n"
                "8888ffff8888888888888888888888888888aaffffff88888888ffffffffffffaa8888888888\n"
                "8888ffff88888888888888888888888888888888ffffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff8888888888888888888888888888888888ffff888888ffff88888888888888888888\n"
                "8888ffff88888888888888888888888888888888aaffff888888ffff88888888888888888888\n"
                "8888ffffffffffffffffff8888ffffffffffffffffff88888888ffff88888888888888888888\n"
                "8888ffffffffffffffffff888888aaffffffffff888888888888ffff88888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                -9, -9,
                0x88,
                "88888888888888aaffffffffaa88888888ffffffffffffffaa88888888888888888888888888\n"
                "8888888888aaffffffffffffffff888888ffffffffffffffffff888888888888888888888888\n"
                "8888888888ffff88888888888888888888ffff8888888888aaffff8888888888888888888888\n"
                "88888888aaffaa88888888888888888888ffff888888888888ffff8888888888888888888888\n"
                "88888888aaffaa88888888888888888888ffff888888888888aaffaa88888888888888888888\n"
                "88888888aaffff88888888888888888888ffff888888888888aaff8888888888888888888888\n"
                "8888888888ffffaa888888888888888888ffff888888888888ffff8888888888888888888888\n"
                "888888888888ffffffaa88888888888888ffff8888888888aaffff8888888888888888888888\n"
                "88888888888888aaffffffff8888888888ffffffffffffffffff888888888888888888888888\n"
                "888888888888888888aaffffff88888888ffffffffffffaa8888888888888888888888888888\n"
                "8888888888888888888888ffffff888888ffff88888888888888888888888888888888888888\n"
                "888888888888888888888888ffff888888ffff88888888888888888888888888888888888888\n"
                "888888888888888888888888ffff888888ffff88888888888888888888888888888888888888\n"
                "888888888888888888888888ffff888888ffff88888888888888888888888888888888888888\n"
                "8888888888888888888888aaffff888888ffff88888888888888888888888888888888888888\n"
                "ffff8888ffffffffffffffffff88888888ffff88888888888888888888888888888888888888\n"
                "ffff888888aaffffffffff888888888888ffff88888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
                "8888888888888888888888888888888888888888888888888888888888888888888888888888\n"
            },
            {
                10, -10,
                0x11,
                "111111111111111111111111ffff55111111111111111111aaffffffffffffffff111111ffff\n"
                "111111111111111111111111ffff55111111111111111155ffff55111111115555111111ffff\n"
                "111111111111111111111111ffff551111111111111111aaffaa11111111111111111111ffff\n"
                "111111111111111111111111ffff551111111111111111aaffaa11111111111111111111ffff\n"
                "111111111111111111111111ffff551111111111111111aaffff11111111111111111111ffff\n"
                "111111111111111111111111ffff55111111111111111111ffffaa111111111111111111ffff\n"
                "111111111111111111111111ffff5511111111111111111155ffffffaa11111111111111ffff\n"
                "111111111111111111111111ffff551111111111111111111111aaffffffff1111111111ffff\n"
                "111111111111111111111111ffff5511111111111111111111111111aaffffff55111111ffff\n"
                "111111111111111111111111ffff55111111111111111111111111111111ffffff111111ffff\n"
                "111111111111111111111111ffff5511111111111111111111111111111111ffff551111ffff\n"
                "111111111111111111111111ffff5511111111111111111111111111111111ffff551111ffff\n"
                "111111111111111111111111ffff5511111111111111111111111111111111ffff111111ffff\n"
                "111111111111111111111111ffff55111111111111111155111111111111aaffff111111ffff\n"
                "111111111111111111111111ffffffffffffffffff5511ffffffffffffffffff11111111ffff\n"
                "111111111111111111111111ffffffffffffffffff551155aaffffffffff551111111111ffff\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
            },
            {
                9, 11,
                0x55,
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555555555555555555555555555555555555555555555555555555555\n"
                "5555555555555555555555ffff555555555555555555555555aaffffffffaa55555555ffffff\n"
                "5555555555555555555555ffff55555555555555555555aaffffffffffffffff555555ffffff\n"
                "5555555555555555555555ffff55555555555555555555ffff55555555555555555555ffff55\n"
                "5555555555555555555555ffff555555555555555555aaffaa55555555555555555555ffff55\n"
                "5555555555555555555555ffff555555555555555555aaffaa55555555555555555555ffff55\n"
                "5555555555555555555555ffff555555555555555555aaffff55555555555555555555ffff55\n"
                "5555555555555555555555ffff55555555555555555555ffffaa555555555555555555ffff55\n"
                "5555555555555555555555ffff5555555555555555555555ffffffaa55555555555555ffff55\n"
                "5555555555555555555555ffff555555555555555555555555aaffffffff5555555555ffffff\n"
                "5555555555555555555555ffff5555555555555555555555555555aaffffff55555555ffffff\n"
                "5555555555555555555555ffff55555555555555555555555555555555ffffff555555ffff55\n"
                "5555555555555555555555ffff5555555555555555555555555555555555ffff555555ffff55\n"
                "5555555555555555555555ffff5555555555555555555555555555555555ffff555555ffff55\n"
            },
            {
                -11, 7,
                0x11,
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111111111111111111111111111111111111111111111111111111111111111111111111\n"
                "1111111155aaffffffffaa55111111ffffffffffffffaa111111111111111111111111111111\n"
                "111111aaffffffffffffffff111111ffffffffffffffffff5511111111111111111111111111\n"
                "111155ffff55111111115555111111ffff5511111111aaffff11111111111111111111111111\n"
                "1111aaffaa11111111111111111111ffff551111111111ffff55111111111111111111111111\n"
                "1111aaffaa11111111111111111111ffff551111111111aaffaa111111111111111111111111\n"
                "1111aaffff11111111111111111111ffff551111111111aaff55111111111111111111111111\n"
                "111111ffffaa111111111111111111ffff551111111111ffff55111111111111111111111111\n"
                "11111155ffffffaa11111111111111ffff5511111111aaffff11111111111111111111111111\n"
                "1111111111aaffffffff1111111111ffffffffffffffffff1111111111111111111111111111\n"
                "11111111111111aaffffff55111111ffffffffffffaa55111111111111111111111111111111\n"
                "111111111111111111ffffff111111ffff551111111111111111111111111111111111111111\n"
                "11111111111111111111ffff551111ffff551111111111111111111111111111111111111111\n"
                "11111111111111111111ffff551111ffff551111111111111111111111111111111111111111\n"
                "11111111111111111111ffff111111ffff551111111111111111111111111111111111111111\n"
                "111155111111111111aaffff111111ffff551111111111111111111111111111111111111111\n"
                "5511ffffffffffffffffff11111111ffff551111111111111111111111111111111111111111\n"
                "551155aaffffffffff551111111111ffff551111111111111111111111111111111111111111\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };

        static const bitmap_pattern_t min_patterns[] =
        {
            {
                0, 0,
                0xee,
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000eeee550000000000000000000055aaeeeeeeeeaa55000000eeeeeeeeeeeeeeaa00000000\n"
                "0000eeee55000000000000000000aaeeeeeeeeeeeeeeee000000eeeeeeeeeeeeeeeeee550000\n"
                "0000eeee55000000000000000055eeee55000000005555000000eeee5500000000aaeeee0000\n"
                "0000eeee550000000000000000aaeeaa00000000000000000000eeee550000000000eeee5500\n"
                "0000eeee550000000000000000aaeeaa00000000000000000000eeee550000000000aaeeaa00\n"
                "0000eeee550000000000000000aaeeee00000000000000000000eeee550000000000aaee5500\n"
                "0000eeee55000000000000000000eeeeaa000000000000000000eeee550000000000eeee5500\n"
                "0000eeee5500000000000000000055eeeeeeaa00000000000000eeee5500000000aaeeee0000\n"
                "0000eeee550000000000000000000000aaeeeeeeee0000000000eeeeeeeeeeeeeeeeee000000\n"
                "0000eeee5500000000000000000000000000aaeeeeee55000000eeeeeeeeeeeeaa5500000000\n"
                "0000eeee55000000000000000000000000000000eeeeee000000eeee55000000000000000000\n"
                "0000eeee5500000000000000000000000000000000eeee550000eeee55000000000000000000\n"
                "0000eeee5500000000000000000000000000000000eeee550000eeee55000000000000000000\n"
                "0000eeee5500000000000000000000000000000000eeee000000eeee55000000000000000000\n"
                "0000eeee55000000000000000055000000000000aaeeee000000eeee55000000000000000000\n"
                "0000eeeeeeeeeeeeeeeeee5500eeeeeeeeeeeeeeeeee00000000eeee55000000000000000000\n"
                "0000eeeeeeeeeeeeeeeeee550055aaeeeeeeeeee550000000000eeee55000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
                "0000000000000000000000000000000000000000000000000000000000000000000000000000\n"
            },
            {
                -9, -9,
                0xee,
                "00000000000055aaeeeeeeeeaa55000000eeeeeeeeeeeeeeaa00000000eeeeeeeeeeeeeeeeee\n"
                "0000000000aaeeeeeeeeeeeeeeee000000eeeeeeeeeeeeeeeeee550000eeeeeeeeeeeeeeeeee\n"
                "0000000055eeee55000000005555000000eeee5500000000aaeeee0000eeeeeeeeeeeeeeeeee\n"
                "00000000aaeeaa00000000000000000000eeee550000000000eeee5500eeeeeeeeeeeeeeeeee\n"
                "00000000aaeeaa00000000000000000000eeee550000000000aaeeaa00eeeeeeeeeeeeeeeeee\n"
                "00000000aaeeee00000000000000000000eeee550000000000aaee5500eeeeeeeeeeeeeeeeee\n"
                "0000000000eeeeaa000000000000000000eeee550000000000eeee5500eeeeeeeeeeeeeeeeee\n"
                "000000000055eeeeeeaa00000000000000eeee5500000000aaeeee0000eeeeeeeeeeeeeeeeee\n"
                "00000000000000aaeeeeeeee0000000000eeeeeeeeeeeeeeeeee000000eeeeeeeeeeeeeeeeee\n"
                "000000000000000000aaeeeeee55000000eeeeeeeeeeeeaa5500000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000eeeeee000000eeee55000000000000000000eeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000eeee550000eeee55000000000000000000eeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000eeee550000eeee55000000000000000000eeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000eeee000000eeee55000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000055000000000000aaeeee000000eeee55000000000000000000eeeeeeeeeeeeeeeeee\n"
                "eeee5500eeeeeeeeeeeeeeeeee00000000eeee55000000000000000000eeeeeeeeeeeeeeeeee\n"
                "eeee550055aaeeeeeeeeee550000000000eeee55000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeee\n"
                "0000000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
            },
            {
                10, -10,
                0xaa,
                "aaaaaaaaaaaaaaaaaaaa0000aaaa55000000000000000000aaaaaaaaaaaaaaaaaa000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa55000000000000000055aaaa55000000005555000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa550000000000000000aaaaaa00000000000000000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa550000000000000000aaaaaa00000000000000000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa550000000000000000aaaaaa00000000000000000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa55000000000000000000aaaaaa000000000000000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa5500000000000000000055aaaaaaaa00000000000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa550000000000000000000000aaaaaaaaaa0000000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa5500000000000000000000000000aaaaaaaa55000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa55000000000000000000000000000000aaaaaa000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa5500000000000000000000000000000000aaaa550000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa5500000000000000000000000000000000aaaa550000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa5500000000000000000000000000000000aaaa000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaa55000000000000000055000000000000aaaaaa000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaaaaaaaaaaaaaaaa5500aaaaaaaaaaaaaaaaaa00000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa0000aaaaaaaaaaaaaaaaaa550055aaaaaaaaaaaa550000000000aaaa\n"
                "aaaaaaaaaaaaaaaaaaaa00000000000000000000000000000000000000000000000000000000\n"
                "aaaaaaaaaaaaaaaaaaaa00000000000000000000000000000000000000000000000000000000\n"
                "aaaaaaaaaaaaaaaaaaaa00000000000000000000000000000000000000000000000000000000\n"
                "aaaaaaaaaaaaaaaaaaaa00000000000000000000000000000000000000000000000000000000\n"
                "aaaaaaaaaaaaaaaaaaaa00000000000000000000000000000000000000000000000000000000\n"
                "aaaaaaaaaaaaaaaaaaaa00000000000000000000000000000000000000000000000000000000\n"
                "aaaaaaaaaaaaaaaaaaaa00000000000000000000000000000000000000000000000000000000\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
                "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa\n"
            },
            {
                9, 11,
                0xee,
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000000000000000000000000000000000000000000000000000000000\n"
                "eeeeeeeeeeeeeeeeee0000eeee550000000000000000000055aaeeeeeeeeaa55000000eeeeee\n"
                "eeeeeeeeeeeeeeeeee0000eeee55000000000000000000aaeeeeeeeeeeeeeeee000000eeeeee\n"
                "eeeeeeeeeeeeeeeeee0000eeee55000000000000000055eeee55000000005555000000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee550000000000000000aaeeaa00000000000000000000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee550000000000000000aaeeaa00000000000000000000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee550000000000000000aaeeee00000000000000000000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee55000000000000000000eeeeaa000000000000000000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee5500000000000000000055eeeeeeaa00000000000000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee550000000000000000000000aaeeeeeeee0000000000eeeeee\n"
                "eeeeeeeeeeeeeeeeee0000eeee5500000000000000000000000000aaeeeeee55000000eeeeee\n"
                "eeeeeeeeeeeeeeeeee0000eeee55000000000000000000000000000000eeeeee000000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee5500000000000000000000000000000000eeee550000eeee55\n"
                "eeeeeeeeeeeeeeeeee0000eeee5500000000000000000000000000000000eeee550000eeee55\n"
            },
            {
                -11, 7,
                0xee,
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "eeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000000000000000000000000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "0000000055aaeeeeeeeeaa55000000eeeeeeeeeeeeeeaa00000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000aaeeeeeeeeeeeeeeee000000eeeeeeeeeeeeeeeeee550000eeeeeeeeeeeeeeeeeeeeee\n"
                "000055eeee55000000005555000000eeee5500000000aaeeee0000eeeeeeeeeeeeeeeeeeeeee\n"
                "0000aaeeaa00000000000000000000eeee550000000000eeee5500eeeeeeeeeeeeeeeeeeeeee\n"
                "0000aaeeaa00000000000000000000eeee550000000000aaeeaa00eeeeeeeeeeeeeeeeeeeeee\n"
                "0000aaeeee00000000000000000000eeee550000000000aaee5500eeeeeeeeeeeeeeeeeeeeee\n"
                "000000eeeeaa000000000000000000eeee550000000000eeee5500eeeeeeeeeeeeeeeeeeeeee\n"
                "00000055eeeeeeaa00000000000000eeee5500000000aaeeee0000eeeeeeeeeeeeeeeeeeeeee\n"
                "0000000000aaeeeeeeee0000000000eeeeeeeeeeeeeeeeee000000eeeeeeeeeeeeeeeeeeeeee\n"
                "00000000000000aaeeeeee55000000eeeeeeeeeeeeaa5500000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000000000000000000eeeeee000000eeee55000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "00000000000000000000eeee550000eeee55000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "00000000000000000000eeee550000eeee55000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "00000000000000000000eeee000000eeee55000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "000055000000000000aaeeee000000eeee55000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "5500eeeeeeeeeeeeeeeeee00000000eeee55000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
                "550055aaeeeeeeeeee550000000000eeee55000000000000000000eeeeeeeeeeeeeeeeeeeeee\n"
            },
            {
                0, 0,
                0,
                NULL
            }
        };


    } /* namespace test */
} /* namespace lsp */

UTEST_BEGIN("dsp.bitmap", b2b8)

    void test_pattern(const char *name, bitmap_func_b1b8_t func, const bitmap_pattern_t *pattern)
    {
        printf("Testing '%s' at point (%d, %d)\n", name, int(pattern->x), int(pattern->y));

        dsp::bitmap_t tmp;
        ByteBuffer bbuf(bitmap.width * bitmap.height);

        tmp.width   = bitmap.width;
        tmp.height  = bitmap.height;
        tmp.stride  = tmp.width;
        tmp.data    = bbuf.data();
        memset(bbuf.data(), pattern->fill, bitmap.width * bitmap.height);

        func(&tmp, &bitmap, pattern->x, pattern->y);

        UTEST_ASSERT(!bbuf.corrupted());

        char *result = rasterize(&tmp);
        UTEST_ASSERT(result != NULL);
        lsp_finally { free(result); };

        if (strcmp(result, pattern->data) != 0)
        {
            UTEST_FAIL_MSG("Faled rasterization:\ngot pattern:\n%s\nexpected pattern:\n%s\n", result, pattern->data);
        }
    }

    void test_patterns(const char *name, bitmap_func_b1b8_t func, const bitmap_pattern_t * list)
    {
        if (!UTEST_SUPPORTED(func))
            return;

        for (; list->data != NULL; ++list)
            test_pattern(name, func, list);
    }


    UTEST_MAIN
    {
        using namespace lsp;

        #define TEST_PATTERNS(func, list) \
            test_patterns(#func, func, list)

        TEST_PATTERNS(generic::bitmap_put_b2b8, put_patterns);
        TEST_PATTERNS(generic::bitmap_add_b2b8, add_patterns);
        TEST_PATTERNS(generic::bitmap_sub_b2b8, sub_patterns);
        TEST_PATTERNS(generic::bitmap_max_b2b8, max_patterns);
        TEST_PATTERNS(generic::bitmap_min_b2b8, min_patterns);
    }

UTEST_END;





