static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2015 Martin Blanchard <tchaik@gmx.com>                           \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_file_path (path, _(\"File\"), \"\")                                  \n"
"  description (_(\"Path of file to load\"))                                   \n"
"property_uri (uri, _(\"URI\"), \"\")                                          \n"
"  description (_(\"URI for file to load\"))                                   \n"
"                                                                              \n"
"property_int(directory, _(\"Directory\"), 1)                                  \n"
"  description (_(\"Image file directory (subfile)\"))                         \n"
"  value_range (1, G_MAXINT)                                                   \n"
"  ui_range (1, 16)                                                            \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_SOURCE                                                        \n"
"#define GEGL_OP_C_SOURCE tiff-load.c                                          \n"
"                                                                              \n"
"#include <gegl-op.h>                                                          \n"
"#include <gegl-gio-private.h>                                                 \n"
"#include <glib/gprintf.h>                                                     \n"
"#include <tiffio.h>                                                           \n"
"                                                                              \n"
"typedef enum {                                                                \n"
"  TIFF_LOADING_RGBA,                                                          \n"
"  TIFF_LOADING_CONTIGUOUS,                                                    \n"
"  TIFF_LOADING_SEPARATED                                                      \n"
"} LoadingMode;                                                                \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  GFile *file;                                                                \n"
"  GInputStream *stream;                                                       \n"
"  gboolean can_seek;                                                          \n"
"                                                                              \n"
"  gchar *buffer;                                                              \n"
"  gsize allocated;                                                            \n"
"  gsize position;                                                             \n"
"  gsize loaded;                                                               \n"
"                                                                              \n"
"  TIFF *tiff;                                                                 \n"
"                                                                              \n"
"  gint directory;                                                             \n"
"                                                                              \n"
"  const Babl *format;                                                         \n"
"  LoadingMode mode;                                                           \n"
"                                                                              \n"
"  gint width;                                                                 \n"
"  gint height;                                                                \n"
"} Priv;                                                                       \n"
"                                                                              \n"
"static void                                                                   \n"
"cleanup(GeglOperation *operation)                                             \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  Priv *p = (Priv*) o->user_data;                                             \n"
"                                                                              \n"
"  if (p != NULL)                                                              \n"
"    {                                                                         \n"
"      if (p->tiff != NULL)                                                    \n"
"        TIFFClose(p->tiff);                                                   \n"
"      else if (p->stream != NULL)                                             \n"
"        g_input_stream_close(G_INPUT_STREAM(p->stream), NULL, NULL);          \n"
"      if (p->stream != NULL)                                                  \n"
"        g_clear_object(&p->stream);                                           \n"
"      p->tiff = NULL;                                                         \n"
"                                                                              \n"
"      if (p->file != NULL)                                                    \n"
"        g_clear_object(&p->file);                                             \n"
"                                                                              \n"
"      p->width = p->height = 0;                                               \n"
"      p->directory = 0;                                                       \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static GSeekType                                                              \n"
"lseek_to_seek_type(int whence)                                                \n"
"{                                                                             \n"
"  switch (whence)                                                             \n"
"    {                                                                         \n"
"    default:                                                                  \n"
"    case SEEK_SET:                                                            \n"
"      return G_SEEK_SET;                                                      \n"
"                                                                              \n"
"    case SEEK_CUR:                                                            \n"
"      return G_SEEK_CUR;                                                      \n"
"                                                                              \n"
"    case SEEK_END:                                                            \n"
"      return G_SEEK_END;                                                      \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"error_handler(const char *module,                                             \n"
"              const char *format,                                             \n"
"              va_list arguments)                                              \n"
"{                                                                             \n"
"  gchar *message;                                                             \n"
"                                                                              \n"
"  g_vasprintf(&message, format, arguments);                                   \n"
"  g_warning(\"%s\", message);                                                 \n"
"                                                                              \n"
"  g_free(message);                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"warning_handler(const char *module,                                           \n"
"                const char *format,                                           \n"
"                va_list arguments)                                            \n"
"{                                                                             \n"
"  gchar *message;                                                             \n"
"                                                                              \n"
"  g_vasprintf(&message, format, arguments);                                   \n"
"  g_message(\"%s\", message);                                                 \n"
"                                                                              \n"
"  g_free(message);                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"static tsize_t                                                                \n"
"read_from_stream(thandle_t handle,                                            \n"
"                 tdata_t buffer,                                              \n"
"                 tsize_t size)                                                \n"
"{                                                                             \n"
"  Priv *p = (Priv*) handle;                                                   \n"
"  GError *error = NULL;                                                       \n"
"  gchar *new_buffer;                                                          \n"
"  gsize new_size = 1;                                                         \n"
"  gsize missing, needed;                                                      \n"
"  gssize read = -1;                                                           \n"
"                                                                              \n"
"  g_assert(p->stream);                                                        \n"
"                                                                              \n"
"  if (p->can_seek)                                                            \n"
"    {                                                                         \n"
"      read = g_input_stream_read(G_INPUT_STREAM(p->stream),                   \n"
"                                 (void *) buffer, (gsize) size,               \n"
"                                 NULL, &error);                               \n"
"      if (read < 0)                                                           \n"
"        {                                                                     \n"
"          g_warning(\"%s\", error->message);                                  \n"
"          g_error_free(error);                                                \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      if (p->position + size > p->loaded)                                     \n"
"        {                                                                     \n"
"          missing = p->position + size - p->loaded;                           \n"
"          if (p->loaded + missing > p->allocated)                             \n"
"            {                                                                 \n"
"              needed = p->loaded + missing - p->allocated;                    \n"
"              while (new_size < p->allocated + needed)                        \n"
"                new_size *= 2;                                                \n"
"                                                                              \n"
"              new_buffer = g_try_realloc(p->buffer, new_size);                \n"
"              if (!new_buffer)                                                \n"
"                return -1;                                                    \n"
"                                                                              \n"
"              p->allocated = new_size;                                        \n"
"              p->buffer = new_buffer;                                         \n"
"            }                                                                 \n"
"                                                                              \n"
"          while (missing > 0)                                                 \n"
"            {                                                                 \n"
"              read = g_input_stream_read(G_INPUT_STREAM(p->stream),           \n"
"                                         (void *) (p->buffer + p->loaded),    \n"
"                                         missing,                             \n"
"                                         NULL, &error);                       \n"
"              if (read < 0)                                                   \n"
"                {                                                             \n"
"                  g_warning(\"%s\", error->message);                          \n"
"                  g_error_free(error);                                        \n"
"                  break;                                                      \n"
"                }                                                             \n"
"                                                                              \n"
"              p->loaded += read;                                              \n"
"              missing -= read;                                                \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      g_assert(p->position + size <= p->loaded);                              \n"
"                                                                              \n"
"      memcpy(buffer, p->buffer + p->position, size);                          \n"
"      p->position += size;                                                    \n"
"      read = size;                                                            \n"
"    }                                                                         \n"
"                                                                              \n"
"  return (tsize_t) read;                                                      \n"
"}                                                                             \n"
"                                                                              \n"
"static tsize_t                                                                \n"
"write_to_stream(thandle_t handle,                                             \n"
"                tdata_t buffer,                                               \n"
"                tsize_t size)                                                 \n"
"{                                                                             \n"
"  Priv *p = (Priv*) handle;                                                   \n"
"                                                                              \n"
"  g_assert(p->stream && FALSE);                                               \n"
"                                                                              \n"
"  return -1;                                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"static toff_t                                                                 \n"
"seek_in_stream(thandle_t handle,                                              \n"
"               toff_t offset,                                                 \n"
"               int whence)                                                    \n"
"{                                                                             \n"
"  Priv *p = (Priv*) handle;                                                   \n"
"  GError *error = NULL;                                                       \n"
"  gboolean sought = FALSE;                                                    \n"
"  goffset position = -1;                                                      \n"
"                                                                              \n"
"  g_assert(p->stream);                                                        \n"
"                                                                              \n"
"  if (p->can_seek)                                                            \n"
"    {                                                                         \n"
"      sought = g_seekable_seek(G_SEEKABLE(p->stream),                         \n"
"                               (goffset) offset, lseek_to_seek_type(whence),  \n"
"                               NULL, &error);                                 \n"
"      if (sought)                                                             \n"
"        position = g_seekable_tell(G_SEEKABLE(p->stream));                    \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          g_warning(\"%s\", error->message);                                  \n"
"          g_error_free(error);                                                \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      switch (whence)                                                         \n"
"        {                                                                     \n"
"        default:                                                              \n"
"        case SEEK_SET:                                                        \n"
"          if (offset <= p->loaded)                                            \n"
"            position = p->position = offset;                                  \n"
"          break;                                                              \n"
"                                                                              \n"
"        case SEEK_CUR:                                                        \n"
"          if (p->position + offset <= p->loaded)                              \n"
"            position = p->position += offset;                                 \n"
"          break;                                                              \n"
"                                                                              \n"
"        case G_SEEK_END:                                                      \n"
"          if (p->loaded + offset <= p->loaded)                                \n"
"            position = p->position = p->loaded + offset;                      \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return (toff_t) position;                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static int                                                                    \n"
"close_stream(thandle_t handle)                                                \n"
"{                                                                             \n"
"  Priv *p = (Priv*) handle;                                                   \n"
"  GError *error = NULL;                                                       \n"
"  gboolean closed = FALSE;                                                    \n"
"                                                                              \n"
"  g_assert(p->stream);                                                        \n"
"                                                                              \n"
"  closed = g_input_stream_close(G_INPUT_STREAM(p->stream),                    \n"
"                                NULL, &error);                                \n"
"  if (!closed)                                                                \n"
"    {                                                                         \n"
"      g_warning(\"%s\", error->message);                                      \n"
"      g_error_free(error);                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_clear_object(&p->stream);                                                 \n"
"                                                                              \n"
"  p->loaded = 0;                                                              \n"
"  p->position = 0;                                                            \n"
"                                                                              \n"
"  if (p->buffer != NULL)                                                      \n"
"    g_free(p->buffer);                                                        \n"
"  p->buffer = NULL;                                                           \n"
"                                                                              \n"
"  p->allocated = 0;                                                           \n"
"                                                                              \n"
"  return (closed) ? 0 : -1;                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static toff_t                                                                 \n"
"get_file_size(thandle_t handle)                                               \n"
"{                                                                             \n"
"  Priv *p = (Priv*) handle;                                                   \n"
"  GError *error = NULL;                                                       \n"
"  GFileInfo *info;                                                            \n"
"  goffset size;                                                               \n"
"                                                                              \n"
"  g_assert(p->stream);                                                        \n"
"                                                                              \n"
"  size = p->loaded;                                                           \n"
"                                                                              \n"
"  if (p->file != NULL)                                                        \n"
"    {                                                                         \n"
"      info = g_file_query_info(p->file,                                       \n"
"                               G_FILE_ATTRIBUTE_STANDARD_SIZE,                \n"
"                               G_FILE_QUERY_INFO_NONE,                        \n"
"                               NULL, &error);                                 \n"
"      if (info == NULL)                                                       \n"
"        {                                                                     \n"
"          g_warning(\"%s\", error->message);                                  \n"
"          g_error_free(error);                                                \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          if (g_file_info_has_attribute(info, G_FILE_ATTRIBUTE_STANDARD_SIZE))\n"
"            size = g_file_info_get_size(info);                                \n"
"          g_object_unref(info);                                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return (toff_t) size;                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"query_tiff(GeglOperation *operation)                                          \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  Priv *p = (Priv*) o->user_data;                                             \n"
"  gshort color_space, compression;                                            \n"
"  gushort bits_per_sample, samples_per_pixel;                                 \n"
"  gushort sample_format;                                                      \n"
"  gboolean has_alpha = FALSE;                                                 \n"
"  gboolean alpha_is_premultiplied = FALSE;                                    \n"
"  gushort *extra_types = NULL;                                                \n"
"  gushort nb_extras, planar_config;                                           \n"
"  gboolean fallback_mode = FALSE;                                             \n"
"  gchar format_string[32];                                                    \n"
"  guint width, height;                                                        \n"
"                                                                              \n"
"  g_return_val_if_fail(p->tiff != NULL, -1);                                  \n"
"                                                                              \n"
"  if (!TIFFGetField(p->tiff, TIFFTAG_IMAGEWIDTH, &width))                     \n"
"    {                                                                         \n"
"      g_warning(\"could not get TIFF image width\");                          \n"
"      return -1;                                                              \n"
"    }                                                                         \n"
"  else if (!TIFFGetField(p->tiff, TIFFTAG_IMAGELENGTH, &height))              \n"
"    {                                                                         \n"
"      g_warning(\"could not get TIFF image height\");                         \n"
"      return -1;                                                              \n"
"    }                                                                         \n"
"                                                                              \n"
"  TIFFGetFieldDefaulted(p->tiff, TIFFTAG_COMPRESSION, &compression);          \n"
"  if (!TIFFGetField(p->tiff, TIFFTAG_PHOTOMETRIC, &color_space))              \n"
"    {                                                                         \n"
"      g_warning(\"could not get photometric from TIFF image\");               \n"
"      if (compression == COMPRESSION_CCITTFAX3 ||                             \n"
"          compression == COMPRESSION_CCITTFAX4 ||                             \n"
"          compression == COMPRESSION_CCITTRLE  ||                             \n"
"          compression == COMPRESSION_CCITTRLEW)                               \n"
"        {                                                                     \n"
"          g_message(\"assuming min-is-white (CCITT compressed)\");            \n"
"          color_space = PHOTOMETRIC_MINISWHITE;                               \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          g_message(\"assuming min-is-black\");                               \n"
"          color_space = PHOTOMETRIC_MINISBLACK;                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  TIFFGetFieldDefaulted(p->tiff, TIFFTAG_SAMPLESPERPIXEL, &samples_per_pixel);\n"
"  if (!TIFFGetField(p->tiff, TIFFTAG_EXTRASAMPLES, &nb_extras, &extra_types)) \n"
"    nb_extras = 0;                                                            \n"
"                                                                              \n"
"  if (nb_extras > 0)                                                          \n"
"    {                                                                         \n"
"      if (extra_types[0] == EXTRASAMPLE_ASSOCALPHA)                           \n"
"        {                                                                     \n"
"          has_alpha = TRUE;                                                   \n"
"          alpha_is_premultiplied = TRUE;                                      \n"
"          nb_extras--;                                                        \n"
"        }                                                                     \n"
"      else if (extra_types[0] == EXTRASAMPLE_UNASSALPHA)                      \n"
"        {                                                                     \n"
"          has_alpha = TRUE;                                                   \n"
"          alpha_is_premultiplied = FALSE;                                     \n"
"          nb_extras--;                                                        \n"
"        }                                                                     \n"
"      else if (extra_types[0] == EXTRASAMPLE_UNSPECIFIED)                     \n"
"        {                                                                     \n"
"          has_alpha = TRUE;                                                   \n"
"          alpha_is_premultiplied = FALSE;                                     \n"
"          nb_extras--;                                                        \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  switch(color_space)                                                         \n"
"    {                                                                         \n"
"    case PHOTOMETRIC_MINISBLACK:                                              \n"
"    case PHOTOMETRIC_MINISWHITE:                                              \n"
"      if (samples_per_pixel > 1 + nb_extras)                                  \n"
"        {                                                                     \n"
"          nb_extras = samples_per_pixel - 2;                                  \n"
"          has_alpha = TRUE;                                                   \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (has_alpha)                                                          \n"
"        {                                                                     \n"
"          if(alpha_is_premultiplied)                                          \n"
"            g_strlcpy(format_string, \"Y'aA \", 32);                          \n"
"          else                                                                \n"
"            g_strlcpy(format_string, \"Y'A \", 32);                           \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        g_strlcpy(format_string, \"Y' \", 32);                                \n"
"      break;                                                                  \n"
"                                                                              \n"
"    case PHOTOMETRIC_RGB:                                                     \n"
"      if (samples_per_pixel > 3 + nb_extras)                                  \n"
"        {                                                                     \n"
"          nb_extras = samples_per_pixel - 4;                                  \n"
"          has_alpha = TRUE;                                                   \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (has_alpha)                                                          \n"
"        {                                                                     \n"
"          if (alpha_is_premultiplied)                                         \n"
"            g_strlcpy(format_string, \"R'aG'aB'aA \", 32);                    \n"
"          else                                                                \n"
"            g_strlcpy(format_string, \"R'G'B'A \", 32);                       \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        g_strlcpy(format_string, \"R'G'B' \", 32);                            \n"
"      break;                                                                  \n"
"                                                                              \n"
"    default:                                                                  \n"
"      fallback_mode = TRUE;                                                   \n"
"      break;                                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  TIFFGetFieldDefaulted(p->tiff, TIFFTAG_SAMPLEFORMAT, &sample_format);       \n"
"  TIFFGetFieldDefaulted(p->tiff, TIFFTAG_BITSPERSAMPLE, &bits_per_sample);    \n"
"                                                                              \n"
"  switch(bits_per_sample)                                                     \n"
"    {                                                                         \n"
"    case 8:                                                                   \n"
"      g_strlcat(format_string, \"u8\", 32);                                   \n"
"      break;                                                                  \n"
"                                                                              \n"
"    case 16:                                                                  \n"
"      if (sample_format == SAMPLEFORMAT_IEEEFP)                               \n"
"        g_strlcat(format_string, \"half\", 32);                               \n"
"      else                                                                    \n"
"        g_strlcat(format_string, \"u16\", 32);                                \n"
"      break;                                                                  \n"
"                                                                              \n"
"    case 32:                                                                  \n"
"      if (sample_format == SAMPLEFORMAT_IEEEFP)                               \n"
"        g_strlcat(format_string, \"float\", 32);                              \n"
"      else                                                                    \n"
"        g_strlcat(format_string, \"u32\", 32);                                \n"
"      break;                                                                  \n"
"                                                                              \n"
"    case 64:                                                                  \n"
"      g_strlcat(format_string, \"double\", 32);                               \n"
"      break;                                                                  \n"
"                                                                              \n"
"    default:                                                                  \n"
"      fallback_mode = TRUE;                                                   \n"
"      break;                                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (fallback_mode == TRUE)                                                  \n"
"    g_strlcpy(format_string, \"R'aG'aB'aA u8\", 32);                          \n"
"                                                                              \n"
"  TIFFGetFieldDefaulted(p->tiff, TIFFTAG_PLANARCONFIG, &planar_config);       \n"
"                                                                              \n"
"  p->format = babl_format(format_string);                                     \n"
"  if (fallback_mode)                                                          \n"
"    p->mode = TIFF_LOADING_RGBA;                                              \n"
"  else if (planar_config == PLANARCONFIG_CONTIG)                              \n"
"    p->mode = TIFF_LOADING_CONTIGUOUS;                                        \n"
"  else                                                                        \n"
"    p->mode = TIFF_LOADING_SEPARATED;                                         \n"
"                                                                              \n"
"  p->height = (gint) height;                                                  \n"
"  p->width = (gint) width;                                                    \n"
"                                                                              \n"
"  return 0;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"load_RGBA(GeglOperation *operation,                                           \n"
"          GeglBuffer    *output)                                              \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  Priv *p = (Priv*) o->user_data;                                             \n"
"  guint32 *buffer;                                                            \n"
"  gint row;                                                                   \n"
"                                                                              \n"
"  g_return_val_if_fail(p->tiff != NULL, -1);                                  \n"
"                                                                              \n"
"  buffer = g_try_new(guint32, p->width * p->height * sizeof(guint32));        \n"
"                                                                              \n"
"  g_assert(buffer != NULL);                                                   \n"
"                                                                              \n"
"  if (!TIFFReadRGBAImage(p->tiff, p->width, p->height, buffer, 0))            \n"
"    {                                                                         \n"
"      g_message(\"unsupported layout, RGBA loader failed\");                  \n"
"      g_free(buffer);                                                         \n"
"      return -1;                                                              \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (row = 0; row < p->height; row++)                                       \n"
"    {                                                                         \n"
"      GeglRectangle line = { 0, p->height - row - 1, p->width, 1 };           \n"
"#if G_BYTE_ORDER != G_LITTLE_ENDIAN                                           \n"
"      guint row_start = row * p->width;                                       \n"
"      guint row_end = row * p->width + p->width;                              \n"
"      guint i;                                                                \n"
"                                                                              \n"
"      for (i = row_start; i < row_end; i++)                                   \n"
"        buffer[i] = GUINT32_TO_LE(buffer[i]);                                 \n"
"#endif                                                                        \n"
"                                                                              \n"
"      gegl_buffer_set(output, &line, 0, p->format,                            \n"
"                      ((guchar *) buffer) + (row * p->width * 4),             \n"
"                      GEGL_AUTO_ROWSTRIDE);                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free(buffer);                                                             \n"
"  return 0;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"load_contiguous(GeglOperation *operation,                                     \n"
"                GeglBuffer    *output)                                        \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  Priv *p = (Priv*) o->user_data;                                             \n"
"  guint32 tile_width = (guint32) p->width;                                    \n"
"  guint32 tile_height = 1;                                                    \n"
"  guchar *buffer;                                                             \n"
"  gint x, y;                                                                  \n"
"                                                                              \n"
"  g_return_val_if_fail(p->tiff != NULL, -1);                                  \n"
"                                                                              \n"
"  if (!TIFFIsTiled(p->tiff))                                                  \n"
"      buffer = g_try_new(guchar, TIFFScanlineSize(p->tiff));                  \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      TIFFGetField(p->tiff, TIFFTAG_TILEWIDTH, &tile_width);                  \n"
"      TIFFGetField(p->tiff, TIFFTAG_TILELENGTH, &tile_height);                \n"
"                                                                              \n"
"      buffer = g_try_new(guchar, TIFFTileSize(p->tiff));                      \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_assert(buffer != NULL);                                                   \n"
"                                                                              \n"
"  for (y = 0; y < p->height; y += tile_height)                                \n"
"    {                                                                         \n"
"      for (x = 0; x < p->width; x += tile_width)                              \n"
"        {                                                                     \n"
"          GeglRectangle tile = { x, y, tile_width, tile_height };             \n"
"                                                                              \n"
"          if (TIFFIsTiled(p->tiff))                                           \n"
"            TIFFReadTile(p->tiff, buffer, x, y, 0, 0);                        \n"
"          else                                                                \n"
"            TIFFReadScanline(p->tiff, buffer, y, 0);                          \n"
"                                                                              \n"
"          gegl_buffer_set(output, &tile, 0, p->format,                        \n"
"                          (guchar *) buffer,                                  \n"
"                          GEGL_AUTO_ROWSTRIDE);                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free(buffer);                                                             \n"
"  return 0;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"load_separated(GeglOperation *operation,                                      \n"
"               GeglBuffer    *output)                                         \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  Priv *p = (Priv*) o->user_data;                                             \n"
"  guint32 tile_width = (guint32) p->width;                                    \n"
"  guint32 tile_height = 1;                                                    \n"
"  gint output_bytes_per_pixel;                                                \n"
"  gint nb_components, offset = 0;                                             \n"
"  guchar *buffer;                                                             \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  g_return_val_if_fail(p->tiff != NULL, -1);                                  \n"
"                                                                              \n"
"  if (!TIFFIsTiled(p->tiff))                                                  \n"
"    buffer = g_try_new(guchar, TIFFScanlineSize(p->tiff));                    \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      TIFFGetField(p->tiff, TIFFTAG_TILEWIDTH, &tile_width);                  \n"
"      TIFFGetField(p->tiff, TIFFTAG_TILELENGTH, &tile_height);                \n"
"                                                                              \n"
"      buffer = g_try_new(guchar, TIFFTileSize(p->tiff));                      \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_assert(buffer != NULL);                                                   \n"
"                                                                              \n"
"  nb_components = babl_format_get_n_components(p->format);                    \n"
"  output_bytes_per_pixel = babl_format_get_bytes_per_pixel(p->format);        \n"
"                                                                              \n"
"  for (i = 0; i < nb_components; i++)                                         \n"
"    {                                                                         \n"
"      const Babl *plane_format;                                               \n"
"      const Babl *component_type;                                             \n"
"      gint plane_bytes_per_pixel;                                             \n"
"      gint x, y;                                                              \n"
"                                                                              \n"
"      component_type = babl_format_get_type(p->format, i);                    \n"
"                                                                              \n"
"      plane_format = babl_format_n(component_type, 1);                        \n"
"                                                                              \n"
"      plane_bytes_per_pixel = babl_format_get_bytes_per_pixel(plane_format);  \n"
"                                                                              \n"
"      for (y = 0; y < p->height; y += tile_height)                            \n"
"        {                                                                     \n"
"          for (x = 0; x < p->width; x += tile_width)                          \n"
"            {                                                                 \n"
"              GeglRectangle output_tile = { x, y, tile_width, tile_height };  \n"
"              GeglRectangle plane_tile = { 0, 0, tile_width, tile_height };   \n"
"              GeglBufferIterator *iterator;                                   \n"
"              GeglBuffer *linear;                                             \n"
"                                                                              \n"
"              if (TIFFIsTiled(p->tiff))                                       \n"
"                TIFFReadTile(p->tiff, buffer, x, y, 0, i);                    \n"
"              else                                                            \n"
"                TIFFReadScanline(p->tiff, buffer, y, i);                      \n"
"                                                                              \n"
"              linear = gegl_buffer_linear_new_from_data(buffer, plane_format, \n"
"                                                        &plane_tile,          \n"
"                                                        GEGL_AUTO_ROWSTRIDE,  \n"
"                                                        NULL, NULL);          \n"
"                                                                              \n"
"              iterator = gegl_buffer_iterator_new(linear, &plane_tile,        \n"
"                                                  0, NULL,                    \n"
"                                                  GEGL_ACCESS_READ,           \n"
"                                                  GEGL_ABYSS_NONE);           \n"
"                                                                              \n"
"              gegl_buffer_iterator_add(iterator, output, &output_tile,        \n"
"                                       0, p->format,                          \n"
"                                       GEGL_ACCESS_READWRITE,                 \n"
"                                       GEGL_ABYSS_NONE);                      \n"
"                                                                              \n"
"              while (gegl_buffer_iterator_next(iterator))                     \n"
"                {                                                             \n"
"                  guchar *plane_buffer = iterator->data[0];                   \n"
"                  guchar *output_buffer = iterator->data[1];                  \n"
"                  gint nb_pixels = iterator->length;                          \n"
"                                                                              \n"
"                  output_buffer += offset;                                    \n"
"                                                                              \n"
"                  while (nb_pixels--)                                         \n"
"                  {                                                           \n"
"                    memcpy(output_buffer, plane_buffer, plane_bytes_per_pixel);\n"
"                                                                              \n"
"                    output_buffer += output_bytes_per_pixel;                  \n"
"                    plane_buffer += plane_bytes_per_pixel;                    \n"
"                  }                                                           \n"
"                }                                                             \n"
"                                                                              \n"
"              g_object_unref(linear);                                         \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      offset += plane_bytes_per_pixel;                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free(buffer);                                                             \n"
"  return 0;                                                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare(GeglOperation *operation)                                             \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  Priv *p = (o->user_data) ? o->user_data : g_new0(Priv, 1);                  \n"
"  GError *error = NULL;                                                       \n"
"  GFile *file = NULL;                                                         \n"
"  gint directories;                                                           \n"
"                                                                              \n"
"  g_assert(p != NULL);                                                        \n"
"                                                                              \n"
"  if (p->file != NULL && (o->uri || o->path))                                 \n"
"    {                                                                         \n"
"      if (o->uri && strlen(o->uri) > 0)                                       \n"
"        file = g_file_new_for_uri(o->uri);                                    \n"
"      else if (o->path && strlen(o->path) > 0)                                \n"
"        file = g_file_new_for_path(o->path);                                  \n"
"      if (file != NULL)                                                       \n"
"        {                                                                     \n"
"          if (!g_file_equal(p->file, file))                                   \n"
"            cleanup(operation);                                               \n"
"          g_object_unref(file);                                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  o->user_data = (void*) p;                                                   \n"
"                                                                              \n"
"  if (p->stream == NULL)                                                      \n"
"    {                                                                         \n"
"      p->stream = gegl_gio_open_input_stream(o->uri, o->path, &p->file, &error);\n"
"      if (p->stream != NULL && p->file != NULL)                               \n"
"        p->can_seek = g_seekable_can_seek(G_SEEKABLE(p->stream));             \n"
"      if (p->stream == NULL)                                                  \n"
"        {                                                                     \n"
"          g_warning(\"%s\", error->message);                                  \n"
"          g_error_free(error);                                                \n"
"          cleanup(operation);                                                 \n"
"          return;                                                             \n"
"        }                                                                     \n"
"                                                                              \n"
"      TIFFSetErrorHandler(error_handler);                                     \n"
"      TIFFSetWarningHandler(warning_handler);                                 \n"
"                                                                              \n"
"      p->tiff = TIFFClientOpen(\"GEGL-tiff-load\", \"r\", (thandle_t) p,      \n"
"                               read_from_stream, write_to_stream,             \n"
"                               seek_in_stream, close_stream,                  \n"
"                               get_file_size, NULL, NULL);                    \n"
"      if (p->tiff == NULL)                                                    \n"
"        {                                                                     \n"
"          if (o->uri != NULL && strlen(o->uri) > 0)                           \n"
"            g_warning(\"failed to open TIFF from %s\", o->uri);               \n"
"          else                                                                \n"
"            g_warning(\"failed to open TIFF from %s\", o->path);              \n"
"          cleanup(operation);                                                 \n"
"          return;                                                             \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (o->directory != p->directory)                                           \n"
"    {                                                                         \n"
"      directories = TIFFNumberOfDirectories(p->tiff);                         \n"
"      if (o->directory > 1 && o->directory <= directories)                    \n"
"        TIFFSetDirectory(p->tiff, o->directory - 1);                          \n"
"                                                                              \n"
"      if (query_tiff(operation))                                              \n"
"        {                                                                     \n"
"          g_warning(\"could not query TIFF file\");                           \n"
"          cleanup(operation);                                                 \n"
"          return;                                                             \n"
"        }                                                                     \n"
"                                                                              \n"
"        p->directory = o->directory;                                          \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_operation_set_format(operation, \"output\", p->format);                \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_bounding_box(GeglOperation *operation)                                    \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  GeglRectangle result = { 0, 0, 0, 0 };                                      \n"
"  Priv *p = (Priv*) o->user_data;                                             \n"
"                                                                              \n"
"  if (p->tiff != NULL)                                                        \n"
"    {                                                                         \n"
"      result.width = p->width;                                                \n"
"      result.height = p->height;                                              \n"
"    }                                                                         \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process(GeglOperation *operation,                                             \n"
"        GeglBuffer *output,                                                   \n"
"        const GeglRectangle *result,                                          \n"
"        gint level)                                                           \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(operation);                             \n"
"  Priv *p = (Priv*) o->user_data;                                             \n"
"                                                                              \n"
"  if (p->tiff != NULL)                                                        \n"
"    {                                                                         \n"
"      switch (p->mode)                                                        \n"
"      {                                                                       \n"
"      case TIFF_LOADING_RGBA:                                                 \n"
"        if (!load_RGBA(operation, output))                                    \n"
"          return TRUE;                                                        \n"
"        break;                                                                \n"
"                                                                              \n"
"      case TIFF_LOADING_CONTIGUOUS:                                           \n"
"        if (!load_contiguous(operation, output))                              \n"
"          return TRUE;                                                        \n"
"        break;                                                                \n"
"                                                                              \n"
"      case TIFF_LOADING_SEPARATED:                                            \n"
"        if (!load_separated(operation, output))                               \n"
"          return TRUE;                                                        \n"
"        break;                                                                \n"
"                                                                              \n"
"      default:                                                                \n"
"        break;                                                                \n"
"      }                                                                       \n"
"    }                                                                         \n"
"                                                                              \n"
"  return FALSE;                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_cached_region(GeglOperation       *operation,                             \n"
"                  const GeglRectangle *roi)                                   \n"
"{                                                                             \n"
"  return get_bounding_box(operation);                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"finalize(GObject *object)                                                     \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES(object);                                \n"
"                                                                              \n"
"  if (o->user_data != NULL)                                                   \n"
"    {                                                                         \n"
"      cleanup(GEGL_OPERATION(object));                                        \n"
"      if (o->user_data != NULL)                                               \n"
"        g_free(o->user_data);                                                 \n"
"      o->user_data = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  G_OBJECT_CLASS(gegl_op_parent_class)->finalize(object);                     \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init(GeglOpClass *klass)                                        \n"
"{                                                                             \n"
"  GeglOperationClass *operation_class;                                        \n"
"  GeglOperationSourceClass *source_class;                                     \n"
"                                                                              \n"
"  G_OBJECT_CLASS(klass)->finalize = finalize;                                 \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS(klass);                              \n"
"  source_class = GEGL_OPERATION_SOURCE_CLASS(klass);                          \n"
"                                                                              \n"
"  source_class->process = process;                                            \n"
"  operation_class->prepare = prepare;                                         \n"
"  operation_class->get_bounding_box = get_bounding_box;                       \n"
"  operation_class->get_cached_region = get_cached_region;                     \n"
"                                                                              \n"
"  gegl_operation_class_set_keys(operation_class,                              \n"
"    \"name\",          \"gegl:tiff-load\",                                    \n"
"    \"title\",       _(\"TIFF File Loader\"),                                 \n"
"    \"categories\",    \"hidden\",                                            \n"
"    \"description\", _(\"TIFF image loader using libtiff\"),                  \n"
"    NULL);                                                                    \n"
"                                                                              \n"
"  gegl_operation_handlers_register_loader(                                    \n"
"    \"image/tiff\", \"gegl:tiff-load\");                                      \n"
"  gegl_operation_handlers_register_loader(                                    \n"
"    \"image/x-tiff-multipage\", \"gegl:tiff-load\");                          \n"
"  gegl_operation_handlers_register_loader(                                    \n"
"    \".tiff\", \"gegl:tiff-load\");                                           \n"
"  gegl_operation_handlers_register_loader(                                    \n"
"    \".tif\", \"gegl:tiff-load\");                                            \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
