/********************************************************************************

   Fotoxx - edit photos and manage collections

   Copyright 2007-2023 Michael Cornelison
   source code URL: https://kornelix.net
   contact: mkornelix@gmail.com

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version. See https://www.gnu.org/licenses

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
   See the GNU General Public License for more details.

*********************************************************************************

   Fotoxx image edit - file menu and file functions

   m_new_session           start a new parallel session of Fotoxx
   m_open_file             file open dialog
   m_cycle2files           cycle 2 previous image files
   m_cycle3files           cycle 3 previous image files
   m_view360               view a 360 degree panorama image
   m_rename                rename current image file or clicked thumbnail
   m_permissions           show and change file permissions
   m_change_alpha          change image file alpha channel
   f_open                  open and display an image file
   f_open_saved            open the last image file saved
   f_remove                remove file and fix index, gallery, thumbs, albums
   f_preload               preload image files ahead of need
   x_prev_next             open prev/next file in current or prev/next gallery
   m_prev                  menu function - open previous
   m_next                  menu function - open next
   m_prev_next             menu button function - open previous or next file
   m_zoom_menu             menu button function - zoom image/thumb (F/G view)
   m_blank_image           create a new monocolor image
   create_blank_file       callable function to create a new monocolor image
   play_gif                play current GIF file animation

   m_copy_move             copy or move an image file to a new location
   m_copyto_desktop        copy an image file to the desktop
   m_copyto_clip           copy clicked file or current file to the clipboard
   m_delete_trash          delete or trash an image file
   m_convert_adobe         convert Adobe file into one or more jpeg files
   m_wallpaper             set current image file as wallpaper

   m_print                 print an image file
   m_print_calibrated      print an image file with calibrated colors
   m_quit                  menu quit
   quitxx                  callable quit
   m_uninstall             uninstall fotoxx
   m_help                  help menu: user guide, change log, about

   m_file_save             save a (modified) image file to disk
   m_file_save_replace     save file (replace) for KB shortcut
   m_file_save_version     save file (new version) for KB shortcut
   f_save                  save an image file to disk (replace, new version, new file)
   f_save_as               dialog to save an image file with a designated file name

   f_realpath              get real path for filename having symlinks
   file_rootname           get root file name /.../filename
   file_basename           get base file name /.../filename.ext
   file_all_versions       get base file and all versions /.../filename.vNN.ext
   file_new_version        get next available version /.../filename.vNN.ext
   file_newest_version     get newest file version or base file name if none
   file_prior_version      get prior version or base file name for given file

   find_imagefiles         find all image files under a given folder path

   regfile                 test if file exists, is regular file
   dirfile                 test if file exists, is a directory/folder
   hasperm                 test if file exists, has read or write permission

   set_permissions         dialog to show and set file permissions
   conv_permissions        conv. permission formats: external <> mode_t

*********************************************************************************/

#define EX extern                                                                //  disable extern declarations
#include "fotoxx.h"                                                              //  (variables in fotoxx.h are refs)

using namespace zfuncs;


/********************************************************************************/

//  start a new parallel session of fotoxx
//  new window is slightly down and right from old window
//  args = optional command line args
//  current file is appended if it exists

void m_new_session(GtkWidget *, ch *)
{
   ch       *pp;
   int      cc;

   F1_help_topic = "new session"; 
   Plog(1,"m_new_session \n");

   if (curr_file) {
      cc = strlen(curr_file);
      pp = (ch *) zmalloc(cc+100,"new-session");
      repl_1str(curr_file,pp+1,"\"","\\\"");                                     //  replace embedded " with \"
      pp[0] = '"';
      cc = strlen(pp);                                                           //  add " at both ends
      pp[cc] = '"';
      pp[cc+1] = 0;
      new_session(pp);
      zfree(pp);
   }

   else new_session(0);
   return;
}


//  callable new session function

void new_session(ch *args)
{
   int      cc, xx, yy, ww, hh;
   ch       progexe[300];
                                                                                 //  do before new session:
   zdialog_inputs("save");                                                       //  save dialog inputs
   zdialog_geometry("save");                                                     //  save dialogs position/size
   gallery_memory("save");                                                       //  save recent gallery positions
   save_params();                                                                //  save state for next session

   gtk_window_get_position(MWIN,&xx,&yy);                                        //  get window position and size
   gtk_window_get_size(MWIN,&ww,&hh);
   xx += 100;                                                                    //  shift down and right
   yy += 100;

   cc = readlink("/proc/self/exe",progexe,300);                                  //  get own program path
   if (cc <= 0) {
      zmessageACK(Mwin,"cannot get /proc/self/exe");
      return;
   }
   progexe[cc] = 0;
   
   if (! args) args = "";

   zshell("log ack","%s -c %d %d %d %d %s &",progexe,xx,yy,ww,hh,args);
   return;
}


/********************************************************************************/

//  find and open an image file using a file open dialog

void m_open_file(GtkWidget *, ch *)
{
   int      err;

   F1_help_topic = "open image file";
   Plog(1,"m_open_file \n");
   if (Fblock(0,"blocked edits")) return;                                        //  check nothing pending

   err = f_open(0,0,0,1,0);
   if (err) return;

   m_viewmode(0,"F");                                                            //  file view mode
   return;
}


/********************************************************************************/

//  open the previous file opened (not the same as toolbar [prev] button)
//  repeated use will alternate the two most recent files

void m_cycle2files(GtkWidget *, ch *)
{
   FILE     *fid;
   ch       *file, buff[XFCC];
   int      Nth = 0, err;
   float    gzoom;

   F1_help_topic = "cycle 2";

   Plog(1,"m_cycle2files \n");

   if (Fblock(0,"blocked edits")) return;                                        //  check nothing pending

   m_viewmode(0,"F");
   if (! Cstate) return;
   gzoom = Cstate->fzoom;

   fid = fopen(recentfiles_file,"r");
   if (! fid) return;

   file = fgets_trim(buff,XFCC,fid,1);                                           //  skip over first most recent file

   while (true)
   {
      file = fgets_trim(buff,XFCC,fid,1);                                        //  find next most recent file
      if (! file) break;
      err = f_open(file,Nth,0,0,0);
      if (! err) break;
   }

   fclose(fid);

   Cstate->fzoom = gzoom;                                                        //  keep same image scale
   Fpaint2();
   return;
}


/********************************************************************************/

//  open the 2nd previous file opened
//  repeated use will alternate the 3 most recent files

void m_cycle3files(GtkWidget *, ch *) 
{
   FILE     *fid;
   ch       *file, buff[XFCC];
   int      Nth = 0, err;
   float    gzoom;

   F1_help_topic = "cycle 3";

   Plog(1,"m_cycle3files \n");

   if (Fblock(0,"blocked edits")) return;                                        //  check nothing pending

   m_viewmode(0,"F");
   if (! Cstate) return;
   gzoom = Cstate->fzoom;

   fid = fopen(recentfiles_file,"r");
   if (! fid) return;

   file = fgets_trim(buff,XFCC,fid,1);                                           //  skip over 2 most recent files
   file = fgets_trim(buff,XFCC,fid,1);

   while (true)
   {
      file = fgets_trim(buff,XFCC,fid,1);                                        //  find next most recent file
      if (! file) break;
      err = f_open(file,Nth,0,0,0);
      if (! err) break;
   }

   fclose(fid);

   Cstate->fzoom = gzoom;                                                        //  keep same image scale
   Fpaint2();
   return;
}


/********************************************************************************/

//   view a 360 degree panorama image
//   center of view direction can be any angle 0-360 degrees
//   image width is assumed to correspond to a 360 degree view

namespace view360 
{
   void   show();
   void * show_thread(void *arg);
   void   mousefunc();
   void   KB_func(int key);
   void   quit();

   PXB      *filepxb, *viewpxb;
   PIXBUF   *viewpixbuf;
   ch       *filename = 0;
   int      Frecalc;
   int      fww, fhh;                                                            //  file dimensions
   int      dww, dhh;                                                            //  window dimensions
   int      pdww, pdhh;
   int      cx, cy;
   float    *Fx, *Fy, zoom;
   zdialog  *zd;
}


//  menu function

void m_view360(GtkWidget *, ch *)
{
   using namespace view360;
   
   F1_help_topic = "view 360° pano";

   Plog(1,"m_view360 \n");

   if (Fblock("view360","block edits")) return;                                  //  check pend, block

   if (clicked_file) {                                                           //  use clicked file if present
      filename = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                           //  else use current file
      filename = zstrdup(curr_file,"view360");
   else goto retx;

   filepxb = PXB_load(filename,1);
   zfree(filename);

   if (! filepxb) goto retx;
   
   zd = zmessage_post_bold(Mwin,"20/20",10,
                "Mouse drag: pan image 360° \n"
                "L/R mouse click: zoom in/out \n"
                "Escape: quit panorama view");

   Vmenu_block(1);                                                               //  block menus   
   Fview360 = 1;                                                                 //  flag for main window KBpress() 

   fww = filepxb->ww;
   fhh = filepxb->hh;
   cx = fww/2;                                                                   //  initial view center = middle
   cy = fhh/2;
   zoom = 1.0;                                                                   //  initial zoom
   Fx = Fy = 0;                                                                  //  no allocated constants
   pdww = pdhh = 0;
   Frecalc = 1;
   
   m_viewmode(0,"F");
   gdk_window_freeze_updates(gdkwin);
   takeMouse(mousefunc,0);
   show();

retx:
   Fblock("view360",0);
   return;
}


//  show current view region on window

void view360::show()
{
   using namespace view360;

   int      ii, px, py, dx, dy;
   float    sx, sy, R, C, Tx, Ty;

   if (! Fview360) return;

   dww = gdk_window_get_width(gdkwin);                                           //  drawing window size
   dhh = gdk_window_get_height(gdkwin);
   if (dww < 20 || dhh < 20) return;                                             //  too small
   
   if (dww != pdww || dhh != pdhh) {
      pdww = dww;
      pdhh = dhh;
      if (Fx) zfree(Fx);
      if (Fy) zfree(Fy);
      Fx = (float *) zmalloc(dww * dhh * sizeof(float),"view360");
      Fy = (float *) zmalloc(dww * dhh * sizeof(float),"view360");
      Frecalc = 1;
   }

   viewpxb = PXB_make(dww,dhh,3);
   viewpixbuf = viewpxb->pixbuf;
   gdk_pixbuf_fill(viewpixbuf,0);

   if (Frecalc)
   {
      Frecalc = 0;

      for (py = 0; py < dhh; py++)                                               //  loop window pixels
      for (px = 0; px < dww; px++)
      {
         dx = px - dww/2;                                                        //  distance from center
         dy = py - dhh/2;
         dx = dx / zoom;                                                         //  scale for zoom
         dy = dy / zoom;   

         ii = py * dww + px;

         sx = 6.28 * dx / fww;                                                   //  scale (dx,dy) so that fww/4
         sy = 6.28 * dy / fww;                                                   //    (90 deg.) is PI/2

         if (sx > 1.0 || sx < -1.0 || sy > 0.7 || sy < -0.7) {                   //  limit +-90 and +-63 deg. view
            Fx[ii] = Fy[ii] = 0;
            continue;
         }

         R = sqrtf(sx*sx + sy*sy);                                               //  conv. (dx,dy) in rectilinear proj.
         C = atanf(R);                                                           //     to spherical coordinates
         Tx = atanf(sx);
         Ty = asinf(sy/R * sinf(C));
         if (R == 0.0) Tx = Ty = 0;                                              //  avoid Ty = nan
         
         sx = 0.25 * fww * Tx;                                                   //  conv. spherical coordinate (Ty,Tx) 
         sy = 0.25 * fww * Ty * cosf(Tx);                                        //    to flat coordinates (sx,sy)

         Fx[ii] = sx;
         Fy[ii] = sy;
      }
   }
   
   do_wthreads(show_thread,Nsmp);

   cairo_t *cr = draw_context_create(gdkwin,draw_context);
   gdk_cairo_set_source_pixbuf(cr,viewpxb->pixbuf,0,0);
   cairo_paint(cr);
   draw_context_destroy(draw_context);

   PXB_free(viewpxb);
   return;
}


//  thread function to generate the image

void * view360::show_thread(void *arg)
{
   using namespace view360;

   int      index = *((int *) (arg));
   int      ii, stat, px, py;
   float    sx, sy;
   uint8    pixs[4], *pixss, *pixp;
   
   for (py = index; py < dhh; py += Nsmp)                                        //  loop window pixels
   for (px = 0; px < dww; px++)
   {
      ii = py * dww + px;
      sx = Fx[ii];
      sy = Fy[ii];
      if (sx == 0 && sy == 0) continue;
      if (sx == NAN || sy == NAN) continue;

      sx += cx;                                                                  //  add back center
      sy += cy;

      if (sy < 0) continue;                                                      //  off the image
      if (sy >= fhh) continue;

      if (sx < 0) sx = fww + sx;                                                 //  handle x wrap-around
      if (sx >= fww) sx = sx - fww;
      
      pixp = PXBpix(viewpxb,px,py);                                              //  dest. pixel

      stat = vpixel(filepxb,sx,sy,pixs);                                         //  source pixel interpolated
      if (stat) memcpy(pixp,pixs,3);                                             //  fails at image left/right edge
      else {
         pixss = PXBpix(filepxb,int(sx),int(sy));                                //  use no interpolation
         memcpy(pixp,pixss,3);
      }
   }

   return 0;
}


//  mouse function - move center of view with mouse drag

void view360::mousefunc()
{
   using namespace view360;
   
   int      Fshow = 0;
   
   if (Mxdrag || Mydrag) {                                                       //  change center of view
      cx += 2 * Mwdragx;                                                         //  horizontal center
      if (cx < 0) cx = fww + cx;
      if (cx >= fww) cx = cx - fww;
//    cy += Mwdragy;                                                             //  vertical center
      Mxdrag = Mydrag = 0;
      Fshow = 1;
   }
   
   if (LMclick) {                                                                //  zoom-in
      LMclick = 0;
      zoom = zoom * 1.414;
      if (zoom > 1.99) zoom = 2;
      Fshow = 1;
      Frecalc = 1;
   }
   
   if (RMclick) {                                                                //  zoom-out
      RMclick = 0;
      zoom = zoom / 1.414;
      if (zoom < 0.3) zoom = 0.3;
      Fshow = 1;
      Frecalc = 1;
   }

   if (Fshow) {
      if (zd) zdialog_free(zd);
      zd = 0;
      show();
   }

   return;
}


//  keyboard function

void view360::KB_func(int key)
{
   using namespace view360;

   int         ii;   
   
   if (zd) zdialog_free(zd);
   zd = 0;   

   if (key == GDK_KEY_Escape) quit();
   
   for (ii = 0; ii < Nkbsu; ii++) {                                              //  test if KB key is "Quit" shortcut
      if (toupper(key) == kbsutab[ii].key[0]) {
         if (strmatchcase(kbsutab[ii].menu,"Quit")) quit();
         else break;
      }
   }

   if (key == GDK_KEY_Left) {                                                    //  left/right arrow keys pan image
      cx -= 20;
      if (cx < 0) cx = fww + cx;
   }

   if (key == GDK_KEY_Right) {
      cx += 20;
      if (cx >= fww) cx = cx - fww;
   }

   show();
   return;
}


//  quit and free resources

void view360::quit()
{
   if (! Fview360) return;
   freeMouse();
   PXB_free(filepxb);
   PXB_free(viewpxb);
   if (Fx) zfree(Fx);
   if (Fy) zfree(Fy);
   Fview360 = 0;
   Vmenu_block(0);
   gdk_window_thaw_updates(gdkwin);
   Fpaint2();
}


/********************************************************************************/

//  rename menu function
//  activate rename dialog, stuff data from current or clicked file
//  dialog remains active when new file is opened

ch     rename_old[200] = "";                                                     //  base names only
ch     rename_new[200] = "";
ch     rename_prev[200] = "";
ch     *rename_file = 0;

void m_rename(GtkWidget *, ch *menu)
{
   int rename_dialog_event(zdialog *zd, ch *event);

   ch     *pdir, *pfile, *pext;

   F1_help_topic = "rename";

   Plog(1,"m_rename \n");

   if (rename_file) zfree(rename_file);
   rename_file = 0;

   if (clicked_file) {                                                           //  use clicked file if present
      rename_file = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                           //  else current file
      rename_file = zstrdup(curr_file,"rename");
   else return;

   if (FGWM != 'F' && FGWM != 'G') m_viewmode(0,"F");

/***
       ______________________________________
      |         Rename Image File            |
      |                                      |
      | Old Name  [_______________________]  |
      | New Name  [_______________________]  |
      |           [previous name] [Add 1]    |
      |                                      |
      | [x] keep this dialog open            |
      |                                      |
      |                   [apply] [cancel]   |
      |______________________________________|

***/

   if (! zd_rename)                                                              //  restart dialog
   {
      zd_rename = zdialog_new("Rename Image File",Mwin,"Apply","Cancel",null);
      zdialog *zd = zd_rename;

      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
      zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|expand");

      zdialog_add_widget(zd,"label","Lold","vb1","Old Name");
      zdialog_add_widget(zd,"label","Lnew","vb1","New Name");
      zdialog_add_widget(zd,"label","space","vb1");

      zdialog_add_widget(zd,"hbox","hb2","vb2");
      zdialog_add_widget(zd,"label","oldname","hb2");
      zdialog_add_widget(zd,"label","space","hb2",0,"expand");

      zdialog_add_widget(zd,"zentry","newname","vb2",0,"size=20");
      zdialog_add_widget(zd,"hbox","hb3","vb2",0,"space=3");
      zdialog_add_widget(zd,"button","prev","hb3","previous name");
      zdialog_add_widget(zd,"button","Badd1","hb3","Add 1","space=8");

      zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=3");
      zdialog_add_widget(zd,"check","keepopen","hb4","keep this dialog open","space=3");

      zdialog_restore_inputs(zd);
      zdialog_run(zd,rename_dialog_event,"parent");                              //  run dialog
   }

   zdialog *zd = zd_rename;
   parsefile(rename_file,&pdir,&pfile,&pext);
   strncpy0(rename_old,pfile,199);
   strncpy0(rename_new,pfile,199);
   zdialog_stuff(zd,"oldname",rename_old);                                       //  current file name
   zdialog_stuff(zd,"newname",rename_new);                                       //  entered file name (same)

   return;
}


//  dialog event and completion callback function

int rename_dialog_event(zdialog *zd, ch *event)
{
   ch       *pp, *pdir, *pfile, *pext, suffix[16];
   ch       *newfile = 0, *nextfile = 0, namever[200];
   int      nseq, digits, ccp, ccn, ccx, err, Fkeep;
   
   if (strmatch(event,"prev"))                                                   //  previous name >> new name
   {
      if (! *rename_prev) return 1;
      if (! rename_file) return 1;

      pp = strrchr(rename_prev,'.');
      if (pp && strlen(pp) == 4 && pp[1] == 'v'                                  //  look for file version .vNN
         && pp[2] >= '0' && pp[2] <= '9'                                         //    and remove if found
         && pp[3] >= '0' && pp[3] <= '9') *pp = 0;
      zdialog_stuff(zd,"newname",rename_prev);                                   //  stuff prev rename name into dialog

      parsefile(rename_file,&pdir,&pfile,&pext);                                 //  curr. file to be renamed
      pp = strrchr(pfile,'.');                                                   //  look for file version .vNN
      if (pp && strlen(pp) == 4 && pp[1] == 'v'
         && pp[2] >= '0' && pp[2] <= '9'
         && pp[3] >= '0' && pp[3] <= '9') 
      {
         *namever = 0;                                                           //  prev rename name + curr file version
         strncatv(namever,200,rename_prev,pp,null);
         zdialog_stuff(zd,"newname",namever);                                    //  stuff into dialog
      }
   }     

   if (strmatch(event,"Badd1"))                                                  //  increment sequence number
   {
      zdialog_fetch(zd,"newname",rename_new,188);                                //  get entered filename
      pp = strrchr(rename_new,'.');                                              //  find .ext
      if (pp && strlen(pp) < 8) {
         if (strmatchN(pp-4,".v",2) &&                                           //  find .vNN.ext
            (pp[2] >= '0' && pp[2] <= '9') && 
            (pp[3] >= '0' && pp[3] <= '9')) pp -= 4;
         strncpy0(suffix,pp,16);                                                 //  save .ext or .vNN.ext
      }
      else {
         pp = rename_new + strlen(rename_new);
         *suffix = 0;
      }

      digits = 0;
      while (pp[-1] >= '0' && pp[-1] <= '9') {
         pp--;                                                                   //  look for NNN in filenameNNN
         digits++;
      }
      nseq = 1;
      if (digits) nseq += atoi(pp);                                              //  NNN + 1
      if (nseq > 9999) nseq = 1;
      if (digits < 1) digits = 1;
      if (nseq > 9 && digits < 2) digits = 2;
      if (nseq > 99 && digits < 3) digits = 3;
      if (nseq > 999 && digits < 4) digits = 4;
      sprintf(pp,"%0*d",digits,nseq);                                            //  replace NNN with NNN + 1
      strncat(rename_new,suffix,199);                                            //  append .ext or .vNN.ext
      zdialog_stuff(zd,"newname",rename_new);
   }
   
   if (zd->zstat == 0) goto CLEANUP;                                             //  not finished
   if (zd->zstat != 1) goto KILL;                                                //  canceled

   zd->zstat = 0;                                                                //  apply - keep dialog active
   
   if (! rename_file) goto CLEANUP;

   parsefile(rename_file,&pdir,&pfile,&pext);                                    //  existing /folders/file.ext

   zdialog_fetch(zd,"newname",rename_new,194);                                   //  new file name from user

   ccp = strlen(pdir);                                                           //  length of /folders/
   ccn = strlen(rename_new);                                                     //  length of file
   if (pext) ccx = strlen(pext);                                                 //  length of .ext
   else ccx = 0;

   newfile = (ch *) zmalloc(ccp + ccn + ccx + 1,"rename");                       //  put it all together
   strncpy(newfile,rename_file,ccp);                                             //   /folders.../newfilename.ext
   strcpy(newfile+ccp,rename_new);
   if (ccx) strcpy(newfile+ccp+ccn,pext);

   if (regfile(newfile)) {                                                       //  check if new name exists
      zmessageACK(Mwin,"output file exists");
      zfree(newfile);
      newfile = 0;
      goto CLEANUP;
   }

   if (FGWM == 'F')
      nextfile = gallery(0,"get",curr_file_posn+1);                              //  save next file, before rename         23.3
   else nextfile = 0;

   if (Fblock("rename","block edits")) goto CLEANUP;                             //  check pend, block

   err = rename(rename_file,newfile);                                            //  rename the file
   if (err) {
      zmessageACK(Mwin,"file error: %s",strerror(errno));
      Fblock("rename",0);
      goto KILL;
   }
   
   album_purge_replace("ALL",rename_file,newfile);                               //  replace name in albums

   load_filemeta(newfile);                                                       //  add new file to image index
   update_image_index(newfile);
   delete_image_index(rename_file);                                              //  delete old file in image index

   delete_thumbfile(rename_file);                                                //  remove old thumbnail
   update_thumbfile(newfile);                                                    //  add new thumbnail 

   add_recent_file(newfile);                                                     //  first in recent files list
   
   strncpy0(rename_prev,rename_new,199);                                         //  save new name to previous name

   if (curr_file && strmatch(rename_file,curr_file))                             //  current file exists no more
      free_resources();
   
   if (navi::gallerytype == FOLDER) {                                            //  refresh gallery list
      gallery(0,"init",0);
      gallery(0,"sort",-2);                                                      //  recall sort and position
      gallery(0,"paint",-1);
   }
   
   Fblock("rename",0);

   zdialog_fetch(zd,"keepopen",Fkeep);
   if (Fkeep) {                                                                  //  keep dialog open
      if (FGWM == 'F' && nextfile) {
         f_open(nextfile);                                                       //  will call m_rename()
         gtk_window_present(MWIN);                                               //  keep focus on main window
      }
      if (newfile) zfree(newfile);
      newfile = 0;
      goto CLEANUP;
   }
   else {                                                                        //  close dialog
      f_open(newfile);                                                           //  open renamed file
      goto KILL;
   }

KILL:
   zdialog_free(zd);                                                             //  kill dialog
   zd_rename = 0;

CLEANUP:
   if (newfile) zfree(newfile);
   if (nextfile) zfree(nextfile);
   return 1;
}


/********************************************************************************/

//  show and change file permissions

ch  *permissions_file = 0;

void m_permissions(GtkWidget *, ch *menu)
{
   int permissions_dialog_event(zdialog *zd, ch *event);

   ch       permissions[100];
   ch       *pp;
   STATB    statB;
   int      ftype;

   F1_help_topic = "permissions";

   Plog(1,"m_permissions \n");

   if (FGWM != 'F' && FGWM != 'G') return;

   if (permissions_file) zfree(permissions_file);
   permissions_file = 0;

   if (clicked_file) {                                                           //  use clicked file if present
      permissions_file = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                           //  else current file
      permissions_file = zstrdup(curr_file,"permissions");

   if (permissions_file) {
      ftype = image_file_type(permissions_file);
      if (ftype != IMAGE && ftype != RAW && ftype != VIDEO) {                    //  must be image or RAW or VIDEO 
         zfree(permissions_file);
         permissions_file = 0;
      }
   }
   
   if (! permissions_file) return;

/***
       ______________________________________
      |       File Permissions               |
      |                                      |
      | File: [____________________________] |        file name
      |                                      |  
      | owner [read+write |v]                |
      | group [read only  |v]                |        combo box entries: read+write, read only, no access
      | other [no access  |v]                |
      |                                      |
      | [x] keep this dialog open            |
      |                                      |
      |                     [apply] [cancel] |
      |______________________________________|

***/

   if (! zd_permissions)                                                         //  restart dialog
   {
      zd_permissions = zdialog_new("File Permissions",Mwin,"Apply","Cancel",null);
      zdialog *zd = zd_permissions;
      
      zdialog_add_widget(zd,"hbox","hbfile","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labfile","hbfile","File:","space=3");
      zdialog_add_widget(zd,"label","filename","hbfile",0,"space=3");
      
      zdialog_add_widget(zd,"hbox","hbperm","dialog",0,"space=2");
      zdialog_add_widget(zd,"label","space","hbperm",0,"space=3");
      zdialog_add_widget(zd,"vbox","vb1","hbperm",0,"homog");
      zdialog_add_widget(zd,"label","space","hbperm",0,"space=6");
      zdialog_add_widget(zd,"vbox","vb2","hbperm",0,"homog");
      
      zdialog_add_widget(zd,"label","labown","vb1","owner","space=3");
      zdialog_add_widget(zd,"combo","permown","vb2",0,"space=3");
      zdialog_stuff(zd,"permown","read+write");
      zdialog_stuff(zd,"permown","read only");
      zdialog_stuff(zd,"permown","no access");

      zdialog_add_widget(zd,"label","labgrp","vb1","group","space=3");
      zdialog_add_widget(zd,"combo","permgrp","vb2",0,"space=3");
      zdialog_stuff(zd,"permgrp","read+write");
      zdialog_stuff(zd,"permgrp","read only");
      zdialog_stuff(zd,"permgrp","no access");

      zdialog_add_widget(zd,"label","laboth","vb1","other","space=3");
      zdialog_add_widget(zd,"combo","permoth","vb2",0,"space=3");
      zdialog_stuff(zd,"permoth","read+write");
      zdialog_stuff(zd,"permoth","read only");
      zdialog_stuff(zd,"permoth","no access");

      zdialog_add_widget(zd,"hbox","hbkeep","dialog",0,"space=3");
      zdialog_add_widget(zd,"check","keepopen","hbkeep","keep this dialog open","space=3");

      zdialog_resize(zd,280,0);
      zdialog_run(zd,permissions_dialog_event,"parent");                         //  run dialog
   }

   zdialog *zd = zd_permissions;
   
   pp = strrchr(permissions_file,'/');                                           //  stuff file name into dialog
   if (pp) pp++;
   else pp = permissions_file;
   zdialog_stuff(zd,"filename",pp);
   
   stat(permissions_file,&statB);                                                //  stuff curr. permissions
   conv_permissions(statB.st_mode,permissions);

   pp = substring(permissions,',',1);
   if (pp) zdialog_stuff(zd,"permown",pp);

   pp = substring(permissions,',',2);
   if (pp) zdialog_stuff(zd,"permgrp",pp);

   pp = substring(permissions,',',3);
   if (pp) zdialog_stuff(zd,"permoth",pp);

   return;
}


//  dialog event and completion callback function

int permissions_dialog_event(zdialog *zd, ch *event)
{
   int         Fkeep;
   mode_t      mode;
   ch          permown[20], permgrp[20], permoth[20];
   ch          permissions[100];
   
   if (zd->zstat == 0) return 1;                                                 //  wait for dialog finished
   if (zd->zstat != 1) goto KILL;                                                //  [cancel] or [x]
   
   zdialog_fetch(zd,"permown",permown,20);                                       //  [apply]
   zdialog_fetch(zd,"permgrp",permgrp,20);                                       //  construct permissions string
   zdialog_fetch(zd,"permoth",permoth,20);                                       //  e.g. "read+write, read only, no access"
   snprintf(permissions,100,"%s, %s, %s",permown,permgrp,permoth);

   conv_permissions(permissions,mode);                                           //  convert to mode_t
   chmod(permissions_file,mode);                                                 //  set file permissions

   zdialog_fetch(zd,"keepopen",Fkeep);
   if (! Fkeep) goto KILL;
   zd->zstat = 0;                                                                //  keep dialog active
   gtk_window_present(MWIN);                                                     //  keep focus on main window 
   return 1;

KILL:
   if (permissions_file) zfree(permissions_file);                                //  free memory
   permissions_file = 0;
   zdialog_free(zd);                                                             //  kill dialog
   zd_permissions = 0;
   return 1;
}


/********************************************************************************

  Add or remove alpha channel.
  Add: alpha channel is added with value 255.
  Remove: RGB pixels are reduced according to alpha channel value:
    RGBout = RGBin * (alpha/255)
    Transparent margins (e.g. panorama images) will become black.

***/


void m_change_alpha(GtkWidget *, ch *)                                           //  23.3
{
   int change_alpha_dialog_event(zdialog *zd, ch *event);
   
   int      ftype;
   ch       *pp;
   zdialog  *zd;
   
   F1_help_topic = "change alpha";
   Plog(1,"m_change_alpha \n");
   
   if (FGWM != 'F') return;
   if (! curr_file) return;
   
   ftype = image_file_type(curr_file);
   if (ftype != IMAGE) return;

   pp = strrchr(curr_file,'.');
   if (pp && strstr(".jpg .JPG .jpeg .JPEG",pp)) {
      zmessageACK(Mwin,"JPEG images have no alpha channel");
      return;
   }

   E0pxm = PXM_load(curr_file,1);                                                //  reload file with full bit depth
   if (! E0pxm) return;

   if (Fblock("change alpha","block edits")) return;                             //  check for blocking function
   

/***
          ___________________________________
         |      Change Alpha Channel         |
         |                                   |
         | File: [_________________________] |
         | alpha channel: (not) present      |
         |                                   |
         |               [Add] [Remove] [OK] |
         |___________________________________|

***/

   zd = zdialog_new("Change Alpha Channel",Mwin,"Add","Remove","OK",null);
   if (! zd) return;
   
   zdialog_add_widget(zd,"hbox","hbfile","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labfile","hbfile","File:","space=3");
   zdialog_add_widget(zd,"label","filename","hbfile",0,"space=3");

   zdialog_add_widget(zd,"hbox","hbalpha","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labalpha","hbalpha","alpha channel:","space=3");
   zdialog_add_widget(zd,"label","labpresent","hbalpha","(not) present","space=3");
   
   pp = strrchr(curr_file,'/');                                                  //  stuff file name into dialog
   if (pp) pp++;
   else pp = curr_file;
   zdialog_stuff(zd,"filename",pp);

   if (E0pxm->nc == 4)
      zdialog_stuff(zd,"labpresent","present");
   else
      zdialog_stuff(zd,"labpresent","not present");

   zdialog_resize(zd,300,0);
   zdialog_run(zd,change_alpha_dialog_event,"parent");

   return;
}


//  dialog event and completion function

int change_alpha_dialog_event(zdialog *zd, ch *event)
{
   if (! zd->zstat) return 1;                                                    //  wait for completion
   
   if (zd->zstat == 1)                                                           //  [add]                                 23.3
   {
      zd->zstat = 0;                                                             //  keep dialog alive

      E0pxm = PXM_load(curr_file,1);                                             //  reload file with full bit depth
      if (! E0pxm) return 1;
   
      if (E0pxm->nc == 4) return 1;                                              //  have alpha channel, do nothing

      PXM_addalpha(E0pxm);                                                       //  add alpha channel, value = 255
      
      f_save(curr_file,curr_file_type,curr_file_bpc,0,1);                        //  replace file on disk
      f_open(curr_file);                                                         //  refresh in memory

      zdialog_stuff(zd,"labpresent","present");                                  //  update dialog status
      return 1;
   }

   if (zd->zstat == 2)                                                           //  [remove]
   {
      zd->zstat = 0;                                                             //  keep dialog alive

      E0pxm = PXM_load(curr_file,1);                                             //  reload file with full bit depth
      if (! E0pxm) return 1;

      if (E0pxm->nc != 4) return 1;                                              //  no alpha channel, do nothing
      
      PXM_applyalpha(E0pxm);                                                     //  apply and remove alpha channel

      f_save(curr_file,curr_file_type,curr_file_bpc,0,1);                        //  replace file on disk
      f_open(curr_file);                                                         //  refresh in memory

      zdialog_stuff(zd,"labpresent","not present");                              //  update dialog status
      return 1;
   }
   
   zdialog_free(zd);                                                             //  [OK] or [x]

   if (E0pxm) PXM_free(E0pxm);                                                   //  free memory
   E0pxm = 0;

   Fblock("change alpha",0);                                                     //  unblock

   return 1;
}


/********************************************************************************/

//  Open a file and initialize Fpxb pixbuf.
//
//  Nth:    if Nth matches file position in current gallery, curr_file_posn
//          is set to Nth, otherwise it is searched and set correctly.
//          (a file can be present multiple times in an album gallery).
//  Fkeep:  edit undo/redo stack is not purged, and current edits are kept
//          after opening the new file (used by file_save()).
//  Fack:   failure will cause a popup ACK dialog.
//  fzoom:  keep current zoom level and position, otherwise fit window.
//
//  Following are set: curr_file_type, curr_file_bpc, curr_file_size.
//  Returns: 0 = OK, +N = error.
//  errors: 1  reentry
//          2  curr. edit function cannot be restarted or canceled
//          3  file not found or user cancel
//          4  unsupported file type or PXB_load() failure

int f_open(ch *filespec, int Nth, int Fkeep, int Fack, int fzoom)
{
   PXB            *tempxb = 0;
   int            fposn, retcode = 0;
   FTYPE          ftype;
   static int     Freent = 0;
   ch             *pp, *file;
   void           (*editfunc)(GtkWidget *, ch *);

   if (Freent++) {                                                               //  stop re-entry
      Plog(0,"f_open() re-entry \n");
      goto ret1;
   }
   
   poptext_killnow();                                                            //  kill pending popup message

   if (CEF && ! CEF->Frestart) goto ret2;                                        //  edit function not restartable

   if (CEF) editfunc = CEF->menufunc;                                            //  active edit, save edit function
   else editfunc = 0;                                                            //    for possible edit restart

   if (CEF && CEF->zd) zdialog_send_event(CEF->zd,"cancel");                     //  cancel if possible
   if (CEF) goto ret2;                                                           //  cannot

   sa_clear();                                                                   //  clear area if any
   
   file = 0;

   if (filespec)
      file = zstrdup(filespec,"f-open");                                         //  use passed filespec
   else {
      pp = curr_file;                                                            //  use file open dialog
      if (! pp && navi::gallerytype == FOLDER) pp = navi::galleryname;
      file = zgetfile("Open Image File",MWIN,"file",pp);
   }

   if (! file) goto ret3;

   if (! regfile(file)) {                                                        //  check file exists
      if (Fack) zmessage_post_bold(Mwin,"20/20",4,"file not found: %s",file);
      zfree(file);
      goto ret3;
   }

   ftype = image_file_type(file);                                                //  reject thumbnail
   if (ftype == THUMB) {
      if (Fack) zmessageACK(Mwin,"thumbnail file");
      goto ret4;
   }

   if (ftype != IMAGE && ftype != RAW && ftype != VIDEO) {                       //  must be supported image file type
      if (Fack) {
         Plog(0,"%s\n",file);
         zmessageACK(Mwin,"unknown file type");
      }
      zfree(file);
      goto ret4;
   }

   tempxb = PXB_load(file,1);                                                    //  load image as PXB pixbuf

   if (! tempxb) {                                                               //  PXB_load() messages user
      zfree(file);
      goto ret4;
   }

   free_resources(Fkeep);                                                        //  free resources for old image file

   curr_file = file;                                                             //  new current file

   Fpxb = tempxb;                                                                //  pixmap for current image

   strcpy(curr_file_type,f_load_type);                                           //  set curr_file_xxx from f_load_xxx
   curr_file_bpc = f_load_bpc;
   curr_file_size = f_load_size;

   load_filemeta(curr_file);                                                     //  load metadata used by fotoxx
   
   Fmeta_edithist = 0;                                                           //  no edit history data loaded
   
   Fupright = Flevel = 0;                                                        //  file not uprighted or levelled 
   if (ftype == RAW) Fupright = 1;                                               //  assume RAW loaders do upright

   fposn = file_position(file,Nth);                                              //  file position in gallery list
   if (fposn < 0) {                                                              //  not there
      gallery(curr_file,"init",0);                                               //  generate new gallery list
      gallery(0,"sort",-2);                                                      //  recall sort and position 
      gallery(curr_file,"paint",0);                                              //  position at curr. file
      fposn = file_position(curr_file,0);                                        //  position and count in gallery list
   }
   curr_file_posn = fposn;                                                       //  keep track of file position

   if (! fzoom) {                                                                //  discard zoom state
      Fzoom = 0;                                                                 //  zoom level = fit window
      zoomx = zoomy = 0;                                                         //  no zoom center
   }

   add_recent_file(curr_file);                                                   //  most recent file opened

   if (zd_rename) m_rename(0,0);                                                 //  update active rename dialog
   if (zd_permissions) m_permissions(0,0);                                       //    "  permissions dialog
   if (zd_copymove) m_copy_move(0,0);                                            //    "  copy/move dialog
   if (zd_deltrash) m_delete_trash(0,0);                                         //    "  delete/trash dialog
   if (zd_metaview) meta_view(0);                                                //    "  metadata view window
   if (zd_editmeta) m_meta_edit_main(0,0);                                       //    "  edit metadata dialog
   if (zd_editanymeta) m_meta_edit_any(0,0);                                     //    "  edit any metadata dialog
   if (zd_deletemeta) m_meta_delete(0,0);                                        //    "  delete metadata dialog
   if (Fcaps) meta_show_caps(1);                                                 //  show captions on current image

   if (editfunc) editfunc(0,0);                                                  //  restart edit function
   
   if (ftype == VIDEO)                                                           //  offer to play video file
      zmessage_post_bold(Mwin,"20/30",3,"VIDEO  press P to play");
   
   pp = strrchr(curr_file,'.');                                                  //  offer to play animated GIF file
   if (pp && strstr(".gif .GIF",pp))
      zmessage_post_bold(Mwin,"20/30",3,"GIF  press P to play");

   if (FGWM == 'F') {
      Fpaintnow();                                                               //  force repaint
      set_mwin_title();                                                          //  set win title from curr_file info
   }

   if (FGWM == 'G') gallery(0,"paint",-1);                                       //  paint gallery
   goto ret0; 

   ret4: retcode++;
   ret3: retcode++;
   ret2: retcode++;
   ret1: retcode++;
   ret0:
   Freent = 0;

   return retcode;
}


/********************************************************************************/

//  Open a file that was just saved. Used by file_save().
//  The edit undo/redo stack is not purged and current edits are kept.
//  Following are set:
//    curr_file  *_folder  *_file_posn  *_file_type  *_file_bpc  *_file_size
//  Returns: 0 = OK, +N = error.

int f_open_saved()
{
   int      Nth = -1, fposn;
   
   if (! f_save_file) return 1;

   if (E0pxm) {                                                                  //  edits were made
      PXB_free(Fpxb);                                                            //  new window image
      Fpxb = PXM_PXB_copy(E0pxm);
   }
   
   zstrcopy(curr_file,f_save_file,"f-open");                                     //  curr. file = last saved file

   fposn = file_position(curr_file,Nth);                                         //  file position in gallery list
   if (fposn < 0) {                                                              //  not there
      gallery(curr_file,"init",0);                                               //  generate new gallery list
      gallery(0,"sort",-2);                                                      //  recall sort and position
      gallery(curr_file,"paint",-1);                                             //  position at current file
      fposn = file_position(curr_file,0);                                        //  position and count in gallery list
   }
   curr_file_posn = fposn;                                                       //  keep track of file position

   strcpy(curr_file_type,f_save_type);                                           //  set curr_file_xxx from f_save_xxx
   curr_file_bpc = f_save_bpc;
   curr_file_size = f_save_size;

   zoomx = zoomy = 0;                                                            //  no zoom center
   if (FGWM == 'F') set_mwin_title();                                            //  set win title from curr_file info

   if (zd_metaview) meta_view(0);                                                //  update meta view window
   return 0;
}


/********************************************************************************

   Remove a file and clean up index, thumbnails, gallery, curr_file.
   Albums must be handled separately via album_purge_replace().
   'opt' is "delete" or "trash".
   trash logic: Move target file to ~/Desktop/ first, then move to trash.
                Gnome trash fails for files on mounted volumes.
   Returns 0 if OK, else error code.

****/

int f_remove(ch *file, ch *opt)
{
   int      err, Nth;
   ch       *cfile = 0;
   ch       trashfile[400] = "~/Desktop/"; 
   ch       *pp;
   GFile    *gfile;
   GError   *gerror = 0;
   int      gstat;
   
   if (*opt == 'd')                                                              //  delete
   {
      err = remove(file);
      if (err && errno != ENOENT) {                                              //  OK if already deleted
         zmessageACK(Mwin,"delete %s \n %s",file,strerror(errno));
         return err;
      }
   }
   
   if (*opt == 't')                                                              //  trash
   {
      strcpy(trashfile,getenv("HOME"));                                          //  copy to desktop, then Gnome trash
      strcat(trashfile,"/Desktop/");
      pp = strrchr(file,'/');                                                    //  get base file name, filename.jpg
      if (pp) pp++;
      else pp = file;
      strncatv(trashfile,400,pp,0);                                              //  ~/Desktop/filename.jpg
      
      err = cp_copy(file,trashfile);                                             //  copy target file to desktop
      if (err) {
         zmessageACK(Mwin,"copy to Desktop failed: %s",strerror(errno));
         return 1;
      }
      
      gfile = g_file_new_for_path(trashfile);
      gstat = g_file_trash(gfile,0,&gerror);
      g_object_unref(gfile);
      if (! gstat) {
         zmessageACK(Mwin,"move to trash failed: %s",gerror->message);
         return 2;
      }
      
      err = remove(file);                                                        //  delete file
      if (err && errno != ENOENT) {                                              //  OK if already deleted
         zmessageACK(Mwin,"delete %s \n %s",file,strerror(errno));
         return err;
      }
   }

   delete_image_index(file);                                                     //  delete file in image index
   delete_thumbfile(file);                                                       //  delete thumbnail file and cache

   Nth = file_position(file,0);                                                  //  find in gallery list
   if (Nth >= 0) gallery(0,"delete",Nth);                                        //  delete from gallery list
   
   if (curr_file && strmatch(file,curr_file)) {                                  //  current file was removed
      free_resources();
      cfile = gallery(0,"getR",curr_file_posn);                                  //  new current file = next               23.1
      if (cfile) f_open(cfile);
      else {
         Nth = curr_file_posn - 1;
         if (Nth >= navi::Gfolders && Nth < navi::Gfiles) {                      //  no next, try previous
            cfile = gallery(0,"getR",curr_file_posn - 1);                        //  23.1
            if (cfile) f_open(cfile);
         }
      }
   }
   
   return 0;
}


/********************************************************************************/

//  Function to preload image files hopefully ahead of need.
//  Usage: f_preload(next)
//    next = -1 / +1  to read previous / next image file in curr. gallery
//    preload_file will be non-zero if and when preload_pxb is available.

void f_preload(int next, int Flast)
{
   int      fd;
   ch       *file;

   if (! curr_file) return;

   file = prev_next_file(next,Flast);
   if (! file) return;

   if (strmatch(file,curr_file)) return;

   fd = open(file,O_RDONLY);
   if (fd >= 0) {
      posix_fadvise(fd,0,0,POSIX_FADV_WILLNEED);                                 //  preload file in kernel cache
      close(fd);
   }

   return;
}


/********************************************************************************/

//  Open previous or next file in current gallery list, 
//  or jump to previous or next gallery when at the limits.
//  index is -1 / +1 for prev / next file.
//  Fjump: jump to last file version, jump to prev/next folder.

void x_prev_next(int index, int Fjump)
{
   using namespace navi;

   ch              *newfile = 0, *newgallery = 0;
   int             err, Nth = 0;
   static int      xbusy = 0, jumpswitch = 0;
   static zdialog  *zd = 0;

   ch     *mess1 = "Previous gallery";
   ch     *mess2 = "Next gallery";
   ch     *mess3 = "Start of gallery";
   ch     *mess4 = "End of gallery";
   
   if (! curr_file) return;
   if (FGWM != 'F' && FGWM != 'G') return;

   if (zd && zdialog_valid2(zd,"post")) zdialog_free(zd);                        //  clear prior popup message
   zd = 0;
   
   if (Fblock(0,"blocked edits")) return;                                        //  check nothing pending

   if (xbusy) return;                                                            //  avoid "function busy"
   xbusy = 1;
   
   newfile = prev_next_file(index,Fjump);                                        //  get prev/next file (last version)
   if (newfile) {
      Nth = curr_file_posn + index;                                              //  albums can have repeat files
      err = f_open(newfile,Nth,0,1,1);                                           //  open image or RAW file 
      if (! err) f_preload(index,Fjump);                                         //  preload next image
      goto returnx;
   }
   
   if (Fjump) {
      if (index == -1) zd = zmessage_post_bold(Mwin,"5/5",3,mess1,0);            //  notify jump to prev/next gallery
      if (index == +1) zd = zmessage_post_bold(Mwin,"5/5",3,mess2,0);
   }
   else {
      if (index == -1) zd = zmessage_post_bold(Mwin,"5/5",3,mess3,0);            //  notify gallery start/end
      if (index == +1) zd = zmessage_post_bold(Mwin,"5/5",3,mess4,0);
      jumpswitch = 0;
   }

   if (Fjump && jumpswitch) {                                                    //  jump to prev/next gallery
      jumpswitch = 0;
      newgallery = prev_next_gallery(index);
      if (! newgallery) goto returnx;
      gallery(newgallery,"init",0);                                              //  load gallery
      zfree(newgallery);
      gallery(0,"sort",-2);                                                      //  preserve sort
      if (Gfiles - Gfolders > 0) {                                               //  at least one image file present
         if (index == +1) Nth = Gfolders;                                        //  get first or last image file
         if (index == -1) Nth = Gfiles - 1;
         newfile = gallery(0,"getR",Nth);                                        //  23.1
         err = f_open(newfile,Nth,0,1,0);                                        //  open image or RAW file
         if (! err) gallery(newfile,"paint",Nth);
         if (! err) f_preload(index,Fjump);                                      //  preload next image
         goto returnx;
      }
      else {                                                                     //  no image files in gallery
         gallery(0,"paint",0);
         m_viewmode(0,"G");
         free_resources();
      }
   }

   else if (Fjump) jumpswitch = 1;                                               //  hesitate once, jump next time
   
returnx:
   zmainloop();                                                                  //  refresh window (holding arrow key)
   xbusy = 0;
   return;
}


void m_prev(GtkWidget *, ch *menu)
{
   Plog(1,"m_prev \n");
   if (menu) x_prev_next(-1,1);                                                  //  jump option
   else x_prev_next(-1,0);                                                       //  search from curr_file -1
   return;                                                                       //    to first file
}


void m_next(GtkWidget *, ch *menu)
{
   Plog(1,"m_next \n");
   if (menu) x_prev_next(+1,1);                                                  //  jump option 
   else x_prev_next(+1,0);                                                       //  search from curr_file +1
   return;                                                                       //    to last file
}


void m_prev_next(GtkWidget *, ch *menu)                                          //  left/right mouse click >>
{                                                                                //    previous/next image file
   int button = zfuncs::vmenuclickbutton;
   if (button == 1) m_prev(0,0);
   else m_next(0,0);
   return;
}


/********************************************************************************/

//  zoom in/out - left/right mouse click on zoom button or F-view image

void m_zoom_menu(GtkWidget *, ch *menu)                                          //  left/right mouse click
{
   F1_help_topic = "zoom";

   int button = zfuncs::vmenuclickbutton;

   if (FGWM == 'G') {                                                            //  G view, larger/smaller thumbnails
      if (button == 1) navi::menufuncx(0,"Zoom+");
      else navi::menufuncx(0,"Zoom-");
   }
   
   if (FGWM == 'F') {                                                            //  F view, zoom image in/out
      if (button == 1) m_zoom(0,"Zoom+");
      else m_zoom(0,"Zoom-");
   }
   
   return;
}


/********************************************************************************/

//  create a new blank image file with desired background color

void m_blank_image(GtkWidget *, ch *pname)
{
   ch          color[20], fname[100], fext[8], *filespec;
   int         zstat, err, cc, ww, hh, RGB[3];
   zdialog     *zd;
   ch          *pp;
   ch          *curr_folder = 0;

   F1_help_topic = "blank image";

   Plog(1,"m_blank_image \n");

   if (navi::gallerytype == FOLDER) curr_folder = navi::galleryname;
   else curr_folder = topfolders[0];
   if (! curr_folder) {
      Plog(0,"no top image folders defined \n");                                 //  should not happen
      return;
   }

   if (Fblock("blank_image","block edits")) return;                              //  check pend, block

   m_viewmode(0,"F");

/***
       __________________________________________________
      |           Create Blank Image                     |
      |                                                  |
      | file name [___________________________] [.jpg|v] |
      | width [____]  height [____] (pixels)             |
      | color [____]                                     |
      |                                                  |
      |                                  [ OK ] [Cancel] |
      |__________________________________________________|

***/


   zd = zdialog_new("Create Blank Image",Mwin,"OK","Cancel",null);
   zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labf","hbf","file name","space=3");
   zdialog_add_widget(zd,"zentry","file","hbf",0,"space=3|expand");
   zdialog_add_widget(zd,"combo","ext","hbf",".jpg","space=3|size=3");
   zdialog_add_widget(zd,"hbox","hbz","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labw","hbz","Width","space=5");
   zdialog_add_widget(zd,"zspin","width","hbz","100|30000|1|1600");
   zdialog_add_widget(zd,"label","space","hbz",0,"space=5");
   zdialog_add_widget(zd,"label","labh","hbz","Height","space=5");
   zdialog_add_widget(zd,"zspin","height","hbz","100|16000|1|1000");
   zdialog_add_widget(zd,"label","labp","hbz","(pixels)","space=3");
   zdialog_add_widget(zd,"hbox","hbc","dialog",0,"space=1");
   zdialog_add_widget(zd,"label","labc","hbc","Color","space=5");
   zdialog_add_widget(zd,"colorbutt","color","hbc","200|200|200");

   zdialog_stuff(zd,"ext",".jpg");
   zdialog_stuff(zd,"ext",".png");
   
   zdialog_restore_inputs(zd);

   zdialog_set_modal(zd);
   
   zdialog_run(zd,0,"save");                                                     //  run dialog
   zstat = zdialog_wait(zd);                                                     //  wait for completion

   if (zstat != 1) {                                                             //  cancel
      zdialog_free(zd);
      Fblock("blank_image",0);
      return;
   }

   zdialog_fetch(zd,"file",fname,92);                                            //  get new file name
   strTrim2(fname);
   if (*fname <= ' ') {
      zmessageACK(zd->dialog,"supply a file name");
      zdialog_free(zd);
      Fblock("blank_image",0);
      m_blank_image(0,0);                                                        //  retry
      return;
   }

   zdialog_fetch(zd,"ext",fext,8);                                               //  add extension
   strcat(fname,fext);
   
   cc = strlen(fname);
   filespec = zstrdup(curr_folder,"blank-image",cc+4);                           //  make full filespec
   strcat(filespec,"/");
   strcat(filespec,fname);

   zdialog_fetch(zd,"width",ww);                                                 //  get image dimensions
   zdialog_fetch(zd,"height",hh);

   RGB[0] = RGB[1] = RGB[2] = 255;
   zdialog_fetch(zd,"color",color,19);                                           //  get image color
   pp = substring(color,"|",1);
   if (pp) RGB[0] = atoi(pp);
   pp = substring(color,"|",2);
   if (pp) RGB[1] = atoi(pp);
   pp = substring(color,"|",3);
   if (pp) RGB[2] = atoi(pp);

   zdialog_free(zd);

   err = create_blank_file(filespec,ww,hh,RGB);
   if (! err) f_open(filespec);                                                  //  make it the current file
   zfree(filespec);
   Fblock("blank_image",0);
   return;
}


//  function to create a new blank image file
//  file extension must be one of: .jpg .tif .png
//  RGB args are in the range 0 - 255
//  if file exists it is overwritten
//  returns 0 if OK, 1 if file exists, +N if error

int create_blank_file(ch *file, int ww, int hh, int RGB[3])
{
   ch             *pp;
   ch             *fext;
   int            cstat;
   PXB            *tempxb;
   GError         *gerror = 0;
   uint8          *pixel;

   if (regfile(file)) {
      zmessageACK(Mwin,"output file exists");                                    //  file already exists
      return 1;
   }

   pp = strrchr(file,'.');                                                       //  get file .ext
   if (! pp || strlen(pp) > 4) return 1;

   if (strmatch(pp,".jpg")) fext = "jpeg";                                       //  validate and set pixbuf arg.
   else if (strmatch(pp,".png")) fext = "png";
   else return 2;

   tempxb = PXB_make(ww,hh,3);                                                   //  create pixbuf image

   for (int py = 0; py < hh; py++)                                               //  fill with color
   for (int px = 0; px < ww; px++)
   {
      pixel = PXBpix(tempxb,px,py);
      pixel[0] = RGB[0];
      pixel[1] = RGB[1];
      pixel[2] = RGB[2];
   }

   cstat = gdk_pixbuf_save(tempxb->pixbuf,file,fext,&gerror,null);
   if (! cstat) {
      zmessageACK(Mwin,"error: %s",gerror->message);
      PXB_free(tempxb);
      return 3;
   }

   PXB_free(tempxb);
   return 0;
}


/********************************************************************************/

//  animate given GIF file if possible

void play_gif(ch *file)
{
   int gif_animations_dialog_event(zdialog *, ch *event);

   FILE        *fid;
   ch          *file2;
   ch          command[300], buff[20] = "", *pp = 0;
   zdialog     *zd;
   GtkWidget   *image;
   GdkPixbufAnimation *animation;
   
   file2 = zescape_quotes(file);                                                 //  23.4
   snprintf(command,300,"exiftool -s3 -framecount \"%s\"",file2);
   zfree(file2);
   fid = popen(command,"r");
   if (fid) {
      pp = fgets(buff,20,fid);
      pclose(fid);
   }
   if (! pp || atoi(pp) < 2) return;

   zd = zdialog_new("GIF animation",Mwin,0);
   zdialog_add_widget(zd,"image","gif","dialog");
   image = zdialog_gtkwidget(zd,"gif");
   animation = gdk_pixbuf_animation_new_from_file(curr_file,0);
   gtk_image_set_from_animation(GTK_IMAGE(image),animation);
   zdialog_run(zd,gif_animations_dialog_event,"desktop");
   zdialog_can_focus(zd,0);  

   return;
}


//  dialog event and completion function

int gif_animations_dialog_event(zdialog *zd, ch *event)
{
   if (zd->zstat) zdialog_free(zd); 
   return 1;
}


/********************************************************************************/

//  copy or move an image file to a new location

namespace   copymove_names
{
   ch       *sorcfile = 0;                                                       //  source file full name
   ch       *destname = 0;                                                       //  destination file base name
   ch       *destfile = 0;                                                       //  destination file full name
   ch       *nextfile = 0;                                                       //  next file in source gallery
}


//  menu function

void m_copy_move(GtkWidget *, ch *)
{
   using namespace copymove_names;

   int copymove_dialog_event(zdialog *zd, ch *event);

   ch     *title = "Copy or Move Image File";
   ch     *pp;

   F1_help_topic = "copy/move";

   Plog(1,"m_copy_move \n");

   if (sorcfile) zfree(sorcfile);
   sorcfile = 0;

   if (clicked_file) {                                                           //  use clicked file if present
      sorcfile = clicked_file;
      clicked_file = 0;
   }
   else if (curr_file)                                                           //  else use current file
      sorcfile = zstrdup(curr_file,"copy-move");
   else return;

   if (FGWM != 'F' && FGWM != 'G') return;

/***
          ________________________________________________________
         |               Copy or Move Image File                  |
         |                                                        |
         | Image File: [________________________________________] |              sorcname       file base name
         | New Location: [_____________________________] [browse] |              copymove_loc   destination folder
         | New File Name: [_____________________________________] |              destname       file base name
         |                                                        |
         | (o) copy (duplicate file)  (o) move (delete original)  |
         | [x] keep this dialog open                              |
         | [x] move to next input file                            |
         |                                      [apply] [cancel]  |
         |________________________________________________________|
***/

   if (! zd_copymove)                                                            //  globally visible dialog
   {
      zd_copymove = zdialog_new(title,Mwin,"Apply","Cancel",null);
      zdialog *zd = zd_copymove;

      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labf","hb1","Image File:","space=5");
      zdialog_add_widget(zd,"label","sorcname","hb1");                           //  sorcname not editable

      zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=2");
      zdialog_add_widget(zd,"label","labl","hb2","New Location:","space=5");
      zdialog_add_widget(zd,"zentry","copymove_loc","hb2",0,"expand");
      zdialog_add_widget(zd,"button","Browse","hb2","Browse","space=5");
      
      zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=2");
      zdialog_add_widget(zd,"label","labl","hb3","New File Name:","space=5");
      zdialog_add_widget(zd,"zentry","destname","hb3",0,"expand");
      
      zdialog_add_widget(zd,"hbox","hb4","dialog",0,"space=3");
      zdialog_add_widget(zd,"radio","copy","hb4","copy (duplicate file)","space=5");
      zdialog_add_widget(zd,"radio","move","hb4","move (delete original)","space=5");

      zdialog_add_widget(zd,"hbox","hb5","dialog");
      zdialog_add_widget(zd,"check","keepopen","hb5","keep this dialog open","space=3");

      zdialog_add_widget(zd,"hbox","hb6","dialog");
      zdialog_add_widget(zd,"check","next","hb6","move to next input file","space=3");
      
      zdialog_stuff(zd,"copy",1);
      zdialog_stuff(zd,"move",0);
      zdialog_stuff(zd,"keepopen",0);
      zdialog_stuff(zd,"next",0);

      zdialog_restore_inputs(zd);                                                //  default prior inputs
      zdialog_resize(zd,400,0);
      zdialog_run(zd,copymove_dialog_event,"parent");                            //  run dialog
   }

   zdialog *zd = zd_copymove;
   
   zdialog_stuff(zd,"sorcname","");                                              //  clear dialog
   zdialog_stuff(zd,"copymove_loc","");
   zdialog_stuff(zd,"destname","");
   
   if (sorcfile) {
      if (FGWM == 'G') f_open(sorcfile,0,0,1,0);                                 //  show or highlight gallery
      pp = strrchr(sorcfile,'/');
      if (pp) {
         zdialog_stuff(zd,"sorcname",pp+1);                                      //  use given source file
         zdialog_stuff(zd,"destname",pp+1);                                      //  initz. dest file name from source
      }
   }
   
   if (! sorcfile || ! pp) {                                                     //  should not happen
      zmessageACK(Mwin,"invalid source file: %s",sorcfile);
      zdialog_free(zd);
      zd_copymove = 0;
      return;
   }

   if (copymove_loc) zdialog_stuff(zd,"copymove_loc",copymove_loc);              //  last used copy-to location

   return;
}


//  dialog event and completion callback function

int copymove_dialog_event(zdialog *zd, ch *event)
{
   using namespace copymove_names;

   int          Fcopy, Fmove, Fnext;
   ch           *pp, *pext;
   ch           copyloc[XFCC];
   int          cc, err, Fkeep, Nth;
   
   if (strmatch(event,"Browse")) {                                               //  browse for new location
      pp = zgetfile("Select folder",MWIN,"folder",copymove_loc);
      if (! pp) return 1;
      if (copymove_loc) zfree(copymove_loc);                                     //  save new copymove location
      copymove_loc = pp;
      zdialog_stuff(zd,"copymove_loc",copymove_loc);
      return 1;
   }
   
   if (zd->zstat == 0) return 1;                                                 //  wait for dialog finished

   if (zd->zstat != 1) {                                                         //  [cancel] or [x]
      zdialog_free(zd);                                                          //  kill dialog
      zd_copymove = 0;
      return 1;
   }

   zd->zstat = 0;                                                                //  [apply] - keep dialog active

   if (image_file_type(sorcfile) != IMAGE) {
      zmessageACK(Mwin,"invalid source file: %s",sorcfile);                      //  validate source file
      return 1;
   }

   if (Fblock(0,"blocked")) return 1;                                            //  check for blocking function
   if (curr_file && strmatch(curr_file,sorcfile))                                //  current file will be moved
      if (Fblock(0,"edits")) return 1;                                           //    check for edits pending

   zdialog_fetch(zd,"copy",Fcopy);                                               //  get options
   zdialog_fetch(zd,"move",Fmove);
   zdialog_fetch(zd,"next",Fnext);

   if (Fcopy + Fmove != 1) return 1;                                             //  one of copy/move must be set

   zdialog_fetch(zd,"copymove_loc",copyloc,XFCC);                                //  get move-to location from dialog
   if (! dirfile(copyloc)) {                                                     //  check for valid folder
      zmessageACK(Mwin,"new location is not a folder");
      return 1;
   }

   zstrcopy(copymove_loc,copyloc,"copy-move");                                   //  save new copymove location

   if (destname) zfree(destname);
   destname = (ch *) zmalloc(200,"copy-move");                                   //  get dest file name
   zdialog_fetch(zd,"destname",destname,190);

   if (*destname <= ' ' || strchr(destname,'/')) {
      zmessageACK(Mwin,"new file name is invalid");
      return 1;
   }
   
   pext = strrchr(destname,'.');                                                 //  find dest file .ext
   if (! pext || strlen(pext) > 8) 
      pext = destname + strlen(destname);                                        //  try to skip over '.' in file name
   if (strlen(pext) > 3 && pext[1] == 'v') 
      pext = destname + strlen(destname);                                        //  try to skip over version number

   *pext = 0;
   pp = strrchr(sorcfile,'.');
   if (pp && strlen(pp) < 9) strcpy(pext,pp);                                    //  copy source file .ext to dest file

   cc = strlen(copymove_loc) + strlen(destname) + 2;                             //  new file = copymove_loc/destname.ext
   if (destfile) zfree(destfile);
   destfile = (ch *) zmalloc(cc,"copy-move");
   *destfile = 0;
   strncatv(destfile,cc,copymove_loc,"/",destname,0);

   if (strmatch(sorcfile,destfile)) {                                            //  block copy to self
      zmessageACK(Mwin,"cannot copy a file to itself");
      return 1;
   }

   if (regfile(destfile)) {                                                      //  if new file exists
      int yn = zmessageYN(Mwin,"output file exists, overwrite?");                //    optionally overwrite file
      if (! yn) return 1;
   }

   err = cp_copy(sorcfile,destfile);                                             //  copy source file to destination
   if (err) {
      zmessageACK(Mwin,strerror(err));
      return 1;
   }
   
   load_filemeta(destfile);                                                      //  update image index for new file
   update_image_index(destfile);
   add_recent_file(destfile);                                                    //  add to recent files list

   if (nextfile) zfree(nextfile);                                                //  find next file in gallery, if any
   nextfile = 0;
   Nth = file_position(sorcfile,0);                                              //  source file position in gallery
   if (Nth >= 0) nextfile = gallery(0,"get",Nth+1);

   if (Fmove)                                                                    //  move - delete source file
   {
      album_purge_replace("ALL",sorcfile,destfile);                              //  replace in albums where present
                                                                                 //    (must precede f_remove)
      err = f_remove(sorcfile,"delete");                                         //  delete file/index/thumb/gallery
      if (err) {
         zmessageACK(Mwin,"delete failed: \n %s",strerror(errno));
         return 1;
      }
      
      if (curr_file && strmatch(curr_file,sorcfile)) {                           //  current file deleted
         zfree(curr_file);
         curr_file = 0;
      }
   }

   if (sorcfile) zfree(sorcfile);                                                //  initz. no source file
   sorcfile = 0;
   zdialog_stuff(zd,"sorcname","");
   zdialog_stuff(zd,"destname","");

   if (Fnext && nextfile) {                                                      //  next file in gallery
      sorcfile = nextfile;
      nextfile = 0;
      f_open(sorcfile,0,0,1,0);                                                  //  show file or highlight thumb
      pp = strrchr(sorcfile,'/');
      if (pp) {
         zdialog_stuff(zd,"sorcname",pp+1);                                      //  next source and dest file name
         zdialog_stuff(zd,"destname",pp+1);
      }
   }

   else if (curr_file) f_open(curr_file,0,0,1,0);                                //  restore curr. metadata

   if (FGWM == 'G' && navi::gallerytype == FOLDER) {                             //  refresh gallery
      gallery(0,"init",0);
      gallery(0,"sort",-2);                                                      //  recall sort and position
      gallery(0,"paint",-1);
   }
   
   zdialog_fetch(zd,"keepopen",Fkeep);
   if (! Fkeep) {
      zdialog_free(zd);                                                          //  kill dialog
      zd_copymove = 0;
   }
   else gtk_window_present(MWIN);                                                //  keep focus on main window

   return 1;
}


/********************************************************************************/

//  copy selected image file to the desktop

void m_copyto_desktop(GtkWidget *, ch *)
{
   ch       *file, *pp, newfile[XFCC];
   int      err;
   
   F1_help_topic = "copy to desktop";

   Plog(1,"m_copyto_desktop \n");

   if (FGWM != 'F' && FGWM != 'G') return;
   
   file = clicked_file;
   if (! file) file = curr_file;
   if (! file) return;
   
   snprintf(newfile,XFCC,"%s/%s",getenv("HOME"),desktopname);                    //  locale specific desktop name
   pp = strrchr(file,'/');
   if (! pp) return;
   strncatv(newfile,XFCC,pp,0);                                                  //  new desktop file name
   
   if (regfile(newfile)) {                                                       //  check if file exists
      int yn = zmessageYN(Mwin,"output file exists, overwrite?");                //  confirm overwrite
      if (! yn) goto cleanup;
   }
   
   if (curr_file && strmatch(file,curr_file)                                     //  current file is copied
                  && image_file_type(file) == IMAGE)
      f_save(newfile,curr_file_type,curr_file_bpc,0,1);                          //  preserve edits
   
   else {
      err = cp_copy(file,newfile);                                               //  copy other file
      if (err) {
         zmessageACK(Mwin,strerror(err));
         goto cleanup;
      }
      load_filemeta(newfile);                                                    //  update image index
      update_image_index(newfile);
   }

cleanup:
   if (clicked_file) zfree(clicked_file);
   clicked_file = 0;
   return;
}


/********************************************************************************/

//  copy selected image file to the clipboard

void m_copyto_clip(GtkWidget *, ch *)
{
   int file_copytoclipboard(ch *file);
   
   ch       tempfile[200];
   int      err;
   
   F1_help_topic = "copy to clipboard";

   Plog(1,"m_copyto_clip \n");

   if (FGWM != 'F' && FGWM != 'G') return;

   if (clicked_file) {
      file_copytoclipboard(clicked_file);                                        //  get file behind thumbnail
      zfree(clicked_file);
      clicked_file = 0;
      return;
   }

   if (! curr_file) return;                                                      //  no current file
   
   if (image_file_type(curr_file) != IMAGE) {                                    //  non-image file
      file_copytoclipboard(curr_file);
      return;
   }
   
   snprintf(tempfile,200,"%s/clipboard.jpg",temp_folder);                        //  may be edited and unsaved
   err = f_save(tempfile,"jpg",8,0,1);
   if (err) return;
   file_copytoclipboard(tempfile);
   remove(tempfile);

   return;
}


//  copy an image file to the clipboard (as pixbuf)
//  any prior clipboard image is replaced
//  supports copy/paste to other apps (not used within fotoxx)
//  returns 1 if OK, else 0

int file_copytoclipboard(ch *file)
{
   GtkClipboard   *clipboard;
   PIXBUF         *pixbuf;
   GError         *gerror = 0;

   pixbuf = gdk_pixbuf_new_from_file(file,&gerror);
   if (! pixbuf) return 0;

   clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
   if (! clipboard) return 0;
   gtk_clipboard_clear(clipboard);
   gtk_clipboard_set_image(clipboard,pixbuf);
   gtk_clipboard_set_can_store(clipboard,0,0);
   gtk_clipboard_store(clipboard);                                               //  persistence after exit does not work
// GdkWindow *gdkw = gtk_widget_get_window(Mwin);
// gdk_display_store_clipboard(zfuncs::display,gdkw,0,null,0);
   g_object_unref(pixbuf);
   return 1;
}


/********************************************************************************/

//  Delete or Trash an image file.
//  Use the Linux standard trash function.
//  If not available, show diagnostic and do nothing.

ch  *deltrash_file = 0;

void m_delete_trash(GtkWidget *, ch *)                                           //  combined delete/trash function
{
   int delete_trash_dialog_event(zdialog *zd, ch *event);

   ch   *title = "Delete/Trash Image File";
   
   F1_help_topic = "delete/trash";

   Plog(1,"m_delete_trash \n");

   if (FGWM != 'F' && FGWM != 'G') return;

   if (deltrash_file) zfree(deltrash_file);
   deltrash_file = 0;
   
   if (clicked_file) {                                                           //  use clicked file if present 
      deltrash_file = clicked_file;
      clicked_file = 0;
   }

   else if (curr_file)                                                           //  else current file
      deltrash_file = zstrdup(curr_file,"delete-trash");

/***
          ______________________________________
         |        Delete/Trash Image File       |
         |                                      |
         | File: [____________________________] |
         |                                      |
         | [x] keep this dialog open            |
         |                                      |
         |            [delete] [trash] [cancel] |
         |______________________________________|

***/

   if (! zd_deltrash)                                                            //  start dialog if not already
   {
      zd_deltrash = zdialog_new(title,Mwin,"Delete","Trash","Cancel",null);
      zdialog *zd = zd_deltrash;

      zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","labf","hb1","File:","space=3");
      zdialog_add_widget(zd,"label","file","hb1",0,"space=3");
      zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=3");
      zdialog_add_widget(zd,"check","keepopen","hb2","keep this dialog open","space=3");

      zdialog_resize(zd,300,0);
      zdialog_restore_inputs(zd);
      zdialog_run(zd,delete_trash_dialog_event,"parent");                        //  run dialog
   }
   
   if (deltrash_file) 
   {
      ch *pp = strrchr(deltrash_file,'/');
      if (pp) pp++;
      else pp = deltrash_file;
      zdialog_stuff(zd_deltrash,"file",pp);
      if (FGWM == 'G') f_open(deltrash_file,0,0,1,0);                            //  show or highlight gallery
   }
   else zdialog_stuff(zd_deltrash,"file","");
   
   return;
}


//  dialog event and completion callback function

int delete_trash_dialog_event(zdialog *zd, ch *event)
{
   int         err = 0, yn = 0;
   int         ftype, Fkeep;
   STATB       statB;
   ch          *opt = 0;
   ch          *pp;

   if (zd->zstat == 0) return 1;                                                 //  wait for dialog finished
   if (zd->zstat == 1) goto PREP;                                                //  [delete] button
   if (zd->zstat == 2) goto PREP;                                                //  [trash] button
   goto EXIT;                                                                    //  [cancel] or [x]

PREP:

   if (! deltrash_file) goto EXIT;                                               //  no file to delete or trash

   if (! regfile(deltrash_file,&statB)) {                                        //  check file exists
      zmessageACK(Mwin,strerror(errno));
      goto EXIT;
   }

   ftype = image_file_type(deltrash_file);
   if (ftype != IMAGE && ftype != RAW && ftype != VIDEO) {                       //  must be image or RAW or VIDEO 
      zmessageACK(Mwin,"not a known image file");
      goto EXIT;
   }

   if (! hasperm(deltrash_file,'w')) {                                           //  check permission
      if (zd->zstat == 1)
         yn = zmessageYN(Mwin,"Delete read-only file?");
      if (zd->zstat == 2)
         yn = zmessageYN(Mwin,"Trash read-only file?");
      if (! yn) goto NEXT;                                                       //  keep file

      statB.st_mode |= S_IWUSR;                                                  //  make writable
      err = chmod(deltrash_file,statB.st_mode);
      if (err) {
         zmessageACK(Mwin,strerror(errno));
         goto EXIT;
      }
   }

   if (Fblock(0,"blocked")) goto EXIT;                                           //  check for blocking function
   if (curr_file && strmatch(curr_file,deltrash_file))                           //  current file will be deleted
      if (Fblock(0,"edits")) goto EXIT;                                          //    check for edits pending

   album_purge_replace("ALL",deltrash_file,0);                                   //  remove from albums if present

   if (zd->zstat == 1) opt = "delete";                                           //  [delete] button
   if (zd->zstat == 2) opt = "trash";                                            //  [trash] button

   err = f_remove(deltrash_file,opt);                                            //  remove file/index/thumb/gallery
   if (err) goto EXIT;

   if (FGWM == 'F') {                                                            //  F window
      zdialog_fetch(zd,"keepopen",Fkeep);
      if (! Fkeep) goto EXIT;
      zd->zstat = 0;                                                             //  keep dialog active
      gtk_window_present(MWIN);                                                  //  keep focus on main window
      return 1;
   }

NEXT:                                                                            //  step to next file

   if (FGWM == 'G') {                                                            //  gallery view
      gallery(0,"init",0);                                                       //  refresh for removed file
      gallery(0,"sort",-2);                                                      //  recall sort and position 
      gallery(0,"paint",-1);                                                     //  paint
   }

   if (FGWM == 'F') m_next(0,0);                                                 //  file view mode, open next image
   
   zdialog_fetch(zd,"keepopen",Fkeep);                                           //  keep going if wanted
   if (! Fkeep) goto EXIT;

   zd->zstat = 0;                                                                //  keep dialog active
   gtk_window_present(MWIN);                                                     //  keep focus on main window

   zdialog_stuff(zd_deltrash,"file","");                                         //  set no next file in dialog
   if (deltrash_file) zfree(deltrash_file);
   deltrash_file = 0;

   if (! curr_file) goto EXIT;                                                   //  no current file

   deltrash_file = zstrdup(curr_file,"delete-trash");                            //  default delete new current file
   pp = strrchr(deltrash_file,'/');
   if (pp) pp++;
   else pp = deltrash_file;
   zdialog_stuff(zd_deltrash,"file",pp);

   if (FGWM == 'G') f_open(deltrash_file,0,0,1,0);                               //  show or highlight gallery
   return 1;

EXIT:

   if (deltrash_file) zfree(deltrash_file);                                      //  free memory
   deltrash_file = 0;
   zdialog_free(zd);                                                             //  quit dialog
   zd_deltrash = 0;
   return 1;
}


/********************************************************************************/

//  Select an Adobe document file and convert all pages into jpeg image files.
//  The image files are named InputFileName-N.jpg where N is the page number.
//  File types recognized:
//       .pdf              portable document format
//       .ps  .eps         postscript
//       .psd              photoshop image file

void m_convert_adobe(GtkWidget *, ch *)
{
   ch       *pp, *infile = 0, *outfile = 0;
   ch       *f1, *f2;
   zdialog  *zd = 0;
   int      err;

   F1_help_topic = "convert adobe";

   Plog(1,"convert adobe \n");

   if (Fblock(0,"blocked edits")) return;                                        //  check nothing pending
   
   pp = 0;
   if (navi::gallerytype == FOLDER) pp = navi::galleryname;
   infile = zgetfile("Open Adobe File",MWIN,"file",pp);
   if (! infile) goto returnx;

   pp = strrchr(infile,'.');
   if (! strcasestr(".pdf .ps .eps .psd",pp)) {
      zmessageACK(Mwin,"not an Adobe file (.pdf .ps .eps .psd)");
      goto returnx;
   }
   
   if (strmatchcase(".psd",pp)) goto photoshop;                                  //  photoshop image
   
   err = zshell(0,"which gs >/dev/null 2>&1");                                   //  check ghostscript installed
   if (err) {
      zmessageACK(Mwin,"ghostscript must be installed");
      goto returnx;
   }

   outfile = zstrdup(infile,"convert adobe",12);                                 //  file.pdf  >>  file-%d.jpg
   pp = strrchr(outfile,'.');
   if (! pp) goto returnx;
   strcpy(pp,"-%d.jpg");                                                         //  file-1.jpg, file-2.jpg, ...
   
   f1 = zescape_quotes(infile);                                                  //  23.4
   f2 = zescape_quotes(outfile);
   err = zshell("ack","gs -q -sDEVICE=jpeg -r300 -dFitPage "                     //  300 dpi 
                      "-o \"%s\" \"%s\" ", f2, f1);
   zfree(f1);
   zfree(f2);

   if (err) goto returnx;
   
   strcpy(pp,"-1.jpg");                                                          //  open 1st output file
   f_open(outfile,0,0,1,0);
   m_viewmode(0,"G");                                                            //  show gallery at file position
   goto returnx;

photoshop:

   err = zshell(0,"which convert >/dev/null 2>&1");                              //  check imagemagick installed
   if (err) {
      zmessageACK(Mwin,"imagemagick must be installed");
      goto returnx;
   }

   outfile = zstrdup(infile,"convert adobe",8);                                  //  file.psd  >>  file.jpg
   pp = strrchr(outfile,'.');
   if (! pp) goto returnx;
   strcpy(pp,".jpg");                                                            //  output file:  file.jpg

   f1 = zescape_quotes(infile);                                                    //  23.4
   f2 = zescape_quotes(outfile);
   err = zshell("ack","convert \"%s\" \"%s\" ",f1,f2);
   zfree(f1);
   zfree(f2);

   if (err) goto returnx;
   
   f_open(outfile,0,0,1,0);
   m_viewmode(0,"G");                                                            //  show gallery at file position

returnx:

   if (zd) zdialog_free(zd);
   if (infile) zfree(infile);
   if (outfile) zfree(outfile);
   return;
}


/********************************************************************************/

//  set current file as desktop wallpaper (GNOME only)

void m_wallpaper(GtkWidget *, ch *)
{
   ch *  key = "gsettings set org.gnome.desktop.background";
   ch *  id = "picture-uri";
   ch *  cf;

   F1_help_topic = "set wallpaper";

   Plog(1,"m_wallpaper \n");

   if (! curr_file) return;
   cf = zescape_quotes(curr_file);                                               //  23.4
   zshell("ack","%s %s \"file://%s\" ",key,id,cf);
   zfree(cf);

   return;
}


/********************************************************************************/

//  print image file

void m_print(GtkWidget *, ch *)                                                  //  use GTK print
{
   int print_addgrid(PXB *Ppxb);

   int      pstat;
   ch       *printfile = 0;
   PXB      *Ppxb = 0;
   GError   *gerror = 0;

   F1_help_topic = "print";

   Plog(1,"m_print \n");

   if (FGWM != 'F' && FGWM != 'G') return;

   if (clicked_file) Ppxb = PXB_load(clicked_file,1);                            //  clicked thumbnail
   else if (E0pxm) Ppxb = PXM_PXB_copy(E0pxm);                                   //  current edited file
   else if (curr_file) Ppxb = PXB_load(curr_file,1);                             //  current file
   clicked_file = 0;
   if (! Ppxb) return;

   print_addgrid(Ppxb);                                                          //  add grid lines if wanted

   printfile = zstrdup(temp_folder,"print",20);                                  //  make temp print file:
   strcat(printfile,"/printfile.bmp");                                           //    /.../fotoxx-nnnnnn/printfile.bmp

   pstat = gdk_pixbuf_save(Ppxb->pixbuf,printfile,"bmp",&gerror,null);           //  bmp much faster than png
   if (! pstat) {
      zmessageACK(Mwin,"error: %s",gerror->message);
      PXB_free(Ppxb);
      zfree(printfile);
      return;
   }

   print_image_file(Mwin,printfile);

   PXB_free(Ppxb);
   zfree(printfile);
   return;
}


//  add grid lines to print image if wanted

int print_addgrid(PXB *Ppxb)
{
   uint8    *pix;
   int      px, py, ww, hh;
   int      startx, starty, stepx, stepy;

   if (! gridsettings[GON]) return 0;                                            //  grid lines off

   ww = Ppxb->ww;
   hh = Ppxb->hh;

   stepx = gridsettings[GXS];                                                    //  space between grid lines
   stepy = gridsettings[GYS];

   stepx = stepx / Mscale;                                                       //  window scale to image scale
   stepy = stepy / Mscale;

   if (gridsettings[GXC])                                                        //  if line counts specified,
      stepx = ww / (1 + gridsettings[GXC]);                                      //    set spacing accordingly
   if (gridsettings[GYC])
      stepy = hh / ( 1 + gridsettings[GYC]);

   startx = stepx * gridsettings[GXF] / 100;                                     //  variable offsets
   if (startx < 0) startx += stepx;

   starty = stepy * gridsettings[GYF] / 100;
   if (starty < 0) starty += stepy;

   if (gridsettings[GX]) {                                                       //  x-grid enabled
      for (px = startx; px < ww-6; px += stepx)
      for (py = 0; py < hh; py++)
      {
         pix = PXBpix(Ppxb,px,py);                                               //  wwwbbb  fat grid lines
         memset(pix,255,9);                                                      //  wwwbbb
         memset(pix+9,0,9);                                                      //  wwwbbb
      }                                                                          //  ......
   }

   if (gridsettings[GY]) {                                                       //  y-grid enabled
      for (py = starty; py < hh-6; py += stepy)
      for (px = 0; px < ww-3; px += 3)
      {
         pix = PXBpix(Ppxb,px,py);                                               //  www...
         memset(pix,255,9);                                                      //  www...
         pix = PXBpix(Ppxb,px,py+1);                                             //  www...
         memset(pix,255,9);                                                      //  bbb...
         pix = PXBpix(Ppxb,px,py+2);                                             //  bbb...
         memset(pix,255,9);                                                      //  bbb...
         pix = PXBpix(Ppxb,px,py+3);
         memset(pix,0,9);
         pix = PXBpix(Ppxb,px,py+4);
         memset(pix,0,9);
         pix = PXBpix(Ppxb,px,py+5);
         memset(pix,0,9);
      }
   }

   return 1;
}


//  print calibrated image
//  menu function calling print_calibrated() in f.tools.cc

void m_print_calibrated(GtkWidget *, ch *)
{
   Plog(1,"m_print_calibrated \n");
   if (FGWM != 'F' && FGWM != 'G') return;
   print_calibrated();
   return;
}


/********************************************************************************/

//  Quit menu function
//  Click on window [x] button (delete event) also comes here.
//  User can bail out if unsaved edit, open dialog, running function, or second thoughts.

void m_quit(GtkWidget *, ch *)
{
   zdialog  *zd;
   int      yn, nn, ii;
   ch       quitmess[100] = "";

   Plog(1,"m_quit \n");

   if (Fblock("quit","edits")) return;                                           //  unsaved edits, user bailout

   if (zfuncs::zdialog_busy) {
      for (int ii = 0; ii < zfuncs::zdialog_count; ii++) {                       //  loop pending dialogs
         zd = zfuncs::zdialog_list[ii];
         if (strmatch(zd->title,"userguide")) continue;                          //  omit this dialog
         zdialog_show(zd,1);                                                     //  un-hide minimized dialog
         zmainsleep(1);                                                          //  next dialog may hide it again         23.3
         yn = zmessageYN(Mwin,"Kill active dialog? %s",zd->title);               //  allow user bailout
         if (! yn) return;
         zdialog_send_event(zd,"escape");                                        //  kill dialog                           23.1
         zmainsleep(0.1);
      }
   }
   
   if (Fblock(0,"blocked quiet")) {                                              //  blocking function active?             23.3
      yn = zmessageYN(Mwin,"Kill blocking function %s?",Fblock_func);
      if (! yn) return;
   }

   if (Ffuncbusy) {                                                              //  something is still running
      yn = zmessageYN(Mwin,"Kill busy function?");
      if (! yn) return;
      Fescape = 1;                                                               //  tell it to exit
      for (ii = 0; ii < 20; ii++) {                                              //  allow 2 secs.
         if (Fblock(0,"blocked quiet") || Ffuncbusy) 
            zmainsleep(0.1);
         else break;
      }

      if (ii == 20) Plog(0,"busy function will be killed \n");                   //  still running
   }

   if (! Faskquit) goto quitxx;                                                  //  no "ask to quit" option

   strncatv(quitmess,100,"  ","Quit Fotoxx?","  ",0);
   nn = zdialog_choose2(Mwin,"parent",quitmess,"Yes","No",0);                    //  get button or KB key input
   if (nn == 1) goto quitxx;                                                     //  [Yes] button
   if (nn == 2 || nn == -1) return;                                              //  [No] or [x] button

   nn = toupper(nn);
   if (nn == "No"[0]) return;                                                    //  KB input matches [No] button
   if (nn == "Yes"[0]) goto quitxx;                                              //  KB input matches [Yes] button

   for (ii = 0; ii < Nkbsu; ii++)                                                //  test if KB key is "Quit" shortcut
      if (nn == kbsutab[ii].key[0]) break;
   if (ii == Nkbsu) return;                                                      //  not a KB shortcut
   if (! strmatchcase(kbsutab[ii].menu,"Quit")) return;                          //  not Quit

quitxx:
   Plog(1,"Quit Fotoxx\n");
   quitxx();                                                                     //  does not return
}


//  used also for main window destroy event
//  does not return

void quitxx()
{
   Fshutdown = 1;
   Fescape = 1;                                                                  //  stop long-running funcs
   gtk_window_get_position(MWIN,&mwgeom[0],&mwgeom[1]);                          //  save main window position and size
   gtk_window_get_size(MWIN,&mwgeom[2],&mwgeom[3]);                              //    for next session
   zdialog_inputs("save");                                                       //  save dialog inputs
   zdialog_geometry("save");                                                     //  save dialogs position/size
   gallery_memory("save");                                                       //  save recent gallery positions
   free_resources();                                                             //  free memory
   showz_docfile(0,0,"quit");                                                    //  close userguide window
   zshell(0,"rm -R -f %s",temp_folder);                                          //  delete temp files
   save_params();                                                                //  save state for next session
   fflush(null);                                                                 //  flush stdout, stderr
   zexit(0,"Fotoxx exit");
}


/********************************************************************************/

//  uninstall fotoxx from either /usr/bin etc. or $HOME/.local/bin etc.

void m_uninstall(GtkWidget *, ch *)                                              //  23.4
{
   ch    *command = 0;
   ch    progexe[300];
   int   cc, YN;

   F1_help_topic = "help menu";
   
   Plog(1,"m_uninstall \n");
   
   cc = readlink("/proc/self/exe",progexe,300);                                  //  get own program path
   if (cc <= 0) {
      zmessageACK(Mwin,"cannot get /proc/self/exe");
      return;
   }
   progexe[cc] = 0;

   YN = zmessageYN(Mwin,"fotoxx is installed at %s \n"
                        "proceed to delete fotoxx?",progexe);
   if (YN != 1) return;

   YN = zmessageYN(Mwin,"Also delete fotoxx user data at %s",get_zhomedir());
   if (YN == 1) zshell("log","rm -f -d -R %s",get_zhomedir());
   zsleep(2);

   if (strstr(progexe,"/usr/bin")) 
   {
      command = "sudo rm -f /usr/share/man/man1/fotoxx* \n"
                "sudo rm -R -f /usr/share/fotoxx \n"
                "sudo rm -R -f /usr/share/doc/fotoxx \n"
                "sudo rm -R -f /usr/share/metainfo/*fotoxx* \n"
                "sudo rm -R -f /usr/share/applications/fotoxx* \n"
                "sleep 2 \n"
                "sudo rm -f /usr/bin/fotoxx"; 
      runroot(command);
      zsleep(2);
      exit(0);
   }

   if (strstr(progexe,"/.local/bin"))
   {
      command = "find $HOME/.local -path \"*fotoxx*\" -type l,f,d -delete";
      zshell("log",command);
      zsleep(2);
      exit(0);
   }

   return;
}


/********************************************************************************/

//  help menu function

void m_help(GtkWidget *, ch *menu)
{
   Plog(1,"m_help \n");
   
   F1_help_topic = "help menu";
   
   if (! *menu) {
      showz_docfile(Mwin,"userguide","help menu");
      return;
   }

   if (strmatch(menu,"User Guide"))
      showz_docfile(Mwin,"userguide",0);

   if (strmatch(menu,"Video Tutorial"))
      showz_html("https://youtu.be/F5Kwnr4TrwM");

   if (strmatch(menu,"All Edit Funcs"))
      showz_docfile(Mwin,"userguide","edit functions");
   
   if (strmatch(menu,"Outboard Programs"))
      check_outboards(1);
   
   if (strmatch(menu,"Log File"))
      if (! isatty(fileno(stdin)))                                               //  23.4
         showz_logfile(Mwin);
      
   if (strmatch(menu,"Command Params"))
      showz_docfile(Mwin,"userguide","command parameters");

   if (strmatch(menu,"Changelog"))
      showz_textfile("doc","changelog",Mwin);

   if (strmatch(menu,"Copyright"))
      showz_textfile("doc","copyright",Mwin);

   if (strmatch(menu,"GNU License"))
      showz_textfile("doc","GNU License",Mwin);

   if (strmatch(menu,"Pareto License"))
      showz_textfile("doc","Pareto License",Mwin);

   if (strmatch(menu,"Source Build"))
      showz_textfile("doc","Source Build",Mwin);

   if (strmatch(menu,"Privacy"))
      showz_docfile(Mwin,"userguide","privacy");

   if (strmatch(menu,"About Fotoxx"))
      zabout(Mwin);
   
   if (strmatch(menu,"Home Page"))
      showz_html("https://kornelix.net");

   if (strmatch(menu,"Uninstall"))
      m_uninstall(0,0);

   return;
}


/********************************************************************************/

//  save (modified) image file to disk

void m_file_save(GtkWidget *, ch *menu)
{
   int  file_save_dialog_event(zdialog *zd, ch *event);

   ch             *pp;
   zdialog        *zd;

   F1_help_topic = "file save";

   Plog(1,"m_file_save \n");

   if (FGWM != 'F' && FGWM != 'G') return;

   if (! curr_file) {
      if (zd_filesave) zdialog_free(zd_filesave);
      zd_filesave = 0;
      return;
   }
   
   if (strmatch(curr_file_type,"other"))                                         //  if unsupported type, use jpg
      strcpy(curr_file_type,"jpg");

/***
          _______________________________________________
         |        Save Image File                        |
         |                                               |
         |   filename.jpg                                |
         |                                               |
         |  [new version] save as new file version       |
         |  [new file ...] save as new file name or type |
         |  [replace file] replace file (OVERWRITE)      |
         |                                               |
         |                                    [cancel]   |
         |_______________________________________________|

***/

   if (! zd_filesave)
   {
      zd = zdialog_new("Save Image File",Mwin,"Cancel",null);
      zd_filesave = zd;

      zdialog_add_widget(zd,"hbox","hbf","dialog",0,"space=3");
      zdialog_add_widget(zd,"label","filename","hbf",0,"space=10");

      zdialog_add_widget(zd,"hbox","hb0","dialog");
      zdialog_add_widget(zd,"vbox","vb1","hb0",0,"space=3|homog");
      zdialog_add_widget(zd,"vbox","vb2","hb0",0,"space=5|homog");

      zdialog_add_widget(zd,"button","newvers","vb1","new version");
      zdialog_add_widget(zd,"button","newfile","vb1","new file ...");
      zdialog_add_widget(zd,"button","replace","vb1","replace file");

      zdialog_add_widget(zd,"hbox","hb1","vb2");
      zdialog_add_widget(zd,"hbox","hb2","vb2");
      zdialog_add_widget(zd,"hbox","hb3","vb2");

      zdialog_add_widget(zd,"label","labvers","hb1","save as new file version");
      zdialog_add_widget(zd,"label","labfile","hb2","save as new file name or type");
      zdialog_add_widget(zd,"icon","warning","hb3","warning.png","size=30");
      zdialog_add_widget(zd,"label","labrepl","hb3","replace old file (OVERWRITE)","space=3");
   }

   zd = zd_filesave;

   pp = strrchr(curr_file,'/');
   if (pp) zdialog_stuff(zd,"filename",pp+1);

   zdialog_run(zd,file_save_dialog_event,"mouse");
   return;
}


//  dialog event and completion function

int file_save_dialog_event(zdialog *zd, ch *event)
{
   ch       *RP = 0, *newfilename = 0;
   ch       temp[8];
   int      err;
   
   if (zd->zstat) goto retx;                                                     //  cancel
   if (! zstrstr("newvers newfile replace",event)) return 1;                     //  ignore other events
   if (Fblock(0,"blocked")) return 1;                                            //  check nothing pending

   if (strmatch(event,"newvers"))                                                //  save as new version
   {
      RP = f_realpath(curr_file);                                                //  use real path
      if (! RP) {
         zmessageACK(Mwin,"file not found: %s",curr_file);
         goto retx;
      }
      
      newfilename = file_new_version(RP);                                        //  get next avail. file version name
      if (! newfilename) goto retx;
      
      if (strmatch(curr_file_type,"RAW"))                                        //  if RAW file, save as 16-bit TIFF
         err = f_save(newfilename,"tif",16,0,1);
      else if (strcasestr("jp2 heic avif webp",curr_file_type))                  //  save these types as jpg               23.3
         err = f_save(newfilename,"jpg",8,0,1);
      else if (strcasestr("pdf ps eps",curr_file_type))                          //  if pdf/ps/eps, save as jpg
         err = f_save(newfilename,"jpg",8,0,1);
      else 
         err = f_save(newfilename,curr_file_type,curr_file_bpc,0,1);             //  save with same file type

      if (! err) f_open_saved();                                                 //  open saved file with edit hist
      goto retx;
   }

   if (strmatch(event,"replace"))                                                //  replace current file
   {
      if (strcasestr("RAW jp2 heic avif webp pdf ps eps",curr_file_type)) {      //  23.3
         strToUpper(temp,curr_file_type);
         zmessageACK(Mwin,"cannot replace %s file",temp);
         goto retx;
      }

      RP = f_realpath(curr_file);                                                //  use real path
      if (! RP) {
         zmessageACK(Mwin,"file not found: %s",curr_file);
         goto retx;
      }

      err = f_save(RP,curr_file_type,curr_file_bpc,0,1);                         //  save with current edits
      if (! err) f_open_saved();                                                 //  open saved file with edit hist
      goto retx;
   }

   if (strmatch(event,"newfile"))
      err = f_save_as();                                                         //  save-as file chooser dialog

retx:
   if (RP) zfree(RP);
   if (newfilename) zfree(newfilename);
   zdialog_free(zd);                                                             //  kill dialog
   zd_filesave = 0;
   return 1;
}


//  menu entry for KB shortcut Save File (replace)

void m_file_save_replace(GtkWidget *, ch *menu)
{
   int      err;
   ch       *RP;

   Plog(1,"m_file_save_replace \n");

   RP = f_realpath(curr_file);                                                   //  use real path
   if (! RP) {
      zmessageACK(Mwin,"file not found: %s",curr_file);
      return;
   }

   err = f_save(RP,curr_file_type,curr_file_bpc,0,1);                            //  save file
   if (! err) f_open_saved();                                                    //  open saved file with edit hist
   zfree(RP); 
   return;
}


//  menu entry for KB shortcut Save File Version

void m_file_save_version(GtkWidget *, ch *menu)
{
   ch       *newfilename = 0, *RP = 0;
   ch       newfiletype[8];
   int      err;

   Plog(1,"m_file_save_version \n");

   RP = f_realpath(curr_file);                                                   //  use real path
   if (! RP) {
      zmessageACK(Mwin,"file not found: %s",curr_file);
      return;
   }

   strncpy0(newfiletype,curr_file_type,6);
   newfilename = file_new_version(RP);                                           //  get next avail. file version name
   if (! newfilename) goto retx;
   err = f_save(newfilename,newfiletype,curr_file_bpc,0,1);                      //  save file
   if (! err) f_open_saved();                                                    //  open saved file with edit hist

retx:
   if (RP) zfree(RP);
   if (newfilename) zfree(newfilename);
   return;
}


/********************************************************************************/

//  Save current image to specified file (same or new).
//  Update image index file and thumbnail file.
//  Set f_save_type, f_save_bpc, f_save_size
//  Returns 0 if OK, else +N.
//  If Fack is true, failure will cause a popup ACK dialog.

int f_save(ch *outfile, ch *outype, int outbpc, int qual, int Fack)
{
   ch       *metakey[5], *metadata[5];
   ch       *ppv[1], *tempfile, *pext, *outfile2;
   int      nkeys, err, cc1, cc2, cc3, ii;
   int      Fmod, Fcopy, Ftransp, Fnewfile;
   int      px, py;
   void     (*editfunc)(GtkWidget *, ch *);
   STATB    statB;

   ch       *warnalpha = "Transparency map will be lost.\n"
                         "save to TIFF or PNG file to retain.";
   
   if (! curr_file) return 1;

   if (strmatch(outype,"RAW")) {                                                 //  disallow saving as RAW type
      zmessageACK(Mwin,"cannot save as RAW type");
      return 1;
   }

   Fmod = 0;
   editfunc = 0;
   if (! qual) qual = jpeg_def_quality;                                          //  use default jpeg compression

   if (CEF) {                                                                    //  edit function active
      if (CEF->Fmods) Fmod = 1;                                                  //  active edits pending
      editfunc = CEF->menufunc;                                                  //  save menu function for restart
      if (CEF->zd) zdialog_send_event(CEF->zd,"done");                           //  tell it to finish
      if (CEF) return 1;                                                         //  failed (HDR etc.)
   }
   
   if (URS_pos > 0 && URS_saved[URS_pos] == 0) Fmod = 1;                         //  completed edits not yet saved
   
   if (strmatch(outfile,curr_file)) Fnewfile = 0;                                //  replace current file
   else Fnewfile = 1;                                                            //  new file (name/.ext/location)
   
   if (! Fnewfile) {
      err = access(outfile,W_OK);                                                //  test file can be written by me
      if (err) {
         zmessageACK(Mwin,"%s: %s","no write permission",outfile);
         return 1;
      }
   }

   outfile2 = f_realpath(outfile);                                               //  resolve symlinks in file path
   if (! outfile2) {
      zmessageACK(Mwin,"file not found: %s",outfile);
      return 2;
   }   
   
   if (Fnewfile) {                                                               //  if new file, force .ext
      pext = strrchr(outfile2,'/');                                              //    to one of: .jpg .png .tif
      if (pext) pext = strrchr(pext,'.');
      if (! pext) pext = outfile2 + strlen(outfile2);
      pext[0] = '.';
      strcpy(pext+1,outype);
   }

   if (! Fnewfile && ! Fmod) {                                                   //  no edit changes to file
      Fcopy = 1;                                                                 //  direct copy to output file?
      if (E0pxm) Fcopy = 0;                                                      //  no, non-edit change (upright)
      if (curr_file_bpc != outbpc) Fcopy = 0;                                    //  no, BPC change
      if (qual < jpeg_def_quality) Fcopy = 0;                                    //  no, higher compression wanted
      if (Fcopy) {
         err = cp_copy(curr_file,outfile2);                                      //  copy unchanged file to output
         if (! err) goto updateindex;
         zmessageACK(Mwin,strerror(err));
         zfree(outfile2);
         return 1;
      }
   }
   
   if (! E0pxm) {
      E0pxm = PXM_load(curr_file,1);                                             //  no prior edits, load image file
      if (! E0pxm) {
         zfree(outfile2);
         return 1;
      }
   }

   tempfile = zstrdup(outfile2,"file-save",20);                                  //  temp file in same folder
   strcat(tempfile,"-temp.");
   strcat(tempfile,outype);
   
   if (E0pxm->nc == 4 && ! strstr("tif png",outype))                             //  alpha channel will be lost
   {
      Ftransp = 0;
      for (py = 2; py < E0pxm->hh-2; py += 2)                                    //  ignore extreme edges
      for (px = 2; px < E0pxm->ww-2; px += 2)
         if (PXMpix(E0pxm,px,py)[3] < 250) {
            Ftransp = 1;
            goto breakout;
         }
   breakout:                                                                     //  warn transparency lost
      if (Ftransp) {
         ii = zdialog_choose(Mwin,"mouse",warnalpha,"save anyway","Cancel",null);
         if (ii != 1) {
            remove(tempfile);                                                    //  user canceled
            zfree(tempfile);
            zfree(outfile2);
            return 0;
         }
      }
   }

   if (! strstr("tif png",outype)) outbpc = 8;                                   //  force bpc 8 if 16 not supported
   
   err = PXM_save(E0pxm,tempfile,outbpc,qual,Fack);
   if (err) {
      remove(tempfile);                                                          //  failure, clean up
      zfree(tempfile);
      zfree(outfile2);
      return 1;
   }
   
   cc1 = 0;                                                                      //  meta_edithist[] text cc

   if (Fmeta_edithist) cc1 = meta_edithist_cc0;                                  //  meta_edithist at file open
   else {
      metakey[0] = (ch *) meta_edithist_key;
      meta_get1(curr_file,metakey,ppv,1);                                        //  get edits made before 
      if (ppv[0]) {                                                              //    this file was opened
         strncpy0(meta_edithist,ppv[0],metadataXcc-2);
         zfree(ppv[0]);
         cc1 = strlen(meta_edithist);
         meta_edithist_cc0 = cc1;                                                //  edit hist cc when file was opened
      }                                                                          //  (not updated for new edits) 
      Fmeta_edithist = 1;                                                        //  history is loaded
   }

   for (ii = 1; ii <= URS_pos; ii++)                                             //  append edits from undo/redo stack
   {                                                                             //  (omit index 0 = initial image)
      cc2 = strlen(URS_menu[ii]);
      cc3 = strlen(URS_parms[ii]);
      if (cc1 + 7 + cc2 + cc3 + 5 > metadataXcc) break;                          //  no more room
      strcpy(meta_edithist+cc1,Frelease);
      cc1 += strlen(Frelease);                                                   //  append "Fotoxx-nn.n:Menu:parms|"
      meta_edithist[cc1] = ':';                                                  //  colon not semicolon                   23.2
      cc1++;
      strcpy(meta_edithist+cc1,URS_menu[ii]);
      cc1 += cc2;
      if (cc3) {
         strcpy(meta_edithist+cc1,": ");
         cc1 += 2;
         strcpy(meta_edithist+cc1,URS_parms[ii]);
         cc1 += cc3;
      }
      meta_edithist[cc1++] = '|';
   }

   if (CEF && CEF->Fmods)                                                        //  append active edit function 
   {
      cc2 = strlen(CEF->menuname);
      cc3 = strlen(CEF->edit_hist);                                              //  23.0
      if (cc1 + 7 + cc2 + cc3 + 5 < metadataXcc) {
         strcpy(meta_edithist+cc1,"Fotoxx:");
         cc1 += 7;                                                               //  append "Fotoxx:Menu:parms|"
         strcpy(meta_edithist+cc1,CEF->menuname);
         if (cc3) {
            cc1 += cc2;
            strcpy(meta_edithist+cc1,": ");
            cc1 += 2;
            strcpy(meta_edithist+cc1,CEF->edit_hist);
            cc1 += cc3;
         }
         meta_edithist[cc1++] = '|';
      }
   }
   
   meta_edithist[cc1] = 0;
   
   nkeys = 0;

   if (Fupright) {                                                               //  image was uprighted
      metakey[nkeys] = meta_orientation_key;                                     //  remove metadata orientation 
      metadata[nkeys] = "";
      nkeys++;
   }
   
   if (Flevel) {                                                                 //  image was levelled
      metakey[nkeys] = meta_rollangle_key;                                       //  remove metadata rollangle
      metadata[nkeys] = "0";                                                     //  "" fails, use "0"
      nkeys++;
   }

   if (cc1) {                                                                    //  prior and/or curr. edit history
      metakey[nkeys] = meta_edithist_key;                                        //  add to metadata
      metadata[nkeys] = meta_edithist;
      nkeys++;
   }

   err = meta_copy(curr_file,tempfile,metakey,metadata,nkeys);                   //  copy all metadata to
   if (err) {                                                                    //    temp file with above revisions 
      if (Fack) zmessageACK(Mwin,"Unable to copy metadata");
      else Plog(1,"Unable to copy metadata \n");
   }
   
   err = rename(tempfile,outfile2);                                              //  rename temp file to output file
   if (err) {
      if (Fack) zmessageACK(Mwin,strerror(err));
      remove(tempfile);                                                          //  delete temp file
      zfree(tempfile);
      zfree(outfile2);
      return 2;                                                                  //  could not save
   }
   
   zfree(tempfile);                                                              //  free memory

   for (ii = 0; ii <= URS_pos; ii++)                                             //  mark all prior edits as saved
      URS_saved[ii] = 1;

updateindex:

   update_thumbfile(outfile2);                                                   //  refresh thumbnail file

   load_filemeta(outfile2);                                                      //  load output file metadata

   if (Fmod) {
      set_meta_wwhh(E0pxm->ww,E0pxm->hh);                                        //  set meta_wwhh in memory
      save_filemeta(outfile2);                                                   //  includes update_image_index() 
   }
   else update_image_index(outfile2);                                            //  add to image index

   if (samefolder(outfile2,navi::galleryname)) {                                 //  if saving into current gallery
      gallery(curr_file,"init",0);                                               //  update curr. gallery list
      gallery(0,"sort",-2);                                                      //  recall sort and position
      set_mwin_title();                                                          //  update posn, count in title
   }

   add_recent_file(outfile2);                                                    //  first in recent files list

   zstrcopy(f_save_file,outfile2,"f_save");                                      //  for f_open_saved()

   err = stat(curr_file,&statB);                                                 //  get current file permissions
   chmod(outfile2,statB.st_mode);                                                //  set permissions for output file

   stat(outfile2,&statB);                                                        //  set output file size, type, bit depth
   f_save_size = statB.st_size;
   strcpy(f_save_type,outype);
   f_save_bpc = outbpc;
   
   if (strmatch(curr_file,outfile2))                                             //  curr. file is being overwritten
      if (zd_metaview) meta_view(0);                                             //  view meta dialog active

   if (editfunc) editfunc(0,0);                                                  //  restart edit function

   zfree(outfile2);
   return 0;
}


/********************************************************************************/

//  save (modified) image to new file name or type
//  confirm if overwrite of existing file
//  returns 0 if OK, 1 if cancel or error

GtkWidget      *saveas_fchooser;

int f_save_as()
{
   int  f_save_as_dialog_event(zdialog *zd, ch *event);

   zdialog        *zd;
   ch             *save_folder = 0;
   ch             *type;
   ch             *newfile, *fname;
   ch             *outfile = 0, *outfile2 = 0, *pp, *pext;
   ch             permissions[100];
   int            ii, zstat, err, bpc;
   int            jpgqual = jpeg_def_quality;
   int            mkcurr = 0;
   STATB          statB;
   mode_t         mode;

/***
       _____________________________________________________
      |   Save as New File Name or Type                     |
      |   ________________________________________________  |
      |  |                                                | |
      |  |       file chooser dialog                      | |
      |  |                                                | |
      |  |                                                | |
      |  |                                                | |
      |  |                                                | |
      |  |                                                | |
      |  |________________________________________________| |
      |                                                     |
      |  (o) tif  (o) png  (o) jpg  [90] jpg quality        |                    //  sep. file type and color depth
      |  color depth: (o) 8-bit  (o) 16-bit                 |
      |  [x] open the new file (will become current file)   |
      |  permissions: [__________________________] [change] |
      |                                                     |
      |                                    [save] [cancel]  |
      |_____________________________________________________|

***/

   zd = zdialog_new("Save as new file name or type",Mwin,"Save","Cancel",null);

   zdialog_add_widget(zd,"hbox","hbfc","dialog",0,"expand");
   saveas_fchooser = gtk_file_chooser_widget_new(GTK_FILE_CHOOSER_ACTION_SAVE);
   gtk_container_add(GTK_CONTAINER(zdialog_gtkwidget(zd,"hbfc")),saveas_fchooser);

   zdialog_add_widget(zd,"vbox","space","dialog",0,"space=3");

   zdialog_add_widget(zd,"hbox","hbft","dialog");
   zdialog_add_widget(zd,"radio","tif","hbft","tif","space=4");
   zdialog_add_widget(zd,"radio","png","hbft","png","space=4");
   zdialog_add_widget(zd,"radio","jpg","hbft","jpg","space=2");
   zdialog_add_widget(zd,"zspin","jpgqual","hbft","10|100|1|90","size=3");
   zdialog_add_widget(zd,"label","labqual","hbft","jpg quality","space=6");

   zdialog_add_widget(zd,"hbox","hbcd","hbft");
   zdialog_add_widget(zd,"label","space","hbcd","","space=8");
   zdialog_add_widget(zd,"label","labdepth","hbcd","color depth:","space=3");
   zdialog_add_widget(zd,"radio","8-bit","hbcd","8-bit","space=4");
   zdialog_add_widget(zd,"radio","16-bit","hbcd","16-bit","space=4");

   zdialog_add_widget(zd,"hbox","hbmc","dialog");
   zdialog_add_widget(zd,"check","mkcurr","hbmc",0,"space=8");
   zdialog_add_widget(zd,"label","labmc","hbmc","open the new file ");
   zdialog_add_widget(zd,"label","labmc","hbmc","(will become current file)");

   zdialog_add_widget(zd,"hbox","hbperm","dialog");
   zdialog_add_widget(zd,"label","labperm","hbperm","permissions:","space=8");
   zdialog_add_widget(zd,"label","permissions","hbperm");
   zdialog_add_widget(zd,"button","change","hbperm","Change","space=8");

   save_folder = f_realpath(curr_file);                                          //  use real path
   if (! save_folder) save_folder = zstrdup(curr_file,"file-save");
   pp = strrchr(save_folder,'/');
   if (pp) *pp = 0;
   gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(saveas_fchooser),save_folder);
   zfree(save_folder);

   newfile = zstrdup(curr_file,"file-save");
   fname = strrchr(newfile,'/') + 1;
   gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(saveas_fchooser),fname);
   zfree(newfile);

   zdialog_stuff(zd,"tif",0);                                                    //  no file type selected
   zdialog_stuff(zd,"png",0);
   zdialog_stuff(zd,"jpg",0);
   zdialog_stuff(zd,"8-bit",0);
   zdialog_stuff(zd,"16-bit",0);

   zdialog_stuff(zd,"jpgqual",jpeg_def_quality);                                 //  default jpeg quality, user setting

   if (strmatch(curr_file_type,"tif")) {                                         //  if curr. file type is tif,
      zdialog_stuff(zd,"tif",1);                                                 //    set corresp. type and bit depth
      if (curr_file_bpc == 16)
         zdialog_stuff(zd,"16-bit",1);
      else zdialog_stuff(zd,"8-bit",1);
   }

   else if (strmatch(curr_file_type,"png")) {                                    //  same for png
      zdialog_stuff(zd,"png",1);
      if (curr_file_bpc == 16)
         zdialog_stuff(zd,"16-bit",1);
      else zdialog_stuff(zd,"8-bit",1);
   }

   else {                                                                        //  same for jpg
      zdialog_stuff(zd,"jpg",1);
      zdialog_stuff(zd,"8-bit",1);
   }

   zdialog_stuff(zd,"mkcurr",1);                                                 //  "make current" is default

   err = stat(curr_file,&statB);                                                 //  current file permissions
   conv_permissions(statB.st_mode,permissions);                                  //   >> default new file permissions
   zdialog_stuff(zd,"permissions",permissions);

   zdialog_resize(zd,700,500);
   zdialog_run(zd,f_save_as_dialog_event,"parent");

zdialog_wait:

   zstat = zdialog_wait(zd);
   if (zstat != 1) {                                                             //  user cancel
      zdialog_free(zd);
      return 1;
   }

   outfile2 = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! outfile2) {
      zd->zstat = 0;
      goto zdialog_wait;
   }
   
   zstrcopy(outfile,outfile2,"file-save",12);                                    //  add space for possible .vNN and .ext
   g_free(outfile2);

   type = "";
   bpc = -1;
   
   zdialog_fetch(zd,"tif",ii);                                                   //  get selected file type
   if (ii) type = "tif";

   zdialog_fetch(zd,"png",ii);
   if (ii) type = "png";

   zdialog_fetch(zd,"jpg",ii);
   if (ii) type = "jpg";
   
   zdialog_fetch(zd,"8-bit",ii);                                                 //  get selected color depth
   if (ii) bpc = 8;
   
   zdialog_fetch(zd,"16-bit",ii);
   if (ii) bpc = 16;
   
   if (! strstr("tif png jpg",type)) goto zdialog_wait;                          //  should not happen
   if (bpc != 8 && bpc != 16) goto zdialog_wait;
   if (strmatch(type,"jpg") && bpc != 8) goto zdialog_wait;

   zdialog_fetch(zd,"jpgqual",jpgqual);                                          //  jpeg compression level

   pext = strrchr(outfile,'/');                                                  //  locate file .ext
   if (pext) pext = strrchr(pext,'.');

   if (pext) {                                                                   //  validate .ext OK for type
      if (strmatch(type,"jpg") && ! strcasestr(".jpg .jpeg",pext)) *pext = 0;
      if (strmatch(type,"tif") && ! strcasestr(".tif .tiff",pext)) *pext = 0;
      if (strmatch(type,"png") && ! strcasestr(".png",pext)) *pext = 0;
   }

   if (! pext || ! *pext) {
      pext = outfile + strlen(outfile);                                          //  wrong or missing, add new .ext
      *pext = '.';                                                               //  NO replace .JPG with .jpg etc.
      strcpy(pext+1,type);
   }

   zdialog_fetch(zd,"mkcurr",mkcurr);                                            //  get make current option 

   if (regfile(outfile)) {                                                       //  check if file exists
      int yn = zmessageYN(Mwin,"Overwrite file? \n %s",outfile);                 //  confirm overwrite
      if (! yn) {
         zd->zstat = 0;
         goto zdialog_wait;
      }
   }
   
   zdialog_fetch(zd,"permissions",permissions,100);                              //  get permissions from dialog
   conv_permissions(permissions,mode);

   zdialog_free(zd);                                                             //  zdialog_free(zd);

   err = f_save(outfile,type,bpc,jpgqual,1);                                     //  save the file
   if (err) {
      zfree(outfile);
      return 1;
   }
   
   chmod(outfile,mode);                                                          //  set permissions
   
   if (samefolder(outfile,navi::galleryname)) {                                  //  if saving into current gallery
      gallery(outfile,"init",0);                                                 //    refresh gallery list
      gallery(0,"sort",-2);                                                      //    recall sort and position 
      curr_file_posn = file_position(curr_file,curr_file_posn);                  //    update curr. file position
      set_mwin_title();                                                          //    update window title (file count)
   }

   if (mkcurr) f_open_saved();                                                   //  open saved file with edit hist

   zfree(outfile);
   return 0;
}


//  dialog event and completion function

int f_save_as_dialog_event(zdialog *zd, ch *event)
{
   int      ii, err;
   ch       *filespec, *filename, *pp;
   ch       oldperms[100], newperms[100];
   ch       ext[4];
   
   if (zd->zstat) return 1;                                                      //  [ OK ] or [cancel]

   if (strmatch(event,"jpgqual")) {                                              //  if jpg quality edited, set jpg .ext
      zdialog_stuff(zd,"jpg",1);
      event = "jpg";
   }

   if (zstrstr("tif png jpg",event)) {                                           //  file type selection
      zdialog_stuff(zd,"tif",0);                                                 //  turn off all types
      zdialog_stuff(zd,"png",0);
      zdialog_stuff(zd,"jpg",0);
      zdialog_stuff(zd,event,1);                                                 //  turn on selected type
   }
   
   if (zstrstr("8-bit 16-bit",event)) {                                          //  color depth selection
      zdialog_stuff(zd,"8-bit",0);                                               //  turn off all depths
      zdialog_stuff(zd,"16-bit",0);
      zdialog_stuff(zd,event,1);                                                 //  turn on selected depth
   }
   
   zdialog_fetch(zd,"jpg",ii);                                                   //  if jpg, force 8-bit
   if (ii) {
      zdialog_stuff(zd,"16-bit",0);
      zdialog_stuff(zd,"8-bit",1);
   }
   
   zdialog_fetch(zd,"tif",ii);                                                   //  get chosen file type "tif" ...
   if (ii) strcpy(ext,"tif");                                                    //  set corresp. extension ".tif" ...
   zdialog_fetch(zd,"png",ii);
   if (ii) strcpy(ext,"png");
   zdialog_fetch(zd,"jpg",ii);
   if (ii) strcpy(ext,"jpg");

   filespec = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(saveas_fchooser));
   if (! filespec) return 1;
   filename = strrchr(filespec,'/');                                             //  revise file .ext in chooser dialog
   if (! filename) return 1;
   filename = zstrdup(filename+1,"file-save",6);
   pp = strrchr(filename,'.');
   if (! pp || strlen(pp) > 5) pp = filename + strlen(filename);
   *pp = '.';
   strcpy(pp+1,ext);
   gtk_file_chooser_set_current_name(GTK_FILE_CHOOSER(saveas_fchooser),filename);
   zfree(filename);
   g_free(filespec);
   
   if (strmatch(event,"change")) {                                               //  change saved file permissions
      zdialog_fetch(zd,"permissions",oldperms,100);
      err = set_permissions(zd->dialog,"",oldperms,newperms);
      if (! err) zdialog_stuff(zd,"permissions",newperms);
   }

   return 1;
}


/********************************************************************************/

//  get real path for filename possibly having symlinks
//  all folders in input filename path must exist
//  final file name need not exist (f_save() caller)
//  returns real path or null if error
//  returned path is subject for zfree()

ch * f_realpath(ch * infile)
{
   int      cc;
   ch       *pp, *RP, *outfile;
   ch       tempfile[XFCC];
   
   RP = realpath(infile,null);                                                   //  try full file path
   if (RP) {
      outfile = zstrdup(RP,"realpath");                                          //  OK
      free(RP);
      return outfile;
   }

   strcpy(tempfile,infile);                                                      //  fail, try folders only 
   pp = strrchr(tempfile + 2,'/');
   if (! pp) return 0;
   *pp = 0;
   RP = realpath(tempfile,null);
   *pp = '/';
   if (! RP) return 0;                                                           //  fail

   cc = strlen(pp);                                                              //  return folders/file
   outfile = zstrdup(RP,"realpath",cc+2);
   strcat(outfile,pp);
   free(RP);
   return outfile;
}


/********************************************************************************/

//  get the root name of an image file, without version and extension
//     /.../filename.V03.png  -->  /.../filename
//  returned root name has extra length to add .vNN.exxxxt
//  returned root name is subject for zfree()

ch * file_rootname(ch *file)
{
   ch    *rootname, *pp1, *pp2;
   
   rootname = zstrdup(file,"rootname",16);
   pp1 = strrchr(rootname,'/');                                                  //  find file .ext
   if (! pp1) pp1 = rootname;
   pp2 = strrchr(pp1,'.');                                                       //  pp2 --> .ext or ending null
   if (! pp2) pp2 = pp1 + strlen(pp1);

   if (pp2[-4] == '.' && pp2[-3] == 'v' &&                                       //  look for .vNN.ext
       pp2[-2] >= '0' && pp2[-2] <= '9' &&
       pp2[-1] >= '0' && pp2[-1] <= '9') pp2 -= 4;                               //  pp2 -->  .ext or .vNN.ext
   *pp2 = 0;                                                                     //  pp2 -->  new ending null

   return rootname;
}


/********************************************************************************/

//  return base name for given file = original file name /.../filename.ext
//  returns null if base file does not exist.
//  returned base name has extra length to add .vNN.exxxxt
//  returned base name is subject for zfree()

ch * file_basename(ch *file)
{
   ch       *rootname, **flist, *pp = 0;
   int      ii, cc, nf;
   
   flist = file_all_versions(file,nf);                                           //  get base file and all versions
   if (! nf) return 0;

   rootname = file_rootname(file);
   cc = strlen(rootname);
   zfree(rootname);
   
   for (ii = 0; ii < nf; ii++) 
   {
      pp = flist[ii];
      if (strmatchN(pp+cc,".v",2)) continue;
      if (pp[cc] == '.') break;
   }
   
   if (ii < nf) pp = zstrdup(pp,"basename",16);
   else pp = 0;
   
   for (ii = 0; ii < nf; ii++)
      zfree(flist[ii]);
   zfree(flist);
   
   return pp;
}


/********************************************************************************/

//  Get all versions of a given file name in sequence, including base file.
//  Returned list of file names is subject for zfree()
//  (each returned flist[*] member and flist itself)
//  returns null if nothing found

ch ** file_all_versions(ch *file, int &NF)
{
   int  file_comp_versions(ch *f1, ch *f2);

   ch       **flist1, **flist2, *rootname;
   ch       *pext, *pvers;
   int      ii, jj, err, cc1, cc2;
   
   rootname = file_rootname(file);                                               //  /.../fname.vNN.ext  -->  /.../fname
   if (! rootname) return 0;

   cc1 = strlen(rootname);                                                       //  file cc without .vNN.ext
   strcpy(rootname + cc1,".*");
   err = zfind(rootname,flist1,NF);                                              //  find /.../fname.*
   zfree(rootname);
   if (err || NF < 1) return 0;

   flist2 = (ch **) zmalloc(NF * sizeof(ch *),"file-vers");                      //  scan for valid versions

   for (ii = jj = 0; ii < NF; ii++)                                              //  validate fname.ext or fname.vNN.ext
   {
      if (image_file_type(flist1[ii]) > VIDEO) continue;                         //  not image file

      pext = strrchr(flist1[ii],'.');                                            //  get file .ext
      if (! pext) continue;
      cc2 = pext - flist1[ii];                                                   //  file cc without .ext
      if (cc2 > cc1) {                                                           //  version present?  .vNN.ext
         cc2 -= 4;
         pvers = flist1[ii] + cc2;
         if (strncmp(pvers,".v01",4) < 0) continue;                              //  not valid version
         if (strncmp(pvers,".v99",4) > 0) continue;
      }

      flist2[jj++] = flist1[ii];                                                 //  valid version, add to list
   }

   zfree(flist1);
   NF = jj;

   if (NF == 0) {
      zfree(flist2);
      return 0;
   }
   
   if (NF == 1) return flist2;

   if (NF > 99) {                                                                //  screen out problem file
      zmessageACK(Mwin,"file: %s \n exceed 99 versions",file);
      zfree(flist2);
      return 0;
   }

   HeapSort(flist2,NF,file_comp_versions);                                       //  sort in base, version order 
   return flist2;
}


int file_comp_versions(ch *f1, ch *f2)
{
   f1 = strrchr(f1,'.');                                                         //  find .ext in file f1
   if (! f1) return -1;                                                          //  f1 has no .ext
   f2 = strrchr(f2,'.');
   if (! f2) return +1;                                                          //  f2 has no .ext
   f1 -= 4;                                                                      //  point to '.v' in filename.vNN.ext
   if (! strmatchN(f1,".v",2)) return -1;                                        //  f1 not a version
   f2 -= 4;
   if (! strmatchN(f2,".v",2)) return +1;                                        //  f2 not a version
   return strcmp(f1,f2);                                                         //  compare versions
}


/********************************************************************************/

//  Get next available version  /.../filename.vNN.ext  for given file.
//  Returns  "/.../filename.v01.ext"  if no versions are found.
//  Returns null if bad file name or 99 versions already exist.
//  Returned file name is subject for zfree().

ch * file_new_version(ch *file)
{
   ch       *retname, *pp, pext[8];
   ch       **flist = 0;
   int      ii, cc, NF = 0, vers;
   
   pp = (ch *) strrchr(file,'/');                                                //  find .ext for file
   if (! pp) return 0;
   pp = strrchr(pp,'.');
   if (pp) strncpy0(pext,pp,8);
   else strcpy(pext,".jpg");
   
   if (strcasestr(".jp2 .heic .avif .webp",pext))                                //  replace these types with .jpg         23.4
      strcpy(pext,".jpg");
   
   flist = file_all_versions(file,NF);                                           //  get base file and all versions
   if (! NF) {                                                                   //  nothing exists
      retname = file_rootname(file);                                             //  get /.../filename
      retname = zstrdup(retname,"new-version",12);
      cc = strlen(retname);
      strcpy(retname+cc,".v01");                                                 //  return /.../filename.v01.ext
      strcpy(retname+cc+4,pext);
      return retname;
   }
   
   retname = zstrdup(flist[NF-1],"new-version",12);                              //  get last version found

   for (ii = 0; ii < NF; ii++)                                                   //  free memory
      zfree(flist[ii]);
   zfree(flist);

   pp = strrchr(retname,'/');                                                    //  find .ext
   if (! pp) return 0;
   pp = strrchr(pp,'.');
   if (! pp) {                                                                   //  none
      strcat(retname,"v.01.jpg");                                                //  return /.../filename.v01.jpg
      return retname;
   }
   
   if (strmatchN(pp-4,".v",2)) {                                                 //  find /.../filename.vNN.ext
      if (pp[-2] >= '0' && pp[-2] <= '9' &&                                      //                        |
          pp[-1] >= '0' && pp[-1] <= '9')                                        //                        pp
      {
         vers = 10 * (pp[-2] - '0') + pp[-1] - '0';
         if (vers >= 99) {
            zmessageACK(Mwin,"file: %s \n exceed 99 versions",file);
            return 0;
         }
         vers++;                                                                 //  add 1 to .vNN, leave same .ext
         pp[-2] = vers/10 + '0';
         pp[-1] = vers - 10 * (vers/10) + '0';
         return retname;                                                         //  return /.../filename.vNN.ext
      }
   }

   strncpy0(pext,pp,8);                                                          //  keep .ext
   strcpy(pp,".v01");                                                            //  return /.../filename.v01.ext
   strcpy(pp+4,pext);
   return retname;
}


/********************************************************************************/

//  Get the newest version  /.../filename.vNN.ext  for the given file.
//  Returns unversioned file  /.../filename.ext  if found by itself.
//  Returns null if bad file name.
//  Returned file name is subject for zfree().

ch * file_newest_version(ch *file)
{
   ch       *retname;
   ch       **flist;
   int      ii, NF;
   
   flist = file_all_versions(file,NF);                                           //  get base file and all versions
   if (! NF) return 0;

   retname = zstrdup(flist[NF-1],"new-version",12);                              //  get last version found

   for (ii = 0; ii < NF; ii++)                                                   //  free memory
      zfree(flist[ii]);
   zfree(flist);

   return retname;
}


/********************************************************************************/

//  Get the prior version  /.../filename.vNN.ext  for the given file.
//  Returns unversioned file  /.../filename.ext  if no prior .vNN found.
//  Returns null if bad file name or no prior version.

ch * file_prior_version(ch *file)
{
   ch       *retname;
   ch       **flist;
   int      ii, NF;
   
   flist = file_all_versions(file,NF);                                           //  get base file and all versions
   if (! NF) return 0;
   
   for (ii = 0; ii < NF; ii++)                                                   //  find input file in list
      if (strmatch(file,flist[ii])) break;
   
   if (ii == 0) return 0;                                                        //  input is base file (no prior)
   if (ii == NF) return 0;                                                       //  should not happen

   retname = zstrdup(flist[ii-1],"prior-version");                               //  prior file in list

   for (ii = 0; ii < NF; ii++)                                                   //  free memory
      zfree(flist[ii]);
   zfree(flist);

   return retname;
}


/********************************************************************************

   Find all image files within given path.

   int find_imagefiles(ch *folder, int flags, ch **&flist, int &NF)

      folder      folder path to search
      flags       sum of the following:
                     1  include image files (+RAW +video)
                     2  include thumbnails
                     4  include hidden files 
                     8  include folders
                    16  recurse folders
                    32  omit symlinks
      NF          count of files returned
      flist       list of files returned

   Returns 0 if OK, +N if error (errno is set).
   flist and flist[*] are subjects for zfree().

*********************************************************************************/

namespace find_imagefiles_names
{
   ch    **fif_filelist;                  //  list of filespecs returned
   int   fif_max;                         //  filelist slots allocated
   int   fif_count;                       //  filelist slots filled
}


int find_imagefiles(ch *folder, int flags, ch **&flist, int &NF, int Finit) 
{
   using namespace find_imagefiles_names;

   int         globflags, Fimages, Fthumbs, Fdirs, Frecurse, Fnolinks;
   int         err1, err2, cc;
   FTYPE       ftype;
   ch          *file, *mfolder, **templist;
   glob_t      globdata;
   STATB       statB;
   
   if (Finit) {                                                                  //  initial call
      spinlock(1);                                                               //  prevent reentrance
      fif_max = fif_count = 0;
   }

   globflags = GLOB_NOSORT;
   Fimages = Fthumbs = Fdirs = Frecurse = Fnolinks = 0;
   
   if (flags & 1) Fimages = 1;
   if (flags & 2) Fthumbs = 1;
   if (flags & 4) globflags += GLOB_PERIOD;
   if (flags & 8) Fdirs = 1;
   if (flags & 16) Frecurse = 1;
   if (flags & 32) Fnolinks = 1;
   
   if (Fdirs && ! Fimages && ! Fthumbs)
      globflags += GLOB_ONLYDIR;

   globdata.gl_pathc = 0;                                                        //  glob() setup
   globdata.gl_offs = 0;
   globdata.gl_pathc = 0;

   NF = 0;                                                                       //  empty output
   flist = 0;

   mfolder = zstrdup(folder,"find-files",4);                                     //  append /* to input folder
   strcat(mfolder,"/*");

   err1 = glob(mfolder,globflags,null,&globdata);                                //  find all files in folder
   if (err1) {
      if (err1 == GLOB_NOMATCH) err1 = 0;
      else if (err1 == GLOB_ABORTED) err1 = 1;
      else if (err1 == GLOB_NOSPACE) err1 = 2;
      else err1 = 3;
      goto fif_return;
   }

   for (uint ii = 0; ii < globdata.gl_pathc; ii++)                               //  loop found files
   {
      file = globdata.gl_pathv[ii];

      if (Fnolinks) {                                                            //  detect and omit symlinks
         err1 = lstat(file,&statB);
         if (err1) continue;
         if (S_ISLNK(statB.st_mode)) continue;
      }
      
      if (Frecurse && dirfile(file)) {                                           //  folder
         err1 = find_imagefiles(file,flags,flist,NF,0);                          //  process member files
         if (err1) goto fif_return;
      }
      
      ftype = image_file_type(file);
      if (ftype == OTHER) continue;                                              //  unknown file type
      
      if (ftype == FDIR && ! Fdirs) continue;
      if (ftype == THUMB && ! Fthumbs) continue;
      if (ftype == IMAGE || ftype == RAW || ftype == VIDEO)
         if (! Fimages) continue;
      
      if (fif_count == fif_max) {                                                //  output list is full
         if (fif_max == 0) {
            fif_max = 1000;                                                      //  initial space, 1000 files
            cc = fif_max * sizeof(ch *);
            fif_filelist = (ch **) zmalloc(cc,"find-files");
         }
         else {
            templist = fif_filelist;                                             //  expand by 2x each time needed
            cc = fif_max * sizeof(ch *);
            fif_filelist = (ch **) zmalloc(cc+cc,"find-files");
            memcpy(fif_filelist,templist,cc);
            memset(fif_filelist+fif_max,0,cc);
            zfree(templist);
            fif_max *= 2;
         }
      }

      fif_filelist[fif_count] = zstrdup(file,"find-files");                      //  add file to output list
      fif_count += 1;
   }

   err1 = 0;

fif_return:

   err2 = errno;                                                                 //  preserve Linux errno

   globfree(&globdata);                                                          //  free memory
   zfree(mfolder);

   if (Finit) {                                                                  //  user call
      NF = fif_count;
      if (NF) flist = fif_filelist;
      spinlock(0);                                                               //  unlock
   }
   
   errno = err2;                                                                 //  return err1 and preserve errno
   return err1;
}


/********************************************************************************/

//  test if a file exists, is a regular file

int regfile(ch *file, struct stat *statb)
{
   STATB    statB;
   int      err;

   err = stat(file,&statB);
   if (err) return 0;
   if (! S_ISREG(statB.st_mode)) return 0;
   if (statb) *statb = statB;
   return 1;
}


//  test if a file exists, is a directory/folder

int dirfile(ch *file, struct stat *statb)
{
   STATB    statB;
   int      err;
   
   err = stat(file,&statB);
   if (err) return 0;
   if (! S_ISDIR(statB.st_mode)) return 0;
   if (statb) *statb = statB;
   return 1;
}


//  test if file exists, is a regular file, and read or write permission
//  rw: 'r' or read or 'w' for write or ' ' for file exists only
//  returns 0 if file does not exist, is not a regular file, or access is not permitted
//  returns 1 if all is OK

int hasperm(ch *file, ch rw)
{
   STATB    statB;
   
   if (! regfile(file,&statB)) return 0;
   if (rw == 'r' && ! (S_IRUSR & statB.st_mode)) return 0;
   if (rw == 'w' && ! (S_IWUSR & statB.st_mode)) return 0;
   return 1;
}


/********************************************************************************/

//  Show a permissions setting, edit and return changed permissions.
//  'fname' is a file name or permission category, e.g. 'default'
//  Permissions are strings: "read+write", "read only", "no access".
//  'p1' and 'p2' are input and output permissions strings, which are
//    any combination of three of the above strings, comma separated,
//     which define the access permissions for user, group, other.
//  p1 and p2 are char[100] strings to allow for fat translations.
//  Returns 1 if user cancel or error, 0 if changes were made OK.

int set_permissions(GtkWidget *parent, ch *fname, ch *p1, ch *p2)
{
   ch             operm[50], gperm[50], wperm[50];
   ch             *pp;
   zdialog        *zd;
   int            zstat;

/***
          ________________________________
         |           permissions          |
         |                                |
         |  owner    [read+write |v]      |
         |  group    [read only  |v]      |
         |  other    [no access  |v]      |
         |                                |
         |               [apply] [cancel] |
         |________________________________|

***/

   zd = zdialog_new("Permissions",parent,"Apply","Cancel",null);
   zdialog_add_widget(zd,"hbox","space","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"vbox","vb1","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","labowner","vb1","owner");
   zdialog_add_widget(zd,"label","labgroup","vb1","group");
   zdialog_add_widget(zd,"label","labother","vb1","other");
   zdialog_add_widget(zd,"combo","ownerperm","vb2");
   zdialog_add_widget(zd,"combo","groupperm","vb2");
   zdialog_add_widget(zd,"combo","otherperm","vb2");
   
   zdialog_stuff(zd,"ownerperm","read+write");                                   //  initz. combo boxes
   zdialog_stuff(zd,"ownerperm","read only");
   zdialog_stuff(zd,"ownerperm","no access");
   
   zdialog_stuff(zd,"groupperm","read+write");
   zdialog_stuff(zd,"groupperm","read only");
   zdialog_stuff(zd,"groupperm","no access");
   
   zdialog_stuff(zd,"otherperm","read+write");
   zdialog_stuff(zd,"otherperm","read only");
   zdialog_stuff(zd,"otherperm","no access");
   
   pp = substring(p1,',',1);                                                     //  initz. dialog from input permissions
   if (pp) zdialog_stuff(zd,"ownerperm",pp);

   pp = substring(p1,',',2);
   if (pp) zdialog_stuff(zd,"groupperm",pp);

   pp = substring(p1,',',3);
   if (pp) zdialog_stuff(zd,"otherperm",pp);
   
   zdialog_run(zd,0,0);                                                          //  run dialog

   zstat = zdialog_wait(zd);                                                     //  wait for completion
   if (zstat != 1) {
      zdialog_free(zd);                                                          //  [cancel]
      return 1;                                                                  //  error return
   }
   
   *p2 = 0;

   zdialog_fetch(zd,"ownerperm",operm,50);                                       //  get permissions from dialog
   zdialog_fetch(zd,"groupperm",gperm,50);
   zdialog_fetch(zd,"otherperm",wperm,50);
   
   snprintf(p2,100,"%s, %s, %s",operm,gperm,wperm);                              //  output permissions
   
   zdialog_free(zd);
   return 0;                                                                     //  normal return
}


//  convert between permission formats
//  string:  "read+write, read only, no access"
//  mode_t:   bitmap of permissions, e.g. 0640

int conv_permissions(ch *perms, mode_t &mode)                                    //  string permissions to mode_t
{
   int      err = 0;
   ch       *pp;

   mode = 0;

   pp = substring(perms,',',1);
   if (pp) {
      if (strmatch(pp,"read+write")) mode += S_IRUSR + S_IWUSR;
      else if (strmatch(pp,"read only")) mode += S_IRUSR;
      else if (strmatch(pp,"no access")); 
      else err = 1;
   }
   else err = 1;
   
   pp = substring(perms,',',2);
   if (pp) {
      if (strmatch(pp,"read+write")) mode += S_IRGRP + S_IWGRP;
      else if (strmatch(pp,"read only")) mode += S_IRGRP;
      else if (strmatch(pp,"no access")); 
      else err = 1;
   }
   else err = 1;
   
   pp = substring(perms,',',3);
   if (pp) {
      if (strmatch(pp,"read+write")) mode += S_IROTH + S_IWOTH;
      else if (strmatch(pp,"read only")) mode += S_IROTH;
      else if (strmatch(pp,"no access")); 
      else err = 1;
   }
   else err = 1;
   
   return err;
}


int conv_permissions(mode_t mode, ch *perms)                                     //  mode_t to string permissions 
{
   int      err = 0;

   *perms = 0;

   if (mode & S_IRUSR && mode & S_IWUSR) strcat(perms,"read+write");
   else if (mode & S_IRUSR) strcat(perms,"read only");
   else strcat(perms,"no access");
   strcat(perms,", ");

   if (mode & S_IRGRP && mode & S_IWGRP) strcat(perms,"read+write");
   else if (mode & S_IRGRP) strcat(perms,"read only");
   else strcat(perms,"no access");
   strcat(perms,", ");

   if (mode & S_IROTH && mode & S_IWOTH) strcat(perms,"read+write");
   else if (mode & S_IROTH) strcat(perms,"read only");
   else strcat(perms,"no access");
   
   return err;
}


