//------------------------------------------------------------------------------
// emWorldClockPanel.cpp
//
// Copyright (C) 2006-2008,2010,2014,2016-2017 Oliver Hamann.
//
// Homepage: http://eaglemode.sourceforge.net/
//
// This program is free software: you can redistribute it and/or modify it under
// the terms of the GNU General Public License version 3 as published by the
// Free Software Foundation.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE. See the GNU General Public License version 3 for
// more details.
//
// You should have received a copy of the GNU General Public License version 3
// along with this program. If not, see <http://www.gnu.org/licenses/>.
//------------------------------------------------------------------------------

#include <emClock/emWorldClockPanel.h>
#include <emClock/emClockPanel.h>


emWorldClockPanel::emWorldClockPanel(
	ParentArg parent, const emString & name, emClockFileModel * fileModel
)
	: emFilePanel(parent,name,fileModel)
{
	FileModel=fileModel;
	TimeZonesModel=emTimeZonesModel::Acquire(GetRootContext());
	Children.SetTuningLevel(4);
	AddWakeUpSignal(GetVirFileStateSignal());
	AddWakeUpSignal(FileModel->GetChangeSignal());
	AddWakeUpSignal(TimeZonesModel->GetTimeSignal());
	UpdateSunPosition();
}


emWorldClockPanel::~emWorldClockPanel()
{
}


emString emWorldClockPanel::GetTitle() const
{
	return "World Clock";
}


bool emWorldClockPanel::Cycle()
{
	bool busy;

	busy=emFilePanel::Cycle();

	if (IsSignaled(GetVirFileStateSignal())) {
		CreateOrDestroyChildren();
		PreparePolygons(false);
	}

	if (IsSignaled(FileModel->GetChangeSignal())) {
		InvalidatePainting();
	}

	if (IsSignaled(TimeZonesModel->GetTimeSignal())) {
		UpdateSunPosition();
		if (IsVFSGood() && IsViewed()) {
			PreparePolygons(true);
			InvalidatePainting();
		}
	}

	return busy;
}


void emWorldClockPanel::Notice(NoticeFlags flags)
{
	emFilePanel::Notice(flags);
	if ((flags&(NF_VIEWING_CHANGED|NF_SOUGHT_NAME_CHANGED))!=0) {
		CreateOrDestroyChildren();
		PreparePolygons(false);
	}
}


bool emWorldClockPanel::IsOpaque() const
{
	if (!IsVFSGood()) return emFilePanel::IsOpaque();
	return false;
}


void emWorldClockPanel::Paint(const emPainter & painter, emColor canvasColor) const
{
	emColor wcol,lcol,scol;
	int i;

	if (!IsVFSGood()) {
		emFilePanel::Paint(painter,canvasColor);
		return;
	}

	wcol=FileModel->WorldWaterColor;
	lcol=FileModel->WorldLandColor;
	scol=FileModel->WorldShadowColor;

	painter.PaintPolygon(
		WaterPolygon.Get(),
		WaterPolygon.GetCount()/2,
		wcol,
		canvasColor
	);

	for (i=0; i<LandPolygons.GetCount(); i++) {
		painter.PaintPolygon(
			LandPolygons[i].Get(),
			LandPolygons[i].GetCount()/2,
			lcol,
			wcol
		);
	}

	painter.PaintPolygon(
		ShadowPolygon.Get(),
		ShadowPolygon.GetCount()/2,
		scol
	);

	if (TimeZonesModel->GetCityCount()<=0) {
		painter.PaintTextBoxed(
			0.25,
			GetHeight()/4,
			0.5,
			GetHeight()/2,
			"Error:\n\nNo time zones found.",
			0.08,
			emColor(255,0,0),
			0,
			EM_ALIGN_CENTER,
			EM_ALIGN_CENTER,
			1.0
		);
	}
	else {
		painter.PaintTextBoxed(
			0.45,
			GetHeight()-0.025,
			0.1,
			0.007,
			"Hint: If some clocks overlap each other, you can push a\n"
			"clock to the front or to the back by clicking on its\n"
			"border with the left or right mouse button, respectively.",
			1.0,
			FileModel->ClockForegroundColor,
			0,
			EM_ALIGN_CENTER,
			EM_ALIGN_CENTER,
			1.0
		);
	}
}


void emWorldClockPanel::LayoutChildren()
{
	emArray<emClockPanel*> sorted;
	emClockPanel * p, * p2;
	int i,j,n;
	double r,x,y,r2,x2,y2,rn,minR,maxR,ov,dTest;

	emFilePanel::LayoutChildren();

	minR=CalcClockMinRadius();
	maxR=CalcClockMaxRadius();

	n=Children.GetCount();

	for (i=0; i<n; i++) {
		p=Children.Get(i);
		TransformCoords(
			&x,
			&y,
			TimeZonesModel->GetCityLatitude(i),
			TimeZonesModel->GetCityLongitude(i)
		);
		p->Layout(
			x-maxR,
			y-maxR,
			2*maxR,
			2*maxR
		);
	}

	sorted=Children;
	sorted.Sort(CmpClockPanelX);
	ov=1.07;
	dTest=maxR*2/ov;
	for (i=0; i<n; i++) {
		p=sorted.Get(i);
		r=p->GetLayoutWidth()*0.5;
		x=p->GetLayoutX()+r;
		y=p->GetLayoutY()+r;
		for (j=i+1; j<n; j++) {
			p2=sorted.Get(j);
			r2=p2->GetLayoutWidth()*0.5;
			x2=p2->GetLayoutX()+r2;
			if (x2-x>dTest) break;
			y2=p2->GetLayoutY()+r2;
			if (y2-y>dTest) continue;
			rn=sqrt((x2-x)*(x2-x)+(y2-y)*(y2-y))*0.5*ov;
			if (rn<minR) rn=minR;
			if (r>rn) {
				r=rn;
				p->Layout(x-r,y-r,2*r,2*r);
			}
			if (r2>rn) {
				p2->Layout(x2-rn,y2-rn,2*rn,2*rn);
			}
		}
	}
}


void emWorldClockPanel::TransformCoords(
	double * pX, double * pY, double latitude, double longitude
) const
{
	double x,y,a,b,h,f;

	// This is the Winkel Tripel Projection.
	x=longitude*(1.0/180.0);
	y=latitude*(M_PI/180.0);
	a=cos(x*(M_PI*0.5));
	if (a<0.999999) {
		a=acos(a*cos(y));
		b=sin(y)/sin(a);
		y=(y+a*b)*0.5;
		a*=sin(acos(b));
		if (x<0) a=-a;
		x+=a;
	}

	h=GetHeight();
	f=emMin(1.0/(M_PI+2.0),h/M_PI);
	*pX=0.5+x*f;
	*pY=h*0.5-y*f;
}


double emWorldClockPanel::CalcEarthWidth() const
{
	double x1,y1,x2,y2;

	TransformCoords(&x1,&y1,0.0,-180.0);
	TransformCoords(&x2,&y2,0.0,180.0);
	return x2-x1;
}


double emWorldClockPanel::CalcEarthHeight() const
{
	double x1,y1,x2,y2;

	TransformCoords(&x1,&y1,90.0,0.0);
	TransformCoords(&x2,&y2,-90.0,0.0);
	return y2-y1;
}


double emWorldClockPanel::CalcClockMinRadius() const
{
	return CalcEarthWidth()/360.0*FileModel->WorldClockMinRadius;
}


double emWorldClockPanel::CalcClockMaxRadius() const
{
	return
		CalcEarthWidth()/360.0*
		emMax(
			FileModel->WorldClockMinRadius.Get(),
			FileModel->WorldClockMaxRadius.Get()
		)
	;
}


void emWorldClockPanel::CreateOrDestroyChildren()
{
	bool haveChildren;
	int i,n;
	double vr;

	haveChildren=IsVFSGood();
	if (!IsInViewedPath()) {
		haveChildren=false;
	}
	else if (IsViewed() && !GetSoughtName()) {
		vr=PanelToViewDeltaX(CalcClockMaxRadius());
		if (vr<1.2) haveChildren=false;
	}

	if (haveChildren) {
		if (Children.GetCount()==0) {
			n=TimeZonesModel->GetCityCount();
			Children.SetCount(n,true);
			for (i=0; i<n; i++) {
				Children.Set(
					i,
					new emClockPanel(
						this,
						TimeZonesModel->GetCityName(i),
						FileModel,
						TimeZonesModel->GetCityZoneId(i)
					)
				);
			}
		}
	}
	else {
		n=Children.GetCount();
		for (i=0; i<n; i++) {
			delete Children.Get(i);
		}
		Children.SetCount(0,true);
	}
}


void emWorldClockPanel::UpdateSunPosition()
{
	double jd,m,t,la,lo;

	jd=TimeZonesModel->GetJulianDate();

	m=357.5291+0.98560028*(jd-2451545.0);

	t=m*(M_PI/180.0);
	m+=1.9148*sin(t)+0.02*sin(2*t)+0.0003*sin(3*t);

	t=(m+102.9372+180.0)*(M_PI/180.0);

	la=asin(sin(t)*sin(23.45*M_PI/180.0))*(180.0/M_PI);
	while (la>180.0) la-=360.0;
	while (la<-180.0) la+=360.0;

	lo=atan2(sin(t)*cos(23.45*M_PI/180.0),cos(t))*(180.0/M_PI);
	lo-=280.16+360.9856235*(jd-2451545.0);
	lo=fmod(lo,360.0);
	while (lo>180.0) lo-=360.0;
	while (lo<-180.0) lo+=360.0;

	SunLatitude=la;
	SunLongitude=lo;
}


void emWorldClockPanel::PreparePolygons(bool shadowOnly)
{
	double fn;
	int n;

	if (IsViewed()) {
		fn=PanelToViewDeltaY(CalcEarthHeight())*0.7;
		if (fn<=8.0) n=8;
		else if (fn>=150.0) n=150;
		else n=(int)(fn+0.5);
	}
	else {
		n=0;
	}

	if (!shadowOnly) {
		PrepareWaterPolygon(n);
		PrepareLandPolygons();
	}
	PrepareShadowPolygon(n);
}


void emWorldClockPanel::PrepareWaterPolygon(int n)
{
	double * xy;
	int i;

	if (!IsVFSGood() || !IsViewed()) {
		WaterPolygon.Clear(true);
		return;
	}

	WaterPolygon.SetTuningLevel(4);
	WaterPolygon.SetCount(n*4,true);
	xy=WaterPolygon.GetWritable();
	for (i=0; i<n; i++) {
		TransformCoords(
			xy+i*2,
			xy+i*2+1,
			90.0-180.0*i/(n-1),
			180.0
		);
		TransformCoords(
			xy+(n*2-1-i)*2,
			xy+(n*2-1-i)*2+1,
			90.0-180.0*i/(n-1),
			-180.0
		);
	}
}


void emWorldClockPanel::PrepareLandPolygons()
{
	const emInt16 * p;
	emArray<double> * poly;
	double * xy;
	double vew;
	int i,j,n;

	if (!IsVFSGood() || !IsViewed()) {
		LandPolygons.Clear(true);
		return;
	}

	vew=PanelToViewDeltaX(CalcEarthWidth());
	if      (vew<100.0) p=MapData1;
	else if (vew<400.0) p=MapData2;
	else                p=MapData3;
	for (i=0; *p; i++) {
		n=*p++;
		if (LandPolygons.GetCount()<=i) LandPolygons.AddNew();
		poly=&LandPolygons.GetWritable(i);
		poly->SetTuningLevel(4);
		poly->SetCount(n*2,true);
		xy=poly->GetWritable();
		for (j=0; j<n; j++) {
			TransformCoords(xy,xy+1,p[1]/-100.0,p[0]/100.0);
			xy+=2;
			p+=2;
		}
	}
	LandPolygons.SetCount(i,true);
}


void emWorldClockPanel::PrepareShadowPolygon(int n)
{
	double * tempMem, * sunrise, * sunset;
	double * xy;
	int pt[6],ps[6],pe[6],pd[6];
	double cos_sla,sin_sla,t,r,s,la,lo;
	int i,i1,i2,j,k;

	if (!IsVFSGood() || !IsViewed()) {
		ShadowPolygon.Clear(true);
		return;
	}

	ShadowPolygon.SetTuningLevel(4);
	tempMem=new double[n*2];
	sunrise=tempMem;
	sunset=tempMem+n;
	cos_sla=cos(SunLatitude*(M_PI/180.0));
	sin_sla=sin(SunLatitude*(M_PI/180.0));
	i1=-1;
	i2=n;
	for (i=0; i<n; i++) {
		la=(0.5-((double)i)/(n-1))*M_PI;
		t=cos(la)*cos_sla;
		if (fabs(t)<0.00000001) {
			if (i1>=0) {
				i2=i;
				break;
			}
		}
		else {
			t=sin(la)*sin_sla/t;
			if (fabs(t)>=1.0) {
				if (i1>=0) {
					i2=i;
					break;
				}
			}
			else {
				t=acos(-t)*(180.0/M_PI);
				r=SunLongitude-t;
				s=SunLongitude+t;
				while (r> 180) r-=360;
				while (r<-180) r+=360;
				while (s> 180) s-=360;
				while (s<-180) s+=360;
				sunrise[i]=r;
				sunset[i]=s;
				if (i1<0) i1=i;
			}
		}
	}
	if (i1<0) i1=i2=n/2;

	if (SunLatitude<=0.0) {
		for (k=i1; k<i2 && sunrise[k]<=sunset[k]; k++);
		pt[0]=1; ps[0]=0; pe[0]=k; pd[0]=-1;
		pt[1]=0; ps[1]=0; pe[1]=k; pd[1]=1;
		t=1.0;
	}
	else {
		for (k=i2; k>i1 && sunrise[k-1]<=sunset[k-1]; k--);
		pt[0]=0; ps[0]=k; pe[0]=n; pd[0]=1;
		pt[1]=1; ps[1]=k; pe[1]=n; pd[1]=-1;
		t=-1.0;
	}

	if (
		i2>i1 &&
		(sunrise[i1]+sunset[i1]-sunrise[i2-1]-sunset[i2-1])*t<0.0
	) {
		pt[2]=2; ps[2]=i1; pe[2]=k;  pd[2]=-1;
		pt[3]=3; ps[3]=i1; pe[3]=k;  pd[3]=1;
		pt[4]=3; ps[4]=k;  pe[4]=i2; pd[4]=1;
		pt[5]=2; ps[5]=k;  pe[5]=i2; pd[5]=-1;
	}
	else {
		pt[2]=3; ps[2]=k;  pe[2]=i2; pd[2]=1;
		pt[3]=2; ps[3]=k;  pe[3]=i2; pd[3]=-1;
		pt[4]=2; ps[4]=i1; pe[4]=k;  pd[4]=-1;
		pt[5]=3; ps[5]=i1; pe[5]=k;  pd[5]=1;
	}

	for (j=0, i=0; i<6; i++) {
		if (pe[i]<ps[i]) pe[i]=ps[i];
		else j+=pe[i]-ps[i];
	}
	ShadowPolygon.SetCount(j*2,true);
	xy=ShadowPolygon.GetWritable();
	for (i=0; i<6; i++) {
		for (j=0; j<pe[i]-ps[i]; j++) {
			if (pd[i]>0) k=ps[i]+j; else k=pe[i]-1-j;
			la=(0.5-((double)k)/(n-1))*180.0;
			switch (pt[i]) {
				case 0:  lo=-180.0;     break;
				case 1:  lo=180.0;      break;
				case 2:  lo=sunrise[k]; break;
				default: lo=sunset[k];  break;
			}
			TransformCoords(xy,xy+1,la,lo);
			xy+=2;
		}
	}

	delete [] tempMem;
}


int emWorldClockPanel::CmpClockPanelX(
	emClockPanel * const * p1, emClockPanel * const * p2, void * context
)
{
	double d;

	d=(*p1)->GetLayoutX()-(*p2)->GetLayoutX();
	if (d<0) return -1;
	if (d>0) return 1;
	return 0;
}


const emInt16 emWorldClockPanel::MapData1[] = {
	3,4427,1730,5085,1260,4530,2760,6,-6213,6524,-6819,8250,5374,6617,16861,
	6984,18000,9000,-18000,9000,46,10485,-7830,13185,-7155,14940,-7200,18000,
	-6885,18000,-6480,15615,-5175,16155,-6210,13995,-5715,14130,-4815,12780,
	-3825,11970,-3915,12285,-2835,10845,-1935,10800,-855,8910,-2340,7701,
	-820,7020,-2385,6075,-2475,3915,540,4050,1485,2745,3420,1755,3375,945,
	-405,-1125,-540,-1845,-2025,-630,-3465,1035,-3555,1800,-3150,3465,-3105,
	3510,-3555,945,-4410,-630,-3645,-990,-4185,-45,-4500,-900,-5310,-450,
	-5760,360,-5220,1080,-5580,1575,-5310,2475,-5760,2070,-6345,1440,-5580,
	405,-6030,2250,-7110,4725,-6750,7020,-7245,8,-3510,630,-4230,2295,-6795,
	4680,-6795,5535,-7470,5265,-6975,2025,-8146,430,-7200,-1305,3,11565,495,
	10845,45,11790,-675,8,15390,2880,14715,4050,13185,3195,11745,3555,11295,
	2340,13365,1125,14074,1765,14310,1260,3,14265,990,12915,0,15030,630,3,
	10980,900,9495,-360,10215,-675,27,-12771,-7018,-11970,-7785,-7965,-8325,
	-5940,-8190,-1215,-8190,-4365,-5985,-7065,-7920,-7965,-7470,-6120,-6570,
	-6660,-6210,-8100,-6840,-9585,-5940,-7966,-5150,-7739,-6270,-6433,-6003,
	-5579,-5256,-8235,-2925,-9810,-2700,-9495,-1800,-8910,-2070,-8370,-945,
	-10395,-1845,-12375,-3915,-12555,-5130,-14580,-6120,-16065,-5760,-16695,
	-6885,0
};


const emInt16 emWorldClockPanel::MapData2[] = {
	3,10863,-1928,10981,-1816,11111,-1994,27,-7920,-900,-7920,-945,-7686,
	-815,-7226,-1190,-6239,-983,-5153,-463,-4893,37,-3506,550,-3501,871,
	-3883,1311,-4026,2117,-4673,2430,-5446,3530,-6126,3917,-6639,4424,-6693,
	4810,-6919,5164,-6573,5490,-6993,5570,-7393,5324,-7566,4944,-7170,3348,
	-6986,1857,-7623,1427,-8146,430,-8013,-16,-7673,-510,5,17807,3877,16927,
	4670,16694,4624,17494,3904,17310,3475,10,13320,38,13521,330,13805,133,
	14548,452,15062,1085,14534,790,14294,924,13821,837,13848,600,13101,184,4,
	-2366,-6623,-1379,-6550,-1706,-6363,-2119,-6376,5,-8089,-2321,-7391,
	-2005,-6653,-1823,-7119,-1830,-8366,-2256,6,-7793,-8303,-6106,-8256,
	-7246,-7976,-7893,-7676,-8799,-7602,-9693,-8063,3,967,-3896,841,-3916,
	914,-4256,6,9574,-8123,8891,-8120,8699,-7948,9554,-7930,10021,-7823,
	10747,-7823,33,3240,-2970,3967,-1583,4481,-1016,5114,-1176,5081,-930,
	4814,-436,4381,-90,3887,484,4081,1514,3492,1993,3563,2433,3301,2577,3107,
	3064,2645,3385,1974,3444,1614,2847,1174,1690,1414,1064,934,110,994,-341,
	361,-643,-859,-443,-1686,-1310,-1593,-2460,-986,-2953,-959,-3203,-546,
	-3543,-199,-3516,954,-3730,1134,-3336,1947,-3016,2061,-3250,3240,-3105,3,
	8001,-616,8207,-670,8027,-976,19,14621,1870,15316,2552,15297,3135,14995,
	3748,14481,3924,13987,3764,13634,3337,12914,3150,11647,3530,11364,2674,
	11407,2224,12127,1917,12695,1379,12964,1499,13099,1144,13674,1217,13540,
	1483,14074,1765,14250,1075,16,-6054,-8184,-3359,-8356,-2399,-8283,-1186,
	-8176,-2032,-7852,-1813,-7636,-2553,-6910,-3653,-6683,-4106,-6416,-4399,
	-5970,-5017,-6243,-5353,-6710,-5286,-7203,-5853,-7556,-6966,-7656,-7317,
	-7835,3,11987,-2330,12107,-2216,12194,-2536,5,14001,-7636,15074,-7496,
	14061,-7503,14321,-7336,13654,-7563,7,12026,-105,12507,-103,12027,44,
	12274,104,12234,464,12021,497,11874,310,3,15307,450,15027,637,14881,584,
	5,-10906,-7870,-11273,-7663,-10406,-7530,-11331,-7444,-12293,-7616,3,
	12767,157,12867,-96,12781,-250,12,-7099,-6730,-7773,-7010,-8886,-7036,
	-8959,-7283,-9486,-7263,-9593,-7410,-8839,-7390,-7839,-7356,-6793,-6976,
	-6253,-6656,-6606,-6203,-7746,-6463,11,10341,-56,10614,584,11621,810,
	12707,850,12381,1037,12419,895,11939,935,10627,744,10561,610,10147,297,
	9521,-583,4,-9760,-7399,-9648,-7208,-9887,-7133,-10283,-7287,4,-9593,
	-7876,-9293,-7756,-10286,-7778,-10588,-7950,3,1551,-3810,1498,-3653,1246,
	-3795,7,-12326,-7456,-10859,-7263,-10599,-7383,-10133,-6936,-11206,-6870,
	-11799,-7096,-12619,-7190,153,4167,-6740,3967,-6620,3269,-6693,3701,
	-6373,4501,-6733,5321,-6860,6107,-6846,5861,-7033,6914,-6820,6687,-7100,
	7181,-7313,7301,-6813,7041,-6606,7494,-6740,7354,-7180,7874,-7246,9074,
	-7566,10547,-7760,11401,-7606,11047,-7360,12834,-7333,12967,-7106,13801,
	-7106,14341,-7260,15147,-7086,15934,-7093,16501,-6886,17221,-7000,18000,
	-6885,18000,-6480,17747,-6266,16363,-5986,16174,-5506,15677,-5087,15556,
	-5558,16501,-6233,15827,-6193,15301,-5933,14260,-5939,13487,-5480,14101,
	-5326,14041,-4853,13507,-4346,12753,-3963,12967,-3553,12653,-3430,12627,
	-3786,12230,-4072,11707,-3853,12241,-3706,11914,-3516,12236,-3025,11621,
	-2261,10734,-2113,10541,-1926,10890,-1529,10919,-1137,10587,-906,10047,
	-1353,9921,-980,10321,-393,10394,-120,10174,-326,9847,-886,9854,-1533,
	9487,-1626,9127,-2300,8024,-1563,7981,-1040,7701,-820,7274,-1986,6647,
	-2513,5781,-2593,5287,-2720,5018,-3026,4774,-2926,5274,-2346,5574,-2573,
	6003,-2220,5374,-1626,4364,-1266,4234,-1706,3547,-2800,3427,-2773,3240,
	-3015,3240,-3105,3494,-3153,3616,-3681,2907,-3626,2607,-4000,2951,-4066,
	2835,-4095,2835,-4140,4071,-4092,4185,-4185,4860,-4185,4997,-4048,4841,
	-3766,5396,-3657,5341,-4126,5067,-4446,5294,-4566,5174,-4746,4671,-4441,
	4860,-4185,4185,-4185,3654,-4686,3454,-4466,3141,-4706,2767,-4249,2835,
	-4140,2835,-4095,2521,-4086,2266,-4027,2421,-3800,2214,-3693,1927,-4160,
	1341,-4560,1234,-4413,1827,-4033,1594,-3780,1607,-3973,941,-4413,354,
	-4333,-253,-3666,-853,-3700,-879,-4333,-139,-4326,-106,-4626,-473,-4820,
	114,-4973,627,-5346,903,-5380,831,-5676,1036,-5764,1087,-5426,2001,-5433,
	2447,-5940,2947,-6013,2261,-6006,2119,-6259,2581,-6546,2274,-6586,1681,
	-6166,1874,-5953,1361,-5533,1094,-5940,807,-5786,547,-5873,534,-6186,
	1154,-6433,1527,-6840,2781,-7093,88,-15619,-7123,-13675,-6884,-12771,
	-7018,-11059,-6763,-9919,-6816,-9506,-7196,-8639,-6810,-8535,-7000,-8106,
	-6890,-8318,-6597,-8022,-6385,-8233,-6250,-8753,-6463,-9486,-6056,-9326,
	-5796,-8259,-5510,-8159,-5196,-7966,-5150,-7746,-5656,-7739,-6270,-6626,
	-5876,-6433,-6003,-5579,-5256,-5993,-5016,-6779,-4963,-6486,-4863,-6393,
	-4630,-5993,-4576,-7026,-4356,-7028,-4157,-7335,-4095,-7650,-4320,-7629,
	-4402,-8113,-4276,-8099,-4556,-8762,-4889,-9139,-4750,-8713,-4556,-8733,
	-4243,-8426,-4636,-8219,-4110,-7650,-4320,-7335,-4095,-7633,-3536,-8125,
	-3107,-7973,-2583,-8146,-2523,-8353,-3030,-9675,-2835,-9810,-2205,-9495,
	-1800,-9090,-1845,-9039,-2080,-8678,-2147,-8889,-1571,-8339,-1536,-8359,
	-1043,-8078,-877,-7920,-945,-7920,-900,-8038,-710,-8572,-992,-8666,-1230,
	-9306,-1583,-9698,-1561,-10519,-1936,-10559,-2236,-11026,-2683,-11321,
	-3125,-11498,-3167,-10935,-2317,-11059,-2316,-11446,-2803,-11751,-3342,
	-12093,-3490,-12446,-4050,-12406,-4836,-13726,-5890,-14806,-6076,-16006,
	-5556,-17033,-5230,-16039,-5636,-15726,-5810,-16613,-6143,-16092,-6350,
	-16798,-6561,-16029,-6646,-16653,-6856,4,4514,-8063,5841,-8163,6701,
	-7956,6441,-7896,9,174,-5156,-433,-5870,-579,-5576,-966,-5376,-993,-5143,
	-619,-5250,-546,-5543,-300,-5399,-513,-5010,6,5780,-7565,6914,-7683,5927,
	-7463,5529,-7216,5734,-7050,5141,-7156,8,12227,-1810,12167,-1496,12561,
	-1221,12599,-605,12339,-705,12289,-1255,12027,-1316,12054,-1843,7,-9053,
	-7650,-8887,-7549,-7826,-7550,-8057,-7425,-9346,-7470,-10153,-7543,
	-10513,-7703,3,-5619,-5123,-5293,-4716,-5951,-4757,5,11874,-550,11629,
	405,11079,325,10849,25,11702,-701,4,1567,-7703,2334,-7770,2701,-8003,
	1061,-7963,6,4427,2564,4320,2175,4427,1730,4961,1264,5034,1517,4713,2519,
	18,-18000,8415,-15840,8505,-15165,7740,-7605,7380,-6075,6345,-6345,6750,
	-6030,7425,-7695,7920,-6120,8325,-990,7200,5374,6617,7067,7057,9487,6584,
	16861,6984,16101,8110,18000,8460,18000,9000,-18000,9000,4,-16979,-6603,
	-17293,-6470,-18000,-6480,-18000,-6885,3,14467,4110,14854,4124,14694,
	4417,14,14014,-3810,14001,-4230,14207,-4506,14247,-5426,14454,-4930,
	14241,-4563,14547,-4330,14087,-4223,14234,-3956,14034,-3490,13294,-3400,
	13094,-3096,13007,-3270,13267,-3480,0
};


const emInt16 emWorldClockPanel::MapData3[] = {
	3,-15938,-2219,-15932,-2171,-15994,-2213,3,-6135,-1521,-6115,-1553,-6153,
	-1562,3,14385,-7354,14086,-7324,14039,-7440,3,-7757,-1754,-7810,-1855,
	-7612,-1790,7,17082,4292,17283,4067,17325,4135,17436,4141,17153,4428,
	16971,4683,16644,4577,3,-16588,-6045,-16573,-5974,-16730,-5986,3,822,
	-331,881,-304,884,-372,9,-17505,-6750,-17415,-6615,-17280,-6705,-17100,
	-6660,-17010,-6570,-17325,-6435,-17820,-6570,-18000,-6480,-18000,-6885,3,
	-1520,-2775,-1573,-2763,-1567,-2814,3,-15826,7976,-16280,8009,-16213,
	7903,3,-17175,1371,-17208,1406,-17264,1365,3,5387,-8069,6023,-8084,5760,
	-8191,3,16680,-5461,16621,-5496,16612,-5546,3,12393,-944,12470,-983,
	12399,-1057,3,16064,937,16067,1003,15937,943,3,-7796,-2670,-7840,-2638,
	-7876,-2676,3,-9770,-6995,-9528,-6850,-9933,-6909,4,-15218,-5850,-15247,
	-5733,-15413,-5653,-15468,-5735,5,14600,4370,14454,4072,14669,4119,14840,
	4076,14754,4358,3,-2857,-3808,-2860,-3867,-2795,-3843,3,862,-4239,918,
	-4141,959,-4283,23,13320,38,13407,90,13427,261,13521,330,13805,133,14199,
	309,14548,452,14610,555,14776,634,14723,738,15062,1085,14782,1014,14628,
	838,14485,762,14250,955,14004,838,13775,833,13859,720,13848,600,13599,
	462,13320,391,13265,258,13064,119,3,11716,-850,11982,-1036,11940,-1078,3,
	-6103,-1429,-6085,-1476,-6121,-1485,3,18000,-7155,17865,-7110,18000,
	-7065,145,3240,-3105,3240,-2970,3384,-2723,3561,-2388,3555,-2297,3691,
	-2206,3725,-2084,3735,-1886,3865,-1796,3941,-1543,4008,-1573,4012,-1510,
	4107,-1465,4356,-1215,4278,-1163,4342,-1141,4424,-1037,4752,-1108,4929,
	-1130,5077,-1191,5132,-1177,5132,-1067,5053,-886,4923,-669,4819,-464,
	4574,-224,4389,-95,4030,285,3871,607,3947,714,3953,943,4058,1081,4081,
	1514,3949,1692,3735,1755,3510,1440,3468,1006,3375,945,3105,855,3064,699,
	2999,642,2970,450,3240,225,3344,252,3412,4,3208,-11,3182,220,3240,225,
	2970,450,2920,317,2926,650,3105,855,3375,945,3431,1107,3411,1269,3510,
	1440,3735,1755,3717,1801,3475,2002,3546,2215,3555,2428,3285,2561,3252,
	2839,3111,2960,3017,3121,2836,3272,2645,3385,2241,3416,1997,3485,1790,
	3331,1831,3199,1673,2908,1528,2750,1453,2434,1447,2248,1372,2145,1273,
	1928,1183,1816,1173,1593,1222,1508,1269,1327,1387,1207,1376,1045,1301,
	927,1350,816,1194,556,1184,467,871,113,944,5,994,-341,853,-448,585,-425,
	447,-636,159,-626,-168,-476,-471,-519,-788,-433,-1234,-738,-1412,-1012,
	-1699,-1261,-1662,-1374,-1739,-1464,-1643,-1598,-1588,-1832,-1641,-1942,
	-1621,-2019,-1708,-2147,-1590,-2378,-1483,-2463,-1452,-2599,-1355,-2680,
	-1310,-2798,-1180,-2810,-980,-2981,-980,-3166,-894,-3302,-666,-3410,-580,
	-3573,-521,-3576,-465,-3505,-157,-3502,-103,-3560,119,-3652,554,-3664,
	1011,-3718,1036,-3661,1113,-3693,1042,-3595,1122,-3524,990,-3395,1118,
	-3373,1196,-3282,1386,-3294,1535,-3225,1557,-3141,1802,-3096,1918,-3020,
	2033,-3094,1987,-3180,2070,-3263,2195,-3301,2387,-3210,2633,-3152,2927,
	-3079,3100,-3156,3,-2682,-3888,-2703,-3843,-2747,-3876,19,13302,-3546,
	13595,-3555,13672,-3682,13825,-3705,13964,-3847,14039,-4093,14151,-4090,
	14240,-3951,14119,-3798,14065,-3518,13684,-3443,13580,-3315,13468,-3446,
	13433,-3336,13272,-3273,13207,-3403,13140,-3114,13033,-3114,12950,-3327,
	8,5780,-7565,6839,-7704,6869,-7609,5854,-7423,5529,-7216,5748,-7044,5357,
	-7062,5144,-7195,69,14383,1418,14529,1489,14650,1908,14859,2022,14938,
	2202,15074,2257,15092,2340,15316,2552,15313,2695,15357,2858,15297,3135,
	15257,3230,15093,3426,15029,3598,14995,3748,14791,3793,14640,3909,14485,
	3819,14335,3884,14042,3811,13977,3655,13912,3563,13654,3608,13858,3521,
	13833,3456,13708,3527,13804,3280,13571,3492,13412,3251,13181,3161,12914,
	3150,12814,3218,12565,3231,12355,3385,11972,3395,11807,3511,11641,3515,
	11489,3397,11572,3353,11564,3146,11364,2674,11425,2598,11327,2450,11384,
	2213,11668,2081,11800,2075,11900,2006,12088,1975,12238,1817,12208,1720,
	12326,1632,12340,1699,12448,1607,12458,1541,12695,1379,12833,1492,12964,
	1499,12953,1427,13127,1217,13252,1207,13099,1144,13337,1164,13674,1217,
	13540,1483,13972,1771,14074,1765,14177,1526,14147,1359,14250,1075,3,
	-8269,-2185,-8281,-2129,-8313,-2179,3,16153,940,16053,893,16082,843,3,
	17955,1629,17967,1688,17863,1688,3,9780,-40,9789,-122,9715,-144,3,-2505,
	-3793,-2540,-3754,-2573,-3786,9,-2233,-6652,-2076,-6543,-1597,-6643,
	-1342,-6531,-1653,-6365,-2002,-6342,-2230,-6398,-2295,-6525,-2431,-6543,
	10,12325,-850,12426,-832,12577,-967,12668,-685,12580,-696,12588,-595,
	12504,-555,12417,-622,12394,-742,12193,-731,4,1121,-5571,1132,-5476,1209,
	-5476,1254,-5601,3,2638,-3508,2449,-3499,2369,-3555,4,13437,546,13502,
	561,13482,677,13420,677,3,13642,121,13594,136,13580,65,3,-6718,-1772,
	-6526,-1811,-6721,-1849,3,-1691,-2803,-1629,-2787,-1641,-2836,6,10984,
	-1991,10863,-1928,10868,-1837,10981,-1816,11058,-1878,11111,-1994,3,
	-18000,-7200,-17763,-7137,-18000,-7110,3,10764,332,10838,282,10749,264,3,
	14719,192,14704,234,14662,219,3,15834,828,15783,884,15641,769,3,11997,
	-1154,12041,-1207,11991,-1233,3,15698,671,15774,745,15656,704,3,16698,
	1576,16731,1523,16663,1491,5,2273,-3807,2344,-3733,2293,-3656,2178,-3682,
	2122,-3798,3,-2304,-1580,-2259,-1604,-2280,-1642,9,4492,2562,4360,2438,
	4330,2162,4462,1977,4391,1697,4740,1492,4927,1188,5061,1551,4713,2519,3,
	-3656,5401,-3567,5498,-3783,5404,19,11923,-524,11775,-401,11789,-214,
	11886,-115,11787,-63,11766,85,11659,159,11608,393,11318,321,11013,300,
	10989,135,10902,46,10888,-125,11089,-190,11164,-285,11320,-299,11399,
	-440,11559,-530,11702,-701,4,962,-4105,977,-3910,856,-3907,814,-4064,7,
	17606,3751,17848,3826,17531,4165,17502,4017,17396,3923,17473,3787,17310,
	3475,3,-7775,-2524,-7745,-2355,-7866,-2476,4,-10986,-7900,-10897,-7829,
	-11062,-7727,-11483,-7802,3,15079,983,15082,935,15014,947,3,-16724,-5386,
	-16665,-5332,-16630,-5403,18,12221,-1833,12239,-1663,12159,-1530,12193,
	-1424,12420,-1391,12414,-1273,12561,-1221,12522,-997,12415,-1135,12482,
	-1184,12287,-1340,12133,-1351,12145,-1208,12035,-1340,12110,-1359,11966,
	-1588,12043,-1632,12067,-1848,6,13009,285,12802,288,12823,353,13009,362,
	13089,409,13095,347,4,3431,-3543,3372,-3475,3283,-3458,3233,-3505,3,
	13142,715,13189,736,13115,795,3,5798,2008,5759,2064,5727,2014,7,-8089,
	-2321,-7380,-2025,-7654,-1983,-7827,-2119,-8055,-2205,-8244,-2237,-8461,
	-2223,3,-84,-6068,-122,-5988,-181,-6026,4,12183,-1063,12266,-1027,12316,
	-1113,12210,-1154,3,17854,1726,17824,1841,17721,1788,10,14290,-5403,
	14325,-5134,14423,-4930,14299,-4886,14269,-4782,14394,-4630,14269,-4655,
	14190,-4594,14231,-5031,14174,-5312,3,15156,288,15109,299,15091,267,3,
	-8939,-7740,-9090,-7714,-9123,-7779,3,10213,-7966,10701,-7821,9950,-7803,
	3,-7302,-2135,-7325,-2084,-7370,-2102,43,-6495,-7968,-6054,-8184,-5104,
	-8222,-4069,-8340,-3047,-8370,-2136,-8266,-2110,-8142,-1804,-8240,-1127,
	-8148,-1687,-8027,-2032,-7852,-1813,-7636,-2029,-7352,-2343,-7216,-2231,
	-7000,-2713,-6811,-3204,-6820,-3482,-6627,-3790,-6556,-4000,-6533,-4121,
	-6308,-4251,-6281,-4233,-6130,-4322,-5982,-4482,-5979,-4558,-6098,-4777,
	-6074,-5017,-6243,-5209,-6471,-5369,-6610,-5374,-6775,-5261,-6864,-5082,
	-6870,-5109,-6941,-5464,-6959,-5271,-7124,-5493,-7148,-5475,-7314,-6045,
	-7619,-6743,-7613,-7065,-7660,-7095,-7758,-7317,-7835,6,-4486,7809,-4326,
	7996,-5066,8109,-5446,8089,-5053,7983,-4940,7829,3,5461,-1264,5393,-1216,
	5336,-1270,4,7145,-7353,7162,-7312,7008,-7312,7017,-7350,3,12716,311,
	12731,400,12589,326,3,341,-3987,305,-3919,246,-3972,5,-10056,-7403,-9760,
	-7399,-9648,-7208,-9887,-7133,-10283,-7287,3,-7725,-2590,-7698,-2629,
	-7736,-2661,4,-7961,-6242,-7920,-6203,-8003,-6156,-8029,-6209,3,-6514,
	-6121,-6467,-6162,-6517,-6180,3,-16801,-5359,-16733,-5321,-16881,-5303,
	20,-11493,-7340,-10922,-7275,-10768,-7186,-10795,-7319,-10443,-7376,
	-10516,-7243,-10372,-7083,-10063,-7008,-10230,-6879,-10514,-6896,-10659,
	-6938,-10826,-6866,-11337,-6861,-11360,-6923,-11643,-6932,-11741,-7012,
	-11221,-7043,-11821,-7092,-11532,-7150,-11914,-7181,3,2319,-5823,2203,
	-5820,2283,-5888,5,13893,-7649,14549,-7566,14472,-7498,13928,-7457,13688,
	-7531,9,1061,-8004,1271,-7806,1567,-7809,1372,-7708,1768,-7637,2047,
	-7842,2286,-7670,2422,-7794,1777,-8004,10,15102,548,15195,411,15245,420,
	15208,338,15360,452,15267,457,15203,523,15053,642,14949,618,14840,565,8,
	9574,-8123,9237,-8052,8891,-8120,8699,-7948,9160,-7886,9468,-7948,9790,
	-7877,10018,-7969,3,-661,-5689,-607,-5743,-655,-5760,3,-9862,-8016,-9877,
	-7959,-10040,-8019,3,16269,1079,16168,1106,16144,1023,6,-11696,-7429,
	-11550,-7363,-12020,-7208,-12286,-7097,-12557,-7208,-12473,-7452,8,-5596,
	-5161,-5615,-5027,-5578,-4941,-5363,-4932,-5319,-4828,-5573,-4689,-5541,
	-4757,-5951,-4757,19,11901,254,12026,-105,12405,-98,12535,-163,12437,-24,
	12063,-40,12011,50,12094,139,12365,51,12343,157,12155,195,12340,528,
	12225,573,12236,479,12162,482,12106,287,12032,284,12043,569,11971,560,4,
	-9768,-7663,-9717,-7518,-10214,-7492,-10454,-7673,3,12085,-583,12148,
	-580,12130,-621,4,-9460,-7855,-9300,-7758,-9635,-7748,-9863,-7882,3,
	-6413,4303,-6366,4256,-6407,4230,6,-9040,-7403,-9160,-7283,-9351,-7230,
	-9533,-7256,-9604,-7336,-9522,-7405,3,12627,166,12500,223,12426,163,3,
	16480,-5921,16448,-5868,16377,-5880,8,12725,-130,12769,7,12701,54,12761,
	101,12826,-26,12891,-156,12826,-162,12852,-251,4,9244,-1130,9274,-1133,
	9310,-1328,9292,-1352,3,15124,-7517,14829,-7448,14627,-7564,4,-9983,
	-7887,-9890,-7771,-10286,-7778,-10588,-7950,3,13032,177,13009,213,12982,
	186,3,-6135,-1574,-6138,-1636,-6183,-1624,8,12323,1071,12489,864,11990,
	901,12089,1037,11862,965,12010,817,12664,809,12733,866,3,-1422,-2820,
	-1375,-2811,-1384,-2865,14,-11393,-7722,-11325,-7695,-11384,-7645,-11100,
	-7571,-10897,-7563,-11049,-7639,-10875,-7698,-10742,-7600,-10517,-7609,
	-10597,-7497,-11071,-7465,-11331,-7444,-11594,-7470,-11765,-7544,3,
	-15604,-2064,-15421,-1940,-15581,-1884,3,13097,-9,13106,38,13065,32,3,
	-708,-5811,-643,-5790,-628,-5852,3,-5797,5110,-5897,5231,-6102,5204,3,
	12414,-1192,12361,-1195,12346,-1257,11,-614,-5228,-611,-5399,-534,-5453,
	-623,-5521,-827,-5518,-842,-5423,-957,-5423,-1013,-5358,-919,-5290,-1034,
	-5195,-933,-5142,4,10680,313,10600,283,10538,180,10591,162,4,-8337,-6280,
	-8195,-6283,-8296,-6221,-8387,-6239,3,1551,-3810,1498,-3653,1246,-3795,3,
	15526,-5011,15603,-5023,15627,-5070,6,-7037,-1983,-6840,-1845,-7138,
	-1784,-7452,-1840,-7248,-1864,-7316,-1974,3,-13168,-5408,-13146,-5207,
	-13324,-5402,3,12683,-3357,12671,-3313,12624,-3334,3,-10421,-7578,-10353,
	-7498,-10498,-7515,6,-11852,-7753,-11545,-7753,-11591,-7671,-11866,-7538,
	-11976,-7583,-12328,-7606,4,-6292,-4972,-6174,-4916,-6307,-4913,-6422,
	-4984,3,15612,677,15532,695,15476,541,3,16816,1624,16778,1659,16739,1615,
	16,9779,-503,9837,-403,10041,-245,10254,-137,10372,-34,10340,67,10443,
	135,10622,344,10587,595,10440,573,10310,475,10103,274,10056,144,9879,
	-164,9652,-375,9516,-553,14,10644,598,10934,692,11441,694,11279,761,
	11577,832,11927,828,11922,886,11711,919,11565,862,11210,846,10972,779,
	10650,757,10640,700,10544,677,3,-2324,-1512,-2372,-1479,-2387,-1521,6,
	-9318,-7678,-9402,-7567,-9335,-7469,-9597,-7443,-9690,-7580,-9713,-7709,
	5,12010,-2352,12025,-2236,12096,-2218,12188,-2414,12144,-2520,9,-8980,
	-7660,-8887,-7549,-8048,-7580,-7853,-7514,-8057,-7425,-8390,-7460,-8869,
	-7434,-9193,-7465,-9278,-7669,27,-303,-5056,58,-5095,138,-5112,90,-5160,
	191,-5281,22,-5299,-31,-5441,-120,-5476,-176,-5583,-286,-5595,-152,-5763,
	-404,-5772,-291,-5887,-513,-5849,-581,-5731,-620,-5618,-472,-5562,-490,
	-5467,-360,-5482,-300,-5399,-324,-5313,-433,-5331,-419,-5251,-522,-5171,
	-268,-5136,-445,-5089,-578,-4994,3,-7139,-2176,-7186,-2161,-7195,-2185,
	21,-9415,-8144,-8443,-8251,-7564,-8337,-6522,-8290,-6106,-8237,-6460,
	-8135,-6660,-8126,-7082,-7984,-7655,-7799,-7850,-7876,-8178,-7864,-8125,
	-7749,-7761,-7696,-7925,-7620,-8302,-7651,-8799,-7602,-8955,-7691,-8728,
	-7722,-8790,-7838,-9203,-7802,-9691,-8015,284,-16378,-6894,-16179,-7011,
	-15578,-7117,-15159,-7046,-14554,-7008,-14090,-6977,-13675,-6884,-12771,
	-7018,-12582,-6941,-12455,-7002,-12397,-6949,-12120,-6982,-12015,-6930,
	-12015,-6705,-12510,-6615,-12142,-6602,-12327,-6474,-11743,-6604,-12015,
	-6705,-12015,-6930,-11406,-6860,-11536,-6787,-11118,-6757,-10878,-6767,
	-10765,-6675,-10698,-6827,-9977,-6768,-9728,-6807,-9538,-6747,-9368,
	-6857,-9368,-6927,-9618,-6997,-9633,-7150,-9467,-7225,-9198,-7067,-9029,
	-6860,-8888,-6927,-8735,-6774,-8558,-6817,-8535,-7000,-8138,-6907,-8138,
	-6737,-8318,-6597,-8568,-6597,-8022,-6385,-8109,-6324,-8320,-6391,-8501,
	-6306,-8661,-6377,-8603,-6602,-8802,-6436,-9066,-6326,-9278,-6228,-9446,
	-6084,-9488,-5905,-9316,-5851,-9241,-5707,-8979,-5707,-8634,-5568,-8476,
	-5517,-8258,-5508,-8212,-5289,-7998,-5129,-7875,-5149,-7853,-5230,-7942,
	-5472,-7771,-5535,-7634,-5617,-7648,-5757,-7837,-5865,-7718,-6017,-7831,
	-6246,-7522,-6222,-7322,-6232,-7151,-6124,-6952,-6088,-6954,-5899,-6774,
	-5834,-6638,-5847,-6475,-6035,-6140,-5717,-6150,-5598,-5738,-5458,-5881,
	-5401,-5716,-5409,-5560,-5324,-5596,-5186,-5850,-5112,-5987,-5020,-6413,
	-5017,-6658,-4997,-6918,-4847,-7093,-4695,-6883,-4821,-6708,-4877,-6439,
	-4911,-6499,-4789,-6468,-4647,-6228,-4567,-6048,-4687,-5948,-4577,-6288,
	-4457,-6528,-4357,-6611,-4417,-6398,-4557,-6493,-4556,-6836,-4423,-7073,
	-4319,-7088,-4221,-7028,-4157,-7335,-4095,-7650,-4320,-7629,-4402,-7898,
	-4377,-7954,-4322,-8190,-4230,-8235,-4320,-8190,-4320,-8121,-4480,-8008,
	-4459,-8048,-4577,-8251,-4610,-8460,-4590,-8460,-4680,-8545,-4815,-8762,
	-4889,-9208,-4677,-8975,-4649,-8808,-4749,-8821,-4690,-8677,-4635,-8460,
	-4680,-8460,-4590,-8647,-4585,-8761,-4482,-8778,-4187,-8698,-4137,-8588,
	-4207,-8608,-4447,-8438,-4567,-8328,-4537,-8348,-4387,-8271,-4401,-8235,
	-4320,-8190,-4230,-8340,-4190,-8318,-4137,-7978,-4197,-7927,-4308,-7874,
	-4313,-7650,-4320,-7335,-4095,-7515,-3870,-7618,-3727,-7528,-3537,-7857,
	-3382,-8050,-3224,-8125,-3107,-8138,-2977,-8008,-2627,-8021,-2528,-8084,
	-2491,-8248,-2677,-8278,-2862,-8348,-3007,-8506,-2968,-8638,-3027,-8906,
	-3010,-8886,-2889,-9133,-2919,-9333,-2963,-9479,-2928,-9537,-2861,-9643,
	-2839,-9742,-2748,-9707,-2546,-9776,-2511,-9788,-2220,-9721,-2059,-9579,
	-1873,-9429,-1826,-9120,-1853,-9039,-2080,-8877,-2149,-8678,-2147,-8733,
	-2001,-8806,-1752,-8889,-1571,-8420,-1583,-8323,-1502,-8368,-1117,-8268,
	-927,-8078,-877,-7920,-945,-7920,-900,-8018,-816,-8038,-710,-8182,-800,
	-8342,-830,-8387,-929,-8572,-992,-8582,-1110,-8769,-1317,-9173,-1406,
	-9410,-1614,-9698,-1561,-10357,-1848,-10556,-2003,-10527,-2193,-10907,
	-2546,-10954,-2646,-11211,-2901,-11321,-3125,-11498,-3167,-11453,-2993,
	-11323,-2893,-11114,-2538,-10935,-2317,-10992,-2294,-11195,-2477,-11219,
	-2595,-11351,-2680,-11471,-2757,-11398,-2838,-11552,-2974,-11751,-3342,
	-12048,-3462,-12230,-3743,-12421,-4017,-12427,-4360,-12403,-4545,-12397,
	-4682,-12470,-4811,-12378,-4787,-12228,-4787,-12298,-4957,-12516,-5033,
	-12344,-4864,-12378,-4827,-12683,-4970,-12823,-5075,-12667,-5059,-12874,
	-5254,-13008,-5384,-13138,-5517,-13298,-5507,-13438,-5647,-13658,-5817,
	-13970,-5960,-14518,-6037,-14798,-6077,-14805,-5992,-14935,-5980,-15164,
	-5903,-15188,-6007,-14990,-6118,-15184,-6104,-15418,-5907,-15318,-5873,
	-15850,-5615,-16220,-5493,-16478,-5417,-16498,-5475,-16008,-5637,-15769,
	-5751,-15710,-5877,-15889,-5848,-16051,-5893,-16177,-5867,-16198,-6007,
	-16388,-5977,-16516,-6062,-16601,-6173,-16448,-6307,-16092,-6350,-16108,
	-6477,-16569,-6439,-16798,-6561,-16425,-6662,-16358,-6603,-16029,-6646,
	-16335,-6721,-16618,-6837,-16611,-6886,3,-9134,120,-9024,73,-9143,14,4,
	-7743,-7370,-7601,-7287,-7962,-7284,-8051,-7364,3,15062,-4626,14869,
	-4514,14704,-4475,3,-785,-7115,-856,-7059,-921,-7092,3,-15734,-2145,
	-15790,-2106,-15831,-2160,82,-7839,-830,-7920,-900,-7920,-945,-7686,-815,
	-7565,-954,-7546,-1073,-7111,-1256,-7166,-1075,-7054,-1133,-6998,-1210,
	-6787,-1051,-6504,-1018,-6261,-1068,-6202,-982,-6122,-1059,-6072,-867,
	-5935,-822,-5699,-580,-5395,-587,-5152,-431,-4992,-130,-5063,4,-4965,-2,
	-4486,149,-4409,261,-3954,291,-3682,521,-3528,510,-3469,735,-3501,871,
	-3883,1311,-3961,1905,-4198,2283,-4461,2310,-4838,2553,-4861,2836,-5130,
	3106,-5436,3507,-5865,3415,-5655,3662,-5776,3829,-6203,3913,-6235,4108,
	-6516,4095,-6442,4273,-6575,4501,-6791,4596,-6569,4758,-6912,5056,-6903,
	5246,-6676,5455,-6413,5484,-6853,5567,-7226,5449,-7451,5302,-7540,5094,
	-7552,4813,-7436,4734,-7525,4639,-7329,4445,-7273,4235,-7439,4345,-7338,
	3934,-7359,3750,-7170,3348,-7158,3028,-7042,2522,-7007,2137,-7039,1827,
	-7400,1613,-7623,1427,-7641,1317,-8001,665,-8103,619,-8126,419,-8004,298,
	-8091,217,-7999,-61,-7910,-135,-7830,-262,-7727,-327,-7736,-647,4,5680,
	-7963,5860,-8040,6526,-8001,6698,-7907,3,9925,232,9934,153,9854,102,41,
	-8490,-7394,-8687,-7265,-8468,-7099,-8557,-7282,-8370,-7350,-8108,-7363,
	-7975,-7212,-7687,-7252,-7305,-7150,-6737,-7017,-6670,-6915,-6857,-6879,
	-6124,-6671,-6364,-6502,-6715,-6657,-6795,-6591,-6409,-6359,-6506,-6259,
	-6855,-6374,-6590,-6196,-6912,-6250,-7092,-6279,-7355,-6436,-7838,-6444,
	-7749,-6534,-7340,-6546,-7420,-6635,-7198,-6719,-7283,-6799,-7494,-6842,
	-7515,-6735,-7704,-6719,-7645,-6942,-7847,-7040,-7991,-6960,-8163,-7013,
	-8554,-7040,-8829,-7034,-9027,-7149,-8960,-7274,-8863,-7367,3,-16996,
	-6359,-16893,-6291,-17192,-6371,3,-17322,-5211,-17458,-5193,-17417,-5250,
	3,-1747,-2852,-1789,-2852,-1786,-2885,3,16011,828,15937,852,15851,766,4,
	2473,-7859,2792,-8001,2337,-8081,1872,-8013,8,14154,-4371,13991,-4211,
	14053,-4167,14157,-4250,14328,-4199,14553,-4353,14343,-4418,14172,-4552,
	3,-10398,-7714,-10513,-7702,-10614,-7799,646,2668,-7104,2680,-7034,2795,
	-7110,3109,-7045,2938,-6990,3272,-6948,3582,-6915,3914,-6820,4124,-6744,
	4100,-6667,3902,-6602,3642,-6631,3269,-6693,3455,-6606,3464,-6498,3505,
	-6445,3671,-6400,3775,-6380,3799,-6431,3650,-6474,3751,-6525,3804,-6480,
	3955,-6442,4053,-6483,3991,-6542,4248,-6646,4399,-6616,4464,-6699,4375,
	-6738,4437,-6844,4582,-6853,4656,-6812,4511,-6729,4588,-6693,4785,-6684,
	4809,-6767,5111,-6832,5250,-6832,5377,-6900,5350,-6818,5486,-6806,5516,
	-6847,5770,-6856,5939,-6889,5924,-6829,6016,-6838,6128,-6891,6004,-6954,
	5851,-7026,5999,-7014,6053,-6976,6253,-6980,6679,-6886,6847,-6812,6901,
	-6874,6696,-6968,6747,-7007,6741,-7069,6661,-7113,6850,-7164,6918,-7291,
	7087,-7273,7285,-7264,7196,-7140,7291,-7066,7250,-6989,7253,-6874,7353,
	-6850,7306,-6761,7152,-6681,6981,-6650,7229,-6619,7418,-6714,7495,-6797,
	7457,-6868,7673,-6891,7741,-6741,7809,-6758,7795,-6865,7697,-6930,7380,
	-6912,7368,-7007,7439,-7054,7321,-7155,7546,-7255,7567,-7134,7824,-7096,
	7623,-7184,7812,-7181,7750,-7261,8010,-7217,8176,-7149,8303,-7161,8218,
	-7087,8363,-7051,8327,-7178,8215,-7238,8081,-7235,8055,-7353,8724,-7383,
	8576,-7460,8957,-7551,9449,-7625,9804,-7596,9913,-7658,10150,-7643,10091,
	-7702,10351,-7782,10768,-7749,10679,-7661,10966,-7687,11265,-7664,11401,
	-7590,11363,-7522,11020,-7442,10567,-7312,10910,-7338,11014,-7391,11305,
	-7378,11382,-7333,11579,-7374,11860,-7353,11882,-7300,12356,-7297,12341,
	-7386,12782,-7356,12968,-7300,12885,-7175,13128,-7057,13261,-7193,13402,
	-7129,13632,-7167,14007,-7143,13940,-7230,14245,-7277,14928,-7215,15265,
	-7078,15914,-7099,16080,-6954,16700,-6948,16911,-7011,16885,-6916,17100,
	-6889,17050,-6993,17662,-6986,18000,-6885,18000,-6480,17674,-6471,17872,
	-6397,17958,-6247,17709,-6267,17393,-6178,17023,-6007,16890,-6057,16617,
	-5968,16606,-6039,16363,-5986,16304,-5892,16177,-5791,16307,-5770,16316,
	-5616,16194,-5590,16182,-5474,16012,-5422,15982,-5315,15873,-5312,15861,
	-5200,15677,-5087,15660,-5226,15585,-5365,15556,-5558,15668,-5735,15876,
	-5806,16080,-5975,16358,-6078,16485,-6259,16331,-6250,16302,-6149,16204,
	-6161,16065,-6058,16000,-6120,16056,-6191,15751,-6188,15411,-5975,15524,
	-5922,15136,-5871,15210,-5925,14926,-5981,14852,-5916,14651,-5951,14618,
	-5910,14580,-5936,14260,-5939,14014,-5771,13780,-5629,13522,-5496,13552,
	-5425,13670,-5463,13673,-5380,13765,-5434,13789,-5357,13866,-5354,13890,
	-5416,14006,-5404,14133,-5324,14129,-5170,14040,-5076,14025,-4874,13753,
	-4567,13511,-4351,13321,-4259,13200,-4327,12975,-4191,12969,-4073,12753,
	-3963,12780,-3874,12839,-3857,12940,-3718,12937,-3513,12653,-3430,12621,
	-3516,12683,-3599,12647,-3687,12677,-3735,12624,-3803,12514,-3797,12502,
	-3841,12544,-3933,12420,-3977,12266,-3945,12165,-3896,12139,-3961,12213,
	-4013,12230,-4072,12094,-4081,12035,-4010,11952,-3998,11914,-3918,11781,
	-3930,11745,-3880,11795,-3812,11914,-3815,11914,-3726,11994,-3720,12076,
	-3788,12263,-3729,12236,-3667,12112,-3670,11914,-3516,11940,-3457,12029,
	-3439,12094,-3235,12201,-3184,12165,-3143,12189,-3093,12047,-3013,12236,
	-3025,12162,-2903,12142,-2797,12076,-2776,11973,-2640,11964,-2522,11704,
	-2359,11621,-2261,11405,-2264,11254,-2161,11067,-2125,11040,-2037,10990,
	-2046,10969,-2144,10786,-2158,10567,-1954,10576,-1849,10686,-1727,10890,
	-1529,10934,-1285,10919,-1137,10688,-1040,10635,-925,10475,-848,10499,
	-1001,10366,-1051,10285,-1212,10116,-1268,10104,-1327,10011,-1332,9993,
	-1184,9901,-1016,9913,-927,9981,-930,10055,-697,10162,-671,10351,-481,
	10345,-280,10449,-96,10117,-298,10032,-517,10026,-646,9827,-845,9865,
	-1123,9860,-1306,9789,-1463,9724,-1720,9560,-1573,9427,-1587,9459,-1812,
	9371,-1895,9362,-1978,9267,-2028,9190,-2238,9119,-2327,9098,-2202,8713,
	-2161,8687,-2028,8616,-1960,8530,-1963,8373,-1791,8252,-1732,8228,-1634,
	8145,-1629,8113,-1566,8024,-1563,8024,-1342,8018,-1235,7974,-1179,7981,
	-1040,7895,-996,7906,-915,7823,-895,7778,-791,7704,-830,7628,-969,7595,
	-1123,7486,-1291,7453,-1418,7355,-1593,7296,-1827,7275,-2025,7299,-2099,
	7246,-2214,7202,-2126,7095,-2069,6894,-2222,7042,-2263,7039,-2307,6912,
	-2287,6811,-2387,6746,-2384,6648,-2556,6157,-2515,5746,-2568,5693,-2713,
	5471,-2648,5296,-2695,5225,-2772,5137,-2799,5018,-3026,4906,-3029,4805,
	-2929,4908,-2751,5027,-2654,5062,-2479,5136,-2606,5166,-2390,5420,-2405,
	5630,-2615,5657,-2449,5790,-2364,5882,-2352,6003,-2220,5781,-1989,5772,
	-1897,5672,-1873,5636,-1788,5539,-1778,5520,-1702,5240,-1654,5209,-1536,
	4952,-1465,4747,-1359,4591,-1347,4482,-1272,4364,-1266,4272,-1544,4266,
	-1704,4127,-1870,4080,-1997,3965,-2044,3920,-2133,3879,-2334,3742,-2453,
	3719,-2526,3541,-2795,3467,-2819,3488,-2946,3408,-2774,3240,-3015,3240,
	-3105,3432,-3118,3603,-3443,3616,-3681,3468,-3684,3381,-3603,3260,-3597,
	3180,-3677,3068,-3680,3038,-3606,2914,-3618,2908,-3671,2778,-3689,2713,
	-3801,2630,-3819,2692,-3852,2683,-3943,2606,-3943,2634,-4018,2951,-4066,
	2835,-4095,2835,-4140,3125,-4116,3396,-4205,3615,-4154,3772,-4098,4071,
	-4092,4185,-4185,4860,-4185,4997,-4048,4881,-3865,4951,-3743,5151,-3646,
	5396,-3657,5390,-3865,5294,-4007,5304,-4092,5482,-4098,5340,-4217,5287,
	-4128,5230,-4178,5290,-4285,5193,-4294,5006,-4456,5304,-4563,5310,-4590,
	5895,-4590,5823,-4427,5923,-4507,5935,-4406,6051,-4463,6039,-4607,6210,
	-4545,7335,-4545,7418,-4531,7527,-4628,7906,-4643,7920,-4680,10395,-5130,
	10555,-5137,10912,-5332,11034,-5536,10966,-5578,10850,-5403,10596,-5220,
	10427,-5193,10395,-5130,7920,-4680,7450,-4664,7335,-4545,6210,-4545,6169,
	-4631,6000,-4670,5895,-4590,5310,-4590,5310,-4687,5038,-4702,4864,-4572,
	4742,-4557,4671,-4441,4781,-4370,4751,-4267,4860,-4185,4185,-4185,4151,
	-4273,3869,-4421,3689,-4504,3822,-4613,3908,-4717,3485,-4640,3509,-4533,
	3631,-4521,3369,-4430,3272,-4551,3372,-4616,3118,-4669,2952,-4539,2864,
	-4382,2767,-4249,2835,-4140,2835,-4095,2603,-4066,2390,-4078,2266,-4027,
	2440,-3740,2254,-3844,2130,-3841,1958,-4030,1967,-4172,1580,-4376,1464,
	-4527,1387,-4486,1367,-4560,1233,-4551,1260,-4391,1381,-4350,1426,-4231,
	1592,-4172,1852,-4030,1835,-3986,1696,-4048,1663,-3989,1731,-3909,1577,
	-3799,1624,-3900,1553,-4009,1233,-4157,1057,-4299,1026,-4397,887,-4441,
	787,-4379,609,-4302,417,-4347,301,-4323,313,-4172,97,-4104,-18,-3953,-20,
	-3859,-76,-3776,-242,-3669,-441,-3663,-533,-3604,-598,-3613,-669,-3714,
	-865,-3711,-865,-3832,-947,-3867,-853,-4060,-873,-4205,-918,-4308,-743,
	-4362,-400,-4344,-163,-4332,-110,-4427,-122,-4625,-249,-4749,-439,-4794,
	-459,-4867,-152,-4856,-175,-4965,-34,-4930,147,-5004,171,-5083,378,-5146,
	507,-5306,545,-5241,568,-5329,903,-5380,829,-5578,831,-5676,1036,-5764,
	1071,-5646,1003,-5587,1065,-5519,965,-5516,1091,-5387,1343,-5448,1432,
	-5359,1761,-5477,1923,-5430,2116,-5498,2119,-5668,2208,-5762,2385,-5685,
	2444,-5733,2450,-5830,2362,-5878,2403,-5937,2655,-5949,2962,-5975,3288,
	-6032,3261,-6108,3110,-6171,3030,-6123,3107,-6032,2953,-6014,2850,-6064,
	2365,-5990,2134,-6058,2202,-6182,2119,-6259,2187,-6332,2406,-6414,2447,
	-6488,2548,-6494,2542,-6585,2234,-6588,2110,-6494,2131,-6408,1734,-6229,
	1726,-6088,1924,-5976,1817,-5878,1681,-5857,1640,-5701,1687,-5677,1462,
	-5606,1397,-5534,1300,-5560,1285,-5664,1071,-5971,965,-5877,829,-5789,
	687,-5804,560,-5860,628,-5923,539,-5937,551,-5994,491,-6099,545,-6213,
	764,-6307,853,-6278,858,-6342,1024,-6334,1115,-6372,1015,-6395,1131,
	-6476,1249,-6517,1231,-6579,1417,-6700,1648,-6806,1464,-6845,1778,-6880,
	1705,-6934,1905,-6996,2157,-6991,2314,-7065,2562,-7099,2509,-7014,3,8086,
	-8099,7958,-8058,7881,-8108,3,-6753,7829,-7066,8023,-7240,7969,4,-5313,
	-4799,-5248,-4736,-5304,-4668,-5407,-4686,88,-18000,8415,-15165,8550,
	-16155,8415,-15273,8236,-15660,8100,-14926,8076,-15680,7789,-13680,7470,
	-12555,7380,-11320,7456,-11126,7429,-10886,7523,-10093,7503,-10413,7263,
	-9630,7200,-8546,7383,-8100,7380,-7886,7283,-7650,7335,-7515,7155,-7155,
	7065,-7213,6929,-6660,6885,-6926,6743,-6615,6660,-6345,6435,-5760,6345,
	-5670,6435,-6030,6525,-6120,6615,-6480,6750,-6435,6840,-6210,6930,-6093,
	7096,-6013,7369,-6380,7569,-7290,7695,-7425,7830,-7740,7830,-7786,7923,
	-7633,7963,-7515,8100,-6885,8100,-5960,8369,-5445,8235,-4635,8190,-4545,
	8280,-4095,8145,-2973,8009,-3626,7849,-2610,7605,-1840,7576,-1710,7380,
	-686,7109,-440,7189,1347,7056,2540,7116,3240,7020,3330,6885,3900,7023,
	4005,6885,5310,6615,5985,6750,7020,6795,6920,7029,6795,7245,7065,7200,
	7300,7003,7634,6956,8674,6663,9855,6660,10215,6570,10800,6705,11267,6616,
	11655,6750,14000,6676,15210,6930,15427,6856,16200,7065,17100,7176,16307,
	7496,16245,7650,16707,7869,16020,7920,16194,8196,18000,8460,18000,9000,
	-18000,9000,4,12236,-938,12337,-891,12346,-1024,12337,-1086,3,-2484,
	-1692,-2534,-1675,-2517,-1722,3,-649,-6216,-696,-6183,-735,-6216,3,5597,
	2092,5561,2149,5517,2104,3,7061,4938,6922,4994,6880,4902,3,12804,145,
	12778,177,12745,151,4,16701,2219,16671,2257,16440,2088,16428,2014,3,5165,
	-8111,5135,-7984,4511,-8069,3,-6789,-6065,-6801,-6014,-6851,-6035,5,7974,
	-760,8030,-580,8187,-627,8199,-760,8027,-976,3,4819,-6851,5050,-6907,
	4831,-6945,3,1813,-5746,1860,-5704,1911,-5802,0
};
