/* 
 * SHA-512 hash in x86-64 assembly
 * 
 * Copyright (c) 2017 Project Nayuki. (MIT License)
 * https://www.nayuki.io/page/fast-sha2-hashes-in-x86-assembly
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * - The above copyright notice and this permission notice shall be included in
 *   all copies or substantial portions of the Software.
 * - The Software is provided "as is", without warranty of any kind, express or
 *   implied, including but not limited to the warranties of merchantability,
 *   fitness for a particular purpose and noninfringement. In no event shall the
 *   authors or copyright holders be liable for any claim, damages or other
 *   liability, whether in an action of contract, tort or otherwise, arising from,
 *   out of or in connection with the Software or the use or other dealings in the
 *   Software.
 */
 
# Adapted for VeraCrypt
# Adapt to Windows build:
#  - specific calling convention
#  - avoid using xmm6 register since it must be preserved. We use the stack to save RBX, RDI and RSI


/* void sha512_compress_nayuki(uint64_t state[8], const uint8_t block[128]) */
.globl sha512_compress_nayuki
.globl _sha512_compress_nayuki
sha512_compress_nayuki:
_sha512_compress_nayuki:
	/* 
	 * Storage usage:
	 *   Bytes  Location  Description
	 *       8  rax       Temporary for calculation per round
	 *       8  rbx       Temporary for calculation per round
	 *       8  rcx       Temporary for calculation per round
	 *       8  rdx       Temporary for calculation per round
	 *       8  rsi       Base address of block array argument (read-only)
	 *       8  rdi       Base address of state array argument (read-only)
	 *       8  rsp       x86-64 stack pointer
	 *       8  r8        SHA-512 state variable A
	 *       8  r9        SHA-512 state variable B
	 *       8  r10       SHA-512 state variable C
	 *       8  r11       SHA-512 state variable D
	 *       8  r12       SHA-512 state variable E
	 *       8  r13       SHA-512 state variable F
	 *       8  r14       SHA-512 state variable G
	 *       8  r15       SHA-512 state variable H
	 *     128  [rsp+0]   Circular buffer of most recent 16 key schedule items, 8 bytes each
	 *      16  xmm0      Caller's value of r10 (only low 64 bits are used)
	 *      16  xmm1      Caller's value of r11 (only low 64 bits are used)
	 *      16  xmm2      Caller's value of r12 (only low 64 bits are used)
	 *      16  xmm3      Caller's value of r13 (only low 64 bits are used)
	 *      16  xmm4      Caller's value of r14 (only low 64 bits are used)
	 *      16  xmm5      Caller's value of r15 (only low 64 bits are used)
	 *       8  mm0       Caller's value of rbx
	 */
 movq %r10, %xmm0
 movq %r11, %xmm1
 movq %r12, %xmm2
 movq %r13, %xmm3
 movq %r14, %xmm4
 movq %r15, %xmm5

.ifdef WINABI
 subq $152, %rsp
 movq %rbx, (0*8 + 128)(%rsp)
 movq %rdi, (1*8 + 128)(%rsp)
 movq %rsi, (2*8 + 128)(%rsp)
 movq %rcx, %rdi
 movq %rdx, %rsi
 .else
 movq %rbx, %xmm6
 subq $128, %rsp
.endif
 

 movq 0(%rdi), %r8
 movq 8(%rdi), %r9
 movq 16(%rdi), %r10
 movq 24(%rdi), %r11
 movq 32(%rdi), %r12
 movq 40(%rdi), %r13
 movq 48(%rdi), %r14
 movq 56(%rdi), %r15


 movq (0*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((0)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0x428A2F98D728AE22, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (1*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((1)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0x7137449123EF65CD, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (2*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((2)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0xB5C0FBCFEC4D3B2F, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (3*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((3)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0xE9B5DBA58189DBBC, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (4*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((4)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x3956C25BF348B538, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (5*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((5)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0x59F111F1B605D019, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (6*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((6)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x923F82A4AF194F9B, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (7*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((7)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0xAB1C5ED5DA6D8118, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (8*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((8)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0xD807AA98A3030242, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (9*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((9)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0x12835B0145706FBE, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (10*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((10)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0x243185BE4EE4B28C, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (11*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((11)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0x550C7DC3D5FFB4E2, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (12*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((12)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x72BE5D74F27B896F, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (13*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((13)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0x80DEB1FE3B1696B1, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (14*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((14)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x9BDC06A725C71235, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (15*8)(%rsi), %rbx; bswapq %rbx; movq %rbx, (((15)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0xC19BF174CF692694, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((16 -15)&0xF)*8)(%rsp), %rax; movq (((16 -16)&0xF)*8)(%rsp), %rbx; addq (((16 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((16 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((16)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0xE49B69C19EF14AD2, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((17 -15)&0xF)*8)(%rsp), %rax; movq (((17 -16)&0xF)*8)(%rsp), %rbx; addq (((17 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((17 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((17)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0xEFBE4786384F25E3, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((18 -15)&0xF)*8)(%rsp), %rax; movq (((18 -16)&0xF)*8)(%rsp), %rbx; addq (((18 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((18 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((18)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0x0FC19DC68B8CD5B5, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((19 -15)&0xF)*8)(%rsp), %rax; movq (((19 -16)&0xF)*8)(%rsp), %rbx; addq (((19 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((19 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((19)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0x240CA1CC77AC9C65, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((20 -15)&0xF)*8)(%rsp), %rax; movq (((20 -16)&0xF)*8)(%rsp), %rbx; addq (((20 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((20 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((20)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x2DE92C6F592B0275, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((21 -15)&0xF)*8)(%rsp), %rax; movq (((21 -16)&0xF)*8)(%rsp), %rbx; addq (((21 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((21 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((21)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0x4A7484AA6EA6E483, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((22 -15)&0xF)*8)(%rsp), %rax; movq (((22 -16)&0xF)*8)(%rsp), %rbx; addq (((22 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((22 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((22)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x5CB0A9DCBD41FBD4, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((23 -15)&0xF)*8)(%rsp), %rax; movq (((23 -16)&0xF)*8)(%rsp), %rbx; addq (((23 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((23 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((23)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0x76F988DA831153B5, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((24 -15)&0xF)*8)(%rsp), %rax; movq (((24 -16)&0xF)*8)(%rsp), %rbx; addq (((24 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((24 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((24)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0x983E5152EE66DFAB, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((25 -15)&0xF)*8)(%rsp), %rax; movq (((25 -16)&0xF)*8)(%rsp), %rbx; addq (((25 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((25 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((25)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0xA831C66D2DB43210, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((26 -15)&0xF)*8)(%rsp), %rax; movq (((26 -16)&0xF)*8)(%rsp), %rbx; addq (((26 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((26 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((26)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0xB00327C898FB213F, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((27 -15)&0xF)*8)(%rsp), %rax; movq (((27 -16)&0xF)*8)(%rsp), %rbx; addq (((27 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((27 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((27)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0xBF597FC7BEEF0EE4, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((28 -15)&0xF)*8)(%rsp), %rax; movq (((28 -16)&0xF)*8)(%rsp), %rbx; addq (((28 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((28 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((28)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0xC6E00BF33DA88FC2, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((29 -15)&0xF)*8)(%rsp), %rax; movq (((29 -16)&0xF)*8)(%rsp), %rbx; addq (((29 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((29 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((29)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0xD5A79147930AA725, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((30 -15)&0xF)*8)(%rsp), %rax; movq (((30 -16)&0xF)*8)(%rsp), %rbx; addq (((30 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((30 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((30)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x06CA6351E003826F, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((31 -15)&0xF)*8)(%rsp), %rax; movq (((31 -16)&0xF)*8)(%rsp), %rbx; addq (((31 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((31 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((31)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0x142929670A0E6E70, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((32 -15)&0xF)*8)(%rsp), %rax; movq (((32 -16)&0xF)*8)(%rsp), %rbx; addq (((32 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((32 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((32)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0x27B70A8546D22FFC, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((33 -15)&0xF)*8)(%rsp), %rax; movq (((33 -16)&0xF)*8)(%rsp), %rbx; addq (((33 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((33 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((33)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0x2E1B21385C26C926, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((34 -15)&0xF)*8)(%rsp), %rax; movq (((34 -16)&0xF)*8)(%rsp), %rbx; addq (((34 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((34 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((34)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0x4D2C6DFC5AC42AED, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((35 -15)&0xF)*8)(%rsp), %rax; movq (((35 -16)&0xF)*8)(%rsp), %rbx; addq (((35 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((35 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((35)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0x53380D139D95B3DF, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((36 -15)&0xF)*8)(%rsp), %rax; movq (((36 -16)&0xF)*8)(%rsp), %rbx; addq (((36 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((36 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((36)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x650A73548BAF63DE, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((37 -15)&0xF)*8)(%rsp), %rax; movq (((37 -16)&0xF)*8)(%rsp), %rbx; addq (((37 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((37 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((37)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0x766A0ABB3C77B2A8, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((38 -15)&0xF)*8)(%rsp), %rax; movq (((38 -16)&0xF)*8)(%rsp), %rbx; addq (((38 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((38 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((38)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x81C2C92E47EDAEE6, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((39 -15)&0xF)*8)(%rsp), %rax; movq (((39 -16)&0xF)*8)(%rsp), %rbx; addq (((39 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((39 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((39)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0x92722C851482353B, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((40 -15)&0xF)*8)(%rsp), %rax; movq (((40 -16)&0xF)*8)(%rsp), %rbx; addq (((40 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((40 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((40)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0xA2BFE8A14CF10364, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((41 -15)&0xF)*8)(%rsp), %rax; movq (((41 -16)&0xF)*8)(%rsp), %rbx; addq (((41 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((41 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((41)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0xA81A664BBC423001, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((42 -15)&0xF)*8)(%rsp), %rax; movq (((42 -16)&0xF)*8)(%rsp), %rbx; addq (((42 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((42 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((42)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0xC24B8B70D0F89791, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((43 -15)&0xF)*8)(%rsp), %rax; movq (((43 -16)&0xF)*8)(%rsp), %rbx; addq (((43 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((43 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((43)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0xC76C51A30654BE30, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((44 -15)&0xF)*8)(%rsp), %rax; movq (((44 -16)&0xF)*8)(%rsp), %rbx; addq (((44 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((44 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((44)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0xD192E819D6EF5218, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((45 -15)&0xF)*8)(%rsp), %rax; movq (((45 -16)&0xF)*8)(%rsp), %rbx; addq (((45 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((45 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((45)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0xD69906245565A910, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((46 -15)&0xF)*8)(%rsp), %rax; movq (((46 -16)&0xF)*8)(%rsp), %rbx; addq (((46 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((46 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((46)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0xF40E35855771202A, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((47 -15)&0xF)*8)(%rsp), %rax; movq (((47 -16)&0xF)*8)(%rsp), %rbx; addq (((47 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((47 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((47)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0x106AA07032BBD1B8, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((48 -15)&0xF)*8)(%rsp), %rax; movq (((48 -16)&0xF)*8)(%rsp), %rbx; addq (((48 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((48 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((48)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0x19A4C116B8D2D0C8, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((49 -15)&0xF)*8)(%rsp), %rax; movq (((49 -16)&0xF)*8)(%rsp), %rbx; addq (((49 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((49 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((49)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0x1E376C085141AB53, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((50 -15)&0xF)*8)(%rsp), %rax; movq (((50 -16)&0xF)*8)(%rsp), %rbx; addq (((50 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((50 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((50)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0x2748774CDF8EEB99, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((51 -15)&0xF)*8)(%rsp), %rax; movq (((51 -16)&0xF)*8)(%rsp), %rbx; addq (((51 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((51 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((51)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0x34B0BCB5E19B48A8, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((52 -15)&0xF)*8)(%rsp), %rax; movq (((52 -16)&0xF)*8)(%rsp), %rbx; addq (((52 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((52 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((52)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x391C0CB3C5C95A63, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((53 -15)&0xF)*8)(%rsp), %rax; movq (((53 -16)&0xF)*8)(%rsp), %rbx; addq (((53 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((53 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((53)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0x4ED8AA4AE3418ACB, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((54 -15)&0xF)*8)(%rsp), %rax; movq (((54 -16)&0xF)*8)(%rsp), %rbx; addq (((54 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((54 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((54)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x5B9CCA4F7763E373, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((55 -15)&0xF)*8)(%rsp), %rax; movq (((55 -16)&0xF)*8)(%rsp), %rbx; addq (((55 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((55 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((55)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0x682E6FF3D6B2B8A3, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((56 -15)&0xF)*8)(%rsp), %rax; movq (((56 -16)&0xF)*8)(%rsp), %rbx; addq (((56 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((56 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((56)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0x748F82EE5DEFB2FC, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((57 -15)&0xF)*8)(%rsp), %rax; movq (((57 -16)&0xF)*8)(%rsp), %rbx; addq (((57 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((57 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((57)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0x78A5636F43172F60, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((58 -15)&0xF)*8)(%rsp), %rax; movq (((58 -16)&0xF)*8)(%rsp), %rbx; addq (((58 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((58 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((58)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0x84C87814A1F0AB72, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((59 -15)&0xF)*8)(%rsp), %rax; movq (((59 -16)&0xF)*8)(%rsp), %rbx; addq (((59 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((59 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((59)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0x8CC702081A6439EC, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((60 -15)&0xF)*8)(%rsp), %rax; movq (((60 -16)&0xF)*8)(%rsp), %rbx; addq (((60 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((60 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((60)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x90BEFFFA23631E28, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((61 -15)&0xF)*8)(%rsp), %rax; movq (((61 -16)&0xF)*8)(%rsp), %rbx; addq (((61 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((61 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((61)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0xA4506CEBDE82BDE9, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((62 -15)&0xF)*8)(%rsp), %rax; movq (((62 -16)&0xF)*8)(%rsp), %rbx; addq (((62 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((62 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((62)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0xBEF9A3F7B2C67915, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((63 -15)&0xF)*8)(%rsp), %rax; movq (((63 -16)&0xF)*8)(%rsp), %rbx; addq (((63 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((63 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((63)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0xC67178F2E372532B, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((64 -15)&0xF)*8)(%rsp), %rax; movq (((64 -16)&0xF)*8)(%rsp), %rbx; addq (((64 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((64 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((64)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0xCA273ECEEA26619C, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((65 -15)&0xF)*8)(%rsp), %rax; movq (((65 -16)&0xF)*8)(%rsp), %rbx; addq (((65 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((65 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((65)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0xD186B8C721C0C207, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((66 -15)&0xF)*8)(%rsp), %rax; movq (((66 -16)&0xF)*8)(%rsp), %rbx; addq (((66 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((66 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((66)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0xEADA7DD6CDE0EB1E, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((67 -15)&0xF)*8)(%rsp), %rax; movq (((67 -16)&0xF)*8)(%rsp), %rbx; addq (((67 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((67 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((67)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0xF57D4F7FEE6ED178, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((68 -15)&0xF)*8)(%rsp), %rax; movq (((68 -16)&0xF)*8)(%rsp), %rbx; addq (((68 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((68 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((68)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x06F067AA72176FBA, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((69 -15)&0xF)*8)(%rsp), %rax; movq (((69 -16)&0xF)*8)(%rsp), %rbx; addq (((69 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((69 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((69)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0x0A637DC5A2C898A6, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((70 -15)&0xF)*8)(%rsp), %rax; movq (((70 -16)&0xF)*8)(%rsp), %rbx; addq (((70 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((70 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((70)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x113F9804BEF90DAE, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((71 -15)&0xF)*8)(%rsp), %rax; movq (((71 -16)&0xF)*8)(%rsp), %rbx; addq (((71 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((71 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((71)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0x1B710B35131C471B, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;
 movq (((72 -15)&0xF)*8)(%rsp), %rax; movq (((72 -16)&0xF)*8)(%rsp), %rbx; addq (((72 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((72 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((72)&0xF)*8)(%rsp); movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r15; movq %r14, %rcx; xorq %r13, %rcx; andq %r12, %rcx; xorq %r14, %rcx; addq %rax, %r15; movabs $0x28DB77F523047D84, %rax; addq %rcx, %r15; addq %rax, %r15; addq %r15, %r11; movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r10, %rcx; addq %rax, %r15; movq %r10, %rax; orq %r9, %rax; andq %r9, %rcx; andq %r8, %rax; orq %rcx, %rax; addq %rax, %r15;
 movq (((73 -15)&0xF)*8)(%rsp), %rax; movq (((73 -16)&0xF)*8)(%rsp), %rbx; addq (((73 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((73 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((73)&0xF)*8)(%rsp); movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r14; movq %r13, %rcx; xorq %r12, %rcx; andq %r11, %rcx; xorq %r13, %rcx; addq %rax, %r14; movabs $0x32CAAB7B40C72493, %rax; addq %rcx, %r14; addq %rax, %r14; addq %r14, %r10; movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r9, %rcx; addq %rax, %r14; movq %r9, %rax; orq %r8, %rax; andq %r8, %rcx; andq %r15, %rax; orq %rcx, %rax; addq %rax, %r14;
 movq (((74 -15)&0xF)*8)(%rsp), %rax; movq (((74 -16)&0xF)*8)(%rsp), %rbx; addq (((74 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((74 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((74)&0xF)*8)(%rsp); movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r13; movq %r12, %rcx; xorq %r11, %rcx; andq %r10, %rcx; xorq %r12, %rcx; addq %rax, %r13; movabs $0x3C9EBE0A15C9BEBC, %rax; addq %rcx, %r13; addq %rax, %r13; addq %r13, %r9; movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r8, %rcx; addq %rax, %r13; movq %r8, %rax; orq %r15, %rax; andq %r15, %rcx; andq %r14, %rax; orq %rcx, %rax; addq %rax, %r13;
 movq (((75 -15)&0xF)*8)(%rsp), %rax; movq (((75 -16)&0xF)*8)(%rsp), %rbx; addq (((75 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((75 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((75)&0xF)*8)(%rsp); movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r12; movq %r11, %rcx; xorq %r10, %rcx; andq %r9, %rcx; xorq %r11, %rcx; addq %rax, %r12; movabs $0x431D67C49C100D4C, %rax; addq %rcx, %r12; addq %rax, %r12; addq %r12, %r8; movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r15, %rcx; addq %rax, %r12; movq %r15, %rax; orq %r14, %rax; andq %r14, %rcx; andq %r13, %rax; orq %rcx, %rax; addq %rax, %r12;
 movq (((76 -15)&0xF)*8)(%rsp), %rax; movq (((76 -16)&0xF)*8)(%rsp), %rbx; addq (((76 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((76 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((76)&0xF)*8)(%rsp); movq %r8, %rcx; movq %r8, %rdx; movq %r8, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r11; movq %r10, %rcx; xorq %r9, %rcx; andq %r8, %rcx; xorq %r10, %rcx; addq %rax, %r11; movabs $0x4CC5D4BECB3E42B6, %rax; addq %rcx, %r11; addq %rax, %r11; addq %r11, %r15; movq %r12, %rcx; movq %r12, %rdx; movq %r12, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r14, %rcx; addq %rax, %r11; movq %r14, %rax; orq %r13, %rax; andq %r13, %rcx; andq %r12, %rax; orq %rcx, %rax; addq %rax, %r11;
 movq (((77 -15)&0xF)*8)(%rsp), %rax; movq (((77 -16)&0xF)*8)(%rsp), %rbx; addq (((77 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((77 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((77)&0xF)*8)(%rsp); movq %r15, %rcx; movq %r15, %rdx; movq %r15, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r10; movq %r9, %rcx; xorq %r8, %rcx; andq %r15, %rcx; xorq %r9, %rcx; addq %rax, %r10; movabs $0x597F299CFC657E2A, %rax; addq %rcx, %r10; addq %rax, %r10; addq %r10, %r14; movq %r11, %rcx; movq %r11, %rdx; movq %r11, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r13, %rcx; addq %rax, %r10; movq %r13, %rax; orq %r12, %rax; andq %r12, %rcx; andq %r11, %rax; orq %rcx, %rax; addq %rax, %r10;
 movq (((78 -15)&0xF)*8)(%rsp), %rax; movq (((78 -16)&0xF)*8)(%rsp), %rbx; addq (((78 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((78 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((78)&0xF)*8)(%rsp); movq %r14, %rcx; movq %r14, %rdx; movq %r14, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r9; movq %r8, %rcx; xorq %r15, %rcx; andq %r14, %rcx; xorq %r8, %rcx; addq %rax, %r9; movabs $0x5FCB6FAB3AD6FAEC, %rax; addq %rcx, %r9; addq %rax, %r9; addq %r9, %r13; movq %r10, %rcx; movq %r10, %rdx; movq %r10, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r12, %rcx; addq %rax, %r9; movq %r12, %rax; orq %r11, %rax; andq %r11, %rcx; andq %r10, %rax; orq %rcx, %rax; addq %rax, %r9;
 movq (((79 -15)&0xF)*8)(%rsp), %rax; movq (((79 -16)&0xF)*8)(%rsp), %rbx; addq (((79 - 7)&0xF)*8)(%rsp), %rbx; movq %rax, %rcx; movq %rax, %rdx; rorq $8, %rcx; shrq $7, %rdx; rorq $1, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq (((79 - 2)&0xF)*8)(%rsp), %rax; movq %rax, %rcx; movq %rax, %rdx; rorq $61, %rcx; shrq $6, %rdx; rorq $19, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rax, %rbx; movq %rbx, (((79)&0xF)*8)(%rsp); movq %r13, %rcx; movq %r13, %rdx; movq %r13, %rax; rorq $18, %rcx; rorq $41, %rdx; rorq $14, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; addq %rbx, %r8; movq %r15, %rcx; xorq %r14, %rcx; andq %r13, %rcx; xorq %r15, %rcx; addq %rax, %r8; movabs $0x6C44198C4A475817, %rax; addq %rcx, %r8; addq %rax, %r8; addq %r8, %r12; movq %r9, %rcx; movq %r9, %rdx; movq %r9, %rax; rorq $39, %rcx; rorq $34, %rdx; rorq $28, %rax; xorq %rdx, %rcx; xorq %rcx, %rax; movq %r11, %rcx; addq %rax, %r8; movq %r11, %rax; orq %r10, %rax; andq %r10, %rcx; andq %r9, %rax; orq %rcx, %rax; addq %rax, %r8;


 addq %r8 , 0(%rdi)
 addq %r9 , 8(%rdi)
 addq %r10, 16(%rdi)
 addq %r11, 24(%rdi)
 addq %r12, 32(%rdi)
 addq %r13, 40(%rdi)
 addq %r14, 48(%rdi)
 addq %r15, 56(%rdi)


 movq %xmm0, %r10
 movq %xmm1, %r11
 movq %xmm2, %r12
 movq %xmm3, %r13
 movq %xmm4, %r14
 movq %xmm5, %r15

.ifdef WINABI
 movq (0*8 + 128)(%rsp), %rbx
 movq (1*8 + 128)(%rsp), %rdi
 movq (2*8 + 128)(%rsp), %rsi
 addq $152, %rsp
 .else
 movq %xmm6, %rbx
 addq $128, %rsp
.endif

 retq

 .ifndef __YASM__
#if defined(__linux__) && defined(__ELF__)
.section .note.GNU-stack,"",%progbits
#endif
 .endif
