/**
*** \file settings.h
*** \brief Load/change/save settings classes.
*** \details Classes to load, change, and save settings.
**/

/*
** This file is part of moonphase.
** Copyright (C) 2014-2018 by Alan Wise <awisesoftware@gmail.com>.
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/


#ifndef   SETTINGS_H
/**
*** \brief settings.h identifier.
*** \details Identifier for settings.h.
**/
#define   SETTINGS_H


/****
*****
***** INCLUDES
*****
****/

#include  <QSettings>
#include  <QColor>


/****
*****
***** DEFINES
*****
****/


/****
*****
***** DATA TYPES
*****
****/

/**
*** \brief Load/change/save settings classes.
*** \details Classes to load, change, and save settings.
**/
class SETTINGS_C : protected QSettings
{
  public:
    /**
    *** \brief Application preferences.
    *** \details Preferences related to running the application.
    **/
    class APPLICATIONPREFERENCES_C : protected QSettings
    {
      public:
        /**
        *** \brief Default constructor.
        *** \details Default constructor.
        **/
        APPLICATIONPREFERENCES_C(void);

        /**
        *** \brief Copy constructor.
        *** \details Copy constructor.
        *** \param Preferences Preferences to copy.
        **/
        APPLICATIONPREFERENCES_C(APPLICATIONPREFERENCES_C const &Preferences);

        /**
        *** \brief Destructor.
        *** \details Destructor.
        **/
        ~APPLICATIONPREFERENCES_C(void);

        /**
        *** \brief = operator.
        *** \details Copy operator.
        *** \param Preferences Preferences to copy.
        **/
        APPLICATIONPREFERENCES_C & operator=(
            APPLICATIONPREFERENCES_C const &Preferences);

        /**
        *** \brief == operator.
        *** \details Equality operator.
        *** \param RHS Object on right hand side of operator.
        *** \retval true Objects are equivalent.
        *** \retval false Objects are not equivalent.
        **/
        bool operator==(APPLICATIONPREFERENCES_C const &RHS);

        /**
        *** \brief != operator.
        *** \details Inequality operator.
        *** \param RHS Object on right hand side of operator.
        *** \retval true Objects are not equivalent.
        *** \retval false Objects are equivalent.
        **/
        bool operator!=(APPLICATIONPREFERENCES_C const &RHS);

        /**
        *** \brief Returns the allow multiple instance flag.
        *** \details Returns the allow multiple instance flag.
        *** \retval true Allow multiple instances.
        *** \retval false Allow only one instance.
        **/
        bool GetAllowMultipleInstancesFlag(void) const;

        /**
        *** \brief Returns the animation pathname.
        *** \details Returns the pathname of the animation image.
        *** \returns Pathname of the animation image.
        **/
        QString GetAnimationPathname(void) const;

        /**
        *** \brief Returns the background color.
        *** \details Returns the color of the background.
        *** \returns Color of the background.
        **/
        QColor GetBackgroundColor(void) const;

        /**
        *** \brief Returns the confirm discard flag.
        *** \details Returns the confirm discarding of changes flag.
        *** \retval true Prompt for confirmation.
        *** \retval false Discard changes without confirming.
        **/
        bool GetConfirmDiscardFlag(void) const;

        /**
        *** \brief Returns the confirm quit flag.
        *** \details Returns the confirm quit flag.
        *** \retval true Prompt for confirmation.
        *** \retval false Quit without confirming.
        **/
        bool GetConfirmQuitFlag(void) const;

        /**
        *** \brief Returns the remind once per session flag.
        *** \details Returns the still running reminder once per session flag.
        *** \retval true Remind only once per session.
        *** \retval false Remind every time.
        **/
        bool GetRemindOncePerSessionFlag(void) const;

        /**
        *** \brief Returns the still running reminder flag.
        *** \details Returns the still running reminder flag.
        *** \retval true Remind on every close.
        *** \retval false No reminder on close.
        **/
        bool GetStillRunningReminderFlag(void) const;

        /**
        *** \brief Returns the update interval.
        *** \details Returns the interval between tray icon updates.
        *** \returns Interval (in hours).
        **/
        unsigned int GetUpdateInterval(void) const;

        /**
        *** \brief Returns the use opaque background flag.
        *** \details Returns the use opaque background flag.
        *** \retval true Draw background using a color.
        *** \retval false Draw transparent background.
        **/
        bool GetUseOpaqueBackgroundFlag(void) const;

        /**
        *** \brief Load settings.
        *** \details Reads the settings from the configuration file.
        **/
        void Load(void);

        /**
        *** \brief Save settings.
        *** \details Writes the settings to the configuration file.
        **/
        void Save(void);

        /**
        *** \brief Sets the allow multiple instances flag.
        *** \details Sets the allow multiple instances flag.
        *** \param AllowMultipleInstancesFlag true - Allow multiple instances,\n
        ***   false - Allow only one instance.
        **/
        void SetAllowMultipleInstancesFlag(bool AllowMultipleInstancesFlag);

        /**
        *** \brief Sets the animation pathname.
        *** \details Sets the pathname of the animation image.
        *** \param Pathname Pathname of the animation image.
        **/
        void SetAnimationPathname(QString Pathname);

        /**
        *** \brief Sets the background color.
        *** \details Sets the color of the background.
        *** \param Color Color of the background.
        **/
        void SetBackgroundColor(QColor Color);

        /**
        *** \brief Sets the confirm discard flag.
        *** \details Sets the confirm discarding of changes flag.
        *** \param ConfirmDiscardFlag true - Prompt for confirmation,\n
        ***   false - Discard changes without confirming.
        **/
        void SetConfirmDiscardFlag(bool ConfirmDiscardFlag);

        /**
        *** \brief Sets the confirm quit flag.
        *** \details Sets the confirm quit flag.
        *** \param ConfirmQuitFlag true - Prompt for confirmation,\n
        ***   false - Quit without confirming.
        **/
        void SetConfirmQuitFlag(bool ConfirmQuitFlag);

        /**
        *** \brief Sets the remind once per session flag.
        *** \details Sets the still running reminder once per session flag.
        *** \param OnceFlag true - Remind only once per session,\n
        ***   false - Remind every time.
        **/
        void SetRemindOncePerSessionFlag(bool OnceFlag);

        /**
        *** \brief Sets the still running reminder flag.
        *** \details Sets the still running reminder flag.
        *** \param ReminderFlag true - Remind on every close,\n
        ***   false - No reminder on close.
        **/
        void SetStillRunningReminderFlag(bool ReminderFlag);

        /**
        *** \brief Sets the update interval.
        *** \details Sets the interval between tray icon updates.
        *** \param Interval Interval (in hours).
        **/
        void SetUpdateInterval(unsigned int Interval);

        /**
        *** \brief Sets the use opaque background flag.
        *** \details Sets the use opaque background flag.
        *** \param UseFlag true - Draw background using a color,\n
        ***   false - Draw transparent background.
        **/
        void SetUseOpaqueBackgroundFlag(bool UseFlag);

      private:
        /**
        *** \brief Sets the allow multiple instances flag.
        *** \details Sets the allow multiple instances flag.
        **/
        bool m_AllowMultipleInstancesFlag;

        /**
        *** \brief Animation pathname.
        *** \details Pathname of the animation image.
        **/
        QString m_AnimationPathname;

        /**
        *** \brief Background color.
        *** \details Color of the background.
        **/
        QColor m_BackgroundColor;

        /**
        *** \brief Confirm discard flag.
        *** \details Confirm discarding of changes flag.
        **/
        bool m_ConfirmDiscardFlag;

        /**
        *** \brief Confirm quit flag.
        *** \details Confirm quit flag.
        **/
        bool m_ConfirmQuitFlag;

        /**
        *** \brief Remind once per session flag.
        *** \details Still running reminder once per session flag.
        **/
        bool m_RemindOncePerSessionFlag;

        /**
        *** \brief Still running reminder flag.
        *** \details Still running reminder flag.
        **/
        bool m_StillRunningReminderFlag;

        /**
        *** \brief Update interval.
        *** \details Interval between icon updates.
        **/
        unsigned int m_UpdateInterval;

        /**
        *** \brief Use opaque background flag.
        *** \details Use opaque background flag.
        **/
        bool m_UseOpaqueBackgroundFlag;
    };

    /**
    *** \todo Doc
    **/
    class INFORMATIONSETTINGS_C
    {
      public:
        class INFORMATIONPREFERENCES_C : protected QSettings
        {
          public:
            /**
            *** \brief Default constructor.
            *** \details Default constructor.
            **/
            INFORMATIONPREFERENCES_C(void);

            /**
            *** \brief Destructor.
            *** \details Destructor.
            **/
            ~INFORMATIONPREFERENCES_C(void);

            /**
            *** \brief == operator.
            *** \details Equality operator.
            *** \param RHS Object on right hand side of operator.
            *** \retval true Objects are equivalent.
            *** \retval false Objects are not equivalent.
            **/
            bool operator==(INFORMATIONPREFERENCES_C const &RHS) const;

            /**
            *** \brief != operator.
            *** \details Inequality operator.
            *** \param RHS Object on right hand side of operator.
            *** \retval true Objects are equivalent.
            *** \retval false Objects are not equivalent.
            **/
            bool operator!=(INFORMATIONPREFERENCES_C const &RHS) const;

            /**
            *** \brief Returns the default to metric units flag.
            *** \details Returns the default to metric units flag.
            *** \retval true - Default to metric units.
            *** \retval false - Default to imperial units.
            **/
            bool GetDefaultToMetricUnitsFlag(void) const;

            /**
            *** \brief Return latitude.
            *** \details Returns the latitude.
            *** \returns Latitude.
            **/
            double GetLatitude(void) const;

            /**
            *** \brief Return longitude.
            *** \details Returns the longitude.
            *** \returns Longitude.
            **/
            double GetLongitude(void) const;

            /**
            *** \brief Load settings.
            *** \details Reads the settings from the configuration file.
            **/
            void Load(void);

            /**
            *** \brief Save settings.
            *** \details Writes the settings to the configuration file.
            **/
            void Save(void);

            /**
            *** \brief Sets the default to metric units flag.
            *** \details Sets the default to metric units flag.
            *** \param DefaultToMetricUnitsFlag true - Default to metric units,\n
            ***   false - Default to imperial units.
            **/
            void SetDefaultToMetricUnitsFlag(bool DefaultToMetricUnitsFlag);

            /**
            *** \brief Set latitude.
            *** \details Set the latitude.
            *** \returns Latitude.
            **/
            void SetLatitude(double Latitude);

            /**
            *** \brief Set longitude.
            *** \details Set the longitude.
            *** \returns Longitude.
            **/
            void SetLongitude(double Longitude);

          private:
            /**
            *** \brief Default to metric units flag.
            *** \details Default to metric units flag.
            **/
            bool m_DefaultToMetricUnitsFlag;

            /**
            *** \brief Latitude.
            *** \details Laitude.
            **/
            double m_Latitude;

            /**
            *** \brief Longitude.
            *** \details Longitude.
            **/
            double m_Longitude;
        };

        /**
        *** \brief Default constructor.
        *** \details Default constructor.
        **/
        INFORMATIONSETTINGS_C(void);

        /**
        *** \brief Destructor.
        *** \details Destructor.
        **/
        ~INFORMATIONSETTINGS_C(void);

        /**
        *** \brief Get information preferences pointer.
        *** \details Gets a pointer to the information preferences.
        *** \returns Pointer to the information preferences.
        **/
        INFORMATIONPREFERENCES_C * GetInformationPreferencesPointer(void);

        /**
        *** \brief Load settings.
        *** \details Reads the settings from the configuration file.
        **/
        void Load(void);

      private:
        /**
        *** \brief Information preferences.
        *** \details Information preferences.
        **/
        INFORMATIONPREFERENCES_C m_InformationPreferences;
    };

  public:
    /**
    *** \brief Constructor.
    *** \details Constructor.
    **/
    SETTINGS_C(void);

    /**
    *** \brief Copy constructor.
    *** \details Copy constructor.
    *** \param Settings Settings to copy.
    **/
    SETTINGS_C(SETTINGS_C const &Settings);

    /**
    *** \brief Destructor.
    *** \details Destructor.
    **/
    ~SETTINGS_C(void);

    /**
    *** \brief = operator.
    *** \details Copy operator.
    *** \param Settings Settings to copy.
    **/
    SETTINGS_C operator=(SETTINGS_C const &Settings);

    /**
    *** \brief == operator.
    *** \details Equality operator.
    *** \param RHS Object on right hand side of operator.
    *** \retval true Objects are equivalent.
    *** \retval false Objects are not equivalent.
    **/
    bool operator==(SETTINGS_C const &RHS) const;

    /**
    *** \brief != operator.
    *** \details Inequality operator.
    *** \param RHS Object on right hand side of operator.
    *** \retval true Objects are not equivalent.
    *** \retval false Objects are equivalent.
    **/
    bool operator!=(SETTINGS_C const &RHS) const;

    /**
    *** \brief Load settings.
    *** \details Reads the settings from the configuration file.
    **/
    void Load(void);

    /**
    *** \brief Get application preferences pointer.
    *** \details Gets a pointer to the application preferences.
    *** \returns Pointer to the application preferences.
    **/
    APPLICATIONPREFERENCES_C * GetApplicationPreferencesPointer(void);

    /**
    *** \brief Get information settings pointer.
    *** \details Gets a pointer to the information settings.
    *** \returns Pointer to the information settings.
    **/
    INFORMATIONSETTINGS_C * GetInformationSettingsPointer(void);

  private:
    /**
    *** \brief Application preferences.
    *** \details Application preferences.
    **/
    APPLICATIONPREFERENCES_C m_ApplicationPreferences;

    /**
    *** \brief Information settings.
    *** \details Information settings.
    **/
    INFORMATIONSETTINGS_C m_InformationSettings;
};


/****
*****
***** DATA
*****
****/


/****
*****
***** VARIABLES
*****
****/


/****
*****
***** PROTOTYPES
*****
****/


#endif    /* SETTINGS_H */


/**
*** settings.h
**/
