
/*

  KLayout Layout Viewer
  Copyright (C) 2006-2016 Matthias Koefferlein

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*/


#include "extDEFImporter.h"
#include "dbPolygonTools.h"

#include <cmath>

namespace ext
{

DEFImporter::DEFImporter ()
  : LEFDEFImporter ()
{
  //  .. nothing yet ..
}

void 
DEFImporter::read_lef (tl::InputStream &stream, db::Layout &layout, LEFDEFLayerDelegate &ld, const std::string &fn)
{
  m_lef_importer.read (stream, layout, ld, fn);
}


db::FTrans 
DEFImporter::orient_string_to_ftrans (const std::string &s) 
{
  if (s == "N") {
    return db::FTrans::r0;
  } else if (s == "S") {
    return db::FTrans::r180;
  } else if (s == "W") {
    return db::FTrans::r90;
  } else if (s == "E") {
    return db::FTrans::r270;
  } else if (s == "FN") {
    return db::FTrans::m90;
  } else if (s == "FS") {
    return db::FTrans::m0;
  } else if (s == "FW") {
    return db::FTrans::m45;
  } else if (s == "FE") {
    return db::FTrans::m135;
  } else {
    error (tl::to_string (QObject::tr ("Invalid orientation specification: ")) + s);
    return db::FTrans::r0;
  }
}

/**
 *  @brief A structure describing a via
 */
struct ViaDesc
{
  ViaDesc () : cell (0) { }

  db::Cell *cell;
  std::string m1, m2;
};

void
DEFImporter::read_polygon (db::Polygon &poly, double scale)
{
  std::vector<db::Point> points;

  double x = 0.0, y = 0.0;

  while (! peek ("+") && ! peek (";") && ! peek ("-")) {

    test ("(");
    if (! test ("*")) {
      x = get_double ();
    }
    if (! test ("*")) {
      y = get_double ();
    }
    points.push_back (db::Point::from_double (db::DPoint (x * scale, y * scale)));
    test (")");

  }

  poly.assign_hull (points.begin (), points.end ());
}

void
DEFImporter::read_rect (db::Polygon &poly, double scale)
{
  double x = 0.0, y = 0.0;

  test ("(");
  x = get_double ();
  y = get_double ();
  db::Point pt1 = db::Point::from_double (db::DPoint (x * scale, y * scale));
  test (")");

  test ("(");
  x = get_double ();
  y = get_double ();
  db::Point pt2 = db::Point::from_double (db::DPoint (x * scale, y * scale));
  test (")");

  poly = db::Polygon (db::Box (pt1, pt2));
}

void 
DEFImporter::do_read (db::Layout &layout)
{
  double dbu_mic = 1000.0;
  double scale = 1.0 / (dbu_mic * layout.dbu ());
  std::map<int, db::Polygon> styles;
  std::map<std::string, ViaDesc> via_desc;

  db::Cell &design = layout.cell (layout.add_cell ("TOP"));

  while (! at_end ()) {

    bool specialnets = false;

    if (test ("END")) {

      //  END DESIGN terminates the file
      expect ("DESIGN");
      break;

    } else if (test ("DESIGN")) {

      std::string cn = get ();
      layout.rename_cell (design.cell_index (), layout.uniquify_cell_name (cn.c_str ()).c_str ());

      expect (";");

    } else if (test ("VERSION")) {

      //  ignore VERSION statement currently 
      take ();
      expect (";");

    } else if (test ("UNITS")) {

      test ("DISTANCE");
      test ("MICRONS");

      double units = get_double ();
      if (fabs (units) > 1e-6) {
        scale = 1.0 / (units * layout.dbu ());
      }
      expect (";");

    } else if (test ("DIEAREA")) {

      std::vector<db::DPoint> points;

      while (! test (";")) {
        test ("(");
        double x = get_double ();
        double y = get_double ();
        points.push_back (db::Point::from_double (db::DPoint (x * scale, y * scale)));
        test (")");
      }

      if (points.size () >= 2) {

        //  create outline shape
        std::pair <bool, unsigned int> dl = open_layer (layout, std::string (), Outline);
        if (dl.first) {
          if (points.size () == 2) {
            design.shapes (dl.second).insert (db::Box (points [0], points [1]));
          } else {
            db::Polygon p;
            p.assign_hull (points.begin (), points.end ());
            design.shapes (dl.second).insert (p);
          }
        }

      }

    } else if (test ("PROPERTYDEFINITIONS")) {
      //  read over PROPERTYDEFINITIONS sections
      while (! test ("END") || ! test ("PROPERTYDEFINITIONS")) {
        take ();
      }

    } else if (test ("NONDEFAULTRULES")) {

      //  read NONDEFAULTRULES sections
      get_long ();
      expect (";");

      while (test ("-")) {

        std::string n = get ();

        while (test ("+")) {

          if (test ("LAYER")) {

            std::string l = get ();

            //  read the width for the layer
            if (test ("WIDTH")) {
              double w = get_double () * scale;
              m_nondefault_widths[n][l] = w;
            } 

          } 

          //  parse over the rest
          while (! peek ("+") && ! peek ("-") && ! peek (";")) {
            take ();
          }

        }

        test (";");

      }

      test ("END");
      test ("NONDEFAULTRULES");

    } else if (test ("REGIONS")) {
      //  read over REGIONS statements 
      while (! test ("END") || ! test ("REGIONS")) {
        take ();
      }
    } else if (test ("PINPROPERTIES")) {
      //  read over PINPROPERTIES statements 
      while (! test ("END") || ! test ("PINPROPERTIES")) {
        take ();
      }
    } else if (test ("SLOTS")) {
      //  read over SLOTS statements 
      while (! test ("END") || ! test ("SLOTS")) {
        take ();
      }
    } else if (test ("FILLS")) {
      //  read over FILLS statements 
      while (! test ("END") || ! test ("FILLS")) {
        take ();
      }
    } else if (test ("SCANCHAINS")) {
      //  read over SCANCHAINS statements 
      while (! test ("END") || ! test ("SCANCHAINS")) {
        take ();
      }
    } else if (test ("GROUPS")) {
      //  read over GROUPS statements 
      while (! test ("END") || ! test ("GROUPS")) {
        take ();
      }
    } else if (test ("BEGINEXT")) {
      //  read over BEGINEXT sections
      while (! test ("ENDEXT")) {
        take ();
      }

    } else if (test ("BLOCKAGES")) {

      get_long ();
      expect (";");

      while (test ("-")) {

        std::string layer;

        while (! test (";")) {

          if (test ("PLACEMENT")) {

            //  indicates a placement blockage
            layer = std::string ();

          } else if (test ("LAYER")) {

            layer = get ();

          } else if (test ("+")) {

            //  ignore options for now
            while (! peek ("RECT") && ! peek ("POLYGON") && ! peek ("+") && ! peek ("-") && ! peek (";")) {
              take ();
            }

          } else if (test ("POLYGON")) {

            db::Polygon p;
            read_polygon (p, scale);

            std::pair <bool, unsigned int> dl = open_layer (layout, layer, layer.empty () ? PlacementBlockage : Blockage);
            if (dl.first) {
              design.shapes (dl.second).insert (p);
            }

          } else if (test ("RECT")) {

            db::Polygon p;
            read_rect (p, scale);

            std::pair <bool, unsigned int> dl = open_layer (layout, layer, layer.empty () ? PlacementBlockage : Blockage);
            if (dl.first) {
              design.shapes (dl.second).insert (p);
            }

          } else {
            expect (";");
          }

        }

      }

      test ("END");
      test ("BLOCKAGES");

    } else if ((specialnets = test ("SPECIALNETS")) == true || test ("NETS")) {

      get_long ();
      expect (";");

      while (test ("-")) {

        std::string net = get ();
        std::string nondefaultrule;
        std::string stored_netname, stored_nondefaultrule;
        std::string taperrule;
        bool in_subnet = false;

        db::properties_id_type prop_id = 0;
        if (produce_net_props ()) {
          db::PropertiesRepository::properties_set props;
          props.insert (std::make_pair (net_prop_name_id (), tl::Variant (net)));
          prop_id = layout.properties_repository ().properties_id (props);
        }

        while (test ("(")) {
          while (! test (")")) {
            take ();
          }
        }

        while (test ("+")) {

          bool was_shield = false;

          if (! specialnets && test ("SUBNET")) {

            while (test ("(")) {
              while (! test (")")) {
                take ();
              }
            }

            if (! in_subnet) {
              stored_netname = net;
              stored_nondefaultrule = nondefaultrule;
              in_subnet = true;
            }

          } else if (! specialnets && test ("NONDEFAULTRULE")) {

            nondefaultrule = get ();

          } else if ((was_shield = test ("SHIELD")) == true || test ("NOSHIELD") || test ("ROUTED") || test ("FIXED") || test ("COVER")) {

            if (was_shield) {
              take ();
            }

            taperrule.clear ();

            do {

              std::string ln = get ();

              db::Coord w = 0;
              if (specialnets) {
                w = db::coord_traits<db::Coord>::rounded (get_double () * scale);
              } 

              const db::Polygon *style = 0;

              int sn = std::numeric_limits<int>::max ();

              if (specialnets) {

                while (test ("+")) {

                  if (test ("STYLE")) {
                    sn = get_long ();
                  } else if (test ("SHAPE")) {
                    take ();
                  }

                }

              } else {

                while (true) {
                  if (test ("TAPER")) {
                    taperrule.clear ();
                  } else if (test ("TAPERRULE")) {
                    taperrule = get ();
                  } else if (test ("STYLE")) {
                    sn = get_long ();
                  } else {
                    break;
                  }
                }

              }

              if (! specialnets) {

                const std::string *rulename = &taperrule;
                if (rulename->empty ()) {
                  rulename = &nondefaultrule;
                }

                w = db::coord_traits<db::Coord>::rounded (m_lef_importer.layer_width (ln, *rulename, 0.0) / layout.dbu ());

                //  try to find local nondefault rule
                if (! rulename->empty ()) {
                  std::map<std::string, std::map<std::string, double> >::const_iterator nd = m_nondefault_widths.find (*rulename);
                  if (nd != m_nondefault_widths.end ()) {
                    std::map<std::string, double>::const_iterator ld = nd->second.find (ln);
                    if (ld != nd->second.end ()) {
                      w = ld->second;
                    }
                  }
                }

              }

              db::Coord def_ext = 0;
              if (! specialnets) {
                def_ext = db::coord_traits<db::Coord>::rounded (m_lef_importer.layer_ext (ln, w * 0.5 * layout.dbu ()) / layout.dbu ());
              }

              std::map<int, db::Polygon>::const_iterator s = styles.find (sn);
              if (s != styles.end ()) {
                style = &s->second;
              }

              std::vector<db::Coord> ext;
              std::vector<db::Point> pts;

              double x = 0.0, y = 0.0;

              while (true) {

                if (test ("MASK")) {
                  //  ignore mask spec
                  get_long ();
                }

                if (test ("RECT")) {

                  if (! test ("(")) {
                    error (tl::to_string (QObject::tr ("RECT routing specification not followed by coordinate list")));
                  }

                  //  rect spec

                  double x1 = get_double ();
                  double y1 = get_double ();
                  double x2 = get_double ();
                  double y2 = get_double ();

                  test (")");

                  std::pair <bool, unsigned int> dl = open_layer (layout, ln, Routing);
                  if (dl.first) {

                    db::Box rect (db::Point::from_double (db::DPoint ((x + x1) * scale, (y + y1) * scale)),
                                  db::Point::from_double (db::DPoint ((x + x2) * scale, (y + y2) * scale)));

                    if (prop_id != 0) {
                      design.shapes (dl.second).insert (db::object_with_properties<db::Box> (rect, prop_id));
                    } else {
                      design.shapes (dl.second).insert (rect);
                    }

                  }

                } else if (test ("VIRTUAL")) {

                  //  virtual specs simply create a new segment

                } else if (peek ("(")) {

                  ext.clear ();
                  pts.clear ();

                  while (peek ("(") || peek ("MASK")) {

                    if (test ("MASK")) {
                      //  ignore MASK spec
                      get_long ();
                    } 

                    expect ("(");

                    if (! test ("*")) {
                      x = get_double ();
                    }
                    if (! test ("*")) {
                      y = get_double ();
                    }
                    pts.push_back (db::Point::from_double (db::DPoint (x * scale, y * scale)));
                    db::Coord e = def_ext;
                    if (! peek (")")) {
                      e = db::coord_traits<db::Coord>::rounded (get_double () * scale);
                    }
                    ext.push_back (e);

                    test (")");

                  }

                  if (pts.size () > 1) {

                    std::pair <bool, unsigned int> dl = open_layer (layout, ln, Routing);
                    if (dl.first) {

                      if (! style) {

                        //  Use the default style (octagon "pen" for non-manhattan segments, paths for 
                        //  horizontal/vertical segments).

                        db::Coord e = std::max (ext.front (), ext.back ());

                        std::vector<db::Point>::const_iterator pt = pts.begin ();
                        while (pt != pts.end ()) {

                          std::vector<db::Point>::const_iterator pt0 = pt;
                          do {
                            ++pt;
                          } while (pt != pts.end () && (pt[-1].x () == pt[0].x () || pt[-1].y () == pt[0].y()));

                          if (pt - pt0 > 1) {

                            db::Path p (pt0, pt, w, pt0 == pts.begin () ? e : 0, pt == pts.end () ? e : 0, false);
                            if (prop_id != 0) {
                              design.shapes (dl.second).insert (db::object_with_properties<db::Path> (p, prop_id));
                            } else {
                              design.shapes (dl.second).insert (p);
                            }

                            if (pt == pts.end ()) {
                              break;
                            }

                            --pt;

                          } else if (pt != pts.end ()) {

                            db::Coord s = (w + 1) / 2;
                            db::Coord t = db::Coord (ceil (w * (M_SQRT2 - 1) / 2));

                            db::Point octagon[8] = {
                              db::Point (-s, t),
                              db::Point (-t, s),
                              db::Point (t, s),
                              db::Point (s, t),
                              db::Point (s, -t),
                              db::Point (t, -s),
                              db::Point (-t, -s),
                              db::Point (-s, -t)
                            };

                            db::Polygon k;
                            k.assign_hull (octagon, octagon + sizeof (octagon) / sizeof (octagon[0]));

                            db::Polygon p = db::minkowsky_sum (k, db::Edge (*pt0, *pt));
                            if (prop_id != 0) {
                              design.shapes (dl.second).insert (db::object_with_properties<db::Polygon> (p, prop_id));
                            } else {
                              design.shapes (dl.second).insert (p);
                            }

                          }

                        }

                      } else {

                        for (size_t i = 0; i < pts.size () - 1; ++i) {
                          db::Polygon p = db::minkowsky_sum (*style, db::Edge (pts [i], pts [i + 1]));
                          if (prop_id != 0) {
                            design.shapes (dl.second).insert (db::object_with_properties<db::Polygon> (p, prop_id));
                          } else {
                            design.shapes (dl.second).insert (p);
                          }
                        }

                      }

                    }
                    
                  }

                } else if (! peek ("NEW") && ! peek ("+") && ! peek ("-") && ! peek (";")) {

                  //  indicates a via
                  std::string vn = get ();
                  db::FTrans ft;
                  if (! peek ("NEW") && ! peek ("+") && ! peek ("-") && ! peek (";")) {
                    ft = orient_string_to_ftrans (get ());
                  }

                  std::map<std::string, ViaDesc>::const_iterator vd = via_desc.find (vn);
                  if (vd != via_desc.end () && ! pts.empty ()) {
                    design.insert (db::CellInstArray (db::CellInst (vd->second.cell->cell_index ()), db::Trans (ft.rot (), pts.back ())));
                    if (ln == vd->second.m1) {
                      ln = vd->second.m2;
                    } else if (ln == vd->second.m2) {
                      ln = vd->second.m1;
                    }
                  }

                } else {
                  break;
                }

              }

            } while (test ("NEW"));

            if (in_subnet) {
              in_subnet = false;
              net = stored_netname;
              stored_netname.clear ();
              nondefaultrule = stored_nondefaultrule;
              stored_nondefaultrule.clear ();
            }

          } else if (test ("POLYGON")) {

            std::string ln = get ();

            db::Polygon p;
            read_polygon (p, scale);

            std::pair <bool, unsigned int> dl = open_layer (layout, ln, Routing);
            if (dl.first) {
              if (prop_id != 0) {
                design.shapes (dl.second).insert (db::object_with_properties<db::Polygon> (p, prop_id));
              } else {
                design.shapes (dl.second).insert (p);
              }
            }

          } else if (test ("RECT")) {

            std::string ln = get ();

            db::Polygon p;
            read_rect (p, scale);

            std::pair <bool, unsigned int> dl = open_layer (layout, ln, Routing);
            if (dl.first) {
              if (prop_id != 0) {
                design.shapes (dl.second).insert (db::object_with_properties<db::Polygon> (p, prop_id));
              } else {
                design.shapes (dl.second).insert (p);
              }
            }

          } else {
            while (! peek ("+") && ! peek ("-") && ! peek (";")) {
              take ();
            }
          }

        }

        expect (";");

      }

      test ("END");
      if (specialnets) {
        test ("SPECIALNETS");
      } else {
        test ("NETS");
      }

    } else if (test ("VIAS")) {

      get_long ();
      expect (";");

      while (test ("-")) {

        std::string n = get ();
        ViaDesc &vd = via_desc.insert (std::make_pair (n, ViaDesc ())).first->second;

        //  produce a cell for vias
        std::string cellname = "VIA_" + n;
        db::Cell &cell = layout.cell (layout.add_cell (cellname.c_str ()));
        vd.cell = &cell;

        bool has_via_rule = false;

        db::Point cutsize, cutspacing;
        db::Point be, te;
        db::Point bo, to;
        db::Point offset;
        int rows = 1, columns = 1;
        std::string pattern;

        std::map<std::string, std::vector<db::Polygon> > geometry;
        std::vector<db::Polygon> *top = 0, *cut = 0, *bottom = 0;

        while (test ("+")) {

          double x, y;

          if (test ("VIARULE")) {

            has_via_rule = true;
            take ();

          } else if (test ("CUTSIZE")) {

            x = get_double ();
            y = get_double ();
            cutsize = db::Point::from_double (db::DPoint (x * scale, y * scale));

          } else if (test ("CUTSPACING")) {

            x = get_double ();
            y = get_double ();
            cutspacing = db::Point::from_double (db::DPoint (x * scale, y * scale));

          } else if (test ("ORIGIN")) {

            x = get_double ();
            y = get_double ();
            offset = db::Point::from_double (db::DPoint (x * scale, y * scale));

          } else if (test ("ENCLOSURE")) {

            x = get_double ();
            y = get_double ();
            be = db::Point::from_double (db::DPoint (x * scale, y * scale));

            x = get_double ();
            y = get_double ();
            te = db::Point::from_double (db::DPoint (x * scale, y * scale));

          } else if (test ("OFFSET")) {

            x = get_double ();
            y = get_double ();
            bo = db::Point::from_double (db::DPoint (x * scale, y * scale));

            x = get_double ();
            y = get_double ();
            to = db::Point::from_double (db::DPoint (x * scale, y * scale));

          } else if (test ("ROWCOL")) {

            rows = get_long ();
            columns = get_long ();

          } else if (test ("PATTERN")) {

            pattern = get ();

          } else if (test ("LAYERS")) {

            std::string bn = get ();
            std::string cn = get ();
            std::string tn = get ();

            bottom = &geometry.insert (std::make_pair (bn, std::vector<db::Polygon> ())).first->second;
            cut = &geometry.insert (std::make_pair (cn, std::vector<db::Polygon> ())).first->second;
            top = &geometry.insert (std::make_pair (tn, std::vector<db::Polygon> ())).first->second;

            vd.m1 = bn;
            vd.m2 = tn;

          } else if (test ("POLYGON")) {

            std::string ln = get ();

            std::vector<db::Polygon> &polygons = geometry.insert (std::make_pair (ln, std::vector<db::Polygon> ())).first->second;
            polygons.push_back (db::Polygon ());
            read_polygon (polygons.back (), scale);

          } else if (test ("RECT")) {

            std::string ln = get ();

            std::vector<db::Polygon> &polygons = geometry.insert (std::make_pair (ln, std::vector<db::Polygon> ())).first->second;
            polygons.push_back (db::Polygon ());
            read_rect (polygons.back (), scale);

          }

        }

        test (";");

        if (has_via_rule && top && cut && bottom) {
            create_generated_via (*bottom, *cut, *top,
                                  cutsize, cutspacing, be, te, bo, to, offset, rows, columns, pattern);
        }

        for (std::map<std::string, std::vector<db::Polygon> >::const_iterator g = geometry.begin (); g != geometry.end (); ++g) {
          std::pair <bool, unsigned int> dl = open_layer (layout, g->first, ViaGeometry);
          if (dl.first) {
            for (std::vector<db::Polygon>::const_iterator p = g->second.begin (); p != g->second.end (); ++p) {
              cell.shapes (dl.second).insert (*p);
            }
          }
        }

      }

      expect ("END");
      expect ("VIAS");

    } else if (test ("STYLES")) {

      get_long ();
      expect (";");

      while (test ("-")) {

        test ("STYLE");

        int sn = get_long ();

        std::vector<db::Point> points;

        double x = 0.0, y = 0.0;

        while (! test (";")) {

          test ("(");
          if (! test ("*")) {
            x = get_double ();
          }
          if (! test ("*")) {
            y = get_double ();
          }
          points.push_back (db::Point::from_double (db::DPoint (x * scale, y * scale)));
          test (")");

        }

        styles.insert (std::make_pair (sn, db::Polygon ())).first->second.assign_hull (points.begin (), points.end ());

      }

      test ("END");
      test ("STYLES");

    } else if (test ("COMPONENTS")) {

      get_long ();
      expect (";");

      while (test ("-")) {

        take (); // instance name
        std::string model = get ();

        db::Cell *cell = m_lef_importer.macro_by_name (model);

        while (test ("+")) {

          if (test ("PLACED") || test ("FIXED") || test ("COVER")) {

            test ("(");
            double x = get_double ();
            double y = get_double ();
            db::Point pt = db::Point::from_double (db::DPoint (x * scale, y * scale));
            test (")");

            db::FTrans ft = orient_string_to_ftrans (get ());

            pt += -m_lef_importer.macro_bbox_by_name (model).transformed (ft).lower_left ();

            if (cell) {
                design.insert (db::CellInstArray (db::CellInst (cell->cell_index ()), db::Trans (ft.rot (), pt)));
            } else {
                warn (tl::to_string (QObject::tr ("Macro not found in LEF file: ")) + model);
            }

          } else {
            while (! peek ("+") && ! peek ("-") && ! peek (";")) {
              take ();
            }
          }

        }

        expect (";");

      }

      expect ("END");
      expect ("COMPONENTS");

    } else if (test ("PINS")) {

      get_long ();
      expect (";");

      while (test ("-")) {

        take (); // pin name

        std::string net;
        std::string dir;
        std::map <std::string, std::vector <db::Polygon> > geometry;
        db::Trans trans;

        while (test ("+")) {

          bool flush = false;

          if (test ("DIRECTION")) {
            dir = get ();
          } else if (test ("NET")) {
            net = get ();
          } else if (test ("LAYER")) {

            std::string ln = get ();

            while (test ("DESIGNRULEWIDTH") || test ("SPACING")) {
              take ();
            }

            double x, y;

            test ("(");
            x = get_double ();
            y = get_double ();
            db::Point pt1 = db::Point::from_double (db::DPoint (x * scale, y * scale));
            test (")");

            test ("(");
            x = get_double ();
            y = get_double ();
            db::Point pt2 = db::Point::from_double (db::DPoint (x * scale, y * scale));
            test (")");

            geometry.insert (std::make_pair (ln, std::vector<db::Polygon> ())).first->second.push_back (db::Polygon (db::Box (pt1, pt2)));

          } else if (test ("POLYGON")) {

            std::string ln = get ();

            while (test ("DESIGNRULEWIDTH") || test ("SPACING")) {
              take ();
            }

            std::vector<db::Point> points;

            double x = 0.0, y = 0.0;

            while (! test ("+") && ! test (";")) {

              test ("(");
              if (! test ("*")) {
                x = get_double ();
              }
              if (! test ("*")) {
                y = get_double ();
              }
              points.push_back (db::Point::from_double (db::DPoint (x * scale, y * scale)));
              test (")");

            }

            std::vector<db::Polygon> &polygons = geometry.insert (std::make_pair (ln, std::vector<db::Polygon> ())).first->second;
            polygons.push_back (db::Polygon ());
            polygons.back ().assign_hull (points.begin (), points.end ());

          } else if (test ("PLACED") || test ("FIXED") || test ("COVER")) {

            test ("(");
            double x = get_double ();
            double y = get_double ();
            db::Point pt = db::Point::from_double (db::DPoint (x * scale, y * scale));
            test (")");

            db::FTrans ft = orient_string_to_ftrans (get ());
            trans = db::Trans (ft.rot (), pt);

          } else if (test ("PORT")) {

            flush = true;

          } else {
            while (! peek ("+") && ! peek ("-") && ! peek (";")) {
              take ();
            }
          }

          if (flush || ! peek ("+")) {

            //  TODO: put a label on every single object?
            std::string label = net;
            /* don't add the direction currently, a name is sufficient
            if (! dir.empty ()) {
              label += ":";
              label += dir;
            }
            */

            //  Produce geometry collected so far
            for (std::map<std::string, std::vector<db::Polygon> >::const_iterator g = geometry.begin (); g != geometry.end (); ++g) {

              std::pair <bool, unsigned int> dl = open_layer (layout, g->first, Pins);
              if (dl.first) {
                for (std::vector<db::Polygon>::const_iterator p = g->second.begin (); p != g->second.end (); ++p) {
                  db::Polygon pt = p->transformed (trans);
                  design.shapes (dl.second).insert (pt);
                }
              }

              dl = open_layer (layout, g->first, Label);
              if (dl.first) {
                db::Box bbox;
                if (! g->second.empty ()) {
                  bbox = g->second.back ().box ().transformed (trans);
                }
                design.shapes (dl.second).insert (db::Text (label.c_str (), db::Trans (bbox.center ())));
              }

            }

            geometry.clear ();
            trans = db::Trans ();

          }

        }

        expect (";");

      }

      expect ("END");
      expect ("PINS");

    } else {
      while (! test (";")) {
        take ();
      }
    }

  }
}

}

