#import <Cocoa/Cocoa.h>
#import <XCTest/XCTest.h>
#include "playlist.h"
#include "tf.h"
#include "playqueue.h"
#include "streamer.h"
#include "plugins.h"

static int fake_out_state_value = OUTPUT_STATE_STOPPED;

static int fake_out_state (void) {
    return fake_out_state_value;
}

static DB_output_t fake_out = {
    .state = fake_out_state,
};

@interface TitleFormatting : XCTestCase {
    playItem_t *it;
    ddb_tf_context_t ctx;
    char buffer[1000];
}
@end

@implementation TitleFormatting

- (void)setUp {
    [super setUp];

    pl_init ();

    it = pl_item_alloc_init ("testfile.flac", "stdflac");

    memset (&ctx, 0, sizeof (ctx));
    ctx._size = sizeof (ddb_tf_context_t);
    ctx.it = (DB_playItem_t *)it;
    ctx.plt = NULL;

    streamer_set_playing_track (NULL);

    fake_out_state_value = OUTPUT_STATE_STOPPED;
}

- (void)tearDown {
    streamer_set_playing_track (NULL);
    pl_item_unref (it);
    pl_free ();

    [super tearDown];
}

- (void)test_Literal_ReturnsLiteral {
    char *bc = tf_compile("hello world");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("hello world", buffer), @"The actual output is: %s", buffer);
}

- (void)test_AlbumArtistDash4DigitYearSpaceAlbum_ReturnsCorrectResult {
    pl_add_meta (it, "album artist", "TheAlbumArtist");
    pl_add_meta (it, "year", "12345678");
    pl_add_meta (it, "album", "TheNameOfAlbum");

    char *bc = tf_compile("%album artist% - ($left($meta(year),4)) %album%");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("TheAlbumArtist - (1234) TheNameOfAlbum", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Unicode_AlbumArtistDash4DigitYearSpaceAlbum_ReturnsCorrectResult {
    pl_add_meta (it, "album artist", "ИсполнительДанногоАльбома");
    pl_add_meta (it, "year", "12345678");
    pl_add_meta (it, "album", "Альбом");

    char *bc = tf_compile("%album artist% - ($left($meta(year),4)) %album%");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("ИсполнительДанногоАльбома - (1234) Альбом", buffer), @"The actual output is: %s", buffer);
}

- (void)test_TotalDiscsGreaterThan1_ReturnsExpectedResult {
    pl_add_meta (it, "numdiscs", "20");
    pl_add_meta (it, "disc", "18");

    char *bc = tf_compile("$if($greater(%totaldiscs%,1),- Disc: %discnumber%/%totaldiscs%)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("- Disc: 18/20", buffer), @"The actual output is: %s", buffer);
}

- (void)test_AlbumArtistSameAsArtist_ReturnsBlankTrackArtist {
    pl_add_meta (it, "artist", "Artist Name");
    pl_add_meta (it, "album artist", "Artist Name");

    char *bc = tf_compile("%track artist%");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!buffer[0], @"The actual output is: %s", buffer);
}

- (void)test_TrackArtistIsUndef_ReturnsBlankTrackArtist {
    pl_add_meta (it, "album artist", "Artist Name");

    char *bc = tf_compile("%track artist%");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!buffer[0], @"The actual output is: %s", buffer);
}

- (void)test_TrackArtistIsDefined_ReturnsTheTrackArtist {
    pl_add_meta (it, "artist", "Track Artist Name");
    pl_add_meta (it, "album artist", "Album Artist Name");

    char *bc = tf_compile("%track artist%");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("Track Artist Name", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Add10And2_Gives12 {
    char *bc = tf_compile("$add(10,2)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("12", buffer), @"The actual output is: %s", buffer);
}

- (void)test_StrcmpChannelsMono_GivesMo {
    char *bc = tf_compile("$if($strcmp(%channels%,mono),mo,st)");
    pl_replace_meta (it, ":CHANNELS", "1");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("mo", buffer), @"The actual output is: %s", buffer);
}

- (void)test_StrcmpChannelsMono_GivesSt {
    char *bc = tf_compile("$if($strcmp(%channels%,mono),mo,st)");
    pl_replace_meta (it, ":CHANNELS", "2");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("st", buffer), @"The actual output is: %s", buffer);
}

- (void)test_SimpleExpr_Performance {
    char *bc = tf_compile("simple expr");

    [self measureBlock:^{
        tf_eval (&ctx, bc, buffer, sizeof (buffer));
    }];
    
    tf_free (bc);
}

- (void)test_LongCommentOverflowBuffer_DoesntCrash {
    char longcomment[2048];
    for (int i = 0; i < sizeof (longcomment) - 1; i++) {
        longcomment[i] = (i % 33) + 'a';
    }
    longcomment[sizeof (longcomment)-1] = 0;

    pl_add_meta (it, "comment", longcomment);

    char *bc = tf_compile("$meta(comment)");
    XCTAssertNoThrow(tf_eval (&ctx, bc, buffer, 200), @"Crashed!");
    tf_free (bc);
    XCTAssert(!memcmp (buffer, "abcdef", 6), @"The actual output is: %s", buffer);
}

- (void)test_ParticularLongExpressionDoesntAllocateZeroBytes {
    pl_replace_meta (it, "artist", "Frank Schätzing");
    pl_replace_meta (it, "year", "1999");
    pl_replace_meta (it, "album", "Tod und Teufel");
    pl_replace_meta (it, "disc", "1");
    pl_replace_meta (it, "disctotal", "4");
    pl_replace_meta (it, ":FILETYPE", "FLAC");
    char *bc = tf_compile("$if($strcmp(%genre%,Classical),%composer%,$if([%band%],%band%,%album artist%)) | ($left(%year%,4)) %album% $if($greater(%totaldiscs%,1),- Disc: %discnumber%/%totaldiscs%) - \\[%codec%\\]");
    XCTAssertNoThrow(tf_eval (&ctx, bc, buffer, 1000), @"Crashed!");
    tf_free (bc);
}

- (void)test_If2FirstArgIsTrue_EvalToFirstArg {
    pl_replace_meta (it, "title", "a title");
    char *bc = tf_compile("$if2(%title%,def)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("a title", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If2FirstArgIsMixedStringTrue_EvalToFirstArg {
    pl_replace_meta (it, "title", "a title");
    pl_replace_meta (it, "artist", "an artist");
    char *bc = tf_compile("$if2(%title%%artist%,def)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("a titlean artist", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If2FirstArgIsMissingField_EvalToLastArg {
    pl_replace_meta (it, "title", "a title");
    pl_replace_meta (it, "artist", "an artist");
    char *bc = tf_compile("$if2(%garbage%,def)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("def", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If2FirstArgIsMixedWithGarbageTrue_EvalToFirstArg {
    pl_replace_meta (it, "title", "a title");
    pl_replace_meta (it, "artist", "an artist");
    char *bc = tf_compile("$if2(%garbage%xxx%title%,def)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("xxxa title", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If2FirstArgIsMixedWithGarbageTailTrue_EvalToFirstArg {
    pl_replace_meta (it, "title", "a title");
    pl_replace_meta (it, "artist", "an artist");
    char *bc = tf_compile("$if2(%title%%garbage%xxx,def)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("a titlexxx", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If2FirstArgIsFalse_EvalToSecondArg {
    char *bc = tf_compile("$if2(,ghi)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("ghi", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If3FirstArgIsTrue_EvalToFirstArg {
    pl_replace_meta (it, "title", "a title");
    char *bc = tf_compile("$if3(%title%,def)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("a title", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If3AllButLastAreFalse_EvalToLastArg {
    char *bc = tf_compile("$if3(,,,,,lastarg)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("lastarg", buffer), @"The actual output is: %s", buffer);
}

- (void)test_If3OneOfTheArgsBeforeLastIsTrue_EvalToFirstTrueArg {
    char *bc = tf_compile("$if3(,,firstarg,,secondarg,lastarg)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("lastarg", buffer), @"The actual output is: %s", buffer);
}

- (void)test_IfEqualTrue_EvalsToThen {
    char *bc = tf_compile("$ifequal(100,100,then,else)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("then", buffer), @"The actual output is: %s", buffer);
}

- (void)test_IfEqualFalse_EvalsToElse {
    char *bc = tf_compile("$ifequal(100,200,then,else)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("else", buffer), @"The actual output is: %s", buffer);
}

- (void)test_IfGreaterTrue_EvalsToThen {
    char *bc = tf_compile("$ifgreater(200,100,then,else)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("then", buffer), @"The actual output is: %s", buffer);
}

- (void)test_IfGreaterFalse_EvalsToElse {
    char *bc = tf_compile("$ifgreater(100,200,then,else)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("else", buffer), @"The actual output is: %s", buffer);
}

- (void)test_GreaterIsTrue_EvalsToTrue {
    char *bc = tf_compile("$if($greater(2,1),istrue,isfalse)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("istrue", buffer), @"The actual output is: %s", buffer);
}

- (void)test_GreaterIsFalse_EvalsToFalse {
    char *bc = tf_compile("$if($greater(1,2),istrue,isfalse)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("isfalse", buffer), @"The actual output is: %s", buffer);
}

- (void)test_GreaterIsFalseEmptyArguments_EvalsToFalse {
    char *bc = tf_compile("$if($greater(,),istrue,isfalse)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("isfalse", buffer), @"The actual output is: %s", buffer);
}

- (void)test_StrCmpEmptyArguments_EvalsToTrue {
    char *bc = tf_compile("$if($strcmp(,),istrue,isfalse)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("istrue", buffer), @"The actual output is: %s", buffer);
}

- (void)test_StrCmpSameArguments_EvalsToTrue {
    char *bc = tf_compile("$if($strcmp(abc,abc),istrue,isfalse)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("istrue", buffer), @"The actual output is: %s", buffer);
}

- (void)test_IfLongerTrue_EvalsToTrue {
    char *bc = tf_compile("$iflonger(abcd,ef,istrue,isfalse)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("istrue", buffer), @"The actual output is: %s", buffer);
}

- (void)test_IfLongerFalse_EvalsToFalse {
    char *bc = tf_compile("$iflonger(ab,cdef,istrue,isfalse)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("isfalse", buffer), @"The actual output is: %s", buffer);
}

- (void)test_SelectMiddle_EvalsToSelectedValue {
    char *bc = tf_compile("$select(3,10,20,30,40,50)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("30", buffer), @"The actual output is: %s", buffer);
}

- (void)test_SelectLeftmost_EvalsToSelectedValue {
    char *bc = tf_compile("$select(1,10,20,30,40,50)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("10", buffer), @"The actual output is: %s", buffer);
}

- (void)test_SelectRightmost_EvalsToSelectedValue {
    char *bc = tf_compile("$select(5,10,20,30,40,50)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!strcmp ("50", buffer), @"The actual output is: %s", buffer);
}

- (void)test_SelectOutOfBoundsLeft_EvalsToFalse {
    char *bc = tf_compile("$select(0,10,20,30,40,50)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!buffer[0], @"The actual output is: %s", buffer);
}

- (void)test_SelectOutOfBoundsRight_EvalsToFalse {
    char *bc = tf_compile("$select(6,10,20,30,40,50)");
    tf_eval (&ctx, bc, buffer, sizeof (buffer));
    tf_free (bc);
    XCTAssert(!buffer[0], @"The actual output is: %s", buffer);
}

- (void)test_InvalidPercentExpression_WithNullTrack_NoCrash {
    char *bc = tf_compile("begin - %version% - end");

    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("begin -  - end", buffer), @"The actual output is: %s", buffer);
}

- (void)test_InvalidPercentExpression_WithNullTrackAccessingMetadata_NoCrash {
    char *bc = tf_compile("begin - %title% - end");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("begin -  - end", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Index_WithNullPlaylist_NoCrash {
    char *bc = tf_compile("begin - %list_index% - end");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("begin - 0 - end", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Div5by2_Gives3 {
    char *bc = tf_compile("$div(5,2)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("3", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Div4pt9by1pt9_Gives4 {
    char *bc = tf_compile("$div(4.9,1.9)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("4", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Div20by2by5_Gives2 {
    char *bc = tf_compile("$div(20,2,5)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("2", buffer), @"The actual output is: %s", buffer);
}

- (void)test_DivBy0_GivesEmpty {
    char *bc = tf_compile("$div(5,0)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!buffer[0], @"The actual output is: %s", buffer);
}

- (void)test_Max0Arguments_GivesEmpty {
    char *bc = tf_compile("$max()");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!buffer[0], @"The actual output is: %s", buffer);
}

- (void)test_MaxOf1and2_Gives2 {
    char *bc = tf_compile("$max(1,2)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("2", buffer), @"The actual output is: %s", buffer);
}

- (void)test_MaxOf30and50and20_Gives50 {
    char *bc = tf_compile("$max(30,50,20)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("50", buffer), @"The actual output is: %s", buffer);
}

- (void)test_MinOf1and2_Gives1 {
    char *bc = tf_compile("$min(1,2)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("1", buffer), @"The actual output is: %s", buffer);
}

- (void)test_MinOf30and50and20_Gives20 {
    char *bc = tf_compile("$min(30,50,20)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("20", buffer), @"The actual output is: %s", buffer);
}

- (void)test_ModOf3and2_Gives1 {
    char *bc = tf_compile("$mod(3,2)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("1", buffer), @"The actual output is: %s", buffer);
}

- (void)test_ModOf6and3_Gives0 {
    char *bc = tf_compile("$mod(6,3)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("0", buffer), @"The actual output is: %s", buffer);
}

- (void)test_ModOf16and18and9_Gives7 {
    char *bc = tf_compile("$mod(16,18,9)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("7", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Mul2and5_Gives10 {
    char *bc = tf_compile("$mul(2,5)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("10", buffer), @"The actual output is: %s", buffer);
}

- (void)test_MulOf2and3and4_Gives24 {
    char *bc = tf_compile("$mul(2,3,4)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("24", buffer), @"The actual output is: %s", buffer);
}

- (void)test_MulDiv2and10and4_Gives5 {
    char *bc = tf_compile("$muldiv(2,10,4)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("5", buffer), @"The actual output is: %s", buffer);
}

- (void)test_MulDiv2and10and0_GivesEmpty {
    char *bc = tf_compile("$muldiv(2,10,0)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!buffer[0], @"The actual output is: %s", buffer);
}

- (void)test_MulDiv2and3and4_Gives2 {
    char *bc = tf_compile("$muldiv(2,3,4)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp ("2", buffer), @"The actual output is: %s", buffer);
}

- (void)test_Rand_GivesANumber {
    char *bc = tf_compile("$rand()");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    int num_digits = 0;
    for (int i = 0; buffer[i]; i++) {
        if (isdigit(buffer[i])) {
            num_digits++;
        }
    }
    XCTAssert(num_digits == strlen (buffer), @"The actual output is: %s", buffer);
}

- (void)test_RandWithArgs_GivesEmpty {
    char *bc = tf_compile("$rand(1)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    int num_digits = 0;
    for (int i = 0; buffer[i]; i++) {
        if (isdigit(buffer[i])) {
            num_digits++;
        }
    }
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_SubWithoutArgs_GivesEmpty {
    char *bc = tf_compile("$sub()");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    int num_digits = 0;
    for (int i = 0; buffer[i]; i++) {
        if (isdigit(buffer[i])) {
            num_digits++;
        }
    }
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_SubWith1Arg_GivesEmpty {
    char *bc = tf_compile("$sub(2)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    int num_digits = 0;
    for (int i = 0; buffer[i]; i++) {
        if (isdigit(buffer[i])) {
            num_digits++;
        }
    }
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_SubWith3and2_Gives1 {
    char *bc = tf_compile("$sub(3,2)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    int num_digits = 0;
    for (int i = 0; buffer[i]; i++) {
        if (isdigit(buffer[i])) {
            num_digits++;
        }
    }
    XCTAssert(!strcmp (buffer, "1"), @"The actual output is: %s", buffer);
}

- (void)test_SubWith10and5and2_Gives3 {
    char *bc = tf_compile("$sub(10,5,2)");
    ctx.it = NULL;
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    int num_digits = 0;
    for (int i = 0; buffer[i]; i++) {
        if (isdigit(buffer[i])) {
            num_digits++;
        }
    }
    XCTAssert(!strcmp (buffer, "3"), @"The actual output is: %s", buffer);
}

- (void)test_ChannelsFor3ChTrack_Gives3 {
    pl_replace_meta (it, ":CHANNELS", "3");
    char *bc = tf_compile("%channels%");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "3"), @"The actual output is: %s", buffer);
}

- (void)test_ChannelsFuncForMonoTrack_GivesMono {
    pl_replace_meta (it, ":CHANNELS", "1");
    char *bc = tf_compile("$channels()");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "mono"), @"The actual output is: %s", buffer);
}

- (void)test_ChannelsFuncForUnsetChannels_GivesStereo {
    char *bc = tf_compile("$channels()");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "stereo"), @"The actual output is: %s", buffer);
}

- (void)test_AndTrueArgs_ReturnsTrue {
    pl_replace_meta (it, "artist", "artist");
    pl_replace_meta (it, "album", "album");

    char *bc = tf_compile("$if($and(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "true"), @"The actual output is: %s", buffer);
}

- (void)test_AndTrueAndFalseArgs_ReturnsFalse {
    pl_replace_meta (it, "artist", "artist");

    char *bc = tf_compile("$if($and(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "false"), @"The actual output is: %s", buffer);
}

- (void)test_AndFalseArgs_ReturnsFalse {
    char *bc = tf_compile("$if($and(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "false"), @"The actual output is: %s", buffer);
}

- (void)test_OrTrueAndFalseArgs_ReturnsTrue {
    pl_replace_meta (it, "artist", "artist");

    char *bc = tf_compile("$if($or(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "true"), @"The actual output is: %s", buffer);
}

- (void)test_OrTrueArgs_ReturnsTrue {
    pl_replace_meta (it, "artist", "artist");
    pl_replace_meta (it, "album", "album");

    char *bc = tf_compile("$if($or(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "true"), @"The actual output is: %s", buffer);
}

- (void)test_OrFalseArgs_ReturnsFalse {
    char *bc = tf_compile("$if($or(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "false"), @"The actual output is: %s", buffer);
}

- (void)test_NotTrueArg_ReturnsFalse {
    pl_replace_meta (it, "artist", "artist");
    char *bc = tf_compile("$if($not(%artist%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "false"), @"The actual output is: %s", buffer);
}

- (void)test_NotFalseArg_ReturnsTrue {
    char *bc = tf_compile("$if($not(%artist%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "true"), @"The actual output is: %s", buffer);
}

- (void)test_XorTrueAndTrue_ReturnsFalse {
    pl_replace_meta (it, "artist", "artist");
    pl_replace_meta (it, "album", "album");
    char *bc = tf_compile("$if($xor(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "false"), @"The actual output is: %s", buffer);
}

- (void)test_XorTrueAndFalse_ReturnsTrue {
    pl_replace_meta (it, "artist", "artist");
    char *bc = tf_compile("$if($xor(%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "true"), @"The actual output is: %s", buffer);
}

- (void)test_XorFalseTrueFalse_ReturnsTrue {
    pl_replace_meta (it, "artist", "artist");
    char *bc = tf_compile("$if($xor(%album%,%artist%,%album%),true,false)");
    tf_eval (&ctx, bc, buffer, 1000);
    tf_free (bc);
    XCTAssert(!strcmp (buffer, "true"), @"The actual output is: %s", buffer);
}

- (void)test_PlayingColumnWithEmptyFormat_GivesQueueIndexes {
    playqueue_push (it);
    ctx.id = DB_COLUMN_PLAYING;
    ctx.flags |= DDB_TF_CONTEXT_HAS_ID;
    tf_eval (&ctx, NULL, buffer, 1000);
    XCTAssert(!strcmp (buffer, "(1)"), @"The actual output is: %s", buffer);
    playqueue_pop ();
}

- (void)test_LengthSamplesOf100Start300End_Returns200 {
    it->startsample = 100;
    it->endsample = 300;
    char *bc = tf_compile("%length_samples%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "200"), @"The actual output is: %s", buffer);
}

- (void)test_AbbrTestString_ReturnsAbbreviatedString {
    it->startsample = 100;
    it->endsample = 300;
    char *bc = tf_compile("$abbr('This is a Long Title (12-inch version) [needs tags]')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "TiaLT1v[needst"), @"The actual output is: %s", buffer);
}

- (void)test_AbbrTestUnicodeString_ReturnsAbbreviatedString {
    it->startsample = 100;
    it->endsample = 300;
    char *bc = tf_compile("$abbr('This ɀHİJ a русский Title (12-inch version) [needs tags]')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "TɀaрT1v[needst"), @"The actual output is: %s", buffer);
}

- (void)test_AnsiTestString_ReturnsTheSameString {
    char *bc = tf_compile("$ansi(ABCDабвг)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "ABCDабвг"), @"The actual output is: %s", buffer);
}

- (void)test_AsciiTestString_ReturnsAsciiSameStringWithInvalidCharsStripped {
    char *bc = tf_compile("$ascii(олдABCDабвг)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "ABCD"), @"The actual output is: %s", buffer);
}


- (void)test_CapsTestAsciiString_ReturnsCapitalizeEachWordString {
    char *bc = tf_compile("$caps(MY TEST STRING)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "My Test String"), @"The actual output is: %s", buffer);
}


- (void)test_CapsTestAsciiRandomizedString_ReturnsCapitalizeEachWordString {
    char *bc = tf_compile("$caps(MY TesT STriNG)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "My Test String"), @"The actual output is: %s", buffer);
}

- (void)test_CapsTestUnicodeRandomizedString_ReturnsCapitalizeEachWordString {
    char *bc = tf_compile("$caps(AsciiAlbumName РуССкоЕНазВАние ΠΥΘΑΓΌΡΑΣ)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "Asciialbumname Русскоеназвание Πυθαγόρασ"), @"The actual output is: %s", buffer);
}

- (void)test_CapsTestUnicodeStringWithNonMatchinByteLengthsForLowerUpperCaseChars_ReturnsCapitalizeEachWordString {
    char *bc = tf_compile("$caps(ɑBCD ɀHİJ)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "Ɑbcd Ɀhij"), @"The actual output is: %s", buffer);
}

- (void)test_Caps2TestUnicodeRandomizedString_ReturnsCapitalizeEachWordString {
    char *bc = tf_compile("$caps2(ɑBCD ɀHİJ)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "ⱭBCD ⱿHİJ"), @"The actual output is: %s", buffer);
}

- (void)test_Char1055And88And38899_ReturnsCorrespondingUTF8Chars {
    char *bc = tf_compile("$char(1055)$char(88)$char(38899)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "ПX音"), @"The actual output is: %s", buffer);
}

- (void)test_Crc32Of123456789_Returns3421780262 {
    char *bc = tf_compile("$crc32(123456789)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "3421780262"), @"The actual output is: %s", buffer);
}

- (void)test_CrLf_InsertsLinebreak {
    ctx.flags |= DDB_TF_CONTEXT_MULTILINE;
    char *bc = tf_compile("$crlf()");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "\n"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePath_ReturnsDirectory {
    char *bc = tf_compile("$directory(/directory/file.path)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "directory"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathWithMultipleSlashes_ReturnsDirectory {
    char *bc = tf_compile("$directory(/directory///file.path)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "directory"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathWithoutFrontSlash_ReturnsDirectory {
    char *bc = tf_compile("$directory(directory/file.path)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "directory"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathWithMoreNestedness_ReturnsDirectory {
    char *bc = tf_compile("$directory(/path/directory/file.path)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "directory"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathWithoutDirectory_ReturnsEmpty {
    char *bc = tf_compile("$directory(file.path)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathAtRoot_ReturnsEmpty {
    char *bc = tf_compile("$directory(/file.path)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnEmptyAtRoot_ReturnsEmpty {
    char *bc = tf_compile("$directory(/)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnEmpty_ReturnsEmpty {
    char *bc = tf_compile("$directory()");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathLevel0_ReturnsEmpty {
    char *bc = tf_compile("$directory(/directory/file.path,0)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathLevel1_ReturnsDirectory1 {
    char *bc = tf_compile("$directory(/directory3/directory2/directory1/file.path,1)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "directory1"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathLevel2_ReturnsDirectory2 {
    char *bc = tf_compile("$directory(/directory3/directory2/directory1/file.path,2)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "directory2"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathLevel4_ReturnsEmpty {
    char *bc = tf_compile("$directory(/directory3/directory2/directory/file.path,4)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryOnFilePathLevel2MultipleSlashes_ReturnsDirectory2 {
    char *bc = tf_compile("$directory(////directory3////directory2////directory////file.path,2)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "directory2"), @"The actual output is: %s", buffer);
}

- (void)test_MultiLine_LineBreaksIgnored {
    char *bc = tf_compile("hello\nworld");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "helloworld"), @"The actual output is: %s", buffer);
}

- (void)test_MultiLineWithComments_LineBreaksAndCommentedLinesIgnored {
    char *bc = tf_compile("// this is a comment\nhello\nworld\n//another comment\nmore text");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "helloworldmore text"), @"The actual output is: %s", buffer);
}

- (void)test_QuotedSpecialChars_TreatedLiterally {
    char *bc = tf_compile("'blah$blah%blah[][]'");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "blah$blah%blah[][]"), @"The actual output is: %s", buffer);
}

- (void)test_FunctionArgumentsOnMultipleLinesWithComments_LinebreaksAndCommentsIgnored {
    char *bc = tf_compile("$add(1,\n2,\n3,//4,\n5)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "11"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryPathOnFilePath_ReturnsDirectoryPath {
    char *bc = tf_compile("$directory_path('/a/b/c/d.mp3')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "/a/b/c"), @"The actual output is: %s", buffer);
}

- (void)test_DirectoryPathOnPathWithoutFile_ReturnsDirectoryPath {
    char *bc = tf_compile("$directory_path('/a/b/c/d/')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "/a/b/c/d"), @"The actual output is: %s", buffer);
}

- (void)test_ExtOnFilePath_ReturnsExt {
    char *bc = tf_compile("$ext('/a/b/c/d/file.mp3')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "mp3"), @"The actual output is: %s", buffer);
}

- (void)test_ExtOnFileWithoutExtPath_ReturnsEmpty {
    char *bc = tf_compile("$ext('/a/b/c/d/file')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_ExtOnFilePathWithoutFilename_ReturnsEmpty {
    char *bc = tf_compile("$ext('/a/b/c/d/')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_ExtOnFilePathEndingWithDot_ReturnsEmpty {
    char *bc = tf_compile("$ext('/a/b/c/d/file.')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_ExtOnFilePathDotFile_ReturnsExt {
    char *bc = tf_compile("$ext('/a/b/c/d/.ext')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "ext"), @"The actual output is: %s", buffer);
}

- (void)test_ExtOnFileExtWithMultiplePeriod_ReturnsExt {
    char *bc = tf_compile("$ext('/a/b/c/d/file.iso.wv')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "wv"), @"The actual output is: %s", buffer);
}

- (void)test_FilenameOnFilePath_ReturnsFilename {
    char *bc = tf_compile("$filename('/a/b/c/d/file.mp3')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "file.mp3"), @"The actual output is: %s", buffer);
}

- (void)test_FilenameOnFilePathWithoutFile_ReturnsEmpty {
    char *bc = tf_compile("$filename('/a/b/c/d/')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_FilenameOnFilenameWithoutPath_ReturnsFilename {
    char *bc = tf_compile("$filename('file.iso.wv')");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "file.iso.wv"), @"The actual output is: %s", buffer);
}

- (void)test_Date_ReturnsYearValue {
    pl_replace_meta (it, "year", "1980");
    char *bc = tf_compile("%date%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "1980"), @"The actual output is: %s", buffer);
}

- (void)test_CustomField_ReturnsTheFieldValue {
    pl_replace_meta (it, "random_name", "random value");
    char *bc = tf_compile("%random_name%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "random value"), @"The actual output is: %s", buffer);
}

- (void)test_MultipleArtists_ReturnsArtistsSeparatedBySemicolons {
    pl_replace_meta (it, "artist", "Artist1\nArtist2");
    char *bc = tf_compile("%artist%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "Artist1;Artist2"), @"The actual output is: %s", buffer);
}

- (void)test_EmptyTitle_YieldsFilename {
    pl_replace_meta (it, ":URI", "/home/user/filename.mp3");
    char *bc = tf_compile("%title%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "filename"), @"The actual output is: %s", buffer);
}

- (void)test_DoublingPercentDollarApostrophe_OutputsSinglePercentDollarApostrophe {
    char *bc = tf_compile("''$$%%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "'$%"), @"The actual output is: %s", buffer);
}

- (void)test_PlaybackTime_OutputsPlaybackTime {
    streamer_set_playing_track (it);
    char *bc = tf_compile("%playback_time%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "0:00"), @"The actual output is: %s", buffer);
}

- (void)test_NoDynamicFlag_SkipsDynamicFields {
    char *bc = tf_compile("header|%playback_time%|footer");
    ctx.flags |= DDB_TF_CONTEXT_NO_DYNAMIC;
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "header||footer"), @"The actual output is: %s", buffer);
}

- (void)test_Track_Number_SingleDigit_ReturnsNonZeroPaddedTrackNumber {
    pl_replace_meta (it, "track", "5");
    char *bc = tf_compile("%track number%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "5"), @"The actual output is: %s", buffer);
}

// this should not normally happen if the metadata was loaded correctly
// but older playlists can have this
- (void)test_Track_Number_SingleDigitWithTotalTracks_ReturnsNonZeroPaddedTrackNumber {
    pl_replace_meta (it, "track", "5/7");
    char *bc = tf_compile("%track number%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "5"), @"The actual output is: %s", buffer);
}

- (void)test_Length_DoesntGetPaddedWithSpace {
    plt_set_item_duration(NULL, it, 130);
    char *bc = tf_compile("%length%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "2:10"), @"The actual output is: %s", buffer);
}

- (void)test_ImportLegacyDirectAccess_ProducesExpectedData {
    const char *old = "%@disc@";
    char new[100];
    tf_import_legacy (old, new, sizeof (new));
    XCTAssert(!strcmp (new, "%disc%"), @"The actual output is: %s", buffer);
}

- (void)test_NestedSquareBracketsWithUndefVarsAndLiteralData_ReturnEmpty {
    pl_replace_meta (it, "title", "title");
    char *bc = tf_compile("[[%discnumber%]a] normaltext [%title%] [[%title%]a]");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, " normaltext title titlea"), @"The actual output is: %s", buffer);
}

- (void)test_FixEof_PutsIndicatorAfterLineBreak {
    pl_replace_meta (it, "title", "line1\nline2\n");
    char *bc = tf_compile("$fix_eol(%title%)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "line1 (...)"), @"The actual output is: %s", buffer);
}

- (void)test_FixEofTwoArgs_PutsCustomIndicatorAfterLineBreak {
    pl_replace_meta (it, "title", "line1\nline2\n");
    char *bc = tf_compile("$fix_eol(%title%, <...>)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "line1 <...>"), @"The actual output is: %s", buffer);
}

- (void)test_FixEofTwoArgsWithSmallBuffer_DoesntOverflowOffByOne {
    pl_replace_meta (it, "title", "hello\n");
    char *bc = tf_compile("$fix_eol(%title%, <...>)");
    tf_eval (&ctx, bc, buffer, 12);
    XCTAssert(!strcmp (buffer, "hello <...>"), @"The actual output is: %s", buffer);
    tf_eval (&ctx, bc, buffer, 11);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
}

- (void)test_Hex_ReturnsHexConvertedNumber {
    char *bc = tf_compile("$hex(11259375)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "abcdef"), @"The actual output is: %s", buffer);
}

- (void)test_HexPadded_ReturnsHexConvertedNumberWithPadding {
    char *bc = tf_compile("$hex(11259375,10)");
    tf_eval (&ctx, bc, buffer, 10);
    XCTAssert(!strcmp (buffer, ""), @"The actual output is: %s", buffer);
    tf_eval (&ctx, bc, buffer, 11);
    XCTAssert(!strcmp (buffer, "0000abcdef"), @"The actual output is: %s", buffer);
}

- (void)test_HexZero_ReturnsZero {
    char *bc = tf_compile("$hex(0)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "0"), @"The actual output is: %s", buffer);
}

- (void)test_QuotedSquareBrackets_ReturnsSquareBrackets {
    char *bc = tf_compile("'['']'");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "[]"), @"The actual output is: %s", buffer);
}

- (void)test_ImportLegacySquareBrackets_ProducesQuotedSquareBrackets {
    const char *old = "[%y]";
    char new[100];
    tf_import_legacy (old, new, sizeof (new));
    XCTAssert(!strcmp (new, "'['%date%']'"), @"The actual output is: %s", buffer);
}

- (void)test_Num_123_5_Returns_00123 {
    char *bc = tf_compile("$num(123,5)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "00123"), @"The actual output is: %s", buffer);
}

- (void)test_Num_Minus123_5_Returns__0123 {
    char *bc = tf_compile("$num(-123,5)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "-0123"), @"The actual output is: %s", buffer);
}

- (void)test_NumFractional_ReturnsIntegerFloor {
    char *bc = tf_compile("$num(4.8,5)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "00004"), @"The actual output is: %s", buffer);
}

- (void)test_NumNonNumber_ReturnsZero {
    char *bc = tf_compile("$num(A1,5)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "00000"), @"The actual output is: %s", buffer);
}

- (void)test_NumLargeNumber_DoesntTruncate {
    char *bc = tf_compile("$num(1234,3)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "1234"), @"The actual output is: %s", buffer);
}

- (void)test_NumNegativePadding_GivesZeroPadding {
    char *bc = tf_compile("$num(1,-3)");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "1"), @"The actual output is: %s", buffer);
}

- (void)test_IsPlayingReturnValueTrue_CorrespondsToStringValue {
    streamer_set_playing_track (it);
    plug_set_output (&fake_out);
    fake_out_state_value = OUTPUT_STATE_PLAYING;
    char *bc = tf_compile("$if(%isplaying%,YES,NO) %isplaying%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "YES 1"), @"The actual output is: %s", buffer);

}

- (void)test_IsPlayingReturnValueFalse_CorrespondsToStringValue {
    plug_set_output (&fake_out);
    fake_out_state_value = OUTPUT_STATE_STOPPED;
    char *bc = tf_compile("$if(%isplaying%,YES,NO) %isplaying%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "NO "), @"The actual output is: %s", buffer);
}

- (void)test_IsPausedReturnValueTrue_CorrespondsToStringValue {
    streamer_set_playing_track (it);
    plug_set_output (&fake_out);
    fake_out_state_value = OUTPUT_STATE_PAUSED;
    char *bc = tf_compile("$if(%ispaused%,YES,NO) %ispaused%");
    tf_eval (&ctx, bc, buffer, 1000);
    XCTAssert(!strcmp (buffer, "YES 1"), @"The actual output is: %s", buffer);
}

@end
