/*
    ClipGrab³
    Copyright (C) Philipp Schmieder
    http://clipgrab.de
    feedback [at] clipgrab [dot] de

    This file is part of ClipGrab.
    ClipGrab is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.
    
    ClipGrab is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ClipGrab.  If not, see <http://www.gnu.org/licenses/>.
*/



#include "mainwindow.h"

MainWindow::MainWindow(QWidget *parent, Qt::WindowFlags flags)
    : QMainWindow(parent, flags)
{
    ui.setupUi(this);
    currentVideo = nullptr;
    searchReply = nullptr;
}


MainWindow::~MainWindow()
{
    delete this->searchPage;
}

void MainWindow::init()
{
    //*
    //* Adding version info to the footer
    //*
    this->ui.label->setText(ui.label->text().replace("%version", "ClipGrab " + QCoreApplication::applicationVersion()));

    //*
    //* Tray Icon
    //*
    systemTrayIcon.setIcon(QIcon(":/img/icon.png"));
    systemTrayIcon.show();
    connect(&systemTrayIcon, SIGNAL(activated(QSystemTrayIcon::ActivationReason)), this, SLOT(systemTrayIconActivated(QSystemTrayIcon::ActivationReason)));
    connect(&systemTrayIcon, SIGNAL(messageClicked()), this, SLOT(systemTrayMessageClicked()));

    //*
    //* Clipboard Handling
    //*
    connect(cg, SIGNAL(compatiblePortalFound(bool, QString, video*)), this, SLOT(compatiblePortalFound(bool, QString, video*)));
    connect(cg, SIGNAL(compatibleUrlFoundInClipboard(QString)), this, SLOT(compatibleUrlFoundInClipBoard(QString)));

    //*
    //* Download Tab
    //*
    connect(ui.downloadStart, SIGNAL(clicked()), this, SLOT(startDownload()));
    connect(ui.downloadLineEdit, SIGNAL(textChanged(QString)), cg, SLOT(determinePortal(QString)));
    connect(this, SIGNAL(itemToRemove(int)), cg, SLOT(removeDownload(int)));
    ui.downloadTree->header()->setSectionResizeMode(1, QHeaderView::Stretch);
    ui.downloadTree->header()->setStretchLastSection(false);
    ui.downloadTree->header()->setSectionResizeMode(3, QHeaderView::ResizeToContents);
    ui.downloadLineEdit->setFocus(Qt::OtherFocusReason);

    int lastFormat = cg->settings.value("LastFormat", 0).toInt();
    for (int i = 0; i < this->cg->formats.size(); ++i)
    {
        this->ui.downloadComboFormat->addItem(this->cg->formats.at(i)._name);
    }

    this->ui.downloadComboFormat->setCurrentIndex(lastFormat);

    ui.downloadPause->hide(); //Qt does currently not handle throttling downloads properly


    //*
    //* Search Tab
    //*
    this->searchNam = new QNetworkAccessManager();

    connect(ui.searchLineEdit, SIGNAL(textChanged(QString)), cg, SLOT(determinePortal(QString)));
    QWebEngineProfile* profile = new QWebEngineProfile;
    this->searchPage = new SearchWebEnginePage(profile);
    ui.searchWebEngineView->setPage(searchPage);
    ui.searchWebEngineView->settings()->setAttribute(QWebEngineSettings::FocusOnNavigationEnabled, false);
    ui.searchWebEngineView->setContextMenuPolicy(Qt::NoContextMenu);
    connect(ui.searchWebEngineView->page(), SIGNAL(linkClicked(QUrl)), this, SLOT(handleSearchResultClicked(QUrl)));
    connect(ui.searchWebEngineView->page(), SIGNAL(linkIntercepted(QUrl)), this, SLOT(handleSearchResultIntercepted(QUrl)));
    connect(&searchTimer, SIGNAL(timeout()), this, SLOT(searchTimerTimeout()));

    //*
    //* Settings Tab
    //*
    connect(this->ui.settingsRadioClipboardAlways, SIGNAL(toggled(bool)), this, SLOT(settingsClipboard_toggled(bool)));
    connect(this->ui.settingsRadioClipboardNever, SIGNAL(toggled(bool)), this, SLOT(settingsClipboard_toggled(bool)));
    connect(this->ui.settingsRadioClipboardAsk, SIGNAL(toggled(bool)), this, SLOT(settingsClipboard_toggled(bool)));
    connect(this->ui.settingsRadioNotificationsAlways, SIGNAL(toggled(bool)), this, SLOT(settingsNotifications_toggled(bool)));
    connect(this->ui.settingsRadioNotificationsFinish, SIGNAL(toggled(bool)), this, SLOT(settingsNotifications_toggled(bool)));
    connect(this->ui.settingsRadioNotificationsNever, SIGNAL(toggled(bool)), this, SLOT(settingsNotifications_toggled(bool)));


    this->ui.settingsSavedPath->setText(cg->settings.value("savedPath", QStandardPaths::writableLocation(QStandardPaths::DesktopLocation)).toString());
    this->ui.settingsSaveLastPath->setChecked(cg->settings.value("saveLastPath", true).toBool());
    ui.settingsNeverAskForPath->setChecked(cg->settings.value("NeverAskForPath", false).toBool());

    ui.settingsUseMetadata->setChecked(cg->settings.value("UseMetadata", false).toBool());
    connect(this->ui.settingsUseMetadata, SIGNAL(stateChanged(int)), this, SLOT(on_settingsUseMetadata_stateChanged(int)));


    ui.settingsUseProxy->setChecked(cg->settings.value("UseProxy", false).toBool());
    ui.settingsProxyAuthenticationRequired->setChecked(cg->settings.value("ProxyAuthenticationRequired", false).toBool());
    ui.settingsProxyHost->setText(cg->settings.value("ProxyHost", "").toString());
    ui.settingsProxyPassword->setText(cg->settings.value("ProxyPassword", "").toString());
    ui.settingsProxyPort->setValue(cg->settings.value("ProxyPort", "").toInt());
    ui.settingsProxyUsername->setText(cg->settings.value("ProxyUsername", "").toString());
    ui.settingsProxyType->setCurrentIndex(cg->settings.value("ProxyType", 0).toInt());

    connect(this->ui.settingsUseProxy, SIGNAL(toggled(bool)), this, SLOT(settingsProxyChanged()));
    connect(this->ui.settingsProxyAuthenticationRequired, SIGNAL(toggled(bool)), this, SLOT(settingsProxyChanged()));
    connect(this->ui.settingsProxyHost, SIGNAL(textChanged(QString)), this, SLOT(settingsProxyChanged()));
    connect(this->ui.settingsProxyPassword, SIGNAL(textChanged(QString)), this, SLOT(settingsProxyChanged()));
    connect(this->ui.settingsProxyPort, SIGNAL(valueChanged(int)), this, SLOT(settingsProxyChanged()));
    connect(this->ui.settingsProxyUsername, SIGNAL(textChanged(QString)), this, SLOT(settingsProxyChanged()));
    connect(this->ui.settingsProxyType, SIGNAL(currentIndexChanged(int)), this, SLOT(settingsProxyChanged()));
    settingsProxyChanged();
    ui.settingsMinimizeToTray->setChecked(cg->settings.value("MinimizeToTray", false).toBool());

    if (cg->settings.value("Clipboard", "ask") == "always")
    {
        ui.settingsRadioClipboardAlways->setChecked(true);
    }
    else if (cg->settings.value("Clipboard", "ask") == "never")
    {
        ui.settingsRadioClipboardNever->setChecked(true);
    }
    else
    {
        ui.settingsRadioClipboardAsk->setChecked(true);
    }

    if (cg->settings.value("Notifications", "finish") == "finish")
    {
        ui.settingsRadioNotificationsFinish->setChecked(true);
    }
    else if (cg->settings.value("Notifications", "finish") == "always")
    {
        ui.settingsRadioNotificationsAlways->setChecked(true);
    }
    else
    {
        ui.settingsRadioNotificationsNever->setChecked(true);
    }

    ui.settingsRememberLogins->setChecked(cg->settings.value("rememberLogins", true).toBool());
    ui.settingsRememberVideoQuality->setChecked(cg->settings.value("rememberVideoQuality", true).toBool());
    ui.settingsUseWebM->setChecked(cg->settings.value("UseWebM", false).toBool());
    ui.settingsRemoveFinishedDownloads->setChecked(cg->settings.value("RemoveFinishedDownloads", false).toBool());
    ui.settingsIgnoreSSLErrors->setChecked(cg->settings.value(("IgnoreSSLErrors"), false).toBool());


    int langIndex = 0;
    for (int i=0; i < cg->languages.count(); i++)
    {
        if (cg->languages.at(i).code == cg->settings.value("Language", "auto").toString())
        {
            langIndex = i;
        }
    }
    for (int i=0; i < cg->languages.count(); i++)
    {
        ui.settingsLanguage->addItem(cg->languages.at(i).name, cg->languages.at(i).code);
    }
    ui.settingsLanguage->setCurrentIndex(langIndex);


    this->ui.tabWidget->removeTab(2); //fixme!

    startTimer(500);

    //*
    //* About Tab
    //*

    #ifdef Q_OS_MAC
        this->ui.downloadOpen->hide();
        this->cg->settings.setValue("Clipboard", "always");
        this->ui.generalSettingsTabWidget->removeTab(2);
        //FIXME this->setStyleSheet("#label_4{padding-top:25px}#label{font-size:10px}#centralWidget{background:#fff};#mainTab{margin-top;-20px};#label_4{padding:10px}#downloadInfoBox, #settingsGeneralInfoBox, #settingsLanguageInfoBox, #aboutInfoBox, #searchInfoBox{color: background: #00B4DE;}");
        // this->ui.label_4->setMinimumHeight(120);
    #endif

    //*
    //* Drag and Drop
    //*
    this->setAcceptDrops(true);
    this->ui.searchWebEngineView->setAcceptDrops(false);

    //*
    //*Keyboard shortcuts
    //*
    QSignalMapper* tabShortcutMapper = new QSignalMapper(this);

    QShortcut* tabShortcutSearch = new QShortcut(QKeySequence(Qt::ControlModifier + Qt::Key_1), this);
    tabShortcutMapper->setMapping(tabShortcutSearch, 0);
    QShortcut* tabShortcutDownload = new QShortcut(QKeySequence(Qt::ControlModifier + Qt::Key_2), this);
    tabShortcutMapper->setMapping(tabShortcutDownload, 1);
    QShortcut* tabShortcutSettings = new QShortcut(QKeySequence(Qt::ControlModifier + Qt::Key_3), this);
    tabShortcutMapper->setMapping(tabShortcutSettings, 2);
    QShortcut* tabShortcutAbout = new QShortcut(QKeySequence(Qt::ControlModifier + Qt::Key_4), this);
    tabShortcutMapper->setMapping(tabShortcutAbout, 3);

    connect(tabShortcutSearch, SIGNAL(activated()), tabShortcutMapper, SLOT(map()));
    connect(tabShortcutDownload, SIGNAL(activated()), tabShortcutMapper, SLOT(map()));
    connect(tabShortcutSettings, SIGNAL(activated()), tabShortcutMapper, SLOT(map()));
    connect(tabShortcutAbout, SIGNAL(activated()), tabShortcutMapper, SLOT(map()));
    connect(tabShortcutMapper, SIGNAL(mapped(int)), this->ui.mainTab, SLOT(setCurrentIndex(int)));

    QApplication::quit();

    //*
    //*Miscellaneous
    //*
    on_searchLineEdit_textChanged("");
    this->ui.mainTab->setCurrentIndex(cg->settings.value("MainTab", 0).toInt());

    //Prevent updating remembered resolution when updating programatically
    this->updatingComboQuality = false;

    #ifdef Q_OS_MACX
    //if ( QSysInfo::MacintoshVersion > QSysInfo::MV_10_8 )
    //{
        // fix Mac OS X 10.9 (mavericks) font issue
        // https://bugreports.qt-project.org/browse/QTBUG-32789
        QFont::insertSubstitution(".Lucida Grande UI", "Lucida Grande");
    //}
    #endif
}

void MainWindow::startDownload()
{
    QString targetDirectory = cg->settings.value("savedPath", QStandardPaths::writableLocation(QStandardPaths::DesktopLocation)).toString();
    QString fileName = currentVideo->getSaveTitle();

    if (cg->settings.value("NeverAskForPath", false).toBool() == false)
    {
       targetFileSelected(QFileDialog::getSaveFileName(this, tr("Select Target"), targetDirectory +"/" + fileName));
    }
    else
    {
        targetFileSelected(targetDirectory + "/" + fileName);
    }
}

void MainWindow::targetFileSelected(QString target)
{
    if (target.isEmpty())
    {
        return;
    }

    QDir targetDir = QFileInfo(target).absoluteDir();
    if (!targetDir.exists())
    {
        ui.settingsNeverAskForPath->setChecked(false);
        this->startDownload();
        return;
    }

    QTreeWidgetItem* tmpItem = new QTreeWidgetItem(QStringList() << currentVideo->getName() << currentVideo->title() << ui.downloadComboFormat->currentText());
    tmpItem->setSizeHint(0, QSize(16, 24));
    currentVideo->setTreeItem(tmpItem);
    currentVideo->setQuality(ui.downloadComboQuality->currentIndex());
    currentVideo->setConverter(cg->formats.at(ui.downloadComboFormat->currentIndex())._converter->createNewInstance(),cg->formats.at(ui.downloadComboFormat->currentIndex())._mode);

    if (cg->settings.value("saveLastPath", true) == true)
    {
        QString targetDir = target;
        targetDir.remove(targetDir.split("/", QString::SkipEmptyParts).last()).replace(QRegExp("/+$"), "/");
        ui.settingsSavedPath->setText(targetDir);
    }
    currentVideo->setTargetPath(target);

    if (cg->settings.value("UseMetadata", false).toBool() == true)
    {
        if (ui.downloadComboFormat->currentIndex() == 4 || ui.downloadComboFormat->currentIndex() == 5)
        {
            metadataDialog = new QDialog;
            mdui.setupUi(metadataDialog);
            mdui.title->setText(currentVideo->title());
            metadataDialog->setModal(true);
            metadataDialog->exec();

            currentVideo->setMetaTitle(mdui.title->text());
            currentVideo->setMetaArtist(mdui.artist->text());

            delete metadataDialog;
        }
    }

    cg->addDownload(currentVideo);
    ui.downloadTree->insertTopLevelItem(0, tmpItem);

    currentVideo->_progressBar = new QProgressBar();
    currentVideo->_progressBar->setValue(0);
    currentVideo->_progressBar->setMaximum(1);
    ui.downloadTree->setItemWidget(tmpItem, 3, currentVideo->_progressBar);

    ((QProgressBar*) ui.downloadTree->itemWidget(tmpItem, 3))->setMaximum(100);
    connect(currentVideo, SIGNAL(progressChanged(int,int)),ui.downloadTree, SLOT(update()));
    connect(currentVideo, SIGNAL(conversionFinished(video*)), this, SLOT(handleFinishedConversion(video*)));
    currentVideo = nullptr;
    ui.downloadLineEdit->clear();
}

void MainWindow::compatiblePortalFound(bool found, QString url, video* portal)
{
    disableDownloadUi(true);
    this->updatingComboQuality = true;
    ui.downloadComboQuality->clear();
    this->updatingComboQuality = false;
    if (found == true)
    {
        this->ui.mainTab->setCurrentIndex(1);
        ui.downloadLineEdit->setText(url);
        ui.downloadLineEdit->setReadOnly(true);
        ui.downloadInfoBox->setText(tr("Please wait while ClipGrab is loading information about the video ..."));

        if (currentVideo)
        {
            currentVideo->deleteAfterAnalysingFinished();
        }
        currentVideo = portal->createNewInstance();
        currentVideo->setUrl(ui.downloadLineEdit->text());
        connect(currentVideo, SIGNAL(error(QString,video*)), cg, SLOT(errorHandler(QString,video*)));
        connect(currentVideo, SIGNAL(analysingFinished()), this, SLOT(updateVideoInfo()));
        currentVideo->analyse();

    }
    else
    {
        if (ui.downloadLineEdit->text() == "")
        {
            ui.downloadInfoBox->setText(tr("Please enter the link to the video you want to download in the field below."));
        }
        else if (ui.downloadLineEdit->text().startsWith("http://") || ui.downloadLineEdit->text().startsWith("https://"))
        {            
            ui.downloadLineEdit->setReadOnly(true);
            ui.downloadInfoBox->setText(tr("The link you have entered seems to not be recognised by any of the supported portals.<br/>Now ClipGrab will check if it can download a video from that site anyway."));

            if (currentVideo)
            {
                currentVideo->deleteAfterAnalysingFinished();
            }
            currentVideo = cg->heuristic->createNewInstance();
            currentVideo->setUrl(ui.downloadLineEdit->text());
            connect(currentVideo, SIGNAL(error(QString,video*)), cg, SLOT(errorHandler(QString,video*)));
            connect(currentVideo, SIGNAL(analysingFinished()), this, SLOT(updateVideoInfo()));
            currentVideo->analyse();
        }
    }
}

void MainWindow::updateVideoInfo()
{
    QList< QPair<QString, int> > supportedQualities = currentVideo->getSupportedQualities();

    if (currentVideo && currentVideo->title() != "" && !supportedQualities.isEmpty())
    {
        this->updatingComboQuality = true;
        ui.downloadInfoBox->setText("<strong>" + currentVideo->title() + "</strong>");
        ui.downloadLineEdit->setReadOnly(false);
        disableDownloadUi(false);
        ui.downloadComboQuality->clear();
        int rememberedResolution = cg->settings.value("rememberedVideoQuality", -1).toInt();
        int bestResolutionMatch = 0;
        int bestResolutionMatchPosition = 0;
        for (int i = 0; i < supportedQualities.length(); i++)
        {
            int resolution = supportedQualities.at(i).second;
            ui.downloadComboQuality->addItem(supportedQualities.at(i).first, resolution);
            if (resolution <= rememberedResolution && resolution > bestResolutionMatch)
            {
                bestResolutionMatch = resolution;
                bestResolutionMatchPosition = i;
            }
        }
        if (cg->settings.value("rememberVideoQuality", true).toBool())
        {
            ui.downloadComboQuality->setCurrentIndex(bestResolutionMatchPosition);
        }
        this->updatingComboQuality = false;
    }
    else
    {
        ui.downloadInfoBox->setText(tr("No downloadable video could be found.<br />Maybe you have entered the wrong link or there is a problem with your connection."));
        ui.downloadLineEdit->setReadOnly(false);
        disableDownloadUi(true);
    }
}
void MainWindow::on_settingsSavedPath_textChanged(QString string)
{
    this->cg->settings.setValue("savedPath", string);
}

void MainWindow::on_settingsBrowseTargetPath_clicked()
{
    QString newPath;
    newPath = QFileDialog::getExistingDirectory(this, tr("ClipGrab - Select target path"), ui.settingsSavedPath->text());
    if (!newPath.isEmpty())
    {
        ui.settingsSavedPath->setText(newPath);
    }
}

void MainWindow::on_downloadCancel_clicked()
{
    if (ui.downloadTree->currentIndex().row() != -1)
    {
        int item = ui.downloadTree->topLevelItemCount() - ui.downloadTree->currentIndex().row() -1;
        ui.downloadTree->takeTopLevelItem(ui.downloadTree->currentIndex().row());
        emit itemToRemove(item);
    }
}

void MainWindow::on_settingsSaveLastPath_stateChanged(int state)
{
    if (state == Qt::Checked)
    {
        cg->settings.setValue("saveLastPath", true);
    }
    else
    {
        cg->settings.setValue("saveLastPath", false);
    }
}

void MainWindow::on_downloadOpen_clicked()
{
    if (ui.downloadTree->currentIndex().row() != -1)
    {
        int item = ui.downloadTree->topLevelItemCount() - ui.downloadTree->currentIndex().row() -1;
        cg->openTargetFolder(item);
    }
}


void MainWindow::compatibleUrlFoundInClipBoard(QString url)
{
    if (QApplication::activeModalWidget() == nullptr)
    {
        if (cg->settings.value("Clipboard", "ask") == "ask")
        {
            Notifications::showMessage(tr("ClipGrab: Video discovered in your clipboard"), tr("ClipGrab has discovered the address of a compatible video in your clipboard. Click on this message to download it now."), &systemTrayIcon);
        }
        else if (cg->settings.value("Clipboard", "ask") == "always")
        {
            this->ui.downloadLineEdit->setText(url);
            this->ui.mainTab->setCurrentIndex(1);
            this->setWindowState((windowState() & ~Qt::WindowMinimized) | Qt::WindowActive);
            this->show();
            this->activateWindow();
            this->raise();
        }
    }

}

void MainWindow::systemTrayMessageClicked()
{
    if (QApplication::activeModalWidget() == nullptr)
    {
        this->ui.downloadLineEdit->setText(cg->clipboardUrl);
        this->ui.mainTab->setCurrentIndex(1);
        this->setWindowState((windowState() & ~Qt::WindowMinimized) | Qt::WindowActive);
        this->show();
        this->activateWindow();
        this->raise();
    }
    

}

void MainWindow::systemTrayIconActivated(QSystemTrayIcon::ActivationReason)
{
    if (cg->settings.value("MinimizeToTray", false).toBool() && !this->isHidden())
    {
        this->hide();
    }
    else
    {
        this->setWindowState((windowState() & ~Qt::WindowMinimized) | Qt::WindowActive);
        this->show();
        this->activateWindow();
        this->raise();
    }
}

void MainWindow::settingsClipboard_toggled(bool)
{
    if (ui.settingsRadioClipboardAlways->isChecked())
    {
        cg->settings.setValue("Clipboard", "always");
    }
    else if (ui.settingsRadioClipboardNever->isChecked())
    {
        cg->settings.setValue("Clipboard", "never");
    }
    else if (ui.settingsRadioClipboardAsk->isChecked())
    {
        cg->settings.setValue("Clipboard", "ask");
    }
}

void MainWindow::disableDownloadUi(bool disable)
{
    ui.downloadComboFormat->setDisabled(disable);
    ui.downloadComboQuality->setDisabled(disable);
    ui.downloadStart->setDisabled(disable);
    ui.label_2->setDisabled(disable);
    ui.label_3->setDisabled(disable);
}

void MainWindow::disableDownloadTreeButtons(bool disable)
{
    ui.downloadOpen->setDisabled(disable);
    ui.downloadCancel->setDisabled(disable);
    ui.downloadPause->setDisabled(disable);
}

void MainWindow::on_downloadTree_currentItemChanged(QTreeWidgetItem* /*current*/, QTreeWidgetItem* /*previous*/)
{
    if (ui.downloadTree->currentIndex().row() == -1)
    {
        disableDownloadTreeButtons();
    }
    else
    {
        disableDownloadTreeButtons(false);
    }
}

void MainWindow::on_settingsNeverAskForPath_stateChanged(int state)
{
    if (state == Qt::Checked)
    {
        cg->settings.setValue("NeverAskForPath", true);
    }
    else
    {
        cg->settings.setValue("NeverAskForPath", false);
    }
}

 void MainWindow::closeEvent(QCloseEvent *event)
 {
    if (cg->downloadsRunning() > 0)
    {
        QMessageBox* exitBox;
        exitBox = new QMessageBox(QMessageBox::Question, tr("ClipGrab - Exit confirmation"), tr("There is still at least one download in progress.<br />If you exit the program now, all downloads will be canceled and cannot be recovered later.<br />Do you really want to quit ClipGrab now?"), QMessageBox::Yes|QMessageBox::No);
        if (exitBox->exec() == QMessageBox::Yes)
        {
            systemTrayIcon.hide();
            cg->cancelAll();
            event->accept();
        }
        else
        {
            event->ignore();
        }
    }
    else
    {
        systemTrayIcon.hide();
        event->accept();
    }
 }

 void MainWindow::settingsNotifications_toggled(bool)
 {
    if (ui.settingsRadioNotificationsAlways->isChecked())
    {
        cg->settings.setValue("Notifications", "always");
    }
    else if (ui.settingsRadioNotificationsFinish->isChecked())
    {
        cg->settings.setValue("Notifications", "finish");
    }
    else if (ui.settingsRadioNotificationsNever->isChecked())
    {
        cg->settings.setValue("Notifications", "never");
    }
 }

 void MainWindow::handleFinishedConversion(video* finishedVideo)
 {
     if (cg->settings.value("Notifications", "finish") == "always")
     {
         Notifications::showMessage(tr("Download finished"), tr("Downloading and converting “%title” is now finished.").replace("%title", finishedVideo->title()), &systemTrayIcon);
     }
     else if (cg->downloadsRunning() == 0 && cg->settings.value("Notifications", "finish") == "finish")
     {
         Notifications::showMessage(tr("All downloads finished"), tr("ClipGrab has finished downloading and converting all selected videos."), &systemTrayIcon);
     }

     if (cg->settings.value("RemoveFinishedDownloads", false) == true)
     {
         int finishedItemIndex = ui.downloadTree->indexOfTopLevelItem(finishedVideo->treeItem());
         int item = ui.downloadTree->topLevelItemCount() - finishedItemIndex -1;
         ui.downloadTree->takeTopLevelItem(finishedItemIndex);
         emit itemToRemove(item);
     }
 }

void MainWindow::on_settingsRemoveFinishedDownloads_stateChanged(int state)
{
    if (state == Qt::Checked)
    {
        cg->settings.setValue("RemoveFinishedDownloads", true);
    }
    else
    {
        cg->settings.setValue("RemoveFinishedDownloads", false);
    }
}

void MainWindow::on_downloadPause_clicked()
{
    if (ui.downloadTree->currentIndex().row() != -1)
    {
        int item = ui.downloadTree->topLevelItemCount() - ui.downloadTree->currentIndex().row() -1;
        cg->pauseDownload(item);
    }
}

void MainWindow::settingsProxyChanged()
{
    cg->settings.setValue("UseProxy", ui.settingsUseProxy->isChecked());
    cg->settings.setValue("ProxyHost", ui.settingsProxyHost->text());
    cg->settings.setValue("ProxyPort", ui.settingsProxyPort->value());
    cg->settings.setValue("ProxyAuthenticationRequired", ui.settingsProxyAuthenticationRequired->isChecked());
    cg->settings.setValue("ProxyPassword", ui.settingsProxyPassword->text());
    cg->settings.setValue("ProxyUsername", ui.settingsProxyUsername->text());
    cg->settings.setValue("ProxyType", ui.settingsProxyType->currentIndex());
    ui.settingsProxyGroup->setEnabled(ui.settingsUseProxy->isChecked());
    ui.settingsProxyAuthenticationRequired->setEnabled(ui.settingsUseProxy->isChecked());
    if (ui.settingsUseProxy->isChecked())
    {
        if (ui.settingsProxyAuthenticationRequired->isChecked())
        {
            ui.settingsProxyAuthenticationGroup->setEnabled(ui.settingsProxyAuthenticationRequired->isChecked());
        }
    }
    else
    {
        ui.settingsProxyAuthenticationGroup->setEnabled(false);
        ui.settingsProxyAuthenticationRequired->setEnabled(false);
     }
    cg->activateProxySettings();
}

void MainWindow::timerEvent(QTimerEvent *)
{
    QPair<qint64, qint64> downloadProgress = cg->downloadProgress();
    if (downloadProgress.first != 0 && downloadProgress.second != 0)
    {
        #ifdef Q_WS_X11
            systemTrayIcon.setToolTip("<strong style=\"font-size:14px\">" + tr("ClipGrab") + "</strong><br /><span style=\"font-size:13px\">" + QString::number(downloadProgress.first*100/downloadProgress.second) + " %</span><br />" + QString::number((double)downloadProgress.first/1024/1024, QLocale::system().decimalPoint().toAscii(), 1) + tr(" MiB") + "/" + QString::number((double)downloadProgress.second/1024/1024, QLocale::system().decimalPoint().toAscii(), 1) + tr(" MiB"));
        #else
        systemTrayIcon.setToolTip(tr("ClipGrab") + " - " + QString::number(downloadProgress.first*100/downloadProgress.second) + " % - " + QString::number((double)downloadProgress.first/1024/1024, QLocale::system().decimalPoint().toLatin1(), 1) + tr(" MiB") + "/" + QString::number((double)downloadProgress.second/1024/1024, QLocale::system().decimalPoint().toLatin1(), 1) + tr(" KiB"));
        #endif
        setWindowTitle("ClipGrab - " + QString::number(downloadProgress.first*100/downloadProgress.second) + " %");
    }
    else
    {
        #ifdef Q_WS_X11
            systemTrayIcon.setToolTip("<strong style=\"font-size:14px\">" + tr("ClipGrab") + "</strong><br /><span style=\"font-size:13px\">" + tr("Currently no downloads in progress."));
        #endif
        systemTrayIcon.setToolTip(tr("ClipGrab") + tr("Currently no downloads in progress."));
        setWindowTitle(tr("ClipGrab - Download and Convert Online Videos"));
    }
}

void MainWindow::changeEvent(QEvent* event)
{
    if (event->type() == QEvent::WindowStateChange)
    {
        if (isMinimized() && cg->settings.value("MinimizeToTray", false).toBool())
        {
            QTimer::singleShot(500, this, SLOT(hide()));
            event->ignore();
        }
    }
}

void MainWindow::on_settingsMinimizeToTray_stateChanged(int state)
{
    if (state == Qt::Checked)
    {
        cg->settings.setValue("MinimizeToTray", true);
    }
    else
    {
        cg->settings.setValue("MinimizeToTray", false);
    }
}


void MainWindow::on_downloadLineEdit_returnPressed()
{
    if (currentVideo)
    {
        if (!currentVideo->title().isEmpty())
        {
            this->startDownload();
        }
    }
}

void MainWindow::on_label_linkActivated(QString link)
{
    QDesktopServices::openUrl(link);
}

void MainWindow::on_settingsUseMetadata_stateChanged(int state)
{
    if (state == Qt::Checked)
    {
        cg->settings.setValue("UseMetadata", true);
    }
    else
    {
        cg->settings.setValue("UseMetadata", false);
    }
}

void MainWindow::on_searchLineEdit_textChanged(QString /*keywords*/)
{
    bool isTimerActive = searchTimer.isActive();
    searchTimer.stop();
    searchTimer.setSingleShot(true);
    searchTimer.start(1500);

    if (isTimerActive) return;
    this->ui.searchWebEngineView->page()->load(QUrl("qrc:///loading/loading-progress.html"));
}

void MainWindow::updateSearch(QString keywords)
{

    if (this->ui.searchWebEngineView->page()->url().toString() != "qrc:///loading/loading-progress.html")
    {
        this->ui.searchWebEngineView->page()->load(QUrl("qrc:///loading/loading-progress.html"));
    }

     QUrl url;
     if (keywords.isEmpty())
     {
        url = QUrl("https://www.youtube.com");
     }
     else {
        url = QUrl("https://www.youtube.com/results?search_query=" + keywords.replace(QRegExp("[&\\?%\\s]"), "+"));
     }

     if (searchReply)
     {
         QNetworkReply* oldReply = searchReply;
         searchReply = nullptr;
         oldReply->disconnect();
         oldReply->abort();
         oldReply->deleteLater();
     }

     QNetworkRequest request(url);
     http_handler::applyRequestDefaults(&request);
     searchReply = searchNam->get(request);
     connect(searchReply, SIGNAL(finished()), this, SLOT(processSearchReply()));
}

void MainWindow::processSearchReply()
{
    if (!searchReply) return;
    if (searchReply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt() != 200) {
        QUrl url;
        url = QUrl("https://m.youtube.com/results?search_query=" + ui.searchLineEdit->text().replace(QRegExp("[&\\?%\\s]"), "+") + "&persist_app=1&app=m");
        this->ui.searchWebEngineView->page()->load(url);
        return;
    }

    QByteArray searchResult = searchReply->readAll();

    int limit = -1;
    if (searchReply->url().toString() == "https://www.youtube.com") {
        limit = 4;
    }
    QString fn;    
    fn.append("\n(function() {");
    fn.append("\n    let findVideos = function(elements, videos) {");
    fn.append("\n        videos = videos || []");
    fn.append("\n        const keys = Object.getOwnPropertyNames(elements)");
    fn.append("\n        ");
    fn.append("\n        if (Array.isArray(elements)) {");
    fn.append("\n            elements.forEach(el => findVideos(el, videos))");
    fn.append("\n            return videos");
    fn.append("\n        }");
    fn.append("\n");
    fn.append("\n        if (typeof elements !== 'object' || !elements) return videos");
    fn.append("\n");
    fn.append("\n        keys.forEach(key => {");
    fn.append("\n            if (key == 'videoRenderer') {");
    fn.append("\n                console.log(elements[key])");
    fn.append("\n                const v = elements[key]");
    fn.append("\n");
    fn.append("\n                const title = v.title.runs[0].text");
    fn.append("\n                const link =  v.videoId ? 'https://www.youtube.com/watch?v=' + v.videoId : null");
    fn.append("\n                const thumbnail = v.thumbnail.thumbnails[0].url");
    fn.append("\n                const thumbnailOverlay = v.thumbnailOverlays[0].thumbnailOverlayTimeStatusRenderer");
    fn.append("\n                const duration = thumbnailOverlay ? thumbnailOverlay.text.simpleText : ''");
    fn.append("\n                if (title && link && thumbnail) videos.push({title, link, thumbnail, duration})");
    fn.append("\n            } else {");
    fn.append("\n                findVideos(elements[key], videos)");
    fn.append("\n            }");
    fn.append("\n        })");
    fn.append("\n        return videos");
    fn.append("\n    }");
    fn.append("\n");
    fn.append("\n    return window.ytInitialData && findVideos(window.ytInitialData.contents)");
    fn.append("\n})()");
    JSRunner runner;

    QRegExp scriptTagRegExp = QRegExp("window\\[\"ytInitialData\"\\]\\s*=\\s*[^\\n]+");
    if (scriptTagRegExp.indexIn(searchResult) > -1) {
        QString html = "<!doctype html><html><head><meta charset=\"utf-8\"></head><body><script>" + scriptTagRegExp.cap(0) + "</script></body></html>";
        runner.setHtml(html);
    }

    QVariant v = runner.runJavaScript(fn);

    QString searchHtml;
    #ifdef Q_OS_MAC
        QString fontFamily = "Helvetica Neue";
    #else
        QFontDatabase fontDatabase;
        QString font = fontDatabase.systemFont(QFontDatabase::GeneralFont).family();
        QString fontFamily = "'" + font + "',  sans-serif";
    #endif
    searchHtml.append("<!doctype html>");
    searchHtml.append("<html>");
    searchHtml.append("<head>");
    searchHtml.append("<style>body {margin:0;padding:0;width:100%;left:0px;top:0px;font-family: " + fontFamily + ";} img{position:relative;top:-22px;left:-15px} .entry{display:block;position:relative;width:50%;float:left;height:100px;} a{color:#1a1a1a;text-decoration:none;} span.title{display:block;font-weight:bold;font-size:14px;position:absolute;left:140px;top:16px;} span.duration{display:block;font-size:11px;position:absolute;left:140px;top:70px;color:#aaa}  span.thumbnail{width:120px;height:68px;background:#00b2de;display:block;overflow:hidden;position:absolute;left:8px;top:16px;} a:hover{background: #00b2de;color:#fff}</style>");
    searchHtml.append("</head>");
    searchHtml.append("<body>");

    int i=0;
    foreach (QVariant vItem, v.toList())
    {
        QMap<QString, QVariant> item = vItem.toMap();

        QString title = item.value("title").toString();
        QString duration = item.value("duration").toString();

        QString link = item.value("link").toString();
        if (link.isEmpty()) continue;
        if (link.startsWith("/watch")) {
            link.prepend("https://www.youtube.com");
        }

        QString thumbnail = item.value("thumbnail").toString();
        if (thumbnail.isEmpty()) continue;
        if (thumbnail.startsWith("//"))
        {
            thumbnail.prepend("https:");
        }

        searchHtml.append("<a href=\"" + link + "\" class=\"entry\">");
        searchHtml.append("<span class=\"title\">" + title + "</span>");
        searchHtml.append("<span class=\"duration\">" + duration + "</span>");
        searchHtml.append("<span class=\"thumbnail\"><img src=\"" + thumbnail + "\"  /></span>");
        searchHtml.append("</a>");

        i++;
        if (limit > 0 && i == limit) {
            break;
        }
    }
    searchHtml.append("</body>");
    searchHtml.append("</html>");
    if (i == 0)
    {
        QFile loadingFailedHTML(":/loading/loading-failed.html");
        loadingFailedHTML.open(QFile::ReadOnly);
        searchHtml = QString(loadingFailedHTML.readAll()).replace("%NORESULTS%", tr("No results found."));
    }
    this->ui.searchWebEngineView->page()->setHtml(searchHtml);
}

void MainWindow::handleSearchResultClicked(const QUrl & url)
{
    this->ui.downloadLineEdit->setText(url.toString());
    this->ui.mainTab->setCurrentIndex(1);
}

void MainWindow::handleSearchResultIntercepted(const QUrl & url)
{
    updateSearch(this->ui.searchLineEdit->text());
    this->ui.downloadLineEdit->setText(url.toString());
    this->ui.mainTab->setCurrentIndex(1);
}

void MainWindow::on_downloadComboFormat_currentIndexChanged(int index)
{
    cg->settings.setValue("LastFormat", index);
}

void MainWindow::on_mainTab_currentChanged(int index)
{
    cg->settings.setValue("MainTab", index);
}

void MainWindow::on_downloadTree_doubleClicked(QModelIndex modelIndex)
{
    int selectedVideo = ui.downloadTree->topLevelItemCount()-1 -modelIndex.row();
    cg->openDownload(selectedVideo);
}

void MainWindow::on_settingsLanguage_currentIndexChanged(int index)
{
    cg->settings.setValue("Language", cg->languages.at(index).code);

}

void MainWindow::on_buttonDonate_clicked()
{
    QDesktopServices::openUrl(QUrl("https://www.paypal.com/cgi-bin/webscr?cmd=_s-xclick&hosted_button_id=AS6TDMR667GJL"));
}

void MainWindow::on_settingsUseWebM_toggled(bool checked)
{
    cg->settings.setValue("UseWebM", checked);
}

void MainWindow::on_settingsIgnoreSSLErrors_toggled(bool checked)
{
    cg->settings.setValue("IgnoreSSLErrors", checked);
}

void MainWindow::on_downloadTree_customContextMenuRequested(const QPoint &pos)
{
    int selectedVideo = ui.downloadTree->topLevelItemCount()-1 - ui.downloadTree->indexAt(pos).row();
    if (selectedVideo == -1 || ui.downloadTree->indexAt(pos).row() == -1)
    {
        return;
    }

    QMenu contextMenu;
    QAction* openDownload = contextMenu.addAction(tr("&Open downloaded file"));
    QAction* openFolder = contextMenu.addAction(tr("Open &target folder"));
    contextMenu.addSeparator();
    QAction* pauseDownload = contextMenu.addAction(tr("&Pause download"));
    QAction* restartDownload = contextMenu.addAction(tr("&Restart download"));
    QAction* cancelDownload = contextMenu.addAction(tr("&Cancel download"));
    contextMenu.addSeparator();
    QAction* copyLink = contextMenu.addAction(tr("Copy &video link"));
    QAction* openLink = contextMenu.addAction(tr("Open video link in &browser"));


    if (cg->isDownloadPaused(selectedVideo))
    {
        pauseDownload->setText(tr("Resume download"));
    }

    if (cg->isDownloadFinished(selectedVideo))
    {
        contextMenu.removeAction(pauseDownload);
        contextMenu.removeAction(restartDownload);
        contextMenu.removeAction(cancelDownload);
        #ifdef Q_OS_MAC
            openFolder->setText(tr("Show in &Finder"));
        #endif
    }
    else {
        contextMenu.removeAction(openDownload);
    }


    QAction* selectedAction = contextMenu.exec(ui.downloadTree->mapToGlobal(pos));
    if (selectedAction == restartDownload)
    {
        cg->restartDownload(selectedVideo);
    }
    else if (selectedAction == cancelDownload)
    {
        cg->cancelDownload(selectedVideo);
    }
    else if (selectedAction == pauseDownload)
    {
        cg->pauseDownload(selectedVideo);
    }
    else if (selectedAction == openFolder)
    {
        cg->openTargetFolder(selectedVideo);
    }
    else if (selectedAction == openDownload)
    {
        cg->openDownload(selectedVideo);
    }
    else if (selectedAction == openLink)
    {
        QString link = cg->getDownloadOriginalUrl(selectedVideo);
        QDesktopServices::openUrl(QUrl(link));
    }
    else if (selectedAction == copyLink)
    {
        QApplication::clipboard()->setText(cg->getDownloadOriginalUrl(selectedVideo));
    }

    contextMenu.deleteLater();
}

void MainWindow::dragEnterEvent(QDragEnterEvent *event)
{
    if (event->mimeData()->hasFormat("text/uri-list"))
    {
        event->acceptProposedAction();
    }
}

void MainWindow::dropEvent(QDropEvent *event)
{
    ui.downloadLineEdit->setText(event->mimeData()->text());
    ui.mainTab->setCurrentIndex(1);
}

void MainWindow::on_settingsRememberLogins_toggled(bool checked)
{
    cg->settings.setValue("rememberLogins", checked);
    cg->settings.setValue("youtubeRememberLogin", checked);
    cg->settings.setValue("facebookRememberLogin", checked);
    cg->settings.setValue("vimeoRememberLogin", checked);
    if (!checked)
    {
        cg->settings.remove("youtubeCookies");
        cg->settings.remove("facebookCookies");
        cg->settings.remove("vimeoCookies");
    }
}

void MainWindow::on_settingsRememberVideoQuality_toggled(bool checked)
{
    cg->settings.setValue("rememberVideoQuality", checked);
}

void MainWindow::on_downloadComboQuality_currentIndexChanged(int index)
{
    if (this->updatingComboQuality || !cg->settings.value("rememberVideoQuality", true).toBool() || !this->currentVideo)
    {
        return;
    }

    cg->settings.setValue("rememberedVideoQuality", ui.downloadComboQuality->itemData(index, Qt::UserRole).toInt());
}

void MainWindow::searchTimerTimeout()
{
    updateSearch(this->ui.searchLineEdit->text());
}
