(function() {
  var CSON, CompositeDisposable, Disposable, Emitter, File, ScopedPropertyStore, Snippet, SnippetExpansion, _, async, fs, path, ref,
    slice = [].slice;

  path = require('path');

  ref = require('atom'), Emitter = ref.Emitter, Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable, File = ref.File;

  _ = require('underscore-plus');

  async = require('async');

  CSON = require('season');

  fs = require('fs-plus');

  ScopedPropertyStore = require('scoped-property-store');

  Snippet = require('./snippet');

  SnippetExpansion = require('./snippet-expansion');

  module.exports = {
    loaded: false,
    activate: function() {
      var snippets;
      this.userSnippetsPath = null;
      this.snippetIdCounter = 0;
      this.parsedSnippetsById = new Map;
      this.scopedPropertyStore = new ScopedPropertyStore;
      this.subscriptions = new CompositeDisposable;
      this.subscriptions.add(atom.workspace.addOpener((function(_this) {
        return function(uri) {
          if (uri === 'atom://.atom/snippets') {
            return atom.workspace.openTextFile(_this.getUserSnippetsPath());
          }
        };
      })(this)));
      this.loadAll();
      this.watchUserSnippets((function(_this) {
        return function(watchDisposable) {
          return _this.subscriptions.add(watchDisposable);
        };
      })(this));
      snippets = this;
      this.subscriptions.add(atom.commands.add('atom-text-editor', {
        'snippets:expand': function(event) {
          var editor;
          editor = this.getModel();
          if (snippets.snippetToExpandUnderCursor(editor)) {
            snippets.clearExpansions(editor);
            return snippets.expandSnippetsUnderCursors(editor);
          } else {
            return event.abortKeyBinding();
          }
        },
        'snippets:next-tab-stop': function(event) {
          var editor;
          editor = this.getModel();
          if (!snippets.goToNextTabStop(editor)) {
            return event.abortKeyBinding();
          }
        },
        'snippets:previous-tab-stop': function(event) {
          var editor;
          editor = this.getModel();
          if (!snippets.goToPreviousTabStop(editor)) {
            return event.abortKeyBinding();
          }
        },
        'snippets:available': function(event) {
          var SnippetsAvailable, editor;
          editor = this.getModel();
          SnippetsAvailable = require('./snippets-available');
          if (snippets.availableSnippetsView == null) {
            snippets.availableSnippetsView = new SnippetsAvailable(snippets);
          }
          return snippets.availableSnippetsView.toggle(editor);
        }
      }));
      return this.subscriptions.add(atom.workspace.observeTextEditors((function(_this) {
        return function(editor) {
          return _this.clearExpansions(editor);
        };
      })(this)));
    },
    deactivate: function() {
      var ref1;
      if ((ref1 = this.emitter) != null) {
        ref1.dispose();
      }
      this.emitter = null;
      this.editorSnippetExpansions = null;
      return atom.config.transact((function(_this) {
        return function() {
          return _this.subscriptions.dispose();
        };
      })(this));
    },
    getUserSnippetsPath: function() {
      if (this.userSnippetsPath != null) {
        return this.userSnippetsPath;
      }
      this.userSnippetsPath = CSON.resolve(path.join(atom.getConfigDirPath(), 'snippets'));
      if (this.userSnippetsPath == null) {
        this.userSnippetsPath = path.join(atom.getConfigDirPath(), 'snippets.cson');
      }
      return this.userSnippetsPath;
    },
    loadAll: function(callback) {
      return this.loadBundledSnippets((function(_this) {
        return function(bundledSnippets) {
          return _this.loadPackageSnippets(function(packageSnippets) {
            return _this.loadUserSnippets(function(userSnippets) {
              atom.config.transact(function() {
                var filepath, i, len, ref1, results1, snippetSet, snippetsBySelector;
                ref1 = [bundledSnippets, packageSnippets, userSnippets];
                results1 = [];
                for (i = 0, len = ref1.length; i < len; i++) {
                  snippetSet = ref1[i];
                  results1.push((function() {
                    var results2;
                    results2 = [];
                    for (filepath in snippetSet) {
                      snippetsBySelector = snippetSet[filepath];
                      results2.push(this.add(filepath, snippetsBySelector));
                    }
                    return results2;
                  }).call(_this));
                }
                return results1;
              });
              return _this.doneLoading();
            });
          });
        };
      })(this));
    },
    loadBundledSnippets: function(callback) {
      var bundledSnippetsPath;
      bundledSnippetsPath = CSON.resolve(path.join(__dirname, 'snippets'));
      return this.loadSnippetsFile(bundledSnippetsPath, function(snippets) {
        var snippetsByPath;
        snippetsByPath = {};
        snippetsByPath[bundledSnippetsPath] = snippets;
        return callback(snippetsByPath);
      });
    },
    loadUserSnippets: function(callback) {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return fs.stat(userSnippetsPath, (function(_this) {
        return function(error, stat) {
          if (stat != null ? stat.isFile() : void 0) {
            return _this.loadSnippetsFile(userSnippetsPath, function(snippets) {
              var result;
              result = {};
              result[userSnippetsPath] = snippets;
              return callback(result);
            });
          } else {
            return callback({});
          }
        };
      })(this));
    },
    watchUserSnippets: function(callback) {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return fs.stat(userSnippetsPath, (function(_this) {
        return function(error, stat) {
          var e, message, userSnippetsFile, userSnippetsFileDisposable;
          if (stat != null ? stat.isFile() : void 0) {
            userSnippetsFileDisposable = new CompositeDisposable();
            userSnippetsFile = new File(userSnippetsPath);
            try {
              userSnippetsFileDisposable.add(userSnippetsFile.onDidChange(function() {
                return _this.handleUserSnippetsDidChange();
              }));
              userSnippetsFileDisposable.add(userSnippetsFile.onDidDelete(function() {
                return _this.handleUserSnippetsDidChange();
              }));
              userSnippetsFileDisposable.add(userSnippetsFile.onDidRename(function() {
                return _this.handleUserSnippetsDidChange();
              }));
            } catch (error1) {
              e = error1;
              message = "Unable to watch path: `snippets.cson`. Make sure you have permissions\nto the `~/.atom` directory and `" + userSnippetsPath + "`.\n\nOn linux there are currently problems with watch sizes. See\n[this document][watches] for more info.\n[watches]:https://github.com/atom/atom/blob/master/docs/build-instructions/linux.md#typeerror-unable-to-watch-path";
              atom.notifications.addError(message, {
                dismissable: true
              });
            }
            return callback(userSnippetsFileDisposable);
          } else {
            return callback(new Disposable(function() {}));
          }
        };
      })(this));
    },
    handleUserSnippetsDidChange: function() {
      var userSnippetsPath;
      userSnippetsPath = this.getUserSnippetsPath();
      return atom.config.transact((function(_this) {
        return function() {
          _this.clearSnippetsForPath(userSnippetsPath);
          return _this.loadSnippetsFile(userSnippetsPath, function(result) {
            return _this.add(userSnippetsPath, result);
          });
        };
      })(this));
    },
    loadPackageSnippets: function(callback) {
      var pack, packages, snippetsDirPaths;
      packages = atom.packages.getLoadedPackages();
      snippetsDirPaths = ((function() {
        var i, len, results1;
        results1 = [];
        for (i = 0, len = packages.length; i < len; i++) {
          pack = packages[i];
          results1.push(path.join(pack.path, 'snippets'));
        }
        return results1;
      })()).sort(function(a, b) {
        if (/\/app\.asar\/node_modules\//.test(a)) {
          return -1;
        } else {
          return 1;
        }
      });
      return async.map(snippetsDirPaths, this.loadSnippetsDirectory.bind(this), function(error, results) {
        return callback(_.extend.apply(_, [{}].concat(slice.call(results))));
      });
    },
    doneLoading: function() {
      this.loaded = true;
      return this.getEmitter().emit('did-load-snippets');
    },
    onDidLoadSnippets: function(callback) {
      return this.getEmitter().on('did-load-snippets', callback);
    },
    getEmitter: function() {
      return this.emitter != null ? this.emitter : this.emitter = new Emitter;
    },
    loadSnippetsDirectory: function(snippetsDirPath, callback) {
      return fs.isDirectory(snippetsDirPath, (function(_this) {
        return function(isDirectory) {
          if (!isDirectory) {
            return callback(null, {});
          }
          return fs.readdir(snippetsDirPath, function(error, entries) {
            if (error) {
              console.warn("Error reading snippets directory " + snippetsDirPath, error);
              return callback(null, {});
            }
            return async.map(entries, function(entry, done) {
              var filePath;
              filePath = path.join(snippetsDirPath, entry);
              return _this.loadSnippetsFile(filePath, function(snippets) {
                return done(null, {
                  filePath: filePath,
                  snippets: snippets
                });
              });
            }, function(error, results) {
              var filePath, i, len, ref1, snippets, snippetsByPath;
              snippetsByPath = {};
              for (i = 0, len = results.length; i < len; i++) {
                ref1 = results[i], filePath = ref1.filePath, snippets = ref1.snippets;
                snippetsByPath[filePath] = snippets;
              }
              return callback(null, snippetsByPath);
            });
          });
        };
      })(this));
    },
    loadSnippetsFile: function(filePath, callback) {
      if (!CSON.isObjectPath(filePath)) {
        return callback({});
      }
      return CSON.readFile(filePath, function(error, object) {
        var ref1, ref2;
        if (object == null) {
          object = {};
        }
        if (error != null) {
          console.warn("Error reading snippets file '" + filePath + "': " + ((ref1 = error.stack) != null ? ref1 : error));
          if ((ref2 = atom.notifications) != null) {
            ref2.addError("Failed to load snippets from '" + filePath + "'", {
              detail: error.message,
              dismissable: true
            });
          }
        }
        return callback(object);
      });
    },
    add: function(filePath, snippetsBySelector) {
      var attributes, body, name, prefix, selector, snippetsByName, unparsedSnippetsByPrefix;
      for (selector in snippetsBySelector) {
        snippetsByName = snippetsBySelector[selector];
        unparsedSnippetsByPrefix = {};
        for (name in snippetsByName) {
          attributes = snippetsByName[name];
          prefix = attributes.prefix, body = attributes.body;
          attributes.name = name;
          attributes.id = this.snippetIdCounter++;
          if (typeof body === 'string') {
            unparsedSnippetsByPrefix[prefix] = attributes;
          } else if (body == null) {
            unparsedSnippetsByPrefix[prefix] = null;
          }
        }
        this.storeUnparsedSnippets(unparsedSnippetsByPrefix, filePath, selector);
      }
    },
    getScopeChain: function(object) {
      var scopesArray;
      scopesArray = object != null ? typeof object.getScopesArray === "function" ? object.getScopesArray() : void 0 : void 0;
      if (scopesArray == null) {
        scopesArray = object;
      }
      return scopesArray.map(function(scope) {
        if (scope[0] !== '.') {
          scope = "." + scope;
        }
        return scope;
      }).join(' ');
    },
    storeUnparsedSnippets: function(value, path, selector) {
      var unparsedSnippets;
      unparsedSnippets = {};
      unparsedSnippets[selector] = {
        "snippets": value
      };
      return this.scopedPropertyStore.addProperties(path, unparsedSnippets, {
        priority: this.priorityForSource(path)
      });
    },
    clearSnippetsForPath: function(path) {
      var attributes, prefix, ref1, results1, scopeSelector;
      results1 = [];
      for (scopeSelector in this.scopedPropertyStore.propertiesForSource(path)) {
        ref1 = this.scopedPropertyStore.propertiesForSourceAndSelector(path, scopeSelector);
        for (prefix in ref1) {
          attributes = ref1[prefix];
          this.parsedSnippetsById["delete"](attributes.id);
        }
        results1.push(this.scopedPropertyStore.removePropertiesForSourceAndSelector(path, scopeSelector));
      }
      return results1;
    },
    parsedSnippetsForScopes: function(scopeDescriptor) {
      var attributes, body, bodyTree, description, descriptionMoreURL, id, leftLabel, leftLabelHTML, name, prefix, rightLabelHTML, snippet, snippets, unparsedSnippetsByPrefix;
      unparsedSnippetsByPrefix = this.scopedPropertyStore.getPropertyValue(this.getScopeChain(scopeDescriptor), "snippets");
      if (unparsedSnippetsByPrefix == null) {
        unparsedSnippetsByPrefix = {};
      }
      snippets = {};
      for (prefix in unparsedSnippetsByPrefix) {
        attributes = unparsedSnippetsByPrefix[prefix];
        if (typeof (attributes != null ? attributes.body : void 0) !== 'string') {
          continue;
        }
        id = attributes.id, name = attributes.name, body = attributes.body, bodyTree = attributes.bodyTree, description = attributes.description, descriptionMoreURL = attributes.descriptionMoreURL, rightLabelHTML = attributes.rightLabelHTML, leftLabel = attributes.leftLabel, leftLabelHTML = attributes.leftLabelHTML;
        if (!this.parsedSnippetsById.has(id)) {
          if (bodyTree == null) {
            bodyTree = this.getBodyParser().parse(body);
          }
          snippet = new Snippet({
            id: id,
            name: name,
            prefix: prefix,
            bodyTree: bodyTree,
            description: description,
            descriptionMoreURL: descriptionMoreURL,
            rightLabelHTML: rightLabelHTML,
            leftLabel: leftLabel,
            leftLabelHTML: leftLabelHTML,
            bodyText: body
          });
          this.parsedSnippetsById.set(id, snippet);
        }
        snippets[prefix] = this.parsedSnippetsById.get(id);
      }
      return snippets;
    },
    priorityForSource: function(source) {
      if (source === this.getUserSnippetsPath()) {
        return 1000;
      } else {
        return 0;
      }
    },
    getBodyParser: function() {
      return this.bodyParser != null ? this.bodyParser : this.bodyParser = require('./snippet-body-parser');
    },
    getPrefixText: function(snippets, editor) {
      var cursor, cursorSnippetPrefix, cursorWordPrefix, i, len, position, prefixStart, ref1, ref2, snippetPrefix, wordPrefix, wordRegex, wordStart;
      wordRegex = this.wordRegexForSnippets(snippets);
      ref1 = [], snippetPrefix = ref1[0], wordPrefix = ref1[1];
      ref2 = editor.getCursors();
      for (i = 0, len = ref2.length; i < len; i++) {
        cursor = ref2[i];
        position = cursor.getBufferPosition();
        prefixStart = cursor.getBeginningOfCurrentWordBufferPosition({
          wordRegex: wordRegex
        });
        cursorSnippetPrefix = editor.getTextInRange([prefixStart, position]);
        if ((snippetPrefix != null) && cursorSnippetPrefix !== snippetPrefix) {
          return null;
        }
        snippetPrefix = cursorSnippetPrefix;
        wordStart = cursor.getBeginningOfCurrentWordBufferPosition();
        cursorWordPrefix = editor.getTextInRange([wordStart, position]);
        if ((wordPrefix != null) && cursorWordPrefix !== wordPrefix) {
          return null;
        }
        wordPrefix = cursorWordPrefix;
      }
      return {
        snippetPrefix: snippetPrefix,
        wordPrefix: wordPrefix
      };
    },
    wordRegexForSnippets: function(snippets) {
      var character, i, len, prefix, prefixCharacters, prefixes;
      prefixes = {};
      for (prefix in snippets) {
        for (i = 0, len = prefix.length; i < len; i++) {
          character = prefix[i];
          prefixes[character] = true;
        }
      }
      prefixCharacters = Object.keys(prefixes).join('');
      return new RegExp("[" + (_.escapeRegExp(prefixCharacters)) + "]+");
    },
    snippetForPrefix: function(snippets, prefix, wordPrefix) {
      var longestPrefixMatch, snippet, snippetPrefix;
      longestPrefixMatch = null;
      for (snippetPrefix in snippets) {
        snippet = snippets[snippetPrefix];
        if (_.endsWith(prefix, snippetPrefix) && wordPrefix.length <= snippetPrefix.length) {
          if ((longestPrefixMatch == null) || snippetPrefix.length > longestPrefixMatch.prefix.length) {
            longestPrefixMatch = snippet;
          }
        }
      }
      return longestPrefixMatch;
    },
    getSnippets: function(editor) {
      return this.parsedSnippetsForScopes(editor.getLastCursor().getScopeDescriptor());
    },
    snippetToExpandUnderCursor: function(editor) {
      var prefixData, snippets;
      if (!editor.getLastSelection().isEmpty()) {
        return false;
      }
      snippets = this.getSnippets(editor);
      if (_.isEmpty(snippets)) {
        return false;
      }
      if (prefixData = this.getPrefixText(snippets, editor)) {
        return this.snippetForPrefix(snippets, prefixData.snippetPrefix, prefixData.wordPrefix);
      }
    },
    expandSnippetsUnderCursors: function(editor) {
      var snippet;
      if (!(snippet = this.snippetToExpandUnderCursor(editor))) {
        return false;
      }
      editor.transact((function(_this) {
        return function() {
          var cursor, cursorPosition, cursors, i, len, results1, startPoint;
          cursors = editor.getCursors();
          results1 = [];
          for (i = 0, len = cursors.length; i < len; i++) {
            cursor = cursors[i];
            cursorPosition = cursor.getBufferPosition();
            startPoint = cursorPosition.translate([0, -snippet.prefix.length], [0, 0]);
            cursor.selection.setBufferRange([startPoint, cursorPosition]);
            results1.push(_this.insert(snippet, editor, cursor));
          }
          return results1;
        };
      })(this));
      return true;
    },
    goToNextTabStop: function(editor) {
      var expansion, i, len, nextTabStopVisited, ref1;
      nextTabStopVisited = false;
      ref1 = this.getExpansions(editor);
      for (i = 0, len = ref1.length; i < len; i++) {
        expansion = ref1[i];
        if (expansion != null ? expansion.goToNextTabStop() : void 0) {
          nextTabStopVisited = true;
        }
      }
      return nextTabStopVisited;
    },
    goToPreviousTabStop: function(editor) {
      var expansion, i, len, previousTabStopVisited, ref1;
      previousTabStopVisited = false;
      ref1 = this.getExpansions(editor);
      for (i = 0, len = ref1.length; i < len; i++) {
        expansion = ref1[i];
        if (expansion != null ? expansion.goToPreviousTabStop() : void 0) {
          previousTabStopVisited = true;
        }
      }
      return previousTabStopVisited;
    },
    getExpansions: function(editor) {
      var ref1, ref2;
      return (ref1 = (ref2 = this.editorSnippetExpansions) != null ? ref2.get(editor) : void 0) != null ? ref1 : [];
    },
    clearExpansions: function(editor) {
      if (this.editorSnippetExpansions == null) {
        this.editorSnippetExpansions = new WeakMap();
      }
      return this.editorSnippetExpansions.set(editor, []);
    },
    addExpansion: function(editor, snippetExpansion) {
      return this.getExpansions(editor).push(snippetExpansion);
    },
    insert: function(snippet, editor, cursor) {
      var bodyTree;
      if (editor == null) {
        editor = atom.workspace.getActiveTextEditor();
      }
      if (cursor == null) {
        cursor = editor.getLastCursor();
      }
      if (typeof snippet === 'string') {
        bodyTree = this.getBodyParser().parse(snippet);
        snippet = new Snippet({
          name: '__anonymous',
          prefix: '',
          bodyTree: bodyTree,
          bodyText: snippet
        });
      }
      return new SnippetExpansion(snippet, editor, cursor, this);
    },
    getUnparsedSnippets: function() {
      return _.deepClone(this.scopedPropertyStore.propertySets);
    },
    provideSnippets: function() {
      return {
        bundledSnippetsLoaded: (function(_this) {
          return function() {
            return _this.loaded;
          };
        })(this),
        insertSnippet: this.insert.bind(this),
        snippetsForScopes: this.parsedSnippetsForScopes.bind(this),
        getUnparsedSnippets: this.getUnparsedSnippets.bind(this)
      };
    }
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
