(function() {
  var AtomIoClient, fs, glob, path, remote, request;

  fs = require('fs-plus');

  path = require('path');

  remote = require('electron').remote;

  glob = require('glob');

  request = require('request');

  module.exports = AtomIoClient = (function() {
    function AtomIoClient(packageManager, baseURL) {
      this.packageManager = packageManager;
      this.baseURL = baseURL;
      if (this.baseURL == null) {
        this.baseURL = 'https://atom.io/api/';
      }
      this.expiry = 1000 * 60 * 60 * 12;
      this.createAvatarCache();
      this.expireAvatarCache();
    }

    AtomIoClient.prototype.avatar = function(login, callback) {
      return this.cachedAvatar(login, (function(_this) {
        return function(err, cached) {
          var stale;
          if (cached) {
            stale = Date.now() - parseInt(cached.split('-').pop()) > _this.expiry;
          }
          if (cached && (!stale || !_this.online())) {
            return callback(null, cached);
          } else {
            return _this.fetchAndCacheAvatar(login, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype["package"] = function(name, callback) {
      var packagePath;
      packagePath = "packages/" + name;
      return this.fetchFromCache(packagePath, {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.request(packagePath, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.featuredPackages = function(callback) {
      return this.fetchFromCache('packages/featured', {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.getFeatured(false, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.featuredThemes = function(callback) {
      return this.fetchFromCache('themes/featured', {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.getFeatured(true, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.getFeatured = function(loadThemes, callback) {
      return this.packageManager.getFeatured(loadThemes).then((function(_this) {
        return function(packages) {
          var cached, key;
          key = loadThemes ? 'themes/featured' : 'packages/featured';
          cached = {
            data: packages,
            createdOn: Date.now()
          };
          localStorage.setItem(_this.cacheKeyForPath(key), JSON.stringify(cached));
          return callback(null, packages);
        };
      })(this))["catch"](function(error) {
        return callback(error, null);
      });
    };

    AtomIoClient.prototype.request = function(path, callback) {
      var options;
      options = {
        url: "" + this.baseURL + path,
        headers: {
          'User-Agent': navigator.userAgent
        }
      };
      return request(options, (function(_this) {
        return function(err, res, body) {
          var cached, data, error;
          try {
            data = JSON.parse(body);
          } catch (error1) {
            error = error1;
            return callback(error);
          }
          delete data.versions;
          cached = {
            data: data,
            createdOn: Date.now()
          };
          localStorage.setItem(_this.cacheKeyForPath(path), JSON.stringify(cached));
          return callback(err, cached.data);
        };
      })(this));
    };

    AtomIoClient.prototype.cacheKeyForPath = function(path) {
      return "settings-view:" + path;
    };

    AtomIoClient.prototype.online = function() {
      return navigator.onLine;
    };

    AtomIoClient.prototype.fetchFromCache = function(packagePath, options, callback) {
      var cached;
      if (!callback) {
        callback = options;
        options = {};
      }
      if (!options.force) {
        options.force = !this.online();
      }
      cached = localStorage.getItem(this.cacheKeyForPath(packagePath));
      cached = cached ? JSON.parse(cached) : void 0;
      if ((cached != null) && (!this.online() || options.force || (Date.now() - cached.createdOn < this.expiry))) {
        if (cached == null) {
          cached = {
            data: {}
          };
        }
        return callback(null, cached.data);
      } else if ((cached == null) && !this.online()) {
        return callback(null, {});
      } else {
        return callback(null, null);
      }
    };

    AtomIoClient.prototype.createAvatarCache = function() {
      return fs.makeTree(this.getCachePath());
    };

    AtomIoClient.prototype.avatarPath = function(login) {
      return path.join(this.getCachePath(), login + "-" + (Date.now()));
    };

    AtomIoClient.prototype.cachedAvatar = function(login, callback) {
      return glob(this.avatarGlob(login), (function(_this) {
        return function(err, files) {
          var createdOn, filename, i, imagePath, len, ref;
          if (err) {
            return callback(err);
          }
          files.sort().reverse();
          for (i = 0, len = files.length; i < len; i++) {
            imagePath = files[i];
            filename = path.basename(imagePath);
            ref = filename.split('-'), createdOn = ref[ref.length - 1];
            if (Date.now() - parseInt(createdOn) < _this.expiry) {
              return callback(null, imagePath);
            }
          }
          return callback(null, null);
        };
      })(this));
    };

    AtomIoClient.prototype.avatarGlob = function(login) {
      return path.join(this.getCachePath(), login + "-*([0-9])");
    };

    AtomIoClient.prototype.fetchAndCacheAvatar = function(login, callback) {
      var imagePath, requestObject;
      if (!this.online()) {
        return callback(null, null);
      } else {
        imagePath = this.avatarPath(login);
        requestObject = {
          url: "https://avatars.githubusercontent.com/" + login,
          headers: {
            'User-Agent': navigator.userAgent
          }
        };
        return request.head(requestObject, function(error, response, body) {
          var writeStream;
          if ((error != null) || response.statusCode !== 200 || !response.headers['content-type'].startsWith('image/')) {
            return callback(error);
          } else {
            writeStream = fs.createWriteStream(imagePath);
            writeStream.on('finish', function() {
              return callback(null, imagePath);
            });
            writeStream.on('error', function(error) {
              writeStream.close();
              try {
                if (fs.existsSync(imagePath)) {
                  fs.unlinkSync(imagePath);
                }
              } catch (error1) {}
              return callback(error);
            });
            return request(requestObject).pipe(writeStream);
          }
        });
      }
    };

    AtomIoClient.prototype.expireAvatarCache = function() {
      var deleteAvatar;
      deleteAvatar = (function(_this) {
        return function(child) {
          var avatarPath;
          avatarPath = path.join(_this.getCachePath(), child);
          return fs.unlink(avatarPath, function(error) {
            if (error && error.code !== 'ENOENT') {
              return console.warn("Error deleting avatar (" + error.code + "): " + avatarPath);
            }
          });
        };
      })(this);
      return fs.readdir(this.getCachePath(), function(error, _files) {
        var children, filename, files, i, key, len, parts, results, stamp;
        if (_files == null) {
          _files = [];
        }
        files = {};
        for (i = 0, len = _files.length; i < len; i++) {
          filename = _files[i];
          parts = filename.split('-');
          stamp = parts.pop();
          key = parts.join('-');
          if (files[key] == null) {
            files[key] = [];
          }
          files[key].push(key + "-" + stamp);
        }
        results = [];
        for (key in files) {
          children = files[key];
          children.sort();
          children.pop();
          results.push(children.forEach(deleteAvatar));
        }
        return results;
      });
    };

    AtomIoClient.prototype.getCachePath = function() {
      return this.cachePath != null ? this.cachePath : this.cachePath = path.join(remote.app.getPath('userData'), 'Cache', 'settings-view');
    };

    return AtomIoClient;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
