'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _listSelection = require('./list-selection');

var _listSelection2 = _interopRequireDefault(_listSelection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let CompositeListSelection = class CompositeListSelection {
  constructor(_ref) {
    let listsByKey = _ref.listsByKey,
        idForItem = _ref.idForItem;

    this.keysBySelection = new Map();
    this.selections = [];
    this.idForItem = idForItem || (item => item);
    this.resolveNextUpdatePromise = () => {};

    for (const key in listsByKey) {
      const selection = new _listSelection2.default({ items: listsByKey[key] });
      this.keysBySelection.set(selection, key);
      this.selections.push(selection);
    }

    this.selectFirstNonEmptyList();
  }

  updateLists(listsByKey) {
    let wasChanged = false;

    const keys = Object.keys(listsByKey);
    for (let i = 0; i < keys.length; i++) {
      const newItems = listsByKey[keys[i]];
      const selection = this.selections[i];

      const oldItems = selection.getItems();
      if (!wasChanged) {
        if (newItems.length !== oldItems.length) {
          wasChanged = true;
        } else {
          for (let j = 0; j < oldItems.length; j++) {
            if (oldItems[j] !== newItems[j]) {
              wasChanged = true;
              break;
            }
          }
        }
      }

      const oldHeadItem = selection.getHeadItem();
      selection.setItems(newItems);
      let newHeadItem = null;
      if (oldHeadItem) {
        newHeadItem = newItems.find(item => this.idForItem(item) === this.idForItem(oldHeadItem));
      }
      if (newHeadItem) {
        selection.selectItem(newHeadItem);
      }
    }
    if (this.getActiveSelection().getItems().length === 0) {
      this.activateNextSelection() || this.activatePreviousSelection();
    }

    if (wasChanged) {
      this.resolveNextUpdatePromise();
    }
  }

  getNextUpdatePromise() {
    return new Promise((resolve, reject) => {
      this.resolveNextUpdatePromise = resolve;
    });
  }

  selectFirstNonEmptyList() {
    this.activeSelectionIndex = 0;
    for (let i = 0; i < this.selections.length; i++) {
      if (this.selections[i].getItems().length) {
        this.activeSelectionIndex = i;
        break;
      }
    }
  }

  getActiveListKey() {
    return this.keysBySelection.get(this.getActiveSelection());
  }

  getSelectedItems() {
    return this.getActiveSelection().getSelectedItems();
  }

  getHeadItem() {
    return this.getActiveSelection().getHeadItem();
  }

  getActiveSelection() {
    return this.selections[this.activeSelectionIndex];
  }

  activateSelection(selection) {
    const index = this.selections.indexOf(selection);
    if (index === -1) {
      throw new Error('Selection not found');
    }
    this.activeSelectionIndex = index;
  }

  activateNextSelection() {
    for (let i = this.activeSelectionIndex + 1; i < this.selections.length; i++) {
      if (this.selections[i].getItems().length > 0) {
        this.activeSelectionIndex = i;
        return true;
      }
    }
    return false;
  }

  activatePreviousSelection() {
    for (let i = this.activeSelectionIndex - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        this.activeSelectionIndex = i;
        return true;
      }
    }
    return false;
  }

  activateLastSelection() {
    for (let i = this.selections.length - 1; i >= 0; i--) {
      if (this.selections[i].getItems().length > 0) {
        this.activeSelectionIndex = i;
        return true;
      }
    }
    return false;
  }

  selectItem(item) {
    let preserveTail = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

    const selection = this.selectionForItem(item);
    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }
    if (!preserveTail) {
      this.activateSelection(selection);
    }
    if (selection === this.getActiveSelection()) {
      selection.selectItem(item, preserveTail);
    }
  }

  addOrSubtractSelection(item) {
    const selection = this.selectionForItem(item);
    if (!selection) {
      throw new Error(`No item found: ${item}`);
    }

    if (selection === this.getActiveSelection()) {
      selection.addOrSubtractSelection(item);
    } else {
      this.activateSelection(selection);
      selection.selectItem(item);
    }
  }

  selectAllItems() {
    this.getActiveSelection().selectAllItems();
  }

  selectFirstItem(preserveTail) {
    this.getActiveSelection().selectFirstItem(preserveTail);
  }

  selectLastItem(preserveTail) {
    this.getActiveSelection().selectLastItem(preserveTail);
  }

  coalesce() {
    this.getActiveSelection().coalesce();
  }

  selectionForItem(item) {
    return this.selections.find(selection => selection.getItems().indexOf(item) > -1);
  }

  listKeyForItem(item) {
    return this.keysBySelection.get(this.selectionForItem(item));
  }

  selectNextItem() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (!preserveTail && this.getActiveSelection().getHeadItem() === this.getActiveSelection().getLastItem()) {
      if (this.activateNextSelection()) {
        this.getActiveSelection().selectFirstItem();
        return true;
      } else {
        this.getActiveSelection().selectLastItem();
        return false;
      }
    } else {
      this.getActiveSelection().selectNextItem(preserveTail);
      return true;
    }
  }

  selectPreviousItem() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (!preserveTail && this.getActiveSelection().getHeadItem() === this.getActiveSelection().getItems()[0]) {
      if (this.activatePreviousSelection()) {
        this.getActiveSelection().selectLastItem();
        return true;
      } else {
        this.getActiveSelection().selectFirstItem();
        return false;
      }
    } else {
      this.getActiveSelection().selectPreviousItem(preserveTail);
      return true;
    }
  }

  findItem(predicate) {
    for (let i = 0; i < this.selections.length; i++) {
      const selection = this.selections[i];
      const key = this.keysBySelection.get(selection);
      const found = selection.getItems().find(item => predicate(item, key));
      if (found !== undefined) {
        return found;
      }
    }
    return null;
  }
};
exports.default = CompositeListSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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