'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _desc, _value, _class;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _coreDecorators = require('core-decorators');

var _deferredCallbackQueue = require('../deferred-callback-queue');

var _deferredCallbackQueue2 = _interopRequireDefault(_deferredCallbackQueue);

var _repository = require('./repository');

var _repository2 = _interopRequireDefault(_repository);

var _resolutionProgress = require('./conflicts/resolution-progress');

var _resolutionProgress2 = _interopRequireDefault(_resolutionProgress);

var _fileSystemChangeObserver = require('./file-system-change-observer');

var _fileSystemChangeObserver2 = _interopRequireDefault(_fileSystemChangeObserver);

var _workspaceChangeObserver = require('./workspace-change-observer');

var _workspaceChangeObserver2 = _interopRequireDefault(_workspaceChangeObserver);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

function _applyDecoratedDescriptor(target, property, decorators, descriptor, context) {
  var desc = {};
  Object['ke' + 'ys'](descriptor).forEach(function (key) {
    desc[key] = descriptor[key];
  });
  desc.enumerable = !!desc.enumerable;
  desc.configurable = !!desc.configurable;

  if ('value' in desc || desc.initializer) {
    desc.writable = true;
  }

  desc = decorators.slice().reverse().reduce(function (desc, decorator) {
    return decorator(target, property, desc) || desc;
  }, desc);

  if (context && desc.initializer !== void 0) {
    desc.value = desc.initializer ? desc.initializer.call(context) : void 0;
    desc.initializer = undefined;
  }

  if (desc.initializer === void 0) {
    Object['define' + 'Property'](target, property, desc);
    desc = null;
  }

  return desc;
}

const createRepoSym = Symbol('createRepo');

let absentWorkdirContext;

/*
 * Bundle of model objects associated with a git working directory.
 *
 * Provides synchronous access to each model in the form of a getter method that returns the model or `null` if it
 * has not yet been initialized, and asynchronous access in the form of a Promise generation method that will resolve
 * once the model is available. Initializes the platform-appropriate change observer and proxies select filesystem
 * change events.
 */
let WorkdirContext = (_class = class WorkdirContext {

  /*
   * Available options:
   * - `options.window`: Browser window global, used on Linux by the WorkspaceChangeObserver.
   * - `options.workspace`: Atom's workspace singleton, used on Linux by the WorkspaceChangeObserver.
   * - `options.promptCallback`: Callback used to collect information interactively through Atom.
   */
  constructor(directory) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    this.directory = directory;
    this.repository = (options[createRepoSym] || (() => new _repository2.default(directory)))();

    const theWindow = options.window,
          workspace = options.workspace,
          promptCallback = options.promptCallback;


    this.destroyed = false;
    this.emitter = new _eventKit.Emitter();
    this.subs = new _eventKit.CompositeDisposable();

    this.observer = this.useWorkspaceChangeObserver() ? new _workspaceChangeObserver2.default(theWindow, workspace, this.repository) : new _fileSystemChangeObserver2.default(this.repository);
    this.resolutionProgress = new _resolutionProgress2.default();
    this.deferredCallbackQueue = new _deferredCallbackQueue2.default(3000, collection => {
      this.repository.observeFilesystemChange(collection);
    });

    if (promptCallback) {
      this.repository.setPromptCallback(promptCallback);
    }

    // Wire up event forwarding among models
    this.subs.add(this.repository.onDidChangeState(this.repositoryChangedState));
    this.subs.add(this.observer.onDidChange(events => {
      const paths = events.map(e => e.special || _path2.default.join(e.directory, e.file || e.newFile));
      this.deferredCallbackQueue.push(...paths);
    }));
    this.subs.add(this.observer.onDidChangeWorkdirOrHead(() => this.emitter.emit('did-change-workdir-or-head')));
    this.subs.add(new _eventKit.Disposable(() => this.deferredCallbackQueue.destroy()));

    // If a pre-loaded Repository was provided, broadcast an initial state change event.
    this.repositoryChangedState({ from: null, to: this.repository.state });
  }

  static absent() {
    if (!absentWorkdirContext) {
      absentWorkdirContext = new AbsentWorkdirContext();
    }
    return absentWorkdirContext;
  }

  static destroyAbsent() {
    if (absentWorkdirContext) {
      absentWorkdirContext.destroy();
      absentWorkdirContext = null;
    }
  }

  static guess(options) {
    const projectPathCount = options.projectPathCount || 0;
    const initPathCount = options.initPathCount || 0;

    const createRepo = projectPathCount === 1 || projectPathCount === 0 && initPathCount === 1 ? () => _repository2.default.loadingGuess() : () => _repository2.default.absentGuess();

    return new WorkdirContext(null, { [createRepoSym]: createRepo });
  }

  /**
   * Respond to changes in `Repository` state. Load resolution progress and start the change observer when it becomes
   * present. Stop the change observer when it is destroyed. Re-broadcast the event to context subscribers
   * regardless.
   *
   * The ResolutionProgress will be loaded before the change event is re-broadcast, but change observer modifications
   * will not be complete.
   */

  repositoryChangedState(payload) {
    if (this.destroyed) {
      return;
    }

    if (this.repository.isPresent()) {
      this.observer.start().then(() => this.emitter.emit('did-start-observer'));
    } else if (this.repository.isDestroyed()) {
      this.emitter.emit('did-destroy-repository');

      this.observer.destroy();
    }

    this.emitter.emit('did-change-repository-state', payload);
  }

  isPresent() {
    return true;
  }

  isDestroyed() {
    return this.destroyed;
  }

  useWorkspaceChangeObserver() {
    return !!process.env.ATOM_GITHUB_WORKSPACE_OBSERVER || process.platform === 'linux';
  }

  // Event subscriptions

  onDidStartObserver(callback) {
    return this.emitter.on('did-start-observer', callback);
  }

  onDidChangeWorkdirOrHead(callback) {
    return this.emitter.on('did-change-workdir-or-head', callback);
  }

  onDidChangeRepositoryState(callback) {
    return this.emitter.on('did-change-repository-state', callback);
  }

  onDidUpdateRepository(callback) {
    return this.emitter.on('did-update-repository', callback);
  }

  onDidDestroyRepository(callback) {
    return this.emitter.on('did-destroy-repository', callback);
  }

  /**
   * Return a Promise that will resolve the next time that a Repository transitions to the requested state. Most
   * useful for test cases; most callers should prefer subscribing to `onDidChangeRepositoryState`.
   */
  getRepositoryStatePromise(stateName) {
    return new Promise(resolve => {
      const sub = this.onDidChangeRepositoryState(() => {
        if (this.repository.isInState(stateName)) {
          resolve();
          sub.dispose();
        }
      });
    });
  }

  /**
   * Return a Promise that will resolve the next time that a ChangeObserver successfully starts. Most useful for
   * test cases.
   */
  getObserverStartedPromise() {
    return new Promise(resolve => {
      const sub = this.onDidStartObserver(() => {
        resolve();
        sub.dispose();
      });
    });
  }

  getWorkingDirectory() {
    return this.directory;
  }

  getRepository() {
    return this.repository;
  }

  getChangeObserver() {
    return this.observer;
  }

  getResolutionProgress() {
    return this.resolutionProgress;
  }

  /*
   * Cleanly destroy any models that need to be cleaned, including stopping the filesystem watcher.
   */
  destroy() {
    var _this = this;

    return _asyncToGenerator(function* () {
      if (_this.destroyed) {
        return;
      }
      _this.destroyed = true;

      _this.subs.dispose();
      _this.repository.destroy();
      _this.emitter.dispose();

      yield _this.observer.destroy();
    })();
  }
}, (_applyDecoratedDescriptor(_class.prototype, 'repositoryChangedState', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'repositoryChangedState'), _class.prototype)), _class);
exports.default = WorkdirContext;
let AbsentWorkdirContext = class AbsentWorkdirContext extends WorkdirContext {
  constructor() {
    super(null, { [createRepoSym]: () => _repository2.default.absent() });
  }

  isPresent() {
    return false;
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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