(function() {
  var GitRepository, Minimatch, PathLoader, PathsChunkSize, _, async, emittedPaths, fs, path;

  async = require('async');

  fs = require('fs');

  path = require('path');

  _ = require('underscore-plus');

  GitRepository = require('atom').GitRepository;

  Minimatch = require('minimatch').Minimatch;

  PathsChunkSize = 100;

  emittedPaths = new Set;

  PathLoader = (function() {
    function PathLoader(rootPath1, ignoreVcsIgnores, traverseSymlinkDirectories, ignoredNames1) {
      var repo;
      this.rootPath = rootPath1;
      this.traverseSymlinkDirectories = traverseSymlinkDirectories;
      this.ignoredNames = ignoredNames1;
      this.paths = [];
      this.realPathCache = {};
      this.repo = null;
      if (ignoreVcsIgnores) {
        repo = GitRepository.open(this.rootPath, {
          refreshOnWindowFocus: false
        });
        if ((repo != null ? repo.relativize(path.join(this.rootPath, 'test')) : void 0) === 'test') {
          this.repo = repo;
        }
      }
    }

    PathLoader.prototype.load = function(done) {
      return this.loadPath(this.rootPath, true, (function(_this) {
        return function() {
          var ref;
          _this.flushPaths();
          if ((ref = _this.repo) != null) {
            ref.destroy();
          }
          return done();
        };
      })(this));
    };

    PathLoader.prototype.isIgnored = function(loadedPath) {
      var i, ignoredName, len, ref, ref1, relativePath;
      relativePath = path.relative(this.rootPath, loadedPath);
      if ((ref = this.repo) != null ? ref.isPathIgnored(relativePath) : void 0) {
        return true;
      } else {
        ref1 = this.ignoredNames;
        for (i = 0, len = ref1.length; i < len; i++) {
          ignoredName = ref1[i];
          if (ignoredName.match(relativePath)) {
            return true;
          }
        }
      }
    };

    PathLoader.prototype.pathLoaded = function(loadedPath, done) {
      if (!(this.isIgnored(loadedPath) || emittedPaths.has(loadedPath))) {
        this.paths.push(loadedPath);
        emittedPaths.add(loadedPath);
      }
      if (this.paths.length === PathsChunkSize) {
        this.flushPaths();
      }
      return done();
    };

    PathLoader.prototype.flushPaths = function() {
      emit('load-paths:paths-found', this.paths);
      return this.paths = [];
    };

    PathLoader.prototype.loadPath = function(pathToLoad, root, done) {
      if (this.isIgnored(pathToLoad) && !root) {
        return done();
      }
      return fs.lstat(pathToLoad, (function(_this) {
        return function(error, stats) {
          if (error != null) {
            return done();
          }
          if (stats.isSymbolicLink()) {
            return _this.isInternalSymlink(pathToLoad, function(isInternal) {
              if (isInternal) {
                return done();
              }
              return fs.stat(pathToLoad, function(error, stats) {
                if (error != null) {
                  return done();
                }
                if (stats.isFile()) {
                  return _this.pathLoaded(pathToLoad, done);
                } else if (stats.isDirectory()) {
                  if (_this.traverseSymlinkDirectories) {
                    return _this.loadFolder(pathToLoad, done);
                  } else {
                    return done();
                  }
                } else {
                  return done();
                }
              });
            });
          } else if (stats.isDirectory()) {
            return _this.loadFolder(pathToLoad, done);
          } else if (stats.isFile()) {
            return _this.pathLoaded(pathToLoad, done);
          } else {
            return done();
          }
        };
      })(this));
    };

    PathLoader.prototype.loadFolder = function(folderPath, done) {
      return fs.readdir(folderPath, (function(_this) {
        return function(error, children) {
          if (children == null) {
            children = [];
          }
          return async.each(children, function(childName, next) {
            return _this.loadPath(path.join(folderPath, childName), false, next);
          }, done);
        };
      })(this));
    };

    PathLoader.prototype.isInternalSymlink = function(pathToLoad, done) {
      return fs.realpath(pathToLoad, this.realPathCache, (function(_this) {
        return function(err, realPath) {
          if (err) {
            return done(false);
          } else {
            return done(realPath.search(_this.rootPath) === 0);
          }
        };
      })(this));
    };

    return PathLoader;

  })();

  module.exports = function(rootPaths, followSymlinks, ignoreVcsIgnores, ignores) {
    var error, i, ignore, ignoredNames, len;
    if (ignores == null) {
      ignores = [];
    }
    ignoredNames = [];
    for (i = 0, len = ignores.length; i < len; i++) {
      ignore = ignores[i];
      if (ignore) {
        try {
          ignoredNames.push(new Minimatch(ignore, {
            matchBase: true,
            dot: true
          }));
        } catch (error1) {
          error = error1;
          console.warn("Error parsing ignore pattern (" + ignore + "): " + error.message);
        }
      }
    }
    return async.each(rootPaths, function(rootPath, next) {
      return new PathLoader(rootPath, ignoreVcsIgnores, followSymlinks, ignoredNames).load(next);
    }, this.async());
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
