(function() {
  var COMPLETIONS, attributePattern, firstCharsEqual, tagPattern;

  COMPLETIONS = require('../completions.json');

  attributePattern = /\s+([a-zA-Z][-a-zA-Z]*)\s*=\s*$/;

  tagPattern = /<([a-zA-Z][-a-zA-Z]*)(?:\s|$)/;

  module.exports = {
    selector: '.text.html',
    disableForSelector: '.text.html .comment',
    filterSuggestions: true,
    completions: COMPLETIONS,
    getSuggestions: function(request) {
      if (this.isAttributeValueStart(request)) {
        return this.getAttributeValueCompletions(request);
      } else if (this.isAttributeStart(request)) {
        return this.getAttributeNameCompletions(request);
      } else if (this.isTagStart(request)) {
        return this.getTagNameCompletions(request);
      } else {
        return [];
      }
    },
    onDidInsertSuggestion: function(arg) {
      var editor, suggestion;
      editor = arg.editor, suggestion = arg.suggestion;
      if (suggestion.type === 'attribute') {
        return setTimeout(this.triggerAutocomplete.bind(this, editor), 1);
      }
    },
    triggerAutocomplete: function(editor) {
      return atom.commands.dispatch(atom.views.getView(editor), 'autocomplete-plus:activate', {
        activatedManually: false
      });
    },
    isTagStart: function(arg) {
      var bufferPosition, editor, prefix, scopeDescriptor, scopes;
      prefix = arg.prefix, scopeDescriptor = arg.scopeDescriptor, bufferPosition = arg.bufferPosition, editor = arg.editor;
      if (prefix.trim() && prefix.indexOf('<') === -1) {
        return this.hasTagScope(scopeDescriptor.getScopesArray());
      }
      prefix = editor.getTextInRange([[bufferPosition.row, bufferPosition.column - 1], bufferPosition]);
      scopes = scopeDescriptor.getScopesArray();
      return prefix === '<' && scopes[0] === 'text.html.basic' && scopes.length === 1;
    },
    isAttributeStart: function(arg) {
      var bufferPosition, editor, prefix, previousBufferPosition, previousScopes, previousScopesArray, scopeDescriptor, scopes;
      prefix = arg.prefix, scopeDescriptor = arg.scopeDescriptor, bufferPosition = arg.bufferPosition, editor = arg.editor;
      scopes = scopeDescriptor.getScopesArray();
      if (!this.getPreviousAttribute(editor, bufferPosition) && prefix && !prefix.trim()) {
        return this.hasTagScope(scopes);
      }
      previousBufferPosition = [bufferPosition.row, Math.max(0, bufferPosition.column - 1)];
      previousScopes = editor.scopeDescriptorForBufferPosition(previousBufferPosition);
      previousScopesArray = previousScopes.getScopesArray();
      if (previousScopesArray.indexOf('entity.other.attribute-name.html') !== -1) {
        return true;
      }
      if (!this.hasTagScope(scopes)) {
        return false;
      }
      return scopes.indexOf('punctuation.definition.tag.end.html') !== -1 && previousScopesArray.indexOf('punctuation.definition.tag.end.html') === -1;
    },
    isAttributeValueStart: function(arg) {
      var bufferPosition, editor, previousBufferPosition, previousScopes, previousScopesArray, scopeDescriptor, scopes;
      scopeDescriptor = arg.scopeDescriptor, bufferPosition = arg.bufferPosition, editor = arg.editor;
      scopes = scopeDescriptor.getScopesArray();
      previousBufferPosition = [bufferPosition.row, Math.max(0, bufferPosition.column - 1)];
      previousScopes = editor.scopeDescriptorForBufferPosition(previousBufferPosition);
      previousScopesArray = previousScopes.getScopesArray();
      return this.hasStringScope(scopes) && this.hasStringScope(previousScopesArray) && previousScopesArray.indexOf('punctuation.definition.string.end.html') === -1 && this.hasTagScope(scopes) && (this.getPreviousAttribute(editor, bufferPosition) != null);
    },
    hasTagScope: function(scopes) {
      return scopes.indexOf('meta.tag.any.html') !== -1 || scopes.indexOf('meta.tag.other.html') !== -1 || scopes.indexOf('meta.tag.block.any.html') !== -1 || scopes.indexOf('meta.tag.inline.any.html') !== -1 || scopes.indexOf('meta.tag.structure.any.html') !== -1;
    },
    hasStringScope: function(scopes) {
      return scopes.indexOf('string.quoted.double.html') !== -1 || scopes.indexOf('string.quoted.single.html') !== -1;
    },
    getTagNameCompletions: function(arg) {
      var bufferPosition, completions, editor, ignorePrefix, options, prefix, ref, tag;
      prefix = arg.prefix, editor = arg.editor, bufferPosition = arg.bufferPosition;
      ignorePrefix = editor.getTextInRange([[bufferPosition.row, bufferPosition.column - 1], bufferPosition]) === '<';
      completions = [];
      ref = this.completions.tags;
      for (tag in ref) {
        options = ref[tag];
        if (ignorePrefix || firstCharsEqual(tag, prefix)) {
          completions.push(this.buildTagCompletion(tag, options));
        }
      }
      return completions;
    },
    buildTagCompletion: function(tag, arg) {
      var description;
      description = arg.description;
      return {
        text: tag,
        type: 'tag',
        description: description != null ? description : "HTML <" + tag + "> tag",
        descriptionMoreURL: description ? this.getTagDocsURL(tag) : null
      };
    },
    getAttributeNameCompletions: function(arg) {
      var attribute, bufferPosition, completions, editor, i, len, options, prefix, ref, tag, tagAttributes;
      prefix = arg.prefix, editor = arg.editor, bufferPosition = arg.bufferPosition;
      completions = [];
      tag = this.getPreviousTag(editor, bufferPosition);
      tagAttributes = this.getTagAttributes(tag);
      for (i = 0, len = tagAttributes.length; i < len; i++) {
        attribute = tagAttributes[i];
        if (!prefix.trim() || firstCharsEqual(attribute, prefix)) {
          completions.push(this.buildLocalAttributeCompletion(attribute, tag, this.completions.attributes[attribute]));
        }
      }
      ref = this.completions.attributes;
      for (attribute in ref) {
        options = ref[attribute];
        if (!prefix.trim() || firstCharsEqual(attribute, prefix)) {
          if (options.global) {
            completions.push(this.buildGlobalAttributeCompletion(attribute, options));
          }
        }
      }
      return completions;
    },
    buildLocalAttributeCompletion: function(attribute, tag, options) {
      return {
        snippet: (options != null ? options.type : void 0) === 'flag' ? attribute : attribute + "=\"$1\"$0",
        displayText: attribute,
        type: 'attribute',
        rightLabel: "<" + tag + ">",
        description: attribute + " attribute local to <" + tag + "> tags",
        descriptionMoreURL: this.getLocalAttributeDocsURL(attribute, tag)
      };
    },
    buildGlobalAttributeCompletion: function(attribute, arg) {
      var description, type;
      description = arg.description, type = arg.type;
      return {
        snippet: type === 'flag' ? attribute : attribute + "=\"$1\"$0",
        displayText: attribute,
        type: 'attribute',
        description: description != null ? description : "Global " + attribute + " attribute",
        descriptionMoreURL: description ? this.getGlobalAttributeDocsURL(attribute) : null
      };
    },
    getAttributeValueCompletions: function(arg) {
      var attribute, bufferPosition, completions, editor, i, len, prefix, ref, tag, value, values;
      prefix = arg.prefix, editor = arg.editor, bufferPosition = arg.bufferPosition;
      completions = [];
      tag = this.getPreviousTag(editor, bufferPosition);
      attribute = this.getPreviousAttribute(editor, bufferPosition);
      values = this.getAttributeValues(tag, attribute);
      for (i = 0, len = values.length; i < len; i++) {
        value = values[i];
        if (!prefix || firstCharsEqual(value, prefix)) {
          completions.push(this.buildAttributeValueCompletion(tag, attribute, value));
        }
      }
      if (completions.length === 0 && ((ref = this.completions.attributes[attribute]) != null ? ref.type : void 0) === 'boolean') {
        completions.push(this.buildAttributeValueCompletion(tag, attribute, 'true'));
        completions.push(this.buildAttributeValueCompletion(tag, attribute, 'false'));
      }
      return completions;
    },
    buildAttributeValueCompletion: function(tag, attribute, value) {
      if (this.completions.attributes[attribute].global) {
        return {
          text: value,
          type: 'value',
          description: value + " value for global " + attribute + " attribute",
          descriptionMoreURL: this.getGlobalAttributeDocsURL(attribute)
        };
      } else {
        return {
          text: value,
          type: 'value',
          rightLabel: "<" + tag + ">",
          description: value + " value for " + attribute + " attribute local to <" + tag + ">",
          descriptionMoreURL: this.getLocalAttributeDocsURL(attribute, tag)
        };
      }
    },
    getPreviousTag: function(editor, bufferPosition) {
      var ref, row, tag;
      row = bufferPosition.row;
      while (row >= 0) {
        tag = (ref = tagPattern.exec(editor.lineTextForBufferRow(row))) != null ? ref[1] : void 0;
        if (tag) {
          return tag;
        }
        row--;
      }
    },
    getPreviousAttribute: function(editor, bufferPosition) {
      var quoteIndex, ref, scopes, scopesArray;
      quoteIndex = bufferPosition.column - 1;
      while (quoteIndex) {
        scopes = editor.scopeDescriptorForBufferPosition([bufferPosition.row, quoteIndex]);
        scopesArray = scopes.getScopesArray();
        if (!this.hasStringScope(scopesArray) || scopesArray.indexOf('punctuation.definition.string.begin.html') !== -1) {
          break;
        }
        quoteIndex--;
      }
      return (ref = attributePattern.exec(editor.getTextInRange([[bufferPosition.row, 0], [bufferPosition.row, quoteIndex]]))) != null ? ref[1] : void 0;
    },
    getAttributeValues: function(tag, attribute) {
      var ref, ref1, ref2, ref3;
      return (ref = (ref1 = (ref2 = this.completions.attributes[attribute]) != null ? ref2.attribOption : void 0) != null ? ref1 : (ref3 = this.completions.attributes[tag + "/" + attribute]) != null ? ref3.attribOption : void 0) != null ? ref : [];
    },
    getTagAttributes: function(tag) {
      var ref, ref1;
      return (ref = (ref1 = this.completions.tags[tag]) != null ? ref1.attributes : void 0) != null ? ref : [];
    },
    getTagDocsURL: function(tag) {
      return "https://developer.mozilla.org/en-US/docs/Web/HTML/Element/" + tag;
    },
    getLocalAttributeDocsURL: function(attribute, tag) {
      return (this.getTagDocsURL(tag)) + "#attr-" + attribute;
    },
    getGlobalAttributeDocsURL: function(attribute) {
      return "https://developer.mozilla.org/en-US/docs/Web/HTML/Global_attributes/" + attribute;
    }
  };

  firstCharsEqual = function(str1, str2) {
    return str1[0].toLowerCase() === str2[0].toLowerCase();
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
