(function() {
  var CompositeDisposable, IgnoredCommands, PathRE, Reporter, fs, grim, path, stripPath;

  CompositeDisposable = require('atom').CompositeDisposable;

  path = require('path');

  Reporter = require('./reporter');

  fs = require('fs-plus');

  grim = require('grim');

  IgnoredCommands = {
    'vim-mode:move-up': true,
    'vim-mode:move-down': true,
    'vim-mode:move-left': true,
    'vim-mode:move-right': true
  };

  module.exports = {
    activate: function(arg) {
      var sessionLength;
      sessionLength = arg.sessionLength;
      this.subscriptions = new CompositeDisposable;
      this.shouldIncludePanesAndCommands = Math.random() < 0.05;
      return this.ensureClientId((function(_this) {
        return function() {
          return _this.begin(sessionLength);
        };
      })(this));
    },
    deactivate: function() {
      var ref;
      return (ref = this.subscriptions) != null ? ref.dispose() : void 0;
    },
    serialize: function() {
      return {
        sessionLength: Date.now() - this.sessionStart
      };
    },
    provideReporter: function() {
      return {
        sendEvent: Reporter.sendEvent.bind(Reporter),
        sendTiming: Reporter.sendTiming.bind(Reporter),
        sendException: Reporter.sendException.bind(Reporter)
      };
    },
    begin: function(sessionLength) {
      this.sessionStart = Date.now();
      if (sessionLength) {
        Reporter.sendEvent('window', 'ended', null, sessionLength);
      }
      Reporter.sendEvent('window', 'started');
      this.subscriptions.add(atom.onDidThrowError(function(event) {
        var errorMessage;
        errorMessage = event;
        if (typeof event !== 'string') {
          errorMessage = event.message;
        }
        errorMessage = stripPath(errorMessage) || 'Unknown';
        errorMessage = errorMessage.replace('Uncaught ', '').slice(0, 150);
        return Reporter.sendException(errorMessage);
      }));
      this.subscriptions.add(atom.textEditors.observe(function(editor) {
        var grammar;
        grammar = editor.getGrammar();
        if (grammar) {
          return Reporter.sendEvent('file', 'open', grammar.scopeName);
        }
      }));
      this.subscriptions.add(atom.config.onDidChange('core.telemetryConsent', function(arg) {
        var newValue, oldValue;
        newValue = arg.newValue, oldValue = arg.oldValue;
        if (newValue !== 'undecided') {
          return Reporter.sendEvent('setting', 'core.telemetryConsent', newValue);
        }
      }));
      this.watchPaneItems();
      this.watchCommands();
      this.watchDeprecations();
      if (atom.getLoadSettings().shellLoadTime != null) {
        Reporter.sendTiming('shell', 'load', atom.getLoadSettings().shellLoadTime);
      }
      return process.nextTick(function() {
        return Reporter.sendTiming('core', 'load', atom.getWindowLoadTime());
      });
    },
    ensureClientId: function(callback) {
      if (localStorage.getItem('metrics.userId')) {
        return callback();
      } else if (atom.config.get('metrics.userId')) {
        localStorage.setItem('metrics.userId', atom.config.get('metrics.userId'));
        return callback();
      } else {
        return this.createClientId(function(clientId) {
          localStorage.setItem('metrics.userId', clientId);
          return callback();
        });
      }
    },
    createClientId: function(callback) {
      return callback(require('node-uuid').v4());
    },
    getClientId: function() {
      return localStorage.getItem('metrics.userId');
    },
    watchPaneItems: function() {
      return this.subscriptions.add(atom.workspace.onDidAddPaneItem((function(_this) {
        return function(arg) {
          var item;
          item = arg.item;
          if (!_this.shouldIncludePanesAndCommands) {
            return;
          }
          return Reporter.sendPaneItem(item);
        };
      })(this)));
    },
    watchCommands: function() {
      return this.subscriptions.add(atom.commands.onWillDispatch((function(_this) {
        return function(commandEvent) {
          var eventName, ref;
          if (!_this.shouldIncludePanesAndCommands) {
            return;
          }
          eventName = commandEvent.type;
          if ((ref = commandEvent.detail) != null ? ref.jQueryTrigger : void 0) {
            return;
          }
          if (eventName.startsWith('core:') || eventName.startsWith('editor:')) {
            return;
          }
          if (!(eventName.indexOf(':') > -1)) {
            return;
          }
          if (eventName in IgnoredCommands) {
            return;
          }
          return Reporter.sendCommand(eventName);
        };
      })(this)));
    },
    watchDeprecations: function() {
      this.deprecationCache = {};
      this.packageVersionCache = {};
      atom.packages.onDidActivateInitialPackages((function(_this) {
        return function() {
          var j, len, pack, packages, ref;
          packages = atom.packages.getLoadedPackages();
          for (j = 0, len = packages.length; j < len; j++) {
            pack = packages[j];
            _this.packageVersionCache[pack.name] = (pack != null ? (ref = pack.metadata) != null ? ref.version : void 0 : void 0) || 'unknown';
          }
          setImmediate(function() {
            var deprecation, k, len1, ref1;
            ref1 = grim.getDeprecations();
            for (k = 0, len1 = ref1.length; k < len1; k++) {
              deprecation = ref1[k];
              _this.reportDeprecation(deprecation);
            }
          });
        };
      })(this));
      atom.packages.onDidLoadPackage((function(_this) {
        return function(pack) {
          var ref;
          if (!_this.packageVersionCache[pack.name]) {
            return _this.packageVersionCache[pack.name] = (pack != null ? (ref = pack.metadata) != null ? ref.version : void 0 : void 0) || 'unknown';
          }
        };
      })(this));
      return grim.on('updated', (function(_this) {
        return function(deprecation) {
          return setImmediate(function() {
            return _this.reportDeprecation(deprecation);
          });
        };
      })(this));
    },
    reportDeprecation: function(deprecation) {
      var __, message, nameAndVersion, pack, packageName, ref, ref1, ref2, ref3, stack, version;
      message = deprecation.getMessage().slice(0, 500);
      ref = deprecation.stacks;
      for (__ in ref) {
        stack = ref[__];
        packageName = (ref1 = (ref2 = stack.metadata) != null ? ref2.packageName : void 0) != null ? ref1 : (this.getPackageName(stack) || '').toLowerCase();
        if (!packageName) {
          continue;
        }
        if (!this.packageVersionCache[packageName]) {
          pack = atom.packages.getLoadedPackage(packageName);
          this.packageVersionCache[packageName] = (pack != null ? (ref3 = pack.metadata) != null ? ref3.version : void 0 : void 0) || 'unknown';
        }
        version = this.packageVersionCache[packageName];
        nameAndVersion = packageName + "@" + version;
        if (this.deprecationCache[nameAndVersion + message] == null) {
          this.deprecationCache[nameAndVersion + message] = true;
          Reporter.sendEvent('deprecation-v3', nameAndVersion, message);
        }
      }
    },
    getFileNameFromCallSite: function(callsite) {
      var ref;
      return (ref = callsite.fileName) != null ? ref : callsite.getFileName();
    },
    getPackageName: function(stack) {
      var fileName, i, j, packageName, packagePath, packagePaths, ref, relativePath;
      packagePaths = this.getPackagePathsByPackageName();
      for (i = j = 1, ref = stack.length; 1 <= ref ? j < ref : j > ref; i = 1 <= ref ? ++j : --j) {
        fileName = this.getFileNameFromCallSite(stack[i]);
        if (!fileName) {
          return;
        }
        if (fileName.includes(path.sep + "node_modules" + path.sep)) {
          continue;
        }
        for (packageName in packagePaths) {
          packagePath = packagePaths[packageName];
          relativePath = path.relative(packagePath, fileName);
          if (!/^\.\./.test(relativePath)) {
            return packageName;
          }
        }
        if (atom.getUserInitScriptPath() === fileName) {
          return "init-script";
        }
      }
    },
    getPackagePathsByPackageName: function() {
      var j, len, pack, ref;
      if (this.packagePathsByPackageName != null) {
        return this.packagePathsByPackageName;
      }
      this.packagePathsByPackageName = {};
      ref = atom.packages.getLoadedPackages();
      for (j = 0, len = ref.length; j < len; j++) {
        pack = ref[j];
        this.packagePathsByPackageName[pack.name] = pack.path;
        if (pack.path.indexOf('.atom/dev/packages') > -1 || pack.path.indexOf('.atom/packages') > -1) {
          this.packagePathsByPackageName[pack.name] = fs.absolute(pack.path);
        }
      }
      return this.packagePathsByPackageName;
    }
  };

  PathRE = /'?((\/|\\|[a-z]:\\)[^\s']+)+'?/ig;

  stripPath = function(message) {
    return message.replace(PathRE, '<path>');
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
