'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _fsExtra = require('fs-extra');

var _fsExtra2 = _interopRequireDefault(_fsExtra);

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _helpers = require('./helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; } // Measure elapsed durations from specific beginning points.

// The maximum number of marks within a single DurationSet. A DurationSet will be automatically finished if this many
// marks are recorded.
const MAXIMUM_MARKS = 100;

// Flush all non-active DurationSets to disk each time that this many marks have been accumulated.
const PERSIST_INTERVAL = 1000;

// A sequence of durations measured from a fixed beginning point.
let DurationSet = class DurationSet {
  constructor(name) {
    this.name = name;
    this.startTimestamp = performance.now();
    this.marks = [];
    this.markCount = 0;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cbegin %c%s:begin', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name);
    }

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profile(this.name);
    }
  }

  mark(eventName) {
    const duration = performance.now() - this.startTimestamp;

    if (atom.config.get('github.performanceToConsole')) {
      // eslint-disable-next-line no-console
      console.log('%cmark %c%s:%s %c%dms', 'font-weight: bold', 'font-weight: normal; font-style: italic; color: blue', this.name, eventName, 'font-style: normal; color: black', duration);
    }

    if (atom.config.get('github.performanceToDirectory') !== '') {
      this.marks.push({ eventName, duration });
    }

    this.markCount++;
    if (this.markCount >= MAXIMUM_MARKS) {
      this.finish();
    }
  }

  finish() {
    this.mark('finish');

    if (atom.config.get('github.performanceToProfile')) {
      // eslint-disable-next-line no-console
      console.profileEnd(this.name);
    }
  }

  serialize() {
    return {
      name: this.name,
      markers: this.marks
    };
  }

  getCount() {
    return this.marks.length;
  }
};


let durationSets = [];
let totalMarkCount = 0;
const activeSets = new Map();

function shouldCapture(seriesName, eventName) {
  const anyActive = ['Console', 'Directory', 'Profile'].some(kind => {
    const value = atom.config.get(`github.performanceTo${kind}`);
    return value !== '' && value !== false;
  });
  if (!anyActive) {
    return false;
  }

  const mask = new RegExp(atom.config.get('github.performanceMask'));
  if (!mask.test(`${seriesName}:${eventName}`)) {
    return false;
  }

  return true;
}

const yardstick = {
  save() {
    return _asyncToGenerator(function* () {
      const destDir = atom.config.get('github.performanceToDirectory');
      if (destDir === '' || destDir === undefined || destDir === null) {
        return;
      }
      const fileName = _path2.default.join(destDir, `performance-${Date.now()}.json`);

      yield new Promise(function (resolve, reject) {
        _fsExtra2.default.ensureDir(destDir, function (err) {
          return err ? reject(err) : resolve();
        });
      });

      const payload = JSON.stringify(durationSets.map(function (set) {
        return set.serialize();
      }));
      yield (0, _helpers.writeFile)(fileName, payload);

      if (atom.config.get('github.performanceToConsole')) {
        // eslint-disable-next-line no-console
        console.log('%csaved %c%d series to %s', 'font-weight: bold', 'font-weight: normal; color: black', durationSets.length, fileName);
      }

      durationSets = [];
    })();
  },

  begin(seriesName) {
    if (!shouldCapture(seriesName, 'begin')) {
      return;
    }

    const ds = new DurationSet(seriesName);
    activeSets.set(seriesName, ds);
  },

  mark(seriesName, eventName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.mark(seriesName[i], eventName);
      }
      return;
    }

    if (!shouldCapture(seriesName, eventName)) {
      return;
    }

    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.mark(eventName);
  },

  finish(seriesName) {
    if (seriesName instanceof Array) {
      for (let i = 0; i < seriesName.length; i++) {
        this.finish(seriesName[i]);
      }
      return;
    }

    if (!shouldCapture(seriesName, 'finish')) {
      return;
    }

    const ds = activeSets.get(seriesName);
    if (ds === undefined) {
      return;
    }
    ds.finish();

    durationSets.push(ds);
    activeSets.delete(seriesName);

    totalMarkCount += ds.getCount();
    if (totalMarkCount >= PERSIST_INTERVAL) {
      totalMarkCount = 0;
      this.save();
    }
  },

  flush() {
    var _this = this;

    return _asyncToGenerator(function* () {
      durationSets.push(...activeSets.values());
      activeSets.clear();
      yield _this.save();
    })();
  }
};

exports.default = yardstick;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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