'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _hunk = require('./hunk');

var _hunk2 = _interopRequireDefault(_hunk);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let FilePatch = class FilePatch {
  constructor(oldPath, newPath, status, hunks) {
    this.oldPath = oldPath;
    this.newPath = newPath;
    this.status = status;
    this.hunks = hunks;
    this.changedLineCount = this.hunks.reduce((acc, hunk) => {
      return acc + hunk.getLines().filter(line => line.isChanged()).length;
    }, 0);
  }

  getOldPath() {
    return this.oldPath;
  }

  getNewPath() {
    return this.newPath;
  }

  getPath() {
    return this.getOldPath() || this.getNewPath();
  }

  getStatus() {
    return this.status;
  }

  getHunks() {
    return this.hunks;
  }

  getStagePatchForHunk(selectedHunk) {
    return this.getStagePatchForLines(new Set(selectedHunk.getLines()));
  }

  getStagePatchForLines(selectedLines) {
    if (this.changedLineCount === [...selectedLines].filter(line => line.isChanged()).length) {
      return this;
    }

    let delta = 0;
    const hunks = [];
    for (const hunk of this.getHunks()) {
      const newStartRow = (hunk.getNewStartRow() || 1) + delta;
      let newLineNumber = newStartRow;
      const lines = [];
      let hunkContainsSelectedLines = false;
      for (const line of hunk.getLines()) {
        if (line.getStatus() === 'nonewline') {
          lines.push(line.copy({ oldLineNumber: -1, newLineNumber: -1 }));
        } else if (selectedLines.has(line)) {
          hunkContainsSelectedLines = true;
          if (line.getStatus() === 'deleted') {
            lines.push(line.copy());
          } else {
            lines.push(line.copy({ newLineNumber: newLineNumber++ }));
          }
        } else if (line.getStatus() === 'deleted') {
          lines.push(line.copy({ newLineNumber: newLineNumber++, status: 'unchanged' }));
        } else if (line.getStatus() === 'unchanged') {
          lines.push(line.copy({ newLineNumber: newLineNumber++ }));
        }
      }
      const newRowCount = newLineNumber - newStartRow;
      if (hunkContainsSelectedLines) {
        // eslint-disable-next-line max-len
        hunks.push(new _hunk2.default(hunk.getOldStartRow(), newStartRow, hunk.getOldRowCount(), newRowCount, hunk.getSectionHeading(), lines));
      }
      delta += newRowCount - hunk.getNewRowCount();
    }

    return new FilePatch(this.getOldPath(), this.getNewPath() ? this.getNewPath() : this.getOldPath(), this.getStatus(), hunks);
  }

  getUnstagePatch() {
    let invertedStatus;
    switch (this.getStatus()) {
      case 'modified':
        invertedStatus = 'modified';
        break;
      case 'added':
        invertedStatus = 'deleted';
        break;
      case 'deleted':
        invertedStatus = 'added';
        break;
      default:
        throw new Error(`Unknown Status: ${this.getStatus()}`);
    }
    const invertedHunks = this.getHunks().map(h => h.invert());
    return new FilePatch(this.getNewPath(), this.getOldPath(), invertedStatus, invertedHunks);
  }

  getUnstagePatchForHunk(hunk) {
    return this.getUnstagePatchForLines(new Set(hunk.getLines()));
  }

  getUnstagePatchForLines(selectedLines) {
    if (this.changedLineCount === [...selectedLines].filter(line => line.isChanged()).length) {
      return this.getUnstagePatch();
    }

    let delta = 0;
    const hunks = [];
    for (const hunk of this.getHunks()) {
      const oldStartRow = (hunk.getOldStartRow() || 1) + delta;
      let oldLineNumber = oldStartRow;
      const lines = [];
      let hunkContainsSelectedLines = false;
      for (const line of hunk.getLines()) {
        if (line.getStatus() === 'nonewline') {
          lines.push(line.copy({ oldLineNumber: -1, newLineNumber: -1 }));
        } else if (selectedLines.has(line)) {
          hunkContainsSelectedLines = true;
          if (line.getStatus() === 'added') {
            lines.push(line.copy());
          } else {
            lines.push(line.copy({ oldLineNumber: oldLineNumber++ }));
          }
        } else if (line.getStatus() === 'added') {
          lines.push(line.copy({ oldLineNumber: oldLineNumber++, status: 'unchanged' }));
        } else if (line.getStatus() === 'unchanged') {
          lines.push(line.copy({ oldLineNumber: oldLineNumber++ }));
        }
      }
      const oldRowCount = oldLineNumber - oldStartRow;
      if (hunkContainsSelectedLines) {
        // eslint-disable-next-line max-len
        hunks.push(new _hunk2.default(oldStartRow, hunk.getNewStartRow(), oldRowCount, hunk.getNewRowCount(), hunk.getSectionHeading(), lines));
      }
      delta += oldRowCount - hunk.getOldRowCount();
    }

    return new FilePatch(this.getOldPath() ? this.getOldPath() : this.getNewPath(), this.getNewPath(), this.getStatus(), hunks).getUnstagePatch();
  }

  toString() {
    return this.getHunks().map(h => h.toString()).join('');
  }

  getHeaderString() {
    let header = this.getOldPath() ? `--- a/${(0, _helpers.toGitPathSep)(this.getOldPath())}` : '--- /dev/null';
    header += '\n';
    header += this.getNewPath() ? `+++ b/${(0, _helpers.toGitPathSep)(this.getNewPath())}` : '+++ /dev/null';
    header += '\n';
    return header;
  }
};
exports.default = FilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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