'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _markerTools = require('./marker-tools');

let Side = class Side {
  constructor(editor, marker, blockMarker, source, position, banner, originalText) {
    this.editor = editor;
    this.marker = marker;
    this.blockMarker = blockMarker;
    this.source = source;
    this.position = position;
    this.banner = banner;
    this.originalText = originalText;
  }

  getMarker() {
    return this.marker;
  }

  getBannerMarker() {
    return this.banner.getMarker();
  }

  getSource() {
    return this.source;
  }

  getBlockMarker() {
    return this.blockMarker;
  }

  getBlockPosition() {
    return this.position.when({
      top: () => 'before',
      middle: () => 'before',
      bottom: () => 'after'
    });
  }

  getLineCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModified';
    } else {
      return this.source.getCSSClass();
    }
  }

  getBannerCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModifiedBanner';
    } else {
      return this.source.getBannerCSSClass();
    }
  }

  getBlockCSSClasses() {
    const cxs = ['github-ConflictBlock'];
    cxs.push(this.source.getBlockCSSClass());
    cxs.push(this.position.getBlockCSSClass());
    if (this.isModified() || this.isBannerModified()) {
      cxs.push('github-ConflictModifiedBlock');
    }
    return cxs.join(' ');
  }

  getPosition() {
    return this.position;
  }

  getRange() {
    const bannerRange = this.banner.getRange();
    const bodyRange = this.marker.getBufferRange();
    return bannerRange.union(bodyRange);
  }

  includesPoint(point) {
    return this.getRange().containsPoint(point);
  }

  getText() {
    return this.editor.getTextInBufferRange(this.getMarker().getBufferRange());
  }

  isBannerModified() {
    return this.banner.isModified();
  }

  isModified() {
    return this.getText() !== this.originalText;
  }

  isEmpty() {
    return this.marker.getBufferRange().isEmpty();
  }

  revertBanner() {
    this.banner.revert();
  }

  revert() {
    const range = this.getMarker().getBufferRange();
    this.editor.setTextInBufferRange(range, this.originalText);
  }

  deleteBanner() {
    this.banner.delete();
  }

  delete() {
    (0, _markerTools.deleteMarkerIn)(this.getMarker(), this.editor);
  }

  appendText(text) {
    const insertionPoint = this.getMarker().getBufferRange().end;
    return this.editor.setTextInBufferRange([insertionPoint, insertionPoint], text);
  }
};
exports.default = Side;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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