'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ConflictParser = undefined;

var _noopVisitor = require('./noop-visitor');

var _position = require('../position');

let Result = class Result {
  constructor(remainingSteps) {
    this.steps = remainingSteps;
  }

  wasSuccessful() {
    return this.steps.length === 0;
  }
};
let ConflictParser = exports.ConflictParser = class ConflictParser {
  constructor(adapter, visitor, isRebase) {
    this.adapter = adapter;
    this.visitor = visitor;
    this.isRebase = isRebase;

    this.lastBoundary = null;
    this.steps = [];

    if (this.isRebase) {
      this.steps.push(parser => parser.visitHeaderSide(_position.TOP, 'visitTheirSide'));
      this.steps.push(parser => parser.visitBaseAndSeparator());
      this.steps.push(parser => parser.visitFooterSide(_position.BOTTOM, 'visitOurSide'));
    } else {
      this.steps.push(parser => parser.visitHeaderSide(_position.TOP, 'visitOurSide'));
      this.steps.push(parser => parser.visitBaseAndSeparator());
      this.steps.push(parser => parser.visitFooterSide(_position.BOTTOM, 'visitTheirSide'));
    }
  }

  continueFrom(result) {
    this.steps = result.steps;
    return this.parse();
  }

  parse() {
    for (let i = 0; i < this.steps.length; i++) {
      if (!this.steps[i](this)) {
        return new Result(this.steps.slice(i));
      }
    }
    return new Result([]);
  }

  // Visit a side that begins with a banner and description as its first line.
  visitHeaderSide(position, visitMethod) {
    const sideRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();

    if (this.advanceToBoundary('|=') === null) {
      return false;
    }

    const sideRowEnd = this.adapter.getCurrentRow();

    this.visitor[visitMethod](position, sideRowStart, sideRowStart + 1, sideRowEnd);
    return true;
  }

  // Visit the base side from diff3 output, if one is present, then visit the separator.
  visitBaseAndSeparator() {
    if (this.lastBoundary === '|') {
      if (!this.visitBaseSide()) {
        return false;
      }
    }

    return this.visitSeparator();
  }

  // Visit a base side from diff3 output.
  visitBaseSide() {
    const sideRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();

    let b = this.advanceToBoundary('<=');
    if (b === null) {
      return false;
    }

    while (b === '<') {
      // Embedded recursive conflict within a base side, caused by a criss-cross merge.
      // Advance the input adapter beyond it without marking anything.
      const subParser = new ConflictParser(this.adapter, new _noopVisitor.NoopVisitor(), this.isRebase);
      if (!subParser.parse().wasSuccessful()) {
        return false;
      }

      b = this.advanceToBoundary('<=');
      if (b === null) {
        return false;
      }
    }

    const sideRowEnd = this.adapter.getCurrentRow();
    this.visitor.visitBaseSide(sideRowStart, sideRowStart + 1, sideRowEnd);
    return true;
  }

  // Visit a "========" separator.
  visitSeparator() {
    const sepRowStart = this.adapter.getCurrentRow();
    this.adapter.advanceRow();
    const sepRowEnd = this.adapter.getCurrentRow();

    this.visitor.visitSeparator(sepRowStart, sepRowEnd);
    return true;
  }

  // Visit a side with a banner and description as its last line.
  visitFooterSide(position, visitMethod) {
    const sideRowStart = this.adapter.getCurrentRow();
    if (this.advanceToBoundary('>') === null) {
      return false;
    }

    this.adapter.advanceRow();
    const sideRowEnd = this.adapter.getCurrentRow();

    this.visitor[visitMethod](position, sideRowEnd - 1, sideRowStart, sideRowEnd - 1);
    return true;
  }

  // Determine if the current row is a side boundary.
  //
  // boundaryKinds - [String] any combination of <, |, =, or > to limit the kinds of boundary detected.
  //
  // Returns the matching boundaryKinds character, or `null` if none match.
  isAtBoundary() {
    let boundaryKinds = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : '<|=>';

    const line = this.adapter.getCurrentLine();
    for (let i = 0; i < boundaryKinds.length; i++) {
      const b = boundaryKinds[i];
      if (line.startsWith(b.repeat(7))) {
        return b;
      }
    }
    return null;
  }

  // Increment the current row until the current line matches one of the provided boundary kinds, or until there are no
  // more lines in the editor.
  //
  // boundaryKinds - [String] any combination of <, |, =, or > to limit the kinds of boundaries that halt the
  //   progression.
  //
  // Returns the matching boundaryKinds character, or 'null' if there are no matches to the end of the editor.
  advanceToBoundary() {
    let boundaryKinds = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : '<|=>';

    let b = this.isAtBoundary(boundaryKinds);
    while (b === null) {
      this.adapter.advanceRow();
      if (this.adapter.isAtEnd()) {
        return null;
      }
      b = this.isAtBoundary(boundaryKinds);
    }

    this.lastBoundary = b;
    return b;
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImluZGV4LmpzIl0sIm5hbWVzIjpbIlJlc3VsdCIsImNvbnN0cnVjdG9yIiwicmVtYWluaW5nU3RlcHMiLCJzdGVwcyIsIndhc1N1Y2Nlc3NmdWwiLCJsZW5ndGgiLCJDb25mbGljdFBhcnNlciIsImFkYXB0ZXIiLCJ2aXNpdG9yIiwiaXNSZWJhc2UiLCJsYXN0Qm91bmRhcnkiLCJwdXNoIiwicGFyc2VyIiwidmlzaXRIZWFkZXJTaWRlIiwidmlzaXRCYXNlQW5kU2VwYXJhdG9yIiwidmlzaXRGb290ZXJTaWRlIiwiY29udGludWVGcm9tIiwicmVzdWx0IiwicGFyc2UiLCJpIiwic2xpY2UiLCJwb3NpdGlvbiIsInZpc2l0TWV0aG9kIiwic2lkZVJvd1N0YXJ0IiwiZ2V0Q3VycmVudFJvdyIsImFkdmFuY2VSb3ciLCJhZHZhbmNlVG9Cb3VuZGFyeSIsInNpZGVSb3dFbmQiLCJ2aXNpdEJhc2VTaWRlIiwidmlzaXRTZXBhcmF0b3IiLCJiIiwic3ViUGFyc2VyIiwic2VwUm93U3RhcnQiLCJzZXBSb3dFbmQiLCJpc0F0Qm91bmRhcnkiLCJib3VuZGFyeUtpbmRzIiwibGluZSIsImdldEN1cnJlbnRMaW5lIiwic3RhcnRzV2l0aCIsInJlcGVhdCIsImlzQXRFbmQiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFBQTs7QUFDQTs7SUFFTUEsTSxHQUFOLE1BQU1BLE1BQU4sQ0FBYTtBQUNYQyxjQUFZQyxjQUFaLEVBQTRCO0FBQzFCLFNBQUtDLEtBQUwsR0FBYUQsY0FBYjtBQUNEOztBQUVERSxrQkFBZ0I7QUFDZCxXQUFPLEtBQUtELEtBQUwsQ0FBV0UsTUFBWCxLQUFzQixDQUE3QjtBQUNEO0FBUFUsQztJQVVBQyxjLFdBQUFBLGMsR0FBTixNQUFNQSxjQUFOLENBQXFCO0FBQzFCTCxjQUFZTSxPQUFaLEVBQXFCQyxPQUFyQixFQUE4QkMsUUFBOUIsRUFBd0M7QUFDdEMsU0FBS0YsT0FBTCxHQUFlQSxPQUFmO0FBQ0EsU0FBS0MsT0FBTCxHQUFlQSxPQUFmO0FBQ0EsU0FBS0MsUUFBTCxHQUFnQkEsUUFBaEI7O0FBRUEsU0FBS0MsWUFBTCxHQUFvQixJQUFwQjtBQUNBLFNBQUtQLEtBQUwsR0FBYSxFQUFiOztBQUVBLFFBQUksS0FBS00sUUFBVCxFQUFtQjtBQUNqQixXQUFLTixLQUFMLENBQVdRLElBQVgsQ0FBZ0JDLFVBQVVBLE9BQU9DLGVBQVAsZ0JBQTRCLGdCQUE1QixDQUExQjtBQUNBLFdBQUtWLEtBQUwsQ0FBV1EsSUFBWCxDQUFnQkMsVUFBVUEsT0FBT0UscUJBQVAsRUFBMUI7QUFDQSxXQUFLWCxLQUFMLENBQVdRLElBQVgsQ0FBZ0JDLFVBQVVBLE9BQU9HLGVBQVAsbUJBQStCLGNBQS9CLENBQTFCO0FBQ0QsS0FKRCxNQUlPO0FBQ0wsV0FBS1osS0FBTCxDQUFXUSxJQUFYLENBQWdCQyxVQUFVQSxPQUFPQyxlQUFQLGdCQUE0QixjQUE1QixDQUExQjtBQUNBLFdBQUtWLEtBQUwsQ0FBV1EsSUFBWCxDQUFnQkMsVUFBVUEsT0FBT0UscUJBQVAsRUFBMUI7QUFDQSxXQUFLWCxLQUFMLENBQVdRLElBQVgsQ0FBZ0JDLFVBQVVBLE9BQU9HLGVBQVAsbUJBQStCLGdCQUEvQixDQUExQjtBQUNEO0FBQ0Y7O0FBRURDLGVBQWFDLE1BQWIsRUFBcUI7QUFDbkIsU0FBS2QsS0FBTCxHQUFhYyxPQUFPZCxLQUFwQjtBQUNBLFdBQU8sS0FBS2UsS0FBTCxFQUFQO0FBQ0Q7O0FBRURBLFVBQVE7QUFDTixTQUFLLElBQUlDLElBQUksQ0FBYixFQUFnQkEsSUFBSSxLQUFLaEIsS0FBTCxDQUFXRSxNQUEvQixFQUF1Q2MsR0FBdkMsRUFBNEM7QUFDMUMsVUFBSSxDQUFDLEtBQUtoQixLQUFMLENBQVdnQixDQUFYLEVBQWMsSUFBZCxDQUFMLEVBQTBCO0FBQ3hCLGVBQU8sSUFBSW5CLE1BQUosQ0FBVyxLQUFLRyxLQUFMLENBQVdpQixLQUFYLENBQWlCRCxDQUFqQixDQUFYLENBQVA7QUFDRDtBQUNGO0FBQ0QsV0FBTyxJQUFJbkIsTUFBSixDQUFXLEVBQVgsQ0FBUDtBQUNEOztBQUVEO0FBQ0FhLGtCQUFnQlEsUUFBaEIsRUFBMEJDLFdBQTFCLEVBQXVDO0FBQ3JDLFVBQU1DLGVBQWUsS0FBS2hCLE9BQUwsQ0FBYWlCLGFBQWIsRUFBckI7QUFDQSxTQUFLakIsT0FBTCxDQUFha0IsVUFBYjs7QUFFQSxRQUFJLEtBQUtDLGlCQUFMLENBQXVCLElBQXZCLE1BQWlDLElBQXJDLEVBQTJDO0FBQ3pDLGFBQU8sS0FBUDtBQUNEOztBQUVELFVBQU1DLGFBQWEsS0FBS3BCLE9BQUwsQ0FBYWlCLGFBQWIsRUFBbkI7O0FBRUEsU0FBS2hCLE9BQUwsQ0FBYWMsV0FBYixFQUEwQkQsUUFBMUIsRUFBb0NFLFlBQXBDLEVBQWtEQSxlQUFlLENBQWpFLEVBQW9FSSxVQUFwRTtBQUNBLFdBQU8sSUFBUDtBQUNEOztBQUVEO0FBQ0FiLDBCQUF3QjtBQUN0QixRQUFJLEtBQUtKLFlBQUwsS0FBc0IsR0FBMUIsRUFBK0I7QUFDN0IsVUFBSSxDQUFDLEtBQUtrQixhQUFMLEVBQUwsRUFBMkI7QUFDekIsZUFBTyxLQUFQO0FBQ0Q7QUFDRjs7QUFFRCxXQUFPLEtBQUtDLGNBQUwsRUFBUDtBQUNEOztBQUVEO0FBQ0FELGtCQUFnQjtBQUNkLFVBQU1MLGVBQWUsS0FBS2hCLE9BQUwsQ0FBYWlCLGFBQWIsRUFBckI7QUFDQSxTQUFLakIsT0FBTCxDQUFha0IsVUFBYjs7QUFFQSxRQUFJSyxJQUFJLEtBQUtKLGlCQUFMLENBQXVCLElBQXZCLENBQVI7QUFDQSxRQUFJSSxNQUFNLElBQVYsRUFBZ0I7QUFDZCxhQUFPLEtBQVA7QUFDRDs7QUFFRCxXQUFPQSxNQUFNLEdBQWIsRUFBa0I7QUFDaEI7QUFDQTtBQUNBLFlBQU1DLFlBQVksSUFBSXpCLGNBQUosQ0FBbUIsS0FBS0MsT0FBeEIsRUFBaUMsOEJBQWpDLEVBQW9ELEtBQUtFLFFBQXpELENBQWxCO0FBQ0EsVUFBSSxDQUFDc0IsVUFBVWIsS0FBVixHQUFrQmQsYUFBbEIsRUFBTCxFQUF3QztBQUN0QyxlQUFPLEtBQVA7QUFDRDs7QUFFRDBCLFVBQUksS0FBS0osaUJBQUwsQ0FBdUIsSUFBdkIsQ0FBSjtBQUNBLFVBQUlJLE1BQU0sSUFBVixFQUFnQjtBQUNkLGVBQU8sS0FBUDtBQUNEO0FBQ0Y7O0FBRUQsVUFBTUgsYUFBYSxLQUFLcEIsT0FBTCxDQUFhaUIsYUFBYixFQUFuQjtBQUNBLFNBQUtoQixPQUFMLENBQWFvQixhQUFiLENBQTJCTCxZQUEzQixFQUF5Q0EsZUFBZSxDQUF4RCxFQUEyREksVUFBM0Q7QUFDQSxXQUFPLElBQVA7QUFDRDs7QUFFRDtBQUNBRSxtQkFBaUI7QUFDZixVQUFNRyxjQUFjLEtBQUt6QixPQUFMLENBQWFpQixhQUFiLEVBQXBCO0FBQ0EsU0FBS2pCLE9BQUwsQ0FBYWtCLFVBQWI7QUFDQSxVQUFNUSxZQUFZLEtBQUsxQixPQUFMLENBQWFpQixhQUFiLEVBQWxCOztBQUVBLFNBQUtoQixPQUFMLENBQWFxQixjQUFiLENBQTRCRyxXQUE1QixFQUF5Q0MsU0FBekM7QUFDQSxXQUFPLElBQVA7QUFDRDs7QUFFRDtBQUNBbEIsa0JBQWdCTSxRQUFoQixFQUEwQkMsV0FBMUIsRUFBdUM7QUFDckMsVUFBTUMsZUFBZSxLQUFLaEIsT0FBTCxDQUFhaUIsYUFBYixFQUFyQjtBQUNBLFFBQUksS0FBS0UsaUJBQUwsQ0FBdUIsR0FBdkIsTUFBZ0MsSUFBcEMsRUFBMEM7QUFDeEMsYUFBTyxLQUFQO0FBQ0Q7O0FBRUQsU0FBS25CLE9BQUwsQ0FBYWtCLFVBQWI7QUFDQSxVQUFNRSxhQUFhLEtBQUtwQixPQUFMLENBQWFpQixhQUFiLEVBQW5COztBQUVBLFNBQUtoQixPQUFMLENBQWFjLFdBQWIsRUFBMEJELFFBQTFCLEVBQW9DTSxhQUFhLENBQWpELEVBQW9ESixZQUFwRCxFQUFrRUksYUFBYSxDQUEvRTtBQUNBLFdBQU8sSUFBUDtBQUNEOztBQUVEO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQU8saUJBQXFDO0FBQUEsUUFBeEJDLGFBQXdCLHVFQUFSLE1BQVE7O0FBQ25DLFVBQU1DLE9BQU8sS0FBSzdCLE9BQUwsQ0FBYThCLGNBQWIsRUFBYjtBQUNBLFNBQUssSUFBSWxCLElBQUksQ0FBYixFQUFnQkEsSUFBSWdCLGNBQWM5QixNQUFsQyxFQUEwQ2MsR0FBMUMsRUFBK0M7QUFDN0MsWUFBTVcsSUFBSUssY0FBY2hCLENBQWQsQ0FBVjtBQUNBLFVBQUlpQixLQUFLRSxVQUFMLENBQWdCUixFQUFFUyxNQUFGLENBQVMsQ0FBVCxDQUFoQixDQUFKLEVBQWtDO0FBQ2hDLGVBQU9ULENBQVA7QUFDRDtBQUNGO0FBQ0QsV0FBTyxJQUFQO0FBQ0Q7O0FBRUQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQUosc0JBQTBDO0FBQUEsUUFBeEJTLGFBQXdCLHVFQUFSLE1BQVE7O0FBQ3hDLFFBQUlMLElBQUksS0FBS0ksWUFBTCxDQUFrQkMsYUFBbEIsQ0FBUjtBQUNBLFdBQU9MLE1BQU0sSUFBYixFQUFtQjtBQUNqQixXQUFLdkIsT0FBTCxDQUFha0IsVUFBYjtBQUNBLFVBQUksS0FBS2xCLE9BQUwsQ0FBYWlDLE9BQWIsRUFBSixFQUE0QjtBQUMxQixlQUFPLElBQVA7QUFDRDtBQUNEVixVQUFJLEtBQUtJLFlBQUwsQ0FBa0JDLGFBQWxCLENBQUo7QUFDRDs7QUFFRCxTQUFLekIsWUFBTCxHQUFvQm9CLENBQXBCO0FBQ0EsV0FBT0EsQ0FBUDtBQUNEO0FBcEp5QixDIiwiZmlsZSI6ImluZGV4LmpzIiwic291cmNlUm9vdCI6Ii9ob21lL2FuZHJlaS9hdG9tLTEuMTkuMi9vdXQvYXBwL25vZGVfbW9kdWxlcy9naXRodWIiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge05vb3BWaXNpdG9yfSBmcm9tICcuL25vb3AtdmlzaXRvcic7XG5pbXBvcnQge1RPUCwgQk9UVE9NfSBmcm9tICcuLi9wb3NpdGlvbic7XG5cbmNsYXNzIFJlc3VsdCB7XG4gIGNvbnN0cnVjdG9yKHJlbWFpbmluZ1N0ZXBzKSB7XG4gICAgdGhpcy5zdGVwcyA9IHJlbWFpbmluZ1N0ZXBzO1xuICB9XG5cbiAgd2FzU3VjY2Vzc2Z1bCgpIHtcbiAgICByZXR1cm4gdGhpcy5zdGVwcy5sZW5ndGggPT09IDA7XG4gIH1cbn1cblxuZXhwb3J0IGNsYXNzIENvbmZsaWN0UGFyc2VyIHtcbiAgY29uc3RydWN0b3IoYWRhcHRlciwgdmlzaXRvciwgaXNSZWJhc2UpIHtcbiAgICB0aGlzLmFkYXB0ZXIgPSBhZGFwdGVyO1xuICAgIHRoaXMudmlzaXRvciA9IHZpc2l0b3I7XG4gICAgdGhpcy5pc1JlYmFzZSA9IGlzUmViYXNlO1xuXG4gICAgdGhpcy5sYXN0Qm91bmRhcnkgPSBudWxsO1xuICAgIHRoaXMuc3RlcHMgPSBbXTtcblxuICAgIGlmICh0aGlzLmlzUmViYXNlKSB7XG4gICAgICB0aGlzLnN0ZXBzLnB1c2gocGFyc2VyID0+IHBhcnNlci52aXNpdEhlYWRlclNpZGUoVE9QLCAndmlzaXRUaGVpclNpZGUnKSk7XG4gICAgICB0aGlzLnN0ZXBzLnB1c2gocGFyc2VyID0+IHBhcnNlci52aXNpdEJhc2VBbmRTZXBhcmF0b3IoKSk7XG4gICAgICB0aGlzLnN0ZXBzLnB1c2gocGFyc2VyID0+IHBhcnNlci52aXNpdEZvb3RlclNpZGUoQk9UVE9NLCAndmlzaXRPdXJTaWRlJykpO1xuICAgIH0gZWxzZSB7XG4gICAgICB0aGlzLnN0ZXBzLnB1c2gocGFyc2VyID0+IHBhcnNlci52aXNpdEhlYWRlclNpZGUoVE9QLCAndmlzaXRPdXJTaWRlJykpO1xuICAgICAgdGhpcy5zdGVwcy5wdXNoKHBhcnNlciA9PiBwYXJzZXIudmlzaXRCYXNlQW5kU2VwYXJhdG9yKCkpO1xuICAgICAgdGhpcy5zdGVwcy5wdXNoKHBhcnNlciA9PiBwYXJzZXIudmlzaXRGb290ZXJTaWRlKEJPVFRPTSwgJ3Zpc2l0VGhlaXJTaWRlJykpO1xuICAgIH1cbiAgfVxuXG4gIGNvbnRpbnVlRnJvbShyZXN1bHQpIHtcbiAgICB0aGlzLnN0ZXBzID0gcmVzdWx0LnN0ZXBzO1xuICAgIHJldHVybiB0aGlzLnBhcnNlKCk7XG4gIH1cblxuICBwYXJzZSgpIHtcbiAgICBmb3IgKGxldCBpID0gMDsgaSA8IHRoaXMuc3RlcHMubGVuZ3RoOyBpKyspIHtcbiAgICAgIGlmICghdGhpcy5zdGVwc1tpXSh0aGlzKSkge1xuICAgICAgICByZXR1cm4gbmV3IFJlc3VsdCh0aGlzLnN0ZXBzLnNsaWNlKGkpKTtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIG5ldyBSZXN1bHQoW10pO1xuICB9XG5cbiAgLy8gVmlzaXQgYSBzaWRlIHRoYXQgYmVnaW5zIHdpdGggYSBiYW5uZXIgYW5kIGRlc2NyaXB0aW9uIGFzIGl0cyBmaXJzdCBsaW5lLlxuICB2aXNpdEhlYWRlclNpZGUocG9zaXRpb24sIHZpc2l0TWV0aG9kKSB7XG4gICAgY29uc3Qgc2lkZVJvd1N0YXJ0ID0gdGhpcy5hZGFwdGVyLmdldEN1cnJlbnRSb3coKTtcbiAgICB0aGlzLmFkYXB0ZXIuYWR2YW5jZVJvdygpO1xuXG4gICAgaWYgKHRoaXMuYWR2YW5jZVRvQm91bmRhcnkoJ3w9JykgPT09IG51bGwpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICBjb25zdCBzaWRlUm93RW5kID0gdGhpcy5hZGFwdGVyLmdldEN1cnJlbnRSb3coKTtcblxuICAgIHRoaXMudmlzaXRvclt2aXNpdE1ldGhvZF0ocG9zaXRpb24sIHNpZGVSb3dTdGFydCwgc2lkZVJvd1N0YXJ0ICsgMSwgc2lkZVJvd0VuZCk7XG4gICAgcmV0dXJuIHRydWU7XG4gIH1cblxuICAvLyBWaXNpdCB0aGUgYmFzZSBzaWRlIGZyb20gZGlmZjMgb3V0cHV0LCBpZiBvbmUgaXMgcHJlc2VudCwgdGhlbiB2aXNpdCB0aGUgc2VwYXJhdG9yLlxuICB2aXNpdEJhc2VBbmRTZXBhcmF0b3IoKSB7XG4gICAgaWYgKHRoaXMubGFzdEJvdW5kYXJ5ID09PSAnfCcpIHtcbiAgICAgIGlmICghdGhpcy52aXNpdEJhc2VTaWRlKCkpIHtcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB0aGlzLnZpc2l0U2VwYXJhdG9yKCk7XG4gIH1cblxuICAvLyBWaXNpdCBhIGJhc2Ugc2lkZSBmcm9tIGRpZmYzIG91dHB1dC5cbiAgdmlzaXRCYXNlU2lkZSgpIHtcbiAgICBjb25zdCBzaWRlUm93U3RhcnQgPSB0aGlzLmFkYXB0ZXIuZ2V0Q3VycmVudFJvdygpO1xuICAgIHRoaXMuYWRhcHRlci5hZHZhbmNlUm93KCk7XG5cbiAgICBsZXQgYiA9IHRoaXMuYWR2YW5jZVRvQm91bmRhcnkoJzw9Jyk7XG4gICAgaWYgKGIgPT09IG51bGwpIHtcbiAgICAgIHJldHVybiBmYWxzZTtcbiAgICB9XG5cbiAgICB3aGlsZSAoYiA9PT0gJzwnKSB7XG4gICAgICAvLyBFbWJlZGRlZCByZWN1cnNpdmUgY29uZmxpY3Qgd2l0aGluIGEgYmFzZSBzaWRlLCBjYXVzZWQgYnkgYSBjcmlzcy1jcm9zcyBtZXJnZS5cbiAgICAgIC8vIEFkdmFuY2UgdGhlIGlucHV0IGFkYXB0ZXIgYmV5b25kIGl0IHdpdGhvdXQgbWFya2luZyBhbnl0aGluZy5cbiAgICAgIGNvbnN0IHN1YlBhcnNlciA9IG5ldyBDb25mbGljdFBhcnNlcih0aGlzLmFkYXB0ZXIsIG5ldyBOb29wVmlzaXRvcigpLCB0aGlzLmlzUmViYXNlKTtcbiAgICAgIGlmICghc3ViUGFyc2VyLnBhcnNlKCkud2FzU3VjY2Vzc2Z1bCgpKSB7XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH1cblxuICAgICAgYiA9IHRoaXMuYWR2YW5jZVRvQm91bmRhcnkoJzw9Jyk7XG4gICAgICBpZiAoYiA9PT0gbnVsbCkge1xuICAgICAgICByZXR1cm4gZmFsc2U7XG4gICAgICB9XG4gICAgfVxuXG4gICAgY29uc3Qgc2lkZVJvd0VuZCA9IHRoaXMuYWRhcHRlci5nZXRDdXJyZW50Um93KCk7XG4gICAgdGhpcy52aXNpdG9yLnZpc2l0QmFzZVNpZGUoc2lkZVJvd1N0YXJ0LCBzaWRlUm93U3RhcnQgKyAxLCBzaWRlUm93RW5kKTtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIC8vIFZpc2l0IGEgXCI9PT09PT09PVwiIHNlcGFyYXRvci5cbiAgdmlzaXRTZXBhcmF0b3IoKSB7XG4gICAgY29uc3Qgc2VwUm93U3RhcnQgPSB0aGlzLmFkYXB0ZXIuZ2V0Q3VycmVudFJvdygpO1xuICAgIHRoaXMuYWRhcHRlci5hZHZhbmNlUm93KCk7XG4gICAgY29uc3Qgc2VwUm93RW5kID0gdGhpcy5hZGFwdGVyLmdldEN1cnJlbnRSb3coKTtcblxuICAgIHRoaXMudmlzaXRvci52aXNpdFNlcGFyYXRvcihzZXBSb3dTdGFydCwgc2VwUm93RW5kKTtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIC8vIFZpc2l0IGEgc2lkZSB3aXRoIGEgYmFubmVyIGFuZCBkZXNjcmlwdGlvbiBhcyBpdHMgbGFzdCBsaW5lLlxuICB2aXNpdEZvb3RlclNpZGUocG9zaXRpb24sIHZpc2l0TWV0aG9kKSB7XG4gICAgY29uc3Qgc2lkZVJvd1N0YXJ0ID0gdGhpcy5hZGFwdGVyLmdldEN1cnJlbnRSb3coKTtcbiAgICBpZiAodGhpcy5hZHZhbmNlVG9Cb3VuZGFyeSgnPicpID09PSBudWxsKSB7XG4gICAgICByZXR1cm4gZmFsc2U7XG4gICAgfVxuXG4gICAgdGhpcy5hZGFwdGVyLmFkdmFuY2VSb3coKTtcbiAgICBjb25zdCBzaWRlUm93RW5kID0gdGhpcy5hZGFwdGVyLmdldEN1cnJlbnRSb3coKTtcblxuICAgIHRoaXMudmlzaXRvclt2aXNpdE1ldGhvZF0ocG9zaXRpb24sIHNpZGVSb3dFbmQgLSAxLCBzaWRlUm93U3RhcnQsIHNpZGVSb3dFbmQgLSAxKTtcbiAgICByZXR1cm4gdHJ1ZTtcbiAgfVxuXG4gIC8vIERldGVybWluZSBpZiB0aGUgY3VycmVudCByb3cgaXMgYSBzaWRlIGJvdW5kYXJ5LlxuICAvL1xuICAvLyBib3VuZGFyeUtpbmRzIC0gW1N0cmluZ10gYW55IGNvbWJpbmF0aW9uIG9mIDwsIHwsID0sIG9yID4gdG8gbGltaXQgdGhlIGtpbmRzIG9mIGJvdW5kYXJ5IGRldGVjdGVkLlxuICAvL1xuICAvLyBSZXR1cm5zIHRoZSBtYXRjaGluZyBib3VuZGFyeUtpbmRzIGNoYXJhY3Rlciwgb3IgYG51bGxgIGlmIG5vbmUgbWF0Y2guXG4gIGlzQXRCb3VuZGFyeShib3VuZGFyeUtpbmRzID0gJzx8PT4nKSB7XG4gICAgY29uc3QgbGluZSA9IHRoaXMuYWRhcHRlci5nZXRDdXJyZW50TGluZSgpO1xuICAgIGZvciAobGV0IGkgPSAwOyBpIDwgYm91bmRhcnlLaW5kcy5sZW5ndGg7IGkrKykge1xuICAgICAgY29uc3QgYiA9IGJvdW5kYXJ5S2luZHNbaV07XG4gICAgICBpZiAobGluZS5zdGFydHNXaXRoKGIucmVwZWF0KDcpKSkge1xuICAgICAgICByZXR1cm4gYjtcbiAgICAgIH1cbiAgICB9XG4gICAgcmV0dXJuIG51bGw7XG4gIH1cblxuICAvLyBJbmNyZW1lbnQgdGhlIGN1cnJlbnQgcm93IHVudGlsIHRoZSBjdXJyZW50IGxpbmUgbWF0Y2hlcyBvbmUgb2YgdGhlIHByb3ZpZGVkIGJvdW5kYXJ5IGtpbmRzLCBvciB1bnRpbCB0aGVyZSBhcmUgbm9cbiAgLy8gbW9yZSBsaW5lcyBpbiB0aGUgZWRpdG9yLlxuICAvL1xuICAvLyBib3VuZGFyeUtpbmRzIC0gW1N0cmluZ10gYW55IGNvbWJpbmF0aW9uIG9mIDwsIHwsID0sIG9yID4gdG8gbGltaXQgdGhlIGtpbmRzIG9mIGJvdW5kYXJpZXMgdGhhdCBoYWx0IHRoZVxuICAvLyAgIHByb2dyZXNzaW9uLlxuICAvL1xuICAvLyBSZXR1cm5zIHRoZSBtYXRjaGluZyBib3VuZGFyeUtpbmRzIGNoYXJhY3Rlciwgb3IgJ251bGwnIGlmIHRoZXJlIGFyZSBubyBtYXRjaGVzIHRvIHRoZSBlbmQgb2YgdGhlIGVkaXRvci5cbiAgYWR2YW5jZVRvQm91bmRhcnkoYm91bmRhcnlLaW5kcyA9ICc8fD0+Jykge1xuICAgIGxldCBiID0gdGhpcy5pc0F0Qm91bmRhcnkoYm91bmRhcnlLaW5kcyk7XG4gICAgd2hpbGUgKGIgPT09IG51bGwpIHtcbiAgICAgIHRoaXMuYWRhcHRlci5hZHZhbmNlUm93KCk7XG4gICAgICBpZiAodGhpcy5hZGFwdGVyLmlzQXRFbmQoKSkge1xuICAgICAgICByZXR1cm4gbnVsbDtcbiAgICAgIH1cbiAgICAgIGIgPSB0aGlzLmlzQXRCb3VuZGFyeShib3VuZGFyeUtpbmRzKTtcbiAgICB9XG5cbiAgICB0aGlzLmxhc3RCb3VuZGFyeSA9IGI7XG4gICAgcmV0dXJuIGI7XG4gIH1cbn1cbiJdfQ==