'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _desc, _value, _class, _class2, _temp;

var _eventKit = require('event-kit');

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _propTypes = require('prop-types');

var _propTypes2 = _interopRequireDefault(_propTypes);

var _coreDecorators = require('core-decorators');

var _compareSets2 = require('compare-sets');

var _compareSets3 = _interopRequireDefault(_compareSets2);

var _commands = require('../views/commands');

var _commands2 = _interopRequireDefault(_commands);

var _conflict = require('../models/conflicts/conflict');

var _conflict2 = _interopRequireDefault(_conflict);

var _conflictController = require('./conflict-controller');

var _conflictController2 = _interopRequireDefault(_conflictController);

var _source = require('../models/conflicts/source');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _toArray(arr) { return Array.isArray(arr) ? arr : Array.from(arr); }

function _applyDecoratedDescriptor(target, property, decorators, descriptor, context) {
  var desc = {};
  Object['ke' + 'ys'](descriptor).forEach(function (key) {
    desc[key] = descriptor[key];
  });
  desc.enumerable = !!desc.enumerable;
  desc.configurable = !!desc.configurable;

  if ('value' in desc || desc.initializer) {
    desc.writable = true;
  }

  desc = decorators.slice().reverse().reduce(function (desc, decorator) {
    return decorator(target, property, desc) || desc;
  }, desc);

  if (context && desc.initializer !== void 0) {
    desc.value = desc.initializer ? desc.initializer.call(context) : void 0;
    desc.initializer = undefined;
  }

  if (desc.initializer === void 0) {
    Object['define' + 'Property'](target, property, desc);
    desc = null;
  }

  return desc;
}

/**
 * Render a `ConflictController` for each conflict marker within an open TextEditor.
 */
let EditorConflictController = (_class = (_temp = _class2 = class EditorConflictController extends _react2.default.Component {

  constructor(props, context) {
    super(props, context);

    // this.layer = props.editor.addMarkerLayer({
    //   maintainHistory: true,
    //   persistent: false,
    // });

    this.layer = props.editor.getDefaultMarkerLayer();

    this.state = {
      conflicts: new Set(_conflict2.default.allFromEditor(props.editor, this.layer, props.isRebase))
    };

    this.subscriptions = new _eventKit.CompositeDisposable();

    this.updateMarkerCount();
  }

  componentDidMount() {
    const buffer = this.props.editor.getBuffer();

    this.subscriptions.add(this.props.editor.onDidStopChanging(() => this.forceUpdate()), this.props.editor.onDidDestroy(() => this.props.refreshResolutionProgress(this.props.editor.getPath())), buffer.onDidReload(() => this.reparseConflicts()));
  }

  render() {
    this.updateMarkerCount();

    return _react2.default.createElement(
      'div',
      null,
      this.state.conflicts.size > 0 && _react2.default.createElement(
        _commands2.default,
        { registry: this.props.commandRegistry, target: 'atom-text-editor' },
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-ours', callback: this.getResolverUsing([_source.OURS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-theirs', callback: this.getResolverUsing([_source.THEIRS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-base', callback: this.getResolverUsing([_source.BASE]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-ours-then-theirs', callback: this.getResolverUsing([_source.OURS, _source.THEIRS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-theirs-then-ours', callback: this.getResolverUsing([_source.THEIRS, _source.OURS]) }),
        _react2.default.createElement(_commands.Command, { command: 'github:resolve-as-current', callback: this.resolveAsCurrent }),
        _react2.default.createElement(_commands.Command, { command: 'github:revert-conflict-modifications', callback: this.revertConflictModifications }),
        _react2.default.createElement(_commands.Command, { command: 'github:dismiss-conflict', callback: this.dismissCurrent })
      ),
      Array.from(this.state.conflicts, c => _react2.default.createElement(_conflictController2.default, {
        key: c.getKey(),
        editor: this.props.editor,
        conflict: c,
        resolveAsSequence: sources => this.resolveAsSequence(c, sources),
        dismiss: () => this.dismissConflicts([c])
      }))
    );
  }

  componentWillUnmount() {
    // this.layer.destroy();
    this.subscriptions.dispose();
  }

  /*
   * Return an Array containing `Conflict` objects whose marked regions include any cursor position in the current
   * `TextEditor` and the `Sides` that contain a cursor within each.
   *
   * This method is written to have linear complexity with respect to the number of cursors and the number of
   * conflicts, to gracefully handle files with large numbers of both.
   */
  getCurrentConflicts() {
    const cursorPositions = this.props.editor.getCursorBufferPositions();
    cursorPositions.sort((a, b) => a.compare(b));
    const cursorIterator = cursorPositions[Symbol.iterator]();

    const conflictIterator = this.state.conflicts.keys();

    let currentCursor = cursorIterator.next();
    let currentConflict = conflictIterator.next();
    const activeConflicts = [];

    while (!currentCursor.done && !currentConflict.done) {
      // Advance currentCursor to the first cursor beyond the earliest conflict.
      const earliestConflictPosition = currentConflict.value.getRange().start;
      while (!currentCursor.done && currentCursor.value.isLessThan(earliestConflictPosition)) {
        currentCursor = cursorIterator.next();
      }

      // Advance currentConflict until the first conflict that begins at a position after the current cursor.
      // Compare each to the current cursor, and add it to activeConflicts if it contains it.
      while (!currentConflict.done && !currentCursor.done && currentConflict.value.getRange().start.isLessThan(currentCursor.value)) {
        if (currentConflict.value.includesPoint(currentCursor.value)) {
          // Hit; determine which sides of this conflict contain cursors.
          const conflict = currentConflict.value;
          const endPosition = conflict.getRange().end;
          const sides = new Set();
          while (!currentCursor.done && currentCursor.value.isLessThan(endPosition)) {
            const side = conflict.getSideContaining(currentCursor.value);
            if (side) {
              sides.add(side);
            }
            currentCursor = cursorIterator.next();
          }

          activeConflicts.push({ conflict, sides });
        }

        currentConflict = conflictIterator.next();
      }
    }

    return activeConflicts;
  }

  getResolverUsing(sequence) {
    return () => {
      this.getCurrentConflicts().forEach(match => this.resolveAsSequence(match.conflict, sequence));
    };
  }

  resolveAsCurrent() {
    this.getCurrentConflicts().forEach(match => {
      if (match.sides.size === 1) {
        const side = match.sides.keys().next().value;
        this.resolveAs(match.conflict, side.getSource());
      }
    });
  }

  revertConflictModifications() {
    this.getCurrentConflicts().forEach(match => {
      match.sides.forEach(side => {
        side.isModified() && side.revert();
        side.isBannerModified() && side.revertBanner();
      });
    });
  }

  dismissCurrent() {
    this.dismissConflicts(this.getCurrentConflicts().map(match => match.conflict));
  }

  dismissConflicts(conflicts) {
    this.setState((prevState, props) => {
      var _compareSets = (0, _compareSets3.default)(new Set(conflicts), prevState.conflicts);

      const added = _compareSets.added;

      return { conflicts: added };
    });
  }

  resolveAsSequence(conflict, sources) {
    var _sources$map$filter = sources.map(source => conflict.getSide(source)).filter(side => side),
        _sources$map$filter2 = _toArray(_sources$map$filter);

    const firstSide = _sources$map$filter2[0],
          restOfSides = _sources$map$filter2.slice(1);

    const textToAppend = restOfSides.map(side => side.getText()).join('');

    this.props.editor.transact(() => {
      // Append text from all but the first Side to the first Side. Adjust the following DisplayMarker so that only that
      // Side's marker includes the appended text, not the next one.
      const appendedRange = firstSide.appendText(textToAppend);
      const nextMarker = conflict.markerAfter(firstSide.getPosition());
      if (nextMarker) {
        nextMarker.setTailBufferPosition(appendedRange.end);
      }

      this.innerResolveAs(conflict, sources[0]);
    });
  }

  resolveAs(conflict, source) {
    this.props.editor.transact(() => {
      this.innerResolveAs(conflict, source);
    });
  }

  innerResolveAs(conflict, source) {
    conflict.resolveAs(source);

    const chosenSide = conflict.getChosenSide();
    if (!chosenSide.isBannerModified()) {
      chosenSide.deleteBanner();
    }

    const separator = conflict.getSeparator();
    if (!separator.isModified()) {
      separator.delete();
    }

    conflict.getUnchosenSides().forEach(side => {
      side.deleteBanner();
      side.delete();
    });

    this.updateMarkerCount();
  }

  reparseConflicts() {
    const newConflicts = new Set(_conflict2.default.allFromEditor(this.props.editor, this.layer, this.props.isRebase));
    this.setState({ conflicts: newConflicts });
  }

  updateMarkerCount() {
    this.props.resolutionProgress.reportMarkerCount(this.props.editor.getPath(), Array.from(this.state.conflicts, c => !c.isResolved()).filter(b => b).length);
  }
}, _class2.propTypes = {
  editor: _propTypes2.default.object.isRequired,
  commandRegistry: _propTypes2.default.object.isRequired,
  resolutionProgress: _propTypes2.default.object.isRequired,
  isRebase: _propTypes2.default.bool.isRequired,
  refreshResolutionProgress: _propTypes2.default.func
}, _class2.defaultProps = {
  refreshResolutionProgress: () => {}
}, _temp), (_applyDecoratedDescriptor(_class.prototype, 'resolveAsCurrent', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'resolveAsCurrent'), _class.prototype), _applyDecoratedDescriptor(_class.prototype, 'revertConflictModifications', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'revertConflictModifications'), _class.prototype), _applyDecoratedDescriptor(_class.prototype, 'dismissCurrent', [_coreDecorators.autobind], Object.getOwnPropertyDescriptor(_class.prototype, 'dismissCurrent'), _class.prototype)), _class);
exports.default = EditorConflictController;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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