(function() {
  var CompositeDisposable, GitDiffView, repositoryForPath,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  CompositeDisposable = require('atom').CompositeDisposable;

  repositoryForPath = require('./helpers').repositoryForPath;

  module.exports = GitDiffView = (function() {
    function GitDiffView(editor) {
      var editorElement, editorView;
      this.editor = editor;
      this.updateDiffs = bind(this.updateDiffs, this);
      this.subscriptions = new CompositeDisposable();
      this.decorations = {};
      this.markers = [];
      this.subscriptions.add(this.editor.onDidStopChanging(this.updateDiffs));
      this.subscriptions.add(this.editor.onDidChangePath(this.updateDiffs));
      this.subscribeToRepository();
      this.subscriptions.add(atom.project.onDidChangePaths((function(_this) {
        return function() {
          return _this.subscribeToRepository();
        };
      })(this)));
      this.subscriptions.add(this.editor.onDidDestroy((function(_this) {
        return function() {
          _this.cancelUpdate();
          _this.removeDecorations();
          return _this.subscriptions.dispose();
        };
      })(this)));
      editorView = atom.views.getView(this.editor);
      this.subscriptions.add(atom.commands.add(editorView, 'git-diff:move-to-next-diff', (function(_this) {
        return function() {
          return _this.moveToNextDiff();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorView, 'git-diff:move-to-previous-diff', (function(_this) {
        return function() {
          return _this.moveToPreviousDiff();
        };
      })(this)));
      this.subscriptions.add(atom.config.onDidChange('git-diff.showIconsInEditorGutter', (function(_this) {
        return function() {
          return _this.updateIconDecoration();
        };
      })(this)));
      this.subscriptions.add(atom.config.onDidChange('editor.showLineNumbers', (function(_this) {
        return function() {
          return _this.updateIconDecoration();
        };
      })(this)));
      editorElement = atom.views.getView(this.editor);
      this.subscriptions.add(editorElement.onDidAttach((function(_this) {
        return function() {
          return _this.updateIconDecoration();
        };
      })(this)));
      this.updateIconDecoration();
      this.scheduleUpdate();
    }

    GitDiffView.prototype.moveToNextDiff = function() {
      var cursorLineNumber, firstDiffLineNumber, i, len, newStart, nextDiffLineNumber, ref, ref1;
      cursorLineNumber = this.editor.getCursorBufferPosition().row + 1;
      nextDiffLineNumber = null;
      firstDiffLineNumber = null;
      ref1 = (ref = this.diffs) != null ? ref : [];
      for (i = 0, len = ref1.length; i < len; i++) {
        newStart = ref1[i].newStart;
        if (newStart > cursorLineNumber) {
          if (nextDiffLineNumber == null) {
            nextDiffLineNumber = newStart - 1;
          }
          nextDiffLineNumber = Math.min(newStart - 1, nextDiffLineNumber);
        }
        if (firstDiffLineNumber == null) {
          firstDiffLineNumber = newStart - 1;
        }
        firstDiffLineNumber = Math.min(newStart - 1, firstDiffLineNumber);
      }
      if (nextDiffLineNumber == null) {
        nextDiffLineNumber = firstDiffLineNumber;
      }
      return this.moveToLineNumber(nextDiffLineNumber);
    };

    GitDiffView.prototype.updateIconDecoration = function() {
      var gutter;
      gutter = atom.views.getView(this.editor).querySelector('.gutter');
      if (atom.config.get('editor.showLineNumbers') && atom.config.get('git-diff.showIconsInEditorGutter')) {
        return gutter != null ? gutter.classList.add('git-diff-icon') : void 0;
      } else {
        return gutter != null ? gutter.classList.remove('git-diff-icon') : void 0;
      }
    };

    GitDiffView.prototype.moveToPreviousDiff = function() {
      var cursorLineNumber, i, lastDiffLineNumber, len, newStart, previousDiffLineNumber, ref, ref1;
      cursorLineNumber = this.editor.getCursorBufferPosition().row + 1;
      previousDiffLineNumber = -1;
      lastDiffLineNumber = -1;
      ref1 = (ref = this.diffs) != null ? ref : [];
      for (i = 0, len = ref1.length; i < len; i++) {
        newStart = ref1[i].newStart;
        if (newStart < cursorLineNumber) {
          previousDiffLineNumber = Math.max(newStart - 1, previousDiffLineNumber);
        }
        lastDiffLineNumber = Math.max(newStart - 1, lastDiffLineNumber);
      }
      if (previousDiffLineNumber === -1) {
        previousDiffLineNumber = lastDiffLineNumber;
      }
      return this.moveToLineNumber(previousDiffLineNumber);
    };

    GitDiffView.prototype.moveToLineNumber = function(lineNumber) {
      if (lineNumber == null) {
        lineNumber = -1;
      }
      if (lineNumber >= 0) {
        this.editor.setCursorBufferPosition([lineNumber, 0]);
        return this.editor.moveToFirstCharacterOfLine();
      }
    };

    GitDiffView.prototype.subscribeToRepository = function() {
      if (this.repository = repositoryForPath(this.editor.getPath())) {
        this.subscriptions.add(this.repository.onDidChangeStatuses((function(_this) {
          return function() {
            return _this.scheduleUpdate();
          };
        })(this)));
        return this.subscriptions.add(this.repository.onDidChangeStatus((function(_this) {
          return function(changedPath) {
            if (changedPath === _this.editor.getPath()) {
              return _this.scheduleUpdate();
            }
          };
        })(this)));
      }
    };

    GitDiffView.prototype.cancelUpdate = function() {
      return clearImmediate(this.immediateId);
    };

    GitDiffView.prototype.scheduleUpdate = function() {
      this.cancelUpdate();
      return this.immediateId = setImmediate(this.updateDiffs);
    };

    GitDiffView.prototype.updateDiffs = function() {
      var path, ref, ref1;
      if (this.editor.isDestroyed()) {
        return;
      }
      this.removeDecorations();
      if (path = (ref = this.editor) != null ? ref.getPath() : void 0) {
        if (this.diffs = (ref1 = this.repository) != null ? ref1.getLineDiffs(path, this.editor.getText()) : void 0) {
          return this.addDecorations(this.diffs);
        }
      }
    };

    GitDiffView.prototype.addDecorations = function(diffs) {
      var endRow, i, len, newLines, newStart, oldLines, ref, startRow;
      for (i = 0, len = diffs.length; i < len; i++) {
        ref = diffs[i], newStart = ref.newStart, oldLines = ref.oldLines, newLines = ref.newLines;
        startRow = newStart - 1;
        endRow = newStart + newLines - 1;
        if (oldLines === 0 && newLines > 0) {
          this.markRange(startRow, endRow, 'git-line-added');
        } else if (newLines === 0 && oldLines > 0) {
          this.markRange(startRow, startRow, 'git-line-removed');
        } else {
          this.markRange(startRow, endRow, 'git-line-modified');
        }
      }
    };

    GitDiffView.prototype.removeDecorations = function() {
      var i, len, marker, ref;
      ref = this.markers;
      for (i = 0, len = ref.length; i < len; i++) {
        marker = ref[i];
        marker.destroy();
      }
      return this.markers = [];
    };

    GitDiffView.prototype.markRange = function(startRow, endRow, klass) {
      var marker;
      marker = this.editor.markBufferRange([[startRow, 0], [endRow, 0]], {
        invalidate: 'never'
      });
      this.editor.decorateMarker(marker, {
        type: 'line-number',
        "class": klass
      });
      return this.markers.push(marker);
    };

    return GitDiffView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
