(function() {
  var COMMENT_SELECTOR_REGEX, Range, SelfClosingTags, TAG_SELECTOR_REGEX, TagFinder, _, generateTagStartOrEndRegex, tagStartOrEndRegex;

  Range = require('atom').Range;

  _ = require('underscore-plus');

  SelfClosingTags = require('./self-closing-tags');

  TAG_SELECTOR_REGEX = /(\b|\.)(meta\.tag|punctuation\.definition\.tag)/;

  COMMENT_SELECTOR_REGEX = /(\b|\.)comment/;

  generateTagStartOrEndRegex = function(tagNameRegexStr) {
    var notSelfClosingTagEnd, re;
    notSelfClosingTagEnd = "(?:[^>\\/\"']|\"[^\"]*\"|'[^']*')*>";
    return re = new RegExp("<(" + tagNameRegexStr + ")" + notSelfClosingTagEnd + "|<\\/(" + tagNameRegexStr + ")>");
  };

  tagStartOrEndRegex = generateTagStartOrEndRegex("\\w[-\\w]*(?:\\:\\w[-\\w]*)?");

  module.exports = TagFinder = (function() {
    function TagFinder(editor) {
      this.editor = editor;
      this.tagPattern = /(<(\/?))([^\s>]+)([\s>]|$)/;
      this.wordRegex = /[^>\r\n]*/;
    }

    TagFinder.prototype.patternForTagName = function(tagName) {
      tagName = _.escapeRegExp(tagName);
      return new RegExp("(<" + tagName + "([\\s>]|$))|(</" + tagName + ">)", 'gi');
    };

    TagFinder.prototype.isRangeCommented = function(range) {
      return this.scopesForPositionMatchRegex(range.start, COMMENT_SELECTOR_REGEX);
    };

    TagFinder.prototype.isTagRange = function(range) {
      return this.scopesForPositionMatchRegex(range.start, TAG_SELECTOR_REGEX);
    };

    TagFinder.prototype.isCursorOnTag = function() {
      return this.scopesForPositionMatchRegex(this.editor.getCursorBufferPosition(), TAG_SELECTOR_REGEX);
    };

    TagFinder.prototype.scopesForPositionMatchRegex = function(position, regex) {
      var buffer, column, grammar, i, len, line, lineLength, nextColumn, ref, ref1, scopeIds, tag, tokenizedBuffer;
      ref = this.editor, tokenizedBuffer = ref.tokenizedBuffer, buffer = ref.buffer;
      grammar = tokenizedBuffer.grammar;
      column = 0;
      line = tokenizedBuffer.tokenizedLineForRow(position.row);
      if (line == null) {
        return false;
      }
      lineLength = buffer.lineLengthForRow(position.row);
      scopeIds = line.openScopes.slice();
      ref1 = line.tags;
      for (i = 0, len = ref1.length; i < len; i += 1) {
        tag = ref1[i];
        if (tag >= 0) {
          nextColumn = column + tag;
          if (nextColumn > position.column || nextColumn === lineLength) {
            break;
          }
          column = nextColumn;
        } else if ((tag & 1) === 1) {
          scopeIds.push(tag);
        } else {
          scopeIds.pop();
        }
      }
      return scopeIds.some(function(scopeId) {
        return regex.test(grammar.scopeForId(scopeId));
      });
    };

    TagFinder.prototype.findStartTag = function(tagName, endPosition) {
      var pattern, scanRange, startRange, unpairedCount;
      scanRange = new Range([0, 0], endPosition);
      pattern = this.patternForTagName(tagName);
      startRange = null;
      unpairedCount = 0;
      this.editor.backwardsScanInBufferRange(pattern, scanRange, (function(_this) {
        return function(arg) {
          var match, range, stop;
          match = arg.match, range = arg.range, stop = arg.stop;
          if (_this.isRangeCommented(range)) {
            return;
          }
          if (match[1]) {
            unpairedCount--;
            if (unpairedCount < 0) {
              startRange = range.translate([0, 1], [0, -match[2].length]);
              return stop();
            }
          } else {
            return unpairedCount++;
          }
        };
      })(this));
      return startRange;
    };

    TagFinder.prototype.findEndTag = function(tagName, startPosition) {
      var endRange, pattern, scanRange, unpairedCount;
      scanRange = new Range(startPosition, this.editor.buffer.getEndPosition());
      pattern = this.patternForTagName(tagName);
      endRange = null;
      unpairedCount = 0;
      this.editor.scanInBufferRange(pattern, scanRange, (function(_this) {
        return function(arg) {
          var match, range, stop;
          match = arg.match, range = arg.range, stop = arg.stop;
          if (_this.isRangeCommented(range)) {
            return;
          }
          if (match[1]) {
            return unpairedCount++;
          } else {
            unpairedCount--;
            if (unpairedCount < 0) {
              endRange = range.translate([0, 2], [0, -1]);
              return stop();
            }
          }
        };
      })(this));
      return endRange;
    };

    TagFinder.prototype.findStartEndTags = function() {
      var endPosition, ranges;
      ranges = null;
      endPosition = this.editor.getLastCursor().getCurrentWordBufferRange({
        wordRegex: this.wordRegex
      }).end;
      this.editor.backwardsScanInBufferRange(this.tagPattern, [[0, 0], endPosition], (function(_this) {
        return function(arg) {
          var endRange, entireMatch, isClosingTag, match, prefix, range, startRange, stop, suffix, tagName;
          match = arg.match, range = arg.range, stop = arg.stop;
          stop();
          entireMatch = match[0], prefix = match[1], isClosingTag = match[2], tagName = match[3], suffix = match[4];
          if (range.start.row === range.end.row) {
            startRange = range.translate([0, prefix.length], [0, -suffix.length]);
          } else {
            startRange = Range.fromObject([range.start.translate([0, prefix.length]), [range.start.row, 2e308]]);
          }
          if (isClosingTag) {
            endRange = _this.findStartTag(tagName, startRange.start);
          } else {
            endRange = _this.findEndTag(tagName, startRange.end);
          }
          if ((startRange != null) && (endRange != null)) {
            return ranges = {
              startRange: startRange,
              endRange: endRange
            };
          }
        };
      })(this));
      return ranges;
    };

    TagFinder.prototype.findEnclosingTags = function() {
      var ranges;
      if (ranges = this.findStartEndTags()) {
        if (this.isTagRange(ranges.startRange) && this.isTagRange(ranges.endRange)) {
          return ranges;
        }
      }
      return null;
    };

    TagFinder.prototype.findMatchingTags = function() {
      if (this.isCursorOnTag()) {
        return this.findStartEndTags();
      }
    };

    TagFinder.prototype.parseFragment = function(fragment, stack, matchExpr, cond) {
      var match, topElem;
      match = fragment.match(matchExpr);
      while (match && cond(stack)) {
        if (SelfClosingTags.indexOf(match[1]) === -1) {
          topElem = stack[stack.length - 1];
          if (match[2] && topElem === match[2]) {
            stack.pop();
          } else if (match[1]) {
            stack.push(match[1]);
          }
        }
        fragment = fragment.substr(match.index + match[0].length);
        match = fragment.match(matchExpr);
      }
      return stack;
    };

    TagFinder.prototype.tagsNotClosedInFragment = function(fragment) {
      return this.parseFragment(fragment, [], tagStartOrEndRegex, function() {
        return true;
      });
    };

    TagFinder.prototype.tagDoesNotCloseInFragment = function(tags, fragment) {
      var escapedTag, stack, stackLength, tag;
      if (tags.length === 0) {
        return false;
      }
      stack = tags;
      stackLength = stack.length;
      tag = tags[tags.length - 1];
      escapedTag = _.escapeRegExp(tag);
      stack = this.parseFragment(fragment, stack, generateTagStartOrEndRegex(escapedTag), function(s) {
        return s.length >= stackLength || s[s.length - 1] === tag;
      });
      return stack.length > 0 && stack[stack.length - 1] === tag;
    };

    TagFinder.prototype.closingTagForFragments = function(preFragment, postFragment) {
      var tag, tags;
      tags = this.tagsNotClosedInFragment(preFragment);
      tag = tags[tags.length - 1];
      if (this.tagDoesNotCloseInFragment(tags, postFragment)) {
        return tag;
      } else {
        return null;
      }
    };

    return TagFinder;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
