(function() {
  var CSON, Emitter, Package, PackageManager, ServiceHub, ThemePackage, _, fs, getDeprecatedPackageMetadata, isDeprecatedPackage, normalizePackageData, packageJSON, path, ref,
    slice = [].slice,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; };

  path = require('path');

  normalizePackageData = null;

  _ = require('underscore-plus');

  Emitter = require('event-kit').Emitter;

  fs = require('fs-plus');

  CSON = require('season');

  ServiceHub = require('service-hub');

  Package = require('./package');

  ThemePackage = require('./theme-package');

  ref = require('./deprecated-packages'), isDeprecatedPackage = ref.isDeprecatedPackage, getDeprecatedPackageMetadata = ref.getDeprecatedPackageMetadata;

  packageJSON = require('../package.json');

  module.exports = PackageManager = (function() {
    function PackageManager(params) {
      var ref1, ref2;
      this.config = params.config, this.styleManager = params.styleManager, this.notificationManager = params.notificationManager, this.keymapManager = params.keymapManager, this.commandRegistry = params.commandRegistry, this.grammarRegistry = params.grammarRegistry, this.deserializerManager = params.deserializerManager, this.viewRegistry = params.viewRegistry;
      this.emitter = new Emitter;
      this.activationHookEmitter = new Emitter;
      this.packageDirPaths = [];
      this.deferredActivationHooks = [];
      this.triggeredActivationHooks = new Set();
      this.packagesCache = (ref1 = packageJSON._atomPackages) != null ? ref1 : {};
      this.packageDependencies = (ref2 = packageJSON.packageDependencies) != null ? ref2 : {};
      this.initialPackagesLoaded = false;
      this.initialPackagesActivated = false;
      this.preloadedPackages = {};
      this.loadedPackages = {};
      this.activePackages = {};
      this.activatingPackages = {};
      this.packageStates = {};
      this.serviceHub = new ServiceHub;
      this.packageActivators = [];
      this.registerPackageActivator(this, ['atom', 'textmate']);
    }

    PackageManager.prototype.initialize = function(params) {
      var configDirPath, safeMode;
      configDirPath = params.configDirPath, this.devMode = params.devMode, safeMode = params.safeMode, this.resourcePath = params.resourcePath;
      if ((configDirPath != null) && !safeMode) {
        if (this.devMode) {
          this.packageDirPaths.push(path.join(configDirPath, "dev", "packages"));
        }
        return this.packageDirPaths.push(path.join(configDirPath, "packages"));
      }
    };

    PackageManager.prototype.setContextMenuManager = function(contextMenuManager) {
      this.contextMenuManager = contextMenuManager;
    };

    PackageManager.prototype.setMenuManager = function(menuManager) {
      this.menuManager = menuManager;
    };

    PackageManager.prototype.setThemeManager = function(themeManager) {
      this.themeManager = themeManager;
    };

    PackageManager.prototype.reset = function() {
      var ref1, ref2;
      this.serviceHub.clear();
      this.deactivatePackages();
      this.loadedPackages = {};
      this.preloadedPackages = {};
      this.packageStates = {};
      this.packagesCache = (ref1 = packageJSON._atomPackages) != null ? ref1 : {};
      this.packageDependencies = (ref2 = packageJSON.packageDependencies) != null ? ref2 : {};
      return this.triggeredActivationHooks.clear();
    };


    /*
    Section: Event Subscription
     */

    PackageManager.prototype.onDidLoadInitialPackages = function(callback) {
      return this.emitter.on('did-load-initial-packages', callback);
    };

    PackageManager.prototype.onDidActivateInitialPackages = function(callback) {
      return this.emitter.on('did-activate-initial-packages', callback);
    };

    PackageManager.prototype.onDidActivatePackage = function(callback) {
      return this.emitter.on('did-activate-package', callback);
    };

    PackageManager.prototype.onDidDeactivatePackage = function(callback) {
      return this.emitter.on('did-deactivate-package', callback);
    };

    PackageManager.prototype.onDidLoadPackage = function(callback) {
      return this.emitter.on('did-load-package', callback);
    };

    PackageManager.prototype.onDidUnloadPackage = function(callback) {
      return this.emitter.on('did-unload-package', callback);
    };


    /*
    Section: Package system data
     */

    PackageManager.prototype.getApmPath = function() {
      var apmRoot, commandName, configPath;
      configPath = atom.config.get('core.apmPath');
      if (configPath) {
        return configPath;
      }
      if (this.apmPath != null) {
        return this.apmPath;
      }
      commandName = 'apm';
      if (process.platform === 'win32') {
        commandName += '.cmd';
      }
      apmRoot = path.join(process.resourcesPath, 'app', 'apm');
      this.apmPath = path.join(apmRoot, 'bin', commandName);
      if (!fs.isFileSync(this.apmPath)) {
        this.apmPath = path.join(apmRoot, 'node_modules', 'atom-package-manager', 'bin', commandName);
      }
      return this.apmPath;
    };

    PackageManager.prototype.getPackageDirPaths = function() {
      return _.clone(this.packageDirPaths);
    };


    /*
    Section: General package data
     */

    PackageManager.prototype.resolvePackagePath = function(name) {
      var packagePath;
      if (fs.isDirectorySync(name)) {
        return name;
      }
      packagePath = fs.resolve.apply(fs, slice.call(this.packageDirPaths).concat([name]));
      if (fs.isDirectorySync(packagePath)) {
        return packagePath;
      }
      packagePath = path.join(this.resourcePath, 'node_modules', name);
      if (this.hasAtomEngine(packagePath)) {
        return packagePath;
      }
    };

    PackageManager.prototype.isBundledPackage = function(name) {
      return this.getPackageDependencies().hasOwnProperty(name);
    };

    PackageManager.prototype.isDeprecatedPackage = function(name, version) {
      return isDeprecatedPackage(name, version);
    };

    PackageManager.prototype.getDeprecatedPackageMetadata = function(name) {
      return getDeprecatedPackageMetadata(name);
    };


    /*
    Section: Enabling and disabling packages
     */

    PackageManager.prototype.enablePackage = function(name) {
      var pack;
      pack = this.loadPackage(name);
      if (pack != null) {
        pack.enable();
      }
      return pack;
    };

    PackageManager.prototype.disablePackage = function(name) {
      var pack;
      pack = this.loadPackage(name);
      if (!this.isPackageDisabled(name)) {
        if (pack != null) {
          pack.disable();
        }
      }
      return pack;
    };

    PackageManager.prototype.isPackageDisabled = function(name) {
      var ref1;
      return _.include((ref1 = this.config.get('core.disabledPackages')) != null ? ref1 : [], name);
    };


    /*
    Section: Accessing active packages
     */

    PackageManager.prototype.getActivePackages = function() {
      return _.values(this.activePackages);
    };

    PackageManager.prototype.getActivePackage = function(name) {
      return this.activePackages[name];
    };

    PackageManager.prototype.isPackageActive = function(name) {
      return this.getActivePackage(name) != null;
    };

    PackageManager.prototype.hasActivatedInitialPackages = function() {
      return this.initialPackagesActivated;
    };


    /*
    Section: Accessing loaded packages
     */

    PackageManager.prototype.getLoadedPackages = function() {
      return _.values(this.loadedPackages);
    };

    PackageManager.prototype.getLoadedPackagesForTypes = function(types) {
      var i, len, pack, ref1, ref2, results;
      ref1 = this.getLoadedPackages();
      results = [];
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        if (ref2 = pack.getType(), indexOf.call(types, ref2) >= 0) {
          results.push(pack);
        }
      }
      return results;
    };

    PackageManager.prototype.getLoadedPackage = function(name) {
      return this.loadedPackages[name];
    };

    PackageManager.prototype.isPackageLoaded = function(name) {
      return this.getLoadedPackage(name) != null;
    };

    PackageManager.prototype.hasLoadedInitialPackages = function() {
      return this.initialPackagesLoaded;
    };


    /*
    Section: Accessing available packages
     */

    PackageManager.prototype.getAvailablePackagePaths = function() {
      return this.getAvailablePackages().map(function(a) {
        return a.path;
      });
    };

    PackageManager.prototype.getAvailablePackageNames = function() {
      return this.getAvailablePackages().map(function(a) {
        return a.name;
      });
    };

    PackageManager.prototype.getAvailablePackageMetadata = function() {
      var i, len, metadata, pack, packages, ref1, ref2, ref3;
      packages = [];
      ref1 = this.getAvailablePackages();
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        metadata = (ref2 = (ref3 = this.getLoadedPackage(pack.name)) != null ? ref3.metadata : void 0) != null ? ref2 : this.loadPackageMetadata(pack, true);
        packages.push(metadata);
      }
      return packages;
    };

    PackageManager.prototype.getAvailablePackages = function() {
      var i, j, len, len1, packageDirPath, packageName, packagePath, packages, packagesByName, ref1, ref2;
      packages = [];
      packagesByName = new Set();
      ref1 = this.packageDirPaths;
      for (i = 0, len = ref1.length; i < len; i++) {
        packageDirPath = ref1[i];
        if (fs.isDirectorySync(packageDirPath)) {
          ref2 = fs.readdirSync(packageDirPath);
          for (j = 0, len1 = ref2.length; j < len1; j++) {
            packagePath = ref2[j];
            packagePath = path.join(packageDirPath, packagePath);
            packageName = path.basename(packagePath);
            if (!packageName.startsWith('.') && !packagesByName.has(packageName) && fs.isDirectorySync(packagePath)) {
              packages.push({
                name: packageName,
                path: packagePath,
                isBundled: false
              });
              packagesByName.add(packageName);
            }
          }
        }
      }
      for (packageName in this.packageDependencies) {
        if (!packagesByName.has(packageName)) {
          packages.push({
            name: packageName,
            path: path.join(this.resourcePath, 'node_modules', packageName),
            isBundled: true
          });
        }
      }
      return packages.sort(function(a, b) {
        return a.name.toLowerCase().localeCompare(b.name.toLowerCase());
      });
    };


    /*
    Section: Private
     */

    PackageManager.prototype.getPackageState = function(name) {
      return this.packageStates[name];
    };

    PackageManager.prototype.setPackageState = function(name, state) {
      return this.packageStates[name] = state;
    };

    PackageManager.prototype.getPackageDependencies = function() {
      return this.packageDependencies;
    };

    PackageManager.prototype.hasAtomEngine = function(packagePath) {
      var metadata, ref1;
      metadata = this.loadPackageMetadata(packagePath, true);
      return (metadata != null ? (ref1 = metadata.engines) != null ? ref1.atom : void 0 : void 0) != null;
    };

    PackageManager.prototype.unobserveDisabledPackages = function() {
      var ref1;
      if ((ref1 = this.disabledPackagesSubscription) != null) {
        ref1.dispose();
      }
      return this.disabledPackagesSubscription = null;
    };

    PackageManager.prototype.observeDisabledPackages = function() {
      return this.disabledPackagesSubscription != null ? this.disabledPackagesSubscription : this.disabledPackagesSubscription = this.config.onDidChange('core.disabledPackages', (function(_this) {
        return function(arg) {
          var i, j, len, len1, newValue, oldValue, packageName, packagesToDisable, packagesToEnable;
          newValue = arg.newValue, oldValue = arg.oldValue;
          packagesToEnable = _.difference(oldValue, newValue);
          packagesToDisable = _.difference(newValue, oldValue);
          for (i = 0, len = packagesToDisable.length; i < len; i++) {
            packageName = packagesToDisable[i];
            if (_this.getActivePackage(packageName)) {
              _this.deactivatePackage(packageName);
            }
          }
          for (j = 0, len1 = packagesToEnable.length; j < len1; j++) {
            packageName = packagesToEnable[j];
            _this.activatePackage(packageName);
          }
          return null;
        };
      })(this));
    };

    PackageManager.prototype.unobservePackagesWithKeymapsDisabled = function() {
      var ref1;
      if ((ref1 = this.packagesWithKeymapsDisabledSubscription) != null) {
        ref1.dispose();
      }
      return this.packagesWithKeymapsDisabledSubscription = null;
    };

    PackageManager.prototype.observePackagesWithKeymapsDisabled = function() {
      return this.packagesWithKeymapsDisabledSubscription != null ? this.packagesWithKeymapsDisabledSubscription : this.packagesWithKeymapsDisabledSubscription = this.config.onDidChange('core.packagesWithKeymapsDisabled', (function(_this) {
        return function(arg) {
          var disabledPackageNames, i, j, keymapsToDisable, keymapsToEnable, len, len1, newValue, oldValue, packageName, ref1, ref2;
          newValue = arg.newValue, oldValue = arg.oldValue;
          keymapsToEnable = _.difference(oldValue, newValue);
          keymapsToDisable = _.difference(newValue, oldValue);
          disabledPackageNames = new Set(_this.config.get('core.disabledPackages'));
          for (i = 0, len = keymapsToDisable.length; i < len; i++) {
            packageName = keymapsToDisable[i];
            if (!disabledPackageNames.has(packageName)) {
              if ((ref1 = _this.getLoadedPackage(packageName)) != null) {
                ref1.deactivateKeymaps();
              }
            }
          }
          for (j = 0, len1 = keymapsToEnable.length; j < len1; j++) {
            packageName = keymapsToEnable[j];
            if (!disabledPackageNames.has(packageName)) {
              if ((ref2 = _this.getLoadedPackage(packageName)) != null) {
                ref2.activateKeymaps();
              }
            }
          }
          return null;
        };
      })(this));
    };

    PackageManager.prototype.preloadPackages = function() {
      var pack, packageName, ref1, results;
      ref1 = this.packagesCache;
      results = [];
      for (packageName in ref1) {
        pack = ref1[packageName];
        results.push(this.preloadPackage(packageName, pack));
      }
      return results;
    };

    PackageManager.prototype.preloadPackage = function(packageName, pack) {
      var metadata, options, ref1, ref2;
      metadata = (ref1 = pack.metadata) != null ? ref1 : {};
      if (!(typeof metadata.name === 'string' && metadata.name.length > 0)) {
        metadata.name = packageName;
      }
      if (((ref2 = metadata.repository) != null ? ref2.type : void 0) === 'git' && typeof metadata.repository.url === 'string') {
        metadata.repository.url = metadata.repository.url.replace(/(^git\+)|(\.git$)/g, '');
      }
      options = {
        path: pack.rootDirPath,
        name: packageName,
        preloadedPackage: true,
        bundledPackage: true,
        metadata: metadata,
        packageManager: this,
        config: this.config,
        styleManager: this.styleManager,
        commandRegistry: this.commandRegistry,
        keymapManager: this.keymapManager,
        notificationManager: this.notificationManager,
        grammarRegistry: this.grammarRegistry,
        themeManager: this.themeManager,
        menuManager: this.menuManager,
        contextMenuManager: this.contextMenuManager,
        deserializerManager: this.deserializerManager,
        viewRegistry: this.viewRegistry
      };
      if (metadata.theme) {
        pack = new ThemePackage(options);
      } else {
        pack = new Package(options);
      }
      pack.preload();
      return this.preloadedPackages[packageName] = pack;
    };

    PackageManager.prototype.loadPackages = function() {
      var disabledPackageNames;
      require('../exports/atom');
      disabledPackageNames = new Set(this.config.get('core.disabledPackages'));
      this.config.transact((function(_this) {
        return function() {
          var i, len, pack, ref1;
          ref1 = _this.getAvailablePackages();
          for (i = 0, len = ref1.length; i < len; i++) {
            pack = ref1[i];
            _this.loadAvailablePackage(pack, disabledPackageNames);
          }
        };
      })(this));
      this.initialPackagesLoaded = true;
      return this.emitter.emit('did-load-initial-packages');
    };

    PackageManager.prototype.loadPackage = function(nameOrPath) {
      var name, pack, packagePath;
      if (path.basename(nameOrPath)[0].match(/^\./)) {
        return null;
      } else if (pack = this.getLoadedPackage(nameOrPath)) {
        return pack;
      } else if (packagePath = this.resolvePackagePath(nameOrPath)) {
        name = path.basename(nameOrPath);
        return this.loadAvailablePackage({
          name: name,
          path: packagePath,
          isBundled: this.isBundledPackagePath(packagePath)
        });
      } else {
        console.warn("Could not resolve '" + nameOrPath + "' to a package path");
        return null;
      }
    };

    PackageManager.prototype.loadAvailablePackage = function(availablePackage, disabledPackageNames) {
      var error, loadedPackage, metadata, options, pack, preloadedPackage, ref1;
      preloadedPackage = this.preloadedPackages[availablePackage.name];
      if (disabledPackageNames != null ? disabledPackageNames.has(availablePackage.name) : void 0) {
        if (preloadedPackage != null) {
          preloadedPackage.deactivate();
          return delete preloadedPackage[availablePackage.name];
        }
      } else {
        loadedPackage = this.getLoadedPackage(availablePackage.name);
        if (loadedPackage != null) {
          return loadedPackage;
        } else {
          if (preloadedPackage != null) {
            if (availablePackage.isBundled) {
              preloadedPackage.finishLoading();
              this.loadedPackages[availablePackage.name] = preloadedPackage;
              return preloadedPackage;
            } else {
              preloadedPackage.deactivate();
              delete preloadedPackage[availablePackage.name];
            }
          }
          try {
            metadata = (ref1 = this.loadPackageMetadata(availablePackage)) != null ? ref1 : {};
          } catch (error1) {
            error = error1;
            this.handleMetadataError(error, availablePackage.path);
            return null;
          }
          if (!availablePackage.isBundled) {
            if (this.isDeprecatedPackage(metadata.name, metadata.version)) {
              console.warn("Could not load " + metadata.name + "@" + metadata.version + " because it uses deprecated APIs that have been removed.");
              return null;
            }
          }
          options = {
            path: availablePackage.path,
            name: availablePackage.name,
            metadata: metadata,
            bundledPackage: availablePackage.isBundled,
            packageManager: this,
            config: this.config,
            styleManager: this.styleManager,
            commandRegistry: this.commandRegistry,
            keymapManager: this.keymapManager,
            notificationManager: this.notificationManager,
            grammarRegistry: this.grammarRegistry,
            themeManager: this.themeManager,
            menuManager: this.menuManager,
            contextMenuManager: this.contextMenuManager,
            deserializerManager: this.deserializerManager,
            viewRegistry: this.viewRegistry
          };
          if (metadata.theme) {
            pack = new ThemePackage(options);
          } else {
            pack = new Package(options);
          }
          pack.load();
          this.loadedPackages[pack.name] = pack;
          this.emitter.emit('did-load-package', pack);
          return pack;
        }
      }
    };

    PackageManager.prototype.unloadPackages = function() {
      var i, len, name, ref1;
      ref1 = _.keys(this.loadedPackages);
      for (i = 0, len = ref1.length; i < len; i++) {
        name = ref1[i];
        this.unloadPackage(name);
      }
      return null;
    };

    PackageManager.prototype.unloadPackage = function(name) {
      var pack;
      if (this.isPackageActive(name)) {
        throw new Error("Tried to unload active package '" + name + "'");
      }
      if (pack = this.getLoadedPackage(name)) {
        delete this.loadedPackages[pack.name];
        return this.emitter.emit('did-unload-package', pack);
      } else {
        throw new Error("No loaded package for name '" + name + "'");
      }
    };

    PackageManager.prototype.activate = function() {
      var activator, i, len, packages, promises, ref1, ref2, types;
      promises = [];
      ref1 = this.packageActivators;
      for (i = 0, len = ref1.length; i < len; i++) {
        ref2 = ref1[i], activator = ref2[0], types = ref2[1];
        packages = this.getLoadedPackagesForTypes(types);
        promises = promises.concat(activator.activatePackages(packages));
      }
      return Promise.all(promises).then((function(_this) {
        return function() {
          _this.triggerDeferredActivationHooks();
          _this.initialPackagesActivated = true;
          return _this.emitter.emit('did-activate-initial-packages');
        };
      })(this));
    };

    PackageManager.prototype.registerPackageActivator = function(activator, types) {
      return this.packageActivators.push([activator, types]);
    };

    PackageManager.prototype.activatePackages = function(packages) {
      var promises;
      promises = [];
      this.config.transactAsync((function(_this) {
        return function() {
          var i, len, pack, promise;
          for (i = 0, len = packages.length; i < len; i++) {
            pack = packages[i];
            promise = _this.activatePackage(pack.name);
            if (!pack.activationShouldBeDeferred()) {
              promises.push(promise);
            }
          }
          return Promise.all(promises);
        };
      })(this));
      this.observeDisabledPackages();
      this.observePackagesWithKeymapsDisabled();
      return promises;
    };

    PackageManager.prototype.activatePackage = function(name) {
      var activationPromise, pack;
      if (pack = this.getActivePackage(name)) {
        return Promise.resolve(pack);
      } else if (pack = this.loadPackage(name)) {
        this.activatingPackages[pack.name] = pack;
        activationPromise = pack.activate().then((function(_this) {
          return function() {
            if (_this.activatingPackages[pack.name] != null) {
              delete _this.activatingPackages[pack.name];
              _this.activePackages[pack.name] = pack;
              _this.emitter.emit('did-activate-package', pack);
            }
            return pack;
          };
        })(this));
        if (this.deferredActivationHooks == null) {
          this.triggeredActivationHooks.forEach((function(_this) {
            return function(hook) {
              return _this.activationHookEmitter.emit(hook);
            };
          })(this));
        }
        return activationPromise;
      } else {
        return Promise.reject(new Error("Failed to load package '" + name + "'"));
      }
    };

    PackageManager.prototype.triggerDeferredActivationHooks = function() {
      var hook, i, len, ref1;
      if (this.deferredActivationHooks == null) {
        return;
      }
      ref1 = this.deferredActivationHooks;
      for (i = 0, len = ref1.length; i < len; i++) {
        hook = ref1[i];
        this.activationHookEmitter.emit(hook);
      }
      return this.deferredActivationHooks = null;
    };

    PackageManager.prototype.triggerActivationHook = function(hook) {
      if (!((hook != null) && _.isString(hook) && hook.length > 0)) {
        return new Error("Cannot trigger an empty activation hook");
      }
      this.triggeredActivationHooks.add(hook);
      if (this.deferredActivationHooks != null) {
        return this.deferredActivationHooks.push(hook);
      } else {
        return this.activationHookEmitter.emit(hook);
      }
    };

    PackageManager.prototype.onDidTriggerActivationHook = function(hook, callback) {
      if (!((hook != null) && _.isString(hook) && hook.length > 0)) {
        return;
      }
      return this.activationHookEmitter.on(hook, callback);
    };

    PackageManager.prototype.serialize = function() {
      var i, len, pack, ref1;
      ref1 = this.getActivePackages();
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        this.serializePackage(pack);
      }
      return this.packageStates;
    };

    PackageManager.prototype.serializePackage = function(pack) {
      var state;
      if (state = typeof pack.serialize === "function" ? pack.serialize() : void 0) {
        return this.setPackageState(pack.name, state);
      }
    };

    PackageManager.prototype.deactivatePackages = function() {
      this.config.transact((function(_this) {
        return function() {
          var i, len, pack, ref1;
          ref1 = _this.getLoadedPackages();
          for (i = 0, len = ref1.length; i < len; i++) {
            pack = ref1[i];
            _this.deactivatePackage(pack.name, true);
          }
        };
      })(this));
      this.unobserveDisabledPackages();
      return this.unobservePackagesWithKeymapsDisabled();
    };

    PackageManager.prototype.deactivatePackage = function(name, suppressSerialization) {
      var pack;
      pack = this.getLoadedPackage(name);
      if (!suppressSerialization && this.isPackageActive(pack.name)) {
        this.serializePackage(pack);
      }
      pack.deactivate();
      delete this.activePackages[pack.name];
      delete this.activatingPackages[pack.name];
      return this.emitter.emit('did-deactivate-package', pack);
    };

    PackageManager.prototype.handleMetadataError = function(error, packagePath) {
      var detail, message, metadataPath, stack;
      metadataPath = path.join(packagePath, 'package.json');
      detail = error.message + " in " + metadataPath;
      stack = error.stack + "\n  at " + metadataPath + ":1:1";
      message = "Failed to load the " + (path.basename(packagePath)) + " package";
      return this.notificationManager.addError(message, {
        stack: stack,
        detail: detail,
        packageName: path.basename(packagePath),
        dismissable: true
      });
    };

    PackageManager.prototype.uninstallDirectory = function(directory) {
      var dirPromise, symlinkPromise;
      symlinkPromise = new Promise(function(resolve) {
        return fs.isSymbolicLink(directory, function(isSymLink) {
          return resolve(isSymLink);
        });
      });
      dirPromise = new Promise(function(resolve) {
        return fs.isDirectory(directory, function(isDir) {
          return resolve(isDir);
        });
      });
      return Promise.all([symlinkPromise, dirPromise]).then(function(values) {
        var isDir, isSymLink;
        isSymLink = values[0], isDir = values[1];
        if (!isSymLink && isDir) {
          return fs.remove(directory, function() {});
        }
      });
    };

    PackageManager.prototype.reloadActivePackageStyleSheets = function() {
      var i, len, pack, ref1;
      ref1 = this.getActivePackages();
      for (i = 0, len = ref1.length; i < len; i++) {
        pack = ref1[i];
        if (pack.getType() !== 'theme') {
          if (typeof pack.reloadStylesheets === "function") {
            pack.reloadStylesheets();
          }
        }
      }
    };

    PackageManager.prototype.isBundledPackagePath = function(packagePath) {
      if (this.devMode) {
        if (!this.resourcePath.startsWith("" + process.resourcesPath + path.sep)) {
          return false;
        }
      }
      if (this.resourcePathWithTrailingSlash == null) {
        this.resourcePathWithTrailingSlash = "" + this.resourcePath + path.sep;
      }
      return packagePath != null ? packagePath.startsWith(this.resourcePathWithTrailingSlash) : void 0;
    };

    PackageManager.prototype.loadPackageMetadata = function(packagePathOrAvailablePackage, ignoreErrors) {
      var availablePackage, error, isBundled, metadata, metadataPath, packageName, packagePath, ref1, ref2;
      if (ignoreErrors == null) {
        ignoreErrors = false;
      }
      if (typeof packagePathOrAvailablePackage === 'object') {
        availablePackage = packagePathOrAvailablePackage;
        packageName = availablePackage.name;
        packagePath = availablePackage.path;
        isBundled = availablePackage.isBundled;
      } else {
        packagePath = packagePathOrAvailablePackage;
        packageName = path.basename(packagePath);
        isBundled = this.isBundledPackagePath(packagePath);
      }
      if (isBundled) {
        metadata = (ref1 = this.packagesCache[packageName]) != null ? ref1.metadata : void 0;
      }
      if (metadata == null) {
        if (metadataPath = CSON.resolve(path.join(packagePath, 'package'))) {
          try {
            metadata = CSON.readFileSync(metadataPath);
            this.normalizePackageMetadata(metadata);
          } catch (error1) {
            error = error1;
            if (!ignoreErrors) {
              throw error;
            }
          }
        }
      }
      if (metadata == null) {
        metadata = {};
      }
      if (!(typeof metadata.name === 'string' && metadata.name.length > 0)) {
        metadata.name = packageName;
      }
      if (((ref2 = metadata.repository) != null ? ref2.type : void 0) === 'git' && typeof metadata.repository.url === 'string') {
        metadata.repository.url = metadata.repository.url.replace(/(^git\+)|(\.git$)/g, '');
      }
      return metadata;
    };

    PackageManager.prototype.normalizePackageMetadata = function(metadata) {
      if (!(metadata != null ? metadata._id : void 0)) {
        if (normalizePackageData == null) {
          normalizePackageData = require('normalize-package-data');
        }
        return normalizePackageData(metadata);
      }
    };

    return PackageManager;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
