(function() {
  var AtomWindow, BrowserWindow, EventEmitter, app, dialog, fs, ipcMain, path, ref, url,
    slice = [].slice;

  ref = require('electron'), BrowserWindow = ref.BrowserWindow, app = ref.app, dialog = ref.dialog, ipcMain = ref.ipcMain;

  path = require('path');

  fs = require('fs');

  url = require('url');

  EventEmitter = require('events').EventEmitter;

  module.exports = AtomWindow = (function() {
    Object.assign(AtomWindow.prototype, EventEmitter.prototype);

    AtomWindow.iconPath = path.resolve(__dirname, '..', '..', 'resources', 'atom.png');

    AtomWindow.includeShellLoadTime = true;

    AtomWindow.prototype.browserWindow = null;

    AtomWindow.prototype.loaded = null;

    AtomWindow.prototype.isSpec = null;

    function AtomWindow(atomApplication, fileRecoveryService, settings) {
      var hasPathToOpen, loadSettings, locationsToOpen, options, parentDirectory, pathToOpen, stat;
      this.atomApplication = atomApplication;
      this.fileRecoveryService = fileRecoveryService;
      if (settings == null) {
        settings = {};
      }
      this.resourcePath = settings.resourcePath, pathToOpen = settings.pathToOpen, locationsToOpen = settings.locationsToOpen, this.isSpec = settings.isSpec, this.headless = settings.headless, this.safeMode = settings.safeMode, this.devMode = settings.devMode;
      if (pathToOpen) {
        if (locationsToOpen == null) {
          locationsToOpen = [
            {
              pathToOpen: pathToOpen
            }
          ];
        }
      }
      if (locationsToOpen == null) {
        locationsToOpen = [];
      }
      this.loadedPromise = new Promise((function(_this) {
        return function(resolveLoadedPromise) {
          _this.resolveLoadedPromise = resolveLoadedPromise;
        };
      })(this));
      this.closedPromise = new Promise((function(_this) {
        return function(resolveClosedPromise) {
          _this.resolveClosedPromise = resolveClosedPromise;
        };
      })(this));
      options = {
        show: false,
        title: 'Atom',
        webPreferences: {
          backgroundThrottling: !this.isSpec,
          disableBlinkFeatures: 'Auxclick'
        }
      };
      if (process.platform === 'linux') {
        options.icon = this.constructor.iconPath;
      }
      if (this.shouldAddCustomTitleBar()) {
        options.titleBarStyle = 'hidden';
      }
      if (this.shouldAddCustomInsetTitleBar()) {
        options.titleBarStyle = 'hidden-inset';
      }
      if (this.shouldHideTitleBar()) {
        options.frame = false;
      }
      this.browserWindow = new BrowserWindow(options);
      this.handleEvents();
      loadSettings = Object.assign({}, settings);
      loadSettings.appVersion = app.getVersion();
      loadSettings.resourcePath = this.resourcePath;
      if (loadSettings.devMode == null) {
        loadSettings.devMode = false;
      }
      if (loadSettings.safeMode == null) {
        loadSettings.safeMode = false;
      }
      loadSettings.atomHome = process.env.ATOM_HOME;
      if (loadSettings.clearWindowState == null) {
        loadSettings.clearWindowState = false;
      }
      if (loadSettings.initialPaths == null) {
        loadSettings.initialPaths = (function() {
          var i, len, results;
          results = [];
          for (i = 0, len = locationsToOpen.length; i < len; i++) {
            pathToOpen = locationsToOpen[i].pathToOpen;
            if (!(pathToOpen)) {
              continue;
            }
            stat = fs.statSyncNoException(pathToOpen) || null;
            if (stat != null ? stat.isDirectory() : void 0) {
              results.push(pathToOpen);
            } else {
              parentDirectory = path.dirname(pathToOpen);
              if ((stat != null ? stat.isFile() : void 0) || fs.existsSync(parentDirectory)) {
                results.push(parentDirectory);
              } else {
                results.push(pathToOpen);
              }
            }
          }
          return results;
        })();
      }
      loadSettings.initialPaths.sort();
      if (this.constructor.includeShellLoadTime && !this.isSpec) {
        this.constructor.includeShellLoadTime = false;
        if (loadSettings.shellLoadTime == null) {
          loadSettings.shellLoadTime = Date.now() - global.shellStartTime;
        }
      }
      this.representedDirectoryPaths = loadSettings.initialPaths;
      if (loadSettings.env != null) {
        this.env = loadSettings.env;
      }
      this.browserWindow.loadSettingsJSON = JSON.stringify(loadSettings);
      this.browserWindow.on('window:loaded', (function(_this) {
        return function() {
          _this.disableZoom();
          _this.emit('window:loaded');
          return _this.resolveLoadedPromise();
        };
      })(this));
      this.browserWindow.on('window:locations-opened', (function(_this) {
        return function() {
          return _this.emit('window:locations-opened');
        };
      })(this));
      this.browserWindow.on('enter-full-screen', (function(_this) {
        return function() {
          return _this.browserWindow.webContents.send('did-enter-full-screen');
        };
      })(this));
      this.browserWindow.on('leave-full-screen', (function(_this) {
        return function() {
          return _this.browserWindow.webContents.send('did-leave-full-screen');
        };
      })(this));
      this.browserWindow.loadURL(url.format({
        protocol: 'file',
        pathname: this.resourcePath + "/static/index.html",
        slashes: true
      }));
      this.browserWindow.showSaveDialog = this.showSaveDialog.bind(this);
      if (this.isSpec) {
        this.browserWindow.focusOnWebView();
      }
      if (typeof windowDimensions !== "undefined" && windowDimensions !== null) {
        this.browserWindow.temporaryState = {
          windowDimensions: windowDimensions
        };
      }
      hasPathToOpen = !(locationsToOpen.length === 1 && (locationsToOpen[0].pathToOpen == null));
      if (hasPathToOpen && !this.isSpecWindow()) {
        this.openLocations(locationsToOpen);
      }
      this.atomApplication.addWindow(this);
    }

    AtomWindow.prototype.hasProjectPath = function() {
      return this.representedDirectoryPaths.length > 0;
    };

    AtomWindow.prototype.setupContextMenu = function() {
      var ContextMenu;
      ContextMenu = require('./context-menu');
      return this.browserWindow.on('context-menu', (function(_this) {
        return function(menuTemplate) {
          return new ContextMenu(menuTemplate, _this);
        };
      })(this));
    };

    AtomWindow.prototype.containsPaths = function(paths) {
      var i, len, pathToCheck;
      for (i = 0, len = paths.length; i < len; i++) {
        pathToCheck = paths[i];
        if (!this.containsPath(pathToCheck)) {
          return false;
        }
      }
      return true;
    };

    AtomWindow.prototype.containsPath = function(pathToCheck) {
      return this.representedDirectoryPaths.some(function(projectPath) {
        var base;
        if (!projectPath) {
          return false;
        } else if (!pathToCheck) {
          return false;
        } else if (pathToCheck === projectPath) {
          return true;
        } else if (typeof (base = fs.statSyncNoException(pathToCheck)).isDirectory === "function" ? base.isDirectory() : void 0) {
          return false;
        } else if (pathToCheck.indexOf(path.join(projectPath, path.sep)) === 0) {
          return true;
        } else {
          return false;
        }
      });
    };

    AtomWindow.prototype.handleEvents = function() {
      this.browserWindow.on('close', (function(_this) {
        return function(event) {
          if (!(_this.atomApplication.quitting || _this.unloading)) {
            event.preventDefault();
            _this.unloading = true;
            _this.atomApplication.saveState(false);
            return _this.prepareToUnload().then(function(result) {
              if (result) {
                return _this.close();
              }
            });
          }
        };
      })(this));
      this.browserWindow.on('closed', (function(_this) {
        return function() {
          _this.fileRecoveryService.didCloseWindow(_this);
          _this.atomApplication.removeWindow(_this);
          return _this.resolveClosedPromise();
        };
      })(this));
      this.browserWindow.on('unresponsive', (function(_this) {
        return function() {
          var chosen;
          if (_this.isSpec) {
            return;
          }
          chosen = dialog.showMessageBox(_this.browserWindow, {
            type: 'warning',
            buttons: ['Force Close', 'Keep Waiting'],
            message: 'Editor is not responding',
            detail: 'The editor is not responding. Would you like to force close it or just keep waiting?'
          });
          if (chosen === 0) {
            return _this.browserWindow.destroy();
          }
        };
      })(this));
      this.browserWindow.webContents.on('crashed', (function(_this) {
        return function() {
          var chosen;
          if (_this.headless) {
            console.log("Renderer process crashed, exiting");
            _this.atomApplication.exit(100);
            return;
          }
          _this.fileRecoveryService.didCrashWindow(_this);
          chosen = dialog.showMessageBox(_this.browserWindow, {
            type: 'warning',
            buttons: ['Close Window', 'Reload', 'Keep It Open'],
            message: 'The editor has crashed',
            detail: 'Please report this issue to https://github.com/atom/atom'
          });
          switch (chosen) {
            case 0:
              return _this.browserWindow.destroy();
            case 1:
              return _this.browserWindow.reload();
          }
        };
      })(this));
      this.browserWindow.webContents.on('will-navigate', (function(_this) {
        return function(event, url) {
          if (url !== _this.browserWindow.webContents.getURL()) {
            return event.preventDefault();
          }
        };
      })(this));
      this.setupContextMenu();
      if (this.isSpec) {
        return this.browserWindow.on('blur', (function(_this) {
          return function() {
            return _this.browserWindow.focusOnWebView();
          };
        })(this));
      }
    };

    AtomWindow.prototype.prepareToUnload = function() {
      if (this.isSpecWindow()) {
        return Promise.resolve(true);
      }
      return this.lastPrepareToUnloadPromise = new Promise((function(_this) {
        return function(resolve) {
          var callback;
          callback = function(event, result) {
            if (BrowserWindow.fromWebContents(event.sender) === _this.browserWindow) {
              ipcMain.removeListener('did-prepare-to-unload', callback);
              if (!result) {
                _this.unloading = false;
                _this.atomApplication.quitting = false;
              }
              return resolve(result);
            }
          };
          ipcMain.on('did-prepare-to-unload', callback);
          return _this.browserWindow.webContents.send('prepare-to-unload');
        };
      })(this));
    };

    AtomWindow.prototype.openPath = function(pathToOpen, initialLine, initialColumn) {
      return this.openLocations([
        {
          pathToOpen: pathToOpen,
          initialLine: initialLine,
          initialColumn: initialColumn
        }
      ]);
    };

    AtomWindow.prototype.openLocations = function(locationsToOpen) {
      return this.loadedPromise.then((function(_this) {
        return function() {
          return _this.sendMessage('open-locations', locationsToOpen);
        };
      })(this));
    };

    AtomWindow.prototype.replaceEnvironment = function(env) {
      return this.browserWindow.webContents.send('environment', env);
    };

    AtomWindow.prototype.sendMessage = function(message, detail) {
      return this.browserWindow.webContents.send('message', message, detail);
    };

    AtomWindow.prototype.sendCommand = function() {
      var args, command;
      command = arguments[0], args = 2 <= arguments.length ? slice.call(arguments, 1) : [];
      if (this.isSpecWindow()) {
        if (!this.atomApplication.sendCommandToFirstResponder(command)) {
          switch (command) {
            case 'window:reload':
              return this.reload();
            case 'window:toggle-dev-tools':
              return this.toggleDevTools();
            case 'window:close':
              return this.close();
          }
        }
      } else if (this.isWebViewFocused()) {
        return this.sendCommandToBrowserWindow.apply(this, [command].concat(slice.call(args)));
      } else {
        if (!this.atomApplication.sendCommandToFirstResponder(command)) {
          return this.sendCommandToBrowserWindow.apply(this, [command].concat(slice.call(args)));
        }
      }
    };

    AtomWindow.prototype.sendCommandToBrowserWindow = function() {
      var action, args, command, ref1, ref2;
      command = arguments[0], args = 2 <= arguments.length ? slice.call(arguments, 1) : [];
      action = ((ref1 = args[0]) != null ? ref1.contextCommand : void 0) ? 'context-command' : 'command';
      return (ref2 = this.browserWindow.webContents).send.apply(ref2, [action, command].concat(slice.call(args)));
    };

    AtomWindow.prototype.getDimensions = function() {
      var height, ref1, ref2, width, x, y;
      ref1 = this.browserWindow.getPosition(), x = ref1[0], y = ref1[1];
      ref2 = this.browserWindow.getSize(), width = ref2[0], height = ref2[1];
      return {
        x: x,
        y: y,
        width: width,
        height: height
      };
    };

    AtomWindow.prototype.shouldAddCustomTitleBar = function() {
      return !this.isSpec && process.platform === 'darwin' && this.atomApplication.config.get('core.titleBar') === 'custom';
    };

    AtomWindow.prototype.shouldAddCustomInsetTitleBar = function() {
      return !this.isSpec && process.platform === 'darwin' && this.atomApplication.config.get('core.titleBar') === 'custom-inset';
    };

    AtomWindow.prototype.shouldHideTitleBar = function() {
      return !this.isSpec && process.platform === 'darwin' && this.atomApplication.config.get('core.titleBar') === 'hidden';
    };

    AtomWindow.prototype.close = function() {
      return this.browserWindow.close();
    };

    AtomWindow.prototype.focus = function() {
      return this.browserWindow.focus();
    };

    AtomWindow.prototype.minimize = function() {
      return this.browserWindow.minimize();
    };

    AtomWindow.prototype.maximize = function() {
      return this.browserWindow.maximize();
    };

    AtomWindow.prototype.unmaximize = function() {
      return this.browserWindow.unmaximize();
    };

    AtomWindow.prototype.restore = function() {
      return this.browserWindow.restore();
    };

    AtomWindow.prototype.setFullScreen = function(fullScreen) {
      return this.browserWindow.setFullScreen(fullScreen);
    };

    AtomWindow.prototype.setAutoHideMenuBar = function(autoHideMenuBar) {
      return this.browserWindow.setAutoHideMenuBar(autoHideMenuBar);
    };

    AtomWindow.prototype.handlesAtomCommands = function() {
      return !this.isSpecWindow() && this.isWebViewFocused();
    };

    AtomWindow.prototype.isFocused = function() {
      return this.browserWindow.isFocused();
    };

    AtomWindow.prototype.isMaximized = function() {
      return this.browserWindow.isMaximized();
    };

    AtomWindow.prototype.isMinimized = function() {
      return this.browserWindow.isMinimized();
    };

    AtomWindow.prototype.isWebViewFocused = function() {
      return this.browserWindow.isWebViewFocused();
    };

    AtomWindow.prototype.isSpecWindow = function() {
      return this.isSpec;
    };

    AtomWindow.prototype.reload = function() {
      this.loadedPromise = new Promise((function(_this) {
        return function(resolveLoadedPromise) {
          _this.resolveLoadedPromise = resolveLoadedPromise;
        };
      })(this));
      this.prepareToUnload().then((function(_this) {
        return function(result) {
          if (result) {
            return _this.browserWindow.reload();
          }
        };
      })(this));
      return this.loadedPromise;
    };

    AtomWindow.prototype.showSaveDialog = function(params) {
      params = Object.assign({
        title: 'Save File',
        defaultPath: this.representedDirectoryPaths[0]
      }, params);
      return dialog.showSaveDialog(this.browserWindow, params);
    };

    AtomWindow.prototype.toggleDevTools = function() {
      return this.browserWindow.toggleDevTools();
    };

    AtomWindow.prototype.openDevTools = function() {
      return this.browserWindow.openDevTools();
    };

    AtomWindow.prototype.closeDevTools = function() {
      return this.browserWindow.closeDevTools();
    };

    AtomWindow.prototype.setDocumentEdited = function(documentEdited) {
      return this.browserWindow.setDocumentEdited(documentEdited);
    };

    AtomWindow.prototype.setRepresentedFilename = function(representedFilename) {
      return this.browserWindow.setRepresentedFilename(representedFilename);
    };

    AtomWindow.prototype.setRepresentedDirectoryPaths = function(representedDirectoryPaths) {
      this.representedDirectoryPaths = representedDirectoryPaths;
      this.representedDirectoryPaths.sort();
      return this.atomApplication.saveState();
    };

    AtomWindow.prototype.copy = function() {
      return this.browserWindow.copy();
    };

    AtomWindow.prototype.disableZoom = function() {
      return this.browserWindow.webContents.setVisualZoomLevelLimits(1, 1);
    };

    return AtomWindow;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
