(function() {
  var CSON, ContextMenuItemSet, ContextMenuManager, Disposable, MenuHelpers, calculateSpecificity, fs, path, platformContextMenu, ref, ref1, ref2, remote, validateSelector;

  path = require('path');

  CSON = require('season');

  fs = require('fs-plus');

  ref = require('clear-cut'), calculateSpecificity = ref.calculateSpecificity, validateSelector = ref.validateSelector;

  Disposable = require('event-kit').Disposable;

  remote = require('electron').remote;

  MenuHelpers = require('./menu-helpers');

  platformContextMenu = (ref1 = require('../package.json')) != null ? (ref2 = ref1._atomMenu) != null ? ref2['context-menu'] : void 0 : void 0;

  module.exports = ContextMenuManager = (function() {
    function ContextMenuManager(arg) {
      this.keymapManager = arg.keymapManager;
      this.definitions = {
        '.overlayer': []
      };
      this.clear();
      this.keymapManager.onDidLoadBundledKeymaps((function(_this) {
        return function() {
          return _this.loadPlatformItems();
        };
      })(this));
    }

    ContextMenuManager.prototype.initialize = function(arg) {
      this.resourcePath = arg.resourcePath, this.devMode = arg.devMode;
    };

    ContextMenuManager.prototype.loadPlatformItems = function() {
      var map, menusDirPath, platformMenuPath, ref3;
      if (platformContextMenu != null) {
        return this.add(platformContextMenu, (ref3 = this.devMode) != null ? ref3 : false);
      } else {
        menusDirPath = path.join(this.resourcePath, 'menus');
        platformMenuPath = fs.resolve(menusDirPath, process.platform, ['cson', 'json']);
        map = CSON.readFileSync(platformMenuPath);
        return this.add(map['context-menu']);
      }
    };

    ContextMenuManager.prototype.add = function(itemsBySelector, throwOnInvalidSelector) {
      var addedItemSets, itemSet, items, selector;
      if (throwOnInvalidSelector == null) {
        throwOnInvalidSelector = true;
      }
      addedItemSets = [];
      for (selector in itemsBySelector) {
        items = itemsBySelector[selector];
        if (throwOnInvalidSelector) {
          validateSelector(selector);
        }
        itemSet = new ContextMenuItemSet(selector, items);
        addedItemSets.push(itemSet);
        this.itemSets.push(itemSet);
      }
      return new Disposable((function(_this) {
        return function() {
          var i, len;
          for (i = 0, len = addedItemSets.length; i < len; i++) {
            itemSet = addedItemSets[i];
            _this.itemSets.splice(_this.itemSets.indexOf(itemSet), 1);
          }
        };
      })(this));
    };

    ContextMenuManager.prototype.templateForElement = function(target) {
      return this.templateForEvent({
        target: target
      });
    };

    ContextMenuManager.prototype.templateForEvent = function(event) {
      var currentTarget, currentTargetItems, i, item, itemForEvent, itemSet, j, k, len, len1, len2, matchingItemSets, ref3, template;
      template = [];
      currentTarget = event.target;
      while (currentTarget != null) {
        currentTargetItems = [];
        matchingItemSets = this.itemSets.filter(function(itemSet) {
          return currentTarget.webkitMatchesSelector(itemSet.selector);
        });
        for (i = 0, len = matchingItemSets.length; i < len; i++) {
          itemSet = matchingItemSets[i];
          ref3 = itemSet.items;
          for (j = 0, len1 = ref3.length; j < len1; j++) {
            item = ref3[j];
            itemForEvent = this.cloneItemForEvent(item, event);
            if (itemForEvent) {
              MenuHelpers.merge(currentTargetItems, itemForEvent, itemSet.specificity);
            }
          }
        }
        for (k = 0, len2 = currentTargetItems.length; k < len2; k++) {
          item = currentTargetItems[k];
          MenuHelpers.merge(template, item, false);
        }
        currentTarget = currentTarget.parentElement;
      }
      this.pruneRedundantSeparators(template);
      return template;
    };

    ContextMenuManager.prototype.pruneRedundantSeparators = function(menu) {
      var index, keepNextItemIfSeparator, results;
      keepNextItemIfSeparator = false;
      index = 0;
      results = [];
      while (index < menu.length) {
        if (menu[index].type === 'separator') {
          if (!keepNextItemIfSeparator || index === menu.length - 1) {
            results.push(menu.splice(index, 1));
          } else {
            results.push(index++);
          }
        } else {
          keepNextItemIfSeparator = true;
          results.push(index++);
        }
      }
      return results;
    };

    ContextMenuManager.prototype.cloneItemForEvent = function(item, event) {
      if (item.devMode && !this.devMode) {
        return null;
      }
      item = Object.create(item);
      if (typeof item.shouldDisplay === 'function') {
        if (!item.shouldDisplay(event)) {
          return null;
        }
      }
      if (typeof item.created === "function") {
        item.created(event);
      }
      if (Array.isArray(item.submenu)) {
        item.submenu = item.submenu.map((function(_this) {
          return function(submenuItem) {
            return _this.cloneItemForEvent(submenuItem, event);
          };
        })(this)).filter(function(submenuItem) {
          return submenuItem !== null;
        });
      }
      return item;
    };

    ContextMenuManager.prototype.convertLegacyItemsBySelector = function(legacyItemsBySelector, devMode) {
      var commandsByLabel, itemsBySelector, selector;
      itemsBySelector = {};
      for (selector in legacyItemsBySelector) {
        commandsByLabel = legacyItemsBySelector[selector];
        itemsBySelector[selector] = this.convertLegacyItems(commandsByLabel, devMode);
      }
      return itemsBySelector;
    };

    ContextMenuManager.prototype.convertLegacyItems = function(legacyItems, devMode) {
      var commandOrSubmenu, items, label;
      items = [];
      for (label in legacyItems) {
        commandOrSubmenu = legacyItems[label];
        if (typeof commandOrSubmenu === 'object') {
          items.push({
            label: label,
            submenu: this.convertLegacyItems(commandOrSubmenu, devMode),
            devMode: devMode
          });
        } else if (commandOrSubmenu === '-') {
          items.push({
            type: 'separator'
          });
        } else {
          items.push({
            label: label,
            command: commandOrSubmenu,
            devMode: devMode
          });
        }
      }
      return items;
    };

    ContextMenuManager.prototype.showForEvent = function(event) {
      var menuTemplate;
      this.activeElement = event.target;
      menuTemplate = this.templateForEvent(event);
      if (!((menuTemplate != null ? menuTemplate.length : void 0) > 0)) {
        return;
      }
      remote.getCurrentWindow().emit('context-menu', menuTemplate);
    };

    ContextMenuManager.prototype.clear = function() {
      var inspectElement;
      this.activeElement = null;
      this.itemSets = [];
      inspectElement = {
        'atom-workspace': [
          {
            label: 'Inspect Element',
            command: 'application:inspect',
            devMode: true,
            created: function(event) {
              var pageX, pageY;
              pageX = event.pageX, pageY = event.pageY;
              return this.commandDetail = {
                x: pageX,
                y: pageY
              };
            }
          }
        ]
      };
      return this.add(inspectElement, false);
    };

    return ContextMenuManager;

  })();

  ContextMenuItemSet = (function() {
    function ContextMenuItemSet(selector1, items1) {
      this.selector = selector1;
      this.items = items1;
      this.specificity = calculateSpecificity(this.selector);
    }

    return ContextMenuItemSet;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
