(function() {
  var BufferedProcess, Client, CompositeDisposable, Emitter, PackageManager, _, createJsonParseError, createProcessError, handleProcessErrors, ref, semver,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  semver = require('semver');

  Client = require('./atom-io-client');

  module.exports = PackageManager = (function() {
    PackageManager.prototype.CACHE_EXPIRY = 1000 * 60 * 10;

    function PackageManager() {
      this.setProxyServersAsync = bind(this.setProxyServersAsync, this);
      this.setProxyServers = bind(this.setProxyServers, this);
      this.packagePromises = [];
      this.apmCache = {
        loadOutdated: {
          value: null,
          expiry: 0
        }
      };
      this.emitter = new Emitter;
    }

    PackageManager.prototype.getClient = function() {
      return this.client != null ? this.client : this.client = new Client(this);
    };

    PackageManager.prototype.isPackageInstalled = function(packageName) {
      if (atom.packages.isPackageLoaded(packageName)) {
        return true;
      } else {
        return atom.packages.getAvailablePackageNames().indexOf(packageName) > -1;
      }
    };

    PackageManager.prototype.packageHasSettings = function(packageName) {
      var grammar, grammars, i, len, pack, ref1, schema;
      grammars = (ref1 = atom.grammars.getGrammars()) != null ? ref1 : [];
      for (i = 0, len = grammars.length; i < len; i++) {
        grammar = grammars[i];
        if (grammar.path) {
          if (grammar.packageName === packageName) {
            return true;
          }
        }
      }
      pack = atom.packages.getLoadedPackage(packageName);
      if ((pack != null) && !atom.packages.isPackageActive(packageName)) {
        pack.activateConfig();
      }
      schema = atom.config.getSchema(packageName);
      return (schema != null) && (schema.type !== 'any');
    };

    PackageManager.prototype.setProxyServers = function(callback) {
      var session;
      session = atom.getCurrentWindow().webContents.session;
      return session.resolveProxy('http://atom.io', (function(_this) {
        return function(httpProxy) {
          _this.applyProxyToEnv('http_proxy', httpProxy);
          return session.resolveProxy('https://atom.io', function(httpsProxy) {
            _this.applyProxyToEnv('https_proxy', httpsProxy);
            return callback();
          });
        };
      })(this));
    };

    PackageManager.prototype.setProxyServersAsync = function(callback) {
      var httpProxyPromise, httpsProxyPromise;
      httpProxyPromise = atom.resolveProxy('http://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('http_proxy', proxy);
        };
      })(this));
      httpsProxyPromise = atom.resolveProxy('https://atom.io').then((function(_this) {
        return function(proxy) {
          return _this.applyProxyToEnv('https_proxy', proxy);
        };
      })(this));
      return Promise.all([httpProxyPromise, httpsProxyPromise]).then(callback);
    };

    PackageManager.prototype.applyProxyToEnv = function(envName, proxy) {
      if (proxy != null) {
        proxy = proxy.split(' ');
        switch (proxy[0].trim().toUpperCase()) {
          case 'DIRECT':
            delete process.env[envName];
            break;
          case 'PROXY':
            process.env[envName] = 'http://' + proxy[1];
        }
      }
    };

    PackageManager.prototype.runCommand = function(args, callback) {
      var bufferedProcess, command, errorLines, exit, outputLines, stderr, stdout;
      command = atom.packages.getApmPath();
      outputLines = [];
      stdout = function(lines) {
        return outputLines.push(lines);
      };
      errorLines = [];
      stderr = function(lines) {
        return errorLines.push(lines);
      };
      exit = function(code) {
        return callback(code, outputLines.join('\n'), errorLines.join('\n'));
      };
      args.push('--no-color');
      if (atom.config.get('core.useProxySettingsWhenCallingApm')) {
        bufferedProcess = new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit,
          autoStart: false
        });
        if (atom.resolveProxy != null) {
          this.setProxyServersAsync(function() {
            return bufferedProcess.start();
          });
        } else {
          this.setProxyServers(function() {
            return bufferedProcess.start();
          });
        }
        return bufferedProcess;
      } else {
        return new BufferedProcess({
          command: command,
          args: args,
          stdout: stdout,
          stderr: stderr,
          exit: exit
        });
      }
    };

    PackageManager.prototype.loadInstalled = function(callback) {
      var apmProcess, args, errorMessage;
      args = ['ls', '--json'];
      errorMessage = 'Fetching local packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadFeatured = function(loadThemes, callback) {
      var apmProcess, args, errorMessage, version;
      if (!callback) {
        callback = loadThemes;
        loadThemes = false;
      }
      args = ['featured', '--json'];
      version = atom.getVersion();
      if (loadThemes) {
        args.push('--themes');
      }
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching featured packages failed.';
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadOutdated = function(clearCache, callback) {
      var apmProcess, args, errorMessage, version;
      if (clearCache) {
        this.clearOutdatedCache();
      } else if (this.apmCache.loadOutdated.value && this.apmCache.loadOutdated.expiry > Date.now()) {
        return callback(null, this.apmCache.loadOutdated.value);
      }
      args = ['outdated', '--json'];
      version = atom.getVersion();
      if (semver.valid(version)) {
        args.push('--compatible', version);
      }
      errorMessage = 'Fetching outdated packages and themes failed.';
      apmProcess = this.runCommand(args, (function(_this) {
        return function(code, stdout, stderr) {
          var error, i, len, pack, packages, parseError, ref1, updatablePackages;
          if (code === 0) {
            try {
              packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
            } catch (error1) {
              parseError = error1;
              error = createJsonParseError(errorMessage, parseError, stdout);
              return callback(error);
            }
            updatablePackages = (function() {
              var i, len, results;
              results = [];
              for (i = 0, len = packages.length; i < len; i++) {
                pack = packages[i];
                if (!this.getVersionPinnedPackages().includes(pack != null ? pack.name : void 0)) {
                  results.push(pack);
                }
              }
              return results;
            }).call(_this);
            _this.apmCache.loadOutdated = {
              value: updatablePackages,
              expiry: Date.now() + _this.CACHE_EXPIRY
            };
            for (i = 0, len = updatablePackages.length; i < len; i++) {
              pack = updatablePackages[i];
              _this.emitPackageEvent('update-available', pack);
            }
            return callback(null, updatablePackages);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return callback(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getVersionPinnedPackages = function() {
      var ref1;
      return (ref1 = atom.config.get('core.versionPinnedPackages')) != null ? ref1 : [];
    };

    PackageManager.prototype.clearOutdatedCache = function() {
      return this.apmCache.loadOutdated = {
        value: null,
        expiry: 0
      };
    };

    PackageManager.prototype.loadPackage = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json'];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.loadCompatiblePackageVersion = function(packageName, callback) {
      var apmProcess, args, errorMessage;
      args = ['view', packageName, '--json', '--compatible', this.normalizeVersion(atom.getVersion())];
      errorMessage = "Fetching package '" + packageName + "' failed.";
      apmProcess = this.runCommand(args, function(code, stdout, stderr) {
        var error, packages, parseError, ref1;
        if (code === 0) {
          try {
            packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
          } catch (error1) {
            parseError = error1;
            error = createJsonParseError(errorMessage, parseError, stdout);
            return callback(error);
          }
          return callback(null, packages);
        } else {
          error = new Error(errorMessage);
          error.stdout = stdout;
          error.stderr = stderr;
          return callback(error);
        }
      });
      return handleProcessErrors(apmProcess, errorMessage, callback);
    };

    PackageManager.prototype.getInstalled = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadInstalled(function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getFeatured = function(loadThemes) {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadFeatured(!!loadThemes, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getOutdated = function(clearCache) {
      if (clearCache == null) {
        clearCache = false;
      }
      return new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadOutdated(clearCache, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.getPackage = function(packageName) {
      var base;
      return (base = this.packagePromises)[packageName] != null ? base[packageName] : base[packageName] = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.loadPackage(packageName, function(error, result) {
            if (error) {
              return reject(error);
            } else {
              return resolve(result);
            }
          });
        };
      })(this));
    };

    PackageManager.prototype.satisfiesVersion = function(version, metadata) {
      var engine, ref1, ref2;
      engine = (ref1 = (ref2 = metadata.engines) != null ? ref2.atom : void 0) != null ? ref1 : '*';
      if (!semver.validRange(engine)) {
        return false;
      }
      return semver.satisfies(version, engine);
    };

    PackageManager.prototype.normalizeVersion = function(version) {
      if (typeof version === 'string') {
        version = version.split('-')[0];
      }
      return version;
    };

    PackageManager.prototype.search = function(query, options) {
      if (options == null) {
        options = {};
      }
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var apmProcess, args, errorMessage;
          args = ['search', query, '--json'];
          if (options.themes) {
            args.push('--themes');
          } else if (options.packages) {
            args.push('--packages');
          }
          errorMessage = "Searching for \u201C" + query + "\u201D failed.";
          apmProcess = _this.runCommand(args, function(code, stdout, stderr) {
            var error, packages, parseError, ref1;
            if (code === 0) {
              try {
                packages = (ref1 = JSON.parse(stdout)) != null ? ref1 : [];
                if (options.sortBy) {
                  packages = _.sortBy(packages, function(pkg) {
                    return pkg[options.sortBy] * -1;
                  });
                }
                return resolve(packages);
              } catch (error1) {
                parseError = error1;
                error = createJsonParseError(errorMessage, parseError, stdout);
                return reject(error);
              }
            } else {
              error = new Error(errorMessage);
              error.stdout = stdout;
              error.stderr = stderr;
              return reject(error);
            }
          });
          return handleProcessErrors(apmProcess, errorMessage, function(error) {
            return reject(error);
          });
        };
      })(this));
    };

    PackageManager.prototype.update = function(pack, newVersion, callback) {
      var apmInstallSource, apmProcess, args, errorMessage, exit, name, onError, theme;
      name = pack.name, theme = pack.theme, apmInstallSource = pack.apmInstallSource;
      errorMessage = newVersion ? "Updating to \u201C" + name + "@" + newVersion + "\u201D failed." : "Updating to latest sha failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('update-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      if ((apmInstallSource != null ? apmInstallSource.type : void 0) === 'git') {
        args = ['install', apmInstallSource.source];
      } else {
        args = ['install', name + "@" + newVersion];
      }
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('updated', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('updating', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.unload = function(name) {
      if (atom.packages.isPackageLoaded(name)) {
        if (atom.packages.isPackageActive(name)) {
          atom.packages.deactivatePackage(name);
        }
        return atom.packages.unloadPackage(name);
      }
    };

    PackageManager.prototype.install = function(pack, callback) {
      var activateOnFailure, activateOnSuccess, apmProcess, args, errorMessage, exit, name, nameWithVersion, onError, theme, version;
      name = pack.name, version = pack.version, theme = pack.theme;
      activateOnSuccess = !theme && !atom.packages.isPackageDisabled(name);
      activateOnFailure = atom.packages.isPackageActive(name);
      nameWithVersion = version != null ? name + "@" + version : name;
      this.unload(name);
      args = ['install', nameWithVersion, '--json'];
      errorMessage = "Installing \u201C" + nameWithVersion + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          error.packageInstallError = !theme;
          _this.emitPackageEvent('install-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      exit = (function(_this) {
        return function(code, stdout, stderr) {
          var err, error, packageInfo;
          if (code === 0) {
            try {
              packageInfo = JSON.parse(stdout)[0];
              pack = _.extend({}, pack, packageInfo.metadata);
              name = pack.name;
            } catch (error1) {
              err = error1;
            }
            _this.clearOutdatedCache();
            if (activateOnSuccess) {
              atom.packages.activatePackage(name);
            } else {
              atom.packages.loadPackage(name);
            }
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('installed', pack);
          } else {
            if (activateOnFailure) {
              atom.packages.activatePackage(name);
            }
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this);
      this.emitPackageEvent('installing', pack);
      apmProcess = this.runCommand(args, exit);
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.uninstall = function(pack, callback) {
      var apmProcess, errorMessage, name, onError;
      name = pack.name;
      if (atom.packages.isPackageActive(name)) {
        atom.packages.deactivatePackage(name);
      }
      errorMessage = "Uninstalling \u201C" + name + "\u201D failed.";
      onError = (function(_this) {
        return function(error) {
          _this.emitPackageEvent('uninstall-failed', pack, error);
          return typeof callback === "function" ? callback(error) : void 0;
        };
      })(this);
      this.emitPackageEvent('uninstalling', pack);
      apmProcess = this.runCommand(['uninstall', '--hard', name], (function(_this) {
        return function(code, stdout, stderr) {
          var error;
          if (code === 0) {
            _this.clearOutdatedCache();
            _this.unload(name);
            _this.removePackageNameFromDisabledPackages(name);
            if (typeof callback === "function") {
              callback();
            }
            return _this.emitPackageEvent('uninstalled', pack);
          } else {
            error = new Error(errorMessage);
            error.stdout = stdout;
            error.stderr = stderr;
            return onError(error);
          }
        };
      })(this));
      return handleProcessErrors(apmProcess, errorMessage, onError);
    };

    PackageManager.prototype.installAlternative = function(pack, alternativePackageName, callback) {
      var eventArg, installPromise, uninstallPromise;
      eventArg = {
        pack: pack,
        alternative: alternativePackageName
      };
      this.emitter.emit('package-installing-alternative', eventArg);
      uninstallPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.uninstall(pack, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      installPromise = new Promise((function(_this) {
        return function(resolve, reject) {
          return _this.install({
            name: alternativePackageName
          }, function(error) {
            if (error) {
              return reject(error);
            } else {
              return resolve();
            }
          });
        };
      })(this));
      return Promise.all([uninstallPromise, installPromise]).then((function(_this) {
        return function() {
          callback(null, eventArg);
          return _this.emitter.emit('package-installed-alternative', eventArg);
        };
      })(this))["catch"]((function(_this) {
        return function(error) {
          console.error(error.message, error.stack);
          callback(error, eventArg);
          eventArg.error = error;
          return _this.emitter.emit('package-install-alternative-failed', eventArg);
        };
      })(this));
    };

    PackageManager.prototype.canUpgrade = function(installedPackage, availableVersion) {
      var installedVersion;
      if (installedPackage == null) {
        return false;
      }
      installedVersion = installedPackage.metadata.version;
      if (!semver.valid(installedVersion)) {
        return false;
      }
      if (!semver.valid(availableVersion)) {
        return false;
      }
      return semver.gt(availableVersion, installedVersion);
    };

    PackageManager.prototype.getPackageTitle = function(arg) {
      var name;
      name = arg.name;
      return _.undasherize(_.uncamelcase(name));
    };

    PackageManager.prototype.getRepositoryUrl = function(arg) {
      var metadata, ref1, ref2, repoName, repoUrl, repository;
      metadata = arg.metadata;
      repository = metadata.repository;
      repoUrl = (ref1 = (ref2 = repository != null ? repository.url : void 0) != null ? ref2 : repository) != null ? ref1 : '';
      if (repoUrl.match('git@github')) {
        repoName = repoUrl.split(':')[1];
        repoUrl = "https://github.com/" + repoName;
      }
      return repoUrl.replace(/\.git$/, '').replace(/\/+$/, '').replace(/^git\+/, '');
    };

    PackageManager.prototype.checkNativeBuildTools = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var apmProcess;
          apmProcess = _this.runCommand(['install', '--check'], function(code, stdout, stderr) {
            if (code === 0) {
              return resolve();
            } else {
              return reject(new Error());
            }
          });
          return apmProcess.onWillThrowError(function(arg) {
            var error, handle;
            error = arg.error, handle = arg.handle;
            handle();
            return reject(error);
          });
        };
      })(this));
    };

    PackageManager.prototype.removePackageNameFromDisabledPackages = function(packageName) {
      return atom.config.removeAtKeyPath('core.disabledPackages', packageName);
    };

    PackageManager.prototype.emitPackageEvent = function(eventName, pack, error) {
      var ref1, ref2, theme;
      theme = (ref1 = pack.theme) != null ? ref1 : (ref2 = pack.metadata) != null ? ref2.theme : void 0;
      eventName = theme ? "theme-" + eventName : "package-" + eventName;
      return this.emitter.emit(eventName, {
        pack: pack,
        error: error
      });
    };

    PackageManager.prototype.on = function(selectors, callback) {
      var i, len, ref1, selector, subscriptions;
      subscriptions = new CompositeDisposable;
      ref1 = selectors.split(" ");
      for (i = 0, len = ref1.length; i < len; i++) {
        selector = ref1[i];
        subscriptions.add(this.emitter.on(selector, callback));
      }
      return subscriptions;
    };

    return PackageManager;

  })();

  createJsonParseError = function(message, parseError, stdout) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = parseError.message + ": " + stdout;
    return error;
  };

  createProcessError = function(message, processError) {
    var error;
    error = new Error(message);
    error.stdout = '';
    error.stderr = processError.message;
    return error;
  };

  handleProcessErrors = function(apmProcess, message, callback) {
    return apmProcess.onWillThrowError(function(arg) {
      var error, handle;
      error = arg.error, handle = arg.handle;
      handle();
      return callback(createProcessError(message, error));
    });
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
