(function() {
  var AtomIoClient, fs, glob, path, remote, request;

  fs = require('fs-plus');

  path = require('path');

  remote = require('electron').remote;

  glob = require('glob');

  request = require('request');

  module.exports = AtomIoClient = (function() {
    function AtomIoClient(packageManager, baseURL) {
      this.packageManager = packageManager;
      this.baseURL = baseURL;
      if (this.baseURL == null) {
        this.baseURL = 'https://atom.io/api/';
      }
      this.expiry = 1000 * 60 * 60 * 12;
      this.createAvatarCache();
      this.expireAvatarCache();
    }

    AtomIoClient.prototype.avatar = function(login, callback) {
      return this.cachedAvatar(login, (function(_this) {
        return function(err, cached) {
          var stale;
          if (cached) {
            stale = Date.now() - parseInt(cached.split('-').pop()) > _this.expiry;
          }
          if (cached && (!stale || !_this.online())) {
            return callback(null, cached);
          } else {
            return _this.fetchAndCacheAvatar(login, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype["package"] = function(name, callback) {
      var packagePath;
      packagePath = "packages/" + name;
      return this.fetchFromCache(packagePath, {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.request(packagePath, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.featuredPackages = function(callback) {
      return this.fetchFromCache('packages/featured', {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.getFeatured(false, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.featuredThemes = function(callback) {
      return this.fetchFromCache('themes/featured', {}, (function(_this) {
        return function(err, data) {
          if (data) {
            return callback(null, data);
          } else {
            return _this.getFeatured(true, callback);
          }
        };
      })(this));
    };

    AtomIoClient.prototype.getFeatured = function(loadThemes, callback) {
      return this.packageManager.getFeatured(loadThemes).then((function(_this) {
        return function(packages) {
          var cached, key;
          key = loadThemes ? 'themes/featured' : 'packages/featured';
          cached = {
            data: packages,
            createdOn: Date.now()
          };
          localStorage.setItem(_this.cacheKeyForPath(key), JSON.stringify(cached));
          return callback(null, packages);
        };
      })(this))["catch"](function(error) {
        return callback(error, null);
      });
    };

    AtomIoClient.prototype.request = function(path, callback) {
      var options;
      options = {
        url: "" + this.baseURL + path,
        headers: {
          'User-Agent': navigator.userAgent
        }
      };
      return request(options, (function(_this) {
        return function(err, res, body) {
          var cached, data, error;
          try {
            data = JSON.parse(body);
          } catch (error1) {
            error = error1;
            return callback(error);
          }
          delete data.versions;
          cached = {
            data: data,
            createdOn: Date.now()
          };
          localStorage.setItem(_this.cacheKeyForPath(path), JSON.stringify(cached));
          return callback(err, cached.data);
        };
      })(this));
    };

    AtomIoClient.prototype.cacheKeyForPath = function(path) {
      return "settings-view:" + path;
    };

    AtomIoClient.prototype.online = function() {
      return navigator.onLine;
    };

    AtomIoClient.prototype.fetchFromCache = function(packagePath, options, callback) {
      var cached;
      if (!callback) {
        callback = options;
        options = {};
      }
      if (!options.force) {
        options.force = !this.online();
      }
      cached = localStorage.getItem(this.cacheKeyForPath(packagePath));
      cached = cached ? JSON.parse(cached) : void 0;
      if ((cached != null) && (!this.online() || options.force || (Date.now() - cached.createdOn < this.expiry))) {
        if (cached == null) {
          cached = {
            data: {}
          };
        }
        return callback(null, cached.data);
      } else if ((cached == null) && !this.online()) {
        return callback(null, {});
      } else {
        return callback(null, null);
      }
    };

    AtomIoClient.prototype.createAvatarCache = function() {
      return fs.makeTree(this.getCachePath());
    };

    AtomIoClient.prototype.avatarPath = function(login) {
      return path.join(this.getCachePath(), login + "-" + (Date.now()));
    };

    AtomIoClient.prototype.cachedAvatar = function(login, callback) {
      return glob(this.avatarGlob(login), (function(_this) {
        return function(err, files) {
          var createdOn, filename, i, imagePath, len, ref;
          if (err) {
            return callback(err);
          }
          files.sort().reverse();
          for (i = 0, len = files.length; i < len; i++) {
            imagePath = files[i];
            filename = path.basename(imagePath);
            ref = filename.split('-'), createdOn = ref[ref.length - 1];
            if (Date.now() - parseInt(createdOn) < _this.expiry) {
              return callback(null, imagePath);
            }
          }
          return callback(null, null);
        };
      })(this));
    };

    AtomIoClient.prototype.avatarGlob = function(login) {
      return path.join(this.getCachePath(), login + "-*([0-9])");
    };

    AtomIoClient.prototype.fetchAndCacheAvatar = function(login, callback) {
      var imagePath, requestObject;
      if (!this.online()) {
        return callback(null, null);
      } else {
        imagePath = this.avatarPath(login);
        requestObject = {
          url: "https://avatars.githubusercontent.com/" + login,
          headers: {
            'User-Agent': navigator.userAgent
          }
        };
        return request.head(requestObject, function(error, response, body) {
          var writeStream;
          if ((error != null) || response.statusCode !== 200 || !response.headers['content-type'].startsWith('image/')) {
            return callback(error);
          } else {
            writeStream = fs.createWriteStream(imagePath);
            writeStream.on('finish', function() {
              return callback(null, imagePath);
            });
            writeStream.on('error', function(error) {
              writeStream.close();
              try {
                if (fs.existsSync(imagePath)) {
                  fs.unlinkSync(imagePath);
                }
              } catch (error1) {}
              return callback(error);
            });
            return request(requestObject).pipe(writeStream);
          }
        });
      }
    };

    AtomIoClient.prototype.expireAvatarCache = function() {
      var deleteAvatar;
      deleteAvatar = (function(_this) {
        return function(child) {
          var avatarPath;
          avatarPath = path.join(_this.getCachePath(), child);
          return fs.unlink(avatarPath, function(error) {
            if (error && error.code !== 'ENOENT') {
              return console.warn("Error deleting avatar (" + error.code + "): " + avatarPath);
            }
          });
        };
      })(this);
      return fs.readdir(this.getCachePath(), function(error, _files) {
        var children, filename, files, i, key, len, parts, results, stamp;
        if (_files == null) {
          _files = [];
        }
        files = {};
        for (i = 0, len = _files.length; i < len; i++) {
          filename = _files[i];
          parts = filename.split('-');
          stamp = parts.pop();
          key = parts.join('-');
          if (files[key] == null) {
            files[key] = [];
          }
          files[key].push(key + "-" + stamp);
        }
        results = [];
        for (key in files) {
          children = files[key];
          children.sort();
          children.pop();
          results.push(children.forEach(deleteAvatar));
        }
        return results;
      });
    };

    AtomIoClient.prototype.getCachePath = function() {
      return this.cachePath != null ? this.cachePath : this.cachePath = path.join(remote.app.getPath('userData'), 'Cache', 'settings-view');
    };

    return AtomIoClient;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
