(function() {
  var CommandLogger, FileURLRegExp, NotificationIssue, StackTraceParser, TITLE_CHAR_LIMIT, UserUtilities, fs, path,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  fs = require('fs-plus');

  path = require('path');

  StackTraceParser = require('stacktrace-parser');

  CommandLogger = require('./command-logger');

  UserUtilities = require('./user-utilities');

  TITLE_CHAR_LIMIT = 100;

  FileURLRegExp = new RegExp('file://\w*/(.*)');

  module.exports = NotificationIssue = (function() {
    function NotificationIssue(notification) {
      this.notification = notification;
      this.normalizedStackPaths = bind(this.normalizedStackPaths, this);
    }

    NotificationIssue.prototype.findSimilarIssues = function() {
      var githubHeaders, issueTitle, query, repo, repoUrl;
      repoUrl = this.getRepoUrl();
      if (repoUrl == null) {
        repoUrl = 'atom/atom';
      }
      repo = repoUrl.replace(/http(s)?:\/\/(\d+\.)?github.com\//gi, '');
      issueTitle = this.getIssueTitle();
      query = issueTitle + " repo:" + repo;
      githubHeaders = new Headers({
        accept: 'application/vnd.github.v3+json',
        contentType: "application/json"
      });
      return fetch("https://api.github.com/search/issues?q=" + (encodeURIComponent(query)) + "&sort=created", {
        headers: githubHeaders
      }).then(function(r) {
        return r != null ? r.json() : void 0;
      }).then(function(data) {
        var issue, issues, j, len, ref;
        if ((data != null ? data.items : void 0) != null) {
          issues = {};
          ref = data.items;
          for (j = 0, len = ref.length; j < len; j++) {
            issue = ref[j];
            if (issue.title.indexOf(issueTitle) > -1 && (issues[issue.state] == null)) {
              issues[issue.state] = issue;
              if ((issues.open != null) && (issues.closed != null)) {
                return issues;
              }
            }
          }
          if ((issues.open != null) || (issues.closed != null)) {
            return issues;
          }
        }
        return null;
      })["catch"](function(e) {
        return null;
      });
    };

    NotificationIssue.prototype.getIssueUrlForSystem = function() {
      return this.getIssueUrl().then(function(issueUrl) {
        return fetch("https://is.gd/create.php?format=simple", {
          method: 'POST',
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded'
          },
          body: "url=" + (encodeURIComponent(issueUrl))
        }).then(function(r) {
          return r.text();
        })["catch"](function(e) {
          return null;
        });
      });
    };

    NotificationIssue.prototype.getIssueUrl = function() {
      return this.getIssueBody().then((function(_this) {
        return function(issueBody) {
          var repoUrl;
          repoUrl = _this.getRepoUrl();
          if (repoUrl == null) {
            repoUrl = 'https://github.com/atom/atom';
          }
          return repoUrl + "/issues/new?title=" + (_this.encodeURI(_this.getIssueTitle())) + "&body=" + (_this.encodeURI(issueBody));
        };
      })(this));
    };

    NotificationIssue.prototype.encodeURI = function(str) {
      return encodeURI(str).replace(/#/g, '%23').replace(/;/g, '%3B').replace(/%20/g, '+');
    };

    NotificationIssue.prototype.getIssueTitle = function() {
      var title;
      title = this.notification.getMessage();
      title = title.replace(process.env.ATOM_HOME, '$ATOM_HOME');
      if (process.platform === 'win32') {
        title = title.replace(process.env.USERPROFILE, '~');
        title = title.replace(path.sep, path.posix.sep);
      } else {
        title = title.replace(process.env.HOME, '~');
      }
      if (title.length > TITLE_CHAR_LIMIT) {
        title = title.substring(0, TITLE_CHAR_LIMIT - 3) + '...';
      }
      return title.replace(/\r?\n|\r/g, "");
    };

    NotificationIssue.prototype.getIssueBody = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var nonCorePackagesPromise, systemPromise;
          if (_this.issueBody) {
            return resolve(_this.issueBody);
          }
          systemPromise = UserUtilities.getOSVersion();
          nonCorePackagesPromise = UserUtilities.getNonCorePackages();
          return Promise.all([systemPromise, nonCorePackagesPromise]).then(function(all) {
            var copyText, message, nonCorePackages, options, packageMessage, packageName, packageVersion, ref, ref1, repoUrl, rootUserStatus, systemName, systemUser;
            systemName = all[0], nonCorePackages = all[1];
            message = _this.notification.getMessage();
            options = _this.notification.getOptions();
            repoUrl = _this.getRepoUrl();
            packageName = _this.getPackageName();
            if (packageName != null) {
              packageVersion = (ref = atom.packages.getLoadedPackage(packageName)) != null ? (ref1 = ref.metadata) != null ? ref1.version : void 0 : void 0;
            }
            copyText = '';
            systemUser = process.env.USER;
            rootUserStatus = '';
            if (systemUser === 'root') {
              rootUserStatus = '**User**: root';
            }
            if ((packageName != null) && (repoUrl != null)) {
              packageMessage = "[" + packageName + "](" + repoUrl + ") package " + packageVersion;
            } else if (packageName != null) {
              packageMessage = "'" + packageName + "' package v" + packageVersion;
            } else {
              packageMessage = 'Atom Core';
            }
            _this.issueBody = "[Enter steps to reproduce:]\n\n1. ...\n2. ...\n\n**Atom**: " + (atom.getVersion()) + " " + process.arch + "\n**Electron**: " + process.versions.electron + "\n**OS**: " + systemName + "\n**Thrown From**: " + packageMessage + "\n" + rootUserStatus + "\n\n### Stack Trace\n\n" + message + "\n\n```\nAt " + options.detail + "\n\n" + (_this.normalizedStackPaths(options.stack)) + "\n```\n\n### Commands\n\n" + (CommandLogger.instance().getText()) + "\n\n### Non-Core Packages\n\n```\n" + (nonCorePackages.join('\n')) + "\n```\n\n" + copyText;
            return resolve(_this.issueBody);
          });
        };
      })(this));
    };

    NotificationIssue.prototype.normalizedStackPaths = function(stack) {
      return stack.replace(/(^\W+at )([\w.]{2,} [(])?(.*)(:\d+:\d+[)]?)/gm, (function(_this) {
        return function(m, p1, p2, p3, p4) {
          return p1 + (p2 || '') + _this.normalizePath(p3) + p4;
        };
      })(this));
    };

    NotificationIssue.prototype.normalizePath = function(path) {
      return path.replace('file:///', '').replace(/[\/]/g, '\\').replace(fs.getHomeDirectory(), '~').replace(/\\/g, '/').replace(/.*(\/(app\.asar|packages\/).*)/, '$1');
    };

    NotificationIssue.prototype.getRepoUrl = function() {
      var packageName, packagePath, ref, ref1, ref2, ref3, ref4, repo, repoUrl;
      packageName = this.getPackageName();
      if (packageName == null) {
        return;
      }
      repo = (ref = atom.packages.getLoadedPackage(packageName)) != null ? (ref1 = ref.metadata) != null ? ref1.repository : void 0 : void 0;
      repoUrl = (ref2 = repo != null ? repo.url : void 0) != null ? ref2 : repo;
      if (!repoUrl) {
        if (packagePath = atom.packages.resolvePackagePath(packageName)) {
          try {
            repo = (ref3 = JSON.parse(fs.readFileSync(path.join(packagePath, 'package.json')))) != null ? ref3.repository : void 0;
            repoUrl = (ref4 = repo != null ? repo.url : void 0) != null ? ref4 : repo;
          } catch (error) {}
        }
      }
      return repoUrl != null ? repoUrl.replace(/\.git$/, '').replace(/^git\+/, '') : void 0;
    };

    NotificationIssue.prototype.getPackageNameFromFilePath = function(filePath) {
      var packageName, ref, ref1, ref2, ref3;
      if (!filePath) {
        return;
      }
      packageName = (ref = /\/\.atom\/dev\/packages\/([^\/]+)\//.exec(filePath)) != null ? ref[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref1 = /\\\.atom\\dev\\packages\\([^\\]+)\\/.exec(filePath)) != null ? ref1[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref2 = /\/\.atom\/packages\/([^\/]+)\//.exec(filePath)) != null ? ref2[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref3 = /\\\.atom\\packages\\([^\\]+)\\/.exec(filePath)) != null ? ref3[1] : void 0;
      if (packageName) {
        return packageName;
      }
    };

    NotificationIssue.prototype.getPackageName = function() {
      var file, getPackageName, i, j, options, packageName, packagePath, packagePaths, ref, stack;
      options = this.notification.getOptions();
      if (options.packageName != null) {
        return options.packageName;
      }
      if (!((options.stack != null) || (options.detail != null))) {
        return;
      }
      packagePaths = this.getPackagePathsByPackageName();
      for (packageName in packagePaths) {
        packagePath = packagePaths[packageName];
        if (packagePath.indexOf(path.join('.atom', 'dev', 'packages')) > -1 || packagePath.indexOf(path.join('.atom', 'packages')) > -1) {
          packagePaths[packageName] = fs.realpathSync(packagePath);
        }
      }
      getPackageName = (function(_this) {
        return function(filePath) {
          var isSubfolder, match, packName;
          filePath = /\((.+?):\d+|\((.+)\)|(.+)/.exec(filePath)[0];
          if (match = FileURLRegExp.exec(filePath)) {
            filePath = match[1];
          }
          filePath = path.normalize(filePath);
          if (path.isAbsolute(filePath)) {
            for (packName in packagePaths) {
              packagePath = packagePaths[packName];
              if (filePath === 'node.js') {
                continue;
              }
              isSubfolder = filePath.indexOf(path.normalize(packagePath + path.sep)) === 0;
              if (isSubfolder) {
                return packName;
              }
            }
          }
          return _this.getPackageNameFromFilePath(filePath);
        };
      })(this);
      if ((options.detail != null) && (packageName = getPackageName(options.detail))) {
        return packageName;
      }
      if (options.stack != null) {
        stack = StackTraceParser.parse(options.stack);
        for (i = j = 0, ref = stack.length; 0 <= ref ? j < ref : j > ref; i = 0 <= ref ? ++j : --j) {
          file = stack[i].file;
          if (!file) {
            return;
          }
          packageName = getPackageName(file);
          if (packageName != null) {
            return packageName;
          }
        }
      }
    };

    NotificationIssue.prototype.getPackagePathsByPackageName = function() {
      var j, len, pack, packagePathsByPackageName, ref;
      packagePathsByPackageName = {};
      ref = atom.packages.getLoadedPackages();
      for (j = 0, len = ref.length; j < len; j++) {
        pack = ref[j];
        packagePathsByPackageName[pack.name] = pack.path;
      }
      return packagePathsByPackageName;
    };

    return NotificationIssue;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
