'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _hunk = require('./hunk');

var _hunk2 = _interopRequireDefault(_hunk);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

let FilePatch = class FilePatch {
  constructor(oldPath, newPath, status, hunks) {
    this.oldPath = oldPath;
    this.newPath = newPath;
    this.status = status;
    this.hunks = hunks;
    this.changedLineCount = this.hunks.reduce((acc, hunk) => {
      return acc + hunk.getLines().filter(line => line.isChanged()).length;
    }, 0);
  }

  getOldPath() {
    return this.oldPath;
  }

  getNewPath() {
    return this.newPath;
  }

  getPath() {
    return this.getOldPath() || this.getNewPath();
  }

  getStatus() {
    return this.status;
  }

  getHunks() {
    return this.hunks;
  }

  getStagePatchForHunk(selectedHunk) {
    return this.getStagePatchForLines(new Set(selectedHunk.getLines()));
  }

  getStagePatchForLines(selectedLines) {
    if (this.changedLineCount === [...selectedLines].filter(line => line.isChanged()).length) {
      return this;
    }

    let delta = 0;
    const hunks = [];
    for (const hunk of this.getHunks()) {
      const newStartRow = (hunk.getNewStartRow() || 1) + delta;
      let newLineNumber = newStartRow;
      const lines = [];
      let hunkContainsSelectedLines = false;
      for (const line of hunk.getLines()) {
        if (line.getStatus() === 'nonewline') {
          lines.push(line.copy({ oldLineNumber: -1, newLineNumber: -1 }));
        } else if (selectedLines.has(line)) {
          hunkContainsSelectedLines = true;
          if (line.getStatus() === 'deleted') {
            lines.push(line.copy());
          } else {
            lines.push(line.copy({ newLineNumber: newLineNumber++ }));
          }
        } else if (line.getStatus() === 'deleted') {
          lines.push(line.copy({ newLineNumber: newLineNumber++, status: 'unchanged' }));
        } else if (line.getStatus() === 'unchanged') {
          lines.push(line.copy({ newLineNumber: newLineNumber++ }));
        }
      }
      const newRowCount = newLineNumber - newStartRow;
      if (hunkContainsSelectedLines) {
        // eslint-disable-next-line max-len
        hunks.push(new _hunk2.default(hunk.getOldStartRow(), newStartRow, hunk.getOldRowCount(), newRowCount, hunk.getSectionHeading(), lines));
      }
      delta += newRowCount - hunk.getNewRowCount();
    }

    return new FilePatch(this.getOldPath(), this.getNewPath() ? this.getNewPath() : this.getOldPath(), this.getStatus(), hunks);
  }

  getUnstagePatch() {
    let invertedStatus;
    switch (this.getStatus()) {
      case 'modified':
        invertedStatus = 'modified';
        break;
      case 'added':
        invertedStatus = 'deleted';
        break;
      case 'deleted':
        invertedStatus = 'added';
        break;
      default:
        throw new Error(`Unknown Status: ${this.getStatus()}`);
    }
    const invertedHunks = this.getHunks().map(h => h.invert());
    return new FilePatch(this.getNewPath(), this.getOldPath(), invertedStatus, invertedHunks);
  }

  getUnstagePatchForHunk(hunk) {
    return this.getUnstagePatchForLines(new Set(hunk.getLines()));
  }

  getUnstagePatchForLines(selectedLines) {
    if (this.changedLineCount === [...selectedLines].filter(line => line.isChanged()).length) {
      return this.getUnstagePatch();
    }

    let delta = 0;
    const hunks = [];
    for (const hunk of this.getHunks()) {
      const oldStartRow = (hunk.getOldStartRow() || 1) + delta;
      let oldLineNumber = oldStartRow;
      const lines = [];
      let hunkContainsSelectedLines = false;
      for (const line of hunk.getLines()) {
        if (line.getStatus() === 'nonewline') {
          lines.push(line.copy({ oldLineNumber: -1, newLineNumber: -1 }));
        } else if (selectedLines.has(line)) {
          hunkContainsSelectedLines = true;
          if (line.getStatus() === 'added') {
            lines.push(line.copy());
          } else {
            lines.push(line.copy({ oldLineNumber: oldLineNumber++ }));
          }
        } else if (line.getStatus() === 'added') {
          lines.push(line.copy({ oldLineNumber: oldLineNumber++, status: 'unchanged' }));
        } else if (line.getStatus() === 'unchanged') {
          lines.push(line.copy({ oldLineNumber: oldLineNumber++ }));
        }
      }
      const oldRowCount = oldLineNumber - oldStartRow;
      if (hunkContainsSelectedLines) {
        // eslint-disable-next-line max-len
        hunks.push(new _hunk2.default(oldStartRow, hunk.getNewStartRow(), oldRowCount, hunk.getNewRowCount(), hunk.getSectionHeading(), lines));
      }
      delta += oldRowCount - hunk.getOldRowCount();
    }

    return new FilePatch(this.getOldPath() ? this.getOldPath() : this.getNewPath(), this.getNewPath(), this.getStatus(), hunks).getUnstagePatch();
  }

  toString() {
    return this.getHunks().map(h => h.toString()).join('');
  }

  getHeaderString() {
    let header = this.getOldPath() ? `--- a/${(0, _helpers.toGitPathSep)(this.getOldPath())}` : '--- /dev/null';
    header += '\n';
    header += this.getNewPath() ? `+++ b/${(0, _helpers.toGitPathSep)(this.getNewPath())}` : '+++ /dev/null';
    header += '\n';
    return header;
  }
};
exports.default = FilePatch;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImZpbGUtcGF0Y2guanMiXSwibmFtZXMiOlsiRmlsZVBhdGNoIiwiY29uc3RydWN0b3IiLCJvbGRQYXRoIiwibmV3UGF0aCIsInN0YXR1cyIsImh1bmtzIiwiY2hhbmdlZExpbmVDb3VudCIsInJlZHVjZSIsImFjYyIsImh1bmsiLCJnZXRMaW5lcyIsImZpbHRlciIsImxpbmUiLCJpc0NoYW5nZWQiLCJsZW5ndGgiLCJnZXRPbGRQYXRoIiwiZ2V0TmV3UGF0aCIsImdldFBhdGgiLCJnZXRTdGF0dXMiLCJnZXRIdW5rcyIsImdldFN0YWdlUGF0Y2hGb3JIdW5rIiwic2VsZWN0ZWRIdW5rIiwiZ2V0U3RhZ2VQYXRjaEZvckxpbmVzIiwiU2V0Iiwic2VsZWN0ZWRMaW5lcyIsImRlbHRhIiwibmV3U3RhcnRSb3ciLCJnZXROZXdTdGFydFJvdyIsIm5ld0xpbmVOdW1iZXIiLCJsaW5lcyIsImh1bmtDb250YWluc1NlbGVjdGVkTGluZXMiLCJwdXNoIiwiY29weSIsIm9sZExpbmVOdW1iZXIiLCJoYXMiLCJuZXdSb3dDb3VudCIsImdldE9sZFN0YXJ0Um93IiwiZ2V0T2xkUm93Q291bnQiLCJnZXRTZWN0aW9uSGVhZGluZyIsImdldE5ld1Jvd0NvdW50IiwiZ2V0VW5zdGFnZVBhdGNoIiwiaW52ZXJ0ZWRTdGF0dXMiLCJFcnJvciIsImludmVydGVkSHVua3MiLCJtYXAiLCJoIiwiaW52ZXJ0IiwiZ2V0VW5zdGFnZVBhdGNoRm9ySHVuayIsImdldFVuc3RhZ2VQYXRjaEZvckxpbmVzIiwib2xkU3RhcnRSb3ciLCJvbGRSb3dDb3VudCIsInRvU3RyaW5nIiwiam9pbiIsImdldEhlYWRlclN0cmluZyIsImhlYWRlciJdLCJtYXBwaW5ncyI6Ijs7Ozs7OztBQUFBOzs7O0FBQ0E7Ozs7SUFFcUJBLFMsR0FBTixNQUFNQSxTQUFOLENBQWdCO0FBQzdCQyxjQUFZQyxPQUFaLEVBQXFCQyxPQUFyQixFQUE4QkMsTUFBOUIsRUFBc0NDLEtBQXRDLEVBQTZDO0FBQzNDLFNBQUtILE9BQUwsR0FBZUEsT0FBZjtBQUNBLFNBQUtDLE9BQUwsR0FBZUEsT0FBZjtBQUNBLFNBQUtDLE1BQUwsR0FBY0EsTUFBZDtBQUNBLFNBQUtDLEtBQUwsR0FBYUEsS0FBYjtBQUNBLFNBQUtDLGdCQUFMLEdBQXdCLEtBQUtELEtBQUwsQ0FBV0UsTUFBWCxDQUFrQixDQUFDQyxHQUFELEVBQU1DLElBQU4sS0FBZTtBQUN2RCxhQUFPRCxNQUFNQyxLQUFLQyxRQUFMLEdBQWdCQyxNQUFoQixDQUF1QkMsUUFBUUEsS0FBS0MsU0FBTCxFQUEvQixFQUFpREMsTUFBOUQ7QUFDRCxLQUZ1QixFQUVyQixDQUZxQixDQUF4QjtBQUdEOztBQUVEQyxlQUFhO0FBQ1gsV0FBTyxLQUFLYixPQUFaO0FBQ0Q7O0FBRURjLGVBQWE7QUFDWCxXQUFPLEtBQUtiLE9BQVo7QUFDRDs7QUFFRGMsWUFBVTtBQUNSLFdBQU8sS0FBS0YsVUFBTCxNQUFxQixLQUFLQyxVQUFMLEVBQTVCO0FBQ0Q7O0FBRURFLGNBQVk7QUFDVixXQUFPLEtBQUtkLE1BQVo7QUFDRDs7QUFFRGUsYUFBVztBQUNULFdBQU8sS0FBS2QsS0FBWjtBQUNEOztBQUVEZSx1QkFBcUJDLFlBQXJCLEVBQW1DO0FBQ2pDLFdBQU8sS0FBS0MscUJBQUwsQ0FBMkIsSUFBSUMsR0FBSixDQUFRRixhQUFhWCxRQUFiLEVBQVIsQ0FBM0IsQ0FBUDtBQUNEOztBQUVEWSx3QkFBc0JFLGFBQXRCLEVBQXFDO0FBQ25DLFFBQUksS0FBS2xCLGdCQUFMLEtBQTBCLENBQUMsR0FBR2tCLGFBQUosRUFBbUJiLE1BQW5CLENBQTBCQyxRQUFRQSxLQUFLQyxTQUFMLEVBQWxDLEVBQW9EQyxNQUFsRixFQUEwRjtBQUN4RixhQUFPLElBQVA7QUFDRDs7QUFFRCxRQUFJVyxRQUFRLENBQVo7QUFDQSxVQUFNcEIsUUFBUSxFQUFkO0FBQ0EsU0FBSyxNQUFNSSxJQUFYLElBQW1CLEtBQUtVLFFBQUwsRUFBbkIsRUFBb0M7QUFDbEMsWUFBTU8sY0FBYyxDQUFDakIsS0FBS2tCLGNBQUwsTUFBeUIsQ0FBMUIsSUFBK0JGLEtBQW5EO0FBQ0EsVUFBSUcsZ0JBQWdCRixXQUFwQjtBQUNBLFlBQU1HLFFBQVEsRUFBZDtBQUNBLFVBQUlDLDRCQUE0QixLQUFoQztBQUNBLFdBQUssTUFBTWxCLElBQVgsSUFBbUJILEtBQUtDLFFBQUwsRUFBbkIsRUFBb0M7QUFDbEMsWUFBSUUsS0FBS00sU0FBTCxPQUFxQixXQUF6QixFQUFzQztBQUNwQ1csZ0JBQU1FLElBQU4sQ0FBV25CLEtBQUtvQixJQUFMLENBQVUsRUFBQ0MsZUFBZSxDQUFDLENBQWpCLEVBQW9CTCxlQUFlLENBQUMsQ0FBcEMsRUFBVixDQUFYO0FBQ0QsU0FGRCxNQUVPLElBQUlKLGNBQWNVLEdBQWQsQ0FBa0J0QixJQUFsQixDQUFKLEVBQTZCO0FBQ2xDa0Isc0NBQTRCLElBQTVCO0FBQ0EsY0FBSWxCLEtBQUtNLFNBQUwsT0FBcUIsU0FBekIsRUFBb0M7QUFDbENXLGtCQUFNRSxJQUFOLENBQVduQixLQUFLb0IsSUFBTCxFQUFYO0FBQ0QsV0FGRCxNQUVPO0FBQ0xILGtCQUFNRSxJQUFOLENBQVduQixLQUFLb0IsSUFBTCxDQUFVLEVBQUNKLGVBQWVBLGVBQWhCLEVBQVYsQ0FBWDtBQUNEO0FBQ0YsU0FQTSxNQU9BLElBQUloQixLQUFLTSxTQUFMLE9BQXFCLFNBQXpCLEVBQW9DO0FBQ3pDVyxnQkFBTUUsSUFBTixDQUFXbkIsS0FBS29CLElBQUwsQ0FBVSxFQUFDSixlQUFlQSxlQUFoQixFQUFpQ3hCLFFBQVEsV0FBekMsRUFBVixDQUFYO0FBQ0QsU0FGTSxNQUVBLElBQUlRLEtBQUtNLFNBQUwsT0FBcUIsV0FBekIsRUFBc0M7QUFDM0NXLGdCQUFNRSxJQUFOLENBQVduQixLQUFLb0IsSUFBTCxDQUFVLEVBQUNKLGVBQWVBLGVBQWhCLEVBQVYsQ0FBWDtBQUNEO0FBQ0Y7QUFDRCxZQUFNTyxjQUFjUCxnQkFBZ0JGLFdBQXBDO0FBQ0EsVUFBSUkseUJBQUosRUFBK0I7QUFDN0I7QUFDQXpCLGNBQU0wQixJQUFOLENBQVcsbUJBQVN0QixLQUFLMkIsY0FBTCxFQUFULEVBQWdDVixXQUFoQyxFQUE2Q2pCLEtBQUs0QixjQUFMLEVBQTdDLEVBQW9FRixXQUFwRSxFQUFpRjFCLEtBQUs2QixpQkFBTCxFQUFqRixFQUEyR1QsS0FBM0csQ0FBWDtBQUNEO0FBQ0RKLGVBQVNVLGNBQWMxQixLQUFLOEIsY0FBTCxFQUF2QjtBQUNEOztBQUVELFdBQU8sSUFBSXZDLFNBQUosQ0FDTCxLQUFLZSxVQUFMLEVBREssRUFFTCxLQUFLQyxVQUFMLEtBQW9CLEtBQUtBLFVBQUwsRUFBcEIsR0FBd0MsS0FBS0QsVUFBTCxFQUZuQyxFQUdMLEtBQUtHLFNBQUwsRUFISyxFQUlMYixLQUpLLENBQVA7QUFNRDs7QUFFRG1DLG9CQUFrQjtBQUNoQixRQUFJQyxjQUFKO0FBQ0EsWUFBUSxLQUFLdkIsU0FBTCxFQUFSO0FBQ0UsV0FBSyxVQUFMO0FBQ0V1Qix5QkFBaUIsVUFBakI7QUFDQTtBQUNGLFdBQUssT0FBTDtBQUNFQSx5QkFBaUIsU0FBakI7QUFDQTtBQUNGLFdBQUssU0FBTDtBQUNFQSx5QkFBaUIsT0FBakI7QUFDQTtBQUNGO0FBQ0UsY0FBTSxJQUFJQyxLQUFKLENBQVcsbUJBQWtCLEtBQUt4QixTQUFMLEVBQWlCLEVBQTlDLENBQU47QUFYSjtBQWFBLFVBQU15QixnQkFBZ0IsS0FBS3hCLFFBQUwsR0FBZ0J5QixHQUFoQixDQUFvQkMsS0FBS0EsRUFBRUMsTUFBRixFQUF6QixDQUF0QjtBQUNBLFdBQU8sSUFBSTlDLFNBQUosQ0FDTCxLQUFLZ0IsVUFBTCxFQURLLEVBRUwsS0FBS0QsVUFBTCxFQUZLLEVBR0wwQixjQUhLLEVBSUxFLGFBSkssQ0FBUDtBQU1EOztBQUVESSx5QkFBdUJ0QyxJQUF2QixFQUE2QjtBQUMzQixXQUFPLEtBQUt1Qyx1QkFBTCxDQUE2QixJQUFJekIsR0FBSixDQUFRZCxLQUFLQyxRQUFMLEVBQVIsQ0FBN0IsQ0FBUDtBQUNEOztBQUVEc0MsMEJBQXdCeEIsYUFBeEIsRUFBdUM7QUFDckMsUUFBSSxLQUFLbEIsZ0JBQUwsS0FBMEIsQ0FBQyxHQUFHa0IsYUFBSixFQUFtQmIsTUFBbkIsQ0FBMEJDLFFBQVFBLEtBQUtDLFNBQUwsRUFBbEMsRUFBb0RDLE1BQWxGLEVBQTBGO0FBQ3hGLGFBQU8sS0FBSzBCLGVBQUwsRUFBUDtBQUNEOztBQUVELFFBQUlmLFFBQVEsQ0FBWjtBQUNBLFVBQU1wQixRQUFRLEVBQWQ7QUFDQSxTQUFLLE1BQU1JLElBQVgsSUFBbUIsS0FBS1UsUUFBTCxFQUFuQixFQUFvQztBQUNsQyxZQUFNOEIsY0FBYyxDQUFDeEMsS0FBSzJCLGNBQUwsTUFBeUIsQ0FBMUIsSUFBK0JYLEtBQW5EO0FBQ0EsVUFBSVEsZ0JBQWdCZ0IsV0FBcEI7QUFDQSxZQUFNcEIsUUFBUSxFQUFkO0FBQ0EsVUFBSUMsNEJBQTRCLEtBQWhDO0FBQ0EsV0FBSyxNQUFNbEIsSUFBWCxJQUFtQkgsS0FBS0MsUUFBTCxFQUFuQixFQUFvQztBQUNsQyxZQUFJRSxLQUFLTSxTQUFMLE9BQXFCLFdBQXpCLEVBQXNDO0FBQ3BDVyxnQkFBTUUsSUFBTixDQUFXbkIsS0FBS29CLElBQUwsQ0FBVSxFQUFDQyxlQUFlLENBQUMsQ0FBakIsRUFBb0JMLGVBQWUsQ0FBQyxDQUFwQyxFQUFWLENBQVg7QUFDRCxTQUZELE1BRU8sSUFBSUosY0FBY1UsR0FBZCxDQUFrQnRCLElBQWxCLENBQUosRUFBNkI7QUFDbENrQixzQ0FBNEIsSUFBNUI7QUFDQSxjQUFJbEIsS0FBS00sU0FBTCxPQUFxQixPQUF6QixFQUFrQztBQUNoQ1csa0JBQU1FLElBQU4sQ0FBV25CLEtBQUtvQixJQUFMLEVBQVg7QUFDRCxXQUZELE1BRU87QUFDTEgsa0JBQU1FLElBQU4sQ0FBV25CLEtBQUtvQixJQUFMLENBQVUsRUFBQ0MsZUFBZUEsZUFBaEIsRUFBVixDQUFYO0FBQ0Q7QUFDRixTQVBNLE1BT0EsSUFBSXJCLEtBQUtNLFNBQUwsT0FBcUIsT0FBekIsRUFBa0M7QUFDdkNXLGdCQUFNRSxJQUFOLENBQVduQixLQUFLb0IsSUFBTCxDQUFVLEVBQUNDLGVBQWVBLGVBQWhCLEVBQWlDN0IsUUFBUSxXQUF6QyxFQUFWLENBQVg7QUFDRCxTQUZNLE1BRUEsSUFBSVEsS0FBS00sU0FBTCxPQUFxQixXQUF6QixFQUFzQztBQUMzQ1csZ0JBQU1FLElBQU4sQ0FBV25CLEtBQUtvQixJQUFMLENBQVUsRUFBQ0MsZUFBZUEsZUFBaEIsRUFBVixDQUFYO0FBQ0Q7QUFDRjtBQUNELFlBQU1pQixjQUFjakIsZ0JBQWdCZ0IsV0FBcEM7QUFDQSxVQUFJbkIseUJBQUosRUFBK0I7QUFDN0I7QUFDQXpCLGNBQU0wQixJQUFOLENBQVcsbUJBQVNrQixXQUFULEVBQXNCeEMsS0FBS2tCLGNBQUwsRUFBdEIsRUFBNkN1QixXQUE3QyxFQUEwRHpDLEtBQUs4QixjQUFMLEVBQTFELEVBQWlGOUIsS0FBSzZCLGlCQUFMLEVBQWpGLEVBQTJHVCxLQUEzRyxDQUFYO0FBQ0Q7QUFDREosZUFBU3lCLGNBQWN6QyxLQUFLNEIsY0FBTCxFQUF2QjtBQUNEOztBQUVELFdBQU8sSUFBSXJDLFNBQUosQ0FDTCxLQUFLZSxVQUFMLEtBQW9CLEtBQUtBLFVBQUwsRUFBcEIsR0FBd0MsS0FBS0MsVUFBTCxFQURuQyxFQUVMLEtBQUtBLFVBQUwsRUFGSyxFQUdMLEtBQUtFLFNBQUwsRUFISyxFQUlMYixLQUpLLEVBS0xtQyxlQUxLLEVBQVA7QUFNRDs7QUFFRFcsYUFBVztBQUNULFdBQU8sS0FBS2hDLFFBQUwsR0FBZ0J5QixHQUFoQixDQUFvQkMsS0FBS0EsRUFBRU0sUUFBRixFQUF6QixFQUF1Q0MsSUFBdkMsQ0FBNEMsRUFBNUMsQ0FBUDtBQUNEOztBQUVEQyxvQkFBa0I7QUFDaEIsUUFBSUMsU0FBUyxLQUFLdkMsVUFBTCxLQUFxQixTQUFRLDJCQUFhLEtBQUtBLFVBQUwsRUFBYixDQUFnQyxFQUE3RCxHQUFpRSxlQUE5RTtBQUNBdUMsY0FBVSxJQUFWO0FBQ0FBLGNBQVUsS0FBS3RDLFVBQUwsS0FBcUIsU0FBUSwyQkFBYSxLQUFLQSxVQUFMLEVBQWIsQ0FBZ0MsRUFBN0QsR0FBaUUsZUFBM0U7QUFDQXNDLGNBQVUsSUFBVjtBQUNBLFdBQU9BLE1BQVA7QUFDRDtBQWpLNEIsQztrQkFBVnRELFMiLCJmaWxlIjoiZmlsZS1wYXRjaC5qcyIsInNvdXJjZVJvb3QiOiIvaG9tZS90cmF2aXMvYnVpbGQvYXRvbS9hdG9tL291dC9hcHAvbm9kZV9tb2R1bGVzL2dpdGh1YiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBIdW5rIGZyb20gJy4vaHVuayc7XG5pbXBvcnQge3RvR2l0UGF0aFNlcH0gZnJvbSAnLi4vaGVscGVycyc7XG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIEZpbGVQYXRjaCB7XG4gIGNvbnN0cnVjdG9yKG9sZFBhdGgsIG5ld1BhdGgsIHN0YXR1cywgaHVua3MpIHtcbiAgICB0aGlzLm9sZFBhdGggPSBvbGRQYXRoO1xuICAgIHRoaXMubmV3UGF0aCA9IG5ld1BhdGg7XG4gICAgdGhpcy5zdGF0dXMgPSBzdGF0dXM7XG4gICAgdGhpcy5odW5rcyA9IGh1bmtzO1xuICAgIHRoaXMuY2hhbmdlZExpbmVDb3VudCA9IHRoaXMuaHVua3MucmVkdWNlKChhY2MsIGh1bmspID0+IHtcbiAgICAgIHJldHVybiBhY2MgKyBodW5rLmdldExpbmVzKCkuZmlsdGVyKGxpbmUgPT4gbGluZS5pc0NoYW5nZWQoKSkubGVuZ3RoO1xuICAgIH0sIDApO1xuICB9XG5cbiAgZ2V0T2xkUGF0aCgpIHtcbiAgICByZXR1cm4gdGhpcy5vbGRQYXRoO1xuICB9XG5cbiAgZ2V0TmV3UGF0aCgpIHtcbiAgICByZXR1cm4gdGhpcy5uZXdQYXRoO1xuICB9XG5cbiAgZ2V0UGF0aCgpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRPbGRQYXRoKCkgfHwgdGhpcy5nZXROZXdQYXRoKCk7XG4gIH1cblxuICBnZXRTdGF0dXMoKSB7XG4gICAgcmV0dXJuIHRoaXMuc3RhdHVzO1xuICB9XG5cbiAgZ2V0SHVua3MoKSB7XG4gICAgcmV0dXJuIHRoaXMuaHVua3M7XG4gIH1cblxuICBnZXRTdGFnZVBhdGNoRm9ySHVuayhzZWxlY3RlZEh1bmspIHtcbiAgICByZXR1cm4gdGhpcy5nZXRTdGFnZVBhdGNoRm9yTGluZXMobmV3IFNldChzZWxlY3RlZEh1bmsuZ2V0TGluZXMoKSkpO1xuICB9XG5cbiAgZ2V0U3RhZ2VQYXRjaEZvckxpbmVzKHNlbGVjdGVkTGluZXMpIHtcbiAgICBpZiAodGhpcy5jaGFuZ2VkTGluZUNvdW50ID09PSBbLi4uc2VsZWN0ZWRMaW5lc10uZmlsdGVyKGxpbmUgPT4gbGluZS5pc0NoYW5nZWQoKSkubGVuZ3RoKSB7XG4gICAgICByZXR1cm4gdGhpcztcbiAgICB9XG5cbiAgICBsZXQgZGVsdGEgPSAwO1xuICAgIGNvbnN0IGh1bmtzID0gW107XG4gICAgZm9yIChjb25zdCBodW5rIG9mIHRoaXMuZ2V0SHVua3MoKSkge1xuICAgICAgY29uc3QgbmV3U3RhcnRSb3cgPSAoaHVuay5nZXROZXdTdGFydFJvdygpIHx8IDEpICsgZGVsdGE7XG4gICAgICBsZXQgbmV3TGluZU51bWJlciA9IG5ld1N0YXJ0Um93O1xuICAgICAgY29uc3QgbGluZXMgPSBbXTtcbiAgICAgIGxldCBodW5rQ29udGFpbnNTZWxlY3RlZExpbmVzID0gZmFsc2U7XG4gICAgICBmb3IgKGNvbnN0IGxpbmUgb2YgaHVuay5nZXRMaW5lcygpKSB7XG4gICAgICAgIGlmIChsaW5lLmdldFN0YXR1cygpID09PSAnbm9uZXdsaW5lJykge1xuICAgICAgICAgIGxpbmVzLnB1c2gobGluZS5jb3B5KHtvbGRMaW5lTnVtYmVyOiAtMSwgbmV3TGluZU51bWJlcjogLTF9KSk7XG4gICAgICAgIH0gZWxzZSBpZiAoc2VsZWN0ZWRMaW5lcy5oYXMobGluZSkpIHtcbiAgICAgICAgICBodW5rQ29udGFpbnNTZWxlY3RlZExpbmVzID0gdHJ1ZTtcbiAgICAgICAgICBpZiAobGluZS5nZXRTdGF0dXMoKSA9PT0gJ2RlbGV0ZWQnKSB7XG4gICAgICAgICAgICBsaW5lcy5wdXNoKGxpbmUuY29weSgpKTtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgbGluZXMucHVzaChsaW5lLmNvcHkoe25ld0xpbmVOdW1iZXI6IG5ld0xpbmVOdW1iZXIrK30pKTtcbiAgICAgICAgICB9XG4gICAgICAgIH0gZWxzZSBpZiAobGluZS5nZXRTdGF0dXMoKSA9PT0gJ2RlbGV0ZWQnKSB7XG4gICAgICAgICAgbGluZXMucHVzaChsaW5lLmNvcHkoe25ld0xpbmVOdW1iZXI6IG5ld0xpbmVOdW1iZXIrKywgc3RhdHVzOiAndW5jaGFuZ2VkJ30pKTtcbiAgICAgICAgfSBlbHNlIGlmIChsaW5lLmdldFN0YXR1cygpID09PSAndW5jaGFuZ2VkJykge1xuICAgICAgICAgIGxpbmVzLnB1c2gobGluZS5jb3B5KHtuZXdMaW5lTnVtYmVyOiBuZXdMaW5lTnVtYmVyKyt9KSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIGNvbnN0IG5ld1Jvd0NvdW50ID0gbmV3TGluZU51bWJlciAtIG5ld1N0YXJ0Um93O1xuICAgICAgaWYgKGh1bmtDb250YWluc1NlbGVjdGVkTGluZXMpIHtcbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgaHVua3MucHVzaChuZXcgSHVuayhodW5rLmdldE9sZFN0YXJ0Um93KCksIG5ld1N0YXJ0Um93LCBodW5rLmdldE9sZFJvd0NvdW50KCksIG5ld1Jvd0NvdW50LCBodW5rLmdldFNlY3Rpb25IZWFkaW5nKCksIGxpbmVzKSk7XG4gICAgICB9XG4gICAgICBkZWx0YSArPSBuZXdSb3dDb3VudCAtIGh1bmsuZ2V0TmV3Um93Q291bnQoKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEZpbGVQYXRjaChcbiAgICAgIHRoaXMuZ2V0T2xkUGF0aCgpLFxuICAgICAgdGhpcy5nZXROZXdQYXRoKCkgPyB0aGlzLmdldE5ld1BhdGgoKSA6IHRoaXMuZ2V0T2xkUGF0aCgpLFxuICAgICAgdGhpcy5nZXRTdGF0dXMoKSxcbiAgICAgIGh1bmtzLFxuICAgICk7XG4gIH1cblxuICBnZXRVbnN0YWdlUGF0Y2goKSB7XG4gICAgbGV0IGludmVydGVkU3RhdHVzO1xuICAgIHN3aXRjaCAodGhpcy5nZXRTdGF0dXMoKSkge1xuICAgICAgY2FzZSAnbW9kaWZpZWQnOlxuICAgICAgICBpbnZlcnRlZFN0YXR1cyA9ICdtb2RpZmllZCc7XG4gICAgICAgIGJyZWFrO1xuICAgICAgY2FzZSAnYWRkZWQnOlxuICAgICAgICBpbnZlcnRlZFN0YXR1cyA9ICdkZWxldGVkJztcbiAgICAgICAgYnJlYWs7XG4gICAgICBjYXNlICdkZWxldGVkJzpcbiAgICAgICAgaW52ZXJ0ZWRTdGF0dXMgPSAnYWRkZWQnO1xuICAgICAgICBicmVhaztcbiAgICAgIGRlZmF1bHQ6XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihgVW5rbm93biBTdGF0dXM6ICR7dGhpcy5nZXRTdGF0dXMoKX1gKTtcbiAgICB9XG4gICAgY29uc3QgaW52ZXJ0ZWRIdW5rcyA9IHRoaXMuZ2V0SHVua3MoKS5tYXAoaCA9PiBoLmludmVydCgpKTtcbiAgICByZXR1cm4gbmV3IEZpbGVQYXRjaChcbiAgICAgIHRoaXMuZ2V0TmV3UGF0aCgpLFxuICAgICAgdGhpcy5nZXRPbGRQYXRoKCksXG4gICAgICBpbnZlcnRlZFN0YXR1cyxcbiAgICAgIGludmVydGVkSHVua3MsXG4gICAgKTtcbiAgfVxuXG4gIGdldFVuc3RhZ2VQYXRjaEZvckh1bmsoaHVuaykge1xuICAgIHJldHVybiB0aGlzLmdldFVuc3RhZ2VQYXRjaEZvckxpbmVzKG5ldyBTZXQoaHVuay5nZXRMaW5lcygpKSk7XG4gIH1cblxuICBnZXRVbnN0YWdlUGF0Y2hGb3JMaW5lcyhzZWxlY3RlZExpbmVzKSB7XG4gICAgaWYgKHRoaXMuY2hhbmdlZExpbmVDb3VudCA9PT0gWy4uLnNlbGVjdGVkTGluZXNdLmZpbHRlcihsaW5lID0+IGxpbmUuaXNDaGFuZ2VkKCkpLmxlbmd0aCkge1xuICAgICAgcmV0dXJuIHRoaXMuZ2V0VW5zdGFnZVBhdGNoKCk7XG4gICAgfVxuXG4gICAgbGV0IGRlbHRhID0gMDtcbiAgICBjb25zdCBodW5rcyA9IFtdO1xuICAgIGZvciAoY29uc3QgaHVuayBvZiB0aGlzLmdldEh1bmtzKCkpIHtcbiAgICAgIGNvbnN0IG9sZFN0YXJ0Um93ID0gKGh1bmsuZ2V0T2xkU3RhcnRSb3coKSB8fCAxKSArIGRlbHRhO1xuICAgICAgbGV0IG9sZExpbmVOdW1iZXIgPSBvbGRTdGFydFJvdztcbiAgICAgIGNvbnN0IGxpbmVzID0gW107XG4gICAgICBsZXQgaHVua0NvbnRhaW5zU2VsZWN0ZWRMaW5lcyA9IGZhbHNlO1xuICAgICAgZm9yIChjb25zdCBsaW5lIG9mIGh1bmsuZ2V0TGluZXMoKSkge1xuICAgICAgICBpZiAobGluZS5nZXRTdGF0dXMoKSA9PT0gJ25vbmV3bGluZScpIHtcbiAgICAgICAgICBsaW5lcy5wdXNoKGxpbmUuY29weSh7b2xkTGluZU51bWJlcjogLTEsIG5ld0xpbmVOdW1iZXI6IC0xfSkpO1xuICAgICAgICB9IGVsc2UgaWYgKHNlbGVjdGVkTGluZXMuaGFzKGxpbmUpKSB7XG4gICAgICAgICAgaHVua0NvbnRhaW5zU2VsZWN0ZWRMaW5lcyA9IHRydWU7XG4gICAgICAgICAgaWYgKGxpbmUuZ2V0U3RhdHVzKCkgPT09ICdhZGRlZCcpIHtcbiAgICAgICAgICAgIGxpbmVzLnB1c2gobGluZS5jb3B5KCkpO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBsaW5lcy5wdXNoKGxpbmUuY29weSh7b2xkTGluZU51bWJlcjogb2xkTGluZU51bWJlcisrfSkpO1xuICAgICAgICAgIH1cbiAgICAgICAgfSBlbHNlIGlmIChsaW5lLmdldFN0YXR1cygpID09PSAnYWRkZWQnKSB7XG4gICAgICAgICAgbGluZXMucHVzaChsaW5lLmNvcHkoe29sZExpbmVOdW1iZXI6IG9sZExpbmVOdW1iZXIrKywgc3RhdHVzOiAndW5jaGFuZ2VkJ30pKTtcbiAgICAgICAgfSBlbHNlIGlmIChsaW5lLmdldFN0YXR1cygpID09PSAndW5jaGFuZ2VkJykge1xuICAgICAgICAgIGxpbmVzLnB1c2gobGluZS5jb3B5KHtvbGRMaW5lTnVtYmVyOiBvbGRMaW5lTnVtYmVyKyt9KSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIGNvbnN0IG9sZFJvd0NvdW50ID0gb2xkTGluZU51bWJlciAtIG9sZFN0YXJ0Um93O1xuICAgICAgaWYgKGh1bmtDb250YWluc1NlbGVjdGVkTGluZXMpIHtcbiAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgaHVua3MucHVzaChuZXcgSHVuayhvbGRTdGFydFJvdywgaHVuay5nZXROZXdTdGFydFJvdygpLCBvbGRSb3dDb3VudCwgaHVuay5nZXROZXdSb3dDb3VudCgpLCBodW5rLmdldFNlY3Rpb25IZWFkaW5nKCksIGxpbmVzKSk7XG4gICAgICB9XG4gICAgICBkZWx0YSArPSBvbGRSb3dDb3VudCAtIGh1bmsuZ2V0T2xkUm93Q291bnQoKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEZpbGVQYXRjaChcbiAgICAgIHRoaXMuZ2V0T2xkUGF0aCgpID8gdGhpcy5nZXRPbGRQYXRoKCkgOiB0aGlzLmdldE5ld1BhdGgoKSxcbiAgICAgIHRoaXMuZ2V0TmV3UGF0aCgpLFxuICAgICAgdGhpcy5nZXRTdGF0dXMoKSxcbiAgICAgIGh1bmtzLFxuICAgICkuZ2V0VW5zdGFnZVBhdGNoKCk7XG4gIH1cblxuICB0b1N0cmluZygpIHtcbiAgICByZXR1cm4gdGhpcy5nZXRIdW5rcygpLm1hcChoID0+IGgudG9TdHJpbmcoKSkuam9pbignJyk7XG4gIH1cblxuICBnZXRIZWFkZXJTdHJpbmcoKSB7XG4gICAgbGV0IGhlYWRlciA9IHRoaXMuZ2V0T2xkUGF0aCgpID8gYC0tLSBhLyR7dG9HaXRQYXRoU2VwKHRoaXMuZ2V0T2xkUGF0aCgpKX1gIDogJy0tLSAvZGV2L251bGwnO1xuICAgIGhlYWRlciArPSAnXFxuJztcbiAgICBoZWFkZXIgKz0gdGhpcy5nZXROZXdQYXRoKCkgPyBgKysrIGIvJHt0b0dpdFBhdGhTZXAodGhpcy5nZXROZXdQYXRoKCkpfWAgOiAnKysrIC9kZXYvbnVsbCc7XG4gICAgaGVhZGVyICs9ICdcXG4nO1xuICAgIHJldHVybiBoZWFkZXI7XG4gIH1cbn1cbiJdfQ==