'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _markerTools = require('./marker-tools');

let Side = class Side {
  constructor(editor, marker, blockMarker, source, position, banner, originalText) {
    this.editor = editor;
    this.marker = marker;
    this.blockMarker = blockMarker;
    this.source = source;
    this.position = position;
    this.banner = banner;
    this.originalText = originalText;
  }

  getMarker() {
    return this.marker;
  }

  getBannerMarker() {
    return this.banner.getMarker();
  }

  getSource() {
    return this.source;
  }

  getBlockMarker() {
    return this.blockMarker;
  }

  getBlockPosition() {
    return this.position.when({
      top: () => 'before',
      middle: () => 'before',
      bottom: () => 'after'
    });
  }

  getLineCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModified';
    } else {
      return this.source.getCSSClass();
    }
  }

  getBannerCSSClass() {
    if (this.isModified() || this.isBannerModified()) {
      return 'github-ConflictModifiedBanner';
    } else {
      return this.source.getBannerCSSClass();
    }
  }

  getBlockCSSClasses() {
    const cxs = ['github-ConflictBlock'];
    cxs.push(this.source.getBlockCSSClass());
    cxs.push(this.position.getBlockCSSClass());
    if (this.isModified() || this.isBannerModified()) {
      cxs.push('github-ConflictModifiedBlock');
    }
    return cxs.join(' ');
  }

  getPosition() {
    return this.position;
  }

  getRange() {
    const bannerRange = this.banner.getRange();
    const bodyRange = this.marker.getBufferRange();
    return bannerRange.union(bodyRange);
  }

  includesPoint(point) {
    return this.getRange().containsPoint(point);
  }

  getText() {
    return this.editor.getTextInBufferRange(this.getMarker().getBufferRange());
  }

  isBannerModified() {
    return this.banner.isModified();
  }

  isModified() {
    return this.getText() !== this.originalText;
  }

  isEmpty() {
    return this.marker.getBufferRange().isEmpty();
  }

  revertBanner() {
    this.banner.revert();
  }

  revert() {
    const range = this.getMarker().getBufferRange();
    this.editor.setTextInBufferRange(range, this.originalText);
  }

  deleteBanner() {
    this.banner.delete();
  }

  delete() {
    (0, _markerTools.deleteMarkerIn)(this.getMarker(), this.editor);
  }

  appendText(text) {
    const insertionPoint = this.getMarker().getBufferRange().end;
    return this.editor.setTextInBufferRange([insertionPoint, insertionPoint], text);
  }
};
exports.default = Side;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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