'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.GitError = undefined;

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

var _class, _temp;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _child_process = require('child_process');

var _child_process2 = _interopRequireDefault(_child_process);

var _electron = require('electron');

var _eventKit = require('event-kit');

var _dugite = require('dugite');

var _whatTheDiff = require('what-the-diff');

var _whatTheStatus = require('what-the-status');

var _gitPromptServer = require('./git-prompt-server');

var _gitPromptServer2 = _interopRequireDefault(_gitPromptServer);

var _asyncQueue = require('./async-queue');

var _asyncQueue2 = _interopRequireDefault(_asyncQueue);

var _helpers = require('./helpers');

var _gitTimingsView = require('./views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _workerManager = require('./worker-manager');

var _workerManager2 = _interopRequireDefault(_workerManager);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

const LINE_ENDING_REGEX = /\r?\n/;

const GPG_HELPER_PATH = _path2.default.resolve((0, _helpers.getPackageRoot)(), 'bin', 'gpg-no-tty.sh');
const ENV_VARS_TO_COPY = ['GIT_AUTHOR_NAME', 'GIT_AUTHOR_EMAIL', 'GIT_AUTHOR_DATE', 'GIT_COMMITTER_NAME', 'GIT_COMMITTER_EMAIL', 'GIT_COMMITTER_DATE', 'EMAIL'];

let headless = null;
let execPathPromise = null;

let GitError = exports.GitError = class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }
};
let GitShellOutStrategy = (_temp = _class = class GitShellOutStrategy {

  constructor(workingDir) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    this.workingDir = workingDir;
    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os2.default.cpus().length);
      this.commandQueue = new _asyncQueue2.default({ parallelism });
    }

    this.prompt = options.prompt || (query => Promise.reject());
    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }

  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */
  setPromptCallback(prompt) {
    this.prompt = prompt;
  }

  // Execute a command and read the output using the embedded Git environment
  exec(args) {
    var _this = this;

    var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : GitShellOutStrategy.defaultExecArgs;

    let stdin = _ref.stdin,
        useGitPromptServer = _ref.useGitPromptServer,
        writeOperation = _ref.writeOperation;
    return _asyncToGenerator(function* () {
      /* eslint-disable no-console */
      const subscriptions = new _eventKit.CompositeDisposable();
      const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');

      const formattedArgs = `git ${args.join(' ')} in ${_this.workingDir}`;
      const timingMarker = _gitTimingsView2.default.generateMarker(`git ${args.join(' ')}`);
      timingMarker.mark('queued');

      if (execPathPromise === null) {
        // Attempt to collect the --exec-path from a native git installation.
        execPathPromise = new Promise(function (resolve, reject) {
          _child_process2.default.exec('git --exec-path', function (error, stdout, stderr) {
            if (error) {
              // Oh well
              resolve(null);
              return;
            }

            resolve(stdout.trim());
          });
        });
      }
      const execPath = yield execPathPromise;

      return _this.commandQueue.push(_asyncToGenerator(function* () {
        timingMarker.mark('prepare');
        let gitPromptServer;

        const pathParts = [];
        if (process.env.PATH) {
          pathParts.push(process.env.PATH);
        }
        if (execPath) {
          pathParts.push(execPath);
        }

        const env = {
          GIT_TERMINAL_PROMPT: '0',
          PATH: pathParts.join(_path2.default.delimiter)
        };

        ENV_VARS_TO_COPY.forEach(function (envVar) {
          if (process.env[envVar] !== undefined) {
            env[envVar] = process.env[envVar];
          }
        });

        if (useGitPromptServer) {
          gitPromptServer = new _gitPromptServer2.default();

          var _ref3 = yield gitPromptServer.start(_this.prompt);

          const socket = _ref3.socket,
                electron = _ref3.electron,
                credentialHelper = _ref3.credentialHelper,
                askPass = _ref3.askPass,
                sshWrapper = _ref3.sshWrapper;


          env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(askPass.script);
          env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(credentialHelper.script);
          env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)(electron);
          env.ATOM_GITHUB_SOCK_PATH = (0, _helpers.normalizeGitHelperPath)(socket);

          env.ATOM_GITHUB_WORKDIR_PATH = _this.workingDir;
          env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();

          // "ssh" won't respect SSH_ASKPASS unless:
          // (a) it's running without a tty
          // (b) DISPLAY is set to something nonempty
          // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.
          if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
            env.DISPLAY = 'atom-github-placeholder';
          }

          env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
          env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
          env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
          env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
          env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';

          env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(askPass.launcher);
          env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(askPass.launcher);

          if (process.platform === 'linux') {
            env.GIT_SSH_COMMAND = sshWrapper.script;
          } else {
            env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
          }

          args.unshift('-c', `credential.helper=${(0, _helpers.normalizeGitHelperPath)(credentialHelper.launcher)}`);
        }

        if (diagnosticsEnabled) {
          env.GIT_TRACE = 'true';
          env.GIT_TRACE_CURL = 'true';
        }

        const options = { env };

        if (stdin) {
          options.stdin = stdin;
          options.stdinEncoding = 'utf8';
        }

        if (process.env.PRINT_GIT_TIMES) {
          console.time(`git:${formattedArgs}`);
        }
        return new Promise((() => {
          var _ref4 = _asyncToGenerator(function* (resolve, reject) {
            var _executeGitCommand = _this.executeGitCommand(args, options, timingMarker);

            const promise = _executeGitCommand.promise,
                  cancel = _executeGitCommand.cancel;

            let expectCancel = false;
            if (gitPromptServer) {
              subscriptions.add(gitPromptServer.onDidCancel((() => {
                var _ref5 = _asyncToGenerator(function* (_ref6) {
                  let handlerPid = _ref6.handlerPid;

                  expectCancel = true;
                  yield cancel();

                  // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
                  // process does not terminate when the git process is killed.
                  // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
                  // chance to fall back to GIT_ASKPASS from the credential handler.
                  require('tree-kill')(handlerPid);
                });

                return function (_x5) {
                  return _ref5.apply(this, arguments);
                };
              })()));
            }

            var _ref7 = yield promise;

            const stdout = _ref7.stdout,
                  stderr = _ref7.stderr,
                  exitCode = _ref7.exitCode,
                  timing = _ref7.timing;


            if (timing) {
              const execTime = timing.execTime,
                    spawnTime = timing.spawnTime,
                    ipcTime = timing.ipcTime;

              const now = performance.now();
              timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
              timingMarker.mark('execute', now - execTime - ipcTime);
              timingMarker.mark('ipc', now - ipcTime);
            }
            timingMarker.finalize();
            if (process.env.PRINT_GIT_TIMES) {
              console.timeEnd(`git:${formattedArgs}`);
            }
            if (gitPromptServer) {
              gitPromptServer.terminate();
            }
            subscriptions.dispose();

            if (diagnosticsEnabled) {
              if (headless) {
                let summary = `git:${formattedArgs}\n`;
                summary += `exit status: ${exitCode}\n`;
                summary += 'stdout:';
                if (stdout.length === 0) {
                  summary += ' <empty>\n';
                } else {
                  summary += `\n${stdout}\n`;
                }
                summary += 'stderr:';
                if (stderr.length === 0) {
                  summary += ' <empty>\n';
                } else {
                  summary += `\n${stderr}\n`;
                }

                console.log(summary);
              } else {
                const headerStyle = 'font-weight: bold; color: blue;';

                console.groupCollapsed(`git:${formattedArgs}`);
                console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
                console.log('%cstdout', headerStyle);
                console.log(stdout);
                console.log('%cstderr', headerStyle);
                console.log(stderr);
                console.groupEnd();
              }
            }

            if (exitCode !== 0 && !expectCancel) {
              const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
              err.code = exitCode;
              err.stdErr = stderr;
              err.stdOut = stdout;
              err.command = formattedArgs;
              reject(err);
            }
            resolve(stdout);
          });

          return function (_x3, _x4) {
            return _ref4.apply(this, arguments);
          };
        })());
      }), { parallel: !writeOperation });
      /* eslint-enable no-console */
    })();
  }

  executeGitCommand(args, options) {
    let marker = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;

    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager2.default.getInstance().isReady()) {
      marker && marker.mark('nexttick');

      let childPid;
      options.processCallback = child => {
        childPid = child.pid;

        child.on('error', err => {
          /* eslint-disable no-console */
          console.error(`Error spawning: git ${args.join(' ')} in ${this.workingDir}`);
          console.error(err);
          /* eslint-enable no-console */
        });

        child.stdin.on('error', err => {
          /* eslint-disable no-console */
          console.error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}`);
          console.error(err);
          /* eslint-enable no-console */
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);
      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => childPid && require('tree-kill')(childPid)
      };
    } else {
      const workerManager = this.workerManager || _workerManager2.default.getInstance();
      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  /**
   * Execute a git command that may create a commit. If the command fails because the GPG binary was invoked and unable
   * to acquire a passphrase (because the pinentry program attempted to use a tty), retry with a `GitPromptServer`.
   */
  gpgExec(args) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    const gpgArgs = ['-c', `gpg.program=${GPG_HELPER_PATH}`].concat(args);
    return this.exec(gpgArgs, options).catch(err => {
      if (err.code === 128 && /gpg failed/.test(err.stdErr) && !options.useGitPromptServer) {
        // Retry with a GitPromptServer
        options.useGitPromptServer = true;
        return this.exec(gpgArgs, options);
      } else {
        throw err;
      }
    });
  }

  resolveDotGitDir() {
    var _this2 = this;

    return _asyncToGenerator(function* () {
      try {
        yield (0, _helpers.fsStat)(_this2.workingDir); // fails if folder doesn't exist
        const output = yield _this2.exec(['rev-parse', '--resolve-git-dir', _path2.default.join(_this2.workingDir, '.git')]);
        const dotGitDir = output.trim();
        if (_path2.default.isAbsolute(dotGitDir)) {
          return (0, _helpers.toNativePathSep)(dotGitDir);
        } else {
          return (0, _helpers.toNativePathSep)(_path2.default.resolve(_path2.default.join(_this2.workingDir, dotGitDir)));
        }
      } catch (e) {
        return null;
      }
    })();
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }

  /**
   * Staging/Unstaging files and patches and committing
   */
  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  unstageFiles(paths) {
    let commit = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'HEAD';

    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  applyPatch(patch) {
    var _ref8 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let index = _ref8.index;

    const args = ['apply', '-'];
    if (index) {
      args.splice(1, 0, '--cached');
    }
    return this.exec(args, { stdin: patch, writeOperation: true });
  }

  commit(message) {
    var _ref9 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let allowEmpty = _ref9.allowEmpty,
        amend = _ref9.amend;

    const args = ['commit', '-m', message];
    if (amend) {
      args.push('--amend');
    }
    if (allowEmpty) {
      args.push('--allow-empty');
    }
    return this.gpgExec(args, { writeOperation: true });
  }

  /**
   * File Status and Diffs
   */
  getStatusBundle() {
    var _this3 = this;

    return _asyncToGenerator(function* () {
      const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
      const output = yield _this3.exec(args);
      const results = yield (0, _whatTheStatus.parse)(output);

      for (const entryType in results) {
        if (Array.isArray(results[entryType])) {
          _this3.updateNativePathSepForEntries(results[entryType]);
        }
      }

      return results;
    })();
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }
      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  diffFileStatus() {
    var _this4 = this;

    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    return _asyncToGenerator(function* () {
      const args = ['diff', '--name-status', '--no-renames'];
      if (options.staged) {
        args.push('--staged');
      }
      if (options.target) {
        args.push(options.target);
      }
      const output = yield _this4.exec(args);

      const statusMap = {
        A: 'added',
        M: 'modified',
        D: 'deleted',
        U: 'unmerged'
      };

      const fileStatuses = {};
      output && output.trim().split(LINE_ENDING_REGEX).forEach(function (line) {
        var _line$split = line.split('\t'),
            _line$split2 = _slicedToArray(_line$split, 2);

        const status = _line$split2[0],
              rawFilePath = _line$split2[1];

        const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
        fileStatuses[filePath] = statusMap[status];
      });
      if (!options.staged) {
        const untracked = yield _this4.getUntrackedFiles();
        untracked.forEach(function (filePath) {
          fileStatuses[filePath] = 'added';
        });
      }
      return fileStatuses;
    })();
  }

  getUntrackedFiles() {
    var _this5 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this5.exec(['ls-files', '--others', '--exclude-standard']);
      if (output.trim() === '') {
        return [];
      }
      return output.trim().split(LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
    })();
  }

  getDiffForFilePath(filePath) {
    var _this6 = this;

    var _ref10 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let staged = _ref10.staged,
        baseCommit = _ref10.baseCommit;
    return _asyncToGenerator(function* () {
      let args = ['diff', '--no-prefix', '--no-renames', '--diff-filter=u'];
      if (staged) {
        args.push('--staged');
      }
      if (baseCommit) {
        args.push(baseCommit);
      }
      args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
      const output = yield _this6.exec(args);

      let rawDiffs = [];
      if (output) {
        rawDiffs = (0, _whatTheDiff.parse)(output).filter(function (rawDiff) {
          return rawDiff.status !== 'unmerged';
        });

        for (let i = 0; i < rawDiffs.length; i++) {
          const rawDiff = rawDiffs[i];
          if (rawDiff.oldPath) {
            rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
          }
          if (rawDiff.newPath) {
            rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
          }
        }
      }

      if (!staged && (yield _this6.getUntrackedFiles()).includes(filePath)) {
        // add untracked file
        const absPath = _path2.default.join(_this6.workingDir, filePath);
        const executable = yield (0, _helpers.isFileExecutable)(absPath);
        const contents = yield (0, _helpers.readFile)(absPath);
        const binary = (0, _helpers.isBinary)(contents);
        rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, executable));
      }
      if (rawDiffs.length > 1) {
        throw new Error(`Expected 0 or 1 diffs for ${filePath} but got ${rawDiffs.length}`);
      }
      return rawDiffs[0];
    })();
  }

  /**
   * Miscellaneous getters
   */
  getCommit(ref) {
    var _this7 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this7.exec(['log', '--pretty=%H%x00%B%x00', '--no-abbrev-commit', '-1', ref]);

      var _output$split = output.split('\0'),
          _output$split2 = _slicedToArray(_output$split, 2);

      const sha = _output$split2[0],
            message = _output$split2[1];

      return { sha, message: message.trim(), unbornRef: false };
    })();
  }

  getHeadCommit() {
    var _this8 = this;

    return _asyncToGenerator(function* () {
      try {
        const commit = yield _this8.getCommit('HEAD');
        commit.unbornRef = false;
        return commit;
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          return { sha: '', message: '', unbornRef: true };
        } else {
          throw e;
        }
      }
    })();
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }

  /**
   * Merge
   */
  merge(branchName) {
    return this.gpgExec(['merge', branchName], { writeOperation: true });
  }

  isMerging(dotGitDir) {
    return _asyncToGenerator(function* () {
      try {
        yield (0, _helpers.readFile)(_path2.default.join(dotGitDir, 'MERGE_HEAD'));
        return true;
      } catch (e) {
        return false;
      }
    })();
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], { writeOperation: true });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }

  /**
   * Rebase
   */
  isRebasing(dotGitDir) {
    return _asyncToGenerator(function* () {
      const results = yield Promise.all([(0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-apply'))]);
      return results.some(function (r) {
        return r;
      });
    })();
  }

  /**
   * Remote interactions
   */
  clone(remoteUrl) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    const args = ['clone'];
    if (options.noLocal) {
      args.push('--no-local');
    }
    if (options.bare) {
      args.push('--bare');
    }
    if (options.recursive) {
      args.push('--recursive');
    }
    args.push(remoteUrl, this.workingDir);

    return this.exec(args, { writeOperation: true });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], { useGitPromptServer: true, writeOperation: true });
  }

  pull(remoteName, branchName) {
    return this.gpgExec(['pull', remoteName, branchName], { useGitPromptServer: true, writeOperation: true });
  }

  push(remoteName, branchName) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

    const args = ['push', remoteName || 'origin', branchName];
    if (options.setUpstream) {
      args.push('--set-upstream');
    }
    if (options.force) {
      args.push('--force');
    }
    return this.exec(args, { useGitPromptServer: true, writeOperation: true });
  }

  /**
   * Branches
   */
  checkout(branchName) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    const args = ['checkout'];
    if (options.createNew) {
      args.push('-b');
    }
    return this.exec(args.concat(branchName), { writeOperation: true });
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }
    const args = ['checkout'];
    if (revision) {
      args.push(revision);
    }
    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), { writeOperation: true });
  }

  getBranches() {
    var _this9 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this9.exec(['for-each-ref', '--format=%(refname:short)', 'refs/heads/**']);
      return output.trim().split(LINE_ENDING_REGEX);
    })();
  }

  describeHead() {
    var _this10 = this;

    return _asyncToGenerator(function* () {
      return (yield _this10.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
    })();
  }

  getConfig(option) {
    var _this11 = this;

    var _ref11 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let local = _ref11.local;
    return _asyncToGenerator(function* () {
      let output;
      try {
        let args = ['config'];
        if (local) {
          args.push('--local');
        }
        args = args.concat(option);
        output = yield _this11.exec(args);
      } catch (err) {
        if (err.code === 1) {
          // No matching config found
          return null;
        } else {
          throw err;
        }
      }

      return output.trim();
    })();
  }

  setConfig(option, value) {
    var _ref12 = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

    let replaceAll = _ref12.replaceAll;

    let args = ['config'];
    if (replaceAll) {
      args.push('--replace-all');
    }
    args = args.concat(option, value);
    return this.exec(args, { writeOperation: true });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], { writeOperation: true });
  }

  getRemotes() {
    var _this12 = this;

    return _asyncToGenerator(function* () {
      let output = yield _this12.getConfig(['--get-regexp', '^remote\\..*\\.url$'], { local: true });
      if (output) {
        output = output.trim();
        if (!output.length) {
          return [];
        }
        return output.split('\n').map(function (line) {
          const match = line.match(/^remote\.(.*)\.url (.*)$/);
          return {
            name: match[1],
            url: match[2]
          };
        });
      } else {
        return [];
      }
    })();
  }

  createBlob() {
    var _this13 = this;

    var _ref13 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    let filePath = _ref13.filePath,
        stdin = _ref13.stdin;
    return _asyncToGenerator(function* () {
      let output;
      if (filePath) {
        try {
          output = (yield _this13.exec(['hash-object', '-w', filePath], { writeOperation: true })).trim();
        } catch (e) {
          if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
            output = null;
          } else {
            throw e;
          }
        }
      } else if (stdin) {
        output = (yield _this13.exec(['hash-object', '-w', '--stdin'], { stdin, writeOperation: true })).trim();
      } else {
        throw new Error('Must supply file path or stdin');
      }
      return output;
    })();
  }

  expandBlobToFile(absFilePath, sha) {
    var _this14 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this14.exec(['cat-file', '-p', sha]);
      yield (0, _helpers.writeFile)(absFilePath, output);
      return absFilePath;
    })();
  }

  getBlobContents(sha) {
    var _this15 = this;

    return _asyncToGenerator(function* () {
      return yield _this15.exec(['cat-file', '-p', sha]);
    })();
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    var _this16 = this;

    return _asyncToGenerator(function* () {
      const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
      let output;
      let conflict = false;
      try {
        output = yield _this16.exec(args);
      } catch (e) {
        if (e instanceof GitError && e.code === 1) {
          output = e.stdOut;
          conflict = true;
        } else {
          throw e;
        }
      }

      // Interpret a relative resultPath as relative to the repository working directory for consistency with the
      // other arguments.
      const resolvedResultPath = _path2.default.resolve(_this16.workingDir, resultPath);
      yield (0, _helpers.writeFile)(resolvedResultPath, output);

      return { filePath: oursPath, resultPath, conflict };
    })();
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    var _this17 = this;

    return _asyncToGenerator(function* () {
      const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
      const fileMode = yield _this17.getFileMode(filePath);
      let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;
      if (commonBaseSha) {
        indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
      }
      if (oursSha) {
        indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
      }
      if (theirsSha) {
        indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
      }
      return _this17.exec(['update-index', '--index-info'], { stdin: indexInfo, writeOperation: true });
    })();
  }

  getFileMode(filePath) {
    var _this18 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this18.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);
      if (output) {
        return output.slice(0, 6);
      } else {
        const executable = yield (0, _helpers.isFileExecutable)(_path2.default.join(_this18.workingDir, filePath));
        return executable ? '100755' : '100644';
      }
    })();
  }

  destroy() {
    this.commandQueue.dispose();
  }
}, _class.defaultExecArgs = { stdin: null, useGitPromptServer: false, writeOperation: false }, _temp);
exports.default = GitShellOutStrategy;


function buildAddedFilePatch(filePath, contents, executable) {
  const hunks = [];
  if (contents) {
    const noNewLine = contents[contents.length - 1] !== '\n';
    const lines = contents.trim().split(LINE_ENDING_REGEX).map(line => `+${line}`);
    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }
    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }
  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: executable ? '100755' : '100644',
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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