(function() {
  var BufferedProcess, CSON, CompileCache, CompositeDisposable, Emitter, ModuleCache, Package, ScopedProperties, _, async, fs, path, ref,
    slice = [].slice;

  path = require('path');

  _ = require('underscore-plus');

  async = require('async');

  CSON = require('season');

  fs = require('fs-plus');

  ref = require('event-kit'), Emitter = ref.Emitter, CompositeDisposable = ref.CompositeDisposable;

  CompileCache = require('./compile-cache');

  ModuleCache = require('./module-cache');

  ScopedProperties = require('./scoped-properties');

  BufferedProcess = require('./buffered-process');

  module.exports = Package = (function() {
    Package.prototype.keymaps = null;

    Package.prototype.menus = null;

    Package.prototype.stylesheets = null;

    Package.prototype.stylesheetDisposables = null;

    Package.prototype.grammars = null;

    Package.prototype.settings = null;

    Package.prototype.mainModulePath = null;

    Package.prototype.resolvedMainModulePath = false;

    Package.prototype.mainModule = null;

    Package.prototype.mainInitialized = false;

    Package.prototype.mainActivated = false;


    /*
    Section: Construction
     */

    function Package(params) {
      var ref1, ref2, ref3;
      this.path = params.path, this.metadata = params.metadata, this.bundledPackage = params.bundledPackage, this.preloadedPackage = params.preloadedPackage, this.packageManager = params.packageManager, this.config = params.config, this.styleManager = params.styleManager, this.commandRegistry = params.commandRegistry, this.keymapManager = params.keymapManager, this.notificationManager = params.notificationManager, this.grammarRegistry = params.grammarRegistry, this.themeManager = params.themeManager, this.menuManager = params.menuManager, this.contextMenuManager = params.contextMenuManager, this.deserializerManager = params.deserializerManager, this.viewRegistry = params.viewRegistry;
      this.emitter = new Emitter;
      if (this.metadata == null) {
        this.metadata = this.packageManager.loadPackageMetadata(this.path);
      }
      if (this.bundledPackage == null) {
        this.bundledPackage = this.packageManager.isBundledPackagePath(this.path);
      }
      this.name = (ref1 = (ref2 = (ref3 = this.metadata) != null ? ref3.name : void 0) != null ? ref2 : params.name) != null ? ref1 : path.basename(this.path);
      this.reset();
    }


    /*
    Section: Event Subscription
     */

    Package.prototype.onDidDeactivate = function(callback) {
      return this.emitter.on('did-deactivate', callback);
    };


    /*
    Section: Instance Methods
     */

    Package.prototype.enable = function() {
      return this.config.removeAtKeyPath('core.disabledPackages', this.name);
    };

    Package.prototype.disable = function() {
      return this.config.pushAtKeyPath('core.disabledPackages', this.name);
    };

    Package.prototype.isTheme = function() {
      var ref1;
      return ((ref1 = this.metadata) != null ? ref1.theme : void 0) != null;
    };

    Package.prototype.measure = function(key, fn) {
      var startTime, value;
      startTime = Date.now();
      value = fn();
      this[key] = Date.now() - startTime;
      return value;
    };

    Package.prototype.getType = function() {
      return 'atom';
    };

    Package.prototype.getStyleSheetPriority = function() {
      return 0;
    };

    Package.prototype.preload = function() {
      var i, len, ref1, settings;
      this.loadKeymaps();
      this.loadMenus();
      this.registerDeserializerMethods();
      this.activateCoreStartupServices();
      this.configSchemaRegisteredOnLoad = this.registerConfigSchemaFromMetadata();
      this.requireMainModule();
      this.settingsPromise = this.loadSettings();
      this.activationDisposables = new CompositeDisposable;
      this.activateKeymaps();
      this.activateMenus();
      ref1 = this.settings;
      for (i = 0, len = ref1.length; i < len; i++) {
        settings = ref1[i];
        settings.activate();
      }
      return this.settingsActivated = true;
    };

    Package.prototype.finishLoading = function() {
      return this.measure('loadTime', (function(_this) {
        return function() {
          _this.path = path.join(_this.packageManager.resourcePath, _this.path);
          ModuleCache.add(_this.path, _this.metadata);
          _this.loadStylesheets();
          return _this.getMainModulePath();
        };
      })(this));
    };

    Package.prototype.load = function() {
      this.measure('loadTime', (function(_this) {
        return function() {
          var error;
          try {
            ModuleCache.add(_this.path, _this.metadata);
            _this.loadKeymaps();
            _this.loadMenus();
            _this.loadStylesheets();
            _this.registerDeserializerMethods();
            _this.activateCoreStartupServices();
            _this.registerTranspilerConfig();
            _this.configSchemaRegisteredOnLoad = _this.registerConfigSchemaFromMetadata();
            _this.settingsPromise = _this.loadSettings();
            if (_this.shouldRequireMainModuleOnLoad() && (_this.mainModule == null)) {
              return _this.requireMainModule();
            }
          } catch (error1) {
            error = error1;
            return _this.handleError("Failed to load the " + _this.name + " package", error);
          }
        };
      })(this));
      return this;
    };

    Package.prototype.unload = function() {
      return this.unregisterTranspilerConfig();
    };

    Package.prototype.shouldRequireMainModuleOnLoad = function() {
      return !((this.metadata.deserializers != null) || (this.metadata.viewProviders != null) || (this.metadata.configSchema != null) || this.activationShouldBeDeferred() || localStorage.getItem(this.getCanDeferMainModuleRequireStorageKey()) === 'true');
    };

    Package.prototype.reset = function() {
      this.stylesheets = [];
      this.keymaps = [];
      this.menus = [];
      this.grammars = [];
      this.settings = [];
      this.mainInitialized = false;
      return this.mainActivated = false;
    };

    Package.prototype.initializeIfNeeded = function() {
      if (this.mainInitialized) {
        return;
      }
      this.measure('initializeTime', (function(_this) {
        return function() {
          var base, error, ref1;
          try {
            if (_this.mainModule == null) {
              _this.requireMainModule();
            }
            if (typeof (base = _this.mainModule).initialize === "function") {
              base.initialize((ref1 = _this.packageManager.getPackageState(_this.name)) != null ? ref1 : {});
            }
            return _this.mainInitialized = true;
          } catch (error1) {
            error = error1;
            return _this.handleError("Failed to initialize the " + _this.name + " package", error);
          }
        };
      })(this));
    };

    Package.prototype.activate = function() {
      if (this.grammarsPromise == null) {
        this.grammarsPromise = this.loadGrammars();
      }
      if (this.activationPromise == null) {
        this.activationPromise = new Promise((function(_this) {
          return function(resolve, reject) {
            _this.resolveActivationPromise = resolve;
            return _this.measure('activateTime', function() {
              var error;
              try {
                _this.activateResources();
                if (_this.activationShouldBeDeferred()) {
                  return _this.subscribeToDeferredActivation();
                } else {
                  return _this.activateNow();
                }
              } catch (error1) {
                error = error1;
                return _this.handleError("Failed to activate the " + _this.name + " package", error);
              }
            });
          };
        })(this));
      }
      return Promise.all([this.grammarsPromise, this.settingsPromise, this.activationPromise]);
    };

    Package.prototype.activateNow = function() {
      var base, base1, error, ref1, ref2, ref3;
      try {
        if (this.mainModule == null) {
          this.requireMainModule();
        }
        this.configSchemaRegisteredOnActivate = this.registerConfigSchemaFromMainModule();
        this.registerViewProviders();
        this.activateStylesheets();
        if ((this.mainModule != null) && !this.mainActivated) {
          this.initializeIfNeeded();
          if (typeof (base = this.mainModule).activateConfig === "function") {
            base.activateConfig();
          }
          if (typeof (base1 = this.mainModule).activate === "function") {
            base1.activate((ref1 = this.packageManager.getPackageState(this.name)) != null ? ref1 : {});
          }
          this.mainActivated = true;
          this.activateServices();
        }
        if ((ref2 = this.activationCommandSubscriptions) != null) {
          ref2.dispose();
        }
        if ((ref3 = this.activationHookSubscriptions) != null) {
          ref3.dispose();
        }
      } catch (error1) {
        error = error1;
        this.handleError("Failed to activate the " + this.name + " package", error);
      }
      return typeof this.resolveActivationPromise === "function" ? this.resolveActivationPromise() : void 0;
    };

    Package.prototype.registerConfigSchemaFromMetadata = function() {
      var configSchema;
      if (configSchema = this.metadata.configSchema) {
        this.config.setSchema(this.name, {
          type: 'object',
          properties: configSchema
        });
        return true;
      } else {
        return false;
      }
    };

    Package.prototype.registerConfigSchemaFromMainModule = function() {
      if ((this.mainModule != null) && !this.configSchemaRegisteredOnLoad) {
        if ((this.mainModule.config != null) && typeof this.mainModule.config === 'object') {
          this.config.setSchema(this.name, {
            type: 'object',
            properties: this.mainModule.config
          });
          return true;
        }
      }
      return false;
    };

    Package.prototype.activateConfig = function() {
      if (this.configSchemaRegisteredOnLoad) {
        return;
      }
      this.requireMainModule();
      return this.registerConfigSchemaFromMainModule();
    };

    Package.prototype.activateStylesheets = function() {
      var context, i, len, match, priority, ref1, ref2, source, sourcePath;
      if (this.stylesheetsActivated) {
        return;
      }
      this.stylesheetDisposables = new CompositeDisposable;
      priority = this.getStyleSheetPriority();
      ref1 = this.stylesheets;
      for (i = 0, len = ref1.length; i < len; i++) {
        ref2 = ref1[i], sourcePath = ref2[0], source = ref2[1];
        if (match = path.basename(sourcePath).match(/[^.]*\.([^.]*)\./)) {
          context = match[1];
        } else if (this.metadata.theme === 'syntax') {
          context = 'atom-text-editor';
        } else {
          context = void 0;
        }
        this.stylesheetDisposables.add(this.styleManager.addStyleSheet(source, {
          sourcePath: sourcePath,
          priority: priority,
          context: context,
          skipDeprecatedSelectorsTransformation: this.bundledPackage
        }));
      }
      return this.stylesheetsActivated = true;
    };

    Package.prototype.activateResources = function() {
      var grammar, i, j, keymapIsDisabled, len, len1, ref1, ref2, ref3, settings;
      if (this.activationDisposables == null) {
        this.activationDisposables = new CompositeDisposable;
      }
      keymapIsDisabled = _.include((ref1 = this.config.get("core.packagesWithKeymapsDisabled")) != null ? ref1 : [], this.name);
      if (keymapIsDisabled) {
        this.deactivateKeymaps();
      } else if (!this.keymapActivated) {
        this.activateKeymaps();
      }
      if (!this.menusActivated) {
        this.activateMenus();
      }
      if (!this.grammarsActivated) {
        ref2 = this.grammars;
        for (i = 0, len = ref2.length; i < len; i++) {
          grammar = ref2[i];
          grammar.activate();
        }
        this.grammarsActivated = true;
      }
      if (!this.settingsActivated) {
        ref3 = this.settings;
        for (j = 0, len1 = ref3.length; j < len1; j++) {
          settings = ref3[j];
          settings.activate();
        }
        return this.settingsActivated = true;
      }
    };

    Package.prototype.activateKeymaps = function() {
      var i, keymapPath, len, map, ref1, ref2, validateSelectors;
      if (this.keymapActivated) {
        return;
      }
      this.keymapDisposables = new CompositeDisposable();
      validateSelectors = !this.preloadedPackage;
      ref1 = this.keymaps;
      for (i = 0, len = ref1.length; i < len; i++) {
        ref2 = ref1[i], keymapPath = ref2[0], map = ref2[1];
        this.keymapDisposables.add(this.keymapManager.add(keymapPath, map, 0, validateSelectors));
      }
      this.menuManager.update();
      return this.keymapActivated = true;
    };

    Package.prototype.deactivateKeymaps = function() {
      var ref1;
      if (!this.keymapActivated) {
        return;
      }
      if ((ref1 = this.keymapDisposables) != null) {
        ref1.dispose();
      }
      this.menuManager.update();
      return this.keymapActivated = false;
    };

    Package.prototype.hasKeymaps = function() {
      var i, len, map, ref1, ref2;
      ref1 = this.keymaps;
      for (i = 0, len = ref1.length; i < len; i++) {
        ref2 = ref1[i], path = ref2[0], map = ref2[1];
        if (map.length > 0) {
          return true;
        }
      }
      return false;
    };

    Package.prototype.activateMenus = function() {
      var error, i, itemsBySelector, j, len, len1, map, menuPath, ref1, ref2, ref3, ref4, validateSelectors;
      validateSelectors = !this.preloadedPackage;
      ref1 = this.menus;
      for (i = 0, len = ref1.length; i < len; i++) {
        ref2 = ref1[i], menuPath = ref2[0], map = ref2[1];
        if (map['context-menu'] != null) {
          try {
            itemsBySelector = map['context-menu'];
            this.activationDisposables.add(this.contextMenuManager.add(itemsBySelector, validateSelectors));
          } catch (error1) {
            error = error1;
            if (error.code === 'EBADSELECTOR') {
              error.message += " in " + menuPath;
              error.stack += "\n  at " + menuPath + ":1:1";
            }
            throw error;
          }
        }
      }
      ref3 = this.menus;
      for (j = 0, len1 = ref3.length; j < len1; j++) {
        ref4 = ref3[j], menuPath = ref4[0], map = ref4[1];
        if (map['menu'] != null) {
          this.activationDisposables.add(this.menuManager.add(map['menu']));
        }
      }
      return this.menusActivated = true;
    };

    Package.prototype.activateServices = function() {
      var methodName, name, ref1, ref2, servicesByVersion, version, versions;
      ref1 = this.metadata.providedServices;
      for (name in ref1) {
        versions = ref1[name].versions;
        servicesByVersion = {};
        for (version in versions) {
          methodName = versions[version];
          if (typeof this.mainModule[methodName] === 'function') {
            servicesByVersion[version] = this.mainModule[methodName]();
          }
        }
        this.activationDisposables.add(this.packageManager.serviceHub.provide(name, servicesByVersion));
      }
      ref2 = this.metadata.consumedServices;
      for (name in ref2) {
        versions = ref2[name].versions;
        for (version in versions) {
          methodName = versions[version];
          if (typeof this.mainModule[methodName] === 'function') {
            this.activationDisposables.add(this.packageManager.serviceHub.consume(name, version, this.mainModule[methodName].bind(this.mainModule)));
          }
        }
      }
    };

    Package.prototype.registerTranspilerConfig = function() {
      if (this.metadata.atomTranspilers) {
        return CompileCache.addTranspilerConfigForPath(this.path, this.name, this.metadata, this.metadata.atomTranspilers);
      }
    };

    Package.prototype.unregisterTranspilerConfig = function() {
      if (this.metadata.atomTranspilers) {
        return CompileCache.removeTranspilerConfigForPath(this.path);
      }
    };

    Package.prototype.loadKeymaps = function() {
      var keymapObject, keymapPath;
      if (this.bundledPackage && (this.packageManager.packagesCache[this.name] != null)) {
        this.keymaps = (function() {
          var ref1, results;
          ref1 = this.packageManager.packagesCache[this.name].keymaps;
          results = [];
          for (keymapPath in ref1) {
            keymapObject = ref1[keymapPath];
            results.push(["core:" + keymapPath, keymapObject]);
          }
          return results;
        }).call(this);
      } else {
        this.keymaps = this.getKeymapPaths().map(function(keymapPath) {
          var ref1;
          return [
            keymapPath, (ref1 = CSON.readFileSync(keymapPath, {
              allowDuplicateKeys: false
            })) != null ? ref1 : {}
          ];
        });
      }
    };

    Package.prototype.loadMenus = function() {
      var menuObject, menuPath;
      if (this.bundledPackage && (this.packageManager.packagesCache[this.name] != null)) {
        this.menus = (function() {
          var ref1, results;
          ref1 = this.packageManager.packagesCache[this.name].menus;
          results = [];
          for (menuPath in ref1) {
            menuObject = ref1[menuPath];
            results.push(["core:" + menuPath, menuObject]);
          }
          return results;
        }).call(this);
      } else {
        this.menus = this.getMenuPaths().map(function(menuPath) {
          var ref1;
          return [menuPath, (ref1 = CSON.readFileSync(menuPath)) != null ? ref1 : {}];
        });
      }
    };

    Package.prototype.getKeymapPaths = function() {
      var keymapsDirPath;
      keymapsDirPath = path.join(this.path, 'keymaps');
      if (this.metadata.keymaps) {
        return this.metadata.keymaps.map(function(name) {
          return fs.resolve(keymapsDirPath, name, ['json', 'cson', '']);
        });
      } else {
        return fs.listSync(keymapsDirPath, ['cson', 'json']);
      }
    };

    Package.prototype.getMenuPaths = function() {
      var menusDirPath;
      menusDirPath = path.join(this.path, 'menus');
      if (this.metadata.menus) {
        return this.metadata.menus.map(function(name) {
          return fs.resolve(menusDirPath, name, ['json', 'cson', '']);
        });
      } else {
        return fs.listSync(menusDirPath, ['cson', 'json']);
      }
    };

    Package.prototype.loadStylesheets = function() {
      return this.stylesheets = this.getStylesheetPaths().map((function(_this) {
        return function(stylesheetPath) {
          return [stylesheetPath, _this.themeManager.loadStylesheet(stylesheetPath, true)];
        };
      })(this));
    };

    Package.prototype.registerDeserializerMethods = function() {
      if (this.metadata.deserializers != null) {
        Object.keys(this.metadata.deserializers).forEach((function(_this) {
          return function(deserializerName) {
            var methodName;
            methodName = _this.metadata.deserializers[deserializerName];
            return _this.deserializerManager.add({
              name: deserializerName,
              deserialize: function(state, atomEnvironment) {
                _this.registerViewProviders();
                _this.requireMainModule();
                _this.initializeIfNeeded();
                return _this.mainModule[methodName](state, atomEnvironment);
              }
            });
          };
        })(this));
      }
    };

    Package.prototype.activateCoreStartupServices = function() {
      var directoryProviderService, methodName, ref1, ref2, servicesByVersion, version;
      if (directoryProviderService = (ref1 = this.metadata.providedServices) != null ? ref1['atom.directory-provider'] : void 0) {
        this.requireMainModule();
        servicesByVersion = {};
        ref2 = directoryProviderService.versions;
        for (version in ref2) {
          methodName = ref2[version];
          if (typeof this.mainModule[methodName] === 'function') {
            servicesByVersion[version] = this.mainModule[methodName]();
          }
        }
        return this.packageManager.serviceHub.provide('atom.directory-provider', servicesByVersion);
      }
    };

    Package.prototype.registerViewProviders = function() {
      if ((this.metadata.viewProviders != null) && !this.registeredViewProviders) {
        this.requireMainModule();
        this.metadata.viewProviders.forEach((function(_this) {
          return function(methodName) {
            return _this.viewRegistry.addViewProvider(function(model) {
              _this.initializeIfNeeded();
              return _this.mainModule[methodName](model);
            });
          };
        })(this));
        return this.registeredViewProviders = true;
      }
    };

    Package.prototype.getStylesheetsPath = function() {
      return path.join(this.path, 'styles');
    };

    Package.prototype.getStylesheetPaths = function() {
      var indexStylesheet, ref1, styleSheetPaths, stylesheetDirPath;
      if (this.bundledPackage && (((ref1 = this.packageManager.packagesCache[this.name]) != null ? ref1.styleSheetPaths : void 0) != null)) {
        styleSheetPaths = this.packageManager.packagesCache[this.name].styleSheetPaths;
        return styleSheetPaths.map((function(_this) {
          return function(styleSheetPath) {
            return path.join(_this.path, styleSheetPath);
          };
        })(this));
      } else {
        stylesheetDirPath = this.getStylesheetsPath();
        if (this.metadata.mainStyleSheet) {
          return [fs.resolve(this.path, this.metadata.mainStyleSheet)];
        } else if (this.metadata.styleSheets) {
          return this.metadata.styleSheets.map(function(name) {
            return fs.resolve(stylesheetDirPath, name, ['css', 'less', '']);
          });
        } else if (indexStylesheet = fs.resolve(this.path, 'index', ['css', 'less'])) {
          return [indexStylesheet];
        } else {
          return fs.listSync(stylesheetDirPath, ['css', 'less']);
        }
      }
    };

    Package.prototype.loadGrammarsSync = function() {
      var error, grammar, grammarPath, grammarPaths, i, len, ref1;
      if (this.grammarsLoaded) {
        return;
      }
      if (this.preloadedPackage && (this.packageManager.packagesCache[this.name] != null)) {
        grammarPaths = this.packageManager.packagesCache[this.name].grammarPaths;
      } else {
        grammarPaths = fs.listSync(path.join(this.path, 'grammars'), ['json', 'cson']);
      }
      for (i = 0, len = grammarPaths.length; i < len; i++) {
        grammarPath = grammarPaths[i];
        if (this.preloadedPackage && (this.packageManager.packagesCache[this.name] != null)) {
          grammarPath = path.resolve(this.packageManager.resourcePath, grammarPath);
        }
        try {
          grammar = this.grammarRegistry.readGrammarSync(grammarPath);
          grammar.packageName = this.name;
          grammar.bundledPackage = this.bundledPackage;
          this.grammars.push(grammar);
          grammar.activate();
        } catch (error1) {
          error = error1;
          console.warn("Failed to load grammar: " + grammarPath, (ref1 = error.stack) != null ? ref1 : error);
        }
      }
      this.grammarsLoaded = true;
      return this.grammarsActivated = true;
    };

    Package.prototype.loadGrammars = function() {
      var loadGrammar;
      if (this.grammarsLoaded) {
        return Promise.resolve();
      }
      loadGrammar = (function(_this) {
        return function(grammarPath, callback) {
          if (_this.preloadedPackage) {
            grammarPath = path.resolve(_this.packageManager.resourcePath, grammarPath);
          }
          return _this.grammarRegistry.readGrammar(grammarPath, function(error, grammar) {
            var detail, stack;
            if (error != null) {
              detail = error.message + " in " + grammarPath;
              stack = error.stack + "\n  at " + grammarPath + ":1:1";
              _this.notificationManager.addFatalError("Failed to load a " + _this.name + " package grammar", {
                stack: stack,
                detail: detail,
                packageName: _this.name,
                dismissable: true
              });
            } else {
              grammar.packageName = _this.name;
              grammar.bundledPackage = _this.bundledPackage;
              _this.grammars.push(grammar);
              if (_this.grammarsActivated) {
                grammar.activate();
              }
            }
            return callback();
          });
        };
      })(this);
      return new Promise((function(_this) {
        return function(resolve) {
          var grammarPaths, grammarsDirPath;
          if (_this.preloadedPackage && (_this.packageManager.packagesCache[_this.name] != null)) {
            grammarPaths = _this.packageManager.packagesCache[_this.name].grammarPaths;
            return async.each(grammarPaths, loadGrammar, function() {
              return resolve();
            });
          } else {
            grammarsDirPath = path.join(_this.path, 'grammars');
            return fs.exists(grammarsDirPath, function(grammarsDirExists) {
              if (!grammarsDirExists) {
                return resolve();
              }
              return fs.list(grammarsDirPath, ['json', 'cson'], function(error, grammarPaths) {
                if (grammarPaths == null) {
                  grammarPaths = [];
                }
                return async.each(grammarPaths, loadGrammar, function() {
                  return resolve();
                });
              });
            });
          }
        };
      })(this));
    };

    Package.prototype.loadSettings = function() {
      var loadSettingsFile;
      this.settings = [];
      loadSettingsFile = (function(_this) {
        return function(settingsPath, callback) {
          return ScopedProperties.load(settingsPath, _this.config, function(error, settings) {
            var detail, stack;
            if (error != null) {
              detail = error.message + " in " + settingsPath;
              stack = error.stack + "\n  at " + settingsPath + ":1:1";
              _this.notificationManager.addFatalError("Failed to load the " + _this.name + " package settings", {
                stack: stack,
                detail: detail,
                packageName: _this.name,
                dismissable: true
              });
            } else {
              _this.settings.push(settings);
              if (_this.settingsActivated) {
                settings.activate();
              }
            }
            return callback();
          });
        };
      })(this);
      return new Promise((function(_this) {
        return function(resolve) {
          var ref1, scopedProperties, settings, settingsDirPath, settingsPath;
          if (_this.preloadedPackage && (_this.packageManager.packagesCache[_this.name] != null)) {
            ref1 = _this.packageManager.packagesCache[_this.name].settings;
            for (settingsPath in ref1) {
              scopedProperties = ref1[settingsPath];
              settings = new ScopedProperties("core:" + settingsPath, scopedProperties != null ? scopedProperties : {}, _this.config);
              _this.settings.push(settings);
              if (_this.settingsActivated) {
                settings.activate();
              }
            }
            return resolve();
          } else {
            settingsDirPath = path.join(_this.path, 'settings');
            return fs.exists(settingsDirPath, function(settingsDirExists) {
              if (!settingsDirExists) {
                return resolve();
              }
              return fs.list(settingsDirPath, ['json', 'cson'], function(error, settingsPaths) {
                if (settingsPaths == null) {
                  settingsPaths = [];
                }
                return async.each(settingsPaths, loadSettingsFile, function() {
                  return resolve();
                });
              });
            });
          }
        };
      })(this));
    };

    Package.prototype.serialize = function() {
      var e, ref1;
      if (this.mainActivated) {
        try {
          return (ref1 = this.mainModule) != null ? typeof ref1.serialize === "function" ? ref1.serialize() : void 0 : void 0;
        } catch (error1) {
          e = error1;
          return console.error("Error serializing package '" + this.name + "'", e.stack);
        }
      }
    };

    Package.prototype.deactivate = function() {
      var e, ref1, ref2, ref3, ref4;
      this.activationPromise = null;
      this.resolveActivationPromise = null;
      if ((ref1 = this.activationCommandSubscriptions) != null) {
        ref1.dispose();
      }
      if ((ref2 = this.activationHookSubscriptions) != null) {
        ref2.dispose();
      }
      this.configSchemaRegisteredOnActivate = false;
      this.deactivateResources();
      this.deactivateKeymaps();
      if (this.mainActivated) {
        try {
          if ((ref3 = this.mainModule) != null) {
            if (typeof ref3.deactivate === "function") {
              ref3.deactivate();
            }
          }
          if ((ref4 = this.mainModule) != null) {
            if (typeof ref4.deactivateConfig === "function") {
              ref4.deactivateConfig();
            }
          }
          this.mainActivated = false;
          this.mainInitialized = false;
        } catch (error1) {
          e = error1;
          console.error("Error deactivating package '" + this.name + "'", e.stack);
        }
      }
      return this.emitter.emit('did-deactivate');
    };

    Package.prototype.deactivateResources = function() {
      var grammar, i, j, len, len1, ref1, ref2, ref3, ref4, ref5, settings;
      ref1 = this.grammars;
      for (i = 0, len = ref1.length; i < len; i++) {
        grammar = ref1[i];
        grammar.deactivate();
      }
      ref2 = this.settings;
      for (j = 0, len1 = ref2.length; j < len1; j++) {
        settings = ref2[j];
        settings.deactivate();
      }
      if ((ref3 = this.stylesheetDisposables) != null) {
        ref3.dispose();
      }
      if ((ref4 = this.activationDisposables) != null) {
        ref4.dispose();
      }
      if ((ref5 = this.keymapDisposables) != null) {
        ref5.dispose();
      }
      this.stylesheetsActivated = false;
      this.grammarsActivated = false;
      this.settingsActivated = false;
      return this.menusActivated = false;
    };

    Package.prototype.reloadStylesheets = function() {
      var error, ref1;
      try {
        this.loadStylesheets();
      } catch (error1) {
        error = error1;
        this.handleError("Failed to reload the " + this.name + " package stylesheets", error);
      }
      if ((ref1 = this.stylesheetDisposables) != null) {
        ref1.dispose();
      }
      this.stylesheetDisposables = new CompositeDisposable;
      this.stylesheetsActivated = false;
      return this.activateStylesheets();
    };

    Package.prototype.requireMainModule = function() {
      var mainModulePath, previousDeserializerCount, previousViewProviderCount;
      if (this.bundledPackage && (this.packageManager.packagesCache[this.name] != null)) {
        if (this.packageManager.packagesCache[this.name].main != null) {
          return this.mainModule = require(this.packageManager.packagesCache[this.name].main);
        }
      } else if (this.mainModuleRequired) {
        return this.mainModule;
      } else if (!this.isCompatible()) {
        console.warn("Failed to require the main module of '" + this.name + "' because it requires one or more incompatible native modules (" + (_.pluck(this.incompatibleModules, 'name').join(', ')) + ").\nRun `apm rebuild` in the package directory and restart Atom to resolve.");
      } else {
        mainModulePath = this.getMainModulePath();
        if (fs.isFileSync(mainModulePath)) {
          this.mainModuleRequired = true;
          previousViewProviderCount = this.viewRegistry.getViewProviderCount();
          previousDeserializerCount = this.deserializerManager.getDeserializerCount();
          this.mainModule = require(mainModulePath);
          if (this.viewRegistry.getViewProviderCount() === previousViewProviderCount && this.deserializerManager.getDeserializerCount() === previousDeserializerCount) {
            return localStorage.setItem(this.getCanDeferMainModuleRequireStorageKey(), 'true');
          }
        }
      }
    };

    Package.prototype.getMainModulePath = function() {
      var mainModulePath;
      if (this.resolvedMainModulePath) {
        return this.mainModulePath;
      }
      this.resolvedMainModulePath = true;
      if (this.bundledPackage && (this.packageManager.packagesCache[this.name] != null)) {
        if (this.packageManager.packagesCache[this.name].main) {
          return this.mainModulePath = path.resolve(this.packageManager.resourcePath, 'static', this.packageManager.packagesCache[this.name].main);
        } else {
          return this.mainModulePath = null;
        }
      } else {
        mainModulePath = this.metadata.main ? path.join(this.path, this.metadata.main) : path.join(this.path, 'index');
        return this.mainModulePath = fs.resolveExtension(mainModulePath, [""].concat(slice.call(CompileCache.supportedExtensions)));
      }
    };

    Package.prototype.activationShouldBeDeferred = function() {
      return this.hasActivationCommands() || this.hasActivationHooks();
    };

    Package.prototype.hasActivationHooks = function() {
      var ref1;
      return ((ref1 = this.getActivationHooks()) != null ? ref1.length : void 0) > 0;
    };

    Package.prototype.hasActivationCommands = function() {
      var commands, ref1, selector;
      ref1 = this.getActivationCommands();
      for (selector in ref1) {
        commands = ref1[selector];
        if (commands.length > 0) {
          return true;
        }
      }
      return false;
    };

    Package.prototype.subscribeToDeferredActivation = function() {
      this.subscribeToActivationCommands();
      return this.subscribeToActivationHooks();
    };

    Package.prototype.subscribeToActivationCommands = function() {
      var command, commands, fn1, i, len, ref1, selector;
      this.activationCommandSubscriptions = new CompositeDisposable;
      ref1 = this.getActivationCommands();
      for (selector in ref1) {
        commands = ref1[selector];
        fn1 = (function(_this) {
          return function(selector, command) {
            var error, metadataPath;
            try {
              _this.activationCommandSubscriptions.add(_this.commandRegistry.add(selector, command, function() {}));
            } catch (error1) {
              error = error1;
              if (error.code === 'EBADSELECTOR') {
                metadataPath = path.join(_this.path, 'package.json');
                error.message += " in " + metadataPath;
                error.stack += "\n  at " + metadataPath + ":1:1";
              }
              throw error;
            }
            return _this.activationCommandSubscriptions.add(_this.commandRegistry.onWillDispatch(function(event) {
              var currentTarget;
              if (event.type !== command) {
                return;
              }
              currentTarget = event.target;
              while (currentTarget) {
                if (currentTarget.webkitMatchesSelector(selector)) {
                  _this.activationCommandSubscriptions.dispose();
                  _this.activateNow();
                  break;
                }
                currentTarget = currentTarget.parentElement;
              }
            }));
          };
        })(this);
        for (i = 0, len = commands.length; i < len; i++) {
          command = commands[i];
          fn1(selector, command);
        }
      }
    };

    Package.prototype.getActivationCommands = function() {
      var base, commands, ref1, ref2, selector;
      if (this.activationCommands != null) {
        return this.activationCommands;
      }
      this.activationCommands = {};
      if (this.metadata.activationCommands != null) {
        ref1 = this.metadata.activationCommands;
        for (selector in ref1) {
          commands = ref1[selector];
          if ((base = this.activationCommands)[selector] == null) {
            base[selector] = [];
          }
          if (_.isString(commands)) {
            this.activationCommands[selector].push(commands);
          } else if (_.isArray(commands)) {
            (ref2 = this.activationCommands[selector]).push.apply(ref2, commands);
          }
        }
      }
      return this.activationCommands;
    };

    Package.prototype.subscribeToActivationHooks = function() {
      var fn1, hook, i, len, ref1;
      this.activationHookSubscriptions = new CompositeDisposable;
      ref1 = this.getActivationHooks();
      fn1 = (function(_this) {
        return function(hook) {
          if ((hook != null) && _.isString(hook) && hook.trim().length > 0) {
            return _this.activationHookSubscriptions.add(_this.packageManager.onDidTriggerActivationHook(hook, function() {
              return _this.activateNow();
            }));
          }
        };
      })(this);
      for (i = 0, len = ref1.length; i < len; i++) {
        hook = ref1[i];
        fn1(hook);
      }
    };

    Package.prototype.getActivationHooks = function() {
      var ref1;
      if ((this.metadata != null) && (this.activationHooks != null)) {
        return this.activationHooks;
      }
      this.activationHooks = [];
      if (this.metadata.activationHooks != null) {
        if (_.isArray(this.metadata.activationHooks)) {
          (ref1 = this.activationHooks).push.apply(ref1, this.metadata.activationHooks);
        } else if (_.isString(this.metadata.activationHooks)) {
          this.activationHooks.push(this.metadata.activationHooks);
        }
      }
      return this.activationHooks = _.uniq(this.activationHooks);
    };

    Package.prototype.isNativeModule = function(modulePath) {
      var error;
      try {
        return fs.listSync(path.join(modulePath, 'build', 'Release'), ['.node']).length > 0;
      } catch (error1) {
        error = error1;
        return false;
      }
    };

    Package.prototype.getNativeModuleDependencyPaths = function() {
      var i, len, nativeModulePath, nativeModulePaths, ref1, ref2, relativeNativeModuleBindingPath, relativeNativeModuleBindingPaths, traversePath;
      nativeModulePaths = [];
      if (this.metadata._atomModuleCache != null) {
        relativeNativeModuleBindingPaths = (ref1 = (ref2 = this.metadata._atomModuleCache.extensions) != null ? ref2['.node'] : void 0) != null ? ref1 : [];
        for (i = 0, len = relativeNativeModuleBindingPaths.length; i < len; i++) {
          relativeNativeModuleBindingPath = relativeNativeModuleBindingPaths[i];
          nativeModulePath = path.join(this.path, relativeNativeModuleBindingPath, '..', '..', '..');
          nativeModulePaths.push(nativeModulePath);
        }
        return nativeModulePaths;
      }
      traversePath = (function(_this) {
        return function(nodeModulesPath) {
          var j, len1, modulePath, ref3;
          try {
            ref3 = fs.listSync(nodeModulesPath);
            for (j = 0, len1 = ref3.length; j < len1; j++) {
              modulePath = ref3[j];
              if (_this.isNativeModule(modulePath)) {
                nativeModulePaths.push(modulePath);
              }
              traversePath(path.join(modulePath, 'node_modules'));
            }
          } catch (error1) {}
        };
      })(this);
      traversePath(path.join(this.path, 'node_modules'));
      return nativeModulePaths;
    };


    /*
    Section: Native Module Compatibility
     */

    Package.prototype.isCompatible = function() {
      if (this.compatible != null) {
        return this.compatible;
      }
      if (this.preloadedPackage) {
        return this.compatible = true;
      } else if (this.getMainModulePath()) {
        this.incompatibleModules = this.getIncompatibleNativeModules();
        return this.compatible = this.incompatibleModules.length === 0 && (this.getBuildFailureOutput() == null);
      } else {
        return this.compatible = true;
      }
    };

    Package.prototype.rebuild = function() {
      return new Promise((function(_this) {
        return function(resolve) {
          return _this.runRebuildProcess(function(result) {
            if (result.code === 0) {
              global.localStorage.removeItem(_this.getBuildFailureOutputStorageKey());
            } else {
              _this.compatible = false;
              global.localStorage.setItem(_this.getBuildFailureOutputStorageKey(), result.stderr);
            }
            global.localStorage.setItem(_this.getIncompatibleNativeModulesStorageKey(), '[]');
            return resolve(result);
          });
        };
      })(this));
    };

    Package.prototype.getBuildFailureOutput = function() {
      return global.localStorage.getItem(this.getBuildFailureOutputStorageKey());
    };

    Package.prototype.runRebuildProcess = function(callback) {
      var stderr, stdout;
      stderr = '';
      stdout = '';
      return new BufferedProcess({
        command: this.packageManager.getApmPath(),
        args: ['rebuild', '--no-color'],
        options: {
          cwd: this.path
        },
        stderr: function(output) {
          return stderr += output;
        },
        stdout: function(output) {
          return stdout += output;
        },
        exit: function(code) {
          return callback({
            code: code,
            stdout: stdout,
            stderr: stderr
          });
        }
      });
    };

    Package.prototype.getBuildFailureOutputStorageKey = function() {
      return "installed-packages:" + this.name + ":" + this.metadata.version + ":build-error";
    };

    Package.prototype.getIncompatibleNativeModulesStorageKey = function() {
      var electronVersion;
      electronVersion = process.versions.electron;
      return "installed-packages:" + this.name + ":" + this.metadata.version + ":electron-" + electronVersion + ":incompatible-native-modules";
    };

    Package.prototype.getCanDeferMainModuleRequireStorageKey = function() {
      return "installed-packages:" + this.name + ":" + this.metadata.version + ":can-defer-main-module-require";
    };

    Package.prototype.getIncompatibleNativeModules = function() {
      var arrayAsString, error, i, incompatibleNativeModules, len, nativeModulePath, ref1, version;
      if (!this.packageManager.devMode) {
        try {
          if (arrayAsString = global.localStorage.getItem(this.getIncompatibleNativeModulesStorageKey())) {
            return JSON.parse(arrayAsString);
          }
        } catch (error1) {}
      }
      incompatibleNativeModules = [];
      ref1 = this.getNativeModuleDependencyPaths();
      for (i = 0, len = ref1.length; i < len; i++) {
        nativeModulePath = ref1[i];
        try {
          require(nativeModulePath);
        } catch (error1) {
          error = error1;
          try {
            version = require(nativeModulePath + "/package.json").version;
          } catch (error1) {}
          incompatibleNativeModules.push({
            path: nativeModulePath,
            name: path.basename(nativeModulePath),
            version: version,
            error: error.message
          });
        }
      }
      global.localStorage.setItem(this.getIncompatibleNativeModulesStorageKey(), JSON.stringify(incompatibleNativeModules));
      return incompatibleNativeModules;
    };

    Package.prototype.handleError = function(message, error) {
      var detail, location, ref1, stack;
      if (atom.inSpecMode()) {
        throw error;
      }
      if (error.filename && error.location && (error instanceof SyntaxError)) {
        location = error.filename + ":" + (error.location.first_line + 1) + ":" + (error.location.first_column + 1);
        detail = error.message + " in " + location;
        stack = "SyntaxError: " + error.message + "\n  at " + location;
      } else if (error.less && error.filename && (error.column != null) && (error.line != null)) {
        location = error.filename + ":" + error.line + ":" + error.column;
        detail = error.message + " in " + location;
        stack = "LessError: " + error.message + "\n  at " + location;
      } else {
        detail = error.message;
        stack = (ref1 = error.stack) != null ? ref1 : error;
      }
      return this.notificationManager.addFatalError(message, {
        stack: stack,
        detail: detail,
        packageName: this.name,
        dismissable: true
      });
    };

    return Package;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
