(function() {
  var CSON, Color, Config, Emitter, ScopeDescriptor, ScopedPropertyStore, _, async, deleteValueAtKeyPath, fs, getValueAtKeyPath, isPlainObject, path, pathWatcher, pushKeyPath, ref, setValueAtKeyPath, sortObject, splitKeyPath, withoutEmptyObjects,
    slice = [].slice;

  _ = require('underscore-plus');

  fs = require('fs-plus');

  Emitter = require('event-kit').Emitter;

  CSON = require('season');

  path = require('path');

  async = require('async');

  pathWatcher = require('pathwatcher');

  ref = require('key-path-helpers'), getValueAtKeyPath = ref.getValueAtKeyPath, setValueAtKeyPath = ref.setValueAtKeyPath, deleteValueAtKeyPath = ref.deleteValueAtKeyPath, pushKeyPath = ref.pushKeyPath, splitKeyPath = ref.splitKeyPath;

  Color = require('./color');

  ScopedPropertyStore = require('scoped-property-store');

  ScopeDescriptor = require('./scope-descriptor');

  module.exports = Config = (function() {
    Config.schemaEnforcers = {};

    Config.addSchemaEnforcer = function(typeName, enforcerFunction) {
      var base;
      if ((base = this.schemaEnforcers)[typeName] == null) {
        base[typeName] = [];
      }
      return this.schemaEnforcers[typeName].push(enforcerFunction);
    };

    Config.addSchemaEnforcers = function(filters) {
      var enforcerFunction, functions, name, typeName;
      for (typeName in filters) {
        functions = filters[typeName];
        for (name in functions) {
          enforcerFunction = functions[name];
          this.addSchemaEnforcer(typeName, enforcerFunction);
        }
      }
    };

    Config.executeSchemaEnforcers = function(keyPath, value, schema) {
      var e, enforcer, enforcerFunctions, error, j, k, len, len1, type, types;
      error = null;
      types = schema.type;
      if (!Array.isArray(types)) {
        types = [types];
      }
      for (j = 0, len = types.length; j < len; j++) {
        type = types[j];
        try {
          enforcerFunctions = this.schemaEnforcers[type].concat(this.schemaEnforcers['*']);
          for (k = 0, len1 = enforcerFunctions.length; k < len1; k++) {
            enforcer = enforcerFunctions[k];
            value = enforcer.call(this, keyPath, value, schema);
          }
          error = null;
          break;
        } catch (error1) {
          e = error1;
          error = e;
        }
      }
      if (error != null) {
        throw error;
      }
      return value;
    };

    function Config(arg) {
      var ref1;
      ref1 = arg != null ? arg : {}, this.notificationManager = ref1.notificationManager, this.enablePersistence = ref1.enablePersistence;
      this.clear();
    }

    Config.prototype.initialize = function(arg) {
      var projectHomeSchema;
      this.configDirPath = arg.configDirPath, this.resourcePath = arg.resourcePath, projectHomeSchema = arg.projectHomeSchema;
      if (this.enablePersistence != null) {
        this.configFilePath = fs.resolve(this.configDirPath, 'config', ['json', 'cson']);
        if (this.configFilePath == null) {
          this.configFilePath = path.join(this.configDirPath, 'config.cson');
        }
      }
      this.schema.properties.core.properties.projectHome = projectHomeSchema;
      return this.defaultSettings.core.projectHome = projectHomeSchema["default"];
    };

    Config.prototype.clear = function() {
      var debouncedSave, save;
      this.emitter = new Emitter;
      this.schema = {
        type: 'object',
        properties: {}
      };
      this.defaultSettings = {};
      this.settings = {};
      this.scopedSettingsStore = new ScopedPropertyStore;
      this.configFileHasErrors = false;
      this.transactDepth = 0;
      this.savePending = false;
      this.requestLoad = _.debounce(this.loadUserConfig, 100);
      this.requestSave = (function(_this) {
        return function() {
          _this.savePending = true;
          return debouncedSave.call(_this);
        };
      })(this);
      save = (function(_this) {
        return function() {
          _this.savePending = false;
          return _this.save();
        };
      })(this);
      return debouncedSave = _.debounce(save, 100);
    };

    Config.prototype.shouldNotAccessFileSystem = function() {
      return !this.enablePersistence;
    };


    /*
    Section: Config Subscription
     */

    Config.prototype.observe = function() {
      var callback, keyPath, options, scopeDescriptor;
      if (arguments.length === 2) {
        keyPath = arguments[0], callback = arguments[1];
      } else if (arguments.length === 3 && (_.isString(arguments[0]) && _.isObject(arguments[1]))) {
        keyPath = arguments[0], options = arguments[1], callback = arguments[2];
        scopeDescriptor = options.scope;
      } else {
        console.error('An unsupported form of Config::observe is being used. See https://atom.io/docs/api/latest/Config for details');
        return;
      }
      if (scopeDescriptor != null) {
        return this.observeScopedKeyPath(scopeDescriptor, keyPath, callback);
      } else {
        return this.observeKeyPath(keyPath, options != null ? options : {}, callback);
      }
    };

    Config.prototype.onDidChange = function() {
      var callback, keyPath, options, scopeDescriptor;
      if (arguments.length === 1) {
        callback = arguments[0];
      } else if (arguments.length === 2) {
        keyPath = arguments[0], callback = arguments[1];
      } else {
        keyPath = arguments[0], options = arguments[1], callback = arguments[2];
        scopeDescriptor = options.scope;
      }
      if (scopeDescriptor != null) {
        return this.onDidChangeScopedKeyPath(scopeDescriptor, keyPath, callback);
      } else {
        return this.onDidChangeKeyPath(keyPath, callback);
      }
    };


    /*
    Section: Managing Settings
     */

    Config.prototype.get = function() {
      var keyPath, options, scope, value;
      if (arguments.length > 1) {
        if (typeof arguments[0] === 'string' || (arguments[0] == null)) {
          keyPath = arguments[0], options = arguments[1];
          scope = options.scope;
        }
      } else {
        keyPath = arguments[0];
      }
      if (scope != null) {
        value = this.getRawScopedValue(scope, keyPath, options);
        return value != null ? value : this.getRawValue(keyPath, options);
      } else {
        return this.getRawValue(keyPath, options);
      }
    };

    Config.prototype.getAll = function(keyPath, options) {
      var globalValue, result, scope, scopeDescriptor;
      if (options != null) {
        scope = options.scope;
      }
      result = [];
      if (scope != null) {
        scopeDescriptor = ScopeDescriptor.fromObject(scope);
        result = result.concat(this.scopedSettingsStore.getAll(scopeDescriptor.getScopeChain(), keyPath, options));
      }
      if (globalValue = this.getRawValue(keyPath, options)) {
        result.push({
          scopeSelector: '*',
          value: globalValue
        });
      }
      return result;
    };

    Config.prototype.set = function() {
      var e, keyPath, options, ref1, scopeSelector, shouldSave, source, value;
      keyPath = arguments[0], value = arguments[1], options = arguments[2];
      scopeSelector = options != null ? options.scopeSelector : void 0;
      source = options != null ? options.source : void 0;
      shouldSave = (ref1 = options != null ? options.save : void 0) != null ? ref1 : true;
      if (source && !scopeSelector) {
        throw new Error("::set with a 'source' and no 'sourceSelector' is not yet implemented!");
      }
      if (source == null) {
        source = this.getUserConfigPath();
      }
      if (value !== void 0) {
        try {
          value = this.makeValueConformToSchema(keyPath, value);
        } catch (error1) {
          e = error1;
          return false;
        }
      }
      if (scopeSelector != null) {
        this.setRawScopedValue(keyPath, value, source, scopeSelector);
      } else {
        this.setRawValue(keyPath, value);
      }
      if (source === this.getUserConfigPath() && shouldSave && !this.configFileHasErrors) {
        this.requestSave();
      }
      return true;
    };

    Config.prototype.unset = function(keyPath, options) {
      var ref1, scopeSelector, settings, source;
      ref1 = options != null ? options : {}, scopeSelector = ref1.scopeSelector, source = ref1.source;
      if (source == null) {
        source = this.getUserConfigPath();
      }
      if (scopeSelector != null) {
        if (keyPath != null) {
          settings = this.scopedSettingsStore.propertiesForSourceAndSelector(source, scopeSelector);
          if (getValueAtKeyPath(settings, keyPath) != null) {
            this.scopedSettingsStore.removePropertiesForSourceAndSelector(source, scopeSelector);
            setValueAtKeyPath(settings, keyPath, void 0);
            settings = withoutEmptyObjects(settings);
            if (settings != null) {
              this.set(null, settings, {
                scopeSelector: scopeSelector,
                source: source,
                priority: this.priorityForSource(source)
              });
            }
            return this.requestSave();
          }
        } else {
          this.scopedSettingsStore.removePropertiesForSourceAndSelector(source, scopeSelector);
          return this.emitChangeEvent();
        }
      } else {
        for (scopeSelector in this.scopedSettingsStore.propertiesForSource(source)) {
          this.unset(keyPath, {
            scopeSelector: scopeSelector,
            source: source
          });
        }
        if ((keyPath != null) && source === this.getUserConfigPath()) {
          return this.set(keyPath, getValueAtKeyPath(this.defaultSettings, keyPath));
        }
      }
    };

    Config.prototype.getSources = function() {
      return _.uniq(_.pluck(this.scopedSettingsStore.propertySets, 'source')).sort();
    };

    Config.prototype.getSchema = function(keyPath) {
      var childSchema, j, key, keys, len, ref1, schema;
      keys = splitKeyPath(keyPath);
      schema = this.schema;
      for (j = 0, len = keys.length; j < len; j++) {
        key = keys[j];
        if (schema.type === 'object') {
          childSchema = (ref1 = schema.properties) != null ? ref1[key] : void 0;
          if (childSchema == null) {
            if (isPlainObject(schema.additionalProperties)) {
              childSchema = schema.additionalProperties;
            } else if (schema.additionalProperties === false) {
              return null;
            } else {
              return {
                type: 'any'
              };
            }
          }
        } else {
          return null;
        }
        schema = childSchema;
      }
      return schema;
    };

    Config.prototype.getUserConfigPath = function() {
      return this.configFilePath;
    };

    Config.prototype.transact = function(callback) {
      this.beginTransaction();
      try {
        return callback();
      } finally {
        this.endTransaction();
      }
    };


    /*
    Section: Internal methods used by core
     */

    Config.prototype.transactAsync = function(callback) {
      var endTransaction, error, result;
      this.beginTransaction();
      try {
        endTransaction = (function(_this) {
          return function(fn) {
            return function() {
              var args;
              args = 1 <= arguments.length ? slice.call(arguments, 0) : [];
              _this.endTransaction();
              return fn.apply(null, args);
            };
          };
        })(this);
        result = callback();
        return new Promise(function(resolve, reject) {
          return result.then(endTransaction(resolve))["catch"](endTransaction(reject));
        });
      } catch (error1) {
        error = error1;
        this.endTransaction();
        return Promise.reject(error);
      }
    };

    Config.prototype.beginTransaction = function() {
      return this.transactDepth++;
    };

    Config.prototype.endTransaction = function() {
      this.transactDepth--;
      return this.emitChangeEvent();
    };

    Config.prototype.pushAtKeyPath = function(keyPath, value) {
      var arrayValue, ref1, result;
      arrayValue = (ref1 = this.get(keyPath)) != null ? ref1 : [];
      result = arrayValue.push(value);
      this.set(keyPath, arrayValue);
      return result;
    };

    Config.prototype.unshiftAtKeyPath = function(keyPath, value) {
      var arrayValue, ref1, result;
      arrayValue = (ref1 = this.get(keyPath)) != null ? ref1 : [];
      result = arrayValue.unshift(value);
      this.set(keyPath, arrayValue);
      return result;
    };

    Config.prototype.removeAtKeyPath = function(keyPath, value) {
      var arrayValue, ref1, result;
      arrayValue = (ref1 = this.get(keyPath)) != null ? ref1 : [];
      result = _.remove(arrayValue, value);
      this.set(keyPath, arrayValue);
      return result;
    };

    Config.prototype.setSchema = function(keyPath, schema) {
      var j, key, len, properties, ref1, rootSchema;
      if (!isPlainObject(schema)) {
        throw new Error("Error loading schema for " + keyPath + ": schemas can only be objects!");
      }
      if (!typeof (schema.type != null)) {
        throw new Error("Error loading schema for " + keyPath + ": schema objects must have a type attribute");
      }
      rootSchema = this.schema;
      if (keyPath) {
        ref1 = splitKeyPath(keyPath);
        for (j = 0, len = ref1.length; j < len; j++) {
          key = ref1[j];
          rootSchema.type = 'object';
          if (rootSchema.properties == null) {
            rootSchema.properties = {};
          }
          properties = rootSchema.properties;
          if (properties[key] == null) {
            properties[key] = {};
          }
          rootSchema = properties[key];
        }
      }
      Object.assign(rootSchema, schema);
      return this.transact((function(_this) {
        return function() {
          _this.setDefaults(keyPath, _this.extractDefaultsFromSchema(schema));
          _this.setScopedDefaultsFromSchema(keyPath, schema);
          return _this.resetSettingsForSchemaChange();
        };
      })(this));
    };

    Config.prototype.load = function() {
      this.initializeConfigDirectory();
      this.loadUserConfig();
      return this.observeUserConfig();
    };


    /*
    Section: Private methods managing the user's config file
     */

    Config.prototype.initializeConfigDirectory = function(done) {
      var onConfigDirFile, queue, templateConfigDirPath;
      if (fs.existsSync(this.configDirPath) || this.shouldNotAccessFileSystem()) {
        return;
      }
      fs.makeTreeSync(this.configDirPath);
      queue = async.queue(function(arg, callback) {
        var destinationPath, sourcePath;
        sourcePath = arg.sourcePath, destinationPath = arg.destinationPath;
        return fs.copy(sourcePath, destinationPath, callback);
      });
      queue.drain = done;
      templateConfigDirPath = fs.resolve(this.resourcePath, 'dot-atom');
      onConfigDirFile = (function(_this) {
        return function(sourcePath) {
          var destinationPath, relativePath;
          relativePath = sourcePath.substring(templateConfigDirPath.length + 1);
          destinationPath = path.join(_this.configDirPath, relativePath);
          return queue.push({
            sourcePath: sourcePath,
            destinationPath: destinationPath
          });
        };
      })(this);
      return fs.traverseTree(templateConfigDirPath, onConfigDirFile, (function(path) {
        return true;
      }), (function() {}));
    };

    Config.prototype.loadUserConfig = function() {
      var detail, error, message, userConfig;
      if (this.shouldNotAccessFileSystem()) {
        return;
      }
      try {
        if (!fs.existsSync(this.configFilePath)) {
          fs.makeTreeSync(path.dirname(this.configFilePath));
          CSON.writeFileSync(this.configFilePath, {});
        }
      } catch (error1) {
        error = error1;
        this.configFileHasErrors = true;
        this.notifyFailure("Failed to initialize `" + (path.basename(this.configFilePath)) + "`", error.stack);
        return;
      }
      try {
        if (!this.savePending) {
          userConfig = CSON.readFileSync(this.configFilePath);
          this.resetUserSettings(userConfig);
          return this.configFileHasErrors = false;
        }
      } catch (error1) {
        error = error1;
        this.configFileHasErrors = true;
        message = "Failed to load `" + (path.basename(this.configFilePath)) + "`";
        detail = error.location != null ? error.stack : error.message;
        return this.notifyFailure(message, detail);
      }
    };

    Config.prototype.observeUserConfig = function() {
      var error;
      if (this.shouldNotAccessFileSystem()) {
        return;
      }
      try {
        return this.watchSubscription != null ? this.watchSubscription : this.watchSubscription = pathWatcher.watch(this.configFilePath, (function(_this) {
          return function(eventType) {
            if (eventType === 'change' && (_this.watchSubscription != null)) {
              return _this.requestLoad();
            }
          };
        })(this));
      } catch (error1) {
        error = error1;
        return this.notifyFailure("Unable to watch path: `" + (path.basename(this.configFilePath)) + "`. Make sure you have permissions to\n`" + this.configFilePath + "`. On linux there are currently problems with watch\nsizes. See [this document][watches] for more info.\n[watches]:https://github.com/atom/atom/blob/master/docs/build-instructions/linux.md#typeerror-unable-to-watch-path");
      }
    };

    Config.prototype.unobserveUserConfig = function() {
      var ref1;
      if ((ref1 = this.watchSubscription) != null) {
        ref1.close();
      }
      return this.watchSubscription = null;
    };

    Config.prototype.notifyFailure = function(errorMessage, detail) {
      var ref1;
      return (ref1 = this.notificationManager) != null ? ref1.addError(errorMessage, {
        detail: detail,
        dismissable: true
      }) : void 0;
    };

    Config.prototype.save = function() {
      var allSettings, detail, error, message;
      if (this.shouldNotAccessFileSystem()) {
        return;
      }
      allSettings = {
        '*': this.settings
      };
      allSettings = Object.assign(allSettings, this.scopedSettingsStore.propertiesForSource(this.getUserConfigPath()));
      allSettings = sortObject(allSettings);
      try {
        return CSON.writeFileSync(this.configFilePath, allSettings);
      } catch (error1) {
        error = error1;
        message = "Failed to save `" + (path.basename(this.configFilePath)) + "`";
        detail = error.message;
        return this.notifyFailure(message, detail);
      }
    };


    /*
    Section: Private methods managing global settings
     */

    Config.prototype.resetUserSettings = function(newSettings) {
      var scopedSettings;
      if (!isPlainObject(newSettings)) {
        this.settings = {};
        this.emitChangeEvent();
        return;
      }
      if (newSettings.global != null) {
        newSettings['*'] = newSettings.global;
        delete newSettings.global;
      }
      if (newSettings['*'] != null) {
        scopedSettings = newSettings;
        newSettings = newSettings['*'];
        delete scopedSettings['*'];
        this.resetUserScopedSettings(scopedSettings);
      }
      return this.transact((function(_this) {
        return function() {
          var key, value;
          _this.settings = {};
          for (key in newSettings) {
            value = newSettings[key];
            _this.set(key, value, {
              save: false
            });
          }
        };
      })(this));
    };

    Config.prototype.getRawValue = function(keyPath, options) {
      var defaultValue, ref1, ref2, value;
      if (!((options != null ? (ref1 = options.excludeSources) != null ? ref1.indexOf(this.getUserConfigPath()) : void 0 : void 0) >= 0)) {
        value = getValueAtKeyPath(this.settings, keyPath);
      }
      if (!((options != null ? (ref2 = options.sources) != null ? ref2.length : void 0 : void 0) > 0)) {
        defaultValue = getValueAtKeyPath(this.defaultSettings, keyPath);
      }
      if (value != null) {
        value = this.deepClone(value);
        if (isPlainObject(value) && isPlainObject(defaultValue)) {
          this.deepDefaults(value, defaultValue);
        }
      } else {
        value = this.deepClone(defaultValue);
      }
      return value;
    };

    Config.prototype.setRawValue = function(keyPath, value) {
      var defaultValue;
      defaultValue = getValueAtKeyPath(this.defaultSettings, keyPath);
      if (_.isEqual(defaultValue, value)) {
        if (keyPath != null) {
          deleteValueAtKeyPath(this.settings, keyPath);
        } else {
          this.settings = null;
        }
      } else {
        if (keyPath != null) {
          setValueAtKeyPath(this.settings, keyPath, value);
        } else {
          this.settings = value;
        }
      }
      return this.emitChangeEvent();
    };

    Config.prototype.observeKeyPath = function(keyPath, options, callback) {
      callback(this.get(keyPath));
      return this.onDidChangeKeyPath(keyPath, function(event) {
        return callback(event.newValue);
      });
    };

    Config.prototype.onDidChangeKeyPath = function(keyPath, callback) {
      var oldValue;
      oldValue = this.get(keyPath);
      return this.emitter.on('did-change', (function(_this) {
        return function() {
          var event, newValue;
          newValue = _this.get(keyPath);
          if (!_.isEqual(oldValue, newValue)) {
            event = {
              oldValue: oldValue,
              newValue: newValue
            };
            oldValue = newValue;
            return callback(event);
          }
        };
      })(this));
    };

    Config.prototype.isSubKeyPath = function(keyPath, subKeyPath) {
      var pathSubTokens, pathTokens;
      if (!((keyPath != null) && (subKeyPath != null))) {
        return false;
      }
      pathSubTokens = splitKeyPath(subKeyPath);
      pathTokens = splitKeyPath(keyPath).slice(0, pathSubTokens.length);
      return _.isEqual(pathTokens, pathSubTokens);
    };

    Config.prototype.setRawDefault = function(keyPath, value) {
      setValueAtKeyPath(this.defaultSettings, keyPath, value);
      return this.emitChangeEvent();
    };

    Config.prototype.setDefaults = function(keyPath, defaults) {
      var e, keys;
      if ((defaults != null) && isPlainObject(defaults)) {
        keys = splitKeyPath(keyPath);
        this.transact((function(_this) {
          return function() {
            var childValue, key, results;
            results = [];
            for (key in defaults) {
              childValue = defaults[key];
              if (!defaults.hasOwnProperty(key)) {
                continue;
              }
              results.push(_this.setDefaults(keys.concat([key]).join('.'), childValue));
            }
            return results;
          };
        })(this));
      } else {
        try {
          defaults = this.makeValueConformToSchema(keyPath, defaults);
          this.setRawDefault(keyPath, defaults);
        } catch (error1) {
          e = error1;
          console.warn("'" + keyPath + "' could not set the default. Attempted default: " + (JSON.stringify(defaults)) + "; Schema: " + (JSON.stringify(this.getSchema(keyPath))));
        }
      }
    };

    Config.prototype.deepClone = function(object) {
      if (object instanceof Color) {
        return object.clone();
      } else if (_.isArray(object)) {
        return object.map((function(_this) {
          return function(value) {
            return _this.deepClone(value);
          };
        })(this));
      } else if (isPlainObject(object)) {
        return _.mapObject(object, (function(_this) {
          return function(key, value) {
            return [key, _this.deepClone(value)];
          };
        })(this));
      } else {
        return object;
      }
    };

    Config.prototype.deepDefaults = function(target) {
      var i, j, key, len, object, ref1, result;
      result = target;
      i = 0;
      while (++i < arguments.length) {
        object = arguments[i];
        if (isPlainObject(result) && isPlainObject(object)) {
          ref1 = Object.keys(object);
          for (j = 0, len = ref1.length; j < len; j++) {
            key = ref1[j];
            result[key] = this.deepDefaults(result[key], object[key]);
          }
        } else {
          if (result == null) {
            result = this.deepClone(object);
          }
        }
      }
      return result;
    };

    Config.prototype.setScopedDefaultsFromSchema = function(keyPath, schema) {
      var childValue, key, keys, ref1, ref2, scope, scopeSchema, scopedDefaults;
      if ((schema.scopes != null) && isPlainObject(schema.scopes)) {
        scopedDefaults = {};
        ref1 = schema.scopes;
        for (scope in ref1) {
          scopeSchema = ref1[scope];
          if (!scopeSchema.hasOwnProperty('default')) {
            continue;
          }
          scopedDefaults[scope] = {};
          setValueAtKeyPath(scopedDefaults[scope], keyPath, scopeSchema["default"]);
        }
        this.scopedSettingsStore.addProperties('schema-default', scopedDefaults);
      }
      if (schema.type === 'object' && (schema.properties != null) && isPlainObject(schema.properties)) {
        keys = splitKeyPath(keyPath);
        ref2 = schema.properties;
        for (key in ref2) {
          childValue = ref2[key];
          if (!schema.properties.hasOwnProperty(key)) {
            continue;
          }
          this.setScopedDefaultsFromSchema(keys.concat([key]).join('.'), childValue);
        }
      }
    };

    Config.prototype.extractDefaultsFromSchema = function(schema) {
      var defaults, key, properties, value;
      if (schema["default"] != null) {
        return schema["default"];
      } else if (schema.type === 'object' && (schema.properties != null) && isPlainObject(schema.properties)) {
        defaults = {};
        properties = schema.properties || {};
        for (key in properties) {
          value = properties[key];
          defaults[key] = this.extractDefaultsFromSchema(value);
        }
        return defaults;
      }
    };

    Config.prototype.makeValueConformToSchema = function(keyPath, value, options) {
      var e, schema;
      if (options != null ? options.suppressException : void 0) {
        try {
          return this.makeValueConformToSchema(keyPath, value);
        } catch (error1) {
          e = error1;
          return void 0;
        }
      } else {
        if ((schema = this.getSchema(keyPath)) == null) {
          if (schema === false) {
            throw new Error("Illegal key path " + keyPath);
          }
        }
        return this.constructor.executeSchemaEnforcers(keyPath, value, schema);
      }
    };

    Config.prototype.resetSettingsForSchemaChange = function(source) {
      if (source == null) {
        source = this.getUserConfigPath();
      }
      return this.transact((function(_this) {
        return function() {
          var scopeSelector, selectorsAndSettings, settings;
          _this.settings = _this.makeValueConformToSchema(null, _this.settings, {
            suppressException: true
          });
          selectorsAndSettings = _this.scopedSettingsStore.propertiesForSource(source);
          _this.scopedSettingsStore.removePropertiesForSource(source);
          for (scopeSelector in selectorsAndSettings) {
            settings = selectorsAndSettings[scopeSelector];
            settings = _this.makeValueConformToSchema(null, settings, {
              suppressException: true
            });
            _this.setRawScopedValue(null, settings, source, scopeSelector);
          }
        };
      })(this));
    };


    /*
    Section: Private Scoped Settings
     */

    Config.prototype.priorityForSource = function(source) {
      if (source === this.getUserConfigPath()) {
        return 1000;
      } else {
        return 0;
      }
    };

    Config.prototype.emitChangeEvent = function() {
      if (!(this.transactDepth > 0)) {
        return this.emitter.emit('did-change');
      }
    };

    Config.prototype.resetUserScopedSettings = function(newScopedSettings) {
      var priority, scopeSelector, settings, source, validatedSettings;
      source = this.getUserConfigPath();
      priority = this.priorityForSource(source);
      this.scopedSettingsStore.removePropertiesForSource(source);
      for (scopeSelector in newScopedSettings) {
        settings = newScopedSettings[scopeSelector];
        settings = this.makeValueConformToSchema(null, settings, {
          suppressException: true
        });
        validatedSettings = {};
        validatedSettings[scopeSelector] = withoutEmptyObjects(settings);
        if (validatedSettings[scopeSelector] != null) {
          this.scopedSettingsStore.addProperties(source, validatedSettings, {
            priority: priority
          });
        }
      }
      return this.emitChangeEvent();
    };

    Config.prototype.setRawScopedValue = function(keyPath, value, source, selector, options) {
      var newValue, settingsBySelector;
      if (keyPath != null) {
        newValue = {};
        setValueAtKeyPath(newValue, keyPath, value);
        value = newValue;
      }
      settingsBySelector = {};
      settingsBySelector[selector] = value;
      this.scopedSettingsStore.addProperties(source, settingsBySelector, {
        priority: this.priorityForSource(source)
      });
      return this.emitChangeEvent();
    };

    Config.prototype.getRawScopedValue = function(scopeDescriptor, keyPath, options) {
      scopeDescriptor = ScopeDescriptor.fromObject(scopeDescriptor);
      return this.scopedSettingsStore.getPropertyValue(scopeDescriptor.getScopeChain(), keyPath, options);
    };

    Config.prototype.observeScopedKeyPath = function(scope, keyPath, callback) {
      callback(this.get(keyPath, {
        scope: scope
      }));
      return this.onDidChangeScopedKeyPath(scope, keyPath, function(event) {
        return callback(event.newValue);
      });
    };

    Config.prototype.onDidChangeScopedKeyPath = function(scope, keyPath, callback) {
      var oldValue;
      oldValue = this.get(keyPath, {
        scope: scope
      });
      return this.emitter.on('did-change', (function(_this) {
        return function() {
          var event, newValue;
          newValue = _this.get(keyPath, {
            scope: scope
          });
          if (!_.isEqual(oldValue, newValue)) {
            event = {
              oldValue: oldValue,
              newValue: newValue
            };
            oldValue = newValue;
            return callback(event);
          }
        };
      })(this));
    };

    return Config;

  })();

  Config.addSchemaEnforcers({
    'any': {
      coerce: function(keyPath, value, schema) {
        return value;
      }
    },
    'integer': {
      coerce: function(keyPath, value, schema) {
        value = parseInt(value);
        if (isNaN(value) || !isFinite(value)) {
          throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " cannot be coerced into an int");
        }
        return value;
      }
    },
    'number': {
      coerce: function(keyPath, value, schema) {
        value = parseFloat(value);
        if (isNaN(value) || !isFinite(value)) {
          throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " cannot be coerced into a number");
        }
        return value;
      }
    },
    'boolean': {
      coerce: function(keyPath, value, schema) {
        switch (typeof value) {
          case 'string':
            if (value.toLowerCase() === 'true') {
              return true;
            } else if (value.toLowerCase() === 'false') {
              return false;
            } else {
              throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " must be a boolean or the string 'true' or 'false'");
            }
            break;
          case 'boolean':
            return value;
          default:
            throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " must be a boolean or the string 'true' or 'false'");
        }
      }
    },
    'string': {
      validate: function(keyPath, value, schema) {
        if (typeof value !== 'string') {
          throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " must be a string");
        }
        return value;
      },
      validateMaximumLength: function(keyPath, value, schema) {
        if (typeof schema.maximumLength === 'number' && value.length > schema.maximumLength) {
          return value.slice(0, schema.maximumLength);
        } else {
          return value;
        }
      }
    },
    'null': {
      coerce: function(keyPath, value, schema) {
        if (value !== (void 0) && value !== null) {
          throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " must be null");
        }
        return value;
      }
    },
    'object': {
      coerce: function(keyPath, value, schema) {
        var allowsAdditionalProperties, childSchema, defaultChildSchema, error, newValue, prop, propValue, ref1;
        if (!isPlainObject(value)) {
          throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " must be an object");
        }
        if (schema.properties == null) {
          return value;
        }
        defaultChildSchema = null;
        allowsAdditionalProperties = true;
        if (isPlainObject(schema.additionalProperties)) {
          defaultChildSchema = schema.additionalProperties;
        }
        if (schema.additionalProperties === false) {
          allowsAdditionalProperties = false;
        }
        newValue = {};
        for (prop in value) {
          propValue = value[prop];
          childSchema = (ref1 = schema.properties[prop]) != null ? ref1 : defaultChildSchema;
          if (childSchema != null) {
            try {
              newValue[prop] = this.executeSchemaEnforcers(pushKeyPath(keyPath, prop), propValue, childSchema);
            } catch (error1) {
              error = error1;
              console.warn("Error setting item in object: " + error.message);
            }
          } else if (allowsAdditionalProperties) {
            newValue[prop] = propValue;
          } else {
            console.warn("Illegal object key: " + keyPath + "." + prop);
          }
        }
        return newValue;
      }
    },
    'array': {
      coerce: function(keyPath, value, schema) {
        var error, item, itemSchema, j, len, newValue;
        if (!Array.isArray(value)) {
          throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " must be an array");
        }
        itemSchema = schema.items;
        if (itemSchema != null) {
          newValue = [];
          for (j = 0, len = value.length; j < len; j++) {
            item = value[j];
            try {
              newValue.push(this.executeSchemaEnforcers(keyPath, item, itemSchema));
            } catch (error1) {
              error = error1;
              console.warn("Error setting item in array: " + error.message);
            }
          }
          return newValue;
        } else {
          return value;
        }
      }
    },
    'color': {
      coerce: function(keyPath, value, schema) {
        var color;
        color = Color.parse(value);
        if (color == null) {
          throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " cannot be coerced into a color");
        }
        return color;
      }
    },
    '*': {
      coerceMinimumAndMaximum: function(keyPath, value, schema) {
        if (typeof value !== 'number') {
          return value;
        }
        if ((schema.minimum != null) && typeof schema.minimum === 'number') {
          value = Math.max(value, schema.minimum);
        }
        if ((schema.maximum != null) && typeof schema.maximum === 'number') {
          value = Math.min(value, schema.maximum);
        }
        return value;
      },
      validateEnum: function(keyPath, value, schema) {
        var j, len, possibleValue, possibleValues;
        possibleValues = schema["enum"];
        if (Array.isArray(possibleValues)) {
          possibleValues = possibleValues.map(function(value) {
            if (value.hasOwnProperty('value')) {
              return value.value;
            } else {
              return value;
            }
          });
        }
        if (!((possibleValues != null) && Array.isArray(possibleValues) && possibleValues.length)) {
          return value;
        }
        for (j = 0, len = possibleValues.length; j < len; j++) {
          possibleValue = possibleValues[j];
          if (_.isEqual(possibleValue, value)) {
            return value;
          }
        }
        throw new Error("Validation failed at " + keyPath + ", " + (JSON.stringify(value)) + " is not one of " + (JSON.stringify(possibleValues)));
      }
    }
  });

  isPlainObject = function(value) {
    return _.isObject(value) && !_.isArray(value) && !_.isFunction(value) && !_.isString(value) && !(value instanceof Color);
  };

  sortObject = function(value) {
    var j, key, len, ref1, result;
    if (!isPlainObject(value)) {
      return value;
    }
    result = {};
    ref1 = Object.keys(value).sort();
    for (j = 0, len = ref1.length; j < len; j++) {
      key = ref1[j];
      result[key] = sortObject(value[key]);
    }
    return result;
  };

  withoutEmptyObjects = function(object) {
    var key, newValue, resultObject, value;
    resultObject = void 0;
    if (isPlainObject(object)) {
      for (key in object) {
        value = object[key];
        newValue = withoutEmptyObjects(value);
        if (newValue != null) {
          if (resultObject == null) {
            resultObject = {};
          }
          resultObject[key] = newValue;
        }
      }
    } else {
      resultObject = object;
    }
    return resultObject;
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
