(function() {
  var CompositeDisposable, Disposable, FileIcons, TabView, layout, path, ref,
    slice = [].slice;

  path = require('path');

  ref = require('atom'), Disposable = ref.Disposable, CompositeDisposable = ref.CompositeDisposable;

  FileIcons = require('./file-icons');

  layout = require('./layout');

  module.exports = TabView = (function() {
    function TabView(arg) {
      var base, closeIcon, didClickCloseIcon, location;
      this.item = arg.item, this.pane = arg.pane, didClickCloseIcon = arg.didClickCloseIcon, this.tabs = arg.tabs, location = arg.location;
      if (typeof this.item.getPath === 'function') {
        this.path = this.item.getPath();
      }
      this.element = document.createElement('li');
      this.element.setAttribute('is', 'tabs-tab');
      if (['TextEditor', 'TestView'].indexOf(this.item.constructor.name) > -1) {
        this.element.classList.add('texteditor');
      }
      this.element.classList.add('tab', 'sortable');
      this.itemTitle = document.createElement('div');
      this.itemTitle.classList.add('title');
      this.element.appendChild(this.itemTitle);
      if (location === 'center' || !(typeof (base = this.item).isPermanentDockItem === "function" ? base.isPermanentDockItem() : void 0)) {
        closeIcon = document.createElement('div');
        closeIcon.classList.add('close-icon');
        closeIcon.onclick = didClickCloseIcon;
        this.element.appendChild(closeIcon);
      }
      this.subscriptions = new CompositeDisposable();
      this.handleEvents();
      this.updateDataAttributes();
      this.updateTitle();
      this.updateIcon();
      this.updateModifiedStatus();
      this.setupTooltip();
      if (this.isItemPending()) {
        this.itemTitle.classList.add('temp');
        this.element.classList.add('pending-tab');
      }
      this.element.ondrag = function(e) {
        return layout.drag(e);
      };
      this.element.ondragend = function(e) {
        return layout.end(e);
      };
      this.element.pane = this.pane;
      this.element.item = this.item;
      this.element.itemTitle = this.itemTitle;
      this.element.path = this.path;
    }

    TabView.prototype.handleEvents = function() {
      var base, iconChangedHandler, modifiedHandler, onDidChangeIconDisposable, onDidChangeModifiedDisposable, onDidChangePathDisposable, onDidChangeTitleDisposable, onDidSaveDisposable, onDidTerminatePendingStateDisposable, pathChangedHandler, titleChangedHandler;
      titleChangedHandler = (function(_this) {
        return function() {
          return _this.updateTitle();
        };
      })(this);
      this.subscriptions.add(this.pane.onDidDestroy((function(_this) {
        return function() {
          return _this.destroy();
        };
      })(this)));
      if (typeof this.pane.onItemDidTerminatePendingState === 'function') {
        this.subscriptions.add(this.pane.onItemDidTerminatePendingState((function(_this) {
          return function(item) {
            if (item === _this.item) {
              return _this.clearPending();
            }
          };
        })(this)));
      } else if (typeof this.item.onDidTerminatePendingState === 'function') {
        onDidTerminatePendingStateDisposable = this.item.onDidTerminatePendingState((function(_this) {
          return function() {
            return _this.clearPending();
          };
        })(this));
        if (Disposable.isDisposable(onDidTerminatePendingStateDisposable)) {
          this.subscriptions.add(onDidTerminatePendingStateDisposable);
        } else {
          console.warn("::onDidTerminatePendingState does not return a valid Disposable!", this.item);
        }
      }
      if (typeof this.item.onDidChangeTitle === 'function') {
        onDidChangeTitleDisposable = this.item.onDidChangeTitle(titleChangedHandler);
        if (Disposable.isDisposable(onDidChangeTitleDisposable)) {
          this.subscriptions.add(onDidChangeTitleDisposable);
        } else {
          console.warn("::onDidChangeTitle does not return a valid Disposable!", this.item);
        }
      } else if (typeof this.item.on === 'function') {
        this.item.on('title-changed', titleChangedHandler);
        this.subscriptions.add({
          dispose: (function(_this) {
            return function() {
              var base;
              return typeof (base = _this.item).off === "function" ? base.off('title-changed', titleChangedHandler) : void 0;
            };
          })(this)
        });
      }
      pathChangedHandler = (function(_this) {
        return function(path1) {
          _this.path = path1;
          _this.updateDataAttributes();
          _this.updateTitle();
          return _this.updateTooltip();
        };
      })(this);
      if (typeof this.item.onDidChangePath === 'function') {
        onDidChangePathDisposable = this.item.onDidChangePath(pathChangedHandler);
        if (Disposable.isDisposable(onDidChangePathDisposable)) {
          this.subscriptions.add(onDidChangePathDisposable);
        } else {
          console.warn("::onDidChangePath does not return a valid Disposable!", this.item);
        }
      } else if (typeof this.item.on === 'function') {
        this.item.on('path-changed', pathChangedHandler);
        this.subscriptions.add({
          dispose: (function(_this) {
            return function() {
              var base;
              return typeof (base = _this.item).off === "function" ? base.off('path-changed', pathChangedHandler) : void 0;
            };
          })(this)
        });
      }
      iconChangedHandler = (function(_this) {
        return function() {
          return _this.updateIcon();
        };
      })(this);
      if (typeof this.item.onDidChangeIcon === 'function') {
        onDidChangeIconDisposable = typeof (base = this.item).onDidChangeIcon === "function" ? base.onDidChangeIcon((function(_this) {
          return function() {
            return _this.updateIcon();
          };
        })(this)) : void 0;
        if (Disposable.isDisposable(onDidChangeIconDisposable)) {
          this.subscriptions.add(onDidChangeIconDisposable);
        } else {
          console.warn("::onDidChangeIcon does not return a valid Disposable!", this.item);
        }
      } else if (typeof this.item.on === 'function') {
        this.item.on('icon-changed', iconChangedHandler);
        this.subscriptions.add({
          dispose: (function(_this) {
            return function() {
              var base1;
              return typeof (base1 = _this.item).off === "function" ? base1.off('icon-changed', iconChangedHandler) : void 0;
            };
          })(this)
        });
      }
      modifiedHandler = (function(_this) {
        return function() {
          return _this.updateModifiedStatus();
        };
      })(this);
      if (typeof this.item.onDidChangeModified === 'function') {
        onDidChangeModifiedDisposable = this.item.onDidChangeModified(modifiedHandler);
        if (Disposable.isDisposable(onDidChangeModifiedDisposable)) {
          this.subscriptions.add(onDidChangeModifiedDisposable);
        } else {
          console.warn("::onDidChangeModified does not return a valid Disposable!", this.item);
        }
      } else if (typeof this.item.on === 'function') {
        this.item.on('modified-status-changed', modifiedHandler);
        this.subscriptions.add({
          dispose: (function(_this) {
            return function() {
              var base1;
              return typeof (base1 = _this.item).off === "function" ? base1.off('modified-status-changed', modifiedHandler) : void 0;
            };
          })(this)
        });
      }
      if (typeof this.item.onDidSave === 'function') {
        onDidSaveDisposable = this.item.onDidSave((function(_this) {
          return function(event) {
            _this.terminatePendingState();
            if (event.path !== _this.path) {
              _this.path = event.path;
              if (atom.config.get('tabs.enableVcsColoring')) {
                return _this.setupVcsStatus();
              }
            }
          };
        })(this));
        if (Disposable.isDisposable(onDidSaveDisposable)) {
          this.subscriptions.add(onDidSaveDisposable);
        } else {
          console.warn("::onDidSave does not return a valid Disposable!", this.item);
        }
      }
      this.subscriptions.add(atom.config.observe('tabs.showIcons', (function(_this) {
        return function() {
          return _this.updateIconVisibility();
        };
      })(this)));
      return this.subscriptions.add(atom.config.observe('tabs.enableVcsColoring', (function(_this) {
        return function(isEnabled) {
          if (isEnabled && (_this.path != null)) {
            return _this.setupVcsStatus();
          } else {
            return _this.unsetVcsStatus();
          }
        };
      })(this)));
    };

    TabView.prototype.setupTooltip = function() {
      var onMouseEnter;
      onMouseEnter = (function(_this) {
        return function() {
          _this.mouseEnterSubscription.dispose();
          _this.hasBeenMousedOver = true;
          _this.updateTooltip();
          return _this.element.dispatchEvent(new CustomEvent('mouseenter', {
            bubbles: true
          }));
        };
      })(this);
      this.mouseEnterSubscription = {
        dispose: (function(_this) {
          return function() {
            _this.element.removeEventListener('mouseenter', onMouseEnter);
            return _this.mouseEnterSubscription = null;
          };
        })(this)
      };
      return this.element.addEventListener('mouseenter', onMouseEnter);
    };

    TabView.prototype.updateTooltip = function() {
      if (!this.hasBeenMousedOver) {
        return;
      }
      this.destroyTooltip();
      if (this.path) {
        return this.tooltip = atom.tooltips.add(this.element, {
          title: this.path,
          html: false,
          delay: {
            show: 1000,
            hide: 100
          },
          placement: 'bottom'
        });
      }
    };

    TabView.prototype.destroyTooltip = function() {
      var ref1;
      if (!this.hasBeenMousedOver) {
        return;
      }
      return (ref1 = this.tooltip) != null ? ref1.dispose() : void 0;
    };

    TabView.prototype.destroy = function() {
      var ref1, ref2, ref3;
      if ((ref1 = this.subscriptions) != null) {
        ref1.dispose();
      }
      if ((ref2 = this.mouseEnterSubscription) != null) {
        ref2.dispose();
      }
      if ((ref3 = this.repoSubscriptions) != null) {
        ref3.dispose();
      }
      this.destroyTooltip();
      return this.element.remove();
    };

    TabView.prototype.updateDataAttributes = function() {
      var itemClass, ref1;
      if (this.path) {
        this.itemTitle.dataset.name = path.basename(this.path);
        this.itemTitle.dataset.path = this.path;
      } else {
        delete this.itemTitle.dataset.name;
        delete this.itemTitle.dataset.path;
      }
      if (itemClass = (ref1 = this.item.constructor) != null ? ref1.name : void 0) {
        return this.element.dataset.type = itemClass;
      } else {
        return delete this.element.dataset.type;
      }
    };

    TabView.prototype.updateTitle = function(arg) {
      var base, base1, i, len, ref1, ref2, ref3, ref4, tab, title, updateSiblings, useLongTitle;
      ref1 = arg != null ? arg : {}, updateSiblings = ref1.updateSiblings, useLongTitle = ref1.useLongTitle;
      if (this.updatingTitle) {
        return;
      }
      this.updatingTitle = true;
      if (updateSiblings === false) {
        title = this.item.getTitle();
        if (useLongTitle) {
          title = (ref2 = typeof (base = this.item).getLongTitle === "function" ? base.getLongTitle() : void 0) != null ? ref2 : title;
        }
        this.itemTitle.textContent = title;
      } else {
        title = this.item.getTitle();
        useLongTitle = false;
        ref3 = this.tabs;
        for (i = 0, len = ref3.length; i < len; i++) {
          tab = ref3[i];
          if (tab !== this) {
            if (tab.item.getTitle() === title) {
              tab.updateTitle({
                updateSiblings: false,
                useLongTitle: true
              });
              useLongTitle = true;
            }
          }
        }
        if (useLongTitle) {
          title = (ref4 = typeof (base1 = this.item).getLongTitle === "function" ? base1.getLongTitle() : void 0) != null ? ref4 : title;
        }
        this.itemTitle.textContent = title;
      }
      return this.updatingTitle = false;
    };

    TabView.prototype.updateIcon = function() {
      var base, names, ref1, ref2;
      if (this.iconName) {
        names = !Array.isArray(this.iconName) ? this.iconName.split(/\s+/g) : this.iconName;
        (ref1 = this.itemTitle.classList).remove.apply(ref1, ['icon', "icon-" + names[0]].concat(slice.call(names)));
      }
      if (this.iconName = typeof (base = this.item).getIconName === "function" ? base.getIconName() : void 0) {
        return this.itemTitle.classList.add('icon', "icon-" + this.iconName);
      } else if ((this.path != null) && (this.iconName = FileIcons.getService().iconClassForPath(this.path, "tabs"))) {
        if (!Array.isArray(names = this.iconName)) {
          names = names.toString().split(/\s+/g);
        }
        return (ref2 = this.itemTitle.classList).add.apply(ref2, ['icon'].concat(slice.call(names)));
      }
    };

    TabView.prototype.isItemPending = function() {
      if (this.pane.getPendingItem != null) {
        return this.pane.getPendingItem() === this.item;
      } else if (this.item.isPending != null) {
        return this.item.isPending();
      }
    };

    TabView.prototype.terminatePendingState = function() {
      if (this.pane.clearPendingItem != null) {
        if (this.pane.getPendingItem() === this.item) {
          return this.pane.clearPendingItem();
        }
      } else if (this.item.terminatePendingState != null) {
        return this.item.terminatePendingState();
      }
    };

    TabView.prototype.clearPending = function() {
      this.itemTitle.classList.remove('temp');
      return this.element.classList.remove('pending-tab');
    };

    TabView.prototype.updateIconVisibility = function() {
      if (atom.config.get('tabs.showIcons')) {
        return this.itemTitle.classList.remove('hide-icon');
      } else {
        return this.itemTitle.classList.add('hide-icon');
      }
    };

    TabView.prototype.updateModifiedStatus = function() {
      var base;
      if (typeof (base = this.item).isModified === "function" ? base.isModified() : void 0) {
        if (!this.isModified) {
          this.element.classList.add('modified');
        }
        return this.isModified = true;
      } else {
        if (this.isModified) {
          this.element.classList.remove('modified');
        }
        return this.isModified = false;
      }
    };

    TabView.prototype.setupVcsStatus = function() {
      if (this.path == null) {
        return;
      }
      return this.repoForPath(this.path).then((function(_this) {
        return function(repo) {
          _this.subscribeToRepo(repo);
          return _this.updateVcsStatus(repo);
        };
      })(this));
    };

    TabView.prototype.subscribeToRepo = function(repo) {
      var ref1;
      if (repo == null) {
        return;
      }
      if ((ref1 = this.repoSubscriptions) != null) {
        ref1.dispose();
      }
      this.repoSubscriptions = new CompositeDisposable();
      this.repoSubscriptions.add(repo.onDidChangeStatus((function(_this) {
        return function(event) {
          if (event.path === _this.path) {
            return _this.updateVcsStatus(repo, event.pathStatus);
          }
        };
      })(this)));
      return this.repoSubscriptions.add(repo.onDidChangeStatuses((function(_this) {
        return function() {
          return _this.updateVcsStatus(repo);
        };
      })(this)));
    };

    TabView.prototype.repoForPath = function() {
      var dir, i, len, ref1;
      ref1 = atom.project.getDirectories();
      for (i = 0, len = ref1.length; i < len; i++) {
        dir = ref1[i];
        if (dir.contains(this.path)) {
          return atom.project.repositoryForDirectory(dir);
        }
      }
      return Promise.resolve(null);
    };

    TabView.prototype.updateVcsStatus = function(repo, status) {
      var newStatus;
      if (repo == null) {
        return;
      }
      newStatus = null;
      if (repo.isPathIgnored(this.path)) {
        newStatus = 'ignored';
      } else {
        if (status == null) {
          status = repo.getCachedPathStatus(this.path);
        }
        if (repo.isStatusModified(status)) {
          newStatus = 'modified';
        } else if (repo.isStatusNew(status)) {
          newStatus = 'added';
        }
      }
      if (newStatus !== this.status) {
        this.status = newStatus;
        return this.updateVcsColoring();
      }
    };

    TabView.prototype.updateVcsColoring = function() {
      this.itemTitle.classList.remove('status-ignored', 'status-modified', 'status-added');
      if (this.status && atom.config.get('tabs.enableVcsColoring')) {
        return this.itemTitle.classList.add("status-" + this.status);
      }
    };

    TabView.prototype.unsetVcsStatus = function() {
      var ref1;
      if ((ref1 = this.repoSubscriptions) != null) {
        ref1.dispose();
      }
      delete this.status;
      return this.updateVcsColoring();
    };

    return TabView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
